/* Copyright 2013 Chris Wilson

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

// Initialize the MIDI library.
(function (global) {
    'use strict';
    var midiIO, _requestMIDIAccess, _delayedInit, MIDIAccess, _onReady, _onNotReady, MIDIPort, MIDIInput, MIDIOutput, _midiProc;

    function Promise() {

    }

    Promise.prototype.then = function(accept, reject) {
        this.accept = accept;
        this.reject = reject;
    }

    Promise.prototype.succeed = function(access) {
        if (this.accept)
            this.accept(access);
    }

    Promise.prototype.fail = function(error) {
        if (this.reject)
            this.reject(error);
    }

    function _JazzInstance() {
        this.inputInUse = false;
        this.outputInUse = false;

        // load the Jazz plugin
        var o1 = document.createElement("object");
        o1.id = "_Jazz" + Math.random() + "ie";
        o1.classid = "CLSID:1ACE1618-1C7D-4561-AEE1-34842AA85E90";

        this.activeX = o1;

        var o2 = document.createElement("object");
        o2.id = "_Jazz" + Math.random();
        o2.type="audio/x-jazz";
        o1.appendChild(o2);

        this.objRef = o2;

        var e = document.createElement("p");
        e.appendChild(document.createTextNode("This page requires the "));

        var a = document.createElement("a");
        a.appendChild(document.createTextNode("Jazz plugin"));
        a.href = "http://jazz-soft.net/";

        e.appendChild(a);
        e.appendChild(document.createTextNode("."));
        o2.appendChild(e);

        var insertionPoint = document.getElementById("MIDIPlugin");
        if (!insertionPoint) {
            // Create hidden element
            var insertionPoint = document.createElement("div");
            insertionPoint.id = "MIDIPlugin";
            insertionPoint.style.position = "absolute";
            insertionPoint.style.visibility = "hidden";
            insertionPoint.style.left = "-9999px";
            insertionPoint.style.top = "-9999px";
            document.body.appendChild(insertionPoint);
        }
        insertionPoint.appendChild(o1);
    }

    _JazzInstance.prototype._init = function() {
        if (this.objRef.isJazz) {
            this._Jazz = this.objRef;
        } else if (this.activeX.isJazz) {
            this._Jazz = this.activeX;
        } else {
            this._Jazz = null;
        }
        if (this._Jazz) {
            this._Jazz._jazzTimeZero = this._Jazz.Time();
            this._Jazz._perfTimeZero = window.performance.now();
        }
    };

    _JazzInstance.prototype._delayedInit = function(then) {
        var that = this;
        setTimeout(function() {
            that._init();
            then();
        }, 100);
    };

    _requestMIDIAccess = function _requestMIDIAccess() {
        var access = new MIDIAccess();
        return access._promise;
    };

    // API Methods

    MIDIAccess = function() {
        this._jazzInstances = new Array();
        var instance = new _JazzInstance();
        this._jazzInstances.push( instance );
        this._promise = new Promise;
        instance._delayedInit(function() {
            if (instance._Jazz) {
                this._Jazz = instance._Jazz;
                window.setTimeout( _onReady.bind(this), 3 );
            } else {
                window.setTimeout( _onNotReady.bind(this), 3 );
            }
        }.bind(this));
    };

    _onReady = function() {
        if (this._promise)
            this._promise.succeed(this);
    };

    _onNotReady = function() {
        if (this._promise)
            this._promise.fail( { code: 1 } );
    };

    MIDIAccess.prototype.inputs = function(  ) {
        if (!this._Jazz)
              return null;
        var list=this._Jazz.MidiInList();
        var inputs = new Array( list.length );

        for ( var i=0; i<list.length; i++ ) {
            inputs[i] = new MIDIInput( this, list[i], i );
        }
        return inputs;
    }

    MIDIAccess.prototype.outputs = function(  ) {
        if (!this._Jazz)
            return null;
        var list=this._Jazz.MidiOutList();
        var outputs = new Array( list.length );

        for ( var i=0; i<list.length; i++ ) {
            outputs[i] = new MIDIOutput( this, list[i], i );
        }
        return outputs;
    };

    MIDIInput = function MIDIInput( midiAccess, name, index ) {
        this._listeners = [];
        this._midiAccess = midiAccess;
        this._index = index;
        this._inLongSysexMessage = false;
        this._sysexBuffer = new Uint8Array();
        this.id = "" + index + "." + name;
        this.manufacturer = "";
        this.name = name;
        this.type = "input";
        this.version = "";
        this.onmidimessage = null;

        var inputInstance = null;
        var then = function() {
            this._jazzInstance = inputInstance._Jazz;
            this._input = this._jazzInstance.MidiInOpen( this._index, _midiProc.bind(this) );
        };
        for (var i=0; (i<midiAccess._jazzInstances.length)&&(!inputInstance); i++) {
            if (!midiAccess._jazzInstances[i].inputInUse)
                inputInstance=midiAccess._jazzInstances[i];
        }
        if (!inputInstance) {
            inputInstance = new _JazzInstance();
            midiAccess._jazzInstances.push( inputInstance );
            inputInstance.inputInUse = true;
            inputInstance._delayedInit(then.bind(this));
        } else {
            inputInstance.inputInUse = true;
            then.bind(this).apply();
        }
    };

    // Introduced in DOM Level 2:
    MIDIInput.prototype.addEventListener = function (type, listener, useCapture ) {
        if (type !== "midimessage")
            return;
        for (var i=0; i<this._listeners.length; i++)
            if (this._listeners[i] == listener)
                return;
        this._listeners.push( listener );
    };

    MIDIInput.prototype.removeEventListener = function (type, listener, useCapture ) {
        if (type !== "midimessage")
            return;
        for (var i=0; i<this._listeners.length; i++)
            if (this._listeners[i] == listener) {
                this._listeners.splice( i, 1 );  //remove it
                return;
            }
    };

    MIDIInput.prototype.preventDefault = function() {
        this._pvtDef = true;
    };

    MIDIInput.prototype.dispatchEvent = function (evt) {
        this._pvtDef = false;

        // dispatch to listeners
        for (var i=0; i<this._listeners.length; i++)
            if (this._listeners[i].handleEvent)
                this._listeners[i].handleEvent.bind(this)( evt );
            else
                this._listeners[i].bind(this)( evt );

        if (this.onmidimessage)
            this.onmidimessage( evt );

        return this._pvtDef;
    };

    MIDIInput.prototype.appendToSysexBuffer = function ( data ) {
        var oldLength = this._sysexBuffer.length;
        var tmpBuffer = new Uint8Array( oldLength + data.length );
        tmpBuffer.set( this._sysexBuffer );
        tmpBuffer.set( data, oldLength );
        this._sysexBuffer = tmpBuffer;
    };

    MIDIInput.prototype.bufferLongSysex = function ( data, initialOffset ) {
        var j = initialOffset;
        while (j<data.length) {
            if (data[j] == 0xF7) {
                // end of sysex!
                j++;
                this.appendToSysexBuffer( data.slice(initialOffset, j) );
                return j;
            }
            j++;
        }
        // didn't reach the end; just tack it on.
        this.appendToSysexBuffer( data.slice(initialOffset, j) );
        this._inLongSysexMessage = true;
        return j;
    };

    _midiProc = function _midiProc( timestamp, data ) {
        // Have to use createEvent/initEvent because IE10 fails on new CustomEvent.  Thanks, IE!
        var length = 0;
        var i,j;
        var isSysexMessage = false;

        // Jazz sometimes passes us multiple messages at once, so we need to parse them out
        // and pass them one at a time.

        for (i=0; i<data.length; i+=length) {
            if (this._inLongSysexMessage) {
                i = this.bufferLongSysex(data,i);
                if ( data[i-1] != 0xf7 ) {
                    // ran off the end without hitting the end of the sysex message
                    return;
                }
                isSysexMessage = true;
            } else {
                isSysexMessage = false;
                switch (data[i] & 0xF0) {
                    case 0x80:  // note off
                    case 0x90:  // note on
                    case 0xA0:  // polyphonic aftertouch
                    case 0xB0:  // control change
                    case 0xE0:  // channel mode
                        length = 3;
                        break;

                    case 0xC0:  // program change
                    case 0xD0:  // channel aftertouch
                        length = 2;
                        break;

                    case 0xF0:
                        switch (data[i]) {
                            case 0xf0:  // variable-length sysex.
                                i = this.bufferLongSysex(data,i);
                                if ( data[i-1] != 0xf7 ) {
                                    // ran off the end without hitting the end of the sysex message
                                    return;
                                }
                                isSysexMessage = true;
                                break;

                            case 0xF1:  // MTC quarter frame
                            case 0xF3:  // song select
                                length = 2;
                                break;

                            case 0xF2:  // song position pointer
                                length = 3;
                                break;

                            default:
                                length = 1;
                                break;
                        }
                        break;
                }
            }
            var evt = document.createEvent( "Event" );
            evt.initEvent( "midimessage", false, false );
            evt.receivedTime = parseFloat( timestamp.toString()) + this._jazzInstance._perfTimeZero;
            if (isSysexMessage || this._inLongSysexMessage) {
                evt.data = new Uint8Array( this._sysexBuffer );
                this._sysexBuffer = new Uint8Array(0);
                this._inLongSysexMessage = false;
            } else
                evt.data = new Uint8Array(data.slice(i, length+i));
            this.dispatchEvent( evt );
        }
    };

    MIDIOutput = function MIDIOutput( midiAccess, name, index ) {
        this._listeners = [];
        this._midiAccess = midiAccess;
        this._index = index;
        this.id = "" + index + "." + name;
        this.manufacturer = "";
        this.name = name;
        this.type = "output";
        this.version = "";

        var outputInstance = null;
        var then = function() {
            this._jazzInstance = outputInstance._Jazz;
            this._jazzInstance.MidiOutOpen(this.name);
        };
        for (var i=0; (i<midiAccess._jazzInstances.length)&&(!outputInstance); i++) {
            if (!midiAccess._jazzInstances[i].outputInUse)
                outputInstance=midiAccess._jazzInstances[i];
        }
        if (!outputInstance) {
            outputInstance = new _JazzInstance();
            midiAccess._jazzInstances.push( outputInstance );
            outputInstance.outputInUse = true;
            outputInstance._delayedInit(then.bind(this));
        } else {
            outputInstance.outputInUse = true;
            then.bind(this).apply();
        }
    };

    function _sendLater() {
        this.jazz.MidiOutLong( this.data );    // handle send as sysex
    }

    MIDIOutput.prototype.send = function( data, timestamp ) {
        var delayBeforeSend = 0;
        if (data.length === 0)
            return false;

        if (timestamp)
            delayBeforeSend = Math.floor( timestamp - window.performance.now() );

        if (timestamp && (delayBeforeSend>1)) {
            var sendObj = new Object();
            sendObj.jazz = this._jazzInstance;
            sendObj.data = data;

            window.setTimeout( _sendLater.bind(sendObj), delayBeforeSend );
        } else {
            this._jazzInstance.MidiOutLong( data );
        }
        return true;
    };

    //init: create plugin
    if (!window.navigator.requestMIDIAccess)
        window.navigator.requestMIDIAccess = _requestMIDIAccess;

}(window));

// Polyfill window.performance.now() if necessary.
(function (exports) {
    var perf = {}, props;

    function findAlt() {
        var prefix = ['moz', 'webkit', 'o', 'ms'],
        i = prefix.length,
            //worst case, we use Date.now()
            props = {
                value: (function (start) {
                    return function () {
                        return Date.now() - start;
                    };
                }(Date.now()))
            };

        //seach for vendor prefixed version
        for (; i >= 0; i--) {
            if ((prefix[i] + "Now") in exports.performance) {
                props.value = function (method) {
                    return function () {
                        exports.performance[method]();
                    }
                }(prefix[i] + "Now");
                return props;
            }
        }

        //otherwise, try to use connectionStart
        if ("timing" in exports.performance && "connectStart" in exports.performance.timing) {
            //this pretty much approximates performance.now() to the millisecond
            props.value = (function (start) {
                return function() {
                    Date.now() - start;
                };
            }(exports.performance.timing.connectStart));
        }
        return props;
    }

    //if already defined, bail
    if (("performance" in exports) && ("now" in exports.performance))
        return;
    if (!("performance" in exports))
        Object.defineProperty(exports, "performance", {
            get: function () {
                return perf;
            }});
        //otherwise, performance is there, but not "now()"

    props = findAlt();
    Object.defineProperty(exports.performance, "now", props);
}(window));

// MIDIEvents : Read and edit events from various sources (ArrayBuffer, Stream)
function MIDIEvents() {
	throw new Error('MIDIEvents function not intended to be run.');
}

// Static constants
// Event types
MIDIEvents.EVENT_META=0xFF;
MIDIEvents.EVENT_SYSEX=0xF0;
MIDIEvents.EVENT_DIVSYSEX=0xF7;
MIDIEvents.EVENT_MIDI=0x8;
// Meta event types
MIDIEvents.EVENT_META_SEQUENCE_NUMBER=0x00,
MIDIEvents.EVENT_META_TEXT=0x01,
MIDIEvents.EVENT_META_COPYRIGHT_NOTICE=0x02,
MIDIEvents.EVENT_META_TRACK_NAME=0x03,
MIDIEvents.EVENT_META_INSTRUMENT_NAME=0x04,
MIDIEvents.EVENT_META_LYRICS=0x05,
MIDIEvents.EVENT_META_MARKER=0x06,
MIDIEvents.EVENT_META_CUE_POINT=0x07,
MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX=0x20,
MIDIEvents.EVENT_META_END_OF_TRACK=0x2F,
MIDIEvents.EVENT_META_SET_TEMPO=0x51,
MIDIEvents.EVENT_META_SMTPE_OFFSET=0x54,
MIDIEvents.EVENT_META_TIME_SIGNATURE=0x58,
MIDIEvents.EVENT_META_KEY_SIGNATURE=0x59,
MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC=0x7F;
// MIDI event types
MIDIEvents.EVENT_MIDI_NOTE_OFF=0x8,
MIDIEvents.EVENT_MIDI_NOTE_ON=0x9,
MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH=0xA,
MIDIEvents.EVENT_MIDI_CONTROLLER=0xB,
MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE=0xC,
MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH=0xD,
MIDIEvents.EVENT_MIDI_PITCH_BEND=0xE;
// MIDI event sizes
MIDIEvents.MIDI_1PARAM_EVENTS=[
	MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE,
	MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH
];
MIDIEvents.MIDI_2PARAMS_EVENTS=[
	MIDIEvents.EVENT_MIDI_NOTE_OFF,
	MIDIEvents.EVENT_MIDI_NOTE_ON,
	MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH,
	MIDIEvents.EVENT_MIDI_CONTROLLER,
	MIDIEvents.EVENT_MIDI_PITCH_BEND
];

// Create an event stream parser
MIDIEvents.createParser=function(stream, startAt, strictMode) {
	// Wrap DataView into a data stream
	if(stream instanceof DataView) {
		stream={
			'position':startAt||0,
			'buffer':stream,
			'readUint8':function() {
				return this.buffer.getUint8(this.position++);
			},
			'readUint16':function() {
				var v=this.buffer.getUint16(this.position);
				this.position=this.position+2;
				return v;
			},
			'readUint32':function() {
				var v=this.buffer.getUint16(this.position);
				this.position=this.position+2;
				return v;
			},
			'readVarInt':function() {
				var v=0, i=0;
				while(i++<4) {
					var b=this.readUint8();
					if (b&0x80) {
						v+=(b&0x7f);
						v<<=7;
					} else {
						return v+b;
					}
				}
				throw new Error('0x'+this.position.toString(16)+': Variable integer'
					+' length cannot exceed 4 bytes');
			},
			'readBytes':function(l) {
				var bytes=[];
				for(l; l>0; l--) {
					bytes.push(this.readUint8());
				}
				return bytes;
			},
			'pos':function() {
				return '0x'+(this.buffer.byteOffset+this.position).toString(16);
			},
			'end':function(l) {
				return this.position===this.buffer.byteLength;
			}
		}
		startAt=0;
	}
	// Consume stream till not at start index
	if(startAt>0) {
		while(startAt--)
			stream.readUint8();
	}
	// Private vars
	// Common vars
	var deltaTime, eventTypeByte, lastEventTypeByte, event,
	// MIDI events vars
		MIDIEventType, MIDIEventChannel, MIDIEventParam1, MIDIEventParam2;
	// creating the parser object
	return {
		// Read the next event
		'next':function() {
			// Check available datas
			if(stream.end())
				return null;
			// Creating the event
			event={
				// Memoize the event index
				'index':stream.pos(),
				// Read the delta time
				'delta':stream.readVarInt()
			};
			// Read the eventTypeByte
			eventTypeByte=stream.readUint8();
			if((eventTypeByte&0xF0) == 0xF0) {
				// Meta events
				if(eventTypeByte==MIDIEvents.EVENT_META) {
					event.type=MIDIEvents.EVENT_META;
					event.subtype=stream.readUint8();
					event.length=stream.readVarInt();
					switch(event.subtype) {
						case MIDIEvents.EVENT_META_SEQUENCE_NUMBER:
							if(strictMode&&2!==event.length)
								throw new Error(stream.pos()+' Bad metaevent length.');
							event.msb=stream.readUint8();
							event.lsb=stream.readUint8();
							return event;
							break;
						case MIDIEvents.EVENT_META_TEXT:
						case MIDIEvents.EVENT_META_COPYRIGHT_NOTICE:
						case MIDIEvents.EVENT_META_TRACK_NAME:
						case MIDIEvents.EVENT_META_INSTRUMENT_NAME:
						case MIDIEvents.EVENT_META_LYRICS:
						case MIDIEvents.EVENT_META_MARKER:
						case MIDIEvents.EVENT_META_CUE_POINT:
							event.data=stream.readBytes(event.length);
							return event;
							break;
						case MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX:
							if(strictMode&&1!==event.length)
								throw new Error(stream.pos()+' Bad metaevent length.');
							event.prefix=stream.readUint8();
							return event;
							break;
						case MIDIEvents.EVENT_META_END_OF_TRACK:
							if(strictMode&&0!==event.length)
								throw new Error(stream.pos()+' Bad metaevent length.');
							return event;
							break;
						case MIDIEvents.EVENT_META_SET_TEMPO:
							if(strictMode&&3!==event.length) {
								throw new Error(stream.pos()+' Tempo meta event length must'
									+' be 3.');
							}
							event.tempo=((stream.readUint8() << 16)
								+ (stream.readUint8() << 8)
								+ stream.readUint8());
							event.tempoBPM=60000000/event.tempo;
							return event;
							break;
						case MIDIEvents.EVENT_META_SMTPE_OFFSET:
							if(strictMode&&5!==event.length) {
								throw new Error(stream.pos()+' Bad metaevent length.');
							}
							event.hour=stream.readUint8();
							if(strictMode&&event.hour>23) {
								throw new Error(stream.pos()+' SMTPE offset hour value must'
									+' be part of 0-23.');
							}
							event.minutes=stream.readUint8();
							if(strictMode&&event.minutes>59) {
								throw new Error(stream.pos()+' SMTPE offset minutes value'
									+' must be part of 0-59.');
							}
							event.seconds=stream.readUint8();
							if(strictMode&&event.seconds>59) {
								throw new Error(stream.pos()+' SMTPE offset seconds value'
									+' must be part of 0-59.');
							}
							event.frames=stream.readUint8();
							if(strictMode&&event.frames>30) {
								throw new Error(stream.pos()+' SMTPE offset frames value must'
									+' be part of 0-30.');
							}
							event.subframes=stream.readUint8();
							if(strictMode&&event.subframes>99) {
								throw new Error(stream.pos()+' SMTPE offset subframes value'
									+' must be part of 0-99.');
							}
							return event;
							break;
						case MIDIEvents.EVENT_META_KEY_SIGNATURE:
							if(strictMode&&2!==event.length) {
								throw new Error(stream.pos()+' Bad metaevent length.');
							}
							event.key=stream.readUint8();
							if(strictMode&&(event.key<-7||event.key>7)) {
								throw new Error(stream.pos()+' Bad metaevent length.');
							}
							event.scale=stream.readUint8();
							if(strictMode&&event.scale!==0&&event.scale!==1) {
								throw new Error(stream.pos()+' Key signature scale value must'
									+' be 0 or 1.');
							}
							return event;
							break;
						 // Not implemented
						case MIDIEvents.EVENT_META_TIME_SIGNATURE:
							if(strictMode&&4!==event.length)
								throw new Error(stream.pos()+' Bad metaevent length.');
							event.data=stream.readBytes(event.length);
							return event;
							break;
						case MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC:
							event.data=stream.readBytes(event.length);
							return event;
							break;
						default:
							if(strictMode)
								throw new Error(stream.pos()+' Unknown meta event type '
									+'('+event.subtype.toString(16)+').');
							event.data=stream.readBytes(event.length);
							return event;
							break;
					}
				// System events
				} else if(eventTypeByte==MIDIEvents.EVENT_SYSEX
						||eventTypeByte==MIDIEvents.EVENT_DIVSYSEX) {
					event.type=eventTypeByte;
					event.length=stream.readVarInt();
					event.data=stream.readBytes(event.length);
					return event;
				// Unknown event, assuming it's system like event
				} else {
					if(strictMode)
						throw new Error(stream.pos()+' Unknown event type '
							+eventTypeByte.toString(16)+', Delta: '+event.delta+'.');
					event.type=eventTypeByte;
					event.badsubtype=stream.readVarInt();
					event.length=stream.readUint8();
					event.data=stream.readBytes(event.length);
					return event;
				}
			// MIDI eventsdestination[index++]
			} else {
				// running status
				if((eventTypeByte&0x80)==0) {
					if(!(MIDIEventType))
						throw new Error(stream.pos()+' Running status without previous event');
					MIDIEventParam1=eventTypeByte;
				} else {
					MIDIEventType=eventTypeByte>>4;
					MIDIEventChannel=eventTypeByte&0x0F;
					MIDIEventParam1=stream.readUint8();
				}
				event.type=MIDIEvents.EVENT_MIDI;
				event.subtype=MIDIEventType;
				event.channel=MIDIEventChannel;
				event.param1=MIDIEventParam1;
				switch(MIDIEventType) {
					case MIDIEvents.EVENT_MIDI_NOTE_OFF:
						event.param2=stream.readUint8();
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_NOTE_ON:
						var velocity=stream.readUint8();
						// If velocity is 0, it's a note off event in fact
						if(!velocity) {
							event.subtype=MIDIEvents.EVENT_MIDI_NOTE_OFF;
							event.param2=127; // Find a standard telling what to do here
						} else {
							event.param2=velocity;
						}
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_NOTE_AFTERTOUCH:
						event.param2=stream.readUint8();
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_CONTROLLER:
						event.param2=stream.readUint8();
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_PROGRAM_CHANGE:
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_CHANNEL_AFTERTOUCH:
						return event;
						break;
					case MIDIEvents.EVENT_MIDI_PITCH_BEND:
						event.param2=stream.readUint8();
						return event;
						break;
					default:
						if(strictMode)
							throw new Error(stream.pos()+' Unknown MIDI event type '
								+'('+MIDIEventType.toString(16)+').');
						return event;
						break;
				}
			}
		}
	};
};

// Return the buffer length needed to encode the given events
MIDIEvents.writeToTrack=function(events, destination) {
	var index=0;
	// Converting each event to binary MIDI datas
	for(var i=0, j=events.length; i<j; i++) {
		// Writing delta value
		if(events[i].delta>>>28) {
			throw Error('Event #'+i+': Maximum delta time value reached ('
				+events[i].delta+'/134217728 max)');
		}
		if(events[i].delta>>>21) {
			destination[index++]=((events[i].delta>>>21)&0x7F)|0x80;
		}
		if(events[i].delta>>>14) {
			destination[index++]=((events[i].delta>>>14)&0x7F)|0x80;
		}
		if(events[i].delta>>>7) {
			destination[index++]=((events[i].delta>>>7)&0x7F)|0x80;
		}
		destination[index++]=(events[i].delta&0x7F);
		// MIDI Events encoding
		if(events[i].type===MIDIEvents.EVENT_MIDI) {
			// Adding the byte of subtype + channel
			destination[index++]=(events[i].subtype<<4)+events[i].channel
			// Adding the byte of the first params
			destination[index++]=events[i].param1;
			// Adding a byte for the optionnal second param
			if(-1!==MIDIEvents.MIDI_2PARAMS_EVENTS.indexOf(events[i].subtype)) {
				destination[index++]=events[i].param2;
			}
		// META / SYSEX events encoding
		} else {
			// Adding the event type byte
			destination[index++]=events[i].type;
			// Adding the META event subtype byte
			if(events[i].type===MIDIEvents.EVENT_META) {
				destination[index++]=events[i].subtype;
			}
			// Writing the event length bytes
			if(events[i].length>>>28) {
				throw Error('Event #'+i+': Maximum length reached ('
					+events[i].length+'/134217728 max)');
			}
			if(events[i].length>>>21) {
				destination[index++]=((events[i].length>>>21)&0x7F)|0x80;
			}
			if(events[i].length>>>14) {
				destination[index++]=((events[i].length>>>14)&0x7F)|0x80;
			}
			if(events[i].length>>>7) {
				destination[index++]=((events[i].length>>>7)&0x7F)|0x80;
			}
			destination[index++]=(events[i].length&0x7F);
			if(events[i].type===MIDIEvents.EVENT_META) {
				switch(events[i].subtype) {
					case MIDIEvents.EVENT_META_SEQUENCE_NUMBER:
						destination[index++]=events[i].msb;
						destination[index++]=events[i].lsb;
						break;
					case MIDIEvents.EVENT_META_TEXT:
					case MIDIEvents.EVENT_META_COPYRIGHT_NOTICE:
					case MIDIEvents.EVENT_META_TRACK_NAME:
					case MIDIEvents.EVENT_META_INSTRUMENT_NAME:
					case MIDIEvents.EVENT_META_LYRICS:
					case MIDIEvents.EVENT_META_MARKER:
					case MIDIEvents.EVENT_META_CUE_POINT:
						for(var k=0, l=events[i].length; k<l; k++) {
							destination[index++]=events[i].data[k];
						}
						break;
					case MIDIEvents.EVENT_META_MIDI_CHANNEL_PREFIX:
						destination[index++]=events[i].prefix;
						return event;
						break;
					case MIDIEvents.EVENT_META_END_OF_TRACK:
						break;
					case MIDIEvents.EVENT_META_SET_TEMPO:
						destination[index++]=(events[i].tempo >> 16);
						destination[index++]=(events[i].tempo >> 8) & 0xFF;
						destination[index++]=events[i].tempo & 0xFF;
						break;
					case MIDIEvents.EVENT_META_SMTPE_OFFSET:
						if(strictMode&&event.hour>23) {
							throw new Error('Event #'+i+': SMTPE offset hour value must be'
								+' part of 0-23.');
						}
						destination[index++]=events[i].hour;
						if(strictMode&&event.minutes>59) {
							throw new Error('Event #'+i+': SMTPE offset minutes value must'
								+' be part of 0-59.');
						}
						destination[index++]=events[i].minutes;
						if(strictMode&&event.seconds>59) {
							throw new Error('Event #'+i+': SMTPE offset seconds value must'
							+' be part of 0-59.');
						}
						destination[index++]=events[i].seconds;
						if(strictMode&&event.frames>30) {
							throw new Error('Event #'+i+': SMTPE offset frames amount must'
								+' be part of 0-30.');
						}
						destination[index++]=events[i].frames;
						if(strictMode&&event.subframes>99) {
							throw new Error('Event #'+i+': SMTPE offset subframes amount'
								+' must be part of 0-99.');
						}
						destination[index++]=events[i].subframes;
						return event;
						break;
					case MIDIEvents.EVENT_META_KEY_SIGNATURE:
						if('number'!= typeof events[i].key
							|| events[i].key<-7 || events[i].scale>7) {
							throw new Error('Event #'+i+':The key signature key must be'
								+' between -7 and 7');
						}
						if('number'!= typeof events[i].scale
							|| events[i].scale<0 || events[i].scale>1) {
							throw new Error('Event #'+i+':The key signature scale must be'
								+' 0 or 1');
						}
						destination[index++]=events[i].key;
						destination[index++]=events[i].scale;
						break;
					// Not implemented
					case MIDIEvents.EVENT_META_TIME_SIGNATURE:
					case MIDIEvents.EVENT_META_SEQUENCER_SPECIFIC:
					default:
						for(var k=0, l=events[i].length; k<l; k++) {
							destination[index++]=events[i].data[k];
						}
						break;
				}
			// Adding bytes corresponding to the sysex event datas
			} else {
				for(var k=0, l=events[i].length; k<l; k++) {
					destination[index++]=events[i].data[k];
				}
			}
		}
	}
};

// Return the buffer length needed to encode the given events
MIDIEvents.getRequiredBufferLength=function(events) {
	var bufferLength=0, event;
	// Calculating the track size by adding events lengths
	for(var i=0, j=events.length; i<j; i++) {
		// Computing necessary bytes to encode the delta value
		bufferLength+=
				(events[i].delta>>>21?4:
				(events[i].delta>>>14?3:
				(events[i].delta>>>7?2:1)));
		// MIDI Events have various fixed lengths
		if(events[i].type===MIDIEvents.EVENT_MIDI) {
			// Adding a byte for subtype + channel
			bufferLength++;
			// Adding a byte for the first params
			bufferLength++;
			// Adding a byte for the optionnal second param
			if(-1!==MIDIEvents.MIDI_2PARAMS_EVENTS.indexOf(events[i].subtype)) {
				bufferLength++;
			}
		// META / SYSEX events lengths are self defined
		} else {
			// Adding a byte for the event type
			bufferLength++;
			// Adding a byte for META events subtype
			if(events[i].type===MIDIEvents.EVENT_META) {
				bufferLength++;
			}
			// Adding necessary bytes to encode the length
			bufferLength+=
				(events[i].length>>>21?4:
				(events[i].length>>>14?3:
				(events[i].length>>>7?2:1)));
			// Adding bytes corresponding to the event length
			bufferLength+=events[i].length;
		}
	}
	return bufferLength;
};

// MIDIFileHeader : Read and edit a MIDI header chunk in a given ArrayBuffer
function MIDIFileHeader(buffer, strictMode) {
	// No buffer creating him
	if(!buffer) {
		var a=new Uint8Array(MIDIFileHeader.HEADER_LENGTH);
		// Adding the header id (MThd)
		a[0]=0x4D; a[1]=0x54; a[2]=0x68; a[3]=0x64;
		// Adding the header chunk size
		a[4]=0x00; a[5]=0x00; a[6]=0x00; a[7]=0x06;
		// Adding the file format (1 here cause it's the most commonly used)
		a[8]=0x00; a[9]=0x01;
		// Adding the track count (1 cause it's a new file)
		a[10]=0x00; a[11]=0x01;
		// Adding the time division (192 ticks per beat)
		a[12]=0x00; a[13]=0xC0;
		// saving the buffer
		this.datas=new DataView(a.buffer,0,MIDIFileHeader.HEADER_LENGTH);
	// Parsing the given buffer
	} else {
		if(!(buffer instanceof ArrayBuffer))
				throw Error('Invalid buffer received.');
		this.datas=new DataView(buffer,0,MIDIFileHeader.HEADER_LENGTH);
		// Reading MIDI header chunk
		if(!('M'===String.fromCharCode(this.datas.getUint8(0))
			&&'T'===String.fromCharCode(this.datas.getUint8(1))
			&&'h'===String.fromCharCode(this.datas.getUint8(2))
			&&'d'===String.fromCharCode(this.datas.getUint8(3)))) {
			throw new Error('Invalid MIDIFileHeader : MThd prefix not found');
		}
		// Reading chunk length
		if(6!==this.datas.getUint32(4)) {
			throw new Error('Invalid MIDIFileHeader : Chunk length must be 6');
		}
	}
}

// Static constants
MIDIFileHeader.HEADER_LENGTH=14;
MIDIFileHeader.FRAMES_PER_SECONDS=1;
MIDIFileHeader.TICKS_PER_BEAT=2;

// MIDI file format
MIDIFileHeader.prototype.getFormat=function() {
	var format=this.datas.getUint16(8);
	if(0!==format&&1!==format&&2!==format) {
		throw new Error('Invalid MIDI file : MIDI format ('+format+'),'
			+' format can be 0, 1 or 2 only.');
	}
	return format;
};

MIDIFileHeader.prototype.setFormat=function(format) {
	var format=this.datas.getUint16(8);
	if(0!==format&&1!==format&&2!==format) {
		throw new Error('Invalid MIDI format given ('+format+'),'
			+' format can be 0, 1 or 2 only.');
	}
	return format;
};

// Number of tracks
MIDIFileHeader.prototype.getTracksCount=function() {
	return this.datas.getUint16(10);
};

MIDIFileHeader.prototype.setTracksCount=function(n) {
	return this.datas.setUint16(10,n);
};

// Tick compute
MIDIFileHeader.prototype.getTickResolution=function(tempo) {
	// Frames per seconds
	if(this.datas.getUint16(12)&0x8000) {
		return 1000000/(this.getSMPTEFrames() * this.getTicksPerFrame());
	// Ticks per beat
	} else {
		// Default MIDI tempo is 120bpm, 500ms per beat
		tempo=tempo||500000;
		return tempo/this.getTicksPerBeat();
	}
};

// Time division type
MIDIFileHeader.prototype.getTimeDivision=function() {
	if(this.datas.getUint16(12)&0x8000) {
		return MIDIFileHeader.FRAMES_PER_SECONDS;
	}
	return MIDIFileHeader.TICKS_PER_BEAT;
};

// Ticks per beat
MIDIFileHeader.prototype.getTicksPerBeat=function() {
	var divisionWord=this.datas.getUint16(12);
	if(divisionWord&0x8000) {
		throw new Error('Time division is not expressed as ticks per beat.');
	}
	return divisionWord;
};

MIDIFileHeader.prototype.setTicksPerBeat=function(ticksPerBeat) {
	this.datas.setUint16(12,ticksPerBeat&0x7FFF);
};

// Frames per seconds
MIDIFileHeader.prototype.getSMPTEFrames=function() {
	var divisionWord=this.datas.getUint16(12), smpteFrames;
	if(!(divisionWord&0x8000)) {
		throw new Error('Time division is not expressed as frames per seconds.');
	}
	smpteFrames=divisionWord&0x7F00;
	if(smpteFrames!=24&&smpteFrames!=25&&smpteFrames!=29&&smpteFrames!=30) {
		throw new Error('Invalid SMPTE frames value ('+smpteFrames+').');
	}
	return (29===smpteFrames?29.97:smpteFrames);
};

MIDIFileHeader.prototype.getTicksPerFrame=function() {
	var divisionWord=this.datas.getUint16(12);
	if(!(divisionWord&0x8000)) {
		throw new Error('Time division is not expressed as frames per seconds.');
	}
	return divisionWord&0x00FF;
};

MIDIFileHeader.prototype.setSMTPEDivision=function(smpteFrames,ticksPerFrame) {
	if(smpteFrames!=24&&smpteFrames!=25&&smpteFrames!=29.97
			&&smpteFrames!=29&&smpteFrames!=30) {
		throw new Error('Invalid SMPTE frames value given ('+smpteFrames+').');
	}
	if(smpteFrames==29.97)
		smpteFrames=29;
	if(ticksPerFrame<0||ticksPerFrame>0xFF) {
		throw new Error('Invalid ticks per frame value given ('+smpteFrames+').');
	}
	this.datas.setUint8(12,0x80|smpteFrames);
	this.datas.setUint8(13,ticksPerFrame);
};

// MIDIFileTrack : Read and edit a MIDI track chunk in a given ArrayBuffer
function MIDIFileTrack(buffer, start, strictMode) {
	// no buffer, creating him
	if(!buffer) {
		var a=new Uint8Array(12);
		// Adding the empty track header (MTrk)
		a[0]=0x4D; a[1]=0x54; a[2]=0x72; a[3]=0x6B;
		// Adding the empty track size (4)
		a[4]=0x00; a[5]=0x00; a[6]=0x00; a[7]=0x04;
		// Adding the track end event
		a[8]=0x00; a[9]=0xFF; a[10]=0x2F; a[11]=0x00;
		// Saving the buffer
		this.datas=new DataView(a.buffer, 0,
			MIDIFileTrack.HDR_LENGTH+4);
	// parsing the given buffer
	} else {
		if(!(buffer instanceof ArrayBuffer))
				throw Error('Invalid buffer received.');
		// Buffer length must size at least like an  empty track (8+3bytes)
		if(buffer.byteLength-start<12) {
			throw Error('Invalid MIDIFileTrack (0x'+start.toString(16)+') :'
				+' Buffer length must size at least 12bytes');
		}
		// Creating a temporary view to read the track header
		this.datas=new DataView(buffer, start, MIDIFileTrack.HDR_LENGTH);
		// Reading MIDI track header chunk
		if(!('M'===String.fromCharCode(this.datas.getUint8(0))
			&&'T'===String.fromCharCode(this.datas.getUint8(1))
			&&'r'===String.fromCharCode(this.datas.getUint8(2))
			&&'k'===String.fromCharCode(this.datas.getUint8(3)))) {
			throw Error('Invalid MIDIFileTrack (0x'+start.toString(16)+') :'
				+' MTrk prefix not found');
			}
		// Reading the track length
		var trackLength=this.getTrackLength();
		if(buffer.byteLength-start<trackLength) {
			throw Error('Invalid MIDIFileTrack (0x'+start.toString(16)+') :'
				+' The track size exceed the buffer length.');
		}
		// Creating the final DataView
		this.datas=new DataView(buffer, start,
			MIDIFileTrack.HDR_LENGTH+trackLength);
		// Trying to find the end of track event
		if(!(0xFF===this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + trackLength-3)
			&&0x2F===this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + trackLength-2)
			&&0x00===this.datas.getUint8(MIDIFileTrack.HDR_LENGTH + trackLength-1)
			)) {
				throw Error('Invalid MIDIFileTrack (0x'+start.toString(16)+') :'
					+' No track end event found at the expected index'
					+' ('+(MIDIFileTrack.HDR_LENGTH+trackLength-1).toString(16)+').');
		}
	}
}

// Static constants
MIDIFileTrack.HDR_LENGTH=8;

// Track length
MIDIFileTrack.prototype.getTrackLength=function() {
	return this.datas.getUint32(4);
};

MIDIFileTrack.prototype.setTrackLength=function(trackLength) {
	return this.datas.setUint32(trackLength);
};

// Read track contents
MIDIFileTrack.prototype.getTrackContent=function() {
	return new DataView(this.datas.buffer,
		this.datas.byteOffset+MIDIFileTrack.HDR_LENGTH,
		this.datas.byteLength-MIDIFileTrack.HDR_LENGTH);
};

// Set track content
MIDIFileTrack.prototype.setTrackContent=function(dataView) {
	// Calculating the track length
	var trackLength=dataView.byteLength-dataView.byteOffset;
	// Track length must size at least like an  empty track (4bytes)
	if(trackLength<4) {
		throw Error('Invalid track length, must size at least 4bytes');
	}
	this.datas=new DataView(
		new Uint8Array(MIDIFileTrack.HDR_LENGTH+trackLength).buffer);
	// Adding the track header (MTrk)
	this.datas.setUint8(0,0x4D); // M
	this.datas.setUint8(1,0x54); // T
	this.datas.setUint8(2,0x72); // r
	this.datas.setUint8(3,0x6B); // k
	// Adding the track size
	this.datas.setUint32(4,trackLength);
	// Copying the content
	var origin=new Uint8Array(dataView.buffer, dataView.byteOffset,
		dataView.byteLength),
		destination=new Uint8Array(this.datas.buffer,
			MIDIFileTrack.HDR_LENGTH,
			trackLength);
	for(var i=0, j=origin.length; i<j; i++) {
		destination[i]=origin[i];
	}
};

// MIDIFile : Read (and soon edit) a MIDI file in a given ArrayBuffer

// Constructor
function MIDIFile(buffer, strictMode) {
	// If not buffer given, creating a new MIDI file
	if(!buffer) {
		// Creating the content
		this.header=new MIDIFileHeader();
		this.tracks=[new MIDIFileTrack()];
	// if a buffer is provided, parsing him
	} else {
		if(!(buffer instanceof ArrayBuffer)) {
			throw new Error('Invalid buffer received.');
		}
		// Minimum MIDI file size is a headerChunk size (14bytes)
		// and an empty track (8+3bytes)
		if(buffer.byteLength<25) {
			throw new Error('A buffer of a valid MIDI file must have, at least, a'
				+' size of 25bytes.');
		}
		// Reading header
		this.header=new MIDIFileHeader(buffer, strictMode);
		this.tracks=[];
		var track;
		var curIndex=14;
		// Reading tracks
		for(var i=0, j=this.header.getTracksCount(); i<j; i++) {
			// Testing the buffer length
			if(strictMode&&curIndex>=buffer.byteLength-1) {
				throw new Error('Couldn\'t find datas corresponding to the track #'+i+'.');
			}
			// Creating the track object
			var track=new MIDIFileTrack(buffer, curIndex, strictMode);
			this.tracks.push(track);
			// Updating index to the track end
			curIndex+=track.getTrackLength()+8;
		}
		// Testing integrity : curIndex should be at the end of the buffer
		if(strictMode&&curIndex!=buffer.byteLength) {
			throw new Error('It seems that the buffer contains too much datas.');
		}
	}
}

// Events reading helpers
MIDIFile.prototype.getEvents = function(type, subtype) {
	var events, event, playTime=0, filteredEvents=[],
		format=this.header.getFormat(),
		tickResolution=this.header.getTickResolution();
	// Reading events
	// if the read is sequential
	if(1!==format||1===this.tracks.length) {
		for(var i=0, j=this.tracks.length; i<j; i++) {
			// reset playtime if format is 2
			playTime=(2==format&&playTime?playTime:0);
			events=new MIDIEvents.createParser(this.tracks[i].getTrackContent(),0,false);
			// loooping throught events
			 while(event=events.next()) {
			 	playTime+=(event.delta?(event.delta*tickResolution)/1000:0);
				if(event.type===MIDIEvents.EVENT_META) {
					// tempo change events
					if(event.subtype===MIDIEvents.EVENT_META_SET_TEMPO) {
						tickResolution=this.header.getTickResolution(event.tempo);
					}
				}
				// push the asked events
				if(((!type)||event.type===type)
					&&((!subtype)||(event.subtype&&event.subtype===type))) {
					event.playTime=playTime;
					filteredEvents.push(event);
				}
			}
		}
	// the read is concurrent
	} else {
		var trackParsers=[], smallestDelta=-1, i, j;
		// Creating parsers
		for(i=0, j=this.tracks.length; i<j; i++) {
			trackParsers[i]={};
			trackParsers[i].parser=new MIDIEvents.createParser(
					this.tracks[i].getTrackContent(),0,false);
			trackParsers[i].curEvent=trackParsers[i].parser.next();
		}
		// Filling events
		do {
			smallestDelta=-1;
			// finding the smallest event
			for(i=0, j=trackParsers.length; i<j; i++) {
				if(trackParsers[i].curEvent) {
					if(-1===smallestDelta||trackParsers[i].curEvent.delta
						<trackParsers[smallestDelta].curEvent.delta) {
						smallestDelta=i;
					}
				}
			}
			if(-1!==smallestDelta) {
				// removing the delta of previous events
				for(i=0, j=trackParsers.length; i<j; i++) {
					if(i!==smallestDelta&&trackParsers[i].curEvent) {
						trackParsers[i].curEvent.delta-=trackParsers[smallestDelta].curEvent.delta;
					}
				}
				// filling values
				event=trackParsers[smallestDelta].curEvent;
			 	playTime+=(event.delta?(event.delta*tickResolution)/1000:0);
				if(event.type===MIDIEvents.EVENT_META) {
					// tempo change events
					if(event.subtype===MIDIEvents.EVENT_META_SET_TEMPO) {
						tickResolution=this.header.getTickResolution(event.tempo);
					}
				}
				// push midi events
				if(((!type)||event.type===type)
					&&((!subtype)||(event.subtype&&event.subtype===type))) {
					event.playTime=playTime;
					event.track=smallestDelta;
					filteredEvents.push(event);
				}
				// getting next event
				trackParsers[smallestDelta].curEvent=trackParsers[smallestDelta].parser.next();
			}
		} while(-1!==smallestDelta);
	}
	return filteredEvents;
};

MIDIFile.prototype.getMidiEvents = function() {
	return this.getEvents(MIDIEvents.EVENT_MIDI);
};

MIDIFile.prototype.getLyrics = function() {
	var events=this.getEvents(MIDIEvents.EVENT_META),
		texts=[], lyrics=[], event, karaoke=-1, format=this.header.getFormat();
	for(var i=0, j=events.length; i<j; i++) {
		event=events[i];
		// Lyrics
		if(event.subtype===MIDIEvents.EVENT_META_LYRICS) {
			lyrics.push(event);
		// Texts
		} else if(event.subtype===MIDIEvents.EVENT_META_TEXT) {
			// Ignore special texts
			if('@'===String.fromCharCode(event.data[0])) {
				if('T'===String.fromCharCode(event.data[1])) {
					//console.log('Title : '+event.text.substring(2));
				} else if('I'===String.fromCharCode(event.data[1])) {
					//console.log('Info : '+event.text.substring(2));
				} else if('L'===String.fromCharCode(event.data[1])) {
					//console.log('Lang : '+event.text.substring(2));
				}
			// karaoke text follows, remove all previous text
			} else if(0===event.data.map(function(c) {
					return String.fromCharCode(c);
				}).join('').indexOf('words')) {
				texts.length=0;
				//console.log('Word marker found');
			// Karaoke texts
			} else {
				// If playtime is greater than 0
				if(0!==event.playTime) {
					texts.push(event);
				}
			}
		}
	}
	// Choosing the right lyrics
	if(lyrics.length>2) {
		texts=lyrics;
	} else if(!texts.length) {
		texts=[];
	}
	// Convert texts and detect encoding
	try {
		texts.forEach(function(event) {
			event.text=UTF8.getStringFromBytes(event.data,0,event.length,true);
		});
	} catch (e) {
		texts.forEach(function(event) {
			event.text=event.data.map(function(c) {
				return String.fromCharCode(c);
			}).join('');
		});
	}
	return texts;
};

// Basic events reading
MIDIFile.prototype.getTrackEvents = function(index) {
	var event, events=[], parser;
	if(index>this.tracks.length||index<0) {
		throw Error('Invalid track index ('+index+')');
	}
	parser=new MIDIEvents.createParser(
		this.tracks[index].getTrackContent(),0,false);
	event=parser.next();
	do {
		events.push(event);
		event=parser.next();
	} while(event);
	return events;
};

// Basic events writting
MIDIFile.prototype.setTrackEvents = function(index, events) {
	var bufferLength, destination;
	if(index>this.tracks.length||index<0) {
		throw Error('Invalid track index ('+index+')');
	}
	if((!events)||(!events.length)) {
		throw Error('A track must contain at least one event, none given.');
	}
	bufferLength=MIDIEvents.getRequiredBufferLength(events);
	destination = new Uint8Array(bufferLength);
	MIDIEvents.writeToTrack(events, destination);
	this.tracks[index].setTrackContent(destination);
};

// Remove a track
MIDIFile.prototype.deleteTrack = function(index) {
	if(index>this.tracks.length||index<0) {
		throw Error('Invalid track index ('+index+')');
	}
	this.tracks.splice(index,1);
	this.header.setTracksCount(this.tracks.length);
};

// Add a track
MIDIFile.prototype.addTrack = function(index) {
	if(index>this.tracks.length||index<0) {
		throw Error('Invalid track index ('+index+')');
	}
	var track = new MIDIFileTrack();
	if(index==this.tracks.length) {
		this.tracks.push(track);
	} else {
		this.tracks.splice(index,0,track);
	}
	this.header.setTracksCount(this.tracks.length);
};

// Retrieve the content in a buffer
MIDIFile.prototype.getContent = function() {
	var bufferLength, destination, origin, lastOffset=0;
	// Calculating the buffer content
	// - initialize with the header length
	bufferLength=MIDIFileHeader.HEADER_LENGTH;
	// - add tracks length
	for(var i=0, j=this.tracks.length; i<j; i++) {
		bufferLength+=this.tracks[i].getTrackLength()+8;
	}
	// Creating the destination buffer
	destination=new Uint8Array(bufferLength);
	// Adding header
	origin=new Uint8Array(this.header.datas.buffer,
		this.header.datas.byteOffset,
		MIDIFileHeader.HEADER_LENGTH);
	for(var i=0, j=MIDIFileHeader.HEADER_LENGTH; i<j; i++) {
		destination[i]=origin[i];
	}
	// Adding tracks
	for(var k=0, l=this.tracks.length; k<l; k++) {
		origin=new Uint8Array(this.tracks[k].datas.buffer,
			this.tracks[k].datas.byteOffset,
			this.tracks[k].datas.byteLength);
		for(var m=0, n=this.tracks[k].datas.byteLength; m<n; m++) {
			destination[i++]=origin[m];
		}
	}
	return destination.buffer;
};

// MIDIPlayer : Play MIDI files

	// Constants
	var PLAY_BUFFER_DELAY=300,
		PAGE_HIDDEN_BUFFER_RATIO=20;

	// MIDIPlayer constructor
	function MIDIPlayer(options) {
		options=options||{};
		this.output=options.output||null; // midi output
		this.volume=options.volume||100; // volume in percents
		this.startTime=-1; // ms since page load
		this.pauseTime=-1; // ms elapsed before player paused
		this.events=[];
		this.notesOn=new Array(32); // notesOn[channel][note]
		for(var i=31; i>=0; i--) {
			this.notesOn[i]=[];
		}
		this.midiFile=null;
	}

	// Parsing all tracks and add their events in a single event queue
	MIDIPlayer.prototype.load = function(midiFile) {
		this.stop();
		this.position=0
		this.midiFile=midiFile;
		this.events=this.midiFile.getMidiEvents();
	};

	MIDIPlayer.prototype.play = function(endCallback) {
		this.endCallback=endCallback;
		if(0===this.position) {
			this.startTime=performance.now();
			this.timeout=setTimeout(this.processPlay.bind(this),0);
			return 1;
		}
		return 0;
	};

	MIDIPlayer.prototype.processPlay = function() {
		var elapsedTime=performance.now()-this.startTime, event, karaoke, param2,
			bufferDelay=PLAY_BUFFER_DELAY*(document.hidden||document.mozHidden
				||document.webkitHidden||document.msHidden||document.oHidden?
					PAGE_HIDDEN_BUFFER_RATIO:1);
		event=this.events[this.position];
		while(this.events[this.position]&&event.playTime-elapsedTime<bufferDelay) {
			param2=0;
			if(event.subtype==MIDIEvents.EVENT_MIDI_NOTE_ON) {
				param2=Math.floor(event.param1*((this.volume||1)/100));
				this.notesOn[event.channel].push(event.param1);
			} else if(event.subtype==MIDIEvents.EVENT_MIDI_NOTE_OFF) {
				var index=this.notesOn[event.channel].indexOf(event.param1)
				if(-1!==index) {
					this.notesOn[event.channel].splice(index,1);
				}
			}
			var L = [(event.subtype<<4)+event.channel, event.param1, (param2||event.param2||0x00)];
			var oneP = MIDIEvents.MIDI_1PARAM_EVENTS;
			for(var i=0;i<oneP.length;i++) {
				if(oneP[i]==event.subtype) {
					L.length = 2;
					break;
				}
			}

			this.output.send(L, Math.floor(event.playTime+this.startTime));
			this.lastPlayTime=event.playTime+this.startTime;
			this.position++
			event=this.events[this.position];
		}
		if(this.position<this.events.length-1) {
			this.timeout=setTimeout(this.processPlay.bind(this),PLAY_BUFFER_DELAY-250);
		} else {
			setTimeout(this.endCallback,PLAY_BUFFER_DELAY+100);
			this.position=0;
		}
	};

	MIDIPlayer.prototype.pause = function() {
		if(this.timeout) {
			clearTimeout(this.timeout);
			this.timeout=null;
			this.pauseTime=performance.now();
			for(var i=this.notesOn.length-1; i>=0; i--) {
				for(var j=this.notesOn[i].length-1; j>=0; j--) {
					this.output.send([(MIDIEvents.EVENT_MIDI_NOTE_OFF<<4)+i, this.notesOn[i][j],
						0x00],this.lastPlayTime+100);
				}
			}
			return true;
		}
		return false;
	};

	MIDIPlayer.prototype.resume = function(endCallback) {
		this.endCallback=endCallback;
		if(this.events&&this.events[this.position]&&!this.timeout) {
			this.startTime+=performance.now()-this.pauseTime
			this.timeout=setTimeout(this.processPlay.bind(this),0);
			return this.events[this.position].playTime;
		}
		return 0;
	};

	MIDIPlayer.prototype.stop = function() {
		if(this.pause()) {
			this.position=0;
			for(var i=31; i>=0; i--) {
				this.notesOn[i]=[];
			}
			return true;
		}
		return false;
	};

// Note: For maximum-speed code, see "Optimizing Code" on the Emscripten wiki, https://github.com/kripken/emscripten/wiki/Optimizing-Code
// Note: Some Emscripten settings may limit the speed of the generated code.
// The Module object: Our interface to the outside world. We import
// and export values on it, and do the work to get that through
// closure compiler if necessary. There are various ways Module can be used:
// 1. Not defined. We create it here
// 2. A function parameter, function(Module) { ..generated code.. }
// 3. pre-run appended it, var Module = {}; ..generated code..
// 4. External script tag defines var Module.
// We need to do an eval in order to handle the closure compiler
// case, where this code here is minified but Module was defined
// elsewhere (e.g. case 4 above). We also need to check if Module
// already exists (e.g. case 3 above).
// Note that if you want to run closure, and also to use Module
// after the generated code, you will need to define   var Module = {};
// before the code. Then that object will be used in the code, and you
// can continue to use Module afterwards as well.
var Module;
if (!Module) Module = eval('(function() { try { return Module || {} } catch(e) { return {} } })()');
// Sometimes an existing Module object exists with properties
// meant to overwrite the default module functionality. Here
// we collect those properties and reapply _after_ we configure
// the current environment's defaults to avoid having to be so
// defensive during initialization.
var moduleOverrides = {};
for (var key in Module) {
  if (Module.hasOwnProperty(key)) {
    moduleOverrides[key] = Module[key];
  }
}
// The environment setup code below is customized to use Module.
// *** Environment setup code ***
var ENVIRONMENT_IS_NODE = typeof process === 'object' && typeof require === 'function';
var ENVIRONMENT_IS_WEB = typeof window === 'object';
var ENVIRONMENT_IS_WORKER = typeof importScripts === 'function';
var ENVIRONMENT_IS_SHELL = !ENVIRONMENT_IS_WEB && !ENVIRONMENT_IS_NODE && !ENVIRONMENT_IS_WORKER;
if (ENVIRONMENT_IS_NODE) {
  // Expose functionality in the same simple way that the shells work
  // Note that we pollute the global namespace here, otherwise we break in node
  Module['print'] = function(x) {
    process['stdout'].write(x + '\n');
  };
  Module['printErr'] = function(x) {
    process['stderr'].write(x + '\n');
  };
  var nodeFS = require('fs');
  var nodePath = require('path');
  Module['read'] = function(filename, binary) {
    filename = nodePath['normalize'](filename);
    var ret = nodeFS['readFileSync'](filename);
    // The path is absolute if the normalized version is the same as the resolved.
    if (!ret && filename != nodePath['resolve'](filename)) {
      filename = path.join(__dirname, '..', 'src', filename);
      ret = nodeFS['readFileSync'](filename);
    }
    if (ret && !binary) ret = ret.toString();
    return ret;
  };
  Module['readBinary'] = function(filename) { return Module['read'](filename, true) };
  Module['load'] = function(f) {
    globalEval(read(f));
  };
  Module['arguments'] = process['argv'].slice(2);
  module.exports = Module;
}
else if (ENVIRONMENT_IS_SHELL) {
  Module['print'] = print;
  if (typeof printErr != 'undefined') Module['printErr'] = printErr; // not present in v8 or older sm
  if (typeof read != 'undefined') {
    Module['read'] = read;
  } else {
    Module['read'] = function() { throw 'no read() available (jsc?)' };
  }
  Module['readBinary'] = function(f) {
    return read(f, 'binary');
  };
  if (typeof scriptArgs != 'undefined') {
    Module['arguments'] = scriptArgs;
  } else if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }
  this['Module'] = Module;
}
else if (ENVIRONMENT_IS_WEB || ENVIRONMENT_IS_WORKER) {
  Module['read'] = function(url) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', url, false);
    xhr.send(null);
    return xhr.responseText;
  };
  if (typeof arguments != 'undefined') {
    Module['arguments'] = arguments;
  }
  if (ENVIRONMENT_IS_WEB) {
    Module['print'] = function(x) {
      console.log(x);
    };
    Module['printErr'] = function(x) {
      console.log(x);
    };
    this['Module'] = Module;
  } else if (ENVIRONMENT_IS_WORKER) {
    // We can do very little here...
    var TRY_USE_DUMP = false;
    Module['print'] = (TRY_USE_DUMP && (typeof(dump) !== "undefined") ? (function(x) {
      dump(x);
    }) : (function(x) {
      // self.postMessage(x); // enable this if you want stdout to be sent as messages
    }));
    Module['load'] = importScripts;
  }
}
else {
  // Unreachable because SHELL is dependant on the others
  throw 'Unknown runtime environment. Where are we?';
}
function globalEval(x) {
  eval.call(null, x);
}
if (!Module['load'] == 'undefined' && Module['read']) {
  Module['load'] = function(f) {
    globalEval(Module['read'](f));
  };
}
if (!Module['print']) {
  Module['print'] = function(){};
}
if (!Module['printErr']) {
  Module['printErr'] = Module['print'];
}
if (!Module['arguments']) {
  Module['arguments'] = [];
}
// *** Environment setup code ***
// Closure helpers
Module.print = Module['print'];
Module.printErr = Module['printErr'];
// Callbacks
Module['preRun'] = [];
Module['postRun'] = [];
// Merge back in the overrides
for (var key in moduleOverrides) {
  if (moduleOverrides.hasOwnProperty(key)) {
    Module[key] = moduleOverrides[key];
  }
}
// === Auto-generated preamble library stuff ===
//========================================
// Runtime code shared with compiler
//========================================
var Runtime = {
  stackSave: function () {
    return STACKTOP;
  },
  stackRestore: function (stackTop) {
    STACKTOP = stackTop;
  },
  forceAlign: function (target, quantum) {
    quantum = quantum || 4;
    if (quantum == 1) return target;
    if (isNumber(target) && isNumber(quantum)) {
      return Math.ceil(target/quantum)*quantum;
    } else if (isNumber(quantum) && isPowerOfTwo(quantum)) {
      var logg = log2(quantum);
      return '((((' +target + ')+' + (quantum-1) + ')>>' + logg + ')<<' + logg + ')';
    }
    return 'Math.ceil((' + target + ')/' + quantum + ')*' + quantum;
  },
  isNumberType: function (type) {
    return type in Runtime.INT_TYPES || type in Runtime.FLOAT_TYPES;
  },
  isPointerType: function isPointerType(type) {
  return type[type.length-1] == '*';
},
  isStructType: function isStructType(type) {
  if (isPointerType(type)) return false;
  if (isArrayType(type)) return true;
  if (/<?{ ?[^}]* ?}>?/.test(type)) return true; // { i32, i8 } etc. - anonymous struct types
  // See comment in isStructPointerType()
  return type[0] == '%';
},
  INT_TYPES: {"i1":0,"i8":0,"i16":0,"i32":0,"i64":0},
  FLOAT_TYPES: {"float":0,"double":0},
  or64: function (x, y) {
    var l = (x | 0) | (y | 0);
    var h = (Math.round(x / 4294967296) | Math.round(y / 4294967296)) * 4294967296;
    return l + h;
  },
  and64: function (x, y) {
    var l = (x | 0) & (y | 0);
    var h = (Math.round(x / 4294967296) & Math.round(y / 4294967296)) * 4294967296;
    return l + h;
  },
  xor64: function (x, y) {
    var l = (x | 0) ^ (y | 0);
    var h = (Math.round(x / 4294967296) ^ Math.round(y / 4294967296)) * 4294967296;
    return l + h;
  },
  getNativeTypeSize: function (type, quantumSize) {
    if (Runtime.QUANTUM_SIZE == 1) return 1;
    var size = {
      '%i1': 1,
      '%i8': 1,
      '%i16': 2,
      '%i32': 4,
      '%i64': 8,
      "%float": 4,
      "%double": 8
    }['%'+type]; // add '%' since float and double confuse Closure compiler as keys, and also spidermonkey as a compiler will remove 's from '_i8' etc
    if (!size) {
      if (type.charAt(type.length-1) == '*') {
        size = Runtime.QUANTUM_SIZE; // A pointer
      } else if (type[0] == 'i') {
        var bits = parseInt(type.substr(1));
        assert(bits % 8 == 0);
        size = bits/8;
      }
    }
    return size;
  },
  getNativeFieldSize: function (type) {
    return Math.max(Runtime.getNativeTypeSize(type), Runtime.QUANTUM_SIZE);
  },
  dedup: function dedup(items, ident) {
  var seen = {};
  if (ident) {
    return items.filter(function(item) {
      if (seen[item[ident]]) return false;
      seen[item[ident]] = true;
      return true;
    });
  } else {
    return items.filter(function(item) {
      if (seen[item]) return false;
      seen[item] = true;
      return true;
    });
  }
},
  set: function set() {
  var args = typeof arguments[0] === 'object' ? arguments[0] : arguments;
  var ret = {};
  for (var i = 0; i < args.length; i++) {
    ret[args[i]] = 0;
  }
  return ret;
},
  STACK_ALIGN: 8,
  getAlignSize: function (type, size, vararg) {
    // we align i64s and doubles on 64-bit boundaries, unlike x86
    if (type == 'i64' || type == 'double' || vararg) return 8;
    if (!type) return Math.min(size, 8); // align structures internally to 64 bits
    return Math.min(size || (type ? Runtime.getNativeFieldSize(type) : 0), Runtime.QUANTUM_SIZE);
  },
  calculateStructAlignment: function calculateStructAlignment(type) {
    type.flatSize = 0;
    type.alignSize = 0;
    var diffs = [];
    var prev = -1;
    var index = 0;
    type.flatIndexes = type.fields.map(function(field) {
      index++;
      var size, alignSize;
      if (Runtime.isNumberType(field) || Runtime.isPointerType(field)) {
        size = Runtime.getNativeTypeSize(field); // pack char; char; in structs, also char[X]s.
        alignSize = Runtime.getAlignSize(field, size);
      } else if (Runtime.isStructType(field)) {
        if (field[1] === '0') {
          // this is [0 x something]. When inside another structure like here, it must be at the end,
          // and it adds no size
          // XXX this happens in java-nbody for example... assert(index === type.fields.length, 'zero-length in the middle!');
          size = 0;
          alignSize = type.alignSize || QUANTUM_SIZE;
        } else {
          size = Types.types[field].flatSize;
          alignSize = Runtime.getAlignSize(null, Types.types[field].alignSize);
        }
      } else if (field[0] == 'b') {
        // bN, large number field, like a [N x i8]
        size = field.substr(1)|0;
        alignSize = 1;
      } else {
        throw 'Unclear type in struct: ' + field + ', in ' + type.name_ + ' :: ' + dump(Types.types[type.name_]);
      }
      if (type.packed) alignSize = 1;
      type.alignSize = Math.max(type.alignSize, alignSize);
      var curr = Runtime.alignMemory(type.flatSize, alignSize); // if necessary, place this on aligned memory
      type.flatSize = curr + size;
      if (prev >= 0) {
        diffs.push(curr-prev);
      }
      prev = curr;
      return curr;
    });
    type.flatSize = Runtime.alignMemory(type.flatSize, type.alignSize);
    if (diffs.length == 0) {
      type.flatFactor = type.flatSize;
    } else if (Runtime.dedup(diffs).length == 1) {
      type.flatFactor = diffs[0];
    }
    type.needsFlattening = (type.flatFactor != 1);
    return type.flatIndexes;
  },
  generateStructInfo: function (struct, typeName, offset) {
    var type, alignment;
    if (typeName) {
      offset = offset || 0;
      type = (typeof Types === 'undefined' ? Runtime.typeInfo : Types.types)[typeName];
      if (!type) return null;
      if (type.fields.length != struct.length) {
        printErr('Number of named fields must match the type for ' + typeName + ': possibly duplicate struct names. Cannot return structInfo');
        return null;
      }
      alignment = type.flatIndexes;
    } else {
      var type = { fields: struct.map(function(item) { return item[0] }) };
      alignment = Runtime.calculateStructAlignment(type);
    }
    var ret = {
      __size__: type.flatSize
    };
    if (typeName) {
      struct.forEach(function(item, i) {
        if (typeof item === 'string') {
          ret[item] = alignment[i] + offset;
        } else {
          // embedded struct
          var key;
          for (var k in item) key = k;
          ret[key] = Runtime.generateStructInfo(item[key], type.fields[i], alignment[i]);
        }
      });
    } else {
      struct.forEach(function(item, i) {
        ret[item[1]] = alignment[i];
      });
    }
    return ret;
  },
  dynCall: function (sig, ptr, args) {
    if (args && args.length) {
      if (!args.splice) args = Array.prototype.slice.call(args);
      args.splice(0, 0, ptr);
      return Module['dynCall_' + sig].apply(null, args);
    } else {
      return Module['dynCall_' + sig].call(null, ptr);
    }
  },
  functionPointers: [],
  addFunction: function (func) {
    for (var i = 0; i < Runtime.functionPointers.length; i++) {
      if (!Runtime.functionPointers[i]) {
        Runtime.functionPointers[i] = func;
        return 2 + 2*i;
      }
    }
    throw 'Finished up all reserved function pointers. Use a higher value for RESERVED_FUNCTION_POINTERS.';
  },
  removeFunction: function (index) {
    Runtime.functionPointers[(index-2)/2] = null;
  },
  warnOnce: function (text) {
    if (!Runtime.warnOnce.shown) Runtime.warnOnce.shown = {};
    if (!Runtime.warnOnce.shown[text]) {
      Runtime.warnOnce.shown[text] = 1;
      Module.printErr(text);
    }
  },
  funcWrappers: {},
  getFuncWrapper: function (func, sig) {
    assert(sig);
    if (!Runtime.funcWrappers[func]) {
      Runtime.funcWrappers[func] = function() {
        return Runtime.dynCall(sig, func, arguments);
      };
    }
    return Runtime.funcWrappers[func];
  },
  UTF8Processor: function () {
    var buffer = [];
    var needed = 0;
    this.processCChar = function (code) {
      code = code & 0xff;
      if (needed) {
        buffer.push(code);
        needed--;
      }
      if (buffer.length == 0) {
        if (code < 128) return String.fromCharCode(code);
        buffer.push(code);
        if (code > 191 && code < 224) {
          needed = 1;
        } else {
          needed = 2;
        }
        return '';
      }
      if (needed > 0) return '';
      var c1 = buffer[0];
      var c2 = buffer[1];
      var c3 = buffer[2];
      var ret;
      if (c1 > 191 && c1 < 224) {
        ret = String.fromCharCode(((c1 & 31) << 6) | (c2 & 63));
      } else {
        ret = String.fromCharCode(((c1 & 15) << 12) | ((c2 & 63) << 6) | (c3 & 63));
      }
      buffer.length = 0;
      return ret;
    }
    this.processJSString = function(string) {
      string = unescape(encodeURIComponent(string));
      var ret = [];
      for (var i = 0; i < string.length; i++) {
        ret.push(string.charCodeAt(i));
      }
      return ret;
    }
  },
  stackAlloc: function (size) { var ret = STACKTOP;STACKTOP = (STACKTOP + size)|0;STACKTOP = ((((STACKTOP)+7)>>3)<<3); return ret; },
  staticAlloc: function (size) { var ret = STATICTOP;STATICTOP = (STATICTOP + size)|0;STATICTOP = ((((STATICTOP)+7)>>3)<<3); return ret; },
  dynamicAlloc: function (size) { var ret = DYNAMICTOP;DYNAMICTOP = (DYNAMICTOP + size)|0;DYNAMICTOP = ((((DYNAMICTOP)+7)>>3)<<3); if (DYNAMICTOP >= TOTAL_MEMORY) enlargeMemory();; return ret; },
  alignMemory: function (size,quantum) { var ret = size = Math.ceil((size)/(quantum ? quantum : 8))*(quantum ? quantum : 8); return ret; },
  makeBigInt: function (low,high,unsigned) { var ret = (unsigned ? ((+(((low)>>>(0))))+((+(((high)>>>(0))))*(+(4294967296)))) : ((+(((low)>>>(0))))+((+(((high)|(0))))*(+(4294967296))))); return ret; },
  GLOBAL_BASE: 8,
  QUANTUM_SIZE: 4,
  __dummy__: 0
}
//========================================
// Runtime essentials
//========================================
var __THREW__ = 0; // Used in checking for thrown exceptions.
var ABORT = false; // whether we are quitting the application. no code should run after this. set in exit() and abort()
var EXITSTATUS = 0;
var undef = 0;
// tempInt is used for 32-bit signed values or smaller. tempBigInt is used
// for 32-bit unsigned values or more than 32 bits. TODO: audit all uses of tempInt
var tempValue, tempInt, tempBigInt, tempInt2, tempBigInt2, tempPair, tempBigIntI, tempBigIntR, tempBigIntS, tempBigIntP, tempBigIntD;
var tempI64, tempI64b;
var tempRet0, tempRet1, tempRet2, tempRet3, tempRet4, tempRet5, tempRet6, tempRet7, tempRet8, tempRet9;
function assert(condition, text) {
  if (!condition) {
    abort('Assertion failed: ' + text);
  }
}
var globalScope = this;
// C calling interface. A convenient way to call C functions (in C files, or
// defined with extern "C").
//
// Note: LLVM optimizations can inline and remove functions, after which you will not be
//       able to call them. Closure can also do so. To avoid that, add your function to
//       the exports using something like
//
//         -s EXPORTED_FUNCTIONS='["_main", "_myfunc"]'
//
// @param ident      The name of the C function (note that C++ functions will be name-mangled - use extern "C")
// @param returnType The return type of the function, one of the JS types 'number', 'string' or 'array' (use 'number' for any C pointer, and
//                   'array' for JavaScript arrays and typed arrays; note that arrays are 8-bit).
// @param argTypes   An array of the types of arguments for the function (if there are no arguments, this can be ommitted). Types are as in returnType,
//                   except that 'array' is not possible (there is no way for us to know the length of the array)
// @param args       An array of the arguments to the function, as native JS values (as in returnType)
//                   Note that string arguments will be stored on the stack (the JS string will become a C string on the stack).
// @return           The return value, as a native JS value (as in returnType)
function ccall(ident, returnType, argTypes, args) {
  return ccallFunc(getCFunc(ident), returnType, argTypes, args);
}
Module["ccall"] = ccall;
// Returns the C function with a specified identifier (for C++, you need to do manual name mangling)
function getCFunc(ident) {
  try {
    var func = Module['_' + ident]; // closure exported function
    if (!func) func = eval('_' + ident); // explicit lookup
  } catch(e) {
  }
  assert(func, 'Cannot call unknown function ' + ident + ' (perhaps LLVM optimizations or closure removed it?)');
  return func;
}
// Internal function that does a C call using a function, not an identifier
function ccallFunc(func, returnType, argTypes, args) {
  var stack = 0;
  function toC(value, type) {
    if (type == 'string') {
      if (value === null || value === undefined || value === 0) return 0; // null string
      if (!stack) stack = Runtime.stackSave();
      var ret = Runtime.stackAlloc(value.length+1);
      writeStringToMemory(value, ret);
      return ret;
    } else if (type == 'array') {
      if (!stack) stack = Runtime.stackSave();
      var ret = Runtime.stackAlloc(value.length);
      writeArrayToMemory(value, ret);
      return ret;
    }
    return value;
  }
  function fromC(value, type) {
    if (type == 'string') {
      return Pointer_stringify(value);
    }
    assert(type != 'array');
    return value;
  }
  var i = 0;
  var cArgs = args ? args.map(function(arg) {
    return toC(arg, argTypes[i++]);
  }) : [];
  var ret = fromC(func.apply(null, cArgs), returnType);
  if (stack) Runtime.stackRestore(stack);
  return ret;
}
// Returns a native JS wrapper for a C function. This is similar to ccall, but
// returns a function you can call repeatedly in a normal way. For example:
//
//   var my_function = cwrap('my_c_function', 'number', ['number', 'number']);
//   alert(my_function(5, 22));
//   alert(my_function(99, 12));
//
function cwrap(ident, returnType, argTypes) {
  var func = getCFunc(ident);
  return function() {
    return ccallFunc(func, returnType, argTypes, Array.prototype.slice.call(arguments));
  }
}
Module["cwrap"] = cwrap;
// Sets a value in memory in a dynamic way at run-time. Uses the
// type data. This is the same as makeSetValue, except that
// makeSetValue is done at compile-time and generates the needed
// code then, whereas this function picks the right code at
// run-time.
// Note that setValue and getValue only do *aligned* writes and reads!
// Note that ccall uses JS types as for defining types, while setValue and
// getValue need LLVM types ('i8', 'i32') - this is a lower-level operation
function setValue(ptr, value, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': HEAP8[(ptr)]=value; break;
      case 'i8': HEAP8[(ptr)]=value; break;
      case 'i16': HEAP16[((ptr)>>1)]=value; break;
      case 'i32': HEAP32[((ptr)>>2)]=value; break;
      case 'i64': (tempI64 = [value>>>0,((Math.min((+(Math.floor((value)/(+(4294967296))))), (+(4294967295))))|0)>>>0],HEAP32[((ptr)>>2)]=tempI64[0],HEAP32[(((ptr)+(4))>>2)]=tempI64[1]); break;
      case 'float': HEAPF32[((ptr)>>2)]=value; break;
      case 'double': HEAPF64[((ptr)>>3)]=value; break;
      default: abort('invalid type for setValue: ' + type);
    }
}
Module['setValue'] = setValue;
// Parallel to setValue.
function getValue(ptr, type, noSafe) {
  type = type || 'i8';
  if (type.charAt(type.length-1) === '*') type = 'i32'; // pointers are 32-bit
    switch(type) {
      case 'i1': return HEAP8[(ptr)];
      case 'i8': return HEAP8[(ptr)];
      case 'i16': return HEAP16[((ptr)>>1)];
      case 'i32': return HEAP32[((ptr)>>2)];
      case 'i64': return HEAP32[((ptr)>>2)];
      case 'float': return HEAPF32[((ptr)>>2)];
      case 'double': return HEAPF64[((ptr)>>3)];
      default: abort('invalid type for setValue: ' + type);
    }
  return null;
}
Module['getValue'] = getValue;
var ALLOC_NORMAL = 0; // Tries to use _malloc()
var ALLOC_STACK = 1; // Lives for the duration of the current function call
var ALLOC_STATIC = 2; // Cannot be freed
var ALLOC_DYNAMIC = 3; // Cannot be freed except through sbrk
var ALLOC_NONE = 4; // Do not allocate
Module['ALLOC_NORMAL'] = ALLOC_NORMAL;
Module['ALLOC_STACK'] = ALLOC_STACK;
Module['ALLOC_STATIC'] = ALLOC_STATIC;
Module['ALLOC_DYNAMIC'] = ALLOC_DYNAMIC;
Module['ALLOC_NONE'] = ALLOC_NONE;
// allocate(): This is for internal use. You can use it yourself as well, but the interface
//             is a little tricky (see docs right below). The reason is that it is optimized
//             for multiple syntaxes to save space in generated code. So you should
//             normally not use allocate(), and instead allocate memory using _malloc(),
//             initialize it with setValue(), and so forth.
// @slab: An array of data, or a number. If a number, then the size of the block to allocate,
//        in *bytes* (note that this is sometimes confusing: the next parameter does not
//        affect this!)
// @types: Either an array of types, one for each byte (or 0 if no type at that position),
//         or a single type which is used for the entire block. This only matters if there
//         is initial data - if @slab is a number, then this does not matter at all and is
//         ignored.
// @allocator: How to allocate memory, see ALLOC_*
function allocate(slab, types, allocator, ptr) {
  var zeroinit, size;
  if (typeof slab === 'number') {
    zeroinit = true;
    size = slab;
  } else {
    zeroinit = false;
    size = slab.length;
  }
  var singleType = typeof types === 'string' ? types : null;
  var ret;
  if (allocator == ALLOC_NONE) {
    ret = ptr;
  } else {
    ret = [_malloc, Runtime.stackAlloc, Runtime.staticAlloc, Runtime.dynamicAlloc][allocator === undefined ? ALLOC_STATIC : allocator](Math.max(size, singleType ? 1 : types.length));
  }
  if (zeroinit) {
    var ptr = ret, stop;
    assert((ret & 3) == 0);
    stop = ret + (size & ~3);
    for (; ptr < stop; ptr += 4) {
      HEAP32[((ptr)>>2)]=0;
    }
    stop = ret + size;
    while (ptr < stop) {
      HEAP8[((ptr++)|0)]=0;
    }
    return ret;
  }
  if (singleType === 'i8') {
    if (slab.subarray || slab.slice) {
      HEAPU8.set(slab, ret);
    } else {
      HEAPU8.set(new Uint8Array(slab), ret);
    }
    return ret;
  }
  var i = 0, type, typeSize, previousType;
  while (i < size) {
    var curr = slab[i];
    if (typeof curr === 'function') {
      curr = Runtime.getFunctionIndex(curr);
    }
    type = singleType || types[i];
    if (type === 0) {
      i++;
      continue;
    }
    if (type == 'i64') type = 'i32'; // special case: we have one i32 here, and one i32 later
    setValue(ret+i, curr, type);
    // no need to look up size unless type changes, so cache it
    if (previousType !== type) {
      typeSize = Runtime.getNativeTypeSize(type);
      previousType = type;
    }
    i += typeSize;
  }
  return ret;
}
Module['allocate'] = allocate;
function Pointer_stringify(ptr, /* optional */ length) {
  // Find the length, and check for UTF while doing so
  var hasUtf = false;
  var t;
  var i = 0;
  while (1) {
    t = HEAPU8[(((ptr)+(i))|0)];
    if (t >= 128) hasUtf = true;
    else if (t == 0 && !length) break;
    i++;
    if (length && i == length) break;
  }
  if (!length) length = i;
  var ret = '';
  if (!hasUtf) {
    var MAX_CHUNK = 1024; // split up into chunks, because .apply on a huge string can overflow the stack
    var curr;
    while (length > 0) {
      curr = String.fromCharCode.apply(String, HEAPU8.subarray(ptr, ptr + Math.min(length, MAX_CHUNK)));
      ret = ret ? ret + curr : curr;
      ptr += MAX_CHUNK;
      length -= MAX_CHUNK;
    }
    return ret;
  }
  var utf8 = new Runtime.UTF8Processor();
  for (i = 0; i < length; i++) {
    t = HEAPU8[(((ptr)+(i))|0)];
    ret += utf8.processCChar(t);
  }
  return ret;
}
Module['Pointer_stringify'] = Pointer_stringify;
// Memory management
var PAGE_SIZE = 4096;
function alignMemoryPage(x) {
  return ((x+4095)>>12)<<12;
}
var HEAP;
var HEAP8, HEAPU8, HEAP16, HEAPU16, HEAP32, HEAPU32, HEAPF32, HEAPF64;
var STATIC_BASE = 0, STATICTOP = 0, staticSealed = false; // static area
var STACK_BASE = 0, STACKTOP = 0, STACK_MAX = 0; // stack area
var DYNAMIC_BASE = 0, DYNAMICTOP = 0; // dynamic area handled by sbrk
function enlargeMemory() {
  abort('Cannot enlarge memory arrays in asm.js. Either (1) compile with -s TOTAL_MEMORY=X with X higher than the current value, or (2) set Module.TOTAL_MEMORY before the program runs.');
}
var TOTAL_STACK = Module['TOTAL_STACK'] || 5242880;
var TOTAL_MEMORY = Module['TOTAL_MEMORY'] || 16777216;
var FAST_MEMORY = Module['FAST_MEMORY'] || 2097152;
// Initialize the runtime's memory
// check for full engine support (use string 'subarray' to avoid closure compiler confusion)
assert(!!Int32Array && !!Float64Array && !!(new Int32Array(1)['subarray']) && !!(new Int32Array(1)['set']),
       'Cannot fallback to non-typed array case: Code is too specialized');
var buffer = new ArrayBuffer(TOTAL_MEMORY);
HEAP8 = new Int8Array(buffer);
HEAP16 = new Int16Array(buffer);
HEAP32 = new Int32Array(buffer);
HEAPU8 = new Uint8Array(buffer);
HEAPU16 = new Uint16Array(buffer);
HEAPU32 = new Uint32Array(buffer);
HEAPF32 = new Float32Array(buffer);
HEAPF64 = new Float64Array(buffer);
// Endianness check (note: assumes compiler arch was little-endian)
HEAP32[0] = 255;
assert(HEAPU8[0] === 255 && HEAPU8[3] === 0, 'Typed arrays 2 must be run on a little-endian system');
Module['HEAP'] = HEAP;
Module['HEAP8'] = HEAP8;
Module['HEAP16'] = HEAP16;
Module['HEAP32'] = HEAP32;
Module['HEAPU8'] = HEAPU8;
Module['HEAPU16'] = HEAPU16;
Module['HEAPU32'] = HEAPU32;
Module['HEAPF32'] = HEAPF32;
Module['HEAPF64'] = HEAPF64;
function callRuntimeCallbacks(callbacks) {
  while(callbacks.length > 0) {
    var callback = callbacks.shift();
    if (typeof callback == 'function') {
      callback();
      continue;
    }
    var func = callback.func;
    if (typeof func === 'number') {
      if (callback.arg === undefined) {
        Runtime.dynCall('v', func);
      } else {
        Runtime.dynCall('vi', func, [callback.arg]);
      }
    } else {
      func(callback.arg === undefined ? null : callback.arg);
    }
  }
}
var __ATPRERUN__  = []; // functions called before the runtime is initialized
var __ATINIT__    = []; // functions called during startup
var __ATMAIN__    = []; // functions called when main() is to be run
var __ATEXIT__    = []; // functions called during shutdown
var __ATPOSTRUN__ = []; // functions called after the runtime has exited
var runtimeInitialized = false;
function preRun() {
  // compatibility - merge in anything from Module['preRun'] at this time
  if (Module['preRun']) {
    if (typeof Module['preRun'] == 'function') Module['preRun'] = [Module['preRun']];
    while (Module['preRun'].length) {
      addOnPreRun(Module['preRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPRERUN__);
}
function ensureInitRuntime() {
  if (runtimeInitialized) return;
  runtimeInitialized = true;
  callRuntimeCallbacks(__ATINIT__);
}
function preMain() {
  callRuntimeCallbacks(__ATMAIN__);
}
function exitRuntime() {
  callRuntimeCallbacks(__ATEXIT__);
}
function postRun() {
  // compatibility - merge in anything from Module['postRun'] at this time
  if (Module['postRun']) {
    if (typeof Module['postRun'] == 'function') Module['postRun'] = [Module['postRun']];
    while (Module['postRun'].length) {
      addOnPostRun(Module['postRun'].shift());
    }
  }
  callRuntimeCallbacks(__ATPOSTRUN__);
}
function addOnPreRun(cb) {
  __ATPRERUN__.unshift(cb);
}
Module['addOnPreRun'] = Module.addOnPreRun = addOnPreRun;
function addOnInit(cb) {
  __ATINIT__.unshift(cb);
}
Module['addOnInit'] = Module.addOnInit = addOnInit;
function addOnPreMain(cb) {
  __ATMAIN__.unshift(cb);
}
Module['addOnPreMain'] = Module.addOnPreMain = addOnPreMain;
function addOnExit(cb) {
  __ATEXIT__.unshift(cb);
}
Module['addOnExit'] = Module.addOnExit = addOnExit;
function addOnPostRun(cb) {
  __ATPOSTRUN__.unshift(cb);
}
Module['addOnPostRun'] = Module.addOnPostRun = addOnPostRun;
// Tools
// This processes a JS string into a C-line array of numbers, 0-terminated.
// For LLVM-originating strings, see parser.js:parseLLVMString function
function intArrayFromString(stringy, dontAddNull, length /* optional */) {
  var ret = (new Runtime.UTF8Processor()).processJSString(stringy);
  if (length) {
    ret.length = length;
  }
  if (!dontAddNull) {
    ret.push(0);
  }
  return ret;
}
Module['intArrayFromString'] = intArrayFromString;
function intArrayToString(array) {
  var ret = [];
  for (var i = 0; i < array.length; i++) {
    var chr = array[i];
    if (chr > 0xFF) {
      chr &= 0xFF;
    }
    ret.push(String.fromCharCode(chr));
  }
  return ret.join('');
}
Module['intArrayToString'] = intArrayToString;
// Write a Javascript array to somewhere in the heap
function writeStringToMemory(string, buffer, dontAddNull) {
  var array = intArrayFromString(string, dontAddNull);
  var i = 0;
  while (i < array.length) {
    var chr = array[i];
    HEAP8[(((buffer)+(i))|0)]=chr
    i = i + 1;
  }
}
Module['writeStringToMemory'] = writeStringToMemory;
function writeArrayToMemory(array, buffer) {
  for (var i = 0; i < array.length; i++) {
    HEAP8[(((buffer)+(i))|0)]=array[i];
  }
}
Module['writeArrayToMemory'] = writeArrayToMemory;
function unSign(value, bits, ignore, sig) {
  if (value >= 0) {
    return value;
  }
  return bits <= 32 ? 2*Math.abs(1 << (bits-1)) + value // Need some trickery, since if bits == 32, we are right at the limit of the bits JS uses in bitshifts
                    : Math.pow(2, bits)         + value;
}
function reSign(value, bits, ignore, sig) {
  if (value <= 0) {
    return value;
  }
  var half = bits <= 32 ? Math.abs(1 << (bits-1)) // abs is needed if bits == 32
                        : Math.pow(2, bits-1);
  if (value >= half && (bits <= 32 || value > half)) { // for huge values, we can hit the precision limit and always get true here. so don't do that
                                                       // but, in general there is no perfect solution here. With 64-bit ints, we get rounding and errors
                                                       // TODO: In i64 mode 1, resign the two parts separately and safely
    value = -2*half + value; // Cannot bitshift half, as it may be at the limit of the bits JS uses in bitshifts
  }
  return value;
}
if (!Math['imul']) Math['imul'] = function(a, b) {
  var ah  = a >>> 16;
  var al = a & 0xffff;
  var bh  = b >>> 16;
  var bl = b & 0xffff;
  return (al*bl + ((ah*bl + al*bh) << 16))|0;
};
Math.imul = Math['imul'];
// A counter of dependencies for calling run(). If we need to
// do asynchronous work before running, increment this and
// decrement it. Incrementing must happen in a place like
// PRE_RUN_ADDITIONS (used by emcc to add file preloading).
// Note that you can add dependencies in preRun, even though
// it happens right before run - run will be postponed until
// the dependencies are met.
var runDependencies = 0;
var runDependencyTracking = {};
var calledInit = false, calledRun = false;
var runDependencyWatcher = null;
function addRunDependency(id) {
  runDependencies++;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (id) {
    assert(!runDependencyTracking[id]);
    runDependencyTracking[id] = 1;
  } else {
    Module.printErr('warning: run dependency added without ID');
  }
}
Module['addRunDependency'] = addRunDependency;
function removeRunDependency(id) {
  runDependencies--;
  if (Module['monitorRunDependencies']) {
    Module['monitorRunDependencies'](runDependencies);
  }
  if (id) {
    assert(runDependencyTracking[id]);
    delete runDependencyTracking[id];
  } else {
    Module.printErr('warning: run dependency removed without ID');
  }
  if (runDependencies == 0) {
    if (runDependencyWatcher !== null) {
      clearInterval(runDependencyWatcher);
      runDependencyWatcher = null;
    } 
    // If run has never been called, and we should call run (INVOKE_RUN is true, and Module.noInitialRun is not false)
    if (!calledRun && shouldRunNow) run();
  }
}
Module['removeRunDependency'] = removeRunDependency;
Module["preloadedImages"] = {}; // maps url to image data
Module["preloadedAudios"] = {}; // maps url to audio data
function loadMemoryInitializer(filename) {
  function applyData(data) {
    HEAPU8.set(data, STATIC_BASE);
  }
  // always do this asynchronously, to keep shell and web as similar as possible
  addOnPreRun(function() {
    if (ENVIRONMENT_IS_NODE || ENVIRONMENT_IS_SHELL) {
      applyData(Module['readBinary'](filename));
    } else {
      Browser.asyncLoad(filename, function(data) {
        applyData(data);
      }, function(data) {
        throw 'could not load memory initializer ' + filename;
      });
    }
  });
}
// === Body ===
STATIC_BASE = 8;
STATICTOP = STATIC_BASE + 37320;
/* global initializers */ __ATINIT__.push({ func: function() { runPostSets() } });
var _stderr;
var __ZTVN10__cxxabiv120__si_class_type_infoE;
var __ZTVN10__cxxabiv117__class_type_infoE;
var __ZN6playerC1EPc;
var _stderr = _stderr=allocate([0,0,0,0,0,0,0,0], "i8", ALLOC_STATIC);
__ZTVN10__cxxabiv120__si_class_type_infoE=allocate([0,0,0,0,200,124,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0], "i8", ALLOC_STATIC);
__ZTVN10__cxxabiv117__class_type_infoE=allocate([0,0,0,0,216,124,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0], "i8", ALLOC_STATIC);
/* memory initializer */ allocate([62,0,0,0,161,0,0,0,9,1,0,0,121,1,0,0,68,2,0,0,6,3,0,0,131,4,0,0,39,6,0,0,212,8,0,0,16,12,0,0,218,17,0,0,89,24,0,0,114,36,0,0,131,51,0,0,228,82,0,0,255,127,0,0,87,1,0,0,107,1,0,0,129,1,0,0,152,1,0,0,176,1,0,0,202,1,0,0,229,1,0,0,2,2,0,0,32,2,0,0,65,2,0,0,99,2,0,0,135,2,0,0,174,2,0,0,0,0,0,0,184,94,0,0,40,0,0,0,178,0,0,0,0,0,0,0,0,0,0,0,24,0,0,0,49,0,0,0,74,0,0,0,97,0,0,0,120,0,0,0,141,0,0,0,161,0,0,0,180,0,0,0,197,0,0,0,212,0,0,0,224,0,0,0,235,0,0,0,244,0,0,0,250,0,0,0,253,0,0,0,255,0,0,0,253,0,0,0,250,0,0,0,244,0,0,0,235,0,0,0,224,0,0,0,212,0,0,0,197,0,0,0,180,0,0,0,161,0,0,0,141,0,0,0,120,0,0,0,97,0,0,0,74,0,0,0,49,0,0,0,24,0,0,0,0,0,0,0,232,255,255,255,207,255,255,255,182,255,255,255,159,255,255,255,136,255,255,255,115,255,255,255,95,255,255,255,76,255,255,255,59,255,255,255,44,255,255,255,32,255,255,255,21,255,255,255,12,255,255,255,6,255,255,255,3,255,255,255,1,255,255,255,3,255,255,255,6,255,255,255,12,255,255,255,21,255,255,255,32,255,255,255,44,255,255,255,59,255,255,255,76,255,255,255,95,255,255,255,115,255,255,255,136,255,255,255,159,255,255,255,182,255,255,255,207,255,255,255,232,255,255,255,0,0,0,0,248,255,255,255,240,255,255,255,232,255,255,255,224,255,255,255,216,255,255,255,208,255,255,255,200,255,255,255,192,255,255,255,184,255,255,255,176,255,255,255,168,255,255,255,160,255,255,255,152,255,255,255,144,255,255,255,136,255,255,255,128,255,255,255,120,255,255,255,112,255,255,255,104,255,255,255,96,255,255,255,88,255,255,255,80,255,255,255,72,255,255,255,64,255,255,255,56,255,255,255,48,255,255,255,40,255,255,255,32,255,255,255,24,255,255,255,16,255,255,255,8,255,255,255,255,0,0,0,248,0,0,0,240,0,0,0,232,0,0,0,224,0,0,0,216,0,0,0,208,0,0,0,200,0,0,0,192,0,0,0,184,0,0,0,176,0,0,0,168,0,0,0,160,0,0,0,152,0,0,0,144,0,0,0,136,0,0,0,128,0,0,0,120,0,0,0,112,0,0,0,104,0,0,0,96,0,0,0,88,0,0,0,80,0,0,0,72,0,0,0,64,0,0,0,56,0,0,0,48,0,0,0,40,0,0,0,32,0,0,0,24,0,0,0,16,0,0,0,8,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,1,255,255,255,0,0,0,0,8,0,0,0,16,0,0,0,24,0,0,0,32,0,0,0,40,0,0,0,48,0,0,0,56,0,0,0,64,0,0,0,72,0,0,0,80,0,0,0,88,0,0,0,96,0,0,0,104,0,0,0,112,0,0,0,120,0,0,0,128,0,0,0,136,0,0,0,144,0,0,0,152,0,0,0,160,0,0,0,168,0,0,0,176,0,0,0,184,0,0,0,192,0,0,0,200,0,0,0,208,0,0,0,216,0,0,0,224,0,0,0,232,0,0,0,240,0,0,0,248,0,0,0,1,255,255,255,8,255,255,255,16,255,255,255,24,255,255,255,32,255,255,255,40,255,255,255,48,255,255,255,56,255,255,255,64,255,255,255,72,255,255,255,80,255,255,255,88,255,255,255,96,255,255,255,104,255,255,255,112,255,255,255,120,255,255,255,128,255,255,255,136,255,255,255,144,255,255,255,152,255,255,255,160,255,255,255,168,255,255,255,176,255,255,255,184,255,255,255,192,255,255,255,200,255,255,255,208,255,255,255,216,255,255,255,224,255,255,255,232,255,255,255,240,255,255,255,248,255,255,255,118,111,114,98,105,115,0,0,0,1,0,0,128,0,0,0,86,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,2,2,2,2,2,2,2,2,3,3,3,3,3,3,4,4,4,4,5,5,5,5,6,6,6,6,7,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,15,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,33,34,36,38,40,42,44,46,48,50,52,54,56,58,60,62,65,68,72,77,80,84,91,95,98,103,109,114,120,126,127,2,3,4,5,6,7,8,0,3,2,4,5,6,7,8,0,4,3,5,2,6,7,8,0,5,4,6,2,3,7,8,0,6,5,7,2,3,4,8,0,7,6,8,2,3,4,5,0,8,7,6,2,3,4,5,0,208,94,0,0,164,0,0,0,56,0,0,0,0,0,0,0,232,94,0,0,20,1,0,0,82,0,0,0,0,0,0,0,136,114,0,0,248,107,0,0,104,103,0,0,112,99,0,0,88,3,40,3,250,2,208,2,166,2,128,2,92,2,58,2,26,2,252,1,224,1,197,1,172,1,148,1,125,1,104,1,83,1,64,1,46,1,29,1,13,1,254,0,240,0,226,0,214,0,202,0,190,0,180,0,170,0,160,0,151,0,143,0,135,0,127,0,120,0,113,0,82,3,34,3,245,2,203,2,162,2,125,2,89,2,55,2,23,2,249,1,221,1,194,1,169,1,145,1,123,1,101,1,81,1,62,1,44,1,28,1,12,1,253,0,239,0,225,0,213,0,201,0,189,0,179,0,169,0,159,0,150,0,142,0,134,0,126,0,119,0,113,0,76,3,28,3,240,2,197,2,158,2,120,2,85,2,51,2,20,2,246,1,218,1,191,1,166,1,142,1,120,1,99,1,79,1,60,1,42,1,26,1,10,1,251,0,237,0,224,0,211,0,199,0,188,0,177,0,167,0,158,0,149,0,141,0,133,0,125,0,118,0,112,0,70,3,23,3,234,2,192,2,153,2,116,2,80,2,47,2,16,2,242,1,214,1,188,1,163,1,139,1,117,1,96,1,76,1,58,1,40,1,24,1,8,1,249,0,235,0,222,0,209,0,198,0,187,0,176,0,166,0,157,0,148,0,140,0,132,0,125,0,118,0,111,0,64,3,17,3,229,2,187,2,148,2,111,2,76,2,43,2,12,2,239,1,211,1,185,1,160,1,136,1,114,1,94,1,74,1,56,1,38,1,22,1,6,1,247,0,233,0,220,0,208,0,196,0,185,0,175,0,165,0,156,0,147,0,139,0,131,0,124,0,117,0,110,0,58,3,11,3,224,2,182,2,143,2,107,2,72,2,39,2,8,2,235,1,207,1,181,1,157,1,134,1,112,1,91,1,72,1,53,1,36,1,20,1,4,1,245,0,232,0,219,0,206,0,195,0,184,0,174,0,164,0,155,0,146,0,138,0,130,0,123,0,116,0,109,0,52,3,6,3,218,2,177,2,139,2,102,2,68,2,35,2,4,2,231,1,204,1,178,1,154,1,131,1,109,1,89,1,69,1,51,1,34,1,18,1,2,1,244,0,230,0,217,0,205,0,193,0,183,0,172,0,163,0,154,0,145,0,137,0,129,0,122,0,115,0,109,0,46,3,0,3,213,2,172,2,134,2,98,2,63,2,31,2,1,2,228,1,201,1,175,1,151,1,128,1,107,1,86,1,67,1,49,1,32,1,16,1,0,1,242,0,228,0,216,0,204,0,192,0,181,0,171,0,161,0,152,0,144,0,136,0,128,0,121,0,114,0,108,0,139,3,88,3,40,3,250,2,208,2,166,2,128,2,92,2,58,2,26,2,252,1,224,1,197,1,172,1,148,1,125,1,104,1,83,1,64,1,46,1,29,1,13,1,254,0,240,0,226,0,214,0,202,0,190,0,180,0,170,0,160,0,151,0,143,0,135,0,127,0,120,0,132,3,82,3,34,3,245,2,203,2,163,2,124,2,89,2,55,2,23,2,249,1,221,1,194,1,169,1,145,1,123,1,101,1,81,1,62,1,44,1,28,1,12,1,253,0,238,0,225,0,212,0,200,0,189,0,179,0,169,0,159,0,150,0,142,0,134,0,126,0,119,0,126,3,76,3,28,3,240,2,197,2,158,2,120,2,85,2,51,2,20,2,246,1,218,1,191,1,166,1,142,1,120,1,99,1,79,1,60,1,42,1,26,1,10,1,251,0,237,0,223,0,211,0,199,0,188,0,177,0,167,0,158,0,149,0,141,0,133,0,125,0,118,0,119,3,70,3,23,3,234,2,192,2,153,2,116,2,80,2,47,2,16,2,242,1,214,1,188,1,163,1,139,1,117,1,96,1,76,1,58,1,40,1,24,1,8,1,249,0,235,0,222,0,209,0,198,0,187,0,176,0,166,0,157,0,148,0,140,0,132,0,125,0,118,0,113,3,64,3,17,3,229,2,187,2,148,2,111,2,76,2,43,2,12,2,238,1,211,1,185,1,160,1,136,1,114,1,94,1,74,1,56,1,38,1,22,1,6,1,247,0,233,0,220,0,208,0,196,0,185,0,175,0,165,0,156,0,147,0,139,0,131,0,123,0,117,0,107,3,58,3,11,3,224,2,182,2,143,2,107,2,72,2,39,2,8,2,235,1,207,1,181,1,157,1,134,1,112,1,91,1,72,1,53,1,36,1,20,1,4,1,245,0,232,0,219,0,206,0,195,0,184,0,174,0,164,0,155,0,146,0,138,0,130,0,123,0,116,0,100,3,52,3,6,3,218,2,177,2,139,2,102,2,68,2,35,2,4,2,231,1,204,1,178,1,154,1,131,1,109,1,89,1,69,1,51,1,34,1,18,1,2,1,244,0,230,0,217,0,205,0,193,0,183,0,172,0,163,0,154,0,145,0,137,0,129,0,122,0,115,0,94,3,46,3,0,3,213,2,172,2,134,2,98,2,63,2,31,2,1,2,228,1,201,1,175,1,151,1,128,1,107,1,86,1,67,1,49,1,32,1,16,1,0,1,242,0,228,0,216,0,203,0,192,0,181,0,171,0,161,0,152,0,144,0,136,0,128,0,121,0,114,0,96,56,96,0,0,160,96,0,1,62,96,0,1,12,72,231,0,95,0,0,254,0,0,0,32,1,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,6,0,0,0,8,0,0,0,12,0,0,0,16,0,0,0,24,0,0,0,32,0,0,0,48,0,0,0,64,0,0,0,96,0,0,0,128,0,0,0,192,0,0,0,0,1,0,0,128,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,6,0,0,0,8,0,0,0,12,0,0,0,16,0,0,0,24,0,0,0,32,0,0,0,48,0,0,0,64,0,0,0,96,0,0,0,128,0,0,0,192,0,0,0,0,0,0,1,0,0,0,3,0,0,0,7,0,0,0,15,0,0,0,31,0,0,0,63,0,0,0,127,0,0,0,255,0,0,0,255,1,0,0,255,3,0,0,255,7,0,0,255,15,0,0,255,31,0,0,255,63,0,0,255,127,0,0,0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13,14,14,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,0,0,0,0,0,0,22,0,0,0,2,0,0,0,22,0,0,0,8,0,0,0,132,0,0,0,2,0,0,0,74,0,0,0,40,0,0,0,44,0,0,0,14,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,0,0,0,20,0,0,0,12,0,0,0,6,0,0,0,62,0,0,0,4,0,0,0,32,0,0,0,14,0,0,0,48,95,0,0,102,0,0,0,114,0,0,0,0,0,0,0,168,95,0,0,4,0,0,0,28,0,0,0,0,0,0,0,192,95,0,0,44,1,0,0,60,0,0,0,0,0,0,0,216,95,0,0,18,1,0,0,246,0,0,0,0,0,0,0,40,96,0,0,134,0,0,0,48,1,0,0,0,0,0,0,128,96,0,0,84,0,0,0,46,1,0,0,0,0,0,0,168,101,0,0,80,0,0,0,78,0,0,0,0,0,0,0,10,0,0,0,4,0,0,0,48,0,0,0,12,0,0,0,34,0,0,0,32,0,0,0,30,0,0,0,36,0,0,0,0,0,0,0,2,0,0,0,4,0,0,0,1,0,0,0,3,0,0,0,5,0,0,0,255,255,255,255,255,255,255,255,6,0,0,0,8,0,0,0,10,0,0,0,7,0,0,0,9,0,0,0,11,0,0,0,255,255,255,255,255,255,255,255,12,0,0,0,14,0,0,0,16,0,0,0,13,0,0,0,15,0,0,0,17,0,0,0,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,49,0,0,0,97,0,0,0,141,0,0,0,180,0,0,0,212,0,0,0,235,0,0,0,250,0,0,0,255,0,0,0,250,0,0,0,235,0,0,0,212,0,0,0,180,0,0,0,141,0,0,0,97,0,0,0,49,0,0,0,0,0,0,0,207,255,255,255,159,255,255,255,115,255,255,255,76,255,255,255,44,255,255,255,21,255,255,255,6,255,255,255,1,255,255,255,6,255,255,255,21,255,255,255,44,255,255,255,76,255,255,255,115,255,255,255,159,255,255,255,207,255,255,255,240,96,0,0,14,0,0,0,152,0,0,0,0,0,0,0,144,97,0,0,0,1,0,0,176,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,255,255,255,255,1,0,0,0,1,0,0,0,254,255,255,255,1,0,0,0,1,0,0,0,252,255,255,255,1,0,0,0,1,0,0,0,248,255,255,255,1,0,0,0,1,0,0,0,240,255,255,255,1,0,0,0,1,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,4,0,0,0,1,0,0,0,1,0,0,0,8,0,0,0,1,0,0,0,1,0,0,0,16,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,3,0,0,0,2,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,200,97,0,0,34,0,0,0,4,1,0,0,0,0,0,0,0,128,64,192,32,160,96,224,16,144,80,208,48,176,112,240,8,136,72,200,40,168,104,232,24,152,88,216,56,184,120,248,4,132,68,196,36,164,100,228,20,148,84,212,52,180,116,244,12,140,76,204,44,172,108,236,28,156,92,220,60,188,124,252,2,130,66,194,34,162,98,226,18,146,82,210,50,178,114,242,10,138,74,202,42,170,106,234,26,154,90,218,58,186,122,250,6,134,70,198,38,166,102,230,22,150,86,214,54,182,118,246,14,142,78,206,46,174,110,238,30,158,94,222,62,190,126,254,1,129,65,193,33,161,97,225,17,145,81,209,49,177,113,241,9,137,73,201,41,169,105,233,25,153,89,217,57,185,121,249,5,133,69,197,37,165,101,229,21,149,85,213,53,181,117,245,13,141,77,205,45,173,109,237,29,157,93,221,61,189,125,253,3,131,67,195,35,163,99,227,19,147,83,211,51,179,115,243,11,139,75,203,43,171,107,235,27,155,91,219,59,187,123,251,7,135,71,199,39,167,103,231,23,151,87,215,55,183,119,247,15,143,79,207,47,175,111,239,31,159,95,223,63,191,127,255,0,0,0,0,1,0,0,0,2,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,11,0,0,0,12,0,0,0,13,0,0,0,19,0,0,0,20,0,0,0,21,0,0,0,32,0,0,0,32,0,0,0,64,0,0,0,64,0,0,0,96,0,0,0,96,0,0,0,128,0,0,0,128,0,0,0,224,0,0,0,224,0,0,0,192,0,0,0,0,0,0,0,224,97,0,0,8,1,0,0,16,0,0,0,0,0,0,0,184,98,0,0,38,0,0,0,70,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,99,0,0,36,0,0,0,28,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,32,99,0,0,234,0,0,0,40,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,200,116,0,0,32,0,0,0,42,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,136,99,0,0,118,0,0,0,42,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,109,0,0,2,0,0,0,42,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,99,0,0,62,1,0,0,68,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,208,99,0,0,204,0,0,0,36,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,100,0,0,62,0,0,0,38,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,100,0,0,8,0,0,0,84,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,100,0,0,92,0,0,0,76,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,88,100,0,0,22,1,0,0,88,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,136,100,0,0,122,0,0,0,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,100,0,0,120,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,152,86,0,0,26,1,0,0,86,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,208,86,0,0,54,0,0,0,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,86,0,0,240,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,86,0,0,172,0,0,0,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,87,0,0,158,0,0,0,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,40,87,0,0,144,0,0,0,50,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,168,86,0,0,196,0,0,0,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,192,86,0,0,12,1,0,0,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,120,87,0,0,6,0,0,0,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,87,0,0,70,0,0,0,54,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,56,87,0,0,216,0,0,0,20,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,88,87,0,0,218,0,0,0,62,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,104,87,0,0,214,0,0,0,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,112,0,0,126,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,87,0,0,226,0,0,0,64,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,8,98,0,0,194,0,0,0,188,0,0,0,0,0,0,0,216,87,0,0,130,0,0,0,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,88,88,0,0,26,0,0,0,26,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,128,88,0,0,228,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,88,0,0,154,0,0,0,46,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,208,88,0,0,16,1,0,0,82,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,20,0,0,176,19,0,0,152,19,0,0,216,18,0,0,168,18,0,0,144,18,0,0,120,18,0,0,80,18,0,0,192,17,0,0,144,17,0,0,168,17,0,0,232,16,0,0,208,16,0,0,24,17,0,0,224,15,0,0,16,16,0,0,88,16,0,0,184,16,0,0,200,15,0,0,56,18,0,0,224,19,0,0,200,19,0,0,120,17,0,0,0,17,0,0,96,17,0,0,72,17,0,0,48,17,0,0,216,17,0,0,8,18,0,0,32,18,0,0,240,17,0,0,176,15,0,0,40,16,0,0,248,15,0,0,64,16,0,0,248,19,0,0,136,16,0,0,160,16,0,0,192,18,0,0,112,16,0,0,0,0,0,0,0,0,0,0,240,88,0,0,166,0,0,0,78,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,224,88,0,0,96,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,89,0,0,54,1,0,0,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,89,0,0,182,0,0,0,56,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,144,88,0,0,98,0,0,0,72,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,40,89,0,0,56,1,0,0,60,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,15,0,0,0,17,0,0,0,18,0,0,0,20,0,0,0,21,0,0,0,22,0,0,0,23,0,0,0,25,0,0,0,26,0,0,0,27,0,0,0,28,0,0,0,29,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,46,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,49,0,0,0,50,0,0,0,51,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,54,0,0,0,54,0,0,0,55,0,0,0,56,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,58,0,0,0,59,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,61,0,0,0,62,0,0,0,63,0,0,0,63,0,0,0,64,0,0,0,64,0,0,0,0,0,0,0,24,98,0,0,180,0,0,0,30,1,0,0,0,0,0,0,0,0,3,88,3,40,2,250,2,208,2,166,2,128,2,92,2,58,2,26,1,252,1,224,1,197,1,172,1,148,1,125,1,104,1,83,1,64,1,46,1,29,1,13,0,254,0,240,0,226,0,214,0,202,0,190,0,180,0,170,0,160,0,151,0,143,0,135,0,127,0,120,0,113,0,0,0,0,0,0,72,98,0,0,238,0,0,0,162,0,0,0,0,0,0,0,128,98,0,0,200,0,0,0,22,0,0,0,0,0,0,0,16,101,0,0,224,0,0,0,2,1,0,0,0,0,0,0,86,28,0,0,34,28,0,0,238,27,0,0,187,27,0,0,135,27,0,0,85,27,0,0,34,27,0,0,240,26,0,0,191,26,0,0,142,26,0,0,93,26,0,0,44,26,0,0,252,25,0,0,204,25,0,0,156,25,0,0,109,25,0,0,62,25,0,0,16,25,0,0,226,24,0,0,180,24,0,0,134,24,0,0,89,24,0,0,44,24,0,0,0,24,0,0,212,23,0,0,168,23,0,0,124,23,0,0,81,23,0,0,38,23,0,0,251,22,0,0,209,22,0,0,167,22,0,0,125,22,0,0,84,22,0,0,43,22,0,0,2,22,0,0,217,21,0,0,177,21,0,0,137,21,0,0,98,21,0,0,58,21,0,0,19,21,0,0,236,20,0,0,198,20,0,0,159,20,0,0,121,20,0,0,84,20,0,0,46,20,0,0,9,20,0,0,228,19,0,0,192,19,0,0,155,19,0,0,119,19,0,0,83,19,0,0,48,19,0,0,12,19,0,0,233,18,0,0,198,18,0,0,164,18,0,0,130,18,0,0,95,18,0,0,62,18,0,0,28,18,0,0,251,17,0,0,218,17,0,0,185,17,0,0,152,17,0,0,120,17,0,0,87,17,0,0,55,17,0,0,24,17,0,0,248,16,0,0,217,16,0,0,186,16,0,0,155,16,0,0,125,16,0,0,94,16,0,0,64,16,0,0,34,16,0,0,4,16,0,0,231,15,0,0,202,15,0,0,173,15,0,0,144,15,0,0,115,15,0,0,87,15,0,0,58,15,0,0,30,15,0,0,2,15,0,0,231,14,0,0,203,14,0,0,176,14,0,0,149,14,0,0,122,14,0,0,95,14,0,0,69,14,0,0,43,14,0,0,17,14,0,0,247,13,0,0,221,13,0,0,195,13,0,0,170,13,0,0,145,13,0,0,120,13,0,0,88,3,0,0,40,3,0,0,250,2,0,0,208,2,0,0,166,2,0,0,128,2,0,0,92,2,0,0,58,2,0,0,26,2,0,0,252,1,0,0,224,1,0,0,197,1,0,0,172,1,0,0,148,1,0,0,125,1,0,0,104,1,0,0,83,1,0,0,64,1,0,0,46,1,0,0,29,1,0,0,13,1,0,0,254,0,0,0,240,0,0,0,226,0,0,0,214,0,0,0,202,0,0,0,190,0,0,0,180,0,0,0,170,0,0,0,160,0,0,0,151,0,0,0,143,0,0,0,135,0,0,0,127,0,0,0,120,0,0,0,113,0,0,0,255,255,255,255,0,0,0,0,184,101,0,0,148,0,0,0,20,0,0,0,0,0,0,0,200,101,0,0,150,0,0,0,138,0,0,0,0,0,0,0,2,0,0,0,28,0,0,0,22,0,0,0,20,0,0,0,2,0,0,0,28,0,0,0,22,0,0,0,20,0,0,0,192,96,0,0,74,0,0,0,104,0,0,0,0,0,0,0,248,101,0,0,160,0,0,0,50,0,0,0,0,0,0,0,54,110,192,250,150,42,235,238,3,74,162,219,170,73,170,234,2,0,0,0,147,241,70,174,183,88,195,157,139,95,188,152,191,35,122,67,4,0,0,0,112,170,3,77,251,47,31,115,217,253,186,254,19,27,183,1,1,0,0,0,233,152,1,44,112,14,180,58,240,50,23,17,48,88,41,178,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,240,102,0,0,0,0,0,0,1,0,0,0,152,117,0,0,4,0,0,0,144,109,0,0,1,0,0,0,1,0,0,0,152,117,0,0,4,0,0,0,144,104,0,0,1,0,0,0,1,0,0,0,112,100,0,0,4,0,0,0,224,96,0,0,1,0,0,0,1,0,0,0,112,100,0,0,4,0,0,0,168,93,0,0,0,0,0,0,0,0,0,0,32,92,0,0,6,0,0,0,216,89,0,0,0,0,0,0,0,0,0,0,32,92,0,0,8,0,0,0,176,87,0,0,1,0,0,0,0,0,0,0,128,121,0,0,6,0,0,0,168,120,0,0,1,0,0,0,0,0,0,0,128,121,0,0,8,0,0,0,104,119,0,0,1,0,0,0,0,0,0,0,48,118,0,0,4,0,0,0,248,116,0,0,1,0,0,0,0,0,0,0,128,115,0,0,4,0,0,0,40,114,0,0,1,0,0,0,0,0,0,0,128,115,0,0,6,0,0,0,184,113,0,0,1,0,0,0,0,0,0,0,128,115,0,0,8,0,0,0,96,112,0,0,1,0,0,0,0,0,0,0,248,111,0,0,4,0,0,0,168,143,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,24,111,0,0,38,1,0,0,220,0,0,0,0,0,0,0,47,0,0,0,43,0,0,0,40,0,0,0,37,0,0,0,35,0,0,0,32,0,0,0,30,0,0,0,29,0,0,0,27,0,0,0,26,0,0,0,56,103,0,0,202,0,0,0,18,0,0,0,0,0,0,0,176,103,0,0,88,0,0,0,132,0,0,0,0,0,0,0,0,104,0,0,252,0,0,0,24,0,0,0,0,0,0,0,88,104,0,0,124,0,0,0,248,0,0,0,0,0,0,0,152,104,0,0,242,0,0,0,34,1,0,0,0,0,0,0,16,105,0,0,236,0,0,0,24,1,0,0,0,0,0,0,88,105,0,0,112,0,0,0,168,0,0,0,0,0,0,0,8,106,0,0,116,0,0,0,110,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,1,0,0,255,3,0,0,255,7,0,0,255,15,0,0,255,31,0,0,96,106,0,0,48,0,0,0,14,1,0,0,0,0,0,0,88,107,0,0,232,0,0,0,106,0,0,0,0,0,0,0,8,0,0,0,18,0,0,0,16,0,0,0,38,0,0,0,6,0,0,0,24,0,0,0,26,0,0,0,46,0,0,0,0,72,100,116,130,138,146,154,162,166,170,174,178,182,234,190,194,196,198,200,202,204,206,208,210,212,214,216,218,220,222,224,226,226,228,228,230,230,232,232,234,234,236,236,238,238,240,240,242,242,244,244,246,246,248,248,250,250,252,252,254,254,254,254,254,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,0,0,0,0,0,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,10,0,0,0,11,0,0,0,13,0,0,0,15,0,0,0,17,0,0,0,19,0,0,0,23,0,0,0,27,0,0,0,31,0,0,0,35,0,0,0,43,0,0,0,51,0,0,0,59,0,0,0,67,0,0,0,83,0,0,0,99,0,0,0,115,0,0,0,131,0,0,0,163,0,0,0,195,0,0,0,227,0,0,0,2,1,0,0,0,0,0,0,144,108,0,0,42,1,0,0,140,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,10,0,0,0,11,0,0,0,13,0,0,0,16,0,0,0,19,0,0,0,22,0,0,0,26,0,0,0,32,0,0,0,43,0,0,0,64,0,0,0,128,0,0,0,62,180,228,51,9,145,243,51,139,178,1,52,60,32,10,52,35,26,19,52,96,169,28,52,167,215,38,52,75,175,49,52,80,59,61,52,112,135,73,52,35,160,86,52,184,146,100,52,85,109,115,52,136,159,129,52,252,11,138,52,147,4,147,52,105,146,156,52,50,191,166,52,63,149,177,52,147,31,189,52,228,105,201,52,173,128,214,52,54,113,228,52,166,73,243,52,136,140,1,53,192,247,9,53,6,239,18,53,118,123,28,53,192,166,38,53,55,123,49,53,218,3,61,53,94,76,73,53,59,97,86,53,185,79,100,53,252,37,115,53,138,121,129,53,134,227,137,53,124,217,146,53,133,100,156,53,82,142,166,53,51,97,177,53,37,232,188,53,220,46,201,53,206,65,214,53,65,46,228,53,87,2,243,53,143,102,1,54,79,207,9,54,245,195,18,54,152,77,28,54,232,117,38,54,50,71,49,54,116,204,60,54,94,17,73,54,101,34,86,54,206,12,100,54,184,222,114,54,151,83,129,54,28,187,137,54,114,174,146,54,175,54,156,54,129,93,166,54,53,45,177,54,199,176,188,54,228,243,200,54,1,3,214,54,96,235,227,54,30,187,242,54,162,64,1,55,235,166,9,55,241,152,18,55,201,31,28,55,30,69,38,55,61,19,49,55,30,149,60,55,111,214,72,55,162,227,85,55,247,201,99,55,137,151,114,55,175,45,129,55,190,146,137,55,116,131,146,55,230,8,156,55,190,44,166,55,71,249,176,55,121,121,188,55,254,184,200,55,71,196,213,55,146,168,227,55,248,115,242,55,192,26,1,56,147,126,9,56,249,109,18,56,6,242,27,56,98,20,38,56,86,223,48,56,216,93,60,56,146,155,72,56,242,164,85,56,51,135,99,56,110,80,114,56,211,7,129,56,107,106,137,56,130,88,146,56,42,219,155,56,9,252,165,56,104,197,176,56,59,66,188,56,41,126,200,56,160,133,213,56,217,101,227,56,232,44,242,56,233,244,0,57,70,86,9,57,14,67,18,57,81,196,27,57,181,227,37,57,127,171,48,57,162,38,60,57,197,96,72,57,83,102,85,57,131,68,99,57,104,9,114,57,1,226,128,57,36,66,137,57,157,45,146,57,123,173,155,57,99,203,165,57,153,145,176,57,13,11,188,57,102,67,200,57,11,71,213,57,50,35,227,57,237,229,241,57,29,207,0,58,5,46,9,58,48,24,18,58,169,150,27,58,21,179,37,58,183,119,48,58,124,239,59,58,10,38,72,58,199,39,85,58,230,1,99,58,120,194,113,58,59,188,128,58,233,25,137,58,198,2,146,58,219,127,155,58,203,154,165,58,216,93,176,58,239,211,187,58,179,8,200,58,136,8,213,58,159,224,226,58,7,159,241,58,92,169,0,59,208,5,9,59,94,237,17,59,15,105,27,59,132,130,37,59,253,67,48,59,103,184,59,59,97,235,71,59,77,233,84,59,93,191,98,59,156,123,113,59,127,150,128,59,186,241,136,59,249,215,145,59,71,82,155,59,65,106,165,59,39,42,176,59,226,156,187,59,18,206,199,59,23,202,212,59,32,158,226,59,53,88,241,59,166,131,0,60,167,221,8,60,152,194,17,60,130,59,27,60,1,82,37,60,84,16,48,60,97,129,59,60,200,176,71,60,229,170,84,60,232,124,98,60,212,52,113,60,207,112,128,60,150,201,136,60,58,173,145,60,192,36,155,60,197,57,165,60,133,246,175,60,229,101,187,60,130,147,199,60,185,139,212,60,180,91,226,60,121,17,241,60,251,93,0,61,137,181,8,61,223,151,17,61,2,14,27,61,141,33,37,61,185,220,47,61,109,74,59,61,64,118,71,61,145,108,84,61,133,58,98,61,34,238,112,61,42,75,128,61,127,161,136,61,136,130,145,61,72,247,154,61,88,9,165,61,242,194,175,61,248,46,187,61,3,89,199,61,109,77,212,61,92,25,226,61,209,202,240,61,91,56,0,62,119,141,8,62,51,109,17,62,144,224,26,62,39,241,36,62,46,169,47,62,135,19,59,62,202,59,71,62,77,46,84,62,55,248,97,62,132,167,112,62,143,37,128,62,115,121,136,62,226,87,145,62,220,201,154,62,249,216,164,62,109,143,175,62,27,248,186,62,149,30,199,62,51,15,212,62,23,215,225,62,61,132,240,62,198,18,0,63,114,101,8,63,147,66,17,63,43,179,26,63,206,192,36,63,177,117,47,63,178,220,58,63,101,1,71,63,29,240,83,63,251,181,97,63,251,96,112,63,0,0,128,63,224,108,0,0,90,0,0,0,146,0,0,0,0,0,0,0,72,109,0,0,46,0,0,0,94,0,0,0,0,0,0,0,0,1,2,2,3,3,3,3,4,4,4,4,4,4,4,4,112,110,0,0,170,0,0,0,60,1,0,0,0,0,0,0,232,110,0,0,206,0,0,0,66,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,6,0,0,0,7,0,0,0,8,0,0,0,9,0,0,0,0,0,0,0,48,111,0,0,190,0,0,0,184,0,0,0,0,0,0,0,47,116,109,112,0,0,0,0,136,111,0,0,52,1,0,0,250,0,0,0,0,0,0,0,40,112,0,0,128,0,0,0,108,0,0,0,0,0,0,0,16,113,0,0,244,0,0,0,212,0,0,0,0,0,0,0,255,15,11,13,10,2,1,3,4,29,0,0,0,0,0,0,255,15,11,13,10,2,1,3,4,29,0,0,0,0,0,0,255,163,11,13,10,2,1,3,4,29,0,6,5,255,255,9,255,27,7,254,171,172,16,255,255,255,255,0,0,0,0,0,255,0,0,0,1,0,0,0,2,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,112,0,0,0,113,0,0,0,114,0,0,0,156,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,157,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,117,0,0,0,255,0,0,0,255,0,0,0,255,0,0,0,11,0,0,0,255,0,0,0,255,0,0,0,15,0,0,0,255,0,0,0,118,0,0,0,12,0,0,0,0,0,13,0,0,0,0,0,0,0,0,0,0,0,0,0,0,171,13,2,255,172,255,255,255,11,255,10,14,3,9,255,255,255,163,7,1,4,255,5,6,0,0,0,0,0,0,0,255,163,11,13,10,2,1,3,4,29,0,6,5,128,129,9,137,27,7,254,135,172,16,17,22,138,132,0,0,0,0,0,255,163,171,3,5,4,6,172,7,0,8,25,12,10,165,166,157,156,1,2,254,253,132,133,9,255,20,27,29,11,13,16,17,14,169,170,0,0,0,0,255,249,248,122,251,254,4,253,252,123,255,255,250,255,255,15,121,120,122,166,123,126,0,0,128,0,0,0,216,25,0,0,176,35,0,0,232,11,0,0,64,28,0,0,56,13,0,0,184,11,0,0,168,11,0,0,72,24,0,0,208,32,0,0,176,32,0,0,48,21,0,0,128,26,0,0,112,5,0,0,216,26,0,0,8,24,0,0,200,26,0,0,64,33,0,0,80,33,0,0,160,21,0,0,152,85,0,0,112,81,0,0,48,33,0,0,32,26,0,0,16,26,0,0,112,26,0,0,96,26,0,0,80,26,0,0,208,43,0,0,24,14,0,0,144,21,0,0,32,10,0,0,192,41,0,0,24,33,0,0,176,41,0,0,48,26,0,0,136,85,0,0,152,11,0,0,208,44,0,0,224,44,0,0,80,41,0,0,248,23,0,0,40,13,0,0,224,40,0,0,96,5,0,0,200,11,0,0,240,80,0,0,56,24,0,0,160,32,0,0,248,11,0,0,160,35,0,0,168,85,0,0,160,15,0,0,224,32,0,0,64,26,0,0,72,86,0,0,176,21,0,0,216,11,0,0,104,18,0,0,0,0,0,0,160,113,0,0,76,0,0,0,58,1,0,0,0,0,0,0,96,115,0,0,100,0,0,0,230,0,0,0,0,0,0,0,3,0,0,0,1,0,0,0,4,0,0,0,12,0,0,0,24,0,0,0,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,0,0,0,1,0,0,0,1,0,0,0,3,0,0,0,6,0,0,0,13,0,0,0,31,0,0,0,78,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,184,255,0,0,112,255,0,0,40,255,0,0,224,254,0,0,152,254,0,0,80,254,0,0,8,254,0,0,192,253,0,0,120,253,0,0,48,253,0,0,232,252,0,0,160,252,0,0,88,252,0,0,16,252,0,0,200,251,0,0,19,208,2,67,43,183,4,67,89,165,6,67,183,154,8,67,96,151,10,67,112,155,12,67,0,167,14,67,47,186,16,67,23,213,18,67,214,247,20,67,137,34,23,67,77,85,25,67,65,144,27,67,131,211,29,67,50,31,32,67,109,115,34,67,84,208,36,67,7,54,39,67,168,164,41,67,87,28,44,67,55,157,46,67,104,39,49,67,15,187,51,67,78,88,54,67,73,255,56,67,37,176,59,67,5,107,62,67,16,48,65,67,106,255,67,67,60,217,70,67,171,189,73,67,223,172,76,67,0,167,79,67,55,172,82,67,174,188,85,67,141,216,88,67,0,0,92,67,49,51,95,67,77,114,98,67,128,189,101,67,246,20,105,67,222,120,108,67,101,233,111,67,187,102,115,67,16,241,118,67,147,136,122,67,119,45,126,67,246,239,128,67,19,208,130,67,43,183,132,67,89,165,134,67,183,154,136,67,96,151,138,67,112,155,140,67,0,167,142,67,47,186,144,67,23,213,146,67,214,247,148,67,137,34,151,67,77,85,153,67,65,144,155,67,131,211,157,67,50,31,160,67,109,115,162,67,84,208,164,67,7,54,167,67,168,164,169,67,87,28,172,67,55,157,174,67,104,39,177,67,15,187,179,67,78,88,182,67,73,255,184,67,37,176,187,67,5,107,190,67,16,48,193,67,106,255,195,67,60,217,198,67,171,189,201,67,223,172,204,67,0,167,207,67,55,172,210,67,174,188,213,67,141,216,216,67,0,0,220,67,49,51,223,67,77,114,226,67,128,189,229,67,246,20,233,67,222,120,236,67,101,233,239,67,187,102,243,67,16,241,246,67,147,136,250,67,119,45,254,67,246,239,0,68,19,208,2,68,43,183,4,68,89,165,6,68,183,154,8,68,96,151,10,68,112,155,12,68,0,167,14,68,47,186,16,68,23,213,18,68,214,247,20,68,137,34,23,68,77,85,25,68,65,144,27,68,131,211,29,68,50,31,32,68,109,115,34,68,84,208,36,68,7,54,39,68,168,164,41,68,87,28,44,68,55,157,46,68,104,39,49,68,15,187,51,68,78,88,54,68,73,255,56,68,37,176,59,68,5,107,62,68,16,48,65,68,106,255,67,68,60,217,70,68,171,189,73,68,223,172,76,68,0,167,79,68,55,172,82,68,174,188,85,68,141,216,88,68,0,0,92,68,49,51,95,68,77,114,98,68,128,189,101,68,246,20,105,68,222,120,108,68,101,233,111,68,187,102,115,68,16,241,118,68,147,136,122,68,119,45,126,68,246,239,128,68,19,208,130,68,43,183,132,68,89,165,134,68,183,154,136,68,96,151,138,68,112,155,140,68,0,167,142,68,47,186,144,68,23,213,146,68,214,247,148,68,137,34,151,68,77,85,153,68,65,144,155,68,131,211,157,68,50,31,160,68,109,115,162,68,84,208,164,68,7,54,167,68,168,164,169,68,87,28,172,68,55,157,174,68,104,39,177,68,15,187,179,68,78,88,182,68,73,255,184,68,37,176,187,68,5,107,190,68,16,48,193,68,106,255,195,68,60,217,198,68,171,189,201,68,223,172,204,68,0,167,207,68,55,172,210,68,174,188,213,68,141,216,216,68,0,0,220,68,49,51,223,68,77,114,226,68,128,189,229,68,246,20,233,68,222,120,236,68,101,233,239,68,187,102,243,68,16,241,246,68,147,136,250,68,119,45,254,68,246,239,0,69,19,208,2,69,43,183,4,69,89,165,6,69,183,154,8,69,96,151,10,69,112,155,12,69,0,167,14,69,47,186,16,69,23,213,18,69,214,247,20,69,137,34,23,69,77,85,25,69,65,144,27,69,131,211,29,69,50,31,32,69,109,115,34,69,84,208,36,69,7,54,39,69,168,164,41,69,87,28,44,69,55,157,46,69,104,39,49,69,15,187,51,69,78,88,54,69,73,255,56,69,37,176,59,69,5,107,62,69,16,48,65,69,106,255,67,69,60,217,70,69,171,189,73,69,223,172,76,69,0,167,79,69,55,172,82,69].concat([174,188,85,69,141,216,88,69,0,0,92,69,49,51,95,69,77,114,98,69,128,189,101,69,246,20,105,69,222,120,108,69,101,233,111,69,187,102,115,69,16,241,118,69,147,136,122,69,119,45,126,69,246,239,128,69,19,208,130,69,43,183,132,69,89,165,134,69,183,154,136,69,96,151,138,69,112,155,140,69,0,167,142,69,47,186,144,69,23,213,146,69,214,247,148,69,137,34,151,69,77,85,153,69,65,144,155,69,131,211,157,69,50,31,160,69,109,115,162,69,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,3,0,0,0,2,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,116,0,0,142,0,0,0,36,1,0,0,0,0,0,0,72,112,0,0,16,120,0,0,160,110,0,0,112,105,0,0,56,101,0,0,72,97,0,0,56,94,0,0,160,92,0,0,104,90,0,0,104,88,0,0,248,121,0,0,176,120,0,0,112,119,0,0,80,118,0,0,0,117,0,0,160,115,0,0,48,114,0,0,248,113,0,0,40,113,0,0,24,112,0,0,168,111,0,0,120,111,0,0,0,0,0,0,0,0,0,0,48,117,0,0,30,0,0,0,50,1,0,0,0,0,0,0,16,0,0,0,17,0,0,0,18,0,0,0,0,0,0,0,8,0,0,0,7,0,0,0,9,0,0,0,6,0,0,0,10,0,0,0,5,0,0,0,11,0,0,0,4,0,0,0,12,0,0,0,3,0,0,0,13,0,0,0,2,0,0,0,14,0,0,0,1,0,0,0,15,0,0,0,0,0,0,0,168,117,0,0,192,0,0,0,52,0,0,0,0,0,0,0,56,119,0,0,186,0,0,0,210,0,0,0,0,0,0,0,0,0,128,63,191,53,123,63,113,130,118,63,166,229,113,63,243,94,109,63,238,237,104,63,48,146,100,63,82,75,96,63,241,24,92,63,170,250,87,63,29,240,83,63,236,248,79,63,185,20,76,63,42,67,72,63,230,131,68,63,149,214,64,63,225,58,61,63,117,176,57,63,255,54,54,63,46,206,50,63,177,117,47,63,59,45,44,63,127,244,40,63,50,203,37,63,9,177,34,63,188,165,31,63,4,169,28,63,155,186,25,63,60,218,22,63,165,7,20,63,147,66,17,63,198,138,14,63,253,223,11,63,251,65,9,63,131,176,6,63,87,43,4,63,62,178,1,63,252,137,254,62,187,198,249,62,74,26,245,62,60,132,240,62,38,4,236,62,159,153,231,62,63,68,227,62,161,3,223,62,97,215,218,62,30,191,214,62,121,186,210,62,19,201,206,62,144,234,202,62,149,30,199,62,202,100,195,62,216,188,191,62,106,38,188,62,42,161,184,62,200,44,181,62,242,200,177,62,88,117,174,62,174,49,171,62,167,253,167,62,248,216,164,62,87,195,161,62,125,188,158,62,34,196,155,62,2,218,152,62,215,253,149,62,96,47,147,62,90,110,144,62,133,186,141,62,162,19,139,62,115,121,136,62,187,235,133,62,62,106,131,62,194,244,128,62,26,22,125,62,207,89,120,62,50,180,115,62,216,36,111,62,85,171,106,62,65,71,102,62,54,248,97,62,206,189,93,62,167,151,89,62,96,133,85,62,153,134,81,62,246,154,77,62,26,194,73,62,171,251,69,62,82,71,66,62,183,164,62,62,135,19,59,62,108,147,55,62,22,36,52,62,51,197,48,62,118,118,45,62,145,55,42,62,56,8,39,62,33,232,35,62,2,215,32,62,147,212,29,62,143,224,26,62,177,250,23,62,180,34,21,62,86,88,18,62,86,155,15,62,116,235,12,62,113,72,10,62,15,178,7,62,18,40,5,62,63,170,2,62,90,56,0,62,88,164,251,61,248,238,246,61,37,80,242,61,116,199,237,61,123,84,233,61,209,246,228,61,17,174,224,61,215,121,220,61,192,89,216,61,108,77,212,61,123,84,208,61,146,110,204,61,85,155,200,61,107,218,196,61,123,43,193,61,48,142,189,61,53,2,186,61,232,120,0,0,40,1,0,0,156,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,9,0,0,0,10,0,0,0,10,0,0,0,11,0,0,0,11,0,0,0,12,0,0,0,12,0,0,0,13,0,0,0,13,0,0,0,1,0,0,0,2,0,0,0,3,0,0,0,4,0,0,0,5,0,0,0,7,0,0,0,9,0,0,0,13,0,0,0,17,0,0,0,25,0,0,0,33,0,0,0,49,0,0,0,65,0,0,0,97,0,0,0,129,0,0,0,193,0,0,0,1,1,0,0,129,1,0,0,1,2,0,0,1,3,0,0,1,4,0,0,1,6,0,0,1,8,0,0,1,12,0,0,1,16,0,0,1,24,0,0,1,32,0,0,1,48,0,0,1,64,0,0,1,96,0,0,104,121,0,0,68,0,0,0,174,0,0,0,0,0,0,0,16,87,0,0,198,0,0,0,10,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,255,255,255,255,255,255,255,255,255,254,255,254,255,254,255,254,255,254,255,253,255,253,255,253,255,252,255,252,255,252,255,251,255,251,255,250,255,250,255,250,255,249,255,249,255,248,255,248,255,247,255,247,255,246,255,246,255,245,255,245,255,244,255,243,255,243,255,242,255,242,255,241,255,240,255,240,255,239,255,238,255,237,255,237,255,236,255,235,255,234,255,234,255,233,255,232,255,231,255,230,255,230,255,229,255,228,255,227,255,226,255,225,255,224,255,223,255,222,255,221,255,220,255,219,255,218,255,217,255,216,255,215,255,214,255,213,255,212,255,211,255,210,255,209,255,208,255,207,255,205,255,204,255,203,255,202,255,201,255,199,255,198,255,197,255,196,255,195,255,193,255,192,255,191,255,189,255,188,255,187,255,186,255,184,255,183,255,181,255,180,255,179,255,177,255,176,255,174,255,173,255,172,255,170,255,169,255,167,255,166,255,164,255,163,255,161,255,160,255,158,255,157,255,155,255,154,255,152,255,150,255,149,255,147,255,146,255,144,255,142,255,141,255,139,255,137,255,136,255,134,255,132,255,131,255,129,255,127,255,126,255,124,255,122,255,120,255,119,255,117,255,115,255,113,255,111,255,110,255,108,255,106,255,104,255,102,255,100,255,99,255,97,255,95,255,93,255,91,255,89,255,87,255,85,255,83,255,81,255,80,255,78,255,76,255,74,255,72,255,70,255,68,255,66,255,64,255,62,255,60,255,58,255,56,255,54,255,51,255,49,255,47,255,45,255,43,255,41,255,39,255,37,255,35,255,33,255,31,255,28,255,26,255,24,255,22,255,20,255,18,255,16,255,13,255,11,255,9,255,7,255,5,255,2,255,0,255,254,254,252,254,249,254,247,254,245,254,243,254,240,254,238,254,236,254,234,254,231,254,229,254,227,254,224,254,222,254,220,254,217,254,215,254,213,254,210,254,208,254,206,254,203,254,201,254,199,254,196,254,194,254,192,254,189,254,187,254,184,254,182,254,180,254,177,254,175,254,172,254,170,254,167,254,165,254,163,254,160,254,158,254,155,254,153,254,150,254,148,254,145,254,143,254,140,254,138,254,135,254,133,254,130,254,128,254,125,254,123,254,120,254,118,254,115,254,113,254,110,254,108,254,105,254,103,254,100,254,98,254,95,254,93,254,90,254,88,254,85,254,82,254,80,254,77,254,75,254,72,254,70,254,67,254,64,254,62,254,59,254,57,254,54,254,51,254,49,254,46,254,44,254,41,254,38,254,36,254,33,254,31,254,28,254,25,254,23,254,20,254,17,254,15,254,12,254,10,254,7,254,4,254,2,254,255,253,252,253,250,253,247,253,245,253,242,253,239,253,237,253,234,253,231,253,229,253,226,253,223,253,221,253,218,253,215,253,213,253,210,253,207,253,205,253,202,253,199,253,197,253,194,253,191,253,189,253,186,253,183,253,181,253,178,253,175,253,173,253,170,253,167,253,165,253,162,253,159,253,157,253,154,253,151,253,149,253,146,253,143,253,141,253,138,253,135,253,133,253,130,253,127,253,125,253,122,253,119,253,117,253,114,253,111,253,109,253,106,253,103,253,101,253,98,253,96,253,93,253,90,253,88,253,85,253,82,253,80,253,77,253,74,253,72,253,69,253,66,253,64,253,61,253,59,253,56,253,53,253,51,253,48,253,45,253,43,253,40,253,38,253,35,253,32,253,30,253,27,253,24,253,22,253,19,253,17,253,14,253,11,253,9,253,6,253,4,253,1,253,255,252,252,252,249,252,247,252,244,252,242,252,239,252,237,252,234,252,231,252,229,252,226,252,224,252,221,252,219,252,216,252,214,252,211,252,209,252,206,252,204,252,201,252,199,252,196,252,194,252,191,252,189,252,186,252,184,252,181,252,179,252,176,252,174,252,171,252,169,252,166,252,164,252,161,252,159,252,157,252,154,252,152,252,149,252,147,252,144,252,142,252,140,252,137,252,135,252,132,252,130,252,128,252,125,252,123,252,121,252,118,252,116,252,113,252,111,252,109,252,106,252,104,252,102,252,100,252,97,252,95,252,93,252,90,252,88,252,86,252,83,252,81,252,79,252,77,252,74,252,72,252,70,252,68,252,66,252,63,252,61,252,59,252,57,252,55,252,52,252,50,252,48,252,46,252,44,252,42,252,39,252,37,252,35,252,33,252,31,252,29,252,27,252,25,252,22,252,20,252,18,252,16,252,14,252,12,252,10,252,8,252,6,252,4,252,2,252,0,252,254,251,252,251,250,251,248,251,246,251,244,251,242,251,240,251,238,251,236,251,234,251,233,251,231,251,229,251,227,251,225,251,223,251,221,251,219,251,218,251,216,251,214,251,212,251,210,251,209,251,207,251,205,251,203,251,201,251,200,251,198,251,196,251,195,251,193,251,191,251,189,251,188,251,186,251,184,251,183,251,181,251,179,251,178,251,176,251,175,251,173,251,171,251,170,251,168,251,167,251,165,251,164,251,162,251,161,251,159,251,158,251,156,251,155,251,153,251,152,251,150,251,149,251,147,251,146,251,145,251,143,251,142,251,140,251,139,251,138,251,136,251,135,251,134,251,132,251,131,251,130,251,129,251,127,251,126,251,125,251,124,251,122,251,121,251,120,251,119,251,118,251,117,251,115,251,114,251,113,251,112,251,111,251,110,251,109,251,108,251,107,251,106,251,105,251,104,251,103,251,102,251,101,251,100,251,99,251,98,251,97,251,96,251,95,251,94,251,93,251,93,251,92,251,91,251,90,251,89,251,88,251,88,251,87,251,86,251,85,251,85,251,84,251,83,251,83,251,82,251,81,251,81,251,80,251,79,251,79,251,78,251,78,251,77,251,76,251,76,251,75,251,75,251,74,251,74,251,73,251,73,251,73,251,72,251,72,251,71,251,71,251,71,251,70,251,70,251,70,251,69,251,69,251,69,251,69,251,68,251,68,251,68,251,68,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,66,251,66,251,66,251,66,251,66,251,66,251,66,251,66,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,68,251,68,251,68,251,68,251,68,251,69,251,69,251,69,251,70,251,70,251,70,251,71,251,71,251,72,251,72,251,72,251,73,251,73,251,74,251,74,251,75,251,75,251,76,251,76,251,77,251,78,251,78,251,79,251,80,251,80,251,81,251,82,251,82,251,83,251,84,251,85,251,85,251,86,251,87,251,88,251,89,251,90,251,91,251,92,251,93,251,93,251,94,251,95,251,96,251,98,251,99,251,100,251,101,251,102,251,103,251,104,251,105,251,106,251,108,251,109,251,110,251,111,251,113,251,114,251,115,251,117,251,118,251,119,251,121,251,122,251,124,251,125,251,127,251,128,251,130,251,131,251,133,251,134,251,136,251,137,251,139,251,141,251,142,251,144,251,146,251,147,251,149,251,151,251,153,251,155,251,156,251,158,251,160,251,162,251,164,251,166,251,168,251,170,251,172,251,174,251,176,251,178,251,180,251,182,251,184,251,186,251,188,251,191,251,193,251,195,251,197,251,200,251,202,251,204,251,206,251,209,251,211,251,214,251,216,251,218,251,221,251,223,251,226,251,228,251,231,251,233,251,236,251,239,251,241,251,244,251,247,251,249,251,252,251,255,251,2,252,4,252,7,252,10,252,13,252,16,252,19,252,22,252,25,252,27,252,30,252,33,252,37,252,40,252,43,252,46,252,49,252,52,252,55,252,58,252,62,252,65,252,68,252,71,252,75,252,78,252,81,252,85,252,88,252,92,252,95,252,99,252,102,252,106,252,109,252,113,252,116,252,120,252,124,252,127,252,131,252,135,252,138,252,142,252,146,252,150,252,154,252,158,252,161,252,165,252,169,252,173,252,177,252,181,252,185,252,189,252,193,252,198,252,202,252,206,252,210,252,214,252,218,252,223,252,227,252,231,252,236,252,240,252,244,252,249,252,253,252,2,253,6,253,11,253,15,253,20,253,24,253,29,253,34,253,38,253,43,253,48,253,53,253,57,253,62,253,67,253,72,253,77,253,82,253,86,253,91,253,96,253,101,253,106,253,112,253,117,253,122,253,127,253,132,253,137,253,142,253,148,253,153,253,158,253,164,253,169,253,174,253,180,253,185,253,191,253,196,253,202,253,207,253,213,253,218,253,224,253,230,253,235,253,241,253,247,253,253,253,2,254,8,254,14,254,20,254,26,254,32,254,38,254,44,254,50,254,56,254,62,254,68,254,74,254,80,254,87,254,93,254,99,254,105,254,112,254,118,254,124,254,131,254,137,254,144,254,150,254,157,254,163,254,170,254,176,254,183,254,190,254,196,254,203,254,210,254,217,254,223,254,230,254,237,254,244,254,251,254,2,255,9,255,16,255,23,255,30,255,37,255,44,255,51,255,58,255,66,255,73,255,80,255,87,255,95,255,102,255,110,255,117,255,124,255,132,255,139,255,147,255,155,255,162,255,170,255,178,255,185,255,193,255,201,255,209,255,216,255,224,255,232,255,240,255,248,255,0,0,8,0,16,0,24,0,32,0,41,0,49,0,58,0,66,0,75,0,83,0,92,0,100,0,109,0,118,0,127,0,136,0,145,0,154,0,163,0,172,0,182,0,191,0,200,0,210,0,219,0,229,0,238,0,248,0,2,1,12,1,21,1,31,1,41,1,51,1,61,1,71,1,82,1,92,1,102,1,113,1,123,1,133,1,144,1,155,1,165,1,176,1,187,1,197,1,208,1,219,1,230,1,241,1,252,1,8,2,19,2,30,2,41,2,53,2,64,2,76,2,87,2,99,2,110,2,122,2,134,2,146,2,157,2,169,2,181,2,193,2,205,2,217,2,230,2,242,2,254,2,10,3,23,3,35,3,48,3,60,3,73,3,86,3,98,3,111,3,124,3,137,3,149,3,162,3,175,3,188,3,202,3,215,3,228,3,241,3,254,3,12,4,25,4,39,4,52,4,66,4,79,4,93,4,107,4,120,4,134,4,148,4,162,4,176,4,190,4,204,4,218,4,232,4,246,4,4,5,18,5,33,5,47,5,62,5,76,5,90,5,105,5,120,5,134,5,149,5,164,5,178,5,193,5,208,5,223,5,238,5,253,5,12,6,27,6,42,6,57,6,72,6,88,6,103,6,118,6,134,6,149,6,165,6,180,6,196,6,211,6,227,6,243,6,2,7,18,7,34,7,50,7,66,7,82,7,98,7,114,7,130,7,146,7,162,7,178,7,194,7,211,7,227,7,243,7,4,8,20,8,36,8,53,8,69,8,86,8,103,8,119,8,136,8,153,8,170,8,186,8,203,8,220,8,237,8,254,8,15,9,32,9,49,9,66,9,83,9,101,9,118,9,135,9,152,9,170,9,187,9,205,9,222,9,239,9,1,10,19,10,36,10,54,10,71,10,89,10,107,10,125,10,142,10,160,10,178,10,196,10,214,10,232,10,250,10,12,11,30,11,48,11,66,11,85,11,103,11,121,11,139,11,158,11,176,11,194,11,213,11,231,11,250,11,12,12,31,12,49,12,68,12,87,12,105,12,124,12,143,12,161,12,180,12,199,12,218,12,237,12,0,13,19,13,38,13,57,13,76,13,95,13,114,13,133,13,152,13,171,13,190,13,210,13,229,13,248,13,11,14,31,14,50,14,70,14,89,14,109,14,128,14,148,14,167,14,187,14,206,14,226,14,245,14,9,15,29,15,49,15,68,15,88,15,108,15,128,15,148,15,167,15,187,15,207,15,227,15,247,15,11,16,31,16,51,16,71,16,92,16,112,16,132,16,152,16,172,16,192,16,213,16,233,16,253,16,18,17,38,17,58,17,79,17,99,17,119,17,140,17,160,17,181,17,201,17,222,17,243,17,7,18,28,18,48,18,69,18,90,18,110,18,131,18,152,18,172,18,193,18,214,18,235,18,0,19,21,19,41,19,62,19,83,19,104,19,125,19,146,19,167,19,188,19,209,19,230,19,251,19,16,20,37,20,58,20,79,20,101,20,122,20,143,20,164,20,185,20,207,20,228,20,249,20,14,21,36,21,57,21,78,21,100,21,121,21,142,21,164,21,185,21,206,21,228,21,249,21,15,22,36,22,58,22,79,22,101,22,122,22,144,22,165,22,187,22,209,22,230,22,252,22,17,23,39,23,61,23,82,23,104,23,126,23,147,23,169,23,191,23,213,23,234,23,0,24,22,24,44,24,65,24,87,24,109,24,131,24,153,24,174,24,196,24,218,24,240,24,6,25,28,25,50,25,72,25,93,25,115,25,137,25,159,25,181,25,203,25,225,25,247,25,13,26,35,26,57,26,79,26,101,26,123,26,145,26,167,26,189,26,211,26,233,26,255,26,22,27,44,27,66,27,88,27,110,27,132,27,154,27,176,27,198,27,220,27,243,27,9,28,31,28,53,28,75,28,97,28,120,28,142,28,164,28,186,28,208,28,230,28,253,28,19,29,41,29,63,29,85,29,107,29,130,29,152,29,174,29,196,29,219,29,241,29,7,30,29,30,51,30,74,30,96,30,118,30,140,30,162,30,185,30,207,30,229,30,251,30,18,31,40,31,62,31,84,31,106,31,129,31,151,31,173,31,195,31,218,31,240,31,6,32,28,32,50,32,73,32,95,32,117,32,139,32,161,32,184,32,206,32,228,32,250,32,16,33,39,33,61,33,83,33,105,33,127,33,149,33,172,33,194,33,216,33,238,33,4,34,26,34,48,34,71,34,93,34,115,34,137,34,159,34,181,34,203,34,225,34,247,34,14,35,36,35,58,35,80,35,102,35,124,35,146,35,168,35,190,35,212,35,234,35,0,36,22,36,44,36,66,36,88,36,110,36,132,36,154,36,176,36,198,36,220,36,242,36,7,37,29,37,51,37,73,37,95,37,117,37,139,37,161,37,181,37,204,37,226,37,248,37,14,38,35,38,56,38,78,38,101,38,123,38,144,38,165,38,187,38,208,38,231,38,253,38,18,39,39,39,61,39,84,39,104,39,126,39,148,39,168,39,191,39,212,39,233,39,255,39,20,40,43,40,63,40,85,40,106,40,128,40,149,40,170,40,191,40,213,40,234,40,0,41,21,41,42,41,64,41,85,41,106,41,127,41,148,41,170,41,191,41,212,41,232,41,254,41,20,42,40,42,62,42,82,42,104,42,124,42,146,42,166,42,187,42,207,42,230,42,250,42,15,43,35,43,56,43,76,43,99,43,119,43,140,43,160,43,181,43,201,43,223,43,243,43,8,44,28,44,49,44,69,44,90,44,110,44,130,44,151,44,172,44,193,44,213,44,234,44,254,44,18,45,39,45,58,45,79,45,99,45,120,45,140,45,160,45,180,45,200,45,221,45,241,45,4,46,25,46,45,46,63,46,84,46,104,46,124,46,144,46,164,46,184,46,204,46,222,46,243,46,7,47,27,47,47,47,65,47,86,47,105,47,125,47,144,47,163,47,183,47,203,47,222,47,241,47,5,48,24,48,44,48,63,48,82,48,102,48,120,48,139,48,160,48,178,48,197,48,216,48,236,48,254,48,18,49,36,49,55,49,74,49,93,49,111,49,131,49,150,49,168,49,187,49,206,49,224,49,243,49,6,50,23,50,42,50,60,50,79,50,98,50,116,50,134,50,153,50,171,50,189,50,208,50,226,50,244,50,6,51,24,51,42,51,60,51,78,51,96,51,114,51,132,51,149,51,168,51,185,51,202,51,221,51,238,51,0,52,18,52,35,52,53,52,71,52,87,52,105,52,123,52,139,52,157,52,174,52,192,52,209,52,226,52,244,52,4,53,21,53,39,53,55,53,73,53,89,53,105,53,123,53,139,53,156,53,173,53,190,53,207,53,223,53,240,53,255,53,16,54,33,54,49,54,66,54,82,54,99,54,115,54,131,54,147,54,162,54,179,54,195,54,211,54,227,54,243,54,2,55,19,55,35,55,50,55,66,55,81,55,97,55,112,55,128,55,143,55,159,55,174,55,190,55,205,55,221,55,236,55,250,55,10,56,25,56,40,56,55,56,70,56,84,56,99,56,114,56,129,56,144,56,158,56,173,56,187,56,202,56,217,56,231,56,246,56,3,57,18,57,33,57,46,57,61,57,76,57,89,57,104,57,117,57,131,57,145,57,160,57,172,57,186,57,201,57,214,57,227,57,241,57,254,57,12,58,26,58,39,58,52,58,65,58,79,58,92,58,105,58,118,58,131,58,144,58,156,58,169,58,183,58,195,58,208,58,219,58,233,58,246,58,2,59,14,59,27,59,39,59,51,59,63,59,76,59,88,59,101,59,112,59,124,59,136,59,148,59,160,59,172,59,183,59,195,59,206,59,217,59,229,59,241,59,252,59,7,60,19,60,30,60,41,60,52,60,63,60,73,60,86,60,96,60,107,60,118,60,129,60,139,60,149,60,160,60,170,60,180,60,192,60,202,60,212,60,222,60,232,60,242,60,253,60,7,61,17,61,26,61,36,61,46,61,56,61,65,61,75,61,84,61,93,61,103,61,113,61,122,61,131,61,140,61,149,61,158,61,167,61,176,61,185,61,194,61,202,61,211,61,220,61,227,61,237,61,245,61,254,61,5,62,14,62,22,62,31,62,38,62,46,62,54,62,62,62,69,62,77,62,85,62,92,62,101,62,108,62,116,62,123,62,130,62,137,62,143,62,151,62,158,62,166,62,172,62,180,62,186,62,192,62,199,62,205,62,213,62,220,62,225,62,232,62,237,62,244,62,250,62,0,63,5,63,12,63,18,63,23,63,30,63,35,63,41,63,46,63,52,63,57,63,63,63,68,63,73,63,78,63,83,63,88,63,92,63,97,63,102,63,107,63,111,63,116,63,119,63,125,63,129,63,134,63,137,63,141,63,146,63,150,63,154,63,157,63,162,63,165,63,169,63,171,63,176,63,179,63,181,63,186,63,189,63,192,63,194,63,198,63,201,63,203,63,207,63,209,63,212,63,214,63,217,63,219,63,221,63,223,63,226,63,229,63,230,63,232,63,233,63,236,63,238,63,238,63,241,63,242,63,243,63,245,63,246,63,248,63,249,63,249,63,251,63,251,63,252,63,253,63,253,63,253,63,254,63,255,63,0,64,0,64,0,64,0,64,0,64,0,64,0,64,0,64,0,64,255,63,254,63,253,63,253,63,253,63,252,63,251,63,251,63,249,63,249,63,248,63,246,63,245,63,243,63,242,63,241,63,238,63,238,63,236,63,233,63,232,63,230,63,229,63,226,63,223,63,221,63,219,63,217,63,214,63,212,63,209,63,207,63,203,63,201,63,198,63,194,63,192,63,189,63,186,63,181,63,179,63,176,63,171,63,169,63,165,63,162,63,157,63,154,63,150,63,146,63,141,63,137,63,134,63,129,63,125,63,119,63,116,63,111,63,107,63,102,63,97,63,92,63,88,63,83,63,78,63,73,63,68,63,63,63,57,63,52,63,46,63,41,63,35,63,30,63,23,63,18,63,12,63,5,63,0,63,250,62,244,62,237,62,232,62,225,62,220,62,213,62,205,62,199,62,192,62,186,62,180,62,172,62,166,62,158,62,151,62,143,62,137,62,130,62,123,62,116,62,108,62,101,62,92,62,85,62,77,62,69,62,62,62,54,62,46,62,38,62,31,62,22,62,14,62,5,62,254,61,245,61,237,61,227,61,220,61,211,61,202,61,194,61,185,61,176,61,167,61,158,61,149,61,140,61,131,61,122,61,113,61,103,61,93,61,84,61,75,61,65,61,56,61,46,61,36,61,26,61,17,61,7,61,253,60,242,60,232,60,222,60,212,60,202,60,192,60,180,60,170,60,160,60,149,60,139,60,129,60,118,60,107,60,96,60,86,60,73,60,63,60,52,60,41,60,30,60,19,60,7,60,252,59,241,59,229,59,217,59,206,59,195,59,183,59,172,59,160,59,148,59,136,59,124,59,112,59,101,59,88,59,76,59,63,59,51,59,39,59,27,59,14,59,2,59,246,58,233,58,219,58,208,58,195,58,183,58,169,58,156,58,144,58,131,58,118,58,105,58,92,58,79,58,65,58,52,58,39,58,26,58,12,58,254,57,241,57,227,57,214,57,201,57,186,57,172,57,160,57,145,57,131,57,117,57,104,57,89,57,76,57,61,57,46,57,33,57,18,57,3,57,246,56,231,56,217,56,202,56,187,56,173,56,158,56,144,56,129,56,114,56,99,56,84,56,70,56,55,56,40,56,25,56,10,56,250,55,236,55,221,55,205,55,190,55,174,55,159,55,143,55,128,55,112,55,97,55,81,55,66,55,50,55,35,55,19,55,2,55,243,54,227,54,211,54,195,54,179,54,162,54,147,54,131,54,115,54,99,54,82,54,66,54,49,54,33,54,16,54,255,53,240,53,223,53,207,53,190,53,173,53,156,53,139,53,123,53,105,53,89,53,73,53,55,53,39,53,21,53,4,53,244,52,226,52,209,52,192,52,174,52,157,52,139,52,123,52,105,52,87,52,71,52,53,52,35,52,18,52,0,52,238,51,221,51,202,51,185,51,168,51,149,51,132,51,114,51,96,51,78,51,60,51,42,51,24,51,6,51,244,50,226,50,208,50,189,50,171,50,153,50,134,50,116,50,98,50,79,50,60,50,42,50,23,50,6,50,243,49,224,49,206,49,187,49,168,49,150,49,131,49,111,49,93,49,74,49,55,49,36,49,18,49,254,48,236,48,216,48,197,48,178,48,160,48,139,48,120,48,102,48,82,48,63,48,44,48,24,48,5,48,241,47,222,47,203,47,183,47,163,47,144,47,125,47,105,47,86,47,65,47,47,47,27,47,7,47,243,46,222,46,204,46,184,46,164,46,144,46,124,46,104,46,84,46,63,46,45,46,25,46,4,46,241,45,221,45,200,45,180,45,160,45,140,45,120,45,99,45,79,45,58,45,39,45,18,45,254,44,234,44,213,44,193,44,172,44,151,44,130,44,110,44,90,44,69,44,49,44,28,44,8,44,243,43,223,43,201,43,181,43,160,43,140,43,119,43,99,43,76,43,56,43,35,43,15,43,250,42,230,42,207,42,187,42,166,42,146,42,124,42,104,42,82,42,62,42,40,42,20,42,254,41,232,41,212,41,191,41,170,41,148,41,127,41,106,41,85,41,64,41,42,41,21,41,0,41,234,40,213,40,191,40,170,40,149,40,128,40,106,40,85,40,63,40,43,40,20,40,255,39,233,39,212,39,191,39,168,39,148,39,126,39,104,39,84,39,61,39,39,39,18,39,253,38,231,38,208,38,187,38,165,38,144,38,123,38,101,38,78,38,56,38,35,38,14,38,248,37,226,37,204,37,181,37,161,37,139,37,117,37,95,37,73,37,51,37,29,37,7,37,242,36,220,36,198,36,176,36,154,36,132,36,110,36,88,36,66,36,44,36,22,36,0,36,234,35,212,35,190,35,168,35,146,35,124,35,102,35,80,35,58,35,36,35,14,35,247,34,225,34,203,34,181,34,159,34,137,34,115,34,93,34,71,34,48,34,26,34,4,34,238,33,216,33,194,33,172,33,149,33,127,33,105,33,83,33,61,33,39,33,16,33,250,32,228,32,206,32,184,32,161,32,139,32,117,32,95,32,73,32,50,32,28,32,6,32,240,31,218,31,195,31,173,31,151,31,129,31,106,31,84,31,62,31,40,31,18,31,251,30,229,30,207,30,185,30,162,30,140,30,118,30,96,30,74,30,51,30,29,30,7,30,241,29,219,29,196,29,174,29,152,29,130,29,107,29,85,29,63,29,41,29,19,29,253,28,230,28,208,28,186,28,164,28,142,28,120,28,97,28,75,28,53,28,31,28,9,28,243,27,220,27,198,27,176,27,154,27,132,27,110,27,88,27,66,27,44,27,22,27,255,26,233,26,211,26,189,26,167,26,145,26,123,26,101,26,79,26,57,26,35,26,13,26,247,25,225,25,203,25,181,25,159,25,137,25,115,25,93,25,72,25,50,25,28,25,6,25,240,24,218,24,196,24,174,24,153,24,131,24,109,24,87,24,65,24,44,24,22,24,0,24,234,23,213,23,191,23,169,23,147,23,126,23,104,23,82,23,61,23,39,23,17,23,252,22,230,22,209,22,187,22,165,22,144,22,122,22,101,22,79,22,58,22,36,22,15,22,249,21,228,21,206,21,185,21,164,21,142,21,121,21,100,21,78,21,57,21,36,21,14,21,249,20,228,20,207,20,185,20,164,20,143,20,122,20,101,20,79,20,58,20,37,20,16,20,251,19,230,19,209,19,188,19,167,19,146,19,125,19,104,19,83,19,62,19,41,19,21,19,0,19,235,18,214,18,193,18,172,18,152,18,131,18,110,18,90,18,69,18,48,18,28,18,7,18,243,17,222,17,201,17,181,17,160,17,140,17,119,17,99,17,79,17,58,17,38,17,18,17,253,16,233,16,213,16,192,16,172,16,152,16,132,16,112,16,92,16,71,16,51,16,31,16,11,16,247,15,227,15,207,15,187,15,167,15,148,15,128,15,108,15,88,15,68,15,49,15,29,15,9,15,245,14,226,14,206,14,187,14,167,14,148,14,128,14,109,14,89,14,70,14,50,14,31,14,11,14,248,13,229,13,210,13,190,13,171,13,152,13,133,13,114,13,95,13,76,13,57,13,38,13,19,13,0,13,237,12,218,12,199,12,180,12,161,12,143,12,124,12,105,12,87,12,68,12,49,12,31,12,12,12,250,11,231,11,213,11,194,11,176,11,158,11,139,11,121,11,103,11,85,11,66,11,48,11,30,11,12,11,250,10,232,10,214,10,196,10,178,10,160,10,142,10,125,10,107,10,89,10,71,10,54,10,36,10,19,10,1,10,239,9,222,9,205,9,187,9,170,9,152,9,135,9,118,9,101,9,83,9,66,9,49,9,32,9,15,9,254,8,237,8,220,8,203,8,186,8,170,8,153,8,136,8,119,8,103,8,86,8,69,8,53,8,36,8,20,8,4,8,243,7,227,7,211,7,194,7,178,7,162,7,146,7,130,7,114,7,98,7,82,7,66,7,50,7,34,7,18,7,2,7,243,6,227,6,211,6,196,6,180,6,165,6,149,6,134,6,118,6,103,6,88,6,72,6,57,6,42,6,27,6,12,6,253,5,238,5,223,5,208,5,193,5,178,5,164,5,149,5,134,5,120,5,105,5,90,5,76,5,62,5,47,5,33,5,18,5,4,5,246,4,232,4,218,4,204,4,190,4,176,4,162,4,148,4,134,4,120,4,107,4,93,4,79,4,66,4,52,4,39,4,25,4,12,4,254,3,241,3,228,3,215,3,202,3,188,3,175,3,162,3,149,3,137,3,124,3,111,3,98,3,86,3,73,3,60,3,48,3,35,3,23,3,10,3,254,2,242,2,230,2,217,2,205,2,193,2,181,2,169,2,157,2,146,2,134,2,122,2,110,2,99,2,87,2,76,2,64,2,53,2,41,2,30,2,19,2,8,2,252,1,241,1,230,1,219,1,208,1,197,1,187,1,176,1,165,1,155,1,144,1,133,1,123,1,113,1,102,1,92,1,82,1,71,1,61,1,51,1,41,1,31,1,21,1,12,1,2,1,248,0,238,0,229,0,219,0,210,0,200,0,191,0,182,0,172,0,163,0,154,0,145,0,136,0,127,0,118,0,109,0,100,0,92,0,83,0,75,0,66,0,58,0,49,0,41,0,32,0,24,0,16,0,8,0,0,0,248,255,240,255,232,255,224,255,216,255,209,255,201,255,193,255,185,255,178,255,170,255,162,255,155,255,147,255,139,255,132,255,124,255,117,255,110,255,102,255,95,255,87,255,80,255,73,255,66,255,58,255,51,255,44,255,37,255,30,255,23,255,16,255,9,255,2,255,251,254,244,254,237,254,230,254,223,254,217,254,210,254,203,254,196,254,190,254,183,254,176,254,170,254,163,254,157,254,150,254,144,254,137,254,131,254,124,254,118,254,112,254,105,254,99,254,93,254,87,254,80,254,74,254,68,254,62,254,56,254,50,254,44,254,38,254,32,254,26,254,20,254,14,254,8,254,2,254,253,253,247,253,241,253,235,253,230,253,224,253,218,253,213,253,207,253,202,253,196,253,191,253,185,253,180,253,174,253,169,253,164,253,158,253,153,253,148,253,142,253,137,253,132,253,127,253,122,253,117,253,112,253,106,253,101,253,96,253,91,253,86,253,82,253,77,253,72,253,67,253,62,253,57,253,53,253,48,253,43,253,38,253,34,253,29,253,24,253,20,253,15,253,11,253,6,253,2,253,253,252,249,252,244,252,240,252,236,252,231,252,227,252,223,252,218,252,214,252,210,252,206,252,202,252,198,252,193,252,189,252,185,252,181,252,177,252,173,252,169,252,165,252,161,252,158,252,154,252,150,252,146,252,142,252,138,252,135,252,131,252,127,252,124,252,120,252,116,252,113,252,109,252,106,252,102,252,99,252,95,252,92,252,88,252,85,252,81,252,78,252,75,252,71,252,68,252,65,252,62,252,58,252,55,252,52,252,49,252,46,252,43,252,40,252,37,252,33,252,30,252,27,252,25,252,22,252,19,252,16,252,13,252,10,252,7,252,4,252,2,252,255,251,252,251,249,251,247,251,244,251,241,251,239,251,236,251,233,251,231,251,228,251,226,251,223,251,221,251,218,251,216,251,214,251,211,251,209,251,206,251,204,251,202,251,200,251,197,251,195,251,193,251,191,251,188,251,186,251,184,251,182,251,180,251,178,251,176,251,174,251,172,251,170,251,168,251,166,251,164,251,162,251,160,251,158,251,156,251,155,251,153,251,151,251,149,251,147,251,146,251,144,251,142,251,141,251,139,251,137,251,136,251,134,251,133,251,131,251,130,251,128,251,127,251,125,251,124,251,122,251,121,251,119,251,118,251,117,251,115,251,114,251,113,251,111,251,110,251,109,251,108,251,106,251,105,251,104,251,103,251,102,251,101,251,100,251,99,251,98,251,96,251,95,251,94,251,93,251,93,251,92,251,91,251,90,251,89,251,88,251,87,251,86,251,85,251,85,251,84,251,83,251,82,251,82,251,81,251,80,251,80,251,79,251,78,251,78,251,77,251,76,251,76,251,75,251,75,251,74,251,74,251,73,251,73,251,72,251,72,251,72,251,71,251,71,251,70,251,70,251,70,251,69,251,69,251,69,251,68,251,68,251,68,251,68,251,68,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,66,251,66,251,66,251,66,251,66,251,66,251,66,251,66,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,67,251,68,251,68,251,68,251,68,251,69,251,69,251,69,251,69,251,70,251,70,251,70,251,71,251,71,251,71,251,72,251,72,251,73,251,73,251,73,251,74,251,74,251,75,251,75,251,76,251,76,251,77,251,78,251,78,251,79,251,79,251,80,251,81,251,81,251,82,251,83,251,83,251,84,251,85,251,85,251,86,251,87,251,88,251,88,251,89,251,90,251,91,251,92,251,93,251,93,251,94,251,95,251,96,251,97,251,98,251,99,251,100,251,101,251,102,251,103,251,104,251,105,251,106,251,107,251,108,251,109,251,110,251,111,251,112,251,113,251,114,251,115,251,117,251,118,251,119,251,120,251,121,251,122,251,124,251,125,251,126,251,127,251,129,251,130,251,131,251,132,251,134,251,135,251,136,251,138,251,139,251,140,251,142,251,143,251,145,251,146,251,147,251,149,251,150,251,152,251,153,251,155,251,156,251,158,251,159,251,161,251,162,251,164,251,165,251,167,251,168,251,170,251,171,251,173,251,175,251,176,251,178,251,179,251,181,251,183,251,184,251,186,251,188,251,189,251,191,251,193,251,195,251,196,251,198,251,200,251,201,251,203,251,205,251,207,251,209,251,210,251,212,251,214,251,216,251,218,251,219,251,221,251,223,251,225,251,227,251,229,251,231,251,233,251,234,251,236,251,238,251,240,251,242,251,244,251,246,251,248,251,250,251,252,251,254,251,0,252,2,252,4,252,6,252,8,252,10,252,12,252,14,252,16,252,18,252,20,252,22,252,25,252,27,252,29,252,31,252,33,252,35,252,37,252,39,252,42,252,44,252,46,252,48,252,50,252,52,252,55,252,57,252,59,252,61,252,63,252,66,252,68,252,70,252,72,252,74,252,77,252,79,252,81,252,83,252,86,252,88,252,90,252,93,252,95,252,97,252,100,252,102,252,104,252,106,252,109,252,111,252,113,252,116,252,118,252,121,252,123,252,125,252,128,252,130,252,132,252,135,252,137,252,140,252,142,252,144,252,147,252,149,252,152,252,154,252,157,252,159,252,161,252,164,252,166,252,169,252,171,252,174,252,176,252,179,252,181,252,184,252,186,252,189,252,191,252,194,252,196,252,199,252,201,252,204,252,206,252,209,252,211,252,214,252,216,252,219,252,221,252,224,252,226,252,229,252,231,252,234,252,237,252,239,252,242,252,244,252,247,252,249,252,252,252,255,252,1,253,4,253,6,253,9,253,11,253,14,253,17,253,19,253,22,253,24,253,27,253,30,253,32,253,35,253,38,253,40,253,43,253,45,253,48,253,51,253,53,253,56,253,59,253,61,253,64,253,66,253,69,253,72,253,74,253,77,253,80,253,82,253,85,253,88,253,90,253,93,253,96,253,98,253,101,253,103,253,106,253,109,253,111,253,114,253,117,253,119,253,122,253,125,253,127,253,130,253,133,253,135,253,138,253,141,253,143,253,146,253,149,253,151,253,154,253,157,253,159,253,162,253,165,253,167,253,170,253,173,253,175,253,178,253,181,253,183,253,186,253,189,253,191,253,194,253,197,253,199,253,202,253,205,253,207,253,210,253,213,253,215,253,218,253,221,253,223,253,226,253,229,253,231,253,234,253,237,253,239,253,242,253,245,253,247,253,250,253,252,253,255,253,2,254,4,254,7,254,10,254,12,254,15,254,17,254,20,254,23,254,25,254,28,254,31,254,33,254,36,254,38,254,41,254,44,254,46,254,49,254,51,254,54,254,57,254,59,254,62,254,64,254,67,254,70,254,72,254,75,254,77,254,80,254,82,254,85,254,88,254,90,254,93,254,95,254,98,254,100,254,103,254,105,254,108,254,110,254,113,254,115,254,118,254,120,254,123,254,125,254,128,254,130,254,133,254,135,254,138,254,140,254,143,254,145,254,148,254,150,254,153,254,155,254,158,254,160,254,163,254,165,254,167,254,170,254,172,254,175,254,177,254,180,254,182,254,184,254,187,254,189,254,192,254,194,254,196,254,199,254,201,254,203,254,206,254,208,254,210,254,213,254,215,254,217,254,220,254,222,254,224,254,227,254,229,254,231,254,234,254,236,254,238,254,240,254,243,254,245,254,247,254,249,254,252,254,254,254,0,255,2,255,5,255,7,255,9,255,11,255,13,255,16,255,18,255,20,255,22,255,24,255,26,255,28,255,31,255,33,255,35,255,37,255,39,255,41,255,43,255,45,255,47,255,49,255,51,255,54,255,56,255,58,255,60,255,62,255,64,255,66,255,68,255,70,255,72,255,74,255,76,255,78,255,80,255,81,255,83,255,85,255,87,255,89,255,91,255,93,255,95,255,97,255,99,255,100,255,102,255,104,255,106,255,108,255,110,255,111,255,113,255,115,255,117,255,119,255,120,255,122,255,124,255,126,255,127,255,129,255,131,255,132,255,134,255,136,255,137,255,139,255,141,255,142,255,144,255,146,255,147,255,149,255,150,255,152,255,154,255,155,255,157,255,158,255,160,255,161,255,163,255,164,255,166,255,167,255,169,255,170,255,172,255,173,255,174,255,176,255,177,255,179,255,180,255,181,255,183,255,184,255,186,255,187,255,188,255,189,255,191,255,192,255,193,255,195,255,196,255,197,255,198,255,199,255,201,255,202,255,203,255,204,255,205,255,207,255,208,255,209,255,210,255,211,255,212,255,213,255,214,255,215,255,216,255,217,255,218,255,219,255,220,255,221,255,222,255,223,255,224,255,225,255,226,255,227,255,228,255,229,255,230,255,230,255,231,255,232,255,233,255,234,255,234,255,235,255,236,255,237,255,237,255,238,255,239,255,240,255,240,255,241,255,242,255,242,255,243,255,243,255,244,255,245,255,245,255,246,255,246,255,247,255,247,255,248,255,248,255,249,255,249,255,250,255,250,255,250,255,251,255,251,255,252,255,252,255,252,255,253,255,253,255,253,255,254,255,254,255,254,255,254,255,254,255,255,255,255,255,255,255,255,255,255,255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,150,48,7,119,44,97,14,238,186,81,9,153,25,196,109,7,143,244,106,112,53,165,99,233,163,149,100,158,50,136,219,14,164,184,220,121,30,233,213,224,136,217,210,151,43,76,182,9,189,124,177,126,7,45,184,231,145,29,191,144,100,16,183,29,242,32,176,106,72,113,185,243,222,65,190,132,125,212,218,26,235,228,221,109,81,181,212,244,199,133,211,131,86,152,108,19,192,168,107,100,122,249,98,253,236,201,101,138,79,92,1,20,217,108,6,99,99,61,15,250,245,13,8,141,200,32,110,59,94,16,105,76,228,65,96,213,114,113,103,162,209,228,3,60,71,212,4,75,253,133,13,210,107,181,10,165,250,168,181,53,108,152,178,66,214,201,187,219,64,249,188,172,227,108,216,50,117,92,223,69,207,13,214,220,89,61,209,171,172,48,217,38,58,0,222,81,128,81,215,200,22,97,208,191,181,244,180,33,35,196,179,86,153,149,186,207,15,165,189,184,158,184,2,40,8,136,5,95,178,217,12,198,36,233,11,177,135,124,111,47,17,76,104,88,171,29,97,193,61,45,102,182,144,65,220,118,6,113,219,1,188,32,210,152,42,16,213,239,137,133,177,113,31,181,182,6,165,228,191,159,51,212,184,232,162,201,7,120,52,249,0,15,142,168,9,150,24,152,14,225,187,13,106,127,45,61,109,8,151,108,100,145,1,92,99,230,244,81,107,107,98,97,108,28,216,48,101,133,78,0,98,242,237,149,6,108,123,165,1,27,193,244,8,130,87,196,15,245,198,217,176,101,80,233,183,18,234,184,190,139,124,136,185,252,223,29,221,98,73,45,218,21,243,124,211,140,101,76,212,251,88,97,178,77,206,81,181,58,116,0,188,163,226,48,187,212,65,165,223,74,215,149,216,61,109,196,209,164,251,244,214,211,106,233,105,67,252,217,110,52,70,136,103,173,208,184,96,218,115,45,4,68,229,29,3,51,95,76,10,170,201,124,13,221,60,113,5,80,170,65,2,39,16,16,11,190,134,32,12,201,37,181,104,87,179,133,111,32,9,212,102,185,159,228,97,206,14,249,222,94,152,201,217,41,34,152,208,176,180,168,215,199,23,61,179,89,129,13,180,46,59,92,189,183,173,108,186,192,32,131,184,237,182,179,191,154,12,226,182,3,154,210,177,116,57,71,213,234,175,119,210,157,21,38,219,4,131,22,220,115,18,11,99,227,132,59,100,148,62,106,109,13,168,90,106,122,11,207,14,228,157,255,9,147,39,174,0,10,177,158,7,125,68,147,15,240,210,163,8,135,104,242,1,30,254,194,6,105,93,87,98,247,203,103,101,128,113,54,108,25,231,6,107,110,118,27,212,254,224,43,211,137,90,122,218,16,204,74,221,103,111,223,185,249,249,239,190,142,67,190,183,23,213,142,176,96,232,163,214,214,126,147,209,161,196,194,216,56,82,242,223,79,241,103,187,209,103,87,188,166,221,6,181,63,75,54,178,72,218,43,13,216,76,27,10,175,246,74,3,54,96,122,4,65,195,239,96,223,85,223,103,168,239,142,110,49,121,190,105,70,140,179,97,203,26,131,102,188,160,210,111,37,54,226,104,82,149,119,12,204,3,71,11,187,185,22,2,34,47,38,5,85,190,59,186,197,40,11,189,178,146,90,180,43,4,106,179,92,167,255,215,194,49,207,208,181,139,158,217,44,29,174,222,91,176,194,100,155,38,242,99,236,156,163,106,117,10,147,109,2,169,6,9,156,63,54,14,235])
.concat([133,103,7,114,19,87,0,5,130,74,191,149,20,122,184,226,174,43,177,123,56,27,182,12,155,142,210,146,13,190,213,229,183,239,220,124,33,223,219,11,212,210,211,134,66,226,212,241,248,179,221,104,110,131,218,31,205,22,190,129,91,38,185,246,225,119,176,111,119,71,183,24,230,90,8,136,112,106,15,255,202,59,6,102,92,11,1,17,255,158,101,143,105,174,98,248,211,255,107,97,69,207,108,22,120,226,10,160,238,210,13,215,84,131,4,78,194,179,3,57,97,38,103,167,247,22,96,208,77,71,105,73,219,119,110,62,74,106,209,174,220,90,214,217,102,11,223,64,240,59,216,55,83,174,188,169,197,158,187,222,127,207,178,71,233,255,181,48,28,242,189,189,138,194,186,202,48,147,179,83,166,163,180,36,5,54,208,186,147,6,215,205,41,87,222,84,191,103,217,35,46,122,102,179,184,74,97,196,2,27,104,93,148,43,111,42,55,190,11,180,161,142,12,195,27,223,5,90,141,239,2,45,200,89,0,0,44,0,0,0,64,0,0,0,0,0,0,0,1,0,0,0,3,0,0,0,7,0,0,0,15,0,0,0,30,0,0,0,60,0,0,0,120,0,0,0,248,0,0,0,1,0,0,0,3,0,0,0,7,0,0,0,15,0,0,0,30,0,0,0,60,0,0,0,120,0,0,0,240,0,0,0,240,1,0,0,240,3,0,0,240,7,0,0,240,15,0,0,240,31,0,0,240,63,0,0,240,127,0,0,240,255,0,0,0,4,4,4,8,8,8,16,16,16,32,32,32,64,64,64,224,90,0,0,6,1,0,0,58,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,2,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,3,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,4,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,5,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,6,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,7,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,8,0,0,0,9,0,0,0,9,0,0,0,9,0,0,0,9,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,10,0,0,0,11,0,0,0,11,0,0,0,11,0,0,0,11,0,0,0,12,0,0,0,12,0,0,0,12,0,0,0,12,0,0,0,13,0,0,0,13,0,0,0,13,0,0,0,13,0,0,0,14,0,0,0,14,0,0,0,14,0,0,0,14,0,0,0,15,0,0,0,15,0,0,0,16,0,0,0,16,0,0,0,17,0,0,0,17,0,0,0,18,0,0,0,18,0,0,0,19,0,0,0,19,0,0,0,20,0,0,0,20,0,0,0,21,0,0,0,21,0,0,0,22,0,0,0,22,0,0,0,23,0,0,0,23,0,0,0,24,0,0,0,24,0,0,0,25,0,0,0,25,0,0,0,26,0,0,0,26,0,0,0,27,0,0,0,27,0,0,0,28,0,0,0,28,0,0,0,29,0,0,0,29,0,0,0,30,0,0,0,30,0,0,0,31,0,0,0,32,0,0,0,33,0,0,0,34,0,0,0,35,0,0,0,36,0,0,0,37,0,0,0,38,0,0,0,39,0,0,0,40,0,0,0,41,0,0,0,42,0,0,0,43,0,0,0,44,0,0,0,45,0,0,0,46,0,0,0,47,0,0,0,48,0,0,0,49,0,0,0,50,0,0,0,51,0,0,0,52,0,0,0,53,0,0,0,54,0,0,0,55,0,0,0,56,0,0,0,57,0,0,0,58,0,0,0,59,0,0,0,60,0,0,0,61,0,0,0,62,0,0,0,64,0,0,0,65,0,0,0,67,0,0,0,69,0,0,0,71,0,0,0,73,0,0,0,75,0,0,0,77,0,0,0,79,0,0,0,81,0,0,0,83,0,0,0,85,0,0,0,87,0,0,0,89,0,0,0,91,0,0,0,93,0,0,0,96,0,0,0,98,0,0,0,100,0,0,0,102,0,0,0,104,0,0,0,106,0,0,0,108,0,0,0,110,0,0,0,112,0,0,0,114,0,0,0,116,0,0,0,118,0,0,0,120,0,0,0,122,0,0,0,124,0,0,0,126,0,0,0,129,0,0,0,131,0,0,0,135,0,0,0,139,0,0,0,143,0,0,0,147,0,0,0,151,0,0,0,155,0,0,0,160,0,0,0,164,0,0,0,168,0,0,0,172,0,0,0,176,0,0,0,180,0,0,0,184,0,0,0,188,0,0,0,193,0,0,0,197,0,0,0,201,0,0,0,205,0,0,0,209,0,0,0,213,0,0,0,217,0,0,0,221,0,0,0,226,0,0,0,230,0,0,0,234,0,0,0,238,0,0,0,242,0,0,0,246,0,0,0,250,0,0,0,255,0,0,0,255,0,0,0,0,0,0,0,136,93,0,0,222,0,0,0,208,0,0,0,0,0,0,0,144,94,0,0,28,1,0,0,86,0,0,0,0,0,0,0,152,96,0,0,136,0,0,0,42,0,0,0,0,0,0,0,0,0,0,0,5,0,0,0,1,0,0,0,6,0,0,0,2,0,0,0,7,0,0,0,3,0,0,0,8,0,0,0,4,0,0,0,9,0,0,0,10,0,0,0,0,0,0,0,0,25,49,71,90,106,117,125,127,125,117,106,90,71,49,25,0,231,207,185,166,150,139,131,129,131,139,150,166,185,207,231,128,136,144,152,160,168,176,184,192,200,208,216,224,232,240,248,0,8,16,24,32,40,48,56,64,72,80,88,96,104,112,120,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,88,99,0,0,12,0,0,0,10,0,0,0,0,0,0,0,115,114,99,47,109,105,120,101,114,46,99,0,0,0,0,0,97,118,97,105,108,97,98,108,101,91,121,93,32,61,61,32,48,0,0,0,0,0,0,0,117,110,108,122,120,58,32,117,110,107,110,111,119,110,32,34,37,115,34,10,0,0,0,0,80,114,111,80,97,99,107,101,114,32,50,46,49,0,0,0,80,114,111,109,105,122,101,114,32,49,46,56,97,0,0,0,83,69,81,85,0,0,0,0,80,114,111,109,105,122,101,114,32,49,46,48,99,0,0,0,80,104,97,32,80,97,99,107,101,114,0,0,0,0,0,0,84,104,101,32,80,108,97,121,101,114,32,54,46,49,97,0,84,104,101,32,80,108,97,121,101,114,32,54,46,48,97,0,84,104,101,32,80,108,97,121,101,114,32,53,46,48,97,0,68,105,103,105,66,111,111,115,116,101,114,32,80,114,111,32,40,68,66,77,41,0,0,0,84,104,101,32,80,108,97,121,101,114,32,52,46,120,0,0,78,111,105,115,101,112,97,99,107,101,114,32,118,51,0,0,117,110,107,110,111,119,110,32,40,37,48,52,120,41,0,0,78,111,105,115,101,112,97,99,107,101,114,32,118,50,0,0,78,111,105,115,101,80,97,99,107,101,114,32,118,49,0,0,78,111,118,111,116,114,97,100,101,32,80,97,99,107,101,114,0,0,0,0,0,0,0,0,79,75,84,65,83,79,78,71,0,0,0,0,0,0,0,0,78,111,105,115,101,82,117,110,110,101,114,0,0,0,0,0,67,68,54,49,0,0,0,0,77,111,100,117,108,101,32,80,114,111,116,101,99,116,111,114,0,0,0,0,0,0,0,0,73,83,0,0,0,0,0,0,75,101,102,114,101,110,115,32,83,111,117,110,100,32,77,97,99,104,105,110,101,0,0,0,112,37,100,32,114,37,100,32,99,37,100,58,32,117,110,107,110,111,119,110,32,101,102,102,101,99,116,32,37,48,50,120,32,37,48,50,120,10,0,0,98,91,48,93,32,61,61,32,39,76,39,32,38,38,32,98,91,49,93,32,61,61,32,39,68,39,32,38,38,32,98,91,50,93,32,61,61,32,39,83,39,32,38,38,32,98,91,51,93,32,61,61,32,39,83,39,0,0,0,0,0,0,0,0,67,104,105,112,84,114,97,99,107,101,114,0,0,0,0,0,42,46,73,78,70,79,0,0,79,112,101,110,77,80,84,32,49,46,49,55,43,0,0,0,72,111,114,110,101,116,32,80,97,99,107,101,114,0,0,0,72,101,97,116,115,101,101,107,101,114,32,49,46,48,0,0,71,97,109,101,32,77,117,115,105,99,32,67,114,101,97,116,111,114,0,0,0,0,0,0,83,84,49,46,51,32,77,111,100,117,108,101,73,78,70,79,0,0,0,0,0,0,0,0,70,117,122,122,97,99,32,80,97,99,107,101,114,0,0,0,70,117,99,104,115,32,84,114,97,99,107,101,114,0,0,0,70,67,45,77,32,80,97,99,107,101,114,0,0,0,0,0,69,117,114,101,107,97,32,80,97,99,107,101,114,0,0,0,68,105,103,105,116,97,108,32,73,108,108,117,115,105,111,110,115,0,0,0,0,0,0,0,65,67,49,68,32,80,97,99,107,101,114,0,0,0,0,0,122,105,82,67,79,78,105,97,0,0,0,0,0,0,0,0,48,0,0,0,0,0,0,0,70,82,101,97,100,40,65,114,99,104,105,118,101,95,72,101,97,100,101,114,41,0,0,0,88,77,80,95,73,78,83,84,82,85,77,69,78,84,95,80,65,84,72,0,0,0,0,0,80,76,69,78,0,0,0,0,69,120,116,101,110,100,101,100,32,77,111,100,117,108,101,58,32,0,0,0,0,0,0,0,89,90,0,0,0,0,0,0,66,101,82,111,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,0,0,0,0,83,66,79,68,0,0,0,0,67,111,99,111,110,105,122,101,114,0,0,0,0,0,0,0,56,67,72,78,0,0,0,0,115,114,99,47,100,101,112,97,99,107,101,114,115,47,118,111,114,98,105,115,46,99,0,0,80,65,0,0,0,0,0,0,112,37,100,32,114,37,100,32,99,37,100,58,32,99,111,109,112,114,101,115,115,101,100,32,101,118,101,110,116,32,37,48,50,120,32,37,48,50,120,10,0,0,0,0,0,0,0,0,101,118,101,110,116,45,62,118,111,108,32,60,61,32,54,53,0,0,0,0,0,0,0,0,108,105,98,120,109,112,58,32,111,117,116,112,117,116,32,101,114,114,111,114,32,105,110,32,108,122,100,40,41,46,10,0,42,46,100,111,99,0,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,83,116,97,114,116,114,101,107,107,101,114,32,49,46,50,0,105,98,117,102,32,33,61,32,78,85,76,76,0,0,0,0,70,83,101,101,107,40,68,97,116,97,41,0,0,0,0,0,65,114,99,104,105,118,101,0,0,0,0,0,0,0,0,0,111,108,100,99,111,100,101,58,45,49,32,99,111,100,101,58,37,105,10,0,0,0,0,0,65,115,121,108,117,109,32,77,117,115,105,99,32,70,111,114,109,97,116,32,40,65,77,70,41,0,0,0,0,0,0,0,70,114,111,109,63,0,0,0,99,45,62,115,111,114,116,101,100,95,101,110,116,114,105,101,115,32,61,61,32,48,0,0,69,79,70,58,32,65,114,99,104,105,118,101,95,72,101,97,100,101,114,10,0,0,0,0,115,114,99,47,100,101,112,97,99,107,101,114,115,47,115,52,48,52,95,100,101,99,46,99,0,0,0,0,0,0,0,0,110,111,109,97,114,99,104,58,32,111,117,116,32,111,102,32,109,101,109,111,114,121,33,10,0,0,0,0,0,0,0,0,80,78,85,77,0,0,0,0,114,101,97,100,95,108,122,119,95,100,121,110,97,109,105,99,0,0,0,0,0,0,0,0,110,111,109,97,114,99,104,58,32,111,117,116,32,111,102,32,109,101,109,111,114,121,33,10,0,0,0,0,0,0,0,0,37,115,32,88,77,32,37,100,46,37,48,50,100,0,0,0,69,120,116,101,110,100,101,100,32,77,111,100,117,108,101,58,0,0,0,0,0,0,0,0,83,99,104,105,115,109,32,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,0,85,110,115,117,112,112,111,114,116,101,100,32,99,111,109,112,114,101,115,115,105,111,110,32,117,115,101,100,46,10,0,0,80,66,79,68,0,0,0,0,70,97,115,116,32,84,114,97,99,107,101,114,0,0,0,0,65,114,99,70,83,58,32,101,114,114,111,114,32,114,101,97,100,105,110,103,32,100,97,116,97,32,40,104,105,116,32,69,79,70,41,10,0,0,0,0,73,73,0,0,0,0,0,0,69,112,105,99,32,77,101,103,97,71,97,109,101,115,32,77,65,83,73,32,80,83,77,0,101,118,101,110,116,45,62,105,110,115,32,60,61,32,49,48,48,0,0,0,0,0,0,0,67,104,105,98,105,32,84,114,97,99,107,101,114,0,0,0,42,46,68,111,99,0,0,0,83,84,49,46,50,32,77,111,100,117,108,101,73,78,70,79,0,0,0,0,0,0,0,0,68,105,103,105,116,97,108,32,84,114,97,99,107,101,114,32,68,84,77,0,0,0,0,0,83,81,83,72,0,0,0,0,115,98,117,102,32,33,61,32,78,85,76,76,0,0,0,0,68,105,103,105,66,111,111,115,116,101,114,32,80,114,111,32,37,100,46,37,48,50,120,32,68,66,77,48,0,0,0,0,108,101,110,32,33,61,32,78,79,95,67,79,68,69,0,0,70,82,101,97,100,40,72,101,97,100,101,114,95,70,105,108,101,110,97,109,101,41,0,0,77,76,69,78,0,0,0,0,111,108,100,32,77,111,100,80,108,117,103,32,84,114,97,99,107,101,114,0,0,0,0,0,77,65,83,95,85,84,114,97,99,107,95,86,48,48,48,0,114,98,0,0,0,0,0,0,65,76,69,89,32,77,79,0,79,112,101,110,77,80,84,32,37,100,46,37,48,50,120,0,65,114,99,104,105,109,101,100,101,115,32,84,114,97,99,107,101,114,0,0,0,0,0,0,80,65,84,84,0,0,0,0,54,67,72,78,0,0,0,0,108,105,98,120,109,112,58,32,99,97,110,39,116,32,111,112,101,110,32,115,97,109,112,108,101,32,102,105,108,101,32,37,115,10,0,0,0,0,0,0,70,69,0,0,0,0,0,0,83,105,110,97,114,105,97,32,80,83,77,0,0,0,0,0,101,118,101,110,116,45,62,110,111,116,101,32,60,61,32,49,49,57,32,124,124,32,101,118,101,110,116,45,62,110,111,116,101,32,61,61,32,88,77,80,95,75,69,89,95,79,70,70,0,0,0,0,0,0,0,0,84,77,80,68,73,82,0,0,67,72,66,73,0,0,0,0,42,46,68,79,67,0,0,0,71,97,108,97,120,121,32,77,117,115,105,99,32,83,121,115,116,101,109,32,53,46,48,0,83,116,97,114,116,114,101,107,107,101,114,0,0,0,0,0,68,65,73,84,0,0,0,0,115,114,99,47,108,111,97,100,101,114,115,47,100,109,102,95,108,111,97,100,46,99,0,0,86,69,78,86,0,0,0,0,68,83,77,73,32,65,100,118,97,110,99,101,100,32,77,111,100,117,108,101,32,70,111,114,109,97,116,32,40,65,77,70,41,0,0,0,0,0,0,0,70,97,115,116,32,84,114,97,99,107,101,114,32,73,73,32,40,88,77,41,0,0,0,0,69,112,105,99,32,71,97,109,101,115,32,85,77,88,0,0,88,80,75,70,0,0,0,0,85,108,116,114,97,32,84,114,97,99,107,101,114,32,40,85,76,84,41,0,0,0,0,0,84,67,66,32,84,114,97,99,107,101,114,0,0,0,0,0,99,45,62,115,111,114,116,101,100,95,99,111,100,101,119,111,114,100,115,91,120,93,32,61,61,32,99,111,100,101,0,0,68,105,103,105,116,97,108,32,83,121,109,112,104,111,110,121,0,0,0,0,0,0,0,0,69,79,70,58,32,72,101,97,100,101,114,95,70,105,108,101,110,97,109,101,10,0,0,0,110,111,109,97,114,99,104,58,32,111,117,116,32,111,102,32,109,101,109,111,114,121,33,10,0,0,0,0,0,0,0,0,83,52,48,52,0,0,0,0,117,110,122,105,112,58,32,100,101,99,111,109,112,114,101,115,115,105,111,110,32,101,114,114,111,114,10,0,0,0,0,0,83,84,77,73,75,32,48,46,50,32,40,83,84,88,41,0,65,78,65,77,0,0,0,0,83,99,114,101,97,109,32,84,114,97,99,107,101,114,32,50,32,40,83,84,77,41,0,0,83,108,97,109,116,105,108,116,0,0,0,0,0,0,0,0,70,97,115,116,84,114,97,99,107,101,114,32,118,32,50,46,48,48,0,0,0,0,0,0,85,108,116,114,97,32,84,114,97,99,107,101,114,32,37,115,32,85,76,84,32,86,37,48,52,100,0,0,0,0,0,0,83,67,82,77,0,0,0,0,90,88,32,83,112,101,99,116,114,117,109,32,83,111,117,110,100,32,84,114,97,99,107,101,114,32,40,83,84,67,41,0,37,115,0,0,0,0,0,0,47,0,0,0,0,0,0,0,114,98,0,0,0,0,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,82,84,83,77,0,0,0,0,83,111,117,110,100,116,114,97,99,107,101,114,32,40,77,79,68,41,0,0,0,0,0,0,65,109,117,115,105,99,32,65,100,108,105,98,32,84,114,97,99,107,101,114,32,40,65,77,68,41,0,0,0,0,0,0,80,76,69,78,0,0,0,0,83,111,117,110,100,115,109,105,116,104,47,77,101,103,97,84,114,97,99,107,101,114,32,40,77,84,80,41,0,0,0,0,78,46,84,46,0,0,0,0,77,77,68,49,0,0,0,0,83,111,117,110,100,70,88,0,114,98,0,0,0,0,0,0,65,68,80,67,77,0,0,0,80,69,0,0,0,0,0,0,80,66,79,68,0,0,0,0,101,118,101,110,116,45,62,102,120,116,32,60,61,32,50,54,0,0,0,0,0,0,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,50,46,49,52,118,53,0,0,42,46,110,102,111,0,0,0,117,110,107,110,111,119,110,32,101,102,102,101,99,116,32,37,48,50,120,32,37,48,50,120,10,0,0,0,0,0,0,0,117,110,107,110,111,119,110,32,101,102,102,101,99,116,32,37,48,50,120,32,37,48,50,120,10,0,0,0,0,0,0,0,83,99,114,101,97,109,32,84,114,97,99,107,101,114,32,51,32,40,83,51,77,41,0,0,37,115,37,115,46,97,115,0,68,65,80,84,0,0,0,0,83,77,80,68,0,0,0,0,83,77,80,76,0,0,0,0,82,101,97,108,32,84,114,97,99,107,101,114,32,40,82,84,77,41,0,0,0,0,0,0,82,101,97,108,105,116,121,32,65,100,108,105,98,32,84,114,97,99,107,101,114,32,40,82,65,68,41,0,0,0,0,0,80,80,50,48,0,0,0,0,112,114,111,119,105,122,97,114,100,0,0,0,0,0,0,0,80,111,108,121,32,84,114,97,99,107,101,114,32,40,80,84,77,41,0,0,0,0,0,0,107,32,61,61,32,99,45,62,115,111,114,116,101,100,95,101,110,116,114,105,101,115,0,0,80,114,111,116,114,97,99,107,101,114,32,51,0,0,0,0,70,82,101,97,100,40,72,101,97,100,101,114,95,67,111,109,109,101,110,116,41,0,0,0,101,114,114,111,114,44,32,37,115,10,0,0,0,0,0,0,80,114,111,116,114,97,99,107,101,114,32,83,116,117,100,105,111,32,40,80,83,77,41,0,40,117,105,110,116,56,32,42,41,98,115,45,62,115,114,99,32,62,61,32,98,115,45,62,111,114,103,115,114,99,0,0,90,101,110,32,80,97,99,107,101,114,0,0,0,0,0,0,117,110,122,105,112,58,32,101,114,114,111,114,32,105,110,32,98,105,116,115,116,114,101,97,109,32,114,101,97,100,105,110,103,32,105,110,32,108,105,116,101,114,97,108,32,99,111,100,101,32,108,101,110,103,116,104,32,37,100,10,0,0,0,0,88,65,78,78,32,80,97,99,107,101,114,0,0,0,0,0,77,78,65,77,0,0,0,0,87,97,110,116,111,110,32,80,97,99,107,101,114,0,0,0,85,110,105,99,32,84,114,97,99,107,101,114,32,50,0,0,77,69,68,50,88,77,32,98,121,32,74,46,80,121,110,110,111,110,101,0,0,0,0,0,65,108,101,121,32,75,101,112,116,114,32,40,65,76,77,41,0,0,0,0,0,0,0,0,49,46,54,0,0,0,0,0,33,83,99,114,101,97,109,33,0,0,0,0,0,0,0,0,85,78,73,67,32,84,114,97,99,107,101,114,0,0,0,0,33,83,99,114,101,97,109,33,0,0,0,0,0,0,0,0,117,110,107,110,111,119,110,32,116,114,97,99,107,101,114,0,73,73,103,115,32,77,101,103,97,84,114,97,99,107,101,114,0,0,0,0,0,0,0,0,84,114,97,99,107,101,114,32,80,97,99,107,101,114,32,118,51,0,0,0,0,0,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,50,46,49,52,118,53,0,0,37,115,46,37,100,0,0,0,82,84,73,78,0,0,0,0,84,105,116,97,110,105,99,115,32,80,108,97,121,101,114,0,37,115,0,0,0,0,0,0,80,84,68,84,0,0,0,0,84,104,101,32,68,97,114,107,32,68,101,109,111,110,0,0,83,116,97,114,116,114,101,107,107,101,114,32,80,97,99,107,101,114,0,0,0,0,0,0,83,75,89,84,32,80,97,99,107,101,114,0,0,0,0,0,83,76,69,78,0,0,0,0,78,111,105,115,101,116,114,97,99,107,101,114,0,0,0,0,77,77,68,51,0,0,0,0,80,114,111,114,117,110,110,101,114,32,50,46,48,0,0,0,77,77,68,48,0,0,0,0,108,105,98,120,109,112,58,32,109,105,115,115,105,110,103,32,102,105,108,101,32,37,115,10,0,0,0,0,0,0,0,0,80,114,111,114,117,110,110,101,114,32,49,46,48,0,0,0,116,111,0,0,0,0,0,0,86,69,0,0,0,0,0,0,114,98,0,0,0,0,0,0,68,83,77,80,0,0,0,0,102,116,101,108,108,32,40,102,41,32,45,32,99,111,117,110,116,32,61,61,32,108,112,46,115,105,122,101,0,0,0,0,80,111,108,108,121,32,84,114,97,99,107,101,114,0,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,50,46,49,52,118,51,0,0,42,46,78,70,79,0,0,0,40,99,32,38,32,48,120,49,102,41,32,60,32,109,111,100,45,62,99,104,110,0,0,0,73,78,83,84,0,0,0,0,73,78,83,84,0,0,0,0,37,115,37,115,46,65,83,0,81,117,97,100,114,97,32,67,111,109,112,111,115,101,114,32,69,77,79,68,32,118,37,100,0,0,0,0,0,0,0,0,73,78,83,84,0,0,0,0,83,77,80,73,0,0,0,0,80,65,84,84,0,0,0,0,67,111,109,112,111,115,101,114,32,54,54,57,0,0,0,0,79,107,116,97,108,121,122,101,114,0,0,0,0,0,0,0,76,105,113,117,105,100,32,84,114,97,99,107,101,114,32,78,79,32,40,76,73,81,41,0,117,110,109,111,51,32,45,115,32,34,37,115,34,32,83,84,68,79,85,84,0,0,0,0,77,117,108,116,105,116,114,97,99,107,101,114,32,40,77,84,77,41,0,0,0,0,0,0,70,97,105,108,32,51,32,111,110,32,109,32,61,32,37,100,0,0,0,0,0,0,0,0,40,105,110,116,41,32,102,108,111,111,114,40,112,111,119,40,40,102,108,111,97,116,41,32,114,44,32,100,105,109,41,41,32,60,61,32,101,110,116,114,105,101,115,0,0,0,0,0,108,105,98,120,109,112,58,32,73,47,79,32,101,114,114,111,114,32,105,110,32,108,122,100,58,114,100,95,100,99,111,100,101,46,10,0,0,0,0,0,69,79,70,58,32,72,101,97,100,101,114,95,67,111,109,109,101,110,116,10,0,0,0,0,114,98,0,0,0,0,0,0,117,110,115,117,112,112,111,114,116,101,100,32,99,111,109,112,114,101,115,115,105,111,110,32,109,101,116,104,111,100,32,37,100,10,0,0,0,0,0,0,40,100,115,116,32,43,32,119,32,43,32,49,41,32,60,32,40,111,114,103,100,115,116,32,43,32,100,115,116,95,108,101,110,103,116,104,41,0,0,0,77,46,75,46,0,0,0,0,69,114,114,111,114,32,40,105,110,102,108,97,116,101,41,58,32,117,110,107,110,111,119,110,32,99,111,109,112,114,101,115,115,105,111,110,32,116,121,112,101,10,0,0,0,0,0,0,83,84,69,82,0,0,0,0,37,115,37,115,0,0,0,0,79,99,116,97,77,69,68,32,40,77,69,68,41,0,0,0,68,105,103,105,116,114,97,107,107,101,114,0,0,0,0,0,82,111,82,0,0,0,0,0,37,115,32,40,37,115,41,0,49,46,53,0,0,0,0,0,65,78,32,67,79,79,76,33,0,0,0,0,0,0,0,0,83,84,77,50,83,84,88,32,49,46,37,100,0,0,0,0,83,99,114,101,97,109,32,84,114,97,99,107,101,114,32,37,100,46,37,48,50,100,32,83,84,77,0,0,0,0,0,0,77,69,68,32,50,46,49,48,47,79,99,116,97,77,69,68,32,40,77,69,68,41,0,0,68,46,79,46,67,32,83,111,117,110,100,116,114,97,99,107,101,114,32,50,46,48,0,0,73,65,78,57,50,97,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,50,46,49,52,118,51,0,0,77,101,103,97,116,114,97,99,107,101,114,32,40,77,71,84,41,0,0,0,0,0,0,0,82,84,78,68,0,0,0,0,65,108,101,121,39,115,32,77,111,100,117,108,101,0,0,0,77,46,75,46,0,0,0,0,117,110,107,110,111,119,110,47,99,111,110,118,101,114,116,101,100,0,0,0,0,0,0,0,67,77,78,84,0,0,0,0,77,97,103,110,101,116,105,99,32,70,105,101,108,100,115,32,80,97,99,107,101,114,0,0,83,80,69,69,0,0,0,0,99,111,110,118,101,114,116,101,100,32,49,53,32,105,110,115,116,114,117,109,101,110,116,0,77,38,75,33,0,0,0,0,77,69,68,32,50,46,49,48,32,77,69,68,52,32,40,77,69,68,41,0,0,0,0,0,77,77,68,50,0,0,0,0,79,99,116,97,77,69,68,32,52,46,48,48,32,77,77,68,49,0,0,0,0,0,0,0,46,115,101,116,0,0,0,0,99,111,110,118,0,0,0,0,114,98,0,0,0,0,0,0,83,65,0,0,0,0,0,0,83,79,78,71,0,0,0,0,70,97,115,116,84,114,97,99,107,101,114,32,49,46,48,49,63,0,0,0,0,0,0,0,77,69,68,32,50,46,48,48,32,77,69,68,51,32,40,77,69,68,41,0,0,0,0,0,112,109,97,103,32,61,61,32,48,120,52,99,53,48,48,48,48,48,0,0,0,0,0,0,77,111,100,80,108,117,103,32,84,114,97,99,107,101,114,32,49,46,49,54,0,0,0,0,77,69,68,32,49,46,49,50,32,77,69,68,50,32,40,77,69,68,41,0,0,0,0,0,42,46,100,105,122,0,0,0,115,114,99,47,108,111,97,100,101,114,115,47,103,100,109,95,108,111,97,100,46,99,0,0,80,114,111,98,97,98,108,121,32,99,111,110,118,101,114,116,101,100,0,0,0,0,0,0,80,65,84,84,0,0,0,0,80,65,84,84,0,0,0,0,70,117,110,107,116,114,97,99,107,101,114,32,68,79,83,51,50,0,0,0,0,0,0,0,37,115,37,115,46,110,116,0,56,83,77,80,0,0,0,0,80,65,84,84,0,0,0,0,80,65,84,84,0,0,0,0,115,116,100,58,58,98,97,100,95,97,108,108,111,99,0,0,73,78,83,84,0,0,0,0,68,105,103,105,116,114,97,107,107,101,114,32,40,77,68,76,41,0,0,0,0,0,0,0,80,114,111,116,114,97,99,107,101,114,32,99,108,111,110,101,0,0,0,0,0,0,0,0,83,99,114,101,97,109,32,84,114,97,99,107,101,114,32,51,32,77,79,68,0,0,0,0,120,109,112,95,88,88,88,88,88,88,0,0,0,0,0,0,69,112,105,99,32,77,101,103,97,71,97,109,101,115,32,77,65,83,73,32,40,80,83,77,41,0,0,0,0,0,0,0,70,97,105,108,32,50,32,111,110,32,109,32,61,32,37,100,0,0,0,0,0,0,0,0,117,110,107,110,111,119,110,32,116,114,97,99,107,101,114,0,112,111,119,40,40,102,108,111,97,116,41,32,114,43,49,44,32,100,105,109,41,32,62,32,101,110,116,114,105,101,115,0,108,105,98,120,109,112,58,32,119,114,105,116,101,32,101,114,114,111,114,32,105,110,32,108,122,100,58,119,114,95,100,99,104,97,114,46,10,0,0,0,65,83,89,76,85,77,32,77,117,115,105,99,32,70,111,114,109,97,116,32,86,49,46,48,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,67,82,67,58,32,65,114,99,104,105,118,101,95,72,101,97,100,101,114,10,0,0,0,0,83,111,117,110,100,116,114,97,99,107,101,114,0,0,0,0,101,114,114,111,114,32,101,120,116,114,97,99,116,105,110,103,32,102,105,108,101,0,0,0,76,105,113,117,105,100,32,84,114,97,99,107,101,114,32,40,76,73,81,41,0,0,0,0,100,115,116,32,62,61,32,111,114,103,100,115,116,0,0,0,69,114,114,111,114,58,32,76,69,78,32,97,110,100,32,78,76,69,78,32,100,111,110,39,116,32,109,97,116,99,104,32,40,37,100,32,37,100,41,10,0,0,0,0,0,0,0,0,101,114,114,111,114,44,32,37,115,10,0,0,0,0,0,0,77,111,100,39,115,32,71,114,97,118,101,0,0,0,0,0,77,86,79,88,0,0,0,0,117,110,107,110,111,119,110,32,116,114,97,99,107,101,114,0,70,108,101,120,116,114,97,120,0,0,0,0,0,0,0,0,49,46,52,0,0,0,0,0,65,78,32,67,79,79,76,46,0,0,0,0,0,0,0,0,60,111,0,0,0,0,0,0,66,77,79,68,50,83,84,77,32,83,84,88,0,0,0,0,66,77,79,68,50,83,84,77,32,83,84,77,0,0,0,0,83,111,117,110,100,116,114,97,99,107,101,114,32,73,88,0,73,73,103,115,32,83,111,117,110,100,83,109,105,116,104,0,83,111,117,110,100,70,88,32,50,46,48,0,0,0,0,0,73,109,97,103,111,32,79,114,112,104,101,117,115,32,37,100,46,37,48,50,120,0,0,0,37,115,32,82,84,77,32,37,120,46,37,48,50,120,0,0,73,109,112,117,108,115,101,32,84,114,97,99,107,101,114,32,40,73,84,41,0,0,0,0,82,65,68,32,98,121,32,82,69,65,76,105,84,89,33,33,0,0,0,0,0,0,0,0,46,0,0,0,0,0,0,0,70,76,69,88,0,0,0,0,119,43,98,0,0,0,0,0,73,78,70,79,0,0,0,0,73,109,97,103,101,115,32,77,117,115,105,99,32,83,121,115,116,101,109,32,40,73,77,83,41,0,0,0,0,0,0,0,85,78,73,67,32,84,114,97,99,107,101,114,32,105,100,48,0,0,0,0,0,0,0,0,85,78,73,83,32,54,54,57,0,0,0,0,0,0,0,0,67,111,117,108,100,110,39,116,32,97,108,108,111,99,97,116,101,32,109,101,109,111,114,121,0,0,0,0,0,0,0,0,73,109,97,103,111,32,79,114,112,104,101,117,115,32,40,73,77,70,41,0,0,0,0,0,83,65,77,80,0,0,0,0,77,84,77,0,0,0,0,0,84,97,107,101,84,114,97,99,107,101,114,47,70,97,115,116,84,114,97,99,107,101,114,32,73,73,0,0,0,0,0,0,77,33,75,33,0,0,0,0,79,99,116,97,77,69,68,32,83,111,117,110,100,115,116,117,100,105,111,32,77,77,68,37,99,0,0,0,0,0,0,0,77,69,68,32,50,46,49,48,32,77,77,68,48,0,0,0,37,115,37,115,0,0,0,0,77,69,68,32,37,100,46,37,48,50,100,32,77,69,68,52,0,0,0,0,0,0,0,0,115,114,99,47,108,111,97,100,101,114,115,47,109,101,100,51,95,108,111,97,100,46,99,0,37,115,47,37,115,0,0,0,84,82,0,0,0,0,0,0,83,68,70,84,0,0,0,0,67,72,78,0,0,0,0,0,115,114,99,47,108,111,97,100,101,114,115,47,108,105,113,95,108,111,97,100,46,99,0,0,82,73,70,70,0,0,0,0,79,112,101,110,83,80,67,32,99,111,110,118,101,114,115,105,111,110,0,0,0,0,0,0,83,111,117,110,100,116,114,97,99,107,101,114,32,50,46,54,32,77,84,78,0,0,0,0,83,111,117,110,100,116,114,97,99,107,101,114,32,50,46,54,47,73,99,101,32,84,114,97,99,107,101,114,32,40,77,84,78,41,0,0,0,0,0,0,71,84,75,0,0,0,0,0,42,46,68,73,90,0,0,0,71,68,77,32,37,100,46,37,48,50,100,32,40,117,110,107,110,111,119,110,32,116,114,97,99,107,101,114,32,37,100,46,37,48,50,100,41,0,0,0,67,72,0,0,0,0,0,0,79,82,68,82,0,0,0,0,79,82,68,82,0,0,0,0,72,83,67,45,84,114,97,99,107,101,114,0,0,0,0,0,70,117,110,107,116,114,97,99,107,101,114,0,0,0,0,0,114,98,0,0,0,0,0,0,80,65,84,84,0,0,0,0,80,114,111,116,114,97,99,107,101,114,32,40,77,79,68,41,0,0,0,0,0,0,0,0,71,114,97,111,117,109,102,32,84,114,97,99,107,101,114,32,40,71,84,75,41,0,0,0,83,46,81,46,0,0,0,0,83,69,81,85,0,0,0,0,68,73,71,73,32,66,111,111,115,116,101,114,32,109,111,100,117,108,101,0,0,0,0,0,83,79,78,71,0,0,0,0,42,46,82,101,97,100,77,101,0,0,0,0,0,0,0,0,71,101,110,101,114,105,99,32,68,105,103,105,116,97,108,32,77,117,115,105,99,32,40,71,68,77,41,0,0,0,0,0,42,46,82,101,97,100,109,101,0,0,0,0,0,0,0,0,115,105,122,101,32,60,61,32,88,77,80,95,77,65,88,95,70,82,65,77,69,83,73,90,69,0,0,0,0,0,0,0,73,78,70,79,0,0,0,0,70,97,105,108,32,49,32,111,110,32,109,32,61,32,37,100,10,0,0,0,0,0,0,0,117,110,107,110,111,119,110,0,77,111,100,117,108,101,32,80,114,111,116,101,99,116,111,114,32,110,111,73,68,0,0,0,42,46,114,101,97,100,109,101,0,0,0,0,0,0,0,0,71,97,108,97,120,121,32,77,117,115,105,99,32,83,121,115,116,101,109,32,53,46,48,32,40,74,50,66,41,0,0,0,82,69,65,68,77,69,0,0,46,0,0,0,0,0,0,0,253,55,122,88,90,0,0,0,78,83,77,83,0,0,0,0,102,45,62,98,121,116,101,115,95,105,110,95,115,101,103,32,62,32,48,0,0,0,0,0,108,105,98,120,109,112,58,32,115,116,97,99,107,32,111,118,101,114,102,108,111,119,32,105,110,32,108,122,100,40,41,46,10,0,0,0,0,0,0,0,37,115,32,73,84,32,37,100,46,37,48,50,120,0,0,0,65,115,121,108,117,109,32,77,117,115,105,99,32,70,111,114,109,97,116,32,86,49,46,48,0,0,0,0,0,0,0,0,77,65,108,108,111,99,40,70,105,108,101,110,97,109,101,95,110,111,100,101,41,10,0,0,117,110,99,111,109,112,114,101,115,115,58,32,99,111,114,114,117,112,116,32,105,110,112,117,116,10,0,0,0,0,0,0,71,97,108,97,120,121,32,77,117,115,105,99,32,83,121,115,116,101,109,32,52,46,48,0,42,46,82,69,65,68,77,69,0,0,0,0,0,0,0,0,110,111,109,97,114,99,104,58,32,101,114,114,111,114,32,114,101,97,100,105,110,103,32,100,97,116,97,32,40,104,105,116,32,69,79,70,41,10,0,0,100,115,116,32,62,32,111,114,103,100,115,116,0,0,0,0,110,111,109,97,114,99,104,58,32,111,117,116,32,111,102,32,109,101,109,111,114,121,33,10,0,0,0,0,0,0,0,0,70,67,72,69,67,75,32,102,97,105,108,115,46,10,0,0,70,117,110,107,116,114,97,99,107,101,114,32,40,70,78,75,41,0,0,0,0,0,0,0,70,65,48,56,0,0,0,0,65,114,99,70,83,58,32,101,114,114,111,114,32,101,120,116,114,97,99,116,105,110,103,32,102,105,108,101,10,0,0,0,66,101,82,111,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,0,0,0,0,42,46,99,111,109,0,0,0,33,99,45,62,115,112,97,114,115,101,0,0,0,0,0,0,84,73,78,70,0,0,0,0,68,83,77,73,32,37,100,46,37,100,32,65,77,70,0,0,70,65,48,54,0,0,0,0,42,46,67,79,77,0,0,0,47,0,0,0,0,0,0,0,32,40,99,111,109,112,97,116,46,41,0,0,0,0,0,0,69,88,79,0,0,0,0,0,37,45,50,48,46,50,48,115,0,0,0,0,0,0,0,0,69,120,116,101,110,100,101,100,32,77,111,100,117,108,101,58,0,0,0,0,0,0,0,0,114,0,0,0,0,0,0,0,60,32,49,46,52,0,0,0,45,45,45,62,32,37,48,50,120,10,0,0,0,0,0,0,99,45,62,115,111,114,116,101,100,95,99,111,100,101,119,111,114,100,115,32,124,124,32,99,45,62,99,111,100,101,119,111,114,100,115,0,0,0,0,0,108,105,98,120,109,112,58,32,117,110,115,117,112,112,111,114,116,101,100,32,115,97,109,112,108,101,32,116,121,112,101,10,0,0,0,0,0,0,0,0,65,109,117,115,105,99,32,65,100,108,105,98,32,84,114,97,99,107,101,114,0,0,0,0,66,77,79,68,50,83,84,77,0,0,0,0,0,0,0,0,66,77,79,68,50,83,84,77,0,0,0,0,0,0,0,0,90,88,32,83,112,101,99,116,114,117,109,32,83,111,117,110,100,32,84,114,97,99,107,101,114,0,0,0,0,0,0,0,85,108,116,105,109,97,116,101,32,83,111,117,110,100,116,114,97,99,107,101,114,0,0,0,83,116,97,114,116,114,101,107,107,101,114,32,40,77,79,68,41,0,0,0,0,0,0,0,83,79,78,71,79,75,0,0,68,105,103,105,116,97,108,32,84,114,97,99,107,101,114,0,83,111,117,110,100,70,88,32,49,46,51,0,0,0,0,0,42,46,101,120,101,0,0,0,108,105,98,120,109,112,58,32,115,104,111,114,116,32,114,101,97,100,32,40,37,100,41,32,105,110,32,115,97,109,112,108,101,32,108,111,97,100,10,0,79,77,80,84,0,0,0,0,83,99,114,101,97,109,32,84,114,97,99,107,101,114,32,37,100,46,37,48,50,120,0,0,70,76,84,0,0,0,0,0,82,84,77,77,0,0,0,0,82,65,68,32,37,100,46,37,100,0,0,0,0,0,0,0,70,97,114,97,110,100,111,108,101,32,67,111,109,112,111,115,101,114,32,40,70,65,82,41,0,0,0,0,0,0,0,0,119,43,98,0,0,0,0,0,120,109,112,95,88,88,88,88,88,88,0,0,0,0,0,0,80,111,108,121,32,84,114,97,99,107,101,114,32,80,84,77,32,37,100,46,37,48,50,120,0,0,0,0,0,0,0,0,65,76,69,89,77,79,68,0,37,45,54,46,54,115,32,73,70,70,77,79,68,76,0,0,33,99,45,62,115,112,97,114,115,101,32,124,124,32,122,32,60,32,99,45,62,115,111,114,116,101,100,95,101,110,116,114,105,101,115,0,0,0,0,0,80,114,111,116,114,97,99,107,101,114,32,83,116,117,100,105,111,32,80,83,77,32,37,100,46,37,48,50,100,0,0,0,85,78,73,67,32,84,114,97,99,107,101,114,32,110,111,105,100,0,0,0,0,0,0,0,67,80,76,88,95,84,80,51,0,0,0,0,0,0,0,0,105,102,0,0,0,0,0,0,70,65,48,52,0,0,0,0,42,46,69,88,69,0,0,0,119,43,98,0,0,0,0,0,79,112,101,110,77,80,84,32,37,100,46,37,48,50,120,0,65,77,0,0,0,0,0,0,67,77,79,68,0,0,0,0,81,117,97,100,114,97,32,67,111,109,112,111,115,101,114,32,40,69,77,79,68,41,0,0,76,105,113,117,105,100,32,84,114,97,99,107,101,114,0,0,77,117,108,116,105,84,114,97,99,107,101,114,32,37,100,46,37,48,50,100,32,77,84,77,0,0,0,0,0,0,0,0,83,52,48,52,0,0,0,0,122,32,60,32,99,45,62,115,111,114,116,101,100,95,101,110,116,114,105,101,115,0,0,0,80,114,111,116,114,97,99,107,101,114,0,0,0,0,0,0,68,101,115,107,116,111,112,32,84,114,97,99,107,101,114,32,40,68,84,84,41,0,0,0,79,99,116,97,77,69,68,32,118,53,32,77,77,68,50,0,79,99,116,97,77,69,68,32,50,46,48,48,32,77,77,68,48,0,0,0,0,0,0,0,77,101,103,97,116,114,97,99,107,101,114,32,77,71,84,32,118,37,100,46,37,100,0,0,108,105,98,120,109,112,58,32,105,110,118,97,108,105,100,32,102,105,108,101,110,97,109,101,32,37,115,10,0,0,0,0,77,69,68,86,0,0,0,4,0,0,0,0,0,0,0,0,84,97,107,101,84,114,97,99,107,101,114,0,0,0,0,0,77,69,68,32,50,46,48,48,32,77,69,68,51,0,0,0,42,46,116,120,116,0,0,0,83,99,104,105,115,109,32,84,114,97,99,107,101,114,32,48,46,37,120,0,0,0,0,0,77,69,68,32,49,46,49,50,32,77,69,68,50,0,0,0,37,115,32,40,37,52,46,52,115,41,0,0,0,0,0,0,73,78,0,0,0,0,0,0,117,110,114,97,114,32,112,32,45,105,110,117,108,32,45,120,114,101,97,100,109,101,32,45,120,42,46,100,105,122,32,45,120,42,46,110,102,111,32,45,120,42,46,116,120,116,32,45,120,42,46,101,120,101,32,45,120,42,46,99,111,109,32,34,37,115,34,0,0,0,0,0,84,73,84,76,0,0,0,0,40,110,32,38,32,51,41,32,61,61,32,48,0,0,0,0,37,115,32,76,73,81,32,37,100,46,37,48,50,100,0,0,69,79,70,58,32,68,97,116,97,10,0,0,0,0,0,0,117,110,109,111,51,0,0,0,73,109,97,103,101,115,32,77,117,115,105,99,32,83,121,115,116,101,109,0,0,0,0,0,68,105,103,105,116,97,108,32,84,114,97,99,107,101,114,32,40,68,84,77,41,0,0,0,73,109,97,103,111,32,79,114,112,104,101,117,115,32,49,46,48,32,73,77,70,0,0,0,84,68,90,52,0,0,0,0,42,46,84,88,84,0,0,0,76,105,113,117,105,100,32,77,111,100,117,108,101,58,0,0,83,99,104,105,115,109,32,84,114,97,99,107,101,114,32,37,48,52,100,45,37,48,50,100,45,37,48,50,100,0,0,0,73,99,101,32,84,114,97,99,107,101,114,32,73,84,49,48,0,0,0,0,0,0,0,0,65,117,100,105,111,83,99,117,108,112,116,117,114,101,32,49,46,48,0,0,0,0,0,0,71,114,97,111,117,109,102,32,84,114,97,99,107,101,114,32,71,84,75,32,118,37,100,0,71,68,77,32,37,100,46,37,48,50,100,32,40,50,71,68,77,32,37,100,46,37,48,50])
.concat([100,41,0,0,0,0,0,0,114,101,97,100,109,101,0,0,82,97,114,0,0,0,0,0,73,78,73,84,0,0,0,0,118,32,61,61,32,98,117,102,50,0,0,0,0,0,0,0,77,65,73,78,0,0,0,0,70,82,101,97,100,40,68,97,116,97,41,0,0,0,0,0,83,65,77,80,0,0,0,0,70,117,110,107,116,114,97,99,107,101,114,71,79,76,68,0,37,115,37,115,46,78,84,0,70,97,114,97,110,100,111,108,101,32,67,111,109,112,111,115,101,114,32,37,100,46,37,100,0,0,0,0,0,0,0,0,69,77,73,67,0,0,0,0,68,101,115,107,116,111,112,32,84,114,97,99,107,101,114,0,67,68,56,49,0,0,0,0,42,46,73,110,102,111,0,0,101,118,101,110,116,45,62,118,111,108,32,60,61,32,54,52,0,0,0,0,0,0,0,0,68,46,84,46,0,0,0,0,117,110,107,110,111,119,110,32,40,37,48,52,120,41,0,0,88,45,84,114,97,99,107,101,114,32,40,68,77,70,41,0,65,117,100,105,111,83,99,117,108,112,116,117,114,101,49,48,0,0,0,0,0,0,0,0,37,115,32,68,77,70,32,118,37,100,0,0,0,0,0,0,68,73,71,73,32,66,111,111,115,116,101,114,32,37,45,52,46,52,115,0,0,0,0,0,76,90,88,0,0,0,0,0,102,45,62,98,121,116,101,115,95,105,110,95,115,101,103,32,61,61,32,48,0,0,0,0,10,0,0,0,0,0,0,0,80,65,84,84,0,0,0,0,68,73,71,73,32,66,111,111,115,116,101,114,0,0,0,0,37,115,32,83,51,77,0,0,79,99,116,97,108,121,115,101,114,0,0,0,0,0,0,0,68,105,103,105,116,114,97,107,107,101,114,32,77,68,76,32,37,100,46,37,100,0,0,0,99,104,97,110,110,101,108,32,37,100,58,32,37,48,50,120,32,37,48,50,120,10,0,0,101,118,101,110,116,45,62,110,111,116,101,32,60,61,32,49,48,55,32,124,124,32,101,118,101,110,116,45,62,110,111,116,101,32,61,61,32,88,77,80,95,75,69,89,95,79,70,70,0,0,0,0,0,0,0,0,42,46,105,110,102,111,0,0,109,117,110,99,104,46,112,121,0,0,0,0,0,0,0,0,83,116,97,114,116,114,101,107,107,101,114,32,49,46,51,0,77,85,83,69,0,0,0,0,114,98,0,0,0,0,0,0,117,110,112,97,99,107,95,98,108,111,99,107,0,0,0,0,110,101,120,116,95,115,101,103,109,101,110,116,0,0,0,0,109,105,120,101,114,95,115,111,102,116,109,105,120,101,114,0,109,101,100,51,95,108,111,97,100,0,0,0,0,0,0,0,108,111,111,107,117,112,49,95,118,97,108,117,101,115,0,0,108,105,113,95,108,111,97,100,0,0,0,0,0,0,0,0,105,110,118,101,114,115,101,95,109,100,99,116,0,0,0,0,105,110,99,108,117,100,101,95,105,110,95,115,111,114,116,0,105,109,100,99,116,95,115,116,101,112,51,95,105,116,101,114,48,95,108,111,111,112,0,0,103,101,116,98,0,0,0,0,103,101,116,95,119,105,110,100,111,119,0,0,0,0,0,0,103,101,116,95,115,109,112,100,0,0,0,0,0,0,0,0,103,101,116,56,95,112,97,99,107,101,116,95,114,97,119,0,103,100,109,95,108,111,97,100,0,0,0,0,0,0,0,0,100,101,99,111,109,112,114,101,115,115,83,52,48,52,0,0,100,101,99,111,100,101,95,101,118,101,110,116,0,0,0,0,99,111,109,112,117,116,101,95,115,111,114,116,101,100,95,104,117,102,102,109,97,110,0,0,99,111,109,112,117,116,101,95,99,111,100,101,119,111,114,100,115,0,0,0,0,0,0,0,99,111,100,101,98,111,111,107,95,100,101,99,111,100,101,95,115,116,97,114,116,0,0,0,99,111,100,101,98,111,111,107,95,100,101,99,111,100,101,95,115,99,97,108,97,114,95,114,97,119,0,0,0,0,0,0,99,111,100,101,98,111,111,107,95,100,101,99,111,100,101,95,100,101,105,110,116,101,114,108,101,97,118,101,95,114,101,112,101,97,116,0,0,0,0,0,0,0,0,0,184,124,0,0,14,0,0,0,8,0,0,0,24,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,248,124,0,0,2,0,0,0,12,0,0,0,0,0,0,0,0,0,0,0,83,116,57,116,121,112,101,95,105,110,102,111,0,0,0,0,83,116,57,101,120,99,101,112,116,105,111,110,0,0,0,0,83,116,57,98,97,100,95,97,108,108,111,99,0,0,0,0,78,49,48,95,95,99,120,120,97,98,105,118,49,50,48,95,95,115,105,95,99,108,97,115,115,95,116,121,112,101,95,105,110,102,111,69,0,0,0,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,55,95,95,99,108,97,115,115,95,116,121,112,101,95,105,110,102,111,69,0,0,0,0,0,0,0,78,49,48,95,95,99,120,120,97,98,105,118,49,49,54,95,95,115,104,105,109,95,116,121,112,101,95,105,110,102,111,69,0,0,0,0,0,0,0,0,54,112,108,97,121,101,114,0,0,0,0,0,248,123,0,0,0,0,0,0,8,124,0,0,0,0,0,0,24,124,0,0,176,124,0,0,0,0,0,0,0,0,0,0,40,124,0,0,216,124,0,0,0,0,0,0,0,0,0,0,80,124,0,0,232,124,0,0,0,0,0,0,0,0,0,0,120,124,0,0,168,124,0,0,0,0,0,0,0,0,0,0,160,124,0,0,0,0,0,16,0,0,128,16,0,0,0,17,0,0,128,17,0,0,0,18,0,0,128,18,0,0,0,19,0,0,128,19,0,0,0,20,0,0,128,20,0,0,0,21,0,0,128,21,0,0,0,22,0,0,128,22,0,0,0,23,0,0,128,31,128,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,2,0,0,0,4,0,0,0,6,0,0,0,8,0,0,0,10,0,0,0,12,0,0,0,14,0,0,0,16,0,0,0,18,0,0,0,20,0,0,0,20,0,0,0,24,0,0,0,24,0,0,0,30,0,0,0,30,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,64,0,0,0,96,0,0,0,128,0,0,0,160,0,0,0,192,0,0,0,224,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,96,0,0,0,160,0,0,0,224,0,0,0,0,1,0,0,64,1,0,0,96,1,0,0,128,1,0,0,160,1,0,0,192,1,0,0,224,1,0,0,0,2,0,0,0,0,0,0,0,0,0,0,0,0,0,0,160,0,0,0,0,1,0,0,96,1,0,0,160,1,0,0,224,1,0,0,0,2,0,0,64,2,0,0,96,2,0,0,128,2,0,0,160,2,0,0,192,2,0,0,224,2,0,0,0,3,0,0,0,0,0,0,0,0,0,0,0,1,0,0,160,1,0,0,0,2,0,0,96,2,0,0,160,2,0,0,224,2,0,0,0,3,0,0,64,3,0,0,96,3,0,0,128,3,0,0,160,3,0,0,192,3,0,0,224,3,0,0,0,4,0,0,0,0,0,0,0,1,0,0,0,2,0,0,160,2,0,0,0,3,0,0,96,3,0,0,160,3,0,0,224,3,0,0,0,4,0,0,64,4,0,0,96,4,0,0,128,4,0,0,160,4,0,0,192,4,0,0,224,4,0,0,0,5,0,0,0,0,0,0,0,2,0,0,0,3,0,0,160,3,0,0,0,4,0,0,96,4,0,0,160,4,0,0,224,4,0,0,0,5,0,0,64,5,0,0,96,5,0,0,128,5,0,0,160,5,0,0,192,5,0,0,224,5,0,0,0,6,0,0,0,0,0,0,0,3,0,0,0,4,0,0,160,4,0,0,0,5,0,0,96,5,0,0,160,5,0,0,224,5,0,0,0,6,0,0,64,6,0,0,96,6,0,0,128,6,0,0,160,6,0,0,192,6,0,0,224,6,0,0,0,7,0,0,32,129,0,0,32,129,0,0,32,129,0,0,32,129,0,0,0,129,0,0,0,129,0,0,0,129,0,0,0,129,0,0,224,128,0,0,192,128,0,0,160,128,0,0,128,128,0,0,96,128,0,0,64,128,0,0,32,128,0,0,0,128,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0])
, "i8", ALLOC_NONE, Runtime.GLOBAL_BASE)
var tempDoublePtr = Runtime.alignMemory(allocate(12, "i8", ALLOC_STATIC), 8);
assert(tempDoublePtr % 8 == 0);
function copyTempFloat(ptr) { // functions, because inlining this code increases code size too much
  HEAP8[tempDoublePtr] = HEAP8[ptr];
  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];
  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];
  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];
}
function copyTempDouble(ptr) {
  HEAP8[tempDoublePtr] = HEAP8[ptr];
  HEAP8[tempDoublePtr+1] = HEAP8[ptr+1];
  HEAP8[tempDoublePtr+2] = HEAP8[ptr+2];
  HEAP8[tempDoublePtr+3] = HEAP8[ptr+3];
  HEAP8[tempDoublePtr+4] = HEAP8[ptr+4];
  HEAP8[tempDoublePtr+5] = HEAP8[ptr+5];
  HEAP8[tempDoublePtr+6] = HEAP8[ptr+6];
  HEAP8[tempDoublePtr+7] = HEAP8[ptr+7];
}
  function ___gxx_personality_v0() {
    }
  function _llvm_umul_with_overflow_i32(x, y) {
      x = x>>>0;
      y = y>>>0;
      return ((asm["setTempRet0"](x*y > 4294967295),(x*y)>>>0)|0);
    }
  var ERRNO_CODES={EPERM:1,ENOENT:2,ESRCH:3,EINTR:4,EIO:5,ENXIO:6,E2BIG:7,ENOEXEC:8,EBADF:9,ECHILD:10,EAGAIN:11,EWOULDBLOCK:11,ENOMEM:12,EACCES:13,EFAULT:14,ENOTBLK:15,EBUSY:16,EEXIST:17,EXDEV:18,ENODEV:19,ENOTDIR:20,EISDIR:21,EINVAL:22,ENFILE:23,EMFILE:24,ENOTTY:25,ETXTBSY:26,EFBIG:27,ENOSPC:28,ESPIPE:29,EROFS:30,EMLINK:31,EPIPE:32,EDOM:33,ERANGE:34,ENOMSG:35,EIDRM:36,ECHRNG:37,EL2NSYNC:38,EL3HLT:39,EL3RST:40,ELNRNG:41,EUNATCH:42,ENOCSI:43,EL2HLT:44,EDEADLK:45,ENOLCK:46,EBADE:50,EBADR:51,EXFULL:52,ENOANO:53,EBADRQC:54,EBADSLT:55,EDEADLOCK:56,EBFONT:57,ENOSTR:60,ENODATA:61,ETIME:62,ENOSR:63,ENONET:64,ENOPKG:65,EREMOTE:66,ENOLINK:67,EADV:68,ESRMNT:69,ECOMM:70,EPROTO:71,EMULTIHOP:74,EDOTDOT:76,EBADMSG:77,ENOTUNIQ:80,EBADFD:81,EREMCHG:82,ELIBACC:83,ELIBBAD:84,ELIBSCN:85,ELIBMAX:86,ELIBEXEC:87,ENOSYS:88,ENOTEMPTY:90,ENAMETOOLONG:91,ELOOP:92,EOPNOTSUPP:95,EPFNOSUPPORT:96,ECONNRESET:104,ENOBUFS:105,EAFNOSUPPORT:106,EPROTOTYPE:107,ENOTSOCK:108,ENOPROTOOPT:109,ESHUTDOWN:110,ECONNREFUSED:111,EADDRINUSE:112,ECONNABORTED:113,ENETUNREACH:114,ENETDOWN:115,ETIMEDOUT:116,EHOSTDOWN:117,EHOSTUNREACH:118,EINPROGRESS:119,EALREADY:120,EDESTADDRREQ:121,EMSGSIZE:122,EPROTONOSUPPORT:123,ESOCKTNOSUPPORT:124,EADDRNOTAVAIL:125,ENETRESET:126,EISCONN:127,ENOTCONN:128,ETOOMANYREFS:129,EUSERS:131,EDQUOT:132,ESTALE:133,ENOTSUP:134,ENOMEDIUM:135,EILSEQ:138,EOVERFLOW:139,ECANCELED:140,ENOTRECOVERABLE:141,EOWNERDEAD:142,ESTRPIPE:143};
  var ERRNO_MESSAGES={0:"Success",1:"Not super-user",2:"No such file or directory",3:"No such process",4:"Interrupted system call",5:"I/O error",6:"No such device or address",7:"Arg list too long",8:"Exec format error",9:"Bad file number",10:"No children",11:"No more processes",12:"Not enough core",13:"Permission denied",14:"Bad address",15:"Block device required",16:"Mount device busy",17:"File exists",18:"Cross-device link",19:"No such device",20:"Not a directory",21:"Is a directory",22:"Invalid argument",23:"Too many open files in system",24:"Too many open files",25:"Not a typewriter",26:"Text file busy",27:"File too large",28:"No space left on device",29:"Illegal seek",30:"Read only file system",31:"Too many links",32:"Broken pipe",33:"Math arg out of domain of func",34:"Math result not representable",35:"No message of desired type",36:"Identifier removed",37:"Channel number out of range",38:"Level 2 not synchronized",39:"Level 3 halted",40:"Level 3 reset",41:"Link number out of range",42:"Protocol driver not attached",43:"No CSI structure available",44:"Level 2 halted",45:"Deadlock condition",46:"No record locks available",50:"Invalid exchange",51:"Invalid request descriptor",52:"Exchange full",53:"No anode",54:"Invalid request code",55:"Invalid slot",56:"File locking deadlock error",57:"Bad font file fmt",60:"Device not a stream",61:"No data (for no delay io)",62:"Timer expired",63:"Out of streams resources",64:"Machine is not on the network",65:"Package not installed",66:"The object is remote",67:"The link has been severed",68:"Advertise error",69:"Srmount error",70:"Communication error on send",71:"Protocol error",74:"Multihop attempted",76:"Cross mount point (not really error)",77:"Trying to read unreadable message",80:"Given log. name not unique",81:"f.d. invalid for this operation",82:"Remote address changed",83:"Can   access a needed shared lib",84:"Accessing a corrupted shared lib",85:".lib section in a.out corrupted",86:"Attempting to link in too many libs",87:"Attempting to exec a shared library",88:"Function not implemented",90:"Directory not empty",91:"File or path name too long",92:"Too many symbolic links",95:"Operation not supported on transport endpoint",96:"Protocol family not supported",104:"Connection reset by peer",105:"No buffer space available",106:"Address family not supported by protocol family",107:"Protocol wrong type for socket",108:"Socket operation on non-socket",109:"Protocol not available",110:"Can't send after socket shutdown",111:"Connection refused",112:"Address already in use",113:"Connection aborted",114:"Network is unreachable",115:"Network interface is not configured",116:"Connection timed out",117:"Host is down",118:"Host is unreachable",119:"Connection already in progress",120:"Socket already connected",121:"Destination address required",122:"Message too long",123:"Unknown protocol",124:"Socket type not supported",125:"Address not available",126:"Connection reset by network",127:"Socket is already connected",128:"Socket is not connected",129:"Too many references",131:"Too many users",132:"Quota exceeded",133:"Stale file handle",134:"Not supported",135:"No medium (in tape drive)",138:"Illegal byte sequence",139:"Value too large for defined data type",140:"Operation canceled",141:"State not recoverable",142:"Previous owner died",143:"Streams pipe error"};
  var ___errno_state=0;function ___setErrNo(value) {
      // For convenient setting and returning of errno.
      HEAP32[((___errno_state)>>2)]=value
      return value;
    }
  var VFS=undefined;
  var PATH={splitPath:function (filename) {
        var splitPathRe = /^(\/?|)([\s\S]*?)((?:\.{1,2}|[^\/]+?|)(\.[^.\/]*|))(?:[\/]*)$/;
        return splitPathRe.exec(filename).slice(1);
      },normalizeArray:function (parts, allowAboveRoot) {
        // if the path tries to go above the root, `up` ends up > 0
        var up = 0;
        for (var i = parts.length - 1; i >= 0; i--) {
          var last = parts[i];
          if (last === '.') {
            parts.splice(i, 1);
          } else if (last === '..') {
            parts.splice(i, 1);
            up++;
          } else if (up) {
            parts.splice(i, 1);
            up--;
          }
        }
        // if the path is allowed to go above the root, restore leading ..s
        if (allowAboveRoot) {
          for (; up--; up) {
            parts.unshift('..');
          }
        }
        return parts;
      },normalize:function (path) {
        var isAbsolute = path.charAt(0) === '/',
            trailingSlash = path.substr(-1) === '/';
        // Normalize the path
        path = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), !isAbsolute).join('/');
        if (!path && !isAbsolute) {
          path = '.';
        }
        if (path && trailingSlash) {
          path += '/';
        }
        return (isAbsolute ? '/' : '') + path;
      },dirname:function (path) {
        var result = PATH.splitPath(path),
            root = result[0],
            dir = result[1];
        if (!root && !dir) {
          // No dirname whatsoever
          return '.';
        }
        if (dir) {
          // It has a dirname, strip trailing slash
          dir = dir.substr(0, dir.length - 1);
        }
        return root + dir;
      },basename:function (path, ext) {
        // EMSCRIPTEN return '/'' for '/', not an empty string
        if (path === '/') return '/';
        var f = PATH.splitPath(path)[2];
        if (ext && f.substr(-1 * ext.length) === ext) {
          f = f.substr(0, f.length - ext.length);
        }
        return f;
      },join:function () {
        var paths = Array.prototype.slice.call(arguments, 0);
        return PATH.normalize(paths.filter(function(p, index) {
          if (typeof p !== 'string') {
            throw new TypeError('Arguments to path.join must be strings');
          }
          return p;
        }).join('/'));
      },resolve:function () {
        var resolvedPath = '',
          resolvedAbsolute = false;
        for (var i = arguments.length - 1; i >= -1 && !resolvedAbsolute; i--) {
          var path = (i >= 0) ? arguments[i] : FS.cwd();
          // Skip empty and invalid entries
          if (typeof path !== 'string') {
            throw new TypeError('Arguments to path.resolve must be strings');
          } else if (!path) {
            continue;
          }
          resolvedPath = path + '/' + resolvedPath;
          resolvedAbsolute = path.charAt(0) === '/';
        }
        // At this point the path should be resolved to a full absolute path, but
        // handle relative paths to be safe (might happen when process.cwd() fails)
        resolvedPath = PATH.normalizeArray(resolvedPath.split('/').filter(function(p) {
          return !!p;
        }), !resolvedAbsolute).join('/');
        return ((resolvedAbsolute ? '/' : '') + resolvedPath) || '.';
      },relative:function (from, to) {
        from = PATH.resolve(from).substr(1);
        to = PATH.resolve(to).substr(1);
        function trim(arr) {
          var start = 0;
          for (; start < arr.length; start++) {
            if (arr[start] !== '') break;
          }
          var end = arr.length - 1;
          for (; end >= 0; end--) {
            if (arr[end] !== '') break;
          }
          if (start > end) return [];
          return arr.slice(start, end - start + 1);
        }
        var fromParts = trim(from.split('/'));
        var toParts = trim(to.split('/'));
        var length = Math.min(fromParts.length, toParts.length);
        var samePartsLength = length;
        for (var i = 0; i < length; i++) {
          if (fromParts[i] !== toParts[i]) {
            samePartsLength = i;
            break;
          }
        }
        var outputParts = [];
        for (var i = samePartsLength; i < fromParts.length; i++) {
          outputParts.push('..');
        }
        outputParts = outputParts.concat(toParts.slice(samePartsLength));
        return outputParts.join('/');
      }};
  var TTY={ttys:[],register:function (dev, ops) {
        TTY.ttys[dev] = { input: [], output: [], ops: ops };
        FS.registerDevice(dev, TTY.stream_ops);
      },stream_ops:{open:function (stream) {
          // this wouldn't be required if the library wasn't eval'd at first...
          if (!TTY.utf8) {
            TTY.utf8 = new Runtime.UTF8Processor();
          }
          var tty = TTY.ttys[stream.node.rdev];
          if (!tty) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          stream.tty = tty;
          stream.seekable = false;
        },close:function (stream) {
          // flush any pending line data
          if (stream.tty.output.length) {
            stream.tty.ops.put_char(stream.tty, 10);
          }
        },read:function (stream, buffer, offset, length, pos /* ignored */) {
          if (!stream.tty || !stream.tty.ops.get_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          var bytesRead = 0;
          for (var i = 0; i < length; i++) {
            var result;
            try {
              result = stream.tty.ops.get_char(stream.tty);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            if (result === undefined && bytesRead === 0) {
              throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
            }
            if (result === null || result === undefined) break;
            bytesRead++;
            buffer[offset+i] = result;
          }
          if (bytesRead) {
            stream.node.timestamp = Date.now();
          }
          return bytesRead;
        },write:function (stream, buffer, offset, length, pos) {
          if (!stream.tty || !stream.tty.ops.put_char) {
            throw new FS.ErrnoError(ERRNO_CODES.ENXIO);
          }
          for (var i = 0; i < length; i++) {
            try {
              stream.tty.ops.put_char(stream.tty, buffer[offset+i]);
            } catch (e) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
          }
          if (length) {
            stream.node.timestamp = Date.now();
          }
          return i;
        }},default_tty_ops:{get_char:function (tty) {
          if (!tty.input.length) {
            var result = null;
            if (ENVIRONMENT_IS_NODE) {
              if (process.stdin.destroyed) {
                return undefined;
              }
              result = process.stdin.read();
            } else if (typeof window != 'undefined' &&
              typeof window.prompt == 'function') {
              // Browser.
              result = window.prompt('Input: ');  // returns null on cancel
              if (result !== null) {
                result += '\n';
              }
            } else if (typeof readline == 'function') {
              // Command line.
              result = readline();
              if (result !== null) {
                result += '\n';
              }
            }
            if (!result) {
              return null;
            }
            tty.input = intArrayFromString(result, true);
          }
          return tty.input.shift();
        },put_char:function (tty, val) {
          if (val === null || val === 10) {
            Module['print'](tty.output.join(''));
            tty.output = [];
          } else {
            tty.output.push(TTY.utf8.processCChar(val));
          }
        }},default_tty1_ops:{put_char:function (tty, val) {
          if (val === null || val === 10) {
            Module['printErr'](tty.output.join(''));
            tty.output = [];
          } else {
            tty.output.push(TTY.utf8.processCChar(val));
          }
        }}};
  var MEMFS={mount:function (mount) {
        return MEMFS.create_node(null, '/', 0040000 | 0777, 0);
      },create_node:function (parent, name, mode, dev) {
        if (FS.isBlkdev(mode) || FS.isFIFO(mode)) {
          // no supported
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        var node = FS.createNode(parent, name, mode, dev);
        if (FS.isDir(node.mode)) {
          node.node_ops = {
            getattr: MEMFS.node_ops.getattr,
            setattr: MEMFS.node_ops.setattr,
            lookup: MEMFS.node_ops.lookup,
            mknod: MEMFS.node_ops.mknod,
            mknod: MEMFS.node_ops.mknod,
            rename: MEMFS.node_ops.rename,
            unlink: MEMFS.node_ops.unlink,
            rmdir: MEMFS.node_ops.rmdir,
            readdir: MEMFS.node_ops.readdir,
            symlink: MEMFS.node_ops.symlink
          };
          node.stream_ops = {
            llseek: MEMFS.stream_ops.llseek
          };
          node.contents = {};
        } else if (FS.isFile(node.mode)) {
          node.node_ops = {
            getattr: MEMFS.node_ops.getattr,
            setattr: MEMFS.node_ops.setattr
          };
          node.stream_ops = {
            llseek: MEMFS.stream_ops.llseek,
            read: MEMFS.stream_ops.read,
            write: MEMFS.stream_ops.write,
            allocate: MEMFS.stream_ops.allocate,
            mmap: MEMFS.stream_ops.mmap
          };
          node.contents = [];
        } else if (FS.isLink(node.mode)) {
          node.node_ops = {
            getattr: MEMFS.node_ops.getattr,
            setattr: MEMFS.node_ops.setattr,
            readlink: MEMFS.node_ops.readlink
          };
          node.stream_ops = {};
        } else if (FS.isChrdev(node.mode)) {
          node.node_ops = {
            getattr: MEMFS.node_ops.getattr,
            setattr: MEMFS.node_ops.setattr
          };
          node.stream_ops = FS.chrdev_stream_ops;
        }
        node.timestamp = Date.now();
        // add the new node to the parent
        if (parent) {
          parent.contents[name] = node;
        }
        return node;
      },node_ops:{getattr:function (node) {
          var attr = {};
          // device numbers reuse inode numbers.
          attr.dev = FS.isChrdev(node.mode) ? node.id : 1;
          attr.ino = node.id;
          attr.mode = node.mode;
          attr.nlink = 1;
          attr.uid = 0;
          attr.gid = 0;
          attr.rdev = node.rdev;
          if (FS.isDir(node.mode)) {
            attr.size = 4096;
          } else if (FS.isFile(node.mode)) {
            attr.size = node.contents.length;
          } else if (FS.isLink(node.mode)) {
            attr.size = node.link.length;
          } else {
            attr.size = 0;
          }
          attr.atime = new Date(node.timestamp);
          attr.mtime = new Date(node.timestamp);
          attr.ctime = new Date(node.timestamp);
          // NOTE: In our implementation, st_blocks = Math.ceil(st_size/st_blksize),
          //       but this is not required by the standard.
          attr.blksize = 4096;
          attr.blocks = Math.ceil(attr.size / attr.blksize);
          return attr;
        },setattr:function (node, attr) {
          if (attr.mode !== undefined) {
            node.mode = attr.mode;
          }
          if (attr.timestamp !== undefined) {
            node.timestamp = attr.timestamp;
          }
          if (attr.size !== undefined) {
            var contents = node.contents;
            if (attr.size < contents.length) contents.length = attr.size;
            else while (attr.size > contents.length) contents.push(0);
          }
        },lookup:function (parent, name) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        },mknod:function (parent, name, mode, dev) {
          return MEMFS.create_node(parent, name, mode, dev);
        },rename:function (old_node, new_dir, new_name) {
          // if we're overwriting a directory at new_name, make sure it's empty.
          if (FS.isDir(old_node.mode)) {
            var new_node;
            try {
              new_node = FS.lookupNode(new_dir, new_name);
            } catch (e) {
            }
            if (new_node) {
              for (var i in new_node.contents) {
                throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
              }
            }
          }
          // do the internal rewiring
          delete old_node.parent.contents[old_node.name];
          old_node.name = new_name;
          new_dir.contents[new_name] = old_node;
        },unlink:function (parent, name) {
          delete parent.contents[name];
        },rmdir:function (parent, name) {
          var node = FS.lookupNode(parent, name);
          for (var i in node.contents) {
            throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
          }
          delete parent.contents[name];
        },readdir:function (node) {
          var entries = ['.', '..']
          for (var key in node.contents) {
            if (!node.contents.hasOwnProperty(key)) {
              continue;
            }
            entries.push(key);
          }
          return entries;
        },symlink:function (parent, newname, oldpath) {
          var node = MEMFS.create_node(parent, newname, 0777 | 0120000, 0);
          node.link = oldpath;
          return node;
        },readlink:function (node) {
          if (!FS.isLink(node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          return node.link;
        }},stream_ops:{read:function (stream, buffer, offset, length, position) {
          var contents = stream.node.contents;
          var size = Math.min(contents.length - position, length);
          if (contents.subarray) { // typed array
            buffer.set(contents.subarray(position, position + size), offset);
          } else
          {
            for (var i = 0; i < size; i++) {
              buffer[offset + i] = contents[position + i];
            }
          }
          return size;
        },write:function (stream, buffer, offset, length, position) {
          var contents = stream.node.contents;
          while (contents.length < position) contents.push(0);
          for (var i = 0; i < length; i++) {
            contents[position + i] = buffer[offset + i];
          }
          stream.node.timestamp = Date.now();
          return length;
        },llseek:function (stream, offset, whence) {
          var position = offset;
          if (whence === 1) {  // SEEK_CUR.
            position += stream.position;
          } else if (whence === 2) {  // SEEK_END.
            if (FS.isFile(stream.node.mode)) {
              position += stream.node.contents.length;
            }
          }
          if (position < 0) {
            throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
          }
          stream.ungotten = [];
          stream.position = position;
          return position;
        },allocate:function (stream, offset, length) {
          var contents = stream.node.contents;
          var limit = offset + length;
          while (limit > contents.length) contents.push(0);
        },mmap:function (stream, buffer, offset, length, position, prot, flags) {
          if (!FS.isFile(stream.node.mode)) {
            throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
          }
          var ptr;
          var allocated;
          var contents = stream.node.contents;
          // Only make a new copy when MAP_PRIVATE is specified.
          if (!(flags & 0x02)) {
            // We can't emulate MAP_SHARED when the file is not backed by the buffer
            // we're mapping to (e.g. the HEAP buffer).
            assert(contents.buffer === buffer || contents.buffer === buffer.buffer);
            allocated = false;
            ptr = contents.byteOffset;
          } else {
            // Try to avoid unnecessary slices.
            if (position > 0 || position + length < contents.length) {
              if (contents.subarray) {
                contents = contents.subarray(position, position + length);
              } else {
                contents = Array.prototype.slice.call(contents, position, position + length);
              }
            }
            allocated = true;
            ptr = _malloc(length);
            if (!ptr) {
              throw new FS.ErrnoError(ERRNO_CODES.ENOMEM);
            }
            buffer.set(contents, ptr);
          }
          return { ptr: ptr, allocated: allocated };
        }}};
  var _stdin=allocate(1, "i32*", ALLOC_STATIC);
  var _stdout=allocate(1, "i32*", ALLOC_STATIC);
  var _stderr=allocate(1, "i32*", ALLOC_STATIC);
  function _fflush(stream) {
      // int fflush(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fflush.html
      // we don't currently perform any user-space buffering of data
    }var FS={root:null,nodes:[null],devices:[null],streams:[null],nextInode:1,name_table:null,currentPath:"/",initialized:false,ignorePermissions:true,ErrnoError:function ErrnoError(errno) {
          this.errno = errno;
          for (var key in ERRNO_CODES) {
            if (ERRNO_CODES[key] === errno) {
              this.code = key;
              break;
            }
          }
          this.message = ERRNO_MESSAGES[errno];
        },handleFSError:function (e) {
        if (!(e instanceof FS.ErrnoError)) throw e + ' : ' + new Error().stack;
        return ___setErrNo(e.errno);
      },hashName:function (parentid, name) {
        var hash = 0;
        for (var i = 0; i < name.length; i++) {
          hash = ((hash << 5) - hash + name.charCodeAt(i)) | 0;
        }
        return ((parentid + hash) >>> 0) % FS.name_table.length;
      },hashAddNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        node.name_next = FS.name_table[hash];
        FS.name_table[hash] = node;
      },hashRemoveNode:function (node) {
        var hash = FS.hashName(node.parent.id, node.name);
        if (FS.name_table[hash] === node) {
          FS.name_table[hash] = node.name_next;
        } else {
          var current = FS.name_table[hash];
          while (current) {
            if (current.name_next === node) {
              current.name_next = node.name_next;
              break;
            }
            current = current.name_next;
          }
        }
      },lookupNode:function (parent, name) {
        var err = FS.mayLookup(parent);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        var hash = FS.hashName(parent.id, name);
        for (var node = FS.name_table[hash]; node; node = node.name_next) {
          if (node.parent.id === parent.id && node.name === name) {
            return node;
          }
        }
        // if we failed to find it in the cache, call into the VFS
        return FS.lookup(parent, name);
      },createNode:function (parent, name, mode, rdev) {
        var node = {
          id: FS.nextInode++,
          name: name,
          mode: mode,
          node_ops: {},
          stream_ops: {},
          rdev: rdev,
          parent: null,
          mount: null
        };
        if (!parent) {
          parent = node;  // root node sets parent to itself
        }
        node.parent = parent;
        node.mount = parent.mount;
        // compatibility
        var readMode = 292 | 73;
        var writeMode = 146;
        // NOTE we must use Object.defineProperties instead of individual calls to
        // Object.defineProperty in order to make closure compiler happy
        Object.defineProperties(node, {
          read: {
            get: function() { return (node.mode & readMode) === readMode; },
            set: function(val) { val ? node.mode |= readMode : node.mode &= ~readMode; }
          },
          write: {
            get: function() { return (node.mode & writeMode) === writeMode; },
            set: function(val) { val ? node.mode |= writeMode : node.mode &= ~writeMode; }
          },
          isFolder: {
            get: function() { return FS.isDir(node.mode); },
          },
          isDevice: {
            get: function() { return FS.isChrdev(node.mode); },
          },
        });
        FS.hashAddNode(node);
        return node;
      },destroyNode:function (node) {
        FS.hashRemoveNode(node);
      },isRoot:function (node) {
        return node === node.parent;
      },isMountpoint:function (node) {
        return node.mounted;
      },isFile:function (mode) {
        return (mode & 0170000) === 0100000;
      },isDir:function (mode) {
        return (mode & 0170000) === 0040000;
      },isLink:function (mode) {
        return (mode & 0170000) === 0120000;
      },isChrdev:function (mode) {
        return (mode & 0170000) === 0020000;
      },isBlkdev:function (mode) {
        return (mode & 0170000) === 0060000;
      },isFIFO:function (mode) {
        return (mode & 0170000) === 0010000;
      },cwd:function () {
        return FS.currentPath;
      },lookupPath:function (path, opts) {
        path = PATH.resolve(FS.currentPath, path);
        opts = opts || { recurse_count: 0 };
        if (opts.recurse_count > 8) {  // max recursive lookup of 8
          throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
        }
        // split the path
        var parts = PATH.normalizeArray(path.split('/').filter(function(p) {
          return !!p;
        }), false);
        // start at the root
        var current = FS.root;
        var current_path = '/';
        for (var i = 0; i < parts.length; i++) {
          var islast = (i === parts.length-1);
          if (islast && opts.parent) {
            // stop resolving
            break;
          }
          current = FS.lookupNode(current, parts[i]);
          current_path = PATH.join(current_path, parts[i]);
          // jump to the mount's root node if this is a mountpoint
          if (FS.isMountpoint(current)) {
            current = current.mount.root;
          }
          // follow symlinks
          // by default, lookupPath will not follow a symlink if it is the final path component.
          // setting opts.follow = true will override this behavior.
          if (!islast || opts.follow) {
            var count = 0;
            while (FS.isLink(current.mode)) {
              var link = FS.readlink(current_path);
              current_path = PATH.resolve(PATH.dirname(current_path), link);
              var lookup = FS.lookupPath(current_path, { recurse_count: opts.recurse_count });
              current = lookup.node;
              if (count++ > 40) {  // limit max consecutive symlinks to 40 (SYMLOOP_MAX).
                throw new FS.ErrnoError(ERRNO_CODES.ELOOP);
              }
            }
          }
        }
        return { path: current_path, node: current };
      },getPath:function (node) {
        var path;
        while (true) {
          if (FS.isRoot(node)) {
            return path ? PATH.join(node.mount.mountpoint, path) : node.mount.mountpoint;
          }
          path = path ? PATH.join(node.name, path) : node.name;
          node = node.parent;
        }
      },flagModes:{"r":0,"rs":8192,"r+":2,"w":1537,"wx":3585,"xw":3585,"w+":1538,"wx+":3586,"xw+":3586,"a":521,"ax":2569,"xa":2569,"a+":522,"ax+":2570,"xa+":2570},modeStringToFlags:function (str) {
        var flags = FS.flagModes[str];
        if (typeof flags === 'undefined') {
          throw new Error('Unknown file open mode: ' + str);
        }
        return flags;
      },flagsToPermissionString:function (flag) {
        var accmode = flag & 3;
        var perms = ['r', 'w', 'rw'][accmode];
        if ((flag & 1024)) {
          perms += 'w';
        }
        return perms;
      },nodePermissions:function (node, perms) {
        if (FS.ignorePermissions) {
          return 0;
        }
        // return 0 if any user, group or owner bits are set.
        if (perms.indexOf('r') !== -1 && !(node.mode & 292)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('w') !== -1 && !(node.mode & 146)) {
          return ERRNO_CODES.EACCES;
        } else if (perms.indexOf('x') !== -1 && !(node.mode & 73)) {
          return ERRNO_CODES.EACCES;
        }
        return 0;
      },mayLookup:function (dir) {
        return FS.nodePermissions(dir, 'x');
      },mayMknod:function (mode) {
        switch (mode & 0170000) {
          case 0100000:
          case 0020000:
          case 0060000:
          case 0010000:
          case 0140000:
            return 0;
          default:
            return ERRNO_CODES.EINVAL;
        }
      },mayCreate:function (dir, name) {
        try {
          var node = FS.lookupNode(dir, name);
          return ERRNO_CODES.EEXIST;
        } catch (e) {
        }
        return FS.nodePermissions(dir, 'wx');
      },mayDelete:function (dir, name, isdir) {
        var node;
        try {
          node = FS.lookupNode(dir, name);
        } catch (e) {
          return e.errno;
        }
        var err = FS.nodePermissions(dir, 'wx');
        if (err) {
          return err;
        }
        if (isdir) {
          if (!FS.isDir(node.mode)) {
            return ERRNO_CODES.ENOTDIR;
          }
          if (FS.isRoot(node) || FS.getPath(node) === FS.currentPath) {
            return ERRNO_CODES.EBUSY;
          }
        } else {
          if (FS.isDir(node.mode)) {
            return ERRNO_CODES.EISDIR;
          }
        }
        return 0;
      },mayOpen:function (node, flags) {
        if (!node) {
          return ERRNO_CODES.ENOENT;
        }
        if (FS.isLink(node.mode)) {
          return ERRNO_CODES.ELOOP;
        } else if (FS.isDir(node.mode)) {
          if ((flags & 3) !== 0 ||  // opening for write
              (flags & 1024)) {
            return ERRNO_CODES.EISDIR;
          }
        }
        return FS.nodePermissions(node, FS.flagsToPermissionString(flags));
      },chrdev_stream_ops:{open:function (stream) {
          var device = FS.getDevice(stream.node.rdev);
          // override node's stream ops with the device's
          stream.stream_ops = device.stream_ops;
          // forward the open call
          if (stream.stream_ops.open) {
            stream.stream_ops.open(stream);
          }
        },llseek:function () {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }},major:function (dev) {
        return ((dev) >> 8);
      },minor:function (dev) {
        return ((dev) & 0xff);
      },makedev:function (ma, mi) {
        return ((ma) << 8 | (mi));
      },registerDevice:function (dev, ops) {
        FS.devices[dev] = { stream_ops: ops };
      },getDevice:function (dev) {
        return FS.devices[dev];
      },MAX_OPEN_FDS:4096,nextfd:function (fd_start, fd_end) {
        fd_start = fd_start || 1;
        fd_end = fd_end || FS.MAX_OPEN_FDS;
        for (var fd = fd_start; fd <= fd_end; fd++) {
          if (!FS.streams[fd]) {
            return fd;
          }
        }
        throw new FS.ErrnoError(ERRNO_CODES.EMFILE);
      },getStream:function (fd) {
        return FS.streams[fd];
      },createStream:function (stream, fd_start, fd_end) {
        var fd = FS.nextfd(fd_start, fd_end);
        stream.fd = fd;
        // compatibility
        Object.defineProperties(stream, {
          object: {
            get: function() { return stream.node; },
            set: function(val) { stream.node = val; }
          },
          isRead: {
            get: function() { return (stream.flags & 3) !== 1; }
          },
          isWrite: {
            get: function() { return (stream.flags & 3) !== 0; }
          },
          isAppend: {
            get: function() { return (stream.flags & 8); }
          }
        });
        FS.streams[fd] = stream;
        return stream;
      },closeStream:function (fd) {
        FS.streams[fd] = null;
      },getMode:function (canRead, canWrite) {
        var mode = 0;
        if (canRead) mode |= 292 | 73;
        if (canWrite) mode |= 146;
        return mode;
      },joinPath:function (parts, forceRelative) {
        var path = PATH.join.apply(null, parts);
        if (forceRelative && path[0] == '/') path = path.substr(1);
        return path;
      },absolutePath:function (relative, base) {
        return PATH.resolve(base, relative);
      },standardizePath:function (path) {
        return PATH.normalize(path);
      },findObject:function (path, dontResolveLastLink) {
        var ret = FS.analyzePath(path, dontResolveLastLink);
        if (ret.exists) {
          return ret.object;
        } else {
          ___setErrNo(ret.error);
          return null;
        }
      },analyzePath:function (path, dontResolveLastLink) {
        // operate from within the context of the symlink's target
        try {
          var lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          path = lookup.path;
        } catch (e) {
        }
        var ret = {
          isRoot: false, exists: false, error: 0, name: null, path: null, object: null,
          parentExists: false, parentPath: null, parentObject: null
        };
        try {
          var lookup = FS.lookupPath(path, { parent: true });
          ret.parentExists = true;
          ret.parentPath = lookup.path;
          ret.parentObject = lookup.node;
          ret.name = PATH.basename(path);
          lookup = FS.lookupPath(path, { follow: !dontResolveLastLink });
          ret.exists = true;
          ret.path = lookup.path;
          ret.object = lookup.node;
          ret.name = lookup.node.name;
          ret.isRoot = lookup.path === '/';
        } catch (e) {
          ret.error = e.errno;
        };
        return ret;
      },createFolder:function (parent, name, canRead, canWrite) {
        var path = PATH.join(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.mkdir(path, mode);
      },createPath:function (parent, path, canRead, canWrite) {
        parent = typeof parent === 'string' ? parent : FS.getPath(parent);
        var parts = path.split('/').reverse();
        while (parts.length) {
          var part = parts.pop();
          if (!part) continue;
          var current = PATH.join(parent, part);
          try {
            FS.mkdir(current, 0777);
          } catch (e) {
            // ignore EEXIST
          }
          parent = current;
        }
        return current;
      },createFile:function (parent, name, properties, canRead, canWrite) {
        var path = PATH.join(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        return FS.create(path, mode);
      },createDataFile:function (parent, name, data, canRead, canWrite) {
        var path = PATH.join(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = FS.getMode(canRead, canWrite);
        var node = FS.create(path, mode);
        if (data) {
          if (typeof data === 'string') {
            var arr = new Array(data.length);
            for (var i = 0, len = data.length; i < len; ++i) arr[i] = data.charCodeAt(i);
            data = arr;
          }
          // make sure we can write to the file
          FS.chmod(path, mode | 146);
          var stream = FS.open(path, 'w');
          FS.write(stream, data, 0, data.length, 0);
          FS.close(stream);
          FS.chmod(path, mode);
        }
        return node;
      },createDevice:function (parent, name, input, output) {
        var path = PATH.join(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        var mode = input && output ? 0777 : (input ? 0333 : 0555);
        if (!FS.createDevice.major) FS.createDevice.major = 64;
        var dev = FS.makedev(FS.createDevice.major++, 0);
        // Create a fake device that a set of stream ops to emulate
        // the old behavior.
        FS.registerDevice(dev, {
          open: function(stream) {
            stream.seekable = false;
          },
          close: function(stream) {
            // flush any pending line data
            if (output && output.buffer && output.buffer.length) {
              output(10);
            }
          },
          read: function(stream, buffer, offset, length, pos /* ignored */) {
            var bytesRead = 0;
            for (var i = 0; i < length; i++) {
              var result;
              try {
                result = input();
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
              if (result === undefined && bytesRead === 0) {
                throw new FS.ErrnoError(ERRNO_CODES.EAGAIN);
              }
              if (result === null || result === undefined) break;
              bytesRead++;
              buffer[offset+i] = result;
            }
            if (bytesRead) {
              stream.node.timestamp = Date.now();
            }
            return bytesRead;
          },
          write: function(stream, buffer, offset, length, pos) {
            for (var i = 0; i < length; i++) {
              try {
                output(buffer[offset+i]);
              } catch (e) {
                throw new FS.ErrnoError(ERRNO_CODES.EIO);
              }
            }
            if (length) {
              stream.node.timestamp = Date.now();
            }
            return i;
          }
        });
        return FS.mkdev(path, mode, dev);
      },createLink:function (parent, name, target, canRead, canWrite) {
        var path = PATH.join(typeof parent === 'string' ? parent : FS.getPath(parent), name);
        return FS.symlink(target, path);
      },forceLoadFile:function (obj) {
        if (obj.isDevice || obj.isFolder || obj.link || obj.contents) return true;
        var success = true;
        if (typeof XMLHttpRequest !== 'undefined') {
          throw new Error("Lazy loading should have been performed (contents set) in createLazyFile, but it was not. Lazy loading only works in web workers. Use --embed-file or --preload-file in emcc on the main thread.");
        } else if (Module['read']) {
          // Command-line.
          try {
            // WARNING: Can't read binary files in V8's d8 or tracemonkey's js, as
            //          read() will try to parse UTF8.
            obj.contents = intArrayFromString(Module['read'](obj.url), true);
          } catch (e) {
            success = false;
          }
        } else {
          throw new Error('Cannot load without read() or XMLHttpRequest.');
        }
        if (!success) ___setErrNo(ERRNO_CODES.EIO);
        return success;
      },createLazyFile:function (parent, name, url, canRead, canWrite) {
        if (typeof XMLHttpRequest !== 'undefined') {
          if (!ENVIRONMENT_IS_WORKER) throw 'Cannot do synchronous binary XHRs outside webworkers in modern browsers. Use --embed-file or --preload-file in emcc';
          // Lazy chunked Uint8Array (implements get and length from Uint8Array). Actual getting is abstracted away for eventual reuse.
          var LazyUint8Array = function() {
            this.lengthKnown = false;
            this.chunks = []; // Loaded chunks. Index is the chunk number
          }
          LazyUint8Array.prototype.get = function(idx) {
            if (idx > this.length-1 || idx < 0) {
              return undefined;
            }
            var chunkOffset = idx % this.chunkSize;
            var chunkNum = Math.floor(idx / this.chunkSize);
            return this.getter(chunkNum)[chunkOffset];
          }
          LazyUint8Array.prototype.setDataGetter = function(getter) {
            this.getter = getter;
          }
          LazyUint8Array.prototype.cacheLength = function() {
              // Find length
              var xhr = new XMLHttpRequest();
              xhr.open('HEAD', url, false);
              xhr.send(null);
              if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
              var datalength = Number(xhr.getResponseHeader("Content-length"));
              var header;
              var hasByteServing = (header = xhr.getResponseHeader("Accept-Ranges")) && header === "bytes";
              var chunkSize = 1024*1024; // Chunk size in bytes
              if (!hasByteServing) chunkSize = datalength;
              // Function to get a range from the remote URL.
              var doXHR = (function(from, to) {
                if (from > to) throw new Error("invalid range (" + from + ", " + to + ") or no bytes requested!");
                if (to > datalength-1) throw new Error("only " + datalength + " bytes available! programmer error!");
                // TODO: Use mozResponseArrayBuffer, responseStream, etc. if available.
                var xhr = new XMLHttpRequest();
                xhr.open('GET', url, false);
                if (datalength !== chunkSize) xhr.setRequestHeader("Range", "bytes=" + from + "-" + to);
                // Some hints to the browser that we want binary data.
                if (typeof Uint8Array != 'undefined') xhr.responseType = 'arraybuffer';
                if (xhr.overrideMimeType) {
                  xhr.overrideMimeType('text/plain; charset=x-user-defined');
                }
                xhr.send(null);
                if (!(xhr.status >= 200 && xhr.status < 300 || xhr.status === 304)) throw new Error("Couldn't load " + url + ". Status: " + xhr.status);
                if (xhr.response !== undefined) {
                  return new Uint8Array(xhr.response || []);
                } else {
                  return intArrayFromString(xhr.responseText || '', true);
                }
              });
              var lazyArray = this;
              lazyArray.setDataGetter(function(chunkNum) {
                var start = chunkNum * chunkSize;
                var end = (chunkNum+1) * chunkSize - 1; // including this byte
                end = Math.min(end, datalength-1); // if datalength-1 is selected, this is the last block
                if (typeof(lazyArray.chunks[chunkNum]) === "undefined") {
                  lazyArray.chunks[chunkNum] = doXHR(start, end);
                }
                if (typeof(lazyArray.chunks[chunkNum]) === "undefined") throw new Error("doXHR failed!");
                return lazyArray.chunks[chunkNum];
              });
              this._length = datalength;
              this._chunkSize = chunkSize;
              this.lengthKnown = true;
          }
          var lazyArray = new LazyUint8Array();
          Object.defineProperty(lazyArray, "length", {
              get: function() {
                  if(!this.lengthKnown) {
                      this.cacheLength();
                  }
                  return this._length;
              }
          });
          Object.defineProperty(lazyArray, "chunkSize", {
              get: function() {
                  if(!this.lengthKnown) {
                      this.cacheLength();
                  }
                  return this._chunkSize;
              }
          });
          var properties = { isDevice: false, contents: lazyArray };
        } else {
          var properties = { isDevice: false, url: url };
        }
        var node = FS.createFile(parent, name, properties, canRead, canWrite);
        // This is a total hack, but I want to get this lazy file code out of the
        // core of MEMFS. If we want to keep this lazy file concept I feel it should
        // be its own thin LAZYFS proxying calls to MEMFS.
        if (properties.contents) {
          node.contents = properties.contents;
        } else if (properties.url) {
          node.contents = null;
          node.url = properties.url;
        }
        // override each stream op with one that tries to force load the lazy file first
        var stream_ops = {};
        var keys = Object.keys(node.stream_ops);
        keys.forEach(function(key) {
          var fn = node.stream_ops[key];
          stream_ops[key] = function() {
            if (!FS.forceLoadFile(node)) {
              throw new FS.ErrnoError(ERRNO_CODES.EIO);
            }
            return fn.apply(null, arguments);
          };
        });
        // use a custom read function
        stream_ops.read = function(stream, buffer, offset, length, position) {
          if (!FS.forceLoadFile(node)) {
            throw new FS.ErrnoError(ERRNO_CODES.EIO);
          }
          var contents = stream.node.contents;
          var size = Math.min(contents.length - position, length);
          if (contents.slice) { // normal array
            for (var i = 0; i < size; i++) {
              buffer[offset + i] = contents[position + i];
            }
          } else {
            for (var i = 0; i < size; i++) { // LazyUint8Array from sync binary XHR
              buffer[offset + i] = contents.get(position + i);
            }
          }
          return size;
        };
        node.stream_ops = stream_ops;
        return node;
      },createPreloadedFile:function (parent, name, url, canRead, canWrite, onload, onerror, dontCreateFile) {
        Browser.init();
        // TODO we should allow people to just pass in a complete filename instead
        // of parent and name being that we just join them anyways
        var fullname = PATH.resolve(PATH.join(parent, name));
        function processData(byteArray) {
          function finish(byteArray) {
            if (!dontCreateFile) {
              FS.createDataFile(parent, name, byteArray, canRead, canWrite);
            }
            if (onload) onload();
            removeRunDependency('cp ' + fullname);
          }
          var handled = false;
          Module['preloadPlugins'].forEach(function(plugin) {
            if (handled) return;
            if (plugin['canHandle'](fullname)) {
              plugin['handle'](byteArray, fullname, finish, function() {
                if (onerror) onerror();
                removeRunDependency('cp ' + fullname);
              });
              handled = true;
            }
          });
          if (!handled) finish(byteArray);
        }
        addRunDependency('cp ' + fullname);
        if (typeof url == 'string') {
          Browser.asyncLoad(url, function(byteArray) {
            processData(byteArray);
          }, onerror);
        } else {
          processData(url);
        }
      },createDefaultDirectories:function () {
        FS.mkdir('/tmp', 0777);
      },createDefaultDevices:function () {
        // create /dev
        FS.mkdir('/dev', 0777);
        // setup /dev/null
        FS.registerDevice(FS.makedev(1, 3), {
          read: function() { return 0; },
          write: function() { return 0; }
        });
        FS.mkdev('/dev/null', 0666, FS.makedev(1, 3));
        // setup /dev/tty and /dev/tty1
        // stderr needs to print output using Module['printErr']
        // so we register a second tty just for it.
        TTY.register(FS.makedev(5, 0), TTY.default_tty_ops);
        TTY.register(FS.makedev(6, 0), TTY.default_tty1_ops);
        FS.mkdev('/dev/tty', 0666, FS.makedev(5, 0));
        FS.mkdev('/dev/tty1', 0666, FS.makedev(6, 0));
        // we're not going to emulate the actual shm device,
        // just create the tmp dirs that reside in it commonly
        FS.mkdir('/dev/shm', 0777);
        FS.mkdir('/dev/shm/tmp', 0777);
      },createStandardStreams:function () {
        // TODO deprecate the old functionality of a single
        // input / output callback and that utilizes FS.createDevice
        // and instead require a unique set of stream ops
        // by default, we symlink the standard streams to the
        // default tty devices. however, if the standard streams
        // have been overwritten we create a unique device for
        // them instead.
        if (Module['stdin']) {
          FS.createDevice('/dev', 'stdin', Module['stdin']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdin');
        }
        if (Module['stdout']) {
          FS.createDevice('/dev', 'stdout', null, Module['stdout']);
        } else {
          FS.symlink('/dev/tty', '/dev/stdout');
        }
        if (Module['stderr']) {
          FS.createDevice('/dev', 'stderr', null, Module['stderr']);
        } else {
          FS.symlink('/dev/tty1', '/dev/stderr');
        }
        // open default streams for the stdin, stdout and stderr devices
        var stdin = FS.open('/dev/stdin', 'r');
        HEAP32[((_stdin)>>2)]=stdin.fd;
        assert(stdin.fd === 1, 'invalid handle for stdin (' + stdin.fd + ')');
        var stdout = FS.open('/dev/stdout', 'w');
        HEAP32[((_stdout)>>2)]=stdout.fd;
        assert(stdout.fd === 2, 'invalid handle for stdout (' + stdout.fd + ')');
        var stderr = FS.open('/dev/stderr', 'w');
        HEAP32[((_stderr)>>2)]=stderr.fd;
        assert(stderr.fd === 3, 'invalid handle for stderr (' + stderr.fd + ')');
      },staticInit:function () {
        FS.name_table = new Array(4096);
        FS.root = FS.createNode(null, '/', 0040000 | 0777, 0);
        FS.mount(MEMFS, {}, '/');
        FS.createDefaultDirectories();
        FS.createDefaultDevices();
      },init:function (input, output, error) {
        assert(!FS.init.initialized, 'FS.init was previously called. If you want to initialize later with custom parameters, remove any earlier calls (note that one is automatically added to the generated code)');
        FS.init.initialized = true;
        // Allow Module.stdin etc. to provide defaults, if none explicitly passed to us here
        Module['stdin'] = input || Module['stdin'];
        Module['stdout'] = output || Module['stdout'];
        Module['stderr'] = error || Module['stderr'];
        FS.createStandardStreams();
      },quit:function () {
        FS.init.initialized = false;
        for (var i = 0; i < FS.streams.length; i++) {
          var stream = FS.streams[i];
          if (!stream) {
            continue;
          }
          FS.close(stream);
        }
      },mount:function (type, opts, mountpoint) {
        var mount = {
          type: type,
          opts: opts,
          mountpoint: mountpoint,
          root: null
        };
        var lookup;
        if (mountpoint) {
          lookup = FS.lookupPath(mountpoint, { follow: false });
        }
        // create a root node for the fs
        var root = type.mount(mount);
        root.mount = mount;
        mount.root = root;
        // assign the mount info to the mountpoint's node
        if (lookup) {
          lookup.node.mount = mount;
          lookup.node.mounted = true;
          // compatibility update FS.root if we mount to /
          if (mountpoint === '/') {
            FS.root = mount.root;
          }
        }
        return root;
      },lookup:function (parent, name) {
        return parent.node_ops.lookup(parent, name);
      },mknod:function (path, mode, dev) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var err = FS.mayCreate(parent, name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.mknod) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.mknod(parent, name, mode, dev);
      },create:function (path, mode) {
        mode &= 4095;
        mode |= 0100000;
        return FS.mknod(path, mode, 0);
      },mkdir:function (path, mode) {
        mode &= 511 | 0001000;
        mode |= 0040000;
        return FS.mknod(path, mode, 0);
      },mkdev:function (path, mode, dev) {
        mode |= 0020000;
        return FS.mknod(path, mode, dev);
      },symlink:function (oldpath, newpath) {
        var lookup = FS.lookupPath(newpath, { parent: true });
        var parent = lookup.node;
        var newname = PATH.basename(newpath);
        var err = FS.mayCreate(parent, newname);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.symlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return parent.node_ops.symlink(parent, newname, oldpath);
      },rename:function (old_path, new_path) {
        var old_dirname = PATH.dirname(old_path);
        var new_dirname = PATH.dirname(new_path);
        var old_name = PATH.basename(old_path);
        var new_name = PATH.basename(new_path);
        // parents must exist
        var lookup, old_dir, new_dir;
        try {
          lookup = FS.lookupPath(old_path, { parent: true });
          old_dir = lookup.node;
          lookup = FS.lookupPath(new_path, { parent: true });
          new_dir = lookup.node;
        } catch (e) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        // need to be part of the same mount
        if (old_dir.mount !== new_dir.mount) {
          throw new FS.ErrnoError(ERRNO_CODES.EXDEV);
        }
        // source must exist
        var old_node = FS.lookupNode(old_dir, old_name);
        // old path should not be an ancestor of the new path
        var relative = PATH.relative(old_path, new_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        // new path should not be an ancestor of the old path
        relative = PATH.relative(new_path, old_dirname);
        if (relative.charAt(0) !== '.') {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTEMPTY);
        }
        // see if the new path already exists
        var new_node;
        try {
          new_node = FS.lookupNode(new_dir, new_name);
        } catch (e) {
          // not fatal
        }
        // early out if nothing needs to change
        if (old_node === new_node) {
          return;
        }
        // we'll need to delete the old entry
        var isdir = FS.isDir(old_node.mode);
        var err = FS.mayDelete(old_dir, old_name, isdir);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        // need delete permissions if we'll be overwriting.
        // need create permissions if new doesn't already exist.
        err = new_node ?
          FS.mayDelete(new_dir, new_name, isdir) :
          FS.mayCreate(new_dir, new_name);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!old_dir.node_ops.rename) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(old_node) || (new_node && FS.isMountpoint(new_node))) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        // if we are going to change the parent, check write permissions
        if (new_dir !== old_dir) {
          err = FS.nodePermissions(old_dir, 'w');
          if (err) {
            throw new FS.ErrnoError(err);
          }
        }
        // remove the node from the lookup hash
        FS.hashRemoveNode(old_node);
        // do the underlying fs rename
        try {
          old_dir.node_ops.rename(old_node, new_dir, new_name);
        } catch (e) {
          throw e;
        } finally {
          // add the node back to the hash (in case node_ops.rename
          // changed its name)
          FS.hashAddNode(old_node);
        }
      },rmdir:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, true);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.rmdir) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        parent.node_ops.rmdir(parent, name);
        FS.destroyNode(node);
      },readdir:function (path) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        if (!node.node_ops.readdir) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOTDIR);
        }
        return node.node_ops.readdir(node);
      },unlink:function (path) {
        var lookup = FS.lookupPath(path, { parent: true });
        var parent = lookup.node;
        var name = PATH.basename(path);
        var node = FS.lookupNode(parent, name);
        var err = FS.mayDelete(parent, name, false);
        if (err) {
          // POSIX says unlink should set EPERM, not EISDIR
          if (err === ERRNO_CODES.EISDIR) err = ERRNO_CODES.EPERM;
          throw new FS.ErrnoError(err);
        }
        if (!parent.node_ops.unlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isMountpoint(node)) {
          throw new FS.ErrnoError(ERRNO_CODES.EBUSY);
        }
        parent.node_ops.unlink(parent, name);
        FS.destroyNode(node);
      },readlink:function (path) {
        var lookup = FS.lookupPath(path, { follow: false });
        var link = lookup.node;
        if (!link.node_ops.readlink) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        return link.node_ops.readlink(link);
      },stat:function (path, dontFollow) {
        var lookup = FS.lookupPath(path, { follow: !dontFollow });
        var node = lookup.node;
        if (!node.node_ops.getattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        return node.node_ops.getattr(node);
      },lstat:function (path) {
        return FS.stat(path, true);
      },chmod:function (path, mode, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          mode: (mode & 4095) | (node.mode & ~4095),
          timestamp: Date.now()
        });
      },lchmod:function (path, mode) {
        FS.chmod(path, mode, true);
      },fchmod:function (fd, mode) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chmod(stream.node, mode);
      },chown:function (path, uid, gid, dontFollow) {
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: !dontFollow });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        node.node_ops.setattr(node, {
          timestamp: Date.now()
          // we ignore the uid / gid for now
        });
      },lchown:function (path, uid, gid) {
        FS.chown(path, uid, gid, true);
      },fchown:function (fd, uid, gid) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        FS.chown(stream.node, uid, gid);
      },truncate:function (path, len) {
        if (len < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var node;
        if (typeof path === 'string') {
          var lookup = FS.lookupPath(path, { follow: true });
          node = lookup.node;
        } else {
          node = path;
        }
        if (!node.node_ops.setattr) {
          throw new FS.ErrnoError(ERRNO_CODES.EPERM);
        }
        if (FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!FS.isFile(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var err = FS.nodePermissions(node, 'w');
        if (err) {
          throw new FS.ErrnoError(err);
        }
        node.node_ops.setattr(node, {
          size: len,
          timestamp: Date.now()
        });
      },ftruncate:function (fd, len) {
        var stream = FS.getStream(fd);
        if (!stream) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if ((stream.flags & 3) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        FS.truncate(stream.node, len);
      },utime:function (path, atime, mtime) {
        var lookup = FS.lookupPath(path, { follow: true });
        var node = lookup.node;
        node.node_ops.setattr(node, {
          timestamp: Math.max(atime, mtime)
        });
      },open:function (path, flags, mode, fd_start, fd_end) {
        path = PATH.normalize(path);
        flags = typeof flags === 'string' ? FS.modeStringToFlags(flags) : flags;
        mode = typeof mode === 'undefined' ? 0666 : mode;
        if ((flags & 512)) {
          mode = (mode & 4095) | 0100000;
        } else {
          mode = 0;
        }
        var node;
        try {
          var lookup = FS.lookupPath(path, {
            follow: !(flags & 0200000)
          });
          node = lookup.node;
          path = lookup.path;
        } catch (e) {
          // ignore
        }
        // perhaps we need to create the node
        if ((flags & 512)) {
          if (node) {
            // if O_CREAT and O_EXCL are set, error out if the node already exists
            if ((flags & 2048)) {
              throw new FS.ErrnoError(ERRNO_CODES.EEXIST);
            }
          } else {
            // node doesn't exist, try to create it
            node = FS.mknod(path, mode, 0);
          }
        }
        if (!node) {
          throw new FS.ErrnoError(ERRNO_CODES.ENOENT);
        }
        // can't truncate a device
        if (FS.isChrdev(node.mode)) {
          flags &= ~1024;
        }
        // check permissions
        var err = FS.mayOpen(node, flags);
        if (err) {
          throw new FS.ErrnoError(err);
        }
        // do truncation if necessary
        if ((flags & 1024)) {
          FS.truncate(node, 0);
        }
        // register the stream with the filesystem
        var stream = FS.createStream({
          path: path,
          node: node,
          flags: flags,
          seekable: true,
          position: 0,
          stream_ops: node.stream_ops,
          // used by the file family libc calls (fopen, fwrite, ferror, etc.)
          ungotten: [],
          error: false
        }, fd_start, fd_end);
        // call the new stream's open function
        if (stream.stream_ops.open) {
          stream.stream_ops.open(stream);
        }
        return stream;
      },close:function (stream) {
        try {
          if (stream.stream_ops.close) {
            stream.stream_ops.close(stream);
          }
        } catch (e) {
          throw e;
        } finally {
          FS.closeStream(stream.fd);
        }
      },llseek:function (stream, offset, whence) {
        if (!stream.seekable || !stream.stream_ops.llseek) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        return stream.stream_ops.llseek(stream, offset, whence);
      },read:function (stream, buffer, offset, length, position) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 3) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.read) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var seeking = true;
        if (typeof position === 'undefined') {
          position = stream.position;
          seeking = false;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        var bytesRead = stream.stream_ops.read(stream, buffer, offset, length, position);
        if (!seeking) stream.position += bytesRead;
        return bytesRead;
      },write:function (stream, buffer, offset, length, position) {
        if (length < 0 || position < 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 3) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (FS.isDir(stream.node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.EISDIR);
        }
        if (!stream.stream_ops.write) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        var seeking = true;
        if (typeof position === 'undefined') {
          position = stream.position;
          seeking = false;
        } else if (!stream.seekable) {
          throw new FS.ErrnoError(ERRNO_CODES.ESPIPE);
        }
        if (stream.flags & 8) {
          // seek to the end before writing in append mode
          FS.llseek(stream, 0, 2);
        }
        var bytesWritten = stream.stream_ops.write(stream, buffer, offset, length, position);
        if (!seeking) stream.position += bytesWritten;
        return bytesWritten;
      },allocate:function (stream, offset, length) {
        if (offset < 0 || length <= 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EINVAL);
        }
        if ((stream.flags & 3) === 0) {
          throw new FS.ErrnoError(ERRNO_CODES.EBADF);
        }
        if (!FS.isFile(stream.node.mode) && !FS.isDir(node.mode)) {
          throw new FS.ErrnoError(ERRNO_CODES.ENODEV);
        }
        if (!stream.stream_ops.allocate) {
          throw new FS.ErrnoError(ERRNO_CODES.EOPNOTSUPP);
        }
        stream.stream_ops.allocate(stream, offset, length);
      },mmap:function (stream, buffer, offset, length, position, prot, flags) {
        // TODO if PROT is PROT_WRITE, make sure we have write access
        if ((stream.flags & 3) === 1) {
          throw new FS.ErrnoError(ERRNO_CODES.EACCES);
        }
        if (!stream.stream_ops.mmap) {
          throw new FS.errnoError(ERRNO_CODES.ENODEV);
        }
        return stream.stream_ops.mmap(stream, buffer, offset, length, position, prot, flags);
      }};
  var ___dirent_struct_layout={__size__:1040,d_ino:0,d_name:4,d_off:1028,d_reclen:1032,d_type:1036};function _open(path, oflag, varargs) {
      // int open(const char *path, int oflag, ...);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/open.html
      var mode = HEAP32[((varargs)>>2)];
      path = Pointer_stringify(path);
      try {
        var stream = FS.open(path, oflag, mode);
        return stream.fd;
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _fopen(filename, mode) {
      // FILE *fopen(const char *restrict filename, const char *restrict mode);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fopen.html
      var flags;
      mode = Pointer_stringify(mode);
      if (mode[0] == 'r') {
        if (mode.indexOf('+') != -1) {
          flags = 2;
        } else {
          flags = 0;
        }
      } else if (mode[0] == 'w') {
        if (mode.indexOf('+') != -1) {
          flags = 2;
        } else {
          flags = 1;
        }
        flags |= 512;
        flags |= 1024;
      } else if (mode[0] == 'a') {
        if (mode.indexOf('+') != -1) {
          flags = 2;
        } else {
          flags = 1;
        }
        flags |= 512;
        flags |= 8;
      } else {
        ___setErrNo(ERRNO_CODES.EINVAL);
        return 0;
      }
      var ret = _open(filename, flags, allocate([0x1FF, 0, 0, 0], 'i32', ALLOC_STACK));  // All creation permissions.
      return (ret == -1) ? 0 : ret;
    }
  function _send(fd, buf, len, flags) {
      var info = FS.getStream(fd);
      if (!info) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      if (info.socket.readyState === WebSocket.CLOSING || info.socket.readyState === WebSocket.CLOSED) {
        ___setErrNo(ERRNO_CODES.ENOTCONN);
        return -1;
      } else if (info.socket.readyState === WebSocket.CONNECTING) {
        ___setErrNo(ERRNO_CODES.EAGAIN);
        return -1;
      }
      info.sender(HEAPU8.subarray(buf, buf+len));
      return len;
    }
  function _pwrite(fildes, buf, nbyte, offset) {
      // ssize_t pwrite(int fildes, const void *buf, size_t nbyte, off_t offset);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/write.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      try {
        var slab = HEAP8;
        return FS.write(stream, slab, buf, nbyte, offset);
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _write(fildes, buf, nbyte) {
      // ssize_t write(int fildes, const void *buf, size_t nbyte);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/write.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      if (stream && ('socket' in stream)) {
        return _send(fildes, buf, nbyte, 0);
      }
      try {
        var slab = HEAP8;
        return FS.write(stream, slab, buf, nbyte);
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _fwrite(ptr, size, nitems, stream) {
      // size_t fwrite(const void *restrict ptr, size_t size, size_t nitems, FILE *restrict stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fwrite.html
      var bytesToWrite = nitems * size;
      if (bytesToWrite == 0) return 0;
      var bytesWritten = _write(stream, ptr, bytesToWrite);
      if (bytesWritten == -1) {
        var streamObj = FS.getStream(stream);
        if (streamObj) streamObj.error = true;
        return 0;
      } else {
        return Math.floor(bytesWritten / size);
      }
    }
  function _close(fildes) {
      // int close(int fildes);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/close.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      try {
        FS.close(stream);
        return 0;
      } catch (e) {
        FS.handleFSError(e);;
        return -1;
      }
    }
  function _fsync(fildes) {
      // int fsync(int fildes);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fsync.html
      var stream = FS.getStream(fildes);
      if (stream) {
        // We write directly to the file system, so there's nothing to do here.
        return 0;
      } else {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
    }function _fclose(stream) {
      // int fclose(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fclose.html
      _fsync(stream);
      return _close(stream);
    }
  Module["_memcpy"] = _memcpy;var _llvm_memcpy_p0i8_p0i8_i32=_memcpy;
  Module["_strlen"] = _strlen;function _strdup(ptr) {
      var len = _strlen(ptr);
      var newStr = _malloc(len + 1);
      (_memcpy(newStr, ptr, len)|0);
      HEAP8[(((newStr)+(len))|0)]=0;
      return newStr;
    }
  function _strncmp(px, py, n) {
      var i = 0;
      while (i < n) {
        var x = HEAPU8[(((px)+(i))|0)];
        var y = HEAPU8[(((py)+(i))|0)];
        if (x == y && x == 0) return 0;
        if (x == 0) return -1;
        if (y == 0) return 1;
        if (x == y) {
          i ++;
          continue;
        } else {
          return x > y ? 1 : -1;
        }
      }
      return 0;
    }function _strcmp(px, py) {
      return _strncmp(px, py, TOTAL_MEMORY);
    }
  var ___stat_struct_layout={__size__:68,st_dev:0,st_ino:4,st_mode:8,st_nlink:12,st_uid:16,st_gid:20,st_rdev:24,st_size:28,st_atime:32,st_spare1:36,st_mtime:40,st_spare2:44,st_ctime:48,st_spare3:52,st_blksize:56,st_blocks:60,st_spare4:64};function _stat(path, buf, dontResolveLastLink) {
      // http://pubs.opengroup.org/onlinepubs/7908799/xsh/stat.html
      // int stat(const char *path, struct stat *buf);
      // NOTE: dontResolveLastLink is a shortcut for lstat(). It should never be
      //       used in client code.
      path = typeof path !== 'string' ? Pointer_stringify(path) : path;
      try {
        var stat = dontResolveLastLink ? FS.lstat(path) : FS.stat(path);
        HEAP32[(((buf)+(___stat_struct_layout.st_dev))>>2)]=stat.dev;
        HEAP32[(((buf)+(___stat_struct_layout.st_ino))>>2)]=stat.ino
        HEAP32[(((buf)+(___stat_struct_layout.st_mode))>>2)]=stat.mode
        HEAP32[(((buf)+(___stat_struct_layout.st_nlink))>>2)]=stat.nlink
        HEAP32[(((buf)+(___stat_struct_layout.st_uid))>>2)]=stat.uid
        HEAP32[(((buf)+(___stat_struct_layout.st_gid))>>2)]=stat.gid
        HEAP32[(((buf)+(___stat_struct_layout.st_rdev))>>2)]=stat.rdev
        HEAP32[(((buf)+(___stat_struct_layout.st_size))>>2)]=stat.size
        HEAP32[(((buf)+(___stat_struct_layout.st_atime))>>2)]=Math.floor(stat.atime.getTime() / 1000)
        HEAP32[(((buf)+(___stat_struct_layout.st_mtime))>>2)]=Math.floor(stat.mtime.getTime() / 1000)
        HEAP32[(((buf)+(___stat_struct_layout.st_ctime))>>2)]=Math.floor(stat.ctime.getTime() / 1000)
        HEAP32[(((buf)+(___stat_struct_layout.st_blksize))>>2)]=4096
        HEAP32[(((buf)+(___stat_struct_layout.st_blocks))>>2)]=stat.blocks
        return 0;
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }
  function ___errno_location() {
      return ___errno_state;
    }var ___errno=___errno_location;
  function _fstat(fildes, buf) {
      // int fstat(int fildes, struct stat *buf);
      // http://pubs.opengroup.org/onlinepubs/7908799/xsh/fstat.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      return _stat(stream.path, buf);
    }
  function _fileno(stream) {
      // int fileno(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fileno.html
      // We use file descriptor numbers and FILE* streams interchangeably.
      return stream;
    }
  function _lseek(fildes, offset, whence) {
      // off_t lseek(int fildes, off_t offset, int whence);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/lseek.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      try {
        return FS.llseek(stream, offset, whence);
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _fseek(stream, offset, whence) {
      // int fseek(FILE *stream, long offset, int whence);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fseek.html
      var ret = _lseek(stream, offset, whence);
      if (ret == -1) {
        return -1;
      }
      stream = FS.getStream(stream);
      stream.eof = false;
      return 0;
    }
  Module["_strncpy"] = _strncpy;
  function _recv(fd, buf, len, flags) {
      var info = FS.getStream(fd);
      if (!info) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      if (!info.hasData()) {
        if (info.socket.readyState === WebSocket.CLOSING || info.socket.readyState === WebSocket.CLOSED) {
          // socket has closed
          return 0;
        } else {
          // else, our socket is in a valid state but truly has nothing available
          ___setErrNo(ERRNO_CODES.EAGAIN);
          return -1;
        }
      }
      var buffer = info.inQueue.shift();
      if (len < buffer.length) {
        if (info.stream) {
          // This is tcp (reliable), so if not all was read, keep it
          info.inQueue.unshift(buffer.subarray(len));
        }
        buffer = buffer.subarray(0, len);
      }
      HEAPU8.set(buffer, buf);
      return buffer.length;
    }
  function _pread(fildes, buf, nbyte, offset) {
      // ssize_t pread(int fildes, void *buf, size_t nbyte, off_t offset);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/read.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      try {
        var slab = HEAP8;
        return FS.read(stream, slab, buf, nbyte, offset);
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _read(fildes, buf, nbyte) {
      // ssize_t read(int fildes, void *buf, size_t nbyte);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/read.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      if (stream && ('socket' in stream)) {
        return _recv(fildes, buf, nbyte, 0);
      }
      try {
        var slab = HEAP8;
        return FS.read(stream, slab, buf, nbyte);
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }function _fread(ptr, size, nitems, stream) {
      // size_t fread(void *restrict ptr, size_t size, size_t nitems, FILE *restrict stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fread.html
      var bytesToRead = nitems * size;
      if (bytesToRead == 0) {
        return 0;
      }
      var bytesRead = 0;
      var streamObj = FS.getStream(stream);
      while (streamObj.ungotten.length && bytesToRead > 0) {
        HEAP8[((ptr++)|0)]=streamObj.ungotten.pop()
        bytesToRead--;
        bytesRead++;
      }
      var err = _read(stream, ptr, bytesToRead);
      if (err == -1) {
        if (streamObj) streamObj.error = true;
        return 0;
      }
      bytesRead += err;
      if (bytesRead < bytesToRead) streamObj.eof = true;
      return Math.floor(bytesRead / size);
    }
  function _strrchr(ptr, chr) {
      var ptr2 = ptr + _strlen(ptr);
      do {
        if (HEAP8[(ptr2)] == chr) return ptr2;
        ptr2--;
      } while (ptr2 >= ptr);
      return 0;
    }
  function _unlink(path) {
      // int unlink(const char *path);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/unlink.html
      path = Pointer_stringify(path);
      try {
        FS.unlink(path);
        return 0;
      } catch (e) {
        FS.handleFSError(e);
        return -1;
      }
    }
  function _strncat(pdest, psrc, num) {
      var len = _strlen(pdest);
      var i = 0;
      while(1) {
        HEAP8[((pdest+len+i)|0)]=HEAP8[((psrc+i)|0)];
        if (HEAP8[(((pdest)+(len+i))|0)] == 0) break;
        i ++;
        if (i == num) {
          HEAP8[(((pdest)+(len+i))|0)]=0
          break;
        }
      }
      return pdest;
    }
  Module["_memcmp"] = _memcmp;
  function _isprint(chr) {
      return 0x1F < chr && chr < 0x7F;
    }
  function _creat(path, mode) {
      // int creat(const char *path, mode_t mode);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/creat.html
      return _open(path, 1 | 512 | 1024, allocate([mode, 0, 0, 0], 'i32', ALLOC_STACK));
    }function _mkstemp(template) {
      if (!_mkstemp.counter) _mkstemp.counter = 0;
      var c = (_mkstemp.counter++).toString();
      var rep = 'XXXXXX';
      while (c.length < rep.length) c = '0' + c;
      writeArrayToMemory(intArrayFromString(c), template + Pointer_stringify(template).indexOf(rep));
      return _creat(template, 0600);
    }
  function _fdopen(fildes, mode) {
      // FILE *fdopen(int fildes, const char *mode);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fdopen.html
      mode = Pointer_stringify(mode);
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return 0;
      }
      if ((mode.indexOf('w') != -1 && !stream.isWrite) ||
          (mode.indexOf('r') != -1 && !stream.isRead) ||
          (mode.indexOf('a') != -1 && !stream.isAppend) ||
          (mode.indexOf('+') != -1 && (!stream.isRead || !stream.isWrite))) {
        ___setErrNo(ERRNO_CODES.EINVAL);
        return 0;
      } else {
        stream.error = false;
        stream.eof = false;
        return fildes;
      }
    }
  function __reallyNegative(x) {
      return x < 0 || (x === 0 && (1/x) === -Infinity);
    }function __formatString(format, varargs) {
      var textIndex = format;
      var argIndex = 0;
      function getNextArg(type) {
        // NOTE: Explicitly ignoring type safety. Otherwise this fails:
        //       int x = 4; printf("%c\n", (char)x);
        var ret;
        if (type === 'double') {
          ret = HEAPF64[(((varargs)+(argIndex))>>3)];
        } else if (type == 'i64') {
          ret = [HEAP32[(((varargs)+(argIndex))>>2)],
                 HEAP32[(((varargs)+(argIndex+8))>>2)]];
          argIndex += 8; // each 32-bit chunk is in a 64-bit block
        } else {
          type = 'i32'; // varargs are always i32, i64, or double
          ret = HEAP32[(((varargs)+(argIndex))>>2)];
        }
        argIndex += Math.max(Runtime.getNativeFieldSize(type), Runtime.getAlignSize(type, null, true));
        return ret;
      }
      var ret = [];
      var curr, next, currArg;
      while(1) {
        var startTextIndex = textIndex;
        curr = HEAP8[(textIndex)];
        if (curr === 0) break;
        next = HEAP8[((textIndex+1)|0)];
        if (curr == 37) {
          // Handle flags.
          var flagAlwaysSigned = false;
          var flagLeftAlign = false;
          var flagAlternative = false;
          var flagZeroPad = false;
          flagsLoop: while (1) {
            switch (next) {
              case 43:
                flagAlwaysSigned = true;
                break;
              case 45:
                flagLeftAlign = true;
                break;
              case 35:
                flagAlternative = true;
                break;
              case 48:
                if (flagZeroPad) {
                  break flagsLoop;
                } else {
                  flagZeroPad = true;
                  break;
                }
              default:
                break flagsLoop;
            }
            textIndex++;
            next = HEAP8[((textIndex+1)|0)];
          }
          // Handle width.
          var width = 0;
          if (next == 42) {
            width = getNextArg('i32');
            textIndex++;
            next = HEAP8[((textIndex+1)|0)];
          } else {
            while (next >= 48 && next <= 57) {
              width = width * 10 + (next - 48);
              textIndex++;
              next = HEAP8[((textIndex+1)|0)];
            }
          }
          // Handle precision.
          var precisionSet = false;
          if (next == 46) {
            var precision = 0;
            precisionSet = true;
            textIndex++;
            next = HEAP8[((textIndex+1)|0)];
            if (next == 42) {
              precision = getNextArg('i32');
              textIndex++;
            } else {
              while(1) {
                var precisionChr = HEAP8[((textIndex+1)|0)];
                if (precisionChr < 48 ||
                    precisionChr > 57) break;
                precision = precision * 10 + (precisionChr - 48);
                textIndex++;
              }
            }
            next = HEAP8[((textIndex+1)|0)];
          } else {
            var precision = 6; // Standard default.
          }
          // Handle integer sizes. WARNING: These assume a 32-bit architecture!
          var argSize;
          switch (String.fromCharCode(next)) {
            case 'h':
              var nextNext = HEAP8[((textIndex+2)|0)];
              if (nextNext == 104) {
                textIndex++;
                argSize = 1; // char (actually i32 in varargs)
              } else {
                argSize = 2; // short (actually i32 in varargs)
              }
              break;
            case 'l':
              var nextNext = HEAP8[((textIndex+2)|0)];
              if (nextNext == 108) {
                textIndex++;
                argSize = 8; // long long
              } else {
                argSize = 4; // long
              }
              break;
            case 'L': // long long
            case 'q': // int64_t
            case 'j': // intmax_t
              argSize = 8;
              break;
            case 'z': // size_t
            case 't': // ptrdiff_t
            case 'I': // signed ptrdiff_t or unsigned size_t
              argSize = 4;
              break;
            default:
              argSize = null;
          }
          if (argSize) textIndex++;
          next = HEAP8[((textIndex+1)|0)];
          // Handle type specifier.
          switch (String.fromCharCode(next)) {
            case 'd': case 'i': case 'u': case 'o': case 'x': case 'X': case 'p': {
              // Integer.
              var signed = next == 100 || next == 105;
              argSize = argSize || 4;
              var currArg = getNextArg('i' + (argSize * 8));
              var origArg = currArg;
              var argText;
              // Flatten i64-1 [low, high] into a (slightly rounded) double
              if (argSize == 8) {
                currArg = Runtime.makeBigInt(currArg[0], currArg[1], next == 117);
              }
              // Truncate to requested size.
              if (argSize <= 4) {
                var limit = Math.pow(256, argSize) - 1;
                currArg = (signed ? reSign : unSign)(currArg & limit, argSize * 8);
              }
              // Format the number.
              var currAbsArg = Math.abs(currArg);
              var prefix = '';
              if (next == 100 || next == 105) {
                if (argSize == 8 && i64Math) argText = i64Math.stringify(origArg[0], origArg[1], null); else
                argText = reSign(currArg, 8 * argSize, 1).toString(10);
              } else if (next == 117) {
                if (argSize == 8 && i64Math) argText = i64Math.stringify(origArg[0], origArg[1], true); else
                argText = unSign(currArg, 8 * argSize, 1).toString(10);
                currArg = Math.abs(currArg);
              } else if (next == 111) {
                argText = (flagAlternative ? '0' : '') + currAbsArg.toString(8);
              } else if (next == 120 || next == 88) {
                prefix = (flagAlternative && currArg != 0) ? '0x' : '';
                if (argSize == 8 && i64Math) {
                  if (origArg[1]) {
                    argText = (origArg[1]>>>0).toString(16);
                    var lower = (origArg[0]>>>0).toString(16);
                    while (lower.length < 8) lower = '0' + lower;
                    argText += lower;
                  } else {
                    argText = (origArg[0]>>>0).toString(16);
                  }
                } else
                if (currArg < 0) {
                  // Represent negative numbers in hex as 2's complement.
                  currArg = -currArg;
                  argText = (currAbsArg - 1).toString(16);
                  var buffer = [];
                  for (var i = 0; i < argText.length; i++) {
                    buffer.push((0xF - parseInt(argText[i], 16)).toString(16));
                  }
                  argText = buffer.join('');
                  while (argText.length < argSize * 2) argText = 'f' + argText;
                } else {
                  argText = currAbsArg.toString(16);
                }
                if (next == 88) {
                  prefix = prefix.toUpperCase();
                  argText = argText.toUpperCase();
                }
              } else if (next == 112) {
                if (currAbsArg === 0) {
                  argText = '(nil)';
                } else {
                  prefix = '0x';
                  argText = currAbsArg.toString(16);
                }
              }
              if (precisionSet) {
                while (argText.length < precision) {
                  argText = '0' + argText;
                }
              }
              // Add sign if needed
              if (flagAlwaysSigned) {
                if (currArg < 0) {
                  prefix = '-' + prefix;
                } else {
                  prefix = '+' + prefix;
                }
              }
              // Add padding.
              while (prefix.length + argText.length < width) {
                if (flagLeftAlign) {
                  argText += ' ';
                } else {
                  if (flagZeroPad) {
                    argText = '0' + argText;
                  } else {
                    prefix = ' ' + prefix;
                  }
                }
              }
              // Insert the result into the buffer.
              argText = prefix + argText;
              argText.split('').forEach(function(chr) {
                ret.push(chr.charCodeAt(0));
              });
              break;
            }
            case 'f': case 'F': case 'e': case 'E': case 'g': case 'G': {
              // Float.
              var currArg = getNextArg('double');
              var argText;
              if (isNaN(currArg)) {
                argText = 'nan';
                flagZeroPad = false;
              } else if (!isFinite(currArg)) {
                argText = (currArg < 0 ? '-' : '') + 'inf';
                flagZeroPad = false;
              } else {
                var isGeneral = false;
                var effectivePrecision = Math.min(precision, 20);
                // Convert g/G to f/F or e/E, as per:
                // http://pubs.opengroup.org/onlinepubs/9699919799/functions/printf.html
                if (next == 103 || next == 71) {
                  isGeneral = true;
                  precision = precision || 1;
                  var exponent = parseInt(currArg.toExponential(effectivePrecision).split('e')[1], 10);
                  if (precision > exponent && exponent >= -4) {
                    next = ((next == 103) ? 'f' : 'F').charCodeAt(0);
                    precision -= exponent + 1;
                  } else {
                    next = ((next == 103) ? 'e' : 'E').charCodeAt(0);
                    precision--;
                  }
                  effectivePrecision = Math.min(precision, 20);
                }
                if (next == 101 || next == 69) {
                  argText = currArg.toExponential(effectivePrecision);
                  // Make sure the exponent has at least 2 digits.
                  if (/[eE][-+]\d$/.test(argText)) {
                    argText = argText.slice(0, -1) + '0' + argText.slice(-1);
                  }
                } else if (next == 102 || next == 70) {
                  argText = currArg.toFixed(effectivePrecision);
                  if (currArg === 0 && __reallyNegative(currArg)) {
                    argText = '-' + argText;
                  }
                }
                var parts = argText.split('e');
                if (isGeneral && !flagAlternative) {
                  // Discard trailing zeros and periods.
                  while (parts[0].length > 1 && parts[0].indexOf('.') != -1 &&
                         (parts[0].slice(-1) == '0' || parts[0].slice(-1) == '.')) {
                    parts[0] = parts[0].slice(0, -1);
                  }
                } else {
                  // Make sure we have a period in alternative mode.
                  if (flagAlternative && argText.indexOf('.') == -1) parts[0] += '.';
                  // Zero pad until required precision.
                  while (precision > effectivePrecision++) parts[0] += '0';
                }
                argText = parts[0] + (parts.length > 1 ? 'e' + parts[1] : '');
                // Capitalize 'E' if needed.
                if (next == 69) argText = argText.toUpperCase();
                // Add sign.
                if (flagAlwaysSigned && currArg >= 0) {
                  argText = '+' + argText;
                }
              }
              // Add padding.
              while (argText.length < width) {
                if (flagLeftAlign) {
                  argText += ' ';
                } else {
                  if (flagZeroPad && (argText[0] == '-' || argText[0] == '+')) {
                    argText = argText[0] + '0' + argText.slice(1);
                  } else {
                    argText = (flagZeroPad ? '0' : ' ') + argText;
                  }
                }
              }
              // Adjust case.
              if (next < 97) argText = argText.toUpperCase();
              // Insert the result into the buffer.
              argText.split('').forEach(function(chr) {
                ret.push(chr.charCodeAt(0));
              });
              break;
            }
            case 's': {
              // String.
              var arg = getNextArg('i8*');
              var argLength = arg ? _strlen(arg) : '(null)'.length;
              if (precisionSet) argLength = Math.min(argLength, precision);
              if (!flagLeftAlign) {
                while (argLength < width--) {
                  ret.push(32);
                }
              }
              if (arg) {
                for (var i = 0; i < argLength; i++) {
                  ret.push(HEAPU8[((arg++)|0)]);
                }
              } else {
                ret = ret.concat(intArrayFromString('(null)'.substr(0, argLength), true));
              }
              if (flagLeftAlign) {
                while (argLength < width--) {
                  ret.push(32);
                }
              }
              break;
            }
            case 'c': {
              // Character.
              if (flagLeftAlign) ret.push(getNextArg('i8'));
              while (--width > 0) {
                ret.push(32);
              }
              if (!flagLeftAlign) ret.push(getNextArg('i8'));
              break;
            }
            case 'n': {
              // Write the length written so far to the next parameter.
              var ptr = getNextArg('i32*');
              HEAP32[((ptr)>>2)]=ret.length
              break;
            }
            case '%': {
              // Literal percent sign.
              ret.push(curr);
              break;
            }
            default: {
              // Unknown specifiers remain untouched.
              for (var i = startTextIndex; i < textIndex + 2; i++) {
                ret.push(HEAP8[(i)]);
              }
            }
          }
          textIndex += 2;
          // TODO: Support a/A (hex float) and m (last error) specifiers.
          // TODO: Support %1${specifier} for arg selection.
        } else {
          ret.push(curr);
          textIndex += 1;
        }
      }
      return ret;
    }function _snprintf(s, n, format, varargs) {
      // int snprintf(char *restrict s, size_t n, const char *restrict format, ...);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/printf.html
      var result = __formatString(format, varargs);
      var limit = (n === undefined) ? result.length
                                    : Math.min(result.length, Math.max(n - 1, 0));
      if (s < 0) {
        s = -s;
        var buf = _malloc(limit+1);
        HEAP32[((s)>>2)]=buf;
        s = buf;
      }
      for (var i = 0; i < limit; i++) {
        HEAP8[(((s)+(i))|0)]=result[i];
      }
      if (limit < n || (n === undefined)) HEAP8[(((s)+(i))|0)]=0;
      return result.length;
    }
  function _popen(command, mode) {
      // FILE *popen(const char *command, const char *mode);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/popen.html
      // We allow only one process, so no pipes.
      ___setErrNo(ERRNO_CODES.EMFILE);
      return 0;
    }
  function _pclose(stream) {
      // int pclose(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/pclose.html
      // We allow only one process, so no pipes.
      ___setErrNo(ERRNO_CODES.ECHILD);
      return -1;
    }
  function _fnmatch() {
  Module['printErr']('missing function: fnmatch'); abort(-1);
  }
  Module["_memset"] = _memset;var _llvm_memset_p0i8_i32=_memset;
  var ___strtok_state=0;
  function _strtok_r(s, delim, lasts) {
      var skip_leading_delim = 1;
      var spanp;
      var c, sc;
      var tok;
      if (s == 0 && (s = getValue(lasts, 'i8*')) == 0) {
        return 0;
      }
      cont: while (1) {
        c = getValue(s++, 'i8');
        for (spanp = delim; (sc = getValue(spanp++, 'i8')) != 0;) {
          if (c == sc) {
            if (skip_leading_delim) {
              continue cont;
            } else {
              setValue(lasts, s, 'i8*');
              setValue(s - 1, 0, 'i8');
              return s - 1;
            }
          }
        }
        break;
      }
      if (c == 0) {
        setValue(lasts, 0, 'i8*');
        return 0;
      }
      tok = s - 1;
      for (;;) {
        c = getValue(s++, 'i8');
        spanp = delim;
        do {
          if ((sc = getValue(spanp++, 'i8')) == c) {
            if (c == 0) {
              s = 0;
            } else {
              setValue(s - 1, 0, 'i8');
            }
            setValue(lasts, s, 'i8*');
            return tok;
          }
        } while (sc != 0);
      }
      abort('strtok_r error!');
    }function _strtok(s, delim) {
      return _strtok_r(s, delim, ___strtok_state);
    }
  function _ftell(stream) {
      // long ftell(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/ftell.html
      stream = FS.getStream(stream);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      if (FS.isChrdev(stream.node.mode)) {
        ___setErrNo(ERRNO_CODES.ESPIPE);
        return -1;
      } else {
        return stream.position;
      }
    }
  function _feof(stream) {
      // int feof(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/feof.html
      stream = FS.getStream(stream);
      return Number(stream && stream.eof);
    }
  function _opendir(dirname) {
      // DIR *opendir(const char *dirname);
      // http://pubs.opengroup.org/onlinepubs/007908799/xsh/opendir.html
      // NOTE: Calculating absolute path redundantly since we need to associate it
      //       with the opened stream.
      var path = Pointer_stringify(dirname);
      if (!path) {
        ___setErrNo(ERRNO_CODES.ENOENT);
        return 0;
      }
      var node;
      try {
        var lookup = FS.lookupPath(path, { follow: true });
        node = lookup.node;
      } catch (e) {
        FS.handleFSError(e);
        return 0;
      }
      if (!FS.isDir(node.mode)) {
        ___setErrNo(ERRNO_CODES.ENOTDIR);
        return 0;
      }
      var err = _open(dirname, 0, allocate([0, 0, 0, 0], 'i32', ALLOC_STACK));
      // open returns 0 on failure, not -1
      return err === -1 ? 0 : err;
    }
  function _readdir_r(dirp, entry, result) {
      // int readdir_r(DIR *dirp, struct dirent *entry, struct dirent **result);
      // http://pubs.opengroup.org/onlinepubs/007908799/xsh/readdir_r.html
      var stream = FS.getStream(dirp);
      if (!stream) {
        return ___setErrNo(ERRNO_CODES.EBADF);
      }
      var entries;
      try {
        entries = FS.readdir(stream.path);
      } catch (e) {
        return FS.handleFSError(e);
      }
      if (stream.position < 0 || stream.position >= entries.length) {
        HEAP32[((result)>>2)]=0
        return 0;
      }
      var id;
      var type;
      var name = entries[stream.position];
      var offset = stream.position + 1;
      if (!name.indexOf('.')) {
        id = 1;
        type = 4;
      } else {
        var child = FS.lookupNode(stream.node, name);
        id = child.id;
        type = FS.isChrdev(child.mode) ? 2 :  // DT_CHR, character device.
               FS.isDir(child.mode) ? 4 :     // DT_DIR, directory.
               FS.isLink(child.mode) ? 10 :   // DT_LNK, symbolic link.
               8;                             // DT_REG, regular file.
      }
      HEAP32[(((entry)+(___dirent_struct_layout.d_ino))>>2)]=id
      HEAP32[(((entry)+(___dirent_struct_layout.d_off))>>2)]=offset
      HEAP32[(((entry)+(___dirent_struct_layout.d_reclen))>>2)]=name.length + 1
      for (var i = 0; i < name.length; i++) {
        HEAP8[(((entry + ___dirent_struct_layout.d_name)+(i))|0)]=name.charCodeAt(i)
      }
      HEAP8[(((entry + ___dirent_struct_layout.d_name)+(i))|0)]=0
      HEAP8[(((entry)+(___dirent_struct_layout.d_type))|0)]=type
      HEAP32[((result)>>2)]=entry
      stream.position++;
      return 0;
    }function _readdir(dirp) {
      // struct dirent *readdir(DIR *dirp);
      // http://pubs.opengroup.org/onlinepubs/007908799/xsh/readdir_r.html
      var stream = FS.getStream(dirp);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return 0;
      }
      // TODO Is it supposed to be safe to execute multiple readdirs?
      if (!_readdir.entry) _readdir.entry = _malloc(___dirent_struct_layout.__size__);
      if (!_readdir.result) _readdir.result = _malloc(4);
      var err = _readdir_r(dirp, _readdir.entry, _readdir.result);
      if (err) {
        ___setErrNo(err);
        return 0;
      }
      return HEAP32[((_readdir.result)>>2)];
    }
  Module["_tolower"] = _tolower; 
  Module["_strncasecmp"] = _strncasecmp; 
  Module["_strcasecmp"] = _strcasecmp;
  function _closedir(dirp) {
      // int closedir(DIR *dirp);
      // http://pubs.opengroup.org/onlinepubs/007908799/xsh/closedir.html
      return _close(dirp);
    }
  var _environ=allocate(1, "i32*", ALLOC_STATIC);var ___environ=_environ;function ___buildEnvironment(env) {
      // WARNING: Arbitrary limit!
      var MAX_ENV_VALUES = 64;
      var TOTAL_ENV_SIZE = 1024;
      // Statically allocate memory for the environment.
      var poolPtr;
      var envPtr;
      if (!___buildEnvironment.called) {
        ___buildEnvironment.called = true;
        // Set default values. Use string keys for Closure Compiler compatibility.
        ENV['USER'] = 'root';
        ENV['PATH'] = '/';
        ENV['PWD'] = '/';
        ENV['HOME'] = '/home/emscripten';
        ENV['LANG'] = 'en_US.UTF-8';
        ENV['_'] = './this.program';
        // Allocate memory.
        poolPtr = allocate(TOTAL_ENV_SIZE, 'i8', ALLOC_STATIC);
        envPtr = allocate(MAX_ENV_VALUES * 4,
                          'i8*', ALLOC_STATIC);
        HEAP32[((envPtr)>>2)]=poolPtr
        HEAP32[((_environ)>>2)]=envPtr;
      } else {
        envPtr = HEAP32[((_environ)>>2)];
        poolPtr = HEAP32[((envPtr)>>2)];
      }
      // Collect key=value lines.
      var strings = [];
      var totalSize = 0;
      for (var key in env) {
        if (typeof env[key] === 'string') {
          var line = key + '=' + env[key];
          strings.push(line);
          totalSize += line.length;
        }
      }
      if (totalSize > TOTAL_ENV_SIZE) {
        throw new Error('Environment size exceeded TOTAL_ENV_SIZE!');
      }
      // Make new.
      var ptrSize = 4;
      for (var i = 0; i < strings.length; i++) {
        var line = strings[i];
        for (var j = 0; j < line.length; j++) {
          HEAP8[(((poolPtr)+(j))|0)]=line.charCodeAt(j);
        }
        HEAP8[(((poolPtr)+(j))|0)]=0;
        HEAP32[(((envPtr)+(i * ptrSize))>>2)]=poolPtr;
        poolPtr += line.length + 1;
      }
      HEAP32[(((envPtr)+(strings.length * ptrSize))>>2)]=0;
    }var ENV={};function _getenv(name) {
      // char *getenv(const char *name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/getenv.html
      if (name === 0) return 0;
      name = Pointer_stringify(name);
      if (!ENV.hasOwnProperty(name)) return 0;
      if (_getenv.ret) _free(_getenv.ret);
      _getenv.ret = allocate(intArrayFromString(ENV[name]), 'i8', ALLOC_NORMAL);
      return _getenv.ret;
    }
  var _llvm_va_start=undefined;
  function _vsnprintf(s, n, format, va_arg) {
      return _snprintf(s, n, format, HEAP32[((va_arg)>>2)]);
    }
  function _llvm_va_end() {}
  function ___assert_func(filename, line, func, condition) {
      throw 'Assertion failed: ' + (condition ? Pointer_stringify(condition) : 'unknown condition') + ', at: ' + [filename ? Pointer_stringify(filename) : 'unknown filename', line, func ? Pointer_stringify(func) : 'unknown function'] + ' at ' + new Error().stack;
    }
  function _rand() {
      return Math.floor(Math.random()*0x80000000);
    }
  var _abs=Math.abs;
  function _fprintf(stream, format, varargs) {
      // int fprintf(FILE *restrict stream, const char *restrict format, ...);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/printf.html
      var result = __formatString(format, varargs);
      var stack = Runtime.stackSave();
      var ret = _fwrite(allocate(result, 'i8', ALLOC_STACK), 1, result.length, stream);
      Runtime.stackRestore(stack);
      return ret;
    }function _printf(format, varargs) {
      // int printf(const char *restrict format, ...);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/printf.html
      var stdout = HEAP32[((_stdout)>>2)];
      return _fprintf(stdout, format, varargs);
    }
  Module["_memmove"] = _memmove;var _llvm_memmove_p0i8_p0i8_i32=_memmove;
  Module["_strcpy"] = _strcpy;
  var ___tm_struct_layout={__size__:44,tm_sec:0,tm_min:4,tm_hour:8,tm_mday:12,tm_mon:16,tm_year:20,tm_wday:24,tm_yday:28,tm_isdst:32,tm_gmtoff:36,tm_zone:40};
  var ___tm_timezones={};
  var __tzname=allocate(8, "i32*", ALLOC_STATIC);
  var __daylight=allocate(1, "i32*", ALLOC_STATIC);
  var __timezone=allocate(1, "i32*", ALLOC_STATIC);function _tzset() {
      // TODO: Use (malleable) environment variables instead of system settings.
      if (_tzset.called) return;
      _tzset.called = true;
      HEAP32[((__timezone)>>2)]=-(new Date()).getTimezoneOffset() * 60
      var winter = new Date(2000, 0, 1);
      var summer = new Date(2000, 6, 1);
      HEAP32[((__daylight)>>2)]=Number(winter.getTimezoneOffset() != summer.getTimezoneOffset())
      var winterName = 'GMT'; // XXX do not rely on browser timezone info, it is very unpredictable | winter.toString().match(/\(([A-Z]+)\)/)[1];
      var summerName = 'GMT'; // XXX do not rely on browser timezone info, it is very unpredictable | summer.toString().match(/\(([A-Z]+)\)/)[1];
      var winterNamePtr = allocate(intArrayFromString(winterName), 'i8', ALLOC_NORMAL);
      var summerNamePtr = allocate(intArrayFromString(summerName), 'i8', ALLOC_NORMAL);
      HEAP32[((__tzname)>>2)]=winterNamePtr
      HEAP32[(((__tzname)+(4))>>2)]=summerNamePtr
    }function _localtime_r(time, tmPtr) {
      _tzset();
      var offsets = ___tm_struct_layout;
      var date = new Date(HEAP32[((time)>>2)]*1000);
      HEAP32[(((tmPtr)+(offsets.tm_sec))>>2)]=date.getSeconds()
      HEAP32[(((tmPtr)+(offsets.tm_min))>>2)]=date.getMinutes()
      HEAP32[(((tmPtr)+(offsets.tm_hour))>>2)]=date.getHours()
      HEAP32[(((tmPtr)+(offsets.tm_mday))>>2)]=date.getDate()
      HEAP32[(((tmPtr)+(offsets.tm_mon))>>2)]=date.getMonth()
      HEAP32[(((tmPtr)+(offsets.tm_year))>>2)]=date.getFullYear()-1900
      HEAP32[(((tmPtr)+(offsets.tm_wday))>>2)]=date.getDay()
      var start = new Date(date.getFullYear(), 0, 1);
      var yday = Math.floor((date.getTime() - start.getTime()) / (1000 * 60 * 60 * 24));
      HEAP32[(((tmPtr)+(offsets.tm_yday))>>2)]=yday
      HEAP32[(((tmPtr)+(offsets.tm_gmtoff))>>2)]=start.getTimezoneOffset() * 60
      var dst = Number(start.getTimezoneOffset() != date.getTimezoneOffset());
      HEAP32[(((tmPtr)+(offsets.tm_isdst))>>2)]=dst
      var timezone = 'GMT'; // XXX do not rely on browser timezone info, it is very unpredictable | date.toString().match(/\(([A-Z]+)\)/)[1];
      if (!(timezone in ___tm_timezones)) {
        ___tm_timezones[timezone] = allocate(intArrayFromString(timezone), 'i8', ALLOC_NORMAL);
      }
      HEAP32[(((tmPtr)+(offsets.tm_zone))>>2)]=___tm_timezones[timezone]
      return tmPtr;
    }
  function _strchr(ptr, chr) {
      ptr--;
      do {
        ptr++;
        var val = HEAP8[(ptr)];
        if (val == chr) return ptr;
      } while (val);
      return 0;
    }
  function _isdigit(chr) {
      return chr >= 48 && chr <= 57;
    }
  var ___flock_struct_layout={__size__:16,l_type:0,l_whence:2,l_start:4,l_len:8,l_pid:12,l_xxx:14};function _fcntl(fildes, cmd, varargs, dup2) {
      // int fcntl(int fildes, int cmd, ...);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/fcntl.html
      var stream = FS.getStream(fildes);
      if (!stream) {
        ___setErrNo(ERRNO_CODES.EBADF);
        return -1;
      }
      switch (cmd) {
        case 0:
          var arg = HEAP32[((varargs)>>2)];
          if (arg < 0) {
            ___setErrNo(ERRNO_CODES.EINVAL);
            return -1;
          }
          var newStream;
          try {
            newStream = FS.open(stream.path, stream.flags, 0, arg);
          } catch (e) {
            FS.handleFSError(e);
            return -1;
          }
          return newStream.fd;
        case 1:
        case 2:
          return 0;  // FD_CLOEXEC makes no sense for a single process.
        case 3:
          return stream.flags;
        case 4:
          var arg = HEAP32[((varargs)>>2)];
          stream.flags |= arg;
          return 0;
        case 7:
        case 20:
          var arg = HEAP32[((varargs)>>2)];
          var offset = ___flock_struct_layout.l_type;
          // We're always unlocked.
          HEAP16[(((arg)+(offset))>>1)]=3
          return 0;
        case 8:
        case 9:
        case 21:
        case 22:
          // Pretend that the locking is successful.
          return 0;
        case 6:
        case 5:
          // These are for sockets. We don't have them fully implemented yet.
          ___setErrNo(ERRNO_CODES.EINVAL);
          return -1;
        default:
          ___setErrNo(ERRNO_CODES.EINVAL);
          return -1;
      }
      // Should never be reached. Only to silence strict warnings.
      return -1;
    }function _dup(fildes) {
      // int dup(int fildes);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/dup.html
      return _fcntl(fildes, 0, allocate([0, 0, 0, 0], 'i32', ALLOC_STACK));  // F_DUPFD.
    }
  function _fputs(s, stream) {
      // int fputs(const char *restrict s, FILE *restrict stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fputs.html
      return _write(stream, s, _strlen(s));
    }
  function _fputc(c, stream) {
      // int fputc(int c, FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fputc.html
      var chr = unSign(c & 0xFF);
      HEAP8[((_fputc.ret)|0)]=chr
      var ret = _write(stream, _fputc.ret, 1);
      if (ret == -1) {
        var streamObj = FS.getStream(stream);
        if (streamObj) streamObj.error = true;
        return -1;
      } else {
        return chr;
      }
    }function _puts(s) {
      // int puts(const char *s);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/puts.html
      // NOTE: puts() always writes an extra newline.
      var stdout = HEAP32[((_stdout)>>2)];
      var ret = _fputs(s, stdout);
      if (ret < 0) {
        return ret;
      } else {
        var newlineRet = _fputc(10, stdout);
        return (newlineRet < 0) ? -1 : ret + 1;
      }
    }
  function _strerror_r(errnum, strerrbuf, buflen) {
      if (errnum in ERRNO_MESSAGES) {
        if (ERRNO_MESSAGES[errnum].length > buflen - 1) {
          return ___setErrNo(ERRNO_CODES.ERANGE);
        } else {
          var msg = ERRNO_MESSAGES[errnum];
          for (var i = 0; i < msg.length; i++) {
            HEAP8[(((strerrbuf)+(i))|0)]=msg.charCodeAt(i)
          }
          HEAP8[(((strerrbuf)+(i))|0)]=0
          return 0;
        }
      } else {
        return ___setErrNo(ERRNO_CODES.EINVAL);
      }
    }function _strerror(errnum) {
      if (!_strerror.buffer) _strerror.buffer = _malloc(256);
      _strerror_r(errnum, _strerror.buffer, 256);
      return _strerror.buffer;
    }function _perror(s) {
      // void perror(const char *s);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/perror.html
      var stdout = HEAP32[((_stdout)>>2)];
      if (s) {
        _fputs(s, stdout);
        _fputc(58, stdout);
        _fputc(32, stdout);
      }
      var errnum = HEAP32[((___errno_location())>>2)];
      _puts(_strerror(errnum));
    }
  function _qsort(base, num, size, cmp) {
      if (num == 0 || size == 0) return;
      // forward calls to the JavaScript sort method
      // first, sort the items logically
      var keys = [];
      for (var i = 0; i < num; i++) keys.push(i);
      keys.sort(function(a, b) {
        return Module['dynCall_iii'](cmp, base+a*size, base+b*size);
      });
      // apply the sort
      var temp = _malloc(num*size);
      _memcpy(temp, base, num*size);
      for (var i = 0; i < num; i++) {
        if (keys[i] == i) continue; // already in place
        _memcpy(base+i*size, temp+keys[i]*size, size);
      }
      _free(temp);
    }
  var _llvm_pow_f64=Math.pow;
  function _round(x) {
      return (x < 0) ? -Math.round(-x) : Math.round(x);
    }
  var _log=Math.log;
  function _fgetc(stream) {
      // int fgetc(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/fgetc.html
      var streamObj = FS.getStream(stream);
      if (!streamObj) return -1;
      if (streamObj.eof || streamObj.error) return -1;
      var ret = _fread(_fgetc.ret, 1, 1, stream);
      if (ret == 0) {
        streamObj.eof = true;
        return -1;
      } else if (ret == -1) {
        streamObj.error = true;
        return -1;
      } else {
        return HEAPU8[((_fgetc.ret)|0)];
      }
    }
  function _putchar(c) {
      // int putchar(int c);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/putchar.html
      return _fputc(c, HEAP32[((_stdout)>>2)]);
    } 
  Module["_saveSetjmp"] = _saveSetjmp;
  Module["_testSetjmp"] = _testSetjmp;var _setjmp=undefined;
  function _longjmp(env, value) {
      asm['setThrew'](env, value || 1);
      throw 'longjmp';
    }
  var _getc=_fgetc;
  function _ungetc(c, stream) {
      // int ungetc(int c, FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/ungetc.html
      stream = FS.getStream(stream);
      if (!stream) {
        return -1;
      }
      if (c === -1) {
        // do nothing for EOF character
        return c;
      }
      c = unSign(c & 0xFF);
      stream.ungotten.push(c);
      stream.eof = false;
      return c;
    }
  function __exit(status) {
      // void _exit(int status);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/exit.html
      Module.print('exit(' + status + ') called');
      Module['exit'](status);
    }function _exit(status) {
      __exit(status);
    }
  function _ferror(stream) {
      // int ferror(FILE *stream);
      // http://pubs.opengroup.org/onlinepubs/000095399/functions/ferror.html
      stream = FS.getStream(stream);
      return Number(stream && stream.error);
    }
  var _sin=Math.sin;
  var _cos=Math.cos;
  var _floor=Math.floor;
  var _exp=Math.exp;
  function _ldexp(x, exp_) {
      return x * Math.pow(2, exp_);
    }
  function _log10(x) {
      return Math.log(x) / Math.LN10;
    }
  var _llvm_memset_p0i8_i64=_memset;
  function _abort() {
      Module['abort']();
    }
  function _sbrk(bytes) {
      // Implement a Linux-like 'memory area' for our 'process'.
      // Changes the size of the memory area by |bytes|; returns the
      // address of the previous top ('break') of the memory area
      // We control the "dynamic" memory - DYNAMIC_BASE to DYNAMICTOP
      var self = _sbrk;
      if (!self.called) {
        DYNAMICTOP = alignMemoryPage(DYNAMICTOP); // make sure we start out aligned
        self.called = true;
        assert(Runtime.dynamicAlloc);
        self.alloc = Runtime.dynamicAlloc;
        Runtime.dynamicAlloc = function() { abort('cannot dynamically allocate, sbrk now has control') };
      }
      var ret = DYNAMICTOP;
      if (bytes != 0) self.alloc(bytes);
      return ret;  // Previous break location.
    }
  function _sysconf(name) {
      // long sysconf(int name);
      // http://pubs.opengroup.org/onlinepubs/009695399/functions/sysconf.html
      switch(name) {
        case 8: return PAGE_SIZE;
        case 54:
        case 56:
        case 21:
        case 61:
        case 63:
        case 22:
        case 67:
        case 23:
        case 24:
        case 25:
        case 26:
        case 27:
        case 69:
        case 28:
        case 101:
        case 70:
        case 71:
        case 29:
        case 30:
        case 199:
        case 75:
        case 76:
        case 32:
        case 43:
        case 44:
        case 80:
        case 46:
        case 47:
        case 45:
        case 48:
        case 49:
        case 42:
        case 82:
        case 33:
        case 7:
        case 108:
        case 109:
        case 107:
        case 112:
        case 119:
        case 121:
          return 200809;
        case 13:
        case 104:
        case 94:
        case 95:
        case 34:
        case 35:
        case 77:
        case 81:
        case 83:
        case 84:
        case 85:
        case 86:
        case 87:
        case 88:
        case 89:
        case 90:
        case 91:
        case 94:
        case 95:
        case 110:
        case 111:
        case 113:
        case 114:
        case 115:
        case 116:
        case 117:
        case 118:
        case 120:
        case 40:
        case 16:
        case 79:
        case 19:
          return -1;
        case 92:
        case 93:
        case 5:
        case 72:
        case 6:
        case 74:
        case 92:
        case 93:
        case 96:
        case 97:
        case 98:
        case 99:
        case 102:
        case 103:
        case 105:
          return 1;
        case 38:
        case 66:
        case 50:
        case 51:
        case 4:
          return 1024;
        case 15:
        case 64:
        case 41:
          return 32;
        case 55:
        case 37:
        case 17:
          return 2147483647;
        case 18:
        case 1:
          return 47839;
        case 59:
        case 57:
          return 99;
        case 68:
        case 58:
          return 2048;
        case 0: return 2097152;
        case 3: return 65536;
        case 14: return 32768;
        case 73: return 32767;
        case 39: return 16384;
        case 60: return 1000;
        case 106: return 700;
        case 52: return 256;
        case 62: return 255;
        case 2: return 100;
        case 65: return 64;
        case 36: return 20;
        case 100: return 16;
        case 20: return 6;
        case 53: return 4;
        case 10: return 1;
      }
      ___setErrNo(ERRNO_CODES.EINVAL);
      return -1;
    }
  function _time(ptr) {
      var ret = Math.floor(Date.now()/1000);
      if (ptr) {
        HEAP32[((ptr)>>2)]=ret
      }
      return ret;
    }
  function ___cxa_allocate_exception(size) {
      return _malloc(size);
    }
  function _llvm_eh_exception() {
      return HEAP32[((_llvm_eh_exception.buf)>>2)];
    }
  function __ZSt18uncaught_exceptionv() { // std::uncaught_exception()
      return !!__ZSt18uncaught_exceptionv.uncaught_exception;
    }
  function ___cxa_is_number_type(type) {
      var isNumber = false;
      try { if (type == __ZTIi) isNumber = true } catch(e){}
      try { if (type == __ZTIj) isNumber = true } catch(e){}
      try { if (type == __ZTIl) isNumber = true } catch(e){}
      try { if (type == __ZTIm) isNumber = true } catch(e){}
      try { if (type == __ZTIx) isNumber = true } catch(e){}
      try { if (type == __ZTIy) isNumber = true } catch(e){}
      try { if (type == __ZTIf) isNumber = true } catch(e){}
      try { if (type == __ZTId) isNumber = true } catch(e){}
      try { if (type == __ZTIe) isNumber = true } catch(e){}
      try { if (type == __ZTIc) isNumber = true } catch(e){}
      try { if (type == __ZTIa) isNumber = true } catch(e){}
      try { if (type == __ZTIh) isNumber = true } catch(e){}
      try { if (type == __ZTIs) isNumber = true } catch(e){}
      try { if (type == __ZTIt) isNumber = true } catch(e){}
      return isNumber;
    }function ___cxa_does_inherit(definiteType, possibilityType, possibility) {
      if (possibility == 0) return false;
      if (possibilityType == 0 || possibilityType == definiteType)
        return true;
      var possibility_type_info;
      if (___cxa_is_number_type(possibilityType)) {
        possibility_type_info = possibilityType;
      } else {
        var possibility_type_infoAddr = HEAP32[((possibilityType)>>2)] - 8;
        possibility_type_info = HEAP32[((possibility_type_infoAddr)>>2)];
      }
      switch (possibility_type_info) {
      case 0: // possibility is a pointer
        // See if definite type is a pointer
        var definite_type_infoAddr = HEAP32[((definiteType)>>2)] - 8;
        var definite_type_info = HEAP32[((definite_type_infoAddr)>>2)];
        if (definite_type_info == 0) {
          // Also a pointer; compare base types of pointers
          var defPointerBaseAddr = definiteType+8;
          var defPointerBaseType = HEAP32[((defPointerBaseAddr)>>2)];
          var possPointerBaseAddr = possibilityType+8;
          var possPointerBaseType = HEAP32[((possPointerBaseAddr)>>2)];
          return ___cxa_does_inherit(defPointerBaseType, possPointerBaseType, possibility);
        } else
          return false; // one pointer and one non-pointer
      case 1: // class with no base class
        return false;
      case 2: // class with base class
        var parentTypeAddr = possibilityType + 8;
        var parentType = HEAP32[((parentTypeAddr)>>2)];
        return ___cxa_does_inherit(definiteType, parentType, possibility);
      default:
        return false; // some unencountered type
      }
    }
  function ___resumeException(ptr) {
      if (HEAP32[((_llvm_eh_exception.buf)>>2)] == 0) HEAP32[((_llvm_eh_exception.buf)>>2)]=ptr;
      throw ptr + " - Exception catching is disabled, this exception cannot be caught. Compile with -s DISABLE_EXCEPTION_CATCHING=0 or DISABLE_EXCEPTION_CATCHING=2 to catch.";;
    }function ___cxa_find_matching_catch(thrown, throwntype) {
      if (thrown == -1) thrown = HEAP32[((_llvm_eh_exception.buf)>>2)];
      if (throwntype == -1) throwntype = HEAP32[(((_llvm_eh_exception.buf)+(4))>>2)];
      var typeArray = Array.prototype.slice.call(arguments, 2);
      // If throwntype is a pointer, this means a pointer has been
      // thrown. When a pointer is thrown, actually what's thrown
      // is a pointer to the pointer. We'll dereference it.
      if (throwntype != 0 && !___cxa_is_number_type(throwntype)) {
        var throwntypeInfoAddr= HEAP32[((throwntype)>>2)] - 8;
        var throwntypeInfo= HEAP32[((throwntypeInfoAddr)>>2)];
        if (throwntypeInfo == 0)
          thrown = HEAP32[((thrown)>>2)];
      }
      // The different catch blocks are denoted by different types.
      // Due to inheritance, those types may not precisely match the
      // type of the thrown object. Find one which matches, and
      // return the type of the catch block which should be called.
      for (var i = 0; i < typeArray.length; i++) {
        if (___cxa_does_inherit(typeArray[i], throwntype, thrown))
          return ((asm["setTempRet0"](typeArray[i]),thrown)|0);
      }
      // Shouldn't happen unless we have bogus data in typeArray
      // or encounter a type for which emscripten doesn't have suitable
      // typeinfo defined. Best-efforts match just in case.
      return ((asm["setTempRet0"](throwntype),thrown)|0);
    }function ___cxa_throw(ptr, type, destructor) {
      if (!___cxa_throw.initialized) {
        try {
          HEAP32[((__ZTVN10__cxxabiv119__pointer_type_infoE)>>2)]=0; // Workaround for libcxxabi integration bug
        } catch(e){}
        try {
          HEAP32[((__ZTVN10__cxxabiv117__class_type_infoE)>>2)]=1; // Workaround for libcxxabi integration bug
        } catch(e){}
        try {
          HEAP32[((__ZTVN10__cxxabiv120__si_class_type_infoE)>>2)]=2; // Workaround for libcxxabi integration bug
        } catch(e){}
        ___cxa_throw.initialized = true;
      }
      HEAP32[((_llvm_eh_exception.buf)>>2)]=ptr
      HEAP32[(((_llvm_eh_exception.buf)+(4))>>2)]=type
      HEAP32[(((_llvm_eh_exception.buf)+(8))>>2)]=destructor
      if (!("uncaught_exception" in __ZSt18uncaught_exceptionv)) {
        __ZSt18uncaught_exceptionv.uncaught_exception = 1;
      } else {
        __ZSt18uncaught_exceptionv.uncaught_exception++;
      }
      throw ptr + " - Exception catching is disabled, this exception cannot be caught. Compile with -s DISABLE_EXCEPTION_CATCHING=0 or DISABLE_EXCEPTION_CATCHING=2 to catch.";;
    }
  function ___cxa_call_unexpected(exception) {
      Module.printErr('Unexpected exception thrown, this is not properly supported - aborting');
      ABORT = true;
      throw exception;
    }
  var Browser={mainLoop:{scheduler:null,shouldPause:false,paused:false,queue:[],pause:function () {
          Browser.mainLoop.shouldPause = true;
        },resume:function () {
          if (Browser.mainLoop.paused) {
            Browser.mainLoop.paused = false;
            Browser.mainLoop.scheduler();
          }
          Browser.mainLoop.shouldPause = false;
        },updateStatus:function () {
          if (Module['setStatus']) {
            var message = Module['statusMessage'] || 'Please wait...';
            var remaining = Browser.mainLoop.remainingBlockers;
            var expected = Browser.mainLoop.expectedBlockers;
            if (remaining) {
              if (remaining < expected) {
                Module['setStatus'](message + ' (' + (expected - remaining) + '/' + expected + ')');
              } else {
                Module['setStatus'](message);
              }
            } else {
              Module['setStatus']('');
            }
          }
        }},isFullScreen:false,pointerLock:false,moduleContextCreatedCallbacks:[],workers:[],init:function () {
        if (!Module["preloadPlugins"]) Module["preloadPlugins"] = []; // needs to exist even in workers
        if (Browser.initted || ENVIRONMENT_IS_WORKER) return;
        Browser.initted = true;
        try {
          new Blob();
          Browser.hasBlobConstructor = true;
        } catch(e) {
          Browser.hasBlobConstructor = false;
          console.log("warning: no blob constructor, cannot create blobs with mimetypes");
        }
        Browser.BlobBuilder = typeof MozBlobBuilder != "undefined" ? MozBlobBuilder : (typeof WebKitBlobBuilder != "undefined" ? WebKitBlobBuilder : (!Browser.hasBlobConstructor ? console.log("warning: no BlobBuilder") : null));
        Browser.URLObject = typeof window != "undefined" ? (window.URL ? window.URL : window.webkitURL) : undefined;
        if (!Module.noImageDecoding && typeof Browser.URLObject === 'undefined') {
          console.log("warning: Browser does not support creating object URLs. Built-in browser image decoding will not be available.");
          Module.noImageDecoding = true;
        }
        // Support for plugins that can process preloaded files. You can add more of these to
        // your app by creating and appending to Module.preloadPlugins.
        //
        // Each plugin is asked if it can handle a file based on the file's name. If it can,
        // it is given the file's raw data. When it is done, it calls a callback with the file's
        // (possibly modified) data. For example, a plugin might decompress a file, or it
        // might create some side data structure for use later (like an Image element, etc.).
        var imagePlugin = {};
        imagePlugin['canHandle'] = function(name) {
          return !Module.noImageDecoding && /\.(jpg|jpeg|png|bmp)$/i.test(name);
        };
        imagePlugin['handle'] = function(byteArray, name, onload, onerror) {
          var b = null;
          if (Browser.hasBlobConstructor) {
            try {
              b = new Blob([byteArray], { type: Browser.getMimetype(name) });
              if (b.size !== byteArray.length) { // Safari bug #118630
                // Safari's Blob can only take an ArrayBuffer
                b = new Blob([(new Uint8Array(byteArray)).buffer], { type: Browser.getMimetype(name) });
              }
            } catch(e) {
              Runtime.warnOnce('Blob constructor present but fails: ' + e + '; falling back to blob builder');
            }
          }
          if (!b) {
            var bb = new Browser.BlobBuilder();
            bb.append((new Uint8Array(byteArray)).buffer); // we need to pass a buffer, and must copy the array to get the right data range
            b = bb.getBlob();
          }
          var url = Browser.URLObject.createObjectURL(b);
          var img = new Image();
          img.onload = function() {
            assert(img.complete, 'Image ' + name + ' could not be decoded');
            var canvas = document.createElement('canvas');
            canvas.width = img.width;
            canvas.height = img.height;
            var ctx = canvas.getContext('2d');
            ctx.drawImage(img, 0, 0);
            Module["preloadedImages"][name] = canvas;
            Browser.URLObject.revokeObjectURL(url);
            if (onload) onload(byteArray);
          };
          img.onerror = function(event) {
            console.log('Image ' + url + ' could not be decoded');
            if (onerror) onerror();
          };
          img.src = url;
        };
        Module['preloadPlugins'].push(imagePlugin);
        var audioPlugin = {};
        audioPlugin['canHandle'] = function(name) {
          return !Module.noAudioDecoding && name.substr(-4) in { '.ogg': 1, '.wav': 1, '.mp3': 1 };
        };
        audioPlugin['handle'] = function(byteArray, name, onload, onerror) {
          var done = false;
          function finish(audio) {
            if (done) return;
            done = true;
            Module["preloadedAudios"][name] = audio;
            if (onload) onload(byteArray);
          }
          function fail() {
            if (done) return;
            done = true;
            Module["preloadedAudios"][name] = new Audio(); // empty shim
            if (onerror) onerror();
          }
          if (Browser.hasBlobConstructor) {
            try {
              var b = new Blob([byteArray], { type: Browser.getMimetype(name) });
            } catch(e) {
              return fail();
            }
            var url = Browser.URLObject.createObjectURL(b); // XXX we never revoke this!
            var audio = new Audio();
            audio.addEventListener('canplaythrough', function() { finish(audio) }, false); // use addEventListener due to chromium bug 124926
            audio.onerror = function(event) {
              if (done) return;
              console.log('warning: browser could not fully decode audio ' + name + ', trying slower base64 approach');
              function encode64(data) {
                var BASE = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
                var PAD = '=';
                var ret = '';
                var leftchar = 0;
                var leftbits = 0;
                for (var i = 0; i < data.length; i++) {
                  leftchar = (leftchar << 8) | data[i];
                  leftbits += 8;
                  while (leftbits >= 6) {
                    var curr = (leftchar >> (leftbits-6)) & 0x3f;
                    leftbits -= 6;
                    ret += BASE[curr];
                  }
                }
                if (leftbits == 2) {
                  ret += BASE[(leftchar&3) << 4];
                  ret += PAD + PAD;
                } else if (leftbits == 4) {
                  ret += BASE[(leftchar&0xf) << 2];
                  ret += PAD;
                }
                return ret;
              }
              audio.src = 'data:audio/x-' + name.substr(-3) + ';base64,' + encode64(byteArray);
              finish(audio); // we don't wait for confirmation this worked - but it's worth trying
            };
            audio.src = url;
            // workaround for chrome bug 124926 - we do not always get oncanplaythrough or onerror
            Browser.safeSetTimeout(function() {
              finish(audio); // try to use it even though it is not necessarily ready to play
            }, 10000);
          } else {
            return fail();
          }
        };
        Module['preloadPlugins'].push(audioPlugin);
        // Canvas event setup
        var canvas = Module['canvas'];
        canvas.requestPointerLock = canvas['requestPointerLock'] ||
                                    canvas['mozRequestPointerLock'] ||
                                    canvas['webkitRequestPointerLock'];
        canvas.exitPointerLock = document['exitPointerLock'] ||
                                 document['mozExitPointerLock'] ||
                                 document['webkitExitPointerLock'] ||
                                 function(){}; // no-op if function does not exist
        canvas.exitPointerLock = canvas.exitPointerLock.bind(document);
        function pointerLockChange() {
          Browser.pointerLock = document['pointerLockElement'] === canvas ||
                                document['mozPointerLockElement'] === canvas ||
                                document['webkitPointerLockElement'] === canvas;
        }
        document.addEventListener('pointerlockchange', pointerLockChange, false);
        document.addEventListener('mozpointerlockchange', pointerLockChange, false);
        document.addEventListener('webkitpointerlockchange', pointerLockChange, false);
        if (Module['elementPointerLock']) {
          canvas.addEventListener("click", function(ev) {
            if (!Browser.pointerLock && canvas.requestPointerLock) {
              canvas.requestPointerLock();
              ev.preventDefault();
            }
          }, false);
        }
      },createContext:function (canvas, useWebGL, setInModule) {
        var ctx;
        try {
          if (useWebGL) {
            ctx = canvas.getContext('experimental-webgl', {
              alpha: false
            });
          } else {
            ctx = canvas.getContext('2d');
          }
          if (!ctx) throw ':(';
        } catch (e) {
          Module.print('Could not create canvas - ' + e);
          return null;
        }
        if (useWebGL) {
          // Set the background of the WebGL canvas to black
          canvas.style.backgroundColor = "black";
          // Warn on context loss
          canvas.addEventListener('webglcontextlost', function(event) {
            alert('WebGL context lost. You will need to reload the page.');
          }, false);
        }
        if (setInModule) {
          Module.ctx = ctx;
          Module.useWebGL = useWebGL;
          Browser.moduleContextCreatedCallbacks.forEach(function(callback) { callback() });
          Browser.init();
        }
        return ctx;
      },destroyContext:function (canvas, useWebGL, setInModule) {},fullScreenHandlersInstalled:false,lockPointer:undefined,resizeCanvas:undefined,requestFullScreen:function (lockPointer, resizeCanvas) {
        Browser.lockPointer = lockPointer;
        Browser.resizeCanvas = resizeCanvas;
        if (typeof Browser.lockPointer === 'undefined') Browser.lockPointer = true;
        if (typeof Browser.resizeCanvas === 'undefined') Browser.resizeCanvas = false;
        var canvas = Module['canvas'];
        function fullScreenChange() {
          Browser.isFullScreen = false;
          if ((document['webkitFullScreenElement'] || document['webkitFullscreenElement'] ||
               document['mozFullScreenElement'] || document['mozFullscreenElement'] ||
               document['fullScreenElement'] || document['fullscreenElement']) === canvas) {
            canvas.cancelFullScreen = document['cancelFullScreen'] ||
                                      document['mozCancelFullScreen'] ||
                                      document['webkitCancelFullScreen'];
            canvas.cancelFullScreen = canvas.cancelFullScreen.bind(document);
            if (Browser.lockPointer) canvas.requestPointerLock();
            Browser.isFullScreen = true;
            if (Browser.resizeCanvas) Browser.setFullScreenCanvasSize();
          } else if (Browser.resizeCanvas){
            Browser.setWindowedCanvasSize();
          }
          if (Module['onFullScreen']) Module['onFullScreen'](Browser.isFullScreen);
        }
        if (!Browser.fullScreenHandlersInstalled) {
          Browser.fullScreenHandlersInstalled = true;
          document.addEventListener('fullscreenchange', fullScreenChange, false);
          document.addEventListener('mozfullscreenchange', fullScreenChange, false);
          document.addEventListener('webkitfullscreenchange', fullScreenChange, false);
        }
        canvas.requestFullScreen = canvas['requestFullScreen'] ||
                                   canvas['mozRequestFullScreen'] ||
                                   (canvas['webkitRequestFullScreen'] ? function() { canvas['webkitRequestFullScreen'](Element['ALLOW_KEYBOARD_INPUT']) } : null);
        canvas.requestFullScreen();
      },requestAnimationFrame:function (func) {
        if (!window.requestAnimationFrame) {
          window.requestAnimationFrame = window['requestAnimationFrame'] ||
                                         window['mozRequestAnimationFrame'] ||
                                         window['webkitRequestAnimationFrame'] ||
                                         window['msRequestAnimationFrame'] ||
                                         window['oRequestAnimationFrame'] ||
                                         window['setTimeout'];
        }
        window.requestAnimationFrame(func);
      },safeCallback:function (func) {
        return function() {
          if (!ABORT) return func.apply(null, arguments);
        };
      },safeRequestAnimationFrame:function (func) {
        return Browser.requestAnimationFrame(function() {
          if (!ABORT) func();
        });
      },safeSetTimeout:function (func, timeout) {
        return setTimeout(function() {
          if (!ABORT) func();
        }, timeout);
      },safeSetInterval:function (func, timeout) {
        return setInterval(function() {
          if (!ABORT) func();
        }, timeout);
      },getMimetype:function (name) {
        return {
          'jpg': 'image/jpeg',
          'jpeg': 'image/jpeg',
          'png': 'image/png',
          'bmp': 'image/bmp',
          'ogg': 'audio/ogg',
          'wav': 'audio/wav',
          'mp3': 'audio/mpeg'
        }[name.substr(name.lastIndexOf('.')+1)];
      },getUserMedia:function (func) {
        if(!window.getUserMedia) {
          window.getUserMedia = navigator['getUserMedia'] ||
                                navigator['mozGetUserMedia'];
        }
        window.getUserMedia(func);
      },getMovementX:function (event) {
        return event['movementX'] ||
               event['mozMovementX'] ||
               event['webkitMovementX'] ||
               0;
      },getMovementY:function (event) {
        return event['movementY'] ||
               event['mozMovementY'] ||
               event['webkitMovementY'] ||
               0;
      },mouseX:0,mouseY:0,mouseMovementX:0,mouseMovementY:0,calculateMouseEvent:function (event) { // event should be mousemove, mousedown or mouseup
        if (Browser.pointerLock) {
          // When the pointer is locked, calculate the coordinates
          // based on the movement of the mouse.
          // Workaround for Firefox bug 764498
          if (event.type != 'mousemove' &&
              ('mozMovementX' in event)) {
            Browser.mouseMovementX = Browser.mouseMovementY = 0;
          } else {
            Browser.mouseMovementX = Browser.getMovementX(event);
            Browser.mouseMovementY = Browser.getMovementY(event);
          }
          // check if SDL is available
          if (typeof SDL != "undefined") {
          	Browser.mouseX = SDL.mouseX + Browser.mouseMovementX;
          	Browser.mouseY = SDL.mouseY + Browser.mouseMovementY;
          } else {
          	// just add the mouse delta to the current absolut mouse position
          	// FIXME: ideally this should be clamped against the canvas size and zero
          	Browser.mouseX += Browser.mouseMovementX;
          	Browser.mouseY += Browser.mouseMovementY;
          }        
        } else {
          // Otherwise, calculate the movement based on the changes
          // in the coordinates.
          var rect = Module["canvas"].getBoundingClientRect();
          var x = event.pageX - (window.scrollX + rect.left);
          var y = event.pageY - (window.scrollY + rect.top);
          // the canvas might be CSS-scaled compared to its backbuffer;
          // SDL-using content will want mouse coordinates in terms
          // of backbuffer units.
          var cw = Module["canvas"].width;
          var ch = Module["canvas"].height;
          x = x * (cw / rect.width);
          y = y * (ch / rect.height);
          Browser.mouseMovementX = x - Browser.mouseX;
          Browser.mouseMovementY = y - Browser.mouseY;
          Browser.mouseX = x;
          Browser.mouseY = y;
        }
      },xhrLoad:function (url, onload, onerror) {
        var xhr = new XMLHttpRequest();
        xhr.open('GET', url, true);
        xhr.responseType = 'arraybuffer';
        xhr.onload = function() {
          if (xhr.status == 200 || (xhr.status == 0 && xhr.response)) { // file URLs can return 0
            onload(xhr.response);
          } else {
            onerror();
          }
        };
        xhr.onerror = onerror;
        xhr.send(null);
      },asyncLoad:function (url, onload, onerror, noRunDep) {
        Browser.xhrLoad(url, function(arrayBuffer) {
          assert(arrayBuffer, 'Loading data file "' + url + '" failed (no arrayBuffer).');
          onload(new Uint8Array(arrayBuffer));
          if (!noRunDep) removeRunDependency('al ' + url);
        }, function(event) {
          if (onerror) {
            onerror();
          } else {
            throw 'Loading data file "' + url + '" failed.';
          }
        });
        if (!noRunDep) addRunDependency('al ' + url);
      },resizeListeners:[],updateResizeListeners:function () {
        var canvas = Module['canvas'];
        Browser.resizeListeners.forEach(function(listener) {
          listener(canvas.width, canvas.height);
        });
      },setCanvasSize:function (width, height, noUpdates) {
        var canvas = Module['canvas'];
        canvas.width = width;
        canvas.height = height;
        if (!noUpdates) Browser.updateResizeListeners();
      },windowedWidth:0,windowedHeight:0,setFullScreenCanvasSize:function () {
        var canvas = Module['canvas'];
        this.windowedWidth = canvas.width;
        this.windowedHeight = canvas.height;
        canvas.width = screen.width;
        canvas.height = screen.height;
        // check if SDL is available   
        if (typeof SDL != "undefined") {
        	var flags = HEAPU32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)];
        	flags = flags | 0x00800000; // set SDL_FULLSCREEN flag
        	HEAP32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)]=flags
        }
        Browser.updateResizeListeners();
      },setWindowedCanvasSize:function () {
        var canvas = Module['canvas'];
        canvas.width = this.windowedWidth;
        canvas.height = this.windowedHeight;
        // check if SDL is available       
        if (typeof SDL != "undefined") {
        	var flags = HEAPU32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)];
        	flags = flags & ~0x00800000; // clear SDL_FULLSCREEN flag
        	HEAP32[((SDL.screen+Runtime.QUANTUM_SIZE*0)>>2)]=flags
        }
        Browser.updateResizeListeners();
      }};
FS.staticInit();__ATINIT__.unshift({ func: function() { if (!Module["noFSInit"] && !FS.init.initialized) FS.init() } });__ATMAIN__.push({ func: function() { FS.ignorePermissions = false } });__ATEXIT__.push({ func: function() { FS.quit() } });Module["FS_createFolder"] = FS.createFolder;Module["FS_createPath"] = FS.createPath;Module["FS_createDataFile"] = FS.createDataFile;Module["FS_createPreloadedFile"] = FS.createPreloadedFile;Module["FS_createLazyFile"] = FS.createLazyFile;Module["FS_createLink"] = FS.createLink;Module["FS_createDevice"] = FS.createDevice;
___errno_state = Runtime.staticAlloc(4); HEAP32[((___errno_state)>>2)]=0;
___strtok_state = Runtime.staticAlloc(4);
___buildEnvironment(ENV);
_fputc.ret = allocate([0], "i8", ALLOC_STATIC);
_fgetc.ret = allocate([0], "i8", ALLOC_STATIC);
_llvm_eh_exception.buf = allocate(12, "void*", ALLOC_STATIC);
Module["requestFullScreen"] = function(lockPointer, resizeCanvas) { Browser.requestFullScreen(lockPointer, resizeCanvas) };
  Module["requestAnimationFrame"] = function(func) { Browser.requestAnimationFrame(func) };
  Module["setCanvasSize"] = function(width, height, noUpdates) { Browser.setCanvasSize(width, height, noUpdates) };
  Module["pauseMainLoop"] = function() { Browser.mainLoop.pause() };
  Module["resumeMainLoop"] = function() { Browser.mainLoop.resume() };
  Module["getUserMedia"] = function() { Browser.getUserMedia() }
STACK_BASE = STACKTOP = Runtime.alignMemory(STATICTOP);
staticSealed = true; // seal the static portion of memory
STACK_MAX = STACK_BASE + 5242880;
DYNAMIC_BASE = DYNAMICTOP = Runtime.alignMemory(STACK_MAX);
assert(DYNAMIC_BASE < TOTAL_MEMORY); // Stack must fit in TOTAL_MEMORY; allocations from here on may enlarge TOTAL_MEMORY
 var ctlz_i8 = allocate([8,7,6,6,5,5,5,5,4,4,4,4,4,4,4,4,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,3,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0], "i8", ALLOC_DYNAMIC);
 var cttz_i8 = allocate([8,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,7,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,6,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,5,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0,4,0,1,0,2,0,1,0,3,0,1,0,2,0,1,0], "i8", ALLOC_DYNAMIC);
var Math_min = Math.min;
function invoke_ii(index,a1) {
  try {
    return Module["dynCall_ii"](index,a1);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_viiiii(index,a1,a2,a3,a4,a5) {
  try {
    Module["dynCall_viiiii"](index,a1,a2,a3,a4,a5);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_vi(index,a1) {
  try {
    Module["dynCall_vi"](index,a1);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_vii(index,a1,a2) {
  try {
    Module["dynCall_vii"](index,a1,a2);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_iiii(index,a1,a2,a3) {
  try {
    return Module["dynCall_iiii"](index,a1,a2,a3);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_viii(index,a1,a2,a3) {
  try {
    Module["dynCall_viii"](index,a1,a2,a3);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_v(index) {
  try {
    Module["dynCall_v"](index);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_viiiiii(index,a1,a2,a3,a4,a5,a6) {
  try {
    Module["dynCall_viiiiii"](index,a1,a2,a3,a4,a5,a6);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_iii(index,a1,a2) {
  try {
    return Module["dynCall_iii"](index,a1,a2);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function invoke_viiii(index,a1,a2,a3,a4) {
  try {
    Module["dynCall_viiii"](index,a1,a2,a3,a4);
  } catch(e) {
    if (typeof e !== 'number' && e !== 'longjmp') throw e;
    asm["setThrew"](1, 0);
  }
}
function asmPrintInt(x, y) {
  Module.print('int ' + x + ',' + y);// + ' ' + new Error().stack);
}
function asmPrintFloat(x, y) {
  Module.print('float ' + x + ',' + y);// + ' ' + new Error().stack);
}
// EMSCRIPTEN_START_ASM
var asm=(function(global,env,buffer){"use asm";var a=new global.Int8Array(buffer);var b=new global.Int16Array(buffer);var c=new global.Int32Array(buffer);var d=new global.Uint8Array(buffer);var e=new global.Uint16Array(buffer);var f=new global.Uint32Array(buffer);var g=new global.Float32Array(buffer);var h=new global.Float64Array(buffer);var i=env.STACKTOP|0;var j=env.STACK_MAX|0;var k=env.tempDoublePtr|0;var l=env.ABORT|0;var m=env.cttz_i8|0;var n=env.ctlz_i8|0;var o=env.__ZTVN10__cxxabiv117__class_type_infoE|0;var p=env.__ZTVN10__cxxabiv120__si_class_type_infoE|0;var q=env._stderr|0;var r=+env.NaN;var s=+env.Infinity;var t=0;var u=0;var v=0;var w=0;var x=0,y=0,z=0,A=0,B=0.0,C=0,D=0,E=0,F=0.0;var G=0;var H=0;var I=0;var J=0;var K=0;var L=0;var M=0;var N=0;var O=0;var P=0;var Q=global.Math.floor;var R=global.Math.abs;var S=global.Math.sqrt;var T=global.Math.pow;var U=global.Math.cos;var V=global.Math.sin;var W=global.Math.tan;var X=global.Math.acos;var Y=global.Math.asin;var Z=global.Math.atan;var _=global.Math.atan2;var $=global.Math.exp;var aa=global.Math.log;var ab=global.Math.ceil;var ac=global.Math.imul;var ad=env.abort;var ae=env.assert;var af=env.asmPrintInt;var ag=env.asmPrintFloat;var ah=env.min;var ai=env.invoke_ii;var aj=env.invoke_viiiii;var ak=env.invoke_vi;var al=env.invoke_vii;var am=env.invoke_iiii;var an=env.invoke_viii;var ao=env.invoke_v;var ap=env.invoke_viiiiii;var aq=env.invoke_iii;var ar=env.invoke_viiii;var as=env._lseek;var at=env._sysconf;var au=env._fclose;var av=env._fflush;var aw=env._fputc;var ax=env._strtok;var ay=env._fwrite;var az=env._send;var aA=env._fputs;var aB=env._llvm_umul_with_overflow_i32;var aC=env._fstat;var aD=env._read;var aE=env._fileno;var aF=env._perror;var aG=env._fsync;var aH=env.___gxx_personality_v0;var aI=env._opendir;var aJ=env.___resumeException;var aK=env._round;var aL=env._strncmp;var aM=env._snprintf;var aN=env._fgetc;var aO=env._pclose;var aP=env._readdir;var aQ=env._close;var aR=env._strchr;var aS=env.___setErrNo;var aT=env._ftell;var aU=env._abs;var aV=env._exit;var aW=env._strrchr;var aX=env._fcntl;var aY=env._ferror;var aZ=env.___cxa_is_number_type;var a_=env._localtime_r;var a$=env._ungetc;var a0=env.___cxa_does_inherit;var a1=env._recv;var a2=env.__ZSt18uncaught_exceptionv;var a3=env._cos;var a4=env._putchar;var a5=env.___cxa_call_unexpected;var a6=env._popen;var a7=env._strcmp;var a8=env.__exit;var a9=env._rand;var ba=env._tzset;var bb=env.___cxa_throw;var bc=env._ldexp;var bd=env._isprint;var be=env._llvm_eh_exception;var bf=env._printf;var bg=env._pread;var bh=env._fopen;var bi=env._open;var bj=env._log;var bk=env._puts;var bl=env._fdopen;var bm=env._qsort;var bn=env._dup;var bo=env.___cxa_find_matching_catch;var bp=env._strdup;var bq=env._log10;var br=env.__reallyNegative;var bs=env.__formatString;var bt=env._getenv;var bu=env._llvm_pow_f64;var bv=env._sbrk;var bw=env.___errno_location;var bx=env._strerror;var by=env._fnmatch;var bz=env._vsnprintf;var bA=env._creat;var bB=env._fread;var bC=env._strtok_r;var bD=env._abort;var bE=env._fprintf;var bF=env._isdigit;var bG=env.___buildEnvironment;var bH=env._feof;var bI=env._strncat;var bJ=env._floor;var bK=env._fseek;var bL=env._write;var bM=env._mkstemp;var bN=env.___cxa_allocate_exception;var bO=env._sin;var bP=env._stat;var bQ=env._longjmp;var bR=env._time;var bS=env._closedir;var bT=env._llvm_va_end;var bU=env._unlink;var bV=env.___assert_func;var bW=env._pwrite;var bX=env._strerror_r;var bY=env._exp;var bZ=env._readdir_r;
// EMSCRIPTEN_START_FUNCS
function b8(a){a=a|0;var b=0;b=i;i=i+a|0;i=i+7>>3<<3;return b|0}function b9(){return i|0}function ca(a){a=a|0;i=a}function cb(a,b){a=a|0;b=b|0;if((t|0)==0){t=a;u=b}}function cc(b){b=b|0;a[k]=a[b];a[k+1|0]=a[b+1|0];a[k+2|0]=a[b+2|0];a[k+3|0]=a[b+3|0]}function cd(b){b=b|0;a[k]=a[b];a[k+1|0]=a[b+1|0];a[k+2|0]=a[b+2|0];a[k+3|0]=a[b+3|0];a[k+4|0]=a[b+4|0];a[k+5|0]=a[b+5|0];a[k+6|0]=a[b+6|0];a[k+7|0]=a[b+7|0]}function ce(a){a=a|0;G=a}function cf(a){a=a|0;H=a}function cg(a){a=a|0;I=a}function ch(a){a=a|0;J=a}function ci(a){a=a|0;K=a}function cj(a){a=a|0;L=a}function ck(a){a=a|0;M=a}function cl(a){a=a|0;N=a}function cm(a){a=a|0;O=a}function cn(a){a=a|0;P=a}function co(){c[p+8>>2]=26;c[p+12>>2]=10;c[p+16>>2]=4;c[p+20>>2]=16;c[p+24>>2]=72;c[p+28>>2]=44;c[p+32>>2]=2;c[p+36>>2]=54;c[o+8>>2]=26;c[o+12>>2]=24;c[o+16>>2]=4;c[o+20>>2]=16;c[o+24>>2]=72;c[o+28>>2]=42;c[o+32>>2]=4;c[o+36>>2]=96;c[7978]=o+8;c[7980]=o+8;c[7982]=p+8;c[7986]=p+8;c[7990]=p+8;c[7994]=p+8;c[7998]=o+8}function cp(b){b=b|0;return(a[b+12|0]&1)==0|0}function cq(b,d){b=b|0;d=d|0;var e=0,f=0,g=0;c[b>>2]=31720;c[b+8>>2]=0;e=b+12|0;a[e]=0;f=cy()|0;g=b+4|0;c[g>>2]=f;if((cE(f,d)|0)==0){d=c[g>>2]|0;hE(d,44100,4)|0;return}else{a[e]=1;cz(c[g>>2]|0);return}}function cr(a){a=a|0;var b=0,d=0,e=0;c[a>>2]=31720;b=a+4|0;d=c[b>>2]|0;if((d|0)==0){e=a;p0(e);return}hN(d);cH(c[b>>2]|0);cz(c[b>>2]|0);e=a;p0(e);return}function cs(a){a=a|0;var b=0;c[a>>2]=31720;b=a+4|0;a=c[b>>2]|0;if((a|0)==0){return}hN(a);cH(c[b>>2]|0);cz(c[b>>2]|0);return}function ct(a,d){a=a|0;d=d|0;var e=0,f=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;e=pY(8)|0;f=e;h=e;c[h>>2]=0;c[h+4>>2]=0;h=e;c[h>>2]=0;i=e+4|0;c[i>>2]=0;e=a+4|0;if((hG(c[e>>2]|0)|0)==0){hM(c[e>>2]|0,a+16|0);do{if(d){j=c[a+72>>2]|0;k=a+8|0}else{e=a+8|0;l=c[a+72>>2]|0;if((c[e>>2]|0)>=(l|0)){j=l;k=e;break}return f|0}}while(0);d=c[a+56>>2]|0;c[h>>2]=d;e=c[a+60>>2]|0;c[i>>2]=e;c[k>>2]=j;m=e;n=d}else{m=0;n=0}d=(m|0)/2|0;e=aB(d|0,4)|0;j=pZ(G?-1:e)|0;e=j;if((m|0)>1){k=0;do{g[e+(k<<2)>>2]=+(b[n+(k<<1)>>1]|0)*30517578125.0e-15;k=k+1|0;}while((k|0)<(d|0))}c[i>>2]=m<<1;c[h>>2]=j;return f|0}function cu(a){a=a|0;var b=0,d=0,e=0;b=pY(1624)|0;d=b;cq(d,a);if(cp(d)|0){e=b;return e|0}if((b|0)==0){e=0;return e|0}b0[c[(c[b>>2]|0)+4>>2]&31](d);e=0;return e|0}function cv(a){a=a|0;if((a|0)==0){return}b0[c[(c[a>>2]|0)+4>>2]&31](a);return}function cw(a,b){a=a|0;b=b|0;return ct(a,b)|0}function cx(a){a=a|0;var b=0;b=c[a>>2]|0;if((b|0)!=0){p1(b)}p0(a);return}function cy(){var a=0,b=0;a=pT(1,8104)|0;if((a|0)==0){b=0}else{b=a}return b|0}function cz(a){a=a|0;pU(a);return}function cA(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;h=i;i=i+22528|0;j=h|0;k=h+1024|0;l=h+5120|0;m=h+6144|0;n=b;b=e;e=f;f=g;g=0;o=0;p=0;if((hz(k|0,4096)|0)<0){q=0;r=q;i=h;return r|0}bI(k|0,27216,4096)|0;bK(c[b>>2]|0,0,0)|0;s=bB(j|0,1,1024,c[b>>2]|0)|0;if((s|0)<100){q=0;r=q;i=h;return r|0}do{if((d[j|0]|0|0)==80){if((d[j+1|0]|0|0)!=75){t=70;break}if((d[j+2|0]|0|0)==3){if((d[j+3|0]|0|0)!=4){t=63}}else{t=63}if((t|0)==63){if((d[j+2|0]|0|0)!=48){t=70;break}if((d[j+3|0]|0|0)!=48){t=70;break}if((d[j+4|0]|0|0)!=80){t=70;break}if((d[j+5|0]|0|0)!=75){t=70;break}if((d[j+6|0]|0|0)!=3){t=70;break}if((d[j+7|0]|0|0)!=4){t=70;break}}p=11}else{t=70}}while(0);if((t|0)==70){do{if((d[j+2|0]|0|0)==45){if((d[j+3|0]|0|0)!=108){t=74;break}if((d[j+4|0]|0|0)!=104){t=74;break}p=16}else{t=74}}while(0);if((t|0)==74){do{if((d[j|0]|0|0)==31){if((d[j+1|0]|0|0)!=139){t=77;break}p=12}else{t=77}}while(0);if((t|0)==77){do{if((d[j|0]|0|0)==66){if((d[j+1|0]|0|0)!=90){t=81;break}if((d[j+2|0]|0|0)!=104){t=81;break}p=14}else{t=81}}while(0);if((t|0)==81){do{if((d[j|0]|0|0)==253){if((d[j+3|0]|0|0)!=88){t=86;break}if((d[j+4|0]|0|0)!=90){t=86;break}if((d[j+5|0]|0|0)!=0){t=86;break}p=15}else{t=86}}while(0);if((t|0)==86){do{if((d[j|0]|0|0)==90){if((d[j+1|0]|0|0)!=79){t=91;break}if((d[j+2|0]|0|0)!=79){t=91;break}if((d[j+3|0]|0|0)!=32){t=91;break}p=17}else{t=91}}while(0);if((t|0)==91){do{if((d[j|0]|0|0)==77){if((d[j+1|0]|0|0)!=79){t=95;break}if((d[j+2|0]|0|0)!=51){t=95;break}g=26080}else{t=95}}while(0);if((t|0)==95){do{if((d[j|0]|0|0)==31){if((d[j+1|0]|0|0)!=157){t=98;break}p=13}else{t=98}}while(0);if((t|0)==98){if((p6(j|0,25088,4)|0)==0){p=1}else{do{if((p6(j|0,24288,4)|0)==0){if((p6(j+8|0,23768,4)|0)!=0){t=103;break}p=2}else{t=103}}while(0);if((t|0)==103){if((p6(j|0,23224,8)|0)!=0){if((p6(j|0,22840,8)|0)==0){p=3}else{do{if((p6(j|0,31264,4)|0)==0){if((jk(j+4|0)|0)!=-559038801){t=110;break}p=9}else{t=110}}while(0);if((t|0)==110){do{if((p6(j|0,31264,4)|0)==0){if((jk(j+4|0)|0)!=-559039810){t=113;break}p=9}else{t=113}}while(0);if((t|0)==113){if((p6(j|0,31032,3)|0)==0){p=10}else{if((p6(j|0,30744,3)|0)==0){g=30360}else{if((p6(j|0,30072,4)|0)==0){p=6}else{if((jT(c[b>>2]|0)|0)==0){p=7}}}}}}}}else{p=5}}}}}}}}}}}do{if((p|0)==0){if((g|0)!=0){break}if((d[j|0]|0|0)!=26){break}s=a[j+1|0]&127;u=0;v=0;while(1){if((v|0)>=13){break}if((d[j+(v+2)|0]|0|0)==0){t=144;break}if((bd(d[j+(v+2)|0]|0|0)|0)==0){t=148;break}v=v+1|0}if((t|0)==144){if((v|0)==0){u=1}}else if((t|0)==148){u=1}w=jj(j+15|0)|0;if((w|0)<0){t=153}else{if((w|0)>524288){t=153}}if((t|0)==153){u=1}if((u|0)==0){do{if((s|0)>=1){if((s|0)>9){t=159;break}if((s|0)==7){t=159;break}p=4}else{t=159}}while(0);if((t|0)==159){if((s|0)==127){p=4}}}}}while(0);bK(c[b>>2]|0,0,0)|0;do{if((p|0)==0){if((g|0)!=0){break}q=0;r=q;i=h;return r|0}}while(0);t=pT(12,1)|0;if((t|0)==0){q=-1;r=q;i=h;return r|0}c[t>>2]=bp(k|0)|0;k=bM(c[t>>2]|0)|0;if((k|0)<0){q=-1;r=q;i=h;return r|0}cJ(t+4|0,n);j=bl(k|0,29744)|0;k=j;if((j|0)==0){q=-1;r=q;i=h;return r|0}do{if((g|0)!=0){j=l|0;u=g;v=c[e>>2]|0;aM(j|0,1024,u|0,(u=i,i=i+8|0,c[u>>2]=v,u)|0)|0;i=u;u=a6(l|0,29312)|0;v=u;if((u|0)==0){u=k;au(u|0)|0;q=-1;r=q;i=h;return r|0}else{while(1){u=bB(m|0,1,16384,v|0)|0;if((u|0)<=0){break}ay(m|0,1,u|0,k|0)|0}aO(v|0)|0;break}}else{s=p;if((s|0)==1){o=jW(c[b>>2]|0,k)|0}else if((s|0)==4){o=km(c[b>>2]|0,k)|0}else if((s|0)==5){o=js(c[b>>2]|0,k)|0}else if((s|0)==2){o=k5(c[b>>2]|0,k)|0}else if((s|0)==3){o=jN(c[b>>2]|0,k)|0}else if((s|0)==9){o=jS(c[b>>2]|0,k)|0}else if((s|0)==10){o=kY(c[b>>2]|0,k)|0}else if((s|0)==6){o=kh(c[b>>2]|0,k)|0}else if((s|0)==11){o=lh(c[b>>2]|0,k)|0}else if((s|0)==12){o=jC(c[b>>2]|0,k)|0}else if((s|0)==13){o=kt(c[b>>2]|0,k)|0}else if((s|0)==14){o=jw(c[b>>2]|0,k)|0}else if((s|0)==15){o=lb(c[b>>2]|0,k)|0}else if((s|0)==16){o=ku(c[b>>2]|0,k)|0}else if((s|0)==17){o=ll(c[b>>2]|0,k)|0}else if((s|0)==7){o=jU(c[b>>2]|0,k)|0}}}while(0);if((o|0)<0){p=k;au(p|0)|0;q=-1;r=q;i=h;return r|0}au(c[b>>2]|0)|0;c[b>>2]=k;k=f-1|0;f=k;if((k|0)!=0){k=bp(c[t>>2]|0)|0;o=cA(n,b,t|0,f)|0;pU(k);q=o;r=q;i=h;return r|0}else{q=-1;r=q;i=h;return r|0}return 0}function cB(a,b){a=a|0;b=b|0;var d=0;d=a;a=b;c[a+4>>2]=d;c[d>>2]=a;return}function cC(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=b;b=d;c[b+4>>2]=e;c[e>>2]=b;c[e+4>>2]=a;c[a>>2]=e;return}function cD(a){a=a|0;var b=0,d=0;b=a;a=c[b>>2]|0;while(1){if((a|0)==(b|0)){break}d=a+(-4|0)|0;bU(c[d>>2]|0)|0;pU(c[d>>2]|0);cI(d+4|0);a=c[a>>2]|0;pU(d)}return}function cE(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;f=i;i=i+96|0;g=f|0;h=f+8|0;j=f+16|0;k=f+88|0;l=b;c[g>>2]=e;e=l;b=e+1536|0;if((bP(c[g>>2]|0,j|0)|0)<0){m=-6;n=m;i=f;return n|0}if((c[j+8>>2]&61440|0)==16384){c[(bw()|0)>>2]=21;m=-6;n=m;i=f;return n|0}o=bh(c[g>>2]|0,31272)|0;c[h>>2]=o;if((o|0)==0){m=-6;n=m;i=f;return n|0}c[k>>2]=k;c[k+4>>2]=k;do{if((cA(k,h,g,5)|0)>=0){if((aC(aE(c[h>>2]|0)|0,j|0)|0)<0){break}if((c[j+28>>2]|0)<256){o=c[h>>2]|0;au(o|0)|0;cD(k);m=-3;n=m;i=f;return n|0}cF(c[g>>2]|0,b+1208|0,b+1212|0);c[b+1216>>2]=c[g>>2];c[b+1240>>2]=c[j+28>>2];cM(e);o=-1;p=-1;q=0;while(1){r=8880+(q<<2)|0;if((d[r]|d[r+1|0]<<8|d[r+2|0]<<16|d[r+3|0]<<24|0)==0){break}bK(c[h>>2]|0,0,0)|0;r=8880+(q<<2)|0;p=b2[c[(d[r]|d[r+1|0]<<8|d[r+2|0]<<16|d[r+3|0]<<24)+4>>2]&511](c[h>>2]|0,0,0)|0;if((p|0)==0){s=259;break}q=q+1|0}if((s|0)==259){r=c[h>>2]|0;bK(r|0,0,0)|0;r=8880+(q<<2)|0;o=b2[c[(d[r]|d[r+1|0]<<8|d[r+2|0]<<16|d[r+3|0]<<24)+8>>2]&511](b,c[h>>2]|0,0)|0}cG(c[h>>2]|0,b+1224|0);au(c[h>>2]|0)|0;cD(k);if((p|0)<0){pU(c[b+1212>>2]|0);pU(c[b+1208>>2]|0);m=-3;n=m;i=f;return n|0}if((o|0)<0){cH(l);m=-4;n=m;i=f;return n|0}hy(b|0)|0;if((a[b|0]|0)==0){r=b|0;t=c[b+1212>>2]|0;p5(r|0,t|0,64)|0}cN(e);m=0;n=m;i=f;return n|0}}while(0);au(c[h>>2]|0)|0;cD(k);m=-5;n=m;i=f;return n|0}function cF(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0;f=b;b=d;d=e;e=aW(f|0,47)|0;g=e;if((e|0)!=0){e=g-f+1|0;c[b>>2]=pS(e+1|0)|0;h=c[b>>2]|0;i=f;j=e;p3(h|0,i|0,j)|0;a[(c[b>>2]|0)+e|0]=0;c[d>>2]=bp(g+1|0)|0;return}else{c[b>>2]=bp(36768)|0;c[d>>2]=bp(f|0)|0;return}}function cG(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0;c=i;i=i+16472|0;d=c|0;e=c+16384|0;f=a;a=b;bK(f|0,0,0)|0;hq(e);while(1){b=bB(d|0,1,16384,f|0)|0;if((b|0)<=0){break}hr(e,d|0,b)}hw(a,e);i=c;return}function cH(a){a=a|0;var b=0,d=0;b=a+1536|0;a=b|0;if((c[b+6556>>2]|0)!=0){pU(c[b+6556>>2]|0)}if((c[b+6548>>2]|0)!=0){d=0;while(1){if((d|0)>=(c[a+140>>2]|0)){break}if((c[(c[b+6548>>2]|0)+(d<<2)>>2]|0)!=0){pU(c[(c[b+6548>>2]|0)+(d<<2)>>2]|0)}d=d+1|0}pU(c[b+6548>>2]|0)}if((c[b+6552>>2]|0)!=0){d=0;while(1){if((d|0)>=(c[a+140>>2]|0)){break}if((c[(c[b+6552>>2]|0)+(d<<2)>>2]|0)!=0){pU(c[(c[b+6552>>2]|0)+(d<<2)>>2]|0)}d=d+1|0}pU(c[b+6552>>2]|0)}d=0;while(1){if((d|0)>=(c[a+132>>2]|0)){break}pU(c[(c[a+172>>2]|0)+(d<<2)>>2]|0);d=d+1|0}d=0;while(1){if((d|0)>=(c[a+128>>2]|0)){break}pU(c[(c[a+168>>2]|0)+(d<<2)>>2]|0);d=d+1|0}d=0;while(1){if((d|0)>=(c[a+140>>2]|0)){break}pU(c[(c[a+176>>2]|0)+(d*764|0)+756>>2]|0);if((c[(c[a+176>>2]|0)+(d*764|0)+760>>2]|0)!=0){pU(c[(c[a+176>>2]|0)+(d*764|0)+760>>2]|0)}d=d+1|0}pU(c[a+172>>2]|0);pU(c[a+168>>2]|0);if((c[a+144>>2]|0)>0){d=0;while(1){if((d|0)>=(c[a+144>>2]|0)){break}if((c[(c[a+180>>2]|0)+(d*52|0)+48>>2]|0)!=0){pU((c[(c[a+180>>2]|0)+(d*52|0)+48>>2]|0)-4|0)}d=d+1|0}pU(c[a+180>>2]|0)}pU(c[a+176>>2]|0);if((c[b+1220>>2]|0)!=0){pU(c[b+1220>>2]|0)}pU(c[b+1208>>2]|0);pU(c[b+1212>>2]|0);return}function cI(a){a=a|0;var b=0;b=a;cB(c[b+4>>2]|0,c[b>>2]|0);return}function cJ(a,b){a=a|0;b=b|0;var d=0;d=b;cC(a,c[d+4>>2]|0,d);return}function cK(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;a=0;while(1){if((c[10480+(a<<2)>>2]|0)==0){d=348;break}if((by(c[10480+(a<<2)>>2]|0,b|0,0)|0)==0){d=345;break}a=a+1|0}if((d|0)==345){e=1;f=e;return f|0}else if((d|0)==348){e=0;f=e;return f|0}return 0}function cL(a){a=a|0;var b=0;b=a;do{if((c[b+20>>2]|0)<(c[b+4>>2]|0)){if((c[b+24>>2]|0)>=(c[b+4>>2]|0)){break}return}}while(0);a=b|0;c[a>>2]=c[a>>2]&-5;return}function cM(a){a=a|0;var b=0;b=a+1536|0;p7(b|0,0,64);p7(b+64|0,0,64);h[b+1248>>3]=250.0;c[b+1264>>2]=8287;c[b+1268>>2]=64;c[b+1272>>2]=64;c[b+1276>>2]=0;c[b+1280>>2]=0;c[b+1284>>2]=0;c[b+1220>>2]=0;c[b+128>>2]=0;c[b+132>>2]=0;c[b+136>>2]=4;c[b+140>>2]=0;c[b+144>>2]=0;c[b+148>>2]=6;c[b+152>>2]=125;c[b+156>>2]=0;c[b+160>>2]=0;c[b+6560>>2]=2904;c[b+6556>>2]=0;h[b+1256>>3]=10.0;c[b+6548>>2]=0;c[b+6552>>2]=0;a=0;while(1){if((a|0)>=64){break}c[b+184+(a*12|0)>>2]=(((a+1|0)/2|0|0)%2|0)*255|0;c[b+184+(a*12|0)+4>>2]=64;c[b+184+(a*12|0)+8>>2]=0;a=a+1|0}return}function cN(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;a=b|0;d=b+1536|0;c[d+164>>2]=c[d+1272>>2];if((c[d+160>>2]|0)>=(c[d+156>>2]|0)){c[d+160>>2]=0}if((c[d+148>>2]|0)==0){c[d+148>>2]=6}if((c[d+152>>2]|0)==0){c[d+152>>2]=125}e=0;while(1){if((e|0)>=(c[d+140>>2]|0)){break}if((~c[d+1280>>2]&16384|0)!=0){c[(c[d+176>>2]|0)+(e*764|0)+32>>2]=c[d+1268>>2]}f=0;while(1){if((f|0)>=(c[(c[d+176>>2]|0)+(e*764|0)+36>>2]|0)){break}if((~c[d+1280>>2]&16384|0)!=0){c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+(f<<6)+4>>2]=c[d+1268>>2]}f=f+1|0}e=e+1|0}e=0;while(1){if((e|0)>=(c[d+140>>2]|0)){break}cL((c[d+176>>2]|0)+(e*764|0)+44|0);cL((c[d+176>>2]|0)+(e*764|0)+356|0);cL((c[d+176>>2]|0)+(e*764|0)+200|0);e=e+1|0}c[a+28>>2]=c[a+24>>2];cO(b);h5(b)|0;return}function cO(a){a=a|0;var b=0,d=0,e=0;b=a;a=b|0;d=b+1536|0;b=0;while(1){if((c[6248+(b*20|0)>>2]|0)==0){break}if((p6(d+1224|0,6232+(b*20|0)|0,16)|0)==0){e=a+28|0;c[e>>2]=c[e>>2]|c[6248+(b*20|0)>>2]}b=b+1|0}return}function cP(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;a=b;b=c;c=jd(d)|0;do{if((c&65535|0)!=26982){if((c&65535|0)==19022){break}e=-1;f=e;return f|0}}while(0);bK(d|0,238,1)|0;if(((ja(d)|0)&255|0)!=255){e=-1;f=e;return f|0}else{c=d;g=b+2|0;bK(c|0,g|0,0)|0;di(d,a,36);e=0;f=e;return f|0}return 0}function cQ(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;g=i;i=i+544|0;h=g|0;j=g+504|0;k=g+536|0;l=b;b=e;e=l|0;bK(b|0,f|0,0)|0;bB(h|0,2,1,b|0)|0;bB(h+2|0,108,1,b|0)|0;a[h+110|0]=ja(b)|0;a[h+111|0]=ja(b)|0;a[h+112|0]=ja(b)|0;bB(h+113|0,128,1,b|0)|0;bB(h+241|0,128,1,b|0)|0;bB(h+369|0,128,1,b|0)|0;c[e+136>>2]=8;c[e+140>>2]=d[h+110|0]|0;c[e+128>>2]=d[h+111|0]|0;c[e+132>>2]=ac(c[e+136>>2]|0,c[e+128>>2]|0)|0;f=0;while(1){if((f|0)>=128){break}if((d[h+113+f|0]|0)>(d[h+111|0]|0)){m=408;break}f=f+1|0}c[e+156>>2]=f;n=e+952|0;o=h+113|0;p=c[e+156>>2]|0;p3(n|0,o|0,p)|0;c[e+148>>2]=6;c[e+152>>2]=76;c[e+144>>2]=c[e+140>>2];p=l+1280|0;c[p>>2]=c[p>>2]|4096;dh(e|0,h+2|0,36)|0;dm(l,(aL(h|0,29936,2)|0)!=0?27928:26024,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0);i=p;c[l+1220>>2]=pS(109)|0;p=c[l+1220>>2]|0;o=h+2|0;p3(p|0,o|0,108)|0;a[(c[l+1220>>2]|0)+108|0]=0;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}o=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=o;bB(j|0,13,1,b|0)|0;c[j+16>>2]=jg(b)|0;c[j+20>>2]=jg(b)|0;c[j+24>>2]=jg(b)|0;o=c[j+16>>2]|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=o;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=((o|0)!=0^1^1)&1;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=c[j+20>>2];if((c[j+24>>2]|0)>>>0>=1048575){q=0}else{q=c[j+24>>2]|0}c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=q;c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0)!=0?2:0;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=64;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=f;dh((c[e+176>>2]|0)+(f*764|0)|0,j|0,13)|0;f=f+1|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;f=0;while(1){if((f|0)>=(c[e+128>>2]|0)){break}j=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(f<<2)>>2]=j;c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]=64;j=0;while(1){if((j|0)>=(c[e+136>>2]|0)){break}q=(ac(f,c[e+136>>2]|0)|0)+j|0;c[(c[(c[e+168>>2]|0)+(f<<2)>>2]|0)+4+(j<<2)>>2]=q;o=pT(((c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(q<<2)>>2]=o;c[c[(c[e+172>>2]|0)+(q<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2];j=j+1|0}a[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+4>>2]<<2)>>2]|0)+9|0]=126;a[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+4>>2]<<2)>>2]|0)+10|0]=a[h+241+f|0]|0;a[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+8>>2]<<2)>>2]|0)+4+(d[h+369+f|0]<<3)+5|0]=13;a[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+8>>2]<<2)>>2]|0)+4+(d[h+369+f|0]<<3)+6|0]=0;j=0;while(1){if((j|0)>=512){break}q=(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+4+(((j|0)%8|0)<<2)>>2]<<2)>>2]|0)+4+(((j|0)/8|0)<<3)|0;bB(k|0,1,3,b|0)|0;if((a[k|0]&254|0)!=254){a[q|0]=(d[k|0]>>2)+37&255;a[q+1|0]=((a[k+1|0]&240)>>4)+1+((a[k|0]&3)<<4)&255}if((d[k|0]|0)!=255){a[q+2|0]=((a[k+1|0]&15)<<2)+1&255}L558:do{if((d[k+2|0]|0)!=255){if(((a[k+2|0]&240)>>4|0)>5){break}do{if((((d[k+2|0]|0)<5&240)>>4|0)!=0){if((a[q+1|0]|0)!=0){break}break L558}}while(0);a[q+3|0]=a[8872+((a[k+2|0]&240)>>4)|0]|0;o=d[q+3|0]|0;if((o|0)==121|(o|0)==120|(o|0)==122){a[q+4|0]=d[k+2|0]&15}else if((o|0)==123){a[q+4|0]=1}else if((o|0)==166){a[q+4|0]=((a[k+2|0]&15)<<4)+128&255}else if((o|0)==126){a[q+4|0]=d[k+2|0]&15;a[q+5|0]=127}m=464}else{m=464}}while(0);if((m|0)==464){m=0}j=j+1|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}if((c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0)>2){m=l;k=b;h=(c[e+180>>2]|0)+(f*52|0)|0;gQ(m,k,2,h,0)|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+136>>2]|0)){break}c[e+184+(f*12|0)>>2]=((f|0)%2|0)*255|0;f=f+1|0}f=l+1280|0;c[f>>2]=c[f>>2]|256;i=g;return 0}function cR(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;c=i;i=i+8|0;d=c|0;e=a;a=b;if((bB(d|0,1,7,e|0)|0)>>>0<7){f=-1;g=f;i=c;return g|0}do{if((p6(d|0,29800,7)|0)!=0){if((p6(d|0,23920,7)|0)==0){break}f=-1;g=f;i=c;return g|0}}while(0);di(e,a,0);f=0;g=f;i=c;return g|0}function cS(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;g=i;i=i+728|0;h=g|0;j=g+144|0;k=g+216|0;l=g+472|0;m=b;b=e;e=m|0;bK(b|0,f|0,0)|0;bB(h|0,7,1,b|0)|0;if((aL(h|0,29800,7)|0)==0){c[e+148>>2]=(d[h+7|0]|0|0)/2|0}p5(l|0,c[m+1216>>2]|0,255)|0;f=ax(l|0,27840)|0;a[h+7|0]=ja(b)|0;a[h+8|0]=ja(b)|0;a[h+9|0]=ja(b)|0;bB(h+10|0,128,1,b|0)|0;c[e+156>>2]=d[h+8|0]|0;c[e+160>>2]=d[h+9|0]|0;l=e+952|0;n=h+10|0;o=c[e+156>>2]|0;p3(l|0,n|0,o)|0;o=0;c[e+128>>2]=0;while(1){if((o|0)>=(c[e+156>>2]|0)){break}if((c[e+128>>2]|0)<(d[h+10+o|0]|0|0)){c[e+128>>2]=d[h+10+o|0]|0}o=o+1|0}h=e+128|0;c[h>>2]=(c[h>>2]|0)+1;c[e+140>>2]=31;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+144>>2]=c[e+140>>2];c[m+1264>>2]=8363;dm(m,26656,(h=i,i=i+1|0,i=i+7>>3<<3,c[h>>2]=0,h)|0);i=h;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;o=0;while(1){if((o|0)>=(c[e+128>>2]|0)){break}n=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(o<<2)>>2]=n;c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2]=64;n=0;while(1){if((n|0)>=(c[e+136>>2]|0)){break}l=(ac(o,c[e+136>>2]|0)|0)+n|0;c[(c[(c[e+168>>2]|0)+(o<<2)>>2]|0)+4+(n<<2)>>2]=l;p=pT(((c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(l<<2)>>2]=p;c[c[(c[e+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2];n=n+1|0}n=0;while(1){if((n|0)>=(c[e+136>>2]<<6|0)){break}l=(c[(c[m+172>>2]|0)+(c[(c[(c[m+168>>2]|0)+(o<<2)>>2]|0)+4+(((n|0)%(c[e+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((n|0)/(c[e+136>>2]|0)|0)<<3)|0;q=ja(b)|0;if(q<<24>>24!=0){if((q&255|0)==37){r=97}else{r=(q&255)+48|0}a[l|0]=r&255}a[l+1|0]=ja(b)|0;n=n+1|0}o=o+1|0}c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}o=0;while(1){if((o|0)>=(c[e+140>>2]|0)){break}r=pT(64,1)|0;c[(c[e+176>>2]|0)+(o*764|0)+756>>2]=r;aM(k|0,255,25600,(h=i,i=i+16|0,c[h>>2]=f,c[h+8>>2]=o+1,h)|0)|0;i=h;r=bh(k|0,24664)|0;n=(r|0)!=0;c[(c[e+176>>2]|0)+(o*764|0)+36>>2]=n&1;if(n){n=aE(r|0)|0;aC(n|0,j|0)|0;q=ja(r)|0;c[(c[e+180>>2]|0)+(o*52|0)+32>>2]=(c[j+28>>2]|0)-(((q<<24>>24!=0^1)&1)*5|0);if(q<<24>>24!=0){n=r;bK(n|0,0,0)|0}else{n=(jc(b)|0)&65535;c[(c[e+180>>2]|0)+(o*52|0)+36>>2]=n;n=(jc(b)|0)&65535;c[(c[e+180>>2]|0)+(o*52|0)+40>>2]=n;c[(c[e+180>>2]|0)+(o*52|0)+44>>2]=(c[(c[e+180>>2]|0)+(o*52|0)+40>>2]|0)>(c[(c[e+180>>2]|0)+(o*52|0)+36>>2]|0)?2:0}c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+8>>2]=128;c[c[(c[e+176>>2]|0)+(o*764|0)+756>>2]>>2]=64;c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]=o;n=m;l=r;p=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(n,l,2,p,0)|0;p=r;au(p|0)|0}o=o+1|0}o=0;while(1){if((o|0)>=(c[e+136>>2]|0)){break}c[e+184+(o*12|0)>>2]=((o|0)%2|0)*255|0;o=o+1|0}i=g;return 0}function cT(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,j=0;d=i;i=i+16|0;e=d|0;f=a;a=b;b=c;bK(f|0,b+1062|0,0)|0;if((bB(e|0,1,9,f|0)|0)>>>0<9){g=-1;h=g;i=d;return h|0}do{if((p6(e|0,27664,2)|0)==0){if((p6(e+6|0,26456,3)|0)!=0){break}c=f;j=b|0;bK(c|0,j|0,0)|0;di(f,a,24);g=0;h=g;i=d;return h|0}}while(0);g=-1;h=g;i=d;return h|0}function cU(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;g=i;i=i+1088|0;h=g|0;j=g+1072|0;k=b;b=e;e=k|0;l=1;bK(b|0,f|0,0)|0;bB(h|0,24,1,b|0)|0;bB(h+24|0,24,1,b|0)|0;f=0;while(1){if((f|0)>=26){break}bB(h+48+(f*34|0)|0,23,1,b|0)|0;bB(h+48+(f*34|0)+23|0,11,1,b|0)|0;f=f+1|0}a[h+932|0]=ja(b)|0;a[h+933|0]=ja(b)|0;bB(h+934|0,128,1,b|0)|0;bB(h+1062|0,9,1,b|0)|0;a[h+1071|0]=ja(b)|0;c[e+136>>2]=9;c[e+152>>2]=125;c[e+148>>2]=6;c[e+156>>2]=d[h+932|0]|0;c[e+128>>2]=(d[h+933|0]|0)+1;c[e+140>>2]=26;c[e+144>>2]=c[e+140>>2];m=e+952|0;n=h+934|0;o=c[e+156>>2]|0;p3(m|0,n|0,o)|0;dm(k,29424,(o=i,i=i+1|0,i=i+7>>3<<3,c[o>>2]=0,o)|0);i=o;p5(e|0,h|0,24)|0;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}o=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=o;dh((c[e+176>>2]|0)+(f*764|0)|0,h+48+(f*34|0)|0,23)|0;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=1;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=64;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=f;p=0;while(1){if((p|0)>=11){break}a[j+p|0]=a[h+48+(f*34|0)+23+(c[21944+(p<<2)>>2]|0)|0]|0;p=p+1|0}gQ(k,b,4096,(c[e+180>>2]|0)+(f*52|0)|0,j|0)|0;f=f+1|0}if((a[h+1071|0]|0)==0){q=-1;r=q;i=g;return r|0}c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;f=0;while(1){if((f|0)>=(c[e+128>>2]|0)){break}h=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(f<<2)>>2]=h;p=0;while(1){if((p|0)>=9){break}s=jc(b)|0;c[(c[(c[e+168>>2]|0)+(f<<2)>>2]|0)+4+(p<<2)>>2]=s&65535;if((s&65535|0)>(c[e+132>>2]|0)){c[e+132>>2]=s&65535}p=p+1|0}c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]=64;f=f+1|0}p=e+132|0;c[p>>2]=(c[p>>2]|0)+1;p=(jc(b)|0)&65535;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;f=0;while(1){if((f|0)>=(p|0)){break}s=jc(b)|0;h=pT(524,1)|0;c[(c[e+172>>2]|0)+((s&65535)<<2)>>2]=h;c[c[(c[e+172>>2]|0)+((s&65535)<<2)>>2]>>2]=64;h=0;while(1){if((h|0)>=64){break}j=(c[(c[e+172>>2]|0)+((s&65535)<<2)>>2]|0)+4+(h<<3)|0;o=ja(b)|0;if((o&128|0)!=0){h=h+((o&127)-1)|0}else{a[j+4|0]=o;o=ja(b)|0;a[j+1|0]=(o&240)>>4&255;n=o&255&15;o=n;m=n&255;do{if(!((m|0)==0|(m|0)==1|(m|0)==2))if((m|0)==8){o=3}else if((m|0)==4){o=12}else if((m|0)==3|(m|0)==9){o=0;a[j+4|0]=0}else if((m|0)==5){o=11}else if((m|0)==6){o=13}else if((m|0)==7){if((a[j+4|0]|0)==0){l=3}if((d[j+4|0]|0)>31){o=0;a[j+4|0]=0;break}else{n=j+4|0;a[n]=(ac(d[n]|0,l)|0)&255;o=15;break}}}while(0);a[j+3|0]=o;o=ja(b)|0;m=j+1|0;a[m]=(d[m]|(o&1)<<4)&255;m=(o&240)>>4&255;a[j|0]=m;if(m<<24>>24!=0){m=j|0;a[m]=(d[m]|0)+((((o&14)>>1)+2|0)*12|0)&255}}h=h+1|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+136>>2]|0)){break}c[e+184+(f*12|0)>>2]=128;c[e+184+(f*12|0)+8>>2]=1;f=f+1|0}c[k+6560>>2]=2936;q=0;r=q;i=g;return r|0}function cV(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0;d=i;i=i+8|0;e=d|0;f=b;b=c;if((bB(e|0,1,3,f|0)|0)>>>0<3){g=-1;h=g;i=d;return h|0}do{if((a[e|0]|0)==65){if((a[e+1|0]|0)!=77){break}if((a[e+2|0]|0)!=70){break}c=(ja(f)|0)&255;do{if((c|0)>=10){if((c|0)>14){break}di(f,b,32);g=0;h=g;i=d;return h|0}}while(0);g=-1;h=g;i=d;return h|0}}while(0);g=-1;h=g;i=d;return h|0}function cW(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;f=i;i=i+1024|0;g=f|0;h=b;b=d;d=h|0;bK(b|0,e|0,0)|0;bB(g|0,1,3,b|0)|0;e=(ja(b)|0)&255;bB(g|0,1,32,b|0)|0;p5(d|0,g|0,32)|0;dm(h,29208,(j=i,i=i+16|0,c[j>>2]=(e|0)/10|0,c[j+8>>2]=(e|0)%10|0,j)|0);i=j;c[d+140>>2]=(ja(b)|0)&255;c[d+156>>2]=(ja(b)|0)&255;c[d+132>>2]=(jc(b)|0)&65535;c[d+136>>2]=(ja(b)|0)&255;c[d+144>>2]=c[d+140>>2];c[d+128>>2]=c[d+156>>2];if((e|0)==10){j=g|0;k=b;bB(j|0,1,16,k|0)|0}if((e|0)>=13){k=g|0;j=b;bB(k|0,1,32,j|0)|0;l=0;while(1){if((l|0)>=32){break}c[d+184>>2]=(a[g+l|0]<<1)+128;l=l+1|0}c[d+152>>2]=(ja(b)|0)&255;c[d+148>>2]=(ja(b)|0)&255}else{if((e|0)>=11){j=g|0;k=b;bB(j|0,1,16,k|0)|0}}l=0;while(1){if((l|0)>=(c[d+156>>2]|0)){break}a[d+952+l|0]=l&255;l=l+1|0}c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;l=0;while(1){if((l|0)>=(c[d+128>>2]|0)){break}k=pT(1,((c[d+136>>2]|0)-1<<2)+8|0)|0;c[(c[d+168>>2]|0)+(l<<2)>>2]=k;if((e|0)>=14){m=(jc(b)|0)&65535}else{m=64}c[c[(c[d+168>>2]|0)+(l<<2)>>2]>>2]=m;n=0;while(1){if((n|0)>=(c[d+136>>2]|0)){break}k=(jc(b)|0)&65535;c[(c[(c[d+168>>2]|0)+(l<<2)>>2]|0)+4+(n<<2)>>2]=k;n=n+1|0}l=l+1|0}c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}if((e|0)<=10){m=aT(b|0)|0;l=0;while(1){if((l|0)>=(c[d+140>>2]|0)){break}k=ja(b)|0;if((k&255|0)!=0){if((k&255|0)!=1){o=706;break}}bK(b|0,45,1)|0;if((jg(b)|0)>>>0>1048576){o=708;break}k=jg(b)|0;if((k|0)>1048576){o=710;break}if(((jc(b)|0)&65535|0)==0){o=712;break}if(((ja(b)|0)&255|0)>64){o=714;break}if((jg(b)|0)>(k|0)){o=716;break}if((jg(b)|0)>(k|0)){o=718;break}l=l+1|0}if((o|0)==706){e=9}else if((o|0)==708){e=9}else if((o|0)==710){e=9}else if((o|0)==712){e=9}else if((o|0)==714){e=9}else if((o|0)==716){e=9}else if((o|0)==718){e=9}bK(b|0,m|0,0)|0}l=0;while(1){if((l|0)>=(c[d+140>>2]|0)){break}m=pT(64,1)|0;c[(c[d+176>>2]|0)+(l*764|0)+756>>2]=m;m=((ja(b)|0)&255|0)!=0?1:0;c[(c[d+176>>2]|0)+(l*764|0)+36>>2]=m;bB(g|0,1,32,b|0)|0;dh((c[d+176>>2]|0)+(l*764|0)|0,g|0,32)|0;bB(g|0,1,13,b|0)|0;jg(b)|0;c[(c[(c[d+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]=l;c[(c[(c[d+176>>2]|0)+(l*764|0)+756>>2]|0)+8>>2]=128;m=jg(b)|0;c[(c[d+180>>2]|0)+(l*52|0)+32>>2]=m;m=(jc(b)|0)&65535;hD(m,(c[(c[d+176>>2]|0)+(l*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(l*764|0)+756>>2]|0)+16|0);m=(ja(b)|0)&255;c[c[(c[d+176>>2]|0)+(l*764|0)+756>>2]>>2]=m;if((e|0)<10){m=(jc(b)|0)&65535;c[(c[d+180>>2]|0)+(l*52|0)+36>>2]=m;c[(c[d+180>>2]|0)+(l*52|0)+40>>2]=(c[(c[d+180>>2]|0)+(l*52|0)+32>>2]|0)-1}else{m=jg(b)|0;c[(c[d+180>>2]|0)+(l*52|0)+36>>2]=m;m=jg(b)|0;c[(c[d+180>>2]|0)+(l*52|0)+40>>2]=m}if((e|0)<10){c[(c[d+180>>2]|0)+(l*52|0)+44>>2]=(c[(c[d+180>>2]|0)+(l*52|0)+36>>2]|0)>0?2:0}else{c[(c[d+180>>2]|0)+(l*52|0)+44>>2]=(c[(c[d+180>>2]|0)+(l*52|0)+40>>2]|0)>(c[(c[d+180>>2]|0)+(l*52|0)+36>>2]|0)?2:0}l=l+1|0}e=pT(4,c[d+132>>2]|0)|0;g=0;l=0;while(1){if((l|0)>=(c[d+132>>2]|0)){break}m=jc(b)|0;c[e+(l<<2)>>2]=m&65535;if((m&65535|0)>(g|0)){g=m&65535}l=l+1|0}l=0;while(1){if((l|0)>=(c[d+128>>2]|0)){break}n=0;while(1){if((n|0)>=(c[d+136>>2]|0)){break}m=(c[(c[(c[d+168>>2]|0)+(l<<2)>>2]|0)+4+(n<<2)>>2]|0)-1|0;if((m|0)<0){o=746}else{if((m|0)>(c[d+132>>2]|0)){o=746}}if((o|0)==746){o=0;m=0}c[(c[(c[d+168>>2]|0)+(l<<2)>>2]|0)+4+(n<<2)>>2]=c[e+(m<<2)>>2];n=n+1|0}l=l+1|0}c[d+132>>2]=g;pU(e);e=d+132|0;c[e>>2]=(c[e>>2]|0)+1;c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;e=pT(523,1)|0;c[c[d+172>>2]>>2]=e;c[c[c[d+172>>2]>>2]>>2]=64;l=1;while(1){if((l|0)>=(c[d+132>>2]|0)){break}e=pT(523,1)|0;c[(c[d+172>>2]|0)+(l<<2)>>2]=e;c[c[(c[d+172>>2]|0)+(l<<2)>>2]>>2]=64;e=je(b)|0;n=0;L924:while(1){if((n|0)>=(e|0)){break}g=ja(b)|0;m=ja(b)|0;k=ja(b)|0;do{if((g&255|0)==255){if((m&255|0)!=255){break}if((k&255|0)==255){o=760;break L924}}}while(0);j=(c[(c[d+172>>2]|0)+(l<<2)>>2]|0)+4+((g&255)<<3)|0;if((m&255|0)<127){if((m&255|0)>0){a[j|0]=(m&255)+1&255}a[j+2|0]=k}else{if((m&255|0)==127){p=j;q=(c[(c[d+172>>2]|0)+(l<<2)>>2]|0)+4+((g&255)-1<<3)|0;p3(p|0,q|0,8)|0}else{if((m&255|0)==128){a[j+1|0]=(k&255)+1&255}else{q=0;p=0;r=m&255;if((r|0)!=133)if((r|0)==134){q=3;p=k}else if((r|0)==135){q=7;p=k}else if((r|0)==136){q=0;p=k}else if((r|0)==137){q=4;p=k}else if((r|0)==138){if((k<<24>>24|0)>0){q=5;p=(k&255)<<4&255}else{q=5;p=-(k<<24>>24)&15}}else if((r|0)==139){if((k<<24>>24|0)>0){q=6;p=(k&255)<<4&255}else{q=6;p=-(k<<24>>24)&15}}else if((r|0)==140){q=13;p=k}else if((r|0)==141){q=11;p=k}else if((r|0)!=142)if((r|0)==143){q=14;p=(144|k&15)&255}else if((r|0)==144){q=9;p=k}else if((r|0)==145){if((k<<24>>24|0)>0){q=14;p=(160|k&15)&255}else{q=14;p=(176|k&15)&255}}else if((r|0)==146){if((k<<24>>24|0)>0){q=2;p=(240|p&15)&255}else{q=1;p=(240|p&15)&255}}else if((r|0)==129){q=15;p=k}else if((r|0)==151){q=8;p=(k<<24>>24<<1)+128&255}else if((r|0)==132){if((k<<24>>24|0)>=0){q=2;p=k}if((k&255|0)==128){q=1;p=0}else{q=1;p=-(k<<24>>24)&255}}else if((r|0)==147){q=14;p=(208|k&15)&255}else if((r|0)==148){q=14;p=(192|k&15)&255}else if((r|0)==149){q=15;if((k&255|0)<33){k=33}p=k}else if((r|0)==150){if((k<<24>>24|0)>0){q=2;p=(224|p&15)&255}else{q=1;p=(224|p&15)&255}}else if((r|0)==130){if((k<<24>>24|0)>0){q=10;p=(k&255)<<4&255}else{q=10;p=-(k<<24>>24)&15}}else if((r|0)==131){a[j+2|0]=k}a[j+3|0]=q;a[j+4|0]=p}}}n=n+1|0}if((o|0)==760){o=0}l=l+1|0}l=0;while(1){if((l|0)>=(c[d+140>>2]|0)){break}gQ(h,b,2,(c[d+180>>2]|0)+((c[(c[(c[d+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;l=l+1|0}l=h+1280|0;c[l>>2]=c[l>>2]|32;i=f;return 0}function cX(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;return}function cY(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;c=a;a=b;if((jl(c)|0)!=1297437528){d=-1;e=d;return e|0}jg(c)|0;while(1){if(!((bH(c|0)|0)!=0^1)){f=840;break}b=jl(c)|0;g=jg(c)|0;if((b|0)==1296974157){f=838;break}bK(c|0,g|0,1)|0}if((f|0)==838){di(c,a,32);d=0;e=d;return e|0}else if((f|0)==840){di(c,a,0);d=0;e=d;return e|0}return 0}function cZ(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+96|0;f=e|0;g=a;a=b;b=g|0;bK(a|0,d|0,0)|0;jl(a)|0;jl(a)|0;c[f+16>>2]=0;c[f+12>>2]=0;c[f+8>>2]=0;c[f+4>>2]=0;c[f>>2]=0;d=es()|0;if((d|0)==0){h=-1;j=h;i=e;return j|0}ev(d,29200,110);ev(d,27600,84);ev(d,26408,142);ev(d,25368,12);ev(d,24504,120);ev(d,23864,98);ev(d,23416,116);ev(d,22912,104);ev(d,22200,88);ev(d,31072,42);ev(d,30800,14);ey(d,1);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(d,g,a,f)}eB(d);d=0;while(1){if((d|0)>=(c[b+136>>2]|0)){break}c[b+184+(d*12|0)>>2]=(((d+3|0)/2|0|0)%2|0)*255|0;d=d+1|0}h=0;j=h;i=e;return j|0}function c_(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;b=d;d=e;e=(ja(b)|0)&255;c[d>>2]=(((e&240)>>4)*10|0)+(e&15);e=(ja(b)|0)&255;a=d|0;c[a>>2]=(c[a>>2]|0)+((((e&240)>>4)*1e3|0)+((e&15)*100|0));e=(ja(b)|0)&255;c[d+4>>2]=(((e&240)>>4)*10|0)+(e&15);e=(ja(b)|0)&255;c[d+8>>2]=(((e&240)>>4)*10|0)+(e&15);return}function c$(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;c[a+136>>2]=jg(d)|0;return}function c0(a,b,e,f){a=a|0;b=b|0;e=e|0;f=f|0;b=a|0;a=f;bB(a+24|0,1,8,e|0)|0;e=0;while(1){if((e|0)>=(c[b+136>>2]|0)){break}do{if((d[a+24+e|0]|0|0)>0){if((d[a+24+e|0]|0|0)>=8){break}c[b+184+(e*12|0)>>2]=((d[a+24+e|0]|0)*42|0)-40}}while(0);e=e+1|0}return}function c1(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;bB(a|0,1,32,c|0)|0;return}function c2(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;c[a+156>>2]=jg(d)|0;return}function c3(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;c[a+128>>2]=jg(d)|0;return}function c4(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;bB(d+32|0,1,64,c|0)|0;return}function c5(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=i;b=a;bB(b+952|0,1,128,d|0)|0;dm(b,23944,(b=i,i=i+1|0,i=i+7>>3<<3,c[b>>2]=0,b)|0);i=b;i=e;return}function c6(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;e=b;b=f;f=e|0;h=g;if((c[h+12>>2]|0)==0){c[h+12>>2]=1;c[8926]=0;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0}g=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(c[8926]<<2)>>2]=g;c[c[(c[f+168>>2]|0)+(c[8926]<<2)>>2]>>2]=d[h+32+(c[8926]|0)|0]|0;g=0;while(1){if((g|0)>=(c[f+136>>2]|0)){break}i=(ac(c[8926]|0,c[f+136>>2]|0)|0)+g|0;c[(c[(c[f+168>>2]|0)+(c[8926]<<2)>>2]|0)+4+(g<<2)>>2]=i;j=pT(((c[c[(c[f+168>>2]|0)+(c[8926]<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(i<<2)>>2]=j;c[c[(c[f+172>>2]|0)+(i<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(c[8926]<<2)>>2]>>2];g=g+1|0}g=0;while(1){if((g|0)>=(d[h+32+(c[8926]|0)|0]|0)){break}i=0;while(1){if((i|0)>=(c[f+136>>2]|0)){break}j=(c[(c[e+172>>2]|0)+(c[(c[(c[e+168>>2]|0)+(c[8926]<<2)>>2]|0)+4+(i<<2)>>2]<<2)>>2]|0)+4+(g<<3)|0;a[j+4|0]=ja(b)|0;a[j+3|0]=ja(b)|0;a[j+1|0]=ja(b)|0;a[j|0]=ja(b)|0;if((a[j|0]|0)!=0){k=j|0;a[k]=(d[k]|0)+48&255}c7(j);i=i+1|0}g=g+1|0}c[8926]=(c[8926]|0)+1;return}function c7(b){b=b|0;var c=0,e=0;c=b;b=d[c+3|0]|0;if((b|0)==0){a[c+3|0]=0;return}else if((b|0)==1){a[c+3|0]=1;return}else if((b|0)==2){a[c+3|0]=2;return}else if((b|0)==3){a[c+3|0]=3;return}else if((b|0)==11){a[c+3|0]=13;return}else if((b|0)==12){if((d[c+4|0]|0|0)<=64){a[c+3|0]=12;a[c+4|0]=a[6920+(d[c+4|0]|0)|0]|0}else{a[c+3|0]=0;a[c+4|0]=0}return}else if((b|0)==14|(b|0)==25){do{if((d[c+4|0]|0|0)>0){if((d[c+4|0]|0|0)>=8){e=919;break}a[c+3|0]=8;a[c+4|0]=((d[c+4|0]|0)*42|0)-40&255}else{e=919}}while(0);if((e|0)==919){a[c+4|0]=0;a[c+3|0]=0}return}else if((b|0)==19){a[c+3|0]=11;return}else if((b|0)==28){a[c+3|0]=15;return}else if((b|0)==31){a[c+3|0]=12;return}else if((b|0)==21){a[c+4|0]=0;a[c+3|0]=0;return}else if((b|0)==16){a[c+3|0]=-96;return}else if((b|0)==17){a[c+3|0]=-95;return}else{a[c+4|0]=0;a[c+3|0]=0;return}}function c8(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;b=a;a=d;d=b|0;f=e;if((c[f+16>>2]|0)==0){c[d+140>>2]=36;c[d+144>>2]=36;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}c[f+16>>2]=1;c[f+20>>2]=0;c[8924]=0}if((c[8924]|0)>=36){return}e=pT(64,1)|0;c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+756>>2]=e;jg(a)|0;e=jg(a)|0;if((e|0)<32){g=(c[d+176>>2]|0)+((c[8924]|0)*764|0)|0;h=e;e=a;bB(g|0,1,h|0,e|0)|0}jg(a)|0;jg(a)|0;e=(jg(a)|0)&255;c[c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+756>>2]>>2]=e;jg(a)|0;jg(a)|0;e=jg(a)|0;c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+32>>2]=e;jg(a)|0;jg(a)|0;e=jg(a)|0;c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+36>>2]=e;jg(a)|0;jg(a)|0;e=jg(a)|0;c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]=e;jg(a)|0;jg(a)|0;jg(a)|0;c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+36>>2]=1;c[(c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+756>>2]|0)+40>>2]=c[8924];c[(c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+756>>2]|0)+8>>2]=128;c[b+1276>>2]=20864;c[b+1268>>2]=255;if((c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]|0)>2){c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+44>>2]=2;c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]=(c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+36>>2]|0)+(c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]|0)}else{do{if((c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]|0)==2){if((c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+36>>2]|0)<=0){break}c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+44>>2]=2;c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+40>>2]=c[(c[d+180>>2]|0)+((c[8924]|0)*52|0)+32>>2]}}while(0)}gQ(b,a,128,(c[d+180>>2]|0)+((c[(c[(c[d+176>>2]|0)+((c[8924]|0)*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;c[8924]=(c[8924]|0)+1;d=f+20|0;c[d>>2]=(c[d>>2]|0)+1;return}function c9(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+32|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,32,e|0)|0)>>>0<32){f=-1}else{if((p6(d|0,27376,32)|0)!=0){f=-1;break}else{di(e,a,0);f=0;break}}}while(0);i=c;return f|0}function da(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0;g=i;i=i+40|0;h=g|0;j=b;b=e;e=f;f=j|0;bK(b|0,e|0,0)|0;bK(b|0,32,1)|0;c[f+148>>2]=(ja(b)|0)&255;c[f+152>>2]=(ja(b)|0)&255;c[f+140>>2]=(ja(b)|0)&255;c[f+128>>2]=(ja(b)|0)&255;c[f+156>>2]=(ja(b)|0)&255;ja(b)|0;bB(f+952|0,1,c[f+156>>2]|0,b|0)|0;bK(b|0,e+294|0,0)|0;c[f+136>>2]=8;c[f+144>>2]=c[f+140>>2];c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;aM(f+64|0,64,28856,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0)|0;i=e;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}e=0;while(1){if((e|0)>=(c[f+140>>2]|0)){break}k=pT(64,1)|0;c[(c[f+176>>2]|0)+(e*764|0)+756>>2]=k;bB(h|0,1,37,b|0)|0;dh((c[f+176>>2]|0)+(e*764|0)|0,h|0,22)|0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+16>>2]=(d[h+22|0]<<4&255)<<24>>24;c[c[(c[f+176>>2]|0)+(e*764|0)+756>>2]>>2]=d[h+23|0]|0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+12>>2]=a[h+24|0]|0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+40>>2]=e;k=jj(h+25|0)|0;c[(c[f+180>>2]|0)+(e*52|0)+32>>2]=k;k=jj(h+29|0)|0;c[(c[f+180>>2]|0)+(e*52|0)+36>>2]=k;k=c[(c[f+180>>2]|0)+(e*52|0)+36>>2]|0;l=k+(jj(h+33|0)|0)|0;c[(c[f+180>>2]|0)+(e*52|0)+40>>2]=l;c[(c[f+176>>2]|0)+(e*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(e*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[f+180>>2]|0)+(e*52|0)+44>>2]=(c[(c[f+180>>2]|0)+(e*52|0)+40>>2]|0)>2?2:0;e=e+1|0}bK(b|0,(64-(c[f+140>>2]|0)|0)*37|0|0,1)|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;e=0;while(1){if((e|0)>=(c[f+128>>2]|0)){break}h=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(e<<2)>>2]=h;c[c[(c[f+168>>2]|0)+(e<<2)>>2]>>2]=64;h=0;while(1){if((h|0)>=(c[f+136>>2]|0)){break}l=(ac(e,c[f+136>>2]|0)|0)+h|0;c[(c[(c[f+168>>2]|0)+(e<<2)>>2]|0)+4+(h<<2)>>2]=l;k=pT(((c[c[(c[f+168>>2]|0)+(e<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(l<<2)>>2]=k;c[c[(c[f+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(e<<2)>>2]>>2];h=h+1|0}h=0;while(1){if((h|0)>=(c[f+136>>2]<<6|0)){break}l=(c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(e<<2)>>2]|0)+4+(((h|0)%(c[f+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((h|0)/(c[f+136>>2]|0)|0)<<3)|0;p7(l|0,0,8);k=ja(b)|0;if((k&255|0)!=0){a[l|0]=(k&255)+13&255}a[l+1|0]=ja(b)|0;a[l+3|0]=ja(b)|0;a[l+4|0]=ja(b)|0;h=h+1|0}e=e+1|0}e=0;while(1){if((e|0)>=(c[f+140>>2]|0)){break}if((c[(c[f+180>>2]|0)+(e*52|0)+32>>2]|0)>1){h=j;l=b;k=(c[f+180>>2]|0)+(e*52|0)|0;gQ(h,l,0,k,0)|0}else{c[(c[f+176>>2]|0)+(e*764|0)+36>>2]=0}e=e+1|0}i=g;return 0}function db(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;d=i;i=i+24|0;e=d|0;f=a;a=b;b=c;c=ja(f)|0;do{if((c&255|0)!=132){if((c&255|0)==136){g=1028;break}h=-1}else{g=1028}}while(0);L1244:do{if((g|0)==1028){c=e|0;j=f;bB(c|0,1,20,j|0)|0;if((dd(e|0,20)|0)!=0){h=-1;break}j=(ja(f)|0)&255;if((j|0)>100){h=-1;break}c=f;ja(c)|0;c=f;ja(c)|0;c=jg(f)|0;do{if(c>>>0>=64){if(c>>>0>1048576){break}c=jg(f)|0;do{if(c>>>0>=64){if(c>>>0>1048576){break}k=0;while(1){if((k|0)>=(j|0)){g=1055;break}l=jg(f)|0;m=jg(f)|0;n=jg(f)|0;o=jg(f)|0;p=jg(f)|0;if((l|0)<64){g=1042;break}if((l|0)>1048576){g=1042;break}if((n|0)>255){g=1044;break}if((m|0)>1048576){g=1048;break}if((o|0)>1048576){g=1048;break}if((p|0)>1048576){g=1048;break}if((o+p-1|0)>(m|0)){g=1050;break}bB(e|0,1,11,f|0)|0;if((dd(e|0,11)|0)!=0){g=1052;break}ja(f)|0;k=k+1|0}if((g|0)==1042){h=-1;break L1244}else if((g|0)==1044){h=-1;break L1244}else if((g|0)==1048){h=-1;break L1244}else if((g|0)==1050){h=-1;break L1244}else if((g|0)==1052){h=-1;break L1244}else if((g|0)==1055){bK(f|0,b+1|0,0)|0;di(f,a,20);h=0;break L1244}}}while(0);h=-1;break L1244}}while(0);h=-1}}while(0);i=d;return h|0}function dc(b){b=b|0;var c=0;c=b;b=d[c+3|0]|0;if((b|0)==13){a[c+3|0]=13;return}else if((b|0)==15){a[c+3|0]=15;return}else if((b|0)==7){a[c+3|0]=8;return}else if((b|0)==14){a[c+3|0]=11;return}else if((b|0)==12){a[c+3|0]=12;a[c+4|0]=255-(d[c+4|0]|0)&255;return}else if((b|0)==0){a[c+3|0]=0;return}else if((b|0)==1|(b|0)==5){a[c+3|0]=1;return}else if((b|0)==2|(b|0)==6){a[c+3|0]=2;return}else if((b|0)==3){a[c+3|0]=-96;return}else if((b|0)==4){a[c+3|0]=-95;return}else if((b|0)==19){a[c+3|0]=-96;return}else if((b|0)==20){a[c+3|0]=-95;return}else if((b|0)==8|(b|0)==9|(b|0)==10|(b|0)==11){a[c+4|0]=0;a[c+3|0]=0;return}else if((b|0)==16){a[c+4|0]=0;a[c+3|0]=0;return}else if((b|0)==17|(b|0)==18){a[c+4|0]=0;a[c+3|0]=0;return}else{a[c+4|0]=0;a[c+3|0]=0;return}}function dd(a,b){a=a|0;b=b|0;var c=0,e=0,f=0,g=0;c=a;a=b;while(1){b=a;a=b-1|0;if((b|0)==0){e=1096;break}b=c;c=b+1|0;if((d[b]|0|0)==13){e=1094;break}}if((e|0)==1096){f=-1;g=f;return g|0}else if((e|0)==1094){f=0;g=f;return g|0}return 0}function de(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;g=i;i=i+400|0;h=g|0;j=b;b=e;e=f;f=j|0;bK(b|0,e|0,0)|0;c[f+136>>2]=(ja(b)|0)&63;di(b,f|0,20);k=0;while(1){if((k|0)>=20){break}if((a[f+k|0]|0)==13){a[f+k|0]=0}k=k+1|0}dm(j,22984,(l=i,i=i+1|0,i=i+7>>3<<3,c[l>>2]=0,l)|0);i=l;l=(ja(b)|0)&255;c[f+144>>2]=l;c[f+140>>2]=l;c[f+156>>2]=(ja(b)|0)&255;c[f+128>>2]=(ja(b)|0)&255;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;l=jg(b)|0;jg(b)|0;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}c[j+1276>>2]=20864;c[j+1268>>2]=255;k=0;while(1){if((k|0)>=(c[f+140>>2]|0)){break}m=pT(64,1)|0;c[(c[f+176>>2]|0)+(k*764|0)+756>>2]=m;c[h+(k<<2)>>2]=jg(b)|0;m=jg(b)|0;c[(c[f+180>>2]|0)+(k*52|0)+32>>2]=m;m=255-(jg(b)|0)|0;c[c[(c[f+176>>2]|0)+(k*764|0)+756>>2]>>2]=m;c[(c[(c[f+176>>2]|0)+(k*764|0)+756>>2]|0)+8>>2]=128;m=jg(b)|0;c[(c[f+180>>2]|0)+(k*52|0)+36>>2]=m;m=c[(c[f+180>>2]|0)+(k*52|0)+36>>2]|0;n=m+(jg(b)|0)|0;c[(c[f+180>>2]|0)+(k*52|0)+40>>2]=n;if((c[(c[f+180>>2]|0)+(k*52|0)+40>>2]|0)!=0){n=(c[f+180>>2]|0)+(k*52|0)+40|0;c[n>>2]=(c[n>>2]|0)-1}c[(c[f+180>>2]|0)+(k*52|0)+44>>2]=(c[(c[f+180>>2]|0)+(k*52|0)+36>>2]|0)>0?2:0;bB((c[f+176>>2]|0)+(k*764|0)|0,1,11,b|0)|0;o=0;while(1){if((o|0)>=11){break}if((a[(c[f+176>>2]|0)+(k*764|0)+o|0]|0)==13){a[(c[f+176>>2]|0)+(k*764|0)+o|0]=0}o=o+1|0}ja(b)|0;c[(c[f+176>>2]|0)+(k*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(k*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[f+176>>2]|0)+(k*764|0)+756>>2]|0)+40>>2]=k;k=k+1|0}bK(b|0,e+l|0,0)|0;k=0;while(1){l=ja(b)|0;if((l&255|0)==255){break}a[f+952+k|0]=l;k=k+1|0}k=k+1|0;while(1){if(((k|0)%4|0|0)==0){break}ja(b)|0;k=k+1|0}c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;k=0;while(1){if((k|0)>=(c[f+128>>2]|0)){break}l=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(k<<2)>>2]=l;c[c[(c[f+168>>2]|0)+(k<<2)>>2]>>2]=64;l=0;while(1){if((l|0)>=(c[f+136>>2]|0)){break}n=(ac(k,c[f+136>>2]|0)|0)+l|0;c[(c[(c[f+168>>2]|0)+(k<<2)>>2]|0)+4+(l<<2)>>2]=n;m=pT(((c[c[(c[f+168>>2]|0)+(k<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(n<<2)>>2]=m;c[c[(c[f+172>>2]|0)+(n<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(k<<2)>>2]>>2];l=l+1|0}o=0;while(1){if((o|0)>=(c[f+136>>2]<<6|0)){break}l=(c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(k<<2)>>2]|0)+4+(((o|0)%(c[f+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((o|0)/(c[f+136>>2]|0)|0)<<3)|0;a[l+4|0]=ja(b)|0;a[l+3|0]=ja(b)|0;a[l+1|0]=ja(b)|0;a[l|0]=ja(b)|0;if((a[l|0]|0)!=0){n=l|0;a[n]=(d[n]|0)+12&255}dc(l);o=o+1|0}k=k+1|0}k=0;while(1){if((k|0)>=(c[f+140>>2]|0)){break}if((c[(c[f+176>>2]|0)+(k*764|0)+36>>2]|0)!=0){o=b;l=e+(c[h+(k<<2)>>2]|0)|0;bK(o|0,l|0,0)|0;l=j;o=b;n=(c[f+180>>2]|0)+((c[(c[(c[f+176>>2]|0)+(k*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(l,o,128,n,0)|0}k=k+1|0}k=0;while(1){if((k|0)>=(c[f+136>>2]|0)){break}c[f+184+(k*12|0)>>2]=(((k+3|0)/2|0|0)%2|0)*255|0;k=k+1|0}i=g;return 0}function df(a,b){a=a|0;b=b|0;var c=0,e=0,f=0,g=0;c=a;a=b;b=0;L1417:while(1){if((b|0)>=(a|0)){e=1186;break}if((d[c+b|0]|0|0)>127){e=1179;break}do{if((d[c+b|0]|0|0)>0){if((d[c+b|0]|0|0)>=32){break}if((d[c+b|0]|0|0)!=8){e=1183;break L1417}}}while(0);b=b+1|0}if((e|0)==1179){f=-1;g=f;return g|0}else if((e|0)==1183){f=-1;g=f;return g|0}else if((e|0)==1186){f=0;g=f;return g|0}return 0}function dg(b){b=b|0;var c=0;c=b;if((a[c+4|0]|0)!=0){return}b=d[c+3|0]|0;if((b|0)==5){a[c+3|0]=3}else if((b|0)==6){a[c+3|0]=4}else if((b|0)==1|(b|0)==2|(b|0)==10){a[c+3|0]=0}return}function dh(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0;f=b;b=e;p7(f|0,0,b+1|0);p5(f|0,c|0,b|0)|0;c=0;while(1){if((a[f+c|0]|0)!=0){g=(c|0)<(b|0)}else{g=0}if(!g){break}if((bd(a[f+c|0]|0)|0)!=0){if((d[f+c|0]|0)>127){h=1206}}else{h=1206}if((h|0)==1206){h=0;a[f+c|0]=46}c=c+1|0}while(1){if((a[f]|0)!=0){i=(a[f+((p4(f|0)|0)-1)|0]|0)==32}else{i=0}if(!i){break}a[f+((p4(f|0)|0)-1)|0]=0}return f|0}function di(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0;e=i;i=i+64|0;f=e|0;g=c;c=d;if((g|0)==0){i=e;return}if((c|0)>=64){c=63}p7(g|0,0,c+1|0);bB(f|0,1,c|0,b|0)|0;a[f+c|0]=0;dh(g,f|0,c)|0;i=e;return}function dj(b,c){b=b|0;c=c|0;var e=0;e=b;b=c;a[e|0]=(ht(((a[b|0]&15)<<8)+(d[b+1|0]|0)|0)|0)&255;a[e+1|0]=((a[b|0]&240)>>4<<4|(a[b+2|0]&240)>>4)&255;a[e+3|0]=(d[b+2|0]|0)&15;a[e+4|0]=a[b+3|0]|0;dg(e);return}function dk(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=b;b=c;c=d;d=0;f=aI(a|0)|0;if((f|0)==0){g=0;h=g;return h|0}while(1){a=aP(f|0)|0;i=a;if((a|0)==0){break}if((qa(i+4|0,e|0)|0)==0){j=1229;break}}if((j|0)==1229){d=1}if((d|0)!=0){j=b;b=i+4|0;i=c;p5(j|0,b|0,i|0)|0}bS(f|0)|0;g=d;h=g;return h|0}function dl(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=a;a=b;b=d;if((c[e+6544>>2]|0)!=0){d=a;f=c[e+6544>>2]|0;e=b;p5(d|0,f|0,e|0)|0;return}if((bt(22888)|0)!=0){e=a;f=bt(22888)|0;d=b;p5(e|0,f|0,d|0)|0}else{d=a;a=b;p5(d|0,28752,a|0)|0}return}function dm(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0;e=i;i=i+16|0;f=e|0;g=f|0;c[g>>2]=d;c[g+4>>2]=0;bz(a+64|0,64,b|0,f|0)|0;i=e;return}function dn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1145195824){d=-1;e=d;return e|0}else{a=c;bK(a|0,12,1)|0;di(c,b,44);d=0;e=d;return e|0}return 0}function dp(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+56|0;f=e|0;g=e+48|0;h=a;a=b;b=h|0;bK(a|0,d|0,0)|0;jl(a)|0;c[g>>2]=0;d=jd(a)|0;bK(a|0,10,1)|0;bB(f|0,1,44,a|0)|0;j=es()|0;if((j|0)==0){k=-1;l=k;i=e;return l|0}ev(j,28632,6);ev(j,28528,72);ev(j,27136,60);ev(j,26016,18);ev(j,25024,78);ev(j,24200,122);p5(b|0,f|0,64)|0;aM(b+64|0,64,23792,(f=i,i=i+16|0,c[f>>2]=(d&65535)>>8,c[f+8>>2]=d&255,f)|0)|0;i=f;while(1){if(!((bH(a|0)|0)!=0^1)){break}et(j,h,a,g)}eB(j);j=0;while(1){if((j|0)>=(c[b+136>>2]|0)){break}c[b+184+(j*12|0)>>2]=128;j=j+1|0}k=0;l=k;i=e;return l|0}function dq(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=d;d=a|0;c[d+140>>2]=(jd(e)|0)&65535;c[d+144>>2]=(jd(e)|0)&65535;jd(e)|0;c[d+128>>2]=(jd(e)|0)&65535;c[d+136>>2]=(jd(e)|0)&65535;c[d+132>>2]=ac(c[d+128>>2]|0,c[d+136>>2]|0)|0;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}return}function dr(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0;d=i;i=i+56|0;g=e;e=b|0;b=f;if((c[b>>2]|0)!=0){i=d;return}c[b>>2]=1;bB(d|0,44,1,g|0)|0;c[e+156>>2]=(jd(g)|0)&65535;b=0;while(1){if((b|0)>=(c[e+156>>2]|0)){break}a[e+952+b|0]=(jd(g)|0)&255;b=b+1|0}i=d;return}function ds(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+56|0;b=e|0;f=d;d=a|0;a=0;while(1){if((a|0)>=(c[d+140>>2]|0)){break}g=pT(64,1)|0;c[(c[d+176>>2]|0)+(a*764|0)+756>>2]=g;c[(c[d+176>>2]|0)+(a*764|0)+36>>2]=1;bB(b|0,30,1,f|0)|0;dh((c[d+176>>2]|0)+(a*764|0)|0,b|0,30)|0;g=(jd(f)|0)&65535;do{if((g|0)==0){h=1298}else{if((g|0)>(c[d+144>>2]|0)){h=1298;break}j=g-1|0;g=j;c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+40>>2]=j;j=(jd(f)|0)&65535;c[c[(c[d+176>>2]|0)+(a*764|0)+756>>2]>>2]=j;j=jl(f)|0;k=jl(f)|0;c[(c[d+180>>2]|0)+(g*52|0)+36>>2]=k;k=c[(c[d+180>>2]|0)+(a*52|0)+36>>2]|0;l=k+(jl(f)|0)|0;c[(c[d+180>>2]|0)+(g*52|0)+40>>2]=l;l=((jd(f)|0)<<16>>16)+128|0;c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+8>>2]=l;if((c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+8>>2]|0)>255){c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+8>>2]=255}l=(jd(f)|0)&65535;c[(c[d+180>>2]|0)+(g*52|0)+44>>2]=(l&3|0)!=0?2:0;k=(c[d+180>>2]|0)+(g*52|0)+44|0;c[k>>2]=c[k>>2]|((l&2|0)!=0?4:0);hD(j,(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+16|0)}}while(0);if((h|0)==1298){h=0}a=a+1|0}i=e;return}function dt(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;g=i;i=i+8|0;e=g|0;h=b;b=f;f=h|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;j=0;while(1){if((j|0)>=(c[f+128>>2]|0)){break}k=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(j<<2)>>2]=k;k=(jd(b)|0)&65535;c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]=k;k=0;while(1){if((k|0)>=(c[f+136>>2]|0)){break}l=(ac(j,c[f+136>>2]|0)|0)+k|0;c[(c[(c[f+168>>2]|0)+(j<<2)>>2]|0)+4+(k<<2)>>2]=l;m=pT(((c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(l<<2)>>2]=m;c[c[(c[f+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2];k=k+1|0}k=jl(b)|0;l=0;m=-1;while(1){if((k|0)<=0){break}m=(ja(b)|0)&255;n=k-1|0;k=n;if((n|0)<=0){o=1323;break}if((m|0)==0){l=l+1|0;m=-1;continue}m=m-1|0;n=(ja(b)|0)&255;p=k-1|0;k=p;if((p|0)<=0){o=1327;break}do{if((m|0)>=(c[f+136>>2]|0)){o=1330}else{if((l|0)>=(c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]|0)){o=1330;break}q=(c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(j<<2)>>2]|0)+4+(m<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}}while(0);if((o|0)==1330){o=0;q=e}if((n&1|0)!=0){p=ja(b)|0;a[q|0]=(((p&240)>>4)*12|0)+13+(p&15)&255;p=k-1|0;k=p;if((p|0)<=0){o=1334;break}}if((n&2|0)!=0){a[q+1|0]=ja(b)|0;p=k-1|0;k=p;if((p|0)<=0){o=1338;break}}if((n&4|0)!=0){a[q+3|0]=ja(b)|0;p=k-1|0;k=p;if((p|0)<=0){o=1342;break}}if((n&8|0)!=0){a[q+4|0]=ja(b)|0;p=k-1|0;k=p;if((p|0)<=0){o=1346;break}}if((n&16|0)!=0){a[q+5|0]=ja(b)|0;p=k-1|0;k=p;if((p|0)<=0){o=1350;break}}if((n&32|0)!=0){a[q+6|0]=ja(b)|0;p=k-1|0;k=p;if((p|0)<=0){o=1354;break}}if((d[q+3|0]|0|0)==28){a[q+3|0]=-85}if((d[q+3|0]|0|0)>28){a[q+6|0]=0;a[q+3|0]=0}if((d[q+5|0]|0|0)==28){a[q+5|0]=-85}if((d[q+5|0]|0|0)>28){a[q+6|0]=0;a[q+5|0]=0}}if((o|0)==1323){o=0}else if((o|0)==1327){o=0}else if((o|0)==1334){o=0}else if((o|0)==1338){o=0}else if((o|0)==1342){o=0}else if((o|0)==1346){o=0}else if((o|0)==1350){o=0}else if((o|0)==1354){o=0}j=j+1|0}i=g;return}function du(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;e=a;a=d;d=e|0;b=0;while(1){if((b|0)>=(c[d+144>>2]|0)){break}f=jl(a)|0;g=jl(a)|0;c[(c[d+180>>2]|0)+(b*52|0)+32>>2]=g;if((f&2|0)!=0){g=(c[d+180>>2]|0)+(b*52|0)+44|0;c[g>>2]=c[g>>2]|1}do{if((f&4|0)!=0){g=(c[d+180>>2]|0)+(b*52|0)+32|0;c[g>>2]=c[g>>2]<<2;g=a;h=c[(c[d+180>>2]|0)+(b*52|0)+32>>2]|0;bK(g|0,h|0,1)|0}else{h=e;g=a;i=(c[d+180>>2]|0)+(b*52|0)|0;gQ(h,g,64,i,0)|0;if((c[(c[d+180>>2]|0)+(b*52|0)+32>>2]|0)==0){break}}}while(0);b=b+1|0}return}function dv(a,d,e,f){a=a|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;f=e;e=a|0;a=(jd(f)|0)&65535;d=0;while(1){if((d|0)>=(a|0)){break}g=((jd(f)|0)&65535)-1|0;h=(ja(f)|0)&7;c[(c[e+176>>2]|0)+(g*764|0)+44>>2]=h;h=(ja(f)|0)&255;c[(c[e+176>>2]|0)+(g*764|0)+48>>2]=h;h=(ja(f)|0)&255;c[(c[e+176>>2]|0)+(g*764|0)+56>>2]=h;h=(ja(f)|0)&255;c[(c[e+176>>2]|0)+(g*764|0)+64>>2]=h;h=(ja(f)|0)&255;c[(c[e+176>>2]|0)+(g*764|0)+68>>2]=h;ja(f)|0;h=0;while(1){if((h|0)>=32){break}i=jd(f)|0;b[(c[e+176>>2]|0)+(g*764|0)+72+(h<<1<<1)>>1]=i;i=jd(f)|0;b[(c[e+176>>2]|0)+(g*764|0)+72+((h<<1)+1<<1)>>1]=i;h=h+1|0}d=d+1|0}return}function dw(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;c=i;i=i+24|0;d=c|0;e=a;if((bB(d|0,1,20,e|0)|0)>>>0<20){f=-1;g=f;i=c;return g|0}if((p6(d|0,28504,19)|0)!=0){f=-1;g=f;i=c;return g|0}else{d=e;bK(d|0,156,1)|0;d=e;bK(d|0,384,1)|0;d=e;bK(d|0,64,1)|0;di(e,b,32);f=0;g=f;i=c;return g|0}return 0}function dx(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;g=i;i=i+1648|0;h=g|0;j=g+1576|0;k=g+1584|0;l=b;b=e;e=l|0;m=0;bK(b|0,f|0,0)|0;bB(h|0,20,1,b|0)|0;bB(h+20|0,4,1,b|0)|0;a[h+24|0]=ja(b)|0;a[h+25|0]=ja(b)|0;a[h+26|0]=ja(b)|0;bB(h+27|0,19,1,b|0)|0;a[h+46|0]=ja(b)|0;a[h+47|0]=ja(b)|0;bB(h+48|0,128,1,b|0)|0;f=0;while(1){if((f|0)>=31){break}c[h+176+(f<<2)>>2]=jl(b)|0;f=f+1|0}f=0;while(1){if((f|0)>=31){break}c[h+300+(f<<2)>>2]=jl(b)|0;f=f+1|0}f=0;while(1){if((f|0)>=31){break}c[h+424+(f<<2)>>2]=jl(b)|0;f=f+1|0}f=0;while(1){if((f|0)>=31){break}a[h+548+f|0]=ja(b)|0;f=f+1|0}f=0;while(1){if((f|0)>=31){break}a[h+579+f|0]=ja(b)|0;f=f+1|0}bB(h+610|0,32,1,b|0)|0;f=0;while(1){if((f|0)>=31){break}bB(h+642+(f*30|0)|0,30,1,b|0)|0;f=f+1|0}c[e+140>>2]=31;c[e+144>>2]=c[e+140>>2];c[e+128>>2]=(d[h+46|0]|0)+1;c[e+136>>2]=d[h+25|0]|0;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+156>>2]=(d[h+47|0]|0)+1;n=l+1280|0;c[n>>2]=c[n>>2]|8192;dh(e|0,h+610|0,32)|0;dm(l,31008,(n=i,i=i+8|0,c[n>>2]=h+20,n)|0);i=n;f=0;while(1){if((f|0)>=(c[e+156>>2]|0)){break}a[e+952+f|0]=a[h+48+f|0]|0;f=f+1|0}c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}n=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=n;n=c[h+176+(f<<2)>>2]|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=n;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=((n|0)!=0^1^1)&1;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=c[h+300+(f<<2)>>2];c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=(c[h+300+(f<<2)>>2]|0)+(c[h+424+(f<<2)>>2]|0);c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0)>0?2:0;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=d[h+548+f|0]|0;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+16>>2]=d[h+579+f|0]|0;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=f;dh((c[e+176>>2]|0)+(f*764|0)|0,h+642+(f*30|0)|0,30)|0;f=f+1|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;f=0;while(1){if((f|0)>=(c[e+128>>2]|0)){break}n=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(f<<2)>>2]=n;c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]=64;n=0;while(1){if((n|0)>=(c[e+136>>2]|0)){break}o=(ac(f,c[e+136>>2]|0)|0)+n|0;c[(c[(c[e+168>>2]|0)+(f<<2)>>2]|0)+4+(n<<2)>>2]=o;p=pT(((c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(o<<2)>>2]=p;c[c[(c[e+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2];n=n+1|0}if((a[h+26|0]|0)!=0){q=((jd(b)|0)&65535)-64>>2&65535;n=k|0;o=b;bB(n|0,1,64,o|0)|0}else{q=c[e+136>>2]<<6&65535;p7(k|0,-1|0,64)}o=0;while(1){if((o|0)>=64){break}n=0;p=128;while(1){if((n|0)>=(c[e+136>>2]|0)){break}if((d[k+o|0]&p|0)!=0){r=j|0;s=b;bB(r|0,4,1,s|0)|0;m=(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(f<<2)>>2]|0)+4+(n<<2)>>2]<<2)>>2]|0)+4+(o<<3)|0;dj(m,j|0);s=d[m+3|0]|0;if((s|0)==8){a[m+4|0]=0;a[m+3|0]=0}else if((s|0)==14){s=(a[m+4|0]&240)>>4;if((s|0)==0|(s|0)==3|(s|0)==8|(s|0)==9){a[m+4|0]=0;a[m+3|0]=0}else if((s|0)==4){a[m+3|0]=12;a[m+4|0]=0}}q=q-1&65535}n=n+1|0;p=p>>1}o=o+1|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}gQ(l,b,0,(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;f=f+1|0}i=g;return 0}function dy(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1145326918){d=-1;e=d;return e|0}else{a=c;bK(a|0,9,1)|0;di(c,b,30);d=0;e=d;return e|0}return 0}function dz(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+288|0;g=f+8|0;h=f+24|0;j=b;b=d;d=j|0;bK(b|0,e|0,0)|0;jl(b)|0;c[h>>2]=(ja(b)|0)&255;bB(g|0,8,1,b|0)|0;a[g+8|0]=0;e=c[h>>2]|0;aM(d+64|0,64,30992,(k=i,i=i+16|0,c[k>>2]=g,c[k+8>>2]=e,k)|0)|0;i=k;a[g+8|0]=0;bB(d|0,30,1,b|0)|0;bK(b|0,20,1)|0;bB(f|0,3,1,b|0)|0;d=es()|0;if((d|0)==0){l=-1;m=l;i=f;return m|0}ev(d,28496,20);ev(d,27112,90);ev(d,26008,80);ev(d,25016,82);ey(d,1);while(1){if(!((bH(b|0)|0)!=0^1)){break}et(d,j,b,h)}c[j+1268>>2]=255;eB(d);l=0;m=l;i=f;return m|0}function dA(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;f=e;e=b|0;jc(f)|0;jc(f)|0;c[e+156>>2]=(d-4|0)/2|0;if((c[e+156>>2]|0)>255){c[e+156>>2]=255}d=0;while(1){if((d|0)>=(c[e+156>>2]|0)){break}a[e+952+d|0]=(jc(f)|0)&255;d=d+1|0}return}function dB(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;f=i;i=i+128|0;d=f|0;g=b;b=e;e=g|0;c[e+128>>2]=(jc(b)|0)&65535;c[e+136>>2]=(ja(b)|0)&255;c[e+132>>2]=ac(c[e+136>>2]|0,c[e+128>>2]|0)|0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[e+128>>2]|0)){break}j=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(h<<2)>>2]=j;j=(ja(b)|0)&255;ja(b)|0;k=(jc(b)|0)&65535;c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2]=k;k=0;while(1){if((k|0)>=(c[e+136>>2]|0)){break}l=(ac(h,c[e+136>>2]|0)|0)+k|0;c[(c[(c[e+168>>2]|0)+(h<<2)>>2]|0)+4+(k<<2)>>2]=l;m=pT(((c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(l<<2)>>2]=m;c[c[(c[e+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2];k=k+1|0}jg(b)|0;k=0;while(1){if((k|0)>=(j|0)){break}c[d+(k<<2)>>2]=0;k=k+1|0}l=0;m=0;while(1){if((l|0)>=(c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2]|0)){break}if((m|0)==0){n=(ja(b)|0)&255;if((n&128|0)!=0){o=(ja(b)|0)&255}else{o=0}m=o;if((n&63|0)!=0){p=(ja(b)|0)&255}else{p=0}}else{m=m-1|0}k=0;while(1){if((k|0)>=(j|0)){break}n=(c[(c[g+172>>2]|0)+(c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(k<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0;if((c[d+(k<<2)>>2]|0)==0){q=(ja(b)|0)&255;if((q&128|0)!=0){c[d+(k<<2)>>2]=(ja(b)|0)&255}if((q&64|0)!=0){a[n+1|0]=ja(b)|0}if((q&32|0)!=0){a[n|0]=((ja(b)|0)&255)+24&255}if((q&16|0)!=0){a[n+2|0]=ja(b)|0}if((q&8|0)!=0){r=(ja(b)|0)&255;s=(ja(b)|0)&255}if((q&4|0)!=0){r=(ja(b)|0)&255;s=(ja(b)|0)&255}if((q&2|0)!=0){r=(ja(b)|0)&255;s=(ja(b)|0)&255;if((r|0)==2){a[n+3|0]=-95;a[n+4|0]=s&255}}}else{n=d+(k<<2)|0;c[n>>2]=(c[n>>2]|0)-1}k=k+1|0}l=l+1|0}h=h+1|0}i=f;return}function dC(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0;e=a;a=b;b=0;f=1;while(1){g=a;a=g-1|0;if((g|0)==0){break}if((c[e+12>>2]|0)!=0){g=e+12|0;c[g>>2]=(c[g>>2]|0)-1}else{if((c[e>>2]|0)>>>0<(c[e+4>>2]|0)>>>0){g=e|0;h=c[g>>2]|0;c[g>>2]=h+1;i=d[h]|0}else{i=0}c[e+8>>2]=i;c[e+12>>2]=7}if((c[e+8>>2]&1|0)!=0){b=(b&255|f&255)&255}f=(f&255)<<1&255;h=e+8|0;c[h>>2]=(c[h>>2]|0)>>>1}return b|0}function dD(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0;d=i;i=i+32|0;g=d|0;h=e;e=b|0;b=f;f=(ja(h)|0)&255;c[e+144>>2]=f;c[e+140>>2]=f;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}j=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=j;j=(ja(h)|0)&255;if((j|0)>30){k=30}else{k=j}l=j-(bB(g|0,1,k|0,h|0)|0)|0;dh((c[e+176>>2]|0)+(f*764|0)|0,g|0,j)|0;a[g+j|0]=0;while(1){j=l;l=j-1|0;if((j|0)==0){break}ja(h)|0}l=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=l;l=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=l;l=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=l;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0)!=0^1^1)&1;l=(jc(h)|0)&65535;hD(l,(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+12|0,(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+16|0);l=(ja(h)|0)&255;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=l;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=f;l=(ja(h)|0)&255;c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(l&1|0)!=0?2:0;if((c[b>>2]|0)>=8){j=h;bK(j|0,8,1)|0}jc(h)|0;jg(h)|0;a[b+4+f|0]=(l&12)>>2&255;f=f+1|0}i=d;return}function dE(d,e,f,g){d=d|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;h=i;i=i+1560|0;j=h|0;k=d;d=e;e=g;g=0;p7(j|0,0,1560);c[j>>2]=d;c[j+4>>2]=f;dF(j);f=0;l=0;while(1){if(l>>>0>=e>>>0){break}m=0;n=dC(j,1)|0;do{if((dC(j,1)|0)<<24>>24!=0){m=b[j+24+(m*6|0)+2>>1]|0}else{m=b[j+24+(m*6|0)>>1]|0}if((m|0)>255){o=1624;break}g=a[j+24+(m*6|0)+4|0]|0;if((c[j>>2]|0)>>>0>=(c[j+4>>2]|0)>>>0){if((c[j+12>>2]|0)==0){o=1627;break}}if((b[j+24+(m*6|0)>>1]|0)>=0){p=(b[j+24+(m*6|0)+2>>1]|0)>=0}else{p=0}}while(p);if((o|0)==1624){o=0}else if((o|0)==1627){o=0}if(n<<24>>24!=0){g=(g&255^255)&255}f=(f&255)+(g&255)&255;if((l|0)!=0){q=f&255}else{q=0}a[k+l|0]=q&255;l=l+1|0}i=h;return(c[j>>2]|0)-d|0}function dF(d){d=d|0;var e=0,f=0,g=0,h=0;e=d;d=c[e+20>>2]|0;if((d|0)>255){return}a[e+24+(d*6|0)+4|0]=dC(e,7)|0;f=dC(e,1)|0;g=dC(e,1)|0;d=c[e+16>>2]|0;if((d|0)>255){return}h=e+20|0;c[h>>2]=(c[h>>2]|0)+1;c[e+16>>2]=c[e+20>>2];if(f<<24>>24!=0){b[e+24+(d*6|0)>>1]=c[e+16>>2]&65535;dF(e)}else{b[e+24+(d*6|0)>>1]=-1}c[e+16>>2]=c[e+20>>2];if(g<<24>>24!=0){b[e+24+(d*6|0)+2>>1]=c[e+16>>2]&65535;dF(e);return}else{b[e+24+(d*6|0)+2>>1]=-1;return}}function dG(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1143886894){d=-1;e=d;return e|0}else{a=c;jl(a)|0;a=c;jd(a)|0;a=c;jd(a)|0;a=c;jd(a)|0;a=c;jd(a)|0;a=c;jd(a)|0;a=c;jl(a)|0;di(c,b,32);d=0;e=d;return e|0}return 0}function dH(a,b,e,f){a=a|0;b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;b=a;a=e;e=b|0;g=f;f=0;h=0;while(1){if((f|0)>=(c[e+144>>2]|0)){break}if((c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0)>(h|0)){h=c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0}f=f+1|0}i=pS(h)|0;if((i|0)==0){bV(24176,326,31456,23776)}j=pS(h)|0;if((j|0)==0){bV(24176,328,31456,23192)}f=0;while(1){if((f|0)>=(c[e+144>>2]|0)){break}h=jg(a)|0;if((h|0)!=0){k=d[g+4+f|0]|0;if((k|0)==0){l=b;m=a;n=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(l,m,0,n,0)|0}else if((k|0)==1){bB(j|0,h|0,1,a|0)|0;dE(i,j,j+h|0,c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0)|0;gQ(b,0,16,(c[e+180>>2]|0)+(f*52|0)|0,i)|0}else{bK(a|0,h|0,1)|0}}f=f+1|0}pU(j);pU(i);return}function dI(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+16|0;f=e|0;g=a;a=b;bK(a|0,d|0,0)|0;c[f+4>>2]=0;c[f>>2]=0;d=es()|0;if((d|0)==0){h=-1;j=h;i=e;return j|0}ev(d,30928,46);ev(d,28488,64);ev(d,27104,134);ev(d,26e3,108);ev(d,25008,152);ev(d,24168,136);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(d,g,a,f)}eB(d);h=0;j=h;i=e;return j|0}function dJ(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;e=i;b=a;a=d;d=b|0;jd(a)|0;jd(a)|0;jd(a)|0;c[d+148>>2]=(jd(a)|0)&65535;f=(jd(a)|0)&65535;if((f|0)>0){c[d+152>>2]=f}jl(a)|0;bB(d|0,32,1,a|0)|0;dm(b,23744,(b=i,i=i+1|0,i=i+7>>3<<3,c[b>>2]=0,b)|0);i=b;i=e;return}function dK(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;g=f;f=b|0;c[f+156>>2]=(jd(g)|0)&65535;c[f+160>>2]=(jd(g)|0)&65535;jl(g)|0;b=0;e=0;while(1){if((b|0)>=128){break}a[f+952+b|0]=ja(g)|0;if((d[f+952+b|0]|0|0)>(e|0)){e=d[f+952+b|0]|0}b=b+1|0}c[f+128>>2]=e+1;return}function dL(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;b=d;d=a|0;c[d+136>>2]=(jd(b)|0)&65535;c[e+8>>2]=(jd(b)|0)&65535;c[d+132>>2]=ac(c[d+136>>2]|0,c[d+128>>2]|0)|0;return}function dM(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;e=i;i=i+32|0;b=e|0;f=d;d=a|0;a=(jd(f)|0)&65535;c[d+144>>2]=a;c[d+140>>2]=a;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}a=0;while(1){if((a|0)>=(c[d+140>>2]|0)){break}g=pT(64,1)|0;c[(c[d+176>>2]|0)+(a*764|0)+756>>2]=g;jl(f)|0;g=jl(f)|0;c[(c[d+180>>2]|0)+(a*52|0)+32>>2]=g;c[(c[d+176>>2]|0)+(a*764|0)+36>>2]=((c[(c[d+180>>2]|0)+(a*52|0)+32>>2]|0)!=0^1^1)&1;g=(jb(f)|0)<<24>>24;h=(ja(f)|0)&255;c[c[(c[d+176>>2]|0)+(a*764|0)+756>>2]>>2]=h;c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+8>>2]=128;h=jl(f)|0;c[(c[d+180>>2]|0)+(a*52|0)+36>>2]=h;h=jl(f)|0;c[(c[d+180>>2]|0)+(a*52|0)+40>>2]=(c[(c[d+180>>2]|0)+(a*52|0)+36>>2]|0)+h-1;c[(c[d+180>>2]|0)+(a*52|0)+44>>2]=(h|0)>2?2:0;bB(b|0,22,1,f|0)|0;dh((c[d+176>>2]|0)+(a*764|0)|0,b|0,22)|0;if(((jd(f)|0)&255|0)>8){h=(c[d+180>>2]|0)+(a*52|0)+44|0;c[h>>2]=c[h>>2]|1;h=(c[d+180>>2]|0)+(a*52|0)+32|0;c[h>>2]=c[h>>2]>>1;h=(c[d+180>>2]|0)+(a*52|0)+36|0;c[h>>2]=c[h>>2]>>1;h=(c[d+180>>2]|0)+(a*52|0)+40|0;c[h>>2]=c[h>>2]>>1}jl(f)|0;h=jl(f)|0;hD(h,(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+16|0);h=(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+16|0;c[h>>2]=(c[h>>2]|0)+g;c[(c[(c[d+176>>2]|0)+(a*764|0)+756>>2]|0)+40>>2]=a;a=a+1|0}i=e;return}function dN(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;d=b;b=e;e=d|0;g=f;if((c[g>>2]|0)==0){c[g>>2]=1;c[8928]=0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0}jl(b)|0;g=(jd(b)|0)&65535;f=g;h=g;g=(jd(b)|0)&65535;h=c[8928]|0;while(1){if((h|0)>(f|0)){break}i=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(h<<2)>>2]=i;c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2]=g;i=0;while(1){if((i|0)>=(c[e+136>>2]|0)){break}j=(ac(h,c[e+136>>2]|0)|0)+i|0;c[(c[(c[e+168>>2]|0)+(h<<2)>>2]|0)+4+(i<<2)>>2]=j;k=pT(((c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(j<<2)>>2]=k;c[c[(c[e+172>>2]|0)+(j<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(h<<2)>>2]>>2];i=i+1|0}h=h+1|0}c[8928]=f+1;h=0;while(1){if((h|0)>=(g|0)){break}i=0;while(1){if((i|0)>=(c[e+136>>2]|0)){break}j=(c[(c[d+172>>2]|0)+(c[(c[(c[d+168>>2]|0)+(f<<2)>>2]|0)+4+(i<<2)>>2]<<2)>>2]|0)+4+(h<<3)|0;k=ja(b)|0;l=ja(b)|0;m=ja(b)|0;n=ja(b)|0;if(k<<24>>24!=0){k=k-1&255;a[j|0]=(((k&255)>>4)*12|0)+(k&15)+12&255}a[j+2|0]=(l&252)>>2&255;a[j+1|0]=((l&3)<<4)+((m&255)>>4)&255;a[j+3|0]=m&255&15;a[j+4|0]=n;i=i+1|0}h=h+1|0}return}function dO(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=a;a=f|0;g=e;if((c[g+4>>2]|0)==0){c[g+4>>2]=1;c[8930]=0}if((b|0)<=2){h=c[8930]|0;i=h+1|0;c[8930]=i;return}gQ(f,d,64,(c[a+180>>2]|0)+((c[(c[(c[a+176>>2]|0)+((c[8930]|0)*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;h=c[8930]|0;i=h+1|0;c[8930]=i;return}function dP(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1148414804){d=-1;e=d;return e|0}else{di(c,b,64);d=0;e=d;return e|0}return 0}function dQ(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;g=i;i=i+1640|0;h=g|0;j=g+104|0;k=g+1128|0;l=g+1384|0;m=b;b=e;e=f;f=m|0;bK(b|0,e|0,0)|0;jl(b)|0;dm(m,30872,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n;bB(h|0,1,64,b|0)|0;p5(f|0,h|0,64)|0;bB(h|0,1,64,b|0)|0;jg(b)|0;c[f+136>>2]=jg(b)|0;c[f+156>>2]=jg(b)|0;bB(h|0,1,8,b|0)|0;c[f+148>>2]=jg(b)|0;c[f+160>>2]=jg(b)|0;c[f+128>>2]=jg(b)|0;n=jg(b)|0;c[f+144>>2]=n;c[f+140>>2]=n;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;bB(f+952|0,1,(c[f+156>>2]|0)+3&-4|0,b|0)|0;n=0;while(1){if((n|0)>=(c[f+128>>2]|0)){break}o=jg(b)|0;if((n|0)<256){c[j+(n<<2)>>2]=o}n=n+1|0}o=(c[f+128>>2]|0)+3&-4;n=0;while(1){if((n|0)>=(o|0)){break}p=(ja(b)|0)&255;if((n|0)<256){a[k+n|0]=p&255}n=n+1|0}c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}n=0;while(1){if((n|0)>=(c[f+140>>2]|0)){break}o=pT(64,1)|0;c[(c[f+176>>2]|0)+(n*764|0)+756>>2]=o;ja(b)|0;o=((ja(b)|0)&255)>>1;c[c[(c[f+176>>2]|0)+(n*764|0)+756>>2]>>2]=o;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+8>>2]=128;jc(b)|0;jg(b)|0;jg(b)|0;jg(b)|0;o=jg(b)|0;c[(c[f+180>>2]|0)+(n*52|0)+36>>2]=o;o=jg(b)|0;c[(c[f+180>>2]|0)+(n*52|0)+44>>2]=(o|0)>0?2:0;c[(c[f+180>>2]|0)+(n*52|0)+40>>2]=(c[(c[f+180>>2]|0)+(n*52|0)+36>>2]|0)+o;o=jg(b)|0;c[(c[f+180>>2]|0)+(n*52|0)+32>>2]=o;bB(h|0,1,32,b|0)|0;dh((c[f+176>>2]|0)+(n*764|0)|0,h|0,32)|0;c[l+(n<<2)>>2]=jg(b)|0;c[(c[f+176>>2]|0)+(n*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(n*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]=n;n=n+1|0}c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;n=0;while(1){if((n|0)>=(c[f+128>>2]|0)){break}h=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(n<<2)>>2]=h;c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]=d[k+n|0]|0;h=0;while(1){if((h|0)>=(c[f+136>>2]|0)){break}o=(ac(n,c[f+136>>2]|0)|0)+h|0;c[(c[(c[f+168>>2]|0)+(n<<2)>>2]|0)+4+(h<<2)>>2]=o;p=pT(((c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(o<<2)>>2]=p;c[c[(c[f+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2];h=h+1|0}bK(b|0,e+(c[j+(n<<2)>>2]|0)|0,0)|0;h=0;while(1){if((h|0)>=(c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]|0)){break}o=0;while(1){if((o|0)>=(c[f+136>>2]|0)){break}p=(c[(c[m+172>>2]|0)+(c[(c[(c[m+168>>2]|0)+(n<<2)>>2]|0)+4+(o<<2)>>2]<<2)>>2]|0)+4+(h<<3)|0;q=jg(b)|0;a[p+1|0]=q&63;a[p|0]=(q&4032)>>>6&255;a[p+3|0]=(q&126976)>>>12&255;if((a[p|0]|0)!=0){r=p|0;a[r]=(d[r]|0)+48&255}if((q&4063232|0)!=0){a[p+6|0]=(q&4063232)>>>17&255;q=jg(b)|0;a[p+4|0]=q&255;a[p+6|0]=(q&65280)>>>8&255}else{a[p+4|0]=(q&-67108864)>>>18&255}o=o+1|0}h=h+1|0}n=n+1|0}n=0;while(1){if((n|0)>=(c[f+140>>2]|0)){break}bK(b|0,e+(c[l+(n<<2)>>2]|0)|0,0)|0;gQ(m,b,128,(c[f+180>>2]|0)+((c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;n=n+1|0}i=g;return 0}function dR(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;a=b;if((jl(c)|0)!=1179603533){d=-1;e=d;return e|0}jl(c)|0;if((jl(c)|0)!=1162694468){d=-1;e=d;return e|0}if((jl(c)|0)==1162692931){b=c;jl(b)|0;b=c;jd(b)|0;di(c,a,20)}else{di(c,a,0)}d=0;e=d;return e|0}function dS(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=a;a=b;bK(a|0,c|0,0)|0;jl(a)|0;jl(a)|0;jl(a)|0;c=es()|0;if((c|0)==0){e=-1;f=e;return f|0}ev(c,30864,154);ev(c,28432,2);ev(c,27096,100);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(c,d,a,0)}eB(c);e=0;f=e;return f|0}function dT(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0;f=i;i=i+256|0;d=f|0;g=b;b=e;e=g|0;h=(jd(b)|0)&65535;bB(e|0,1,20,b|0)|0;bK(b|0,20,1)|0;c[e+152>>2]=(ja(b)|0)&255;c[e+140>>2]=(ja(b)|0)&255;c[e+144>>2]=c[e+140>>2];j=g+1280|0;c[j>>2]=c[j>>2]|8192;aM(e+64|0,64,25968,(j=i,i=i+8|0,c[j>>2]=h,j)|0)|0;i=j;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}j=0;while(1){if((j|0)>=(c[e+140>>2]|0)){break}h=pT(64,1)|0;c[(c[e+176>>2]|0)+(j*764|0)+756>>2]=h;ja(b)|0;h=(ja(b)|0)&255;c[c[(c[e+176>>2]|0)+(j*764|0)+756>>2]>>2]=h;h=((jd(b)|0)&65535)<<1;c[(c[e+180>>2]|0)+(j*52|0)+32>>2]=h;bB((c[e+176>>2]|0)+(j*764|0)|0,1,20,b|0)|0;h=((ja(b)|0)&1|0)!=0?2:0;c[(c[e+180>>2]|0)+(j*52|0)+44>>2]=h;h=(ja(b)|0)&255;c[(c[(c[e+176>>2]|0)+(j*764|0)+756>>2]|0)+16>>2]=h;h=((jd(b)|0)&65535)<<1;c[(c[e+180>>2]|0)+(j*52|0)+36>>2]=h;h=c[(c[e+180>>2]|0)+(j*52|0)+36>>2]|0;g=h+(((jd(b)|0)&65535)<<1)|0;c[(c[e+180>>2]|0)+(j*52|0)+40>>2]=g;jl(b)|0;c[(c[e+176>>2]|0)+(j*764|0)+36>>2]=1;c[(c[(c[e+176>>2]|0)+(j*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(j*764|0)+756>>2]|0)+40>>2]=j;j=j+1|0}ja(b)|0;c[e+128>>2]=(ja(b)|0)&255;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;p7(d|0,0,256);j=0;while(1){if((j|0)>=(c[e+128>>2]|0)){break}a[d+((ja(b)|0)&255)|0]=j&255;g=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(j<<2)>>2]=g;g=((ja(b)|0)&255)+1|0;c[c[(c[e+168>>2]|0)+(j<<2)>>2]>>2]=g;g=0;while(1){if((g|0)>=(c[e+136>>2]|0)){break}h=(ac(j,c[e+136>>2]|0)|0)+g|0;c[(c[(c[e+168>>2]|0)+(j<<2)>>2]|0)+4+(g<<2)>>2]=h;k=pT(((c[c[(c[e+168>>2]|0)+(j<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(h<<2)>>2]=k;c[c[(c[e+172>>2]|0)+(h<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(j<<2)>>2]>>2];g=g+1|0}bK(b|0,20,1)|0;jl(b)|0;j=j+1|0}c[e+156>>2]=(ja(b)|0)&255;j=0;while(1){if((j|0)>=(c[e+156>>2]|0)){break}a[e+952+j|0]=a[d+((ja(b)|0)&255)|0]|0;j=j+1|0}i=f;return}function dU(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0;g=b;b=f;f=g|0;e=0;while(1){if((e|0)>=(c[f+128>>2]|0)){break}h=0;while(1){if((h|0)>=(c[c[(c[f+168>>2]|0)+(e<<2)>>2]>>2]|0)){break}i=0;while(1){if((i|0)>=(c[f+136>>2]|0)){break}j=(c[(c[g+172>>2]|0)+(c[(c[(c[g+168>>2]|0)+(e<<2)>>2]|0)+4+(i<<2)>>2]<<2)>>2]|0)+4+(h<<3)|0;a[j+1|0]=ja(b)|0;a[j|0]=((ja(b)|0)&255)+1&255;if((d[j|0]|0|0)!=0){k=j|0;a[k]=(d[k]|0)+48&255}a[j+3|0]=(ja(b)|0)&255&15;a[j+4|0]=ja(b)|0;k=d[j+3|0]|0;if((k|0)==4){l=a[j+4|0]|0;a[j+4|0]=(l&240|(l&255)<<1&15)&255}else if((k|0)==9){m=j+3|0;a[m]=(d[m]|0)<<1&255}else if((k|0)==11){l=a[j+3|0]|0;a[j+3|0]=(((l&255|0)/10|0)<<4)+((l&255|0)%10|0)&255}i=i+1|0}h=h+1|0}e=e+1|0}return}function dV(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a;a=d;d=e|0;b=0;while(1){if((b|0)>=(c[d+144>>2]|0)){break}gQ(e,a,0,(c[d+180>>2]|0)+(b*52|0)|0,0)|0;b=b+1|0}return}function dW(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1178686206){d=-1;e=d;return e|0}else{di(c,b,40);d=0;e=d;return e|0}return 0}function dX(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;j=i;i=i+944|0;k=j|0;l=j+104|0;m=j+880|0;n=j+936|0;o=f;f=g;g=o|0;p=0;bK(f|0,h|0,0)|0;jl(f)|0;bB(k+4|0,40,1,f|0)|0;bB(k+44|0,3,1,f|0)|0;b[k+48>>1]=jc(f)|0;a[k+50|0]=ja(f)|0;bB(k+51|0,16,1,f|0)|0;bK(f|0,9,1)|0;a[k+76|0]=ja(f)|0;bB(k+77|0,16,1,f|0)|0;jg(f)|0;b[k+98>>1]=jc(f)|0;bK(f|0,e[k+98>>1]|0,1)|0;bB(l|0,256,1,f|0)|0;a[l+256|0]=ja(f)|0;a[l+257|0]=ja(f)|0;a[l+258|0]=ja(f)|0;h=0;while(1){if((h|0)>=256){break}b[l+260+(h<<1)>>1]=jc(f)|0;h=h+1|0}c[g+136>>2]=16;c[g+156>>2]=d[l+257|0]|0;c[g+148>>2]=6;c[g+152>>2]=480/(d[k+76|0]|0)|0;q=g+952|0;r=l|0;s=c[g+156>>2]|0;p3(q|0,r|0,s)|0;h=0;c[g+128>>2]=0;while(1){if((h|0)>=256){break}if((b[l+260+(h<<1)>>1]|0)!=0){c[g+128>>2]=h+1}h=h+1|0}c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;p5(g|0,k+4|0,40)|0;s=a[k+50|0]&15;dm(o,30832,(r=i,i=i+16|0,c[r>>2]=(a[k+50|0]&240)>>4,c[r+8>>2]=s,r)|0);i=r;c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[g+128>>2]|0)){break}r=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(h<<2)>>2]=r;if((b[l+260+(h<<1)>>1]|0)!=0){c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]=((e[l+260+(h<<1)>>1]|0)-2|0)/64|0;r=0;while(1){if((r|0)>=(c[g+136>>2]|0)){break}s=(ac(h,c[g+136>>2]|0)|0)+r|0;c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(r<<2)>>2]=s;k=pT(((c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(s<<2)>>2]=k;c[c[(c[g+172>>2]|0)+(s<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2];r=r+1|0}r=((ja(f)|0)&255)+1&255;ja(f)|0;s=0;while(1){if((s|0)>=(ac(c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]|0,c[g+136>>2]|0)|0)){break}k=(c[(c[o+172>>2]|0)+(c[(c[(c[o+168>>2]|0)+(h<<2)>>2]|0)+4+(((s|0)%(c[g+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((s|0)/(c[g+136>>2]|0)|0)<<3)|0;do{if(((s|0)%(c[g+136>>2]|0)|0|0)==0){if(((s|0)/(c[g+136>>2]|0)|0|0)!=(r&255|0)){break}a[k+5|0]=13}}while(0);q=ja(f)|0;t=ja(f)|0;u=ja(f)|0;v=ja(f)|0;if(q<<24>>24!=0){a[k|0]=(q&255)+48&255}if((d[k|0]|0)!=0){w=1978}else{if((t&255|0)!=0){w=1978}}if((w|0)==1978){w=0;a[k+1|0]=(t&255)+1&255}u=((u&15)<<4)+((u&240)>>4)&255;if(u<<24>>24!=0){a[k+2|0]=(u&255)-16&255}a[k+3|0]=a[8856+((v&240)>>4)|0]|0;a[k+4|0]=v&255&15;v=d[k+3|0]|0;if((v|0)==255){a[k+4|0]=0;a[k+3|0]=0}else if((v|0)==254){p=a[k+4|0]&15;a[k+4|0]=0;a[k+3|0]=0}else if((v|0)==253){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|160)&255}else if((v|0)==249){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|16)&255}else if((v|0)==248){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|32)&255}else if((v|0)==123){a[k+4|0]=(d[k+4|0]<<4)+p&255}else if((v|0)==250){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|208)&255}else if((v|0)==252){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|176)&255}else if((v|0)==15){a[k+4|0]=(480/(d[k+4|0]|0)|0)&255}else if((v|0)==251){a[k+3|0]=14;u=k+4|0;a[u]=(d[u]|144)&255}else if((v|0)==4){a[k+4|0]=(d[k+4|0]<<4)+p&255}s=s+1|0}}h=h+1|0}c[g+140>>2]=-1;bB(n|0,1,8,f|0)|0;h=0;while(1){if((h|0)>=64){break}if((d[n+((h|0)/8|0)|0]&1<<((h|0)%8|0)|0)!=0){c[g+140>>2]=h}h=h+1|0}p=g+140|0;c[p>>2]=(c[p>>2]|0)+1;c[g+144>>2]=c[g+140>>2];c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}if((d[n+((h|0)/8|0)|0]&1<<((h|0)%8|0)|0)!=0){p=pT(64,1)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=p;p=m|0;w=f;bB(p|0,32,1,w|0)|0;c[m+32>>2]=jg(f)|0;a[m+36|0]=ja(f)|0;a[m+37|0]=ja(f)|0;c[m+40>>2]=jg(f)|0;c[m+44>>2]=jg(f)|0;a[m+48|0]=ja(f)|0;a[m+49|0]=ja(f)|0;w=m+32|0;c[w>>2]=c[w>>2]&65535;w=m+40|0;c[w>>2]=c[w>>2]&65535;w=m+44|0;c[w>>2]=c[w>>2]&65535;c[(c[g+180>>2]|0)+(h*52|0)+32>>2]=c[m+32>>2];c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=(c[m+32>>2]|0)>>>0>0?1:0;c[(c[g+180>>2]|0)+(h*52|0)+36>>2]=c[m+40>>2];c[(c[g+180>>2]|0)+(h*52|0)+40>>2]=c[m+44>>2];c[(c[g+180>>2]|0)+(h*52|0)+44>>2]=0;if((d[m+48|0]|0)!=0){w=(c[g+180>>2]|0)+(h*52|0)+44|0;c[w>>2]=c[w>>2]|1;w=(c[g+180>>2]|0)+(h*52|0)+32|0;c[w>>2]=c[w>>2]>>1;w=(c[g+180>>2]|0)+(h*52|0)+36|0;c[w>>2]=c[w>>2]>>1;w=(c[g+180>>2]|0)+(h*52|0)+40|0;c[w>>2]=c[w>>2]>>1}w=(c[g+180>>2]|0)+(h*52|0)+44|0;c[w>>2]=c[w>>2]|((d[m+49|0]|0)!=0?2:0);c[c[(c[g+176>>2]|0)+(h*764|0)+756>>2]>>2]=255;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]=h;w=(c[g+176>>2]|0)+(h*764|0)|0;p=m|0;dh(w,p,32)|0;p=o;w=f;l=(c[g+180>>2]|0)+(h*52|0)|0;gQ(p,w,0,l,0)|0}h=h+1|0}c[o+1268>>2]=255;i=j;return 0}function dY(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+8|0;f=e|0;g=b;b=c;c=d;bK(g|0,c+1080|0,0)|0;if((bB(f|0,1,4,g|0)|0)>>>0<4){h=-1;j=h;i=e;return j|0}do{if((p6(f|0,29680,3)|0)!=0){if((p6(f|0,29264,3)|0)==0){break}h=-1;j=h;i=e;return j|0}}while(0);do{if((a[f+3|0]|0)!=52){if((a[f+3|0]|0)==56){break}if((a[f+3|0]|0)==77){break}h=-1;j=h;i=e;return j|0}}while(0);bK(g|0,c|0,0)|0;di(g,b,20);h=0;j=h;i=e;return j|0}function dZ(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;j=i;i=i+2136|0;k=j|0;l=j+1088|0;m=j+1096|0;n=j+2120|0;o=f;f=g;g=o|0;bK(f|0,h|0,0)|0;h=0;p=c[o+1212>>2]|0;aM(m|0,1024,30824,(q=i,i=i+16|0,c[q>>2]=c[o+1208>>2],c[q+8>>2]=p,q)|0)|0;i=q;p=bh(m|0,28424)|0;r=p;if((p|0)==0){p=m|0;s=c[o+1208>>2]|0;t=c[o+1212>>2]|0;aM(p|0,1024,27088,(q=i,i=i+16|0,c[q>>2]=s,c[q+8>>2]=t,q)|0)|0;i=q;t=bh(m|0,28424)|0;r=t;if((t|0)==0){t=m|0;s=c[o+1208>>2]|0;p=c[o+1212>>2]|0;aM(t|0,1024,25960,(q=i,i=i+16|0,c[q>>2]=s,c[q+8>>2]=p,q)|0)|0;i=q;p=bh(m|0,28424)|0;r=p;if((p|0)==0){p=m|0;s=c[o+1208>>2]|0;t=c[o+1212>>2]|0;aM(p|0,1024,25e3,(q=i,i=i+16|0,c[q>>2]=s,c[q+8>>2]=t,q)|0)|0;i=q;r=bh(m|0,28424)|0}}}m=24152;if((r|0)!=0){t=n|0;s=r;bB(t|0,1,16,s|0)|0;if((p6(n|0,23720,16)|0)==0){h=1;m=23176}else{if((p6(n|0,22712,16)|0)==0){h=1;m=31248}else{if((p6(n|0,30968,16)|0)==0){h=1;m=30656}}}}bB(k|0,20,1,f|0)|0;n=0;while(1){if((n|0)>=31){break}bB(k+20+(n*30|0)|0,22,1,f|0)|0;b[k+20+(n*30|0)+22>>1]=jd(f)|0;a[k+20+(n*30|0)+24|0]=ja(f)|0;a[k+20+(n*30|0)+25|0]=ja(f)|0;b[k+20+(n*30|0)+26>>1]=jd(f)|0;b[k+20+(n*30|0)+28>>1]=jd(f)|0;n=n+1|0}a[k+950|0]=ja(f)|0;a[k+951|0]=ja(f)|0;bB(k+952|0,128,1,f|0)|0;bB(k+1080|0,4,1,f|0)|0;if((d[k+1083|0]|0)==52){c[g+136>>2]=4}else{c[g+136>>2]=8}c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];c[g+156>>2]=d[k+950|0]|0;c[g+160>>2]=d[k+951|0]|0;s=g+952|0;t=k+952|0;p3(s|0,t|0,128)|0;n=0;while(1){if((n|0)>=128){break}if((c[g+136>>2]|0)>4){t=g+952+n|0;a[t]=d[t]>>1&255}if((d[g+952+n|0]|0)>(c[g+128>>2]|0)){c[g+128>>2]=d[g+952+n|0]|0}n=n+1|0}t=g+128|0;c[t>>2]=(c[t>>2]|0)+1;c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;p5(g|0,k|0,20)|0;dm(o,30336,(q=i,i=i+16|0,c[q>>2]=m,c[q+8>>2]=k+1080,q)|0);i=q;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}n=0;while(1){if((n|0)>=(c[g+140>>2]|0)){break}q=pT(64,1)|0;c[(c[g+176>>2]|0)+(n*764|0)+756>>2]=q;c[(c[g+180>>2]|0)+(n*52|0)+32>>2]=e[k+20+(n*30|0)+22>>1]<<1;c[(c[g+180>>2]|0)+(n*52|0)+36>>2]=e[k+20+(n*30|0)+26>>1]<<1;c[(c[g+180>>2]|0)+(n*52|0)+40>>2]=(c[(c[g+180>>2]|0)+(n*52|0)+36>>2]|0)+(e[k+20+(n*30|0)+28>>1]<<1);c[(c[g+180>>2]|0)+(n*52|0)+44>>2]=(e[k+20+(n*30|0)+28>>1]|0)>1?2:0;c[(c[(c[g+176>>2]|0)+(n*764|0)+756>>2]|0)+16>>2]=(a[k+20+(n*30|0)+24|0]<<4&255)<<24>>24;c[c[(c[g+176>>2]|0)+(n*764|0)+756>>2]>>2]=a[k+20+(n*30|0)+25|0]|0;c[(c[(c[g+176>>2]|0)+(n*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]=n;c[(c[g+176>>2]|0)+(n*764|0)+36>>2]=((c[(c[g+180>>2]|0)+(n*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[g+176>>2]|0)+(n*764|0)+40>>2]=4095;dh((c[g+176>>2]|0)+(n*764|0)|0,k+20+(n*30|0)|0,22)|0;do{if((h|0)!=0){if((d_(r,n)|0)==0){u=2083;break}}else{u=2083}}while(0);if((u|0)==2083){u=0;if((a[(c[g+176>>2]|0)+(n*764|0)|0]|0)!=0){u=2085}else{if((c[(c[g+180>>2]|0)+(n*52|0)+32>>2]|0)>2){u=2085}}if((u|0)==2085){u=0}}n=n+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;n=0;while(1){if((n|0)>=(c[g+128>>2]|0)){break}u=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(n<<2)>>2]=u;c[c[(c[g+168>>2]|0)+(n<<2)>>2]>>2]=64;u=0;while(1){if((u|0)>=(c[g+136>>2]|0)){break}k=(ac(n,c[g+136>>2]|0)|0)+u|0;c[(c[(c[g+168>>2]|0)+(n<<2)>>2]|0)+4+(u<<2)>>2]=k;q=pT(((c[c[(c[g+168>>2]|0)+(n<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(k<<2)>>2]=q;c[c[(c[g+172>>2]|0)+(k<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(n<<2)>>2]>>2];u=u+1|0}u=0;while(1){if((u|0)>=256){break}v=(c[(c[o+172>>2]|0)+(c[(c[(c[o+168>>2]|0)+(n<<2)>>2]|0)+4+(((u|0)%4|0)<<2)>>2]<<2)>>2]|0)+4+(((u|0)/4|0)<<3)|0;bB(l|0,1,4,f|0)|0;dj(v,l|0);u=u+1|0}if((c[g+136>>2]|0)>4){u=0;while(1){if((u|0)>=256){break}v=(c[(c[o+172>>2]|0)+(c[(c[(c[o+168>>2]|0)+(n<<2)>>2]|0)+4+(((u|0)%4|0)+4<<2)>>2]<<2)>>2]|0)+4+(((u|0)/4|0)<<3)|0;bB(l|0,1,4,f|0)|0;dj(v,l|0);if((d[v+3|0]|0)==14){a[v+4|0]=0;a[v+3|0]=0}u=u+1|0}}n=n+1|0}n=0;while(1){if((n|0)>=(c[g+144>>2]|0)){break}if((c[(c[g+180>>2]|0)+(n*52|0)+32>>2]|0)==0){do{if((h|0)!=0){if((d_(r,n)|0)==0){break}d$(o,r,n)}}while(0)}else{v=o;l=f;u=(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(v,l,512,u,0)|0}n=n+1|0}if((r|0)==0){i=j;return 0}au(r|0)|0;i=j;return 0}function d_(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;c=i;i=i+8|0;d=c|0;e=a;bK(e|0,(b*120|0)+144|0,0)|0;if((bB(d|0,1,2,e|0)|0)>>>0<2){f=0;g=f;i=c;return g|0}if((p6(d|0,29984,2)|0)!=0){f=0;g=f;i=c;return g|0}bK(e|0,24,1)|0;d=jd(e)|0;do{if((d<<16>>16|0)>=0){if((d<<16>>16|0)>3){break}f=1;g=f;i=c;return g|0}}while(0);f=0;g=f;i=c;return g|0}function d$(d,e,f){d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0;g=i;i=i+1056|0;h=g|0;j=g+32|0;k=d;d=e;e=f;f=k|0;bK(d|0,(e*120|0)+150|0,0)|0;b[h>>1]=jd(d)|0;b[h+2>>1]=jd(d)|0;b[h+4>>1]=jd(d)|0;b[h+6>>1]=jd(d)|0;b[h+8>>1]=jd(d)|0;b[h+10>>1]=jd(d)|0;b[h+12>>1]=jd(d)|0;b[h+14>>1]=jd(d)|0;jd(d)|0;b[h+16>>1]=jd(d)|0;b[h+18>>1]=jd(d)|0;b[h+20>>1]=-((jd(d)|0)<<16>>16)&65535;b[h+22>>1]=jd(d)|0;b[h+24>>1]=jd(d)|0;b[h+26>>1]=jd(d)|0;if((b[h+18>>1]|0)<3){c[(c[f+180>>2]|0)+(e*52|0)+32>>2]=32;c[(c[f+180>>2]|0)+(e*52|0)+36>>2]=0;c[(c[f+180>>2]|0)+(e*52|0)+40>>2]=32;l=21992+(b[h+18>>1]<<5)|0}else{c[(c[f+180>>2]|0)+(e*52|0)+32>>2]=1024;c[(c[f+180>>2]|0)+(e*52|0)+36>>2]=0;c[(c[f+180>>2]|0)+(e*52|0)+40>>2]=1024;d=0;while(1){if((d|0)>=1024){break}a[j+d|0]=((a9()|0)%256|0)&255;d=d+1|0}l=j|0}c[(c[f+180>>2]|0)+(e*52|0)+44>>2]=2;c[c[(c[f+176>>2]|0)+(e*764|0)+756>>2]>>2]=64;c[(c[f+176>>2]|0)+(e*764|0)+36>>2]=1;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+12>>2]=(b[h+26>>1]|0)*-12|0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+20>>2]=0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+24>>2]=b[h+22>>1]|0;c[(c[(c[f+176>>2]|0)+(e*764|0)+756>>2]|0)+28>>2]=b[h+24>>1]|0;if((b[h+4>>1]|0)==0){b[h+4>>1]=1}if((b[h+8>>1]|0)==0){b[h+8>>1]=1}if((b[h+12>>1]|0)==0){b[h+12>>1]=1}if((b[h+16>>1]|0)==0){b[h+16>>1]=1}c[(c[f+176>>2]|0)+(e*764|0)+48>>2]=6;c[(c[f+176>>2]|0)+(e*764|0)+44>>2]=1;b[(c[f+176>>2]|0)+(e*764|0)+72>>1]=0;b[(c[f+176>>2]|0)+(e*764|0)+74>>1]=((b[h>>1]|0)/4|0)&65535;if((b[h+2>>1]|0)>(b[h>>1]|0)){m=(b[h+2>>1]|0)-(b[h>>1]|0)|0;n=256-(b[h>>1]|0)|0}else{m=(b[h>>1]|0)-(b[h+2>>1]|0)|0;n=b[h>>1]|0}if((n|0)==0){n=1}j=b[(c[f+176>>2]|0)+(e*764|0)+72>>1]|0;d=j+((m<<8|0)/(ac(b[h+4>>1]|0,n)|0)|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+76>>1]=d;b[(c[f+176>>2]|0)+(e*764|0)+78>>1]=((b[h+2>>1]|0)/4|0)&65535;if((b[h+6>>1]|0)>(b[h+2>>1]|0)){m=(b[h+6>>1]|0)-(b[h+2>>1]|0)|0;n=256-(b[h+2>>1]|0)|0}else{m=(b[h+2>>1]|0)-(b[h+6>>1]|0)|0;n=b[h+2>>1]|0}if((n|0)==0){n=1}d=b[(c[f+176>>2]|0)+(e*764|0)+76>>1]|0;j=d+((m<<8|0)/(ac(b[h+8>>1]|0,n)|0)|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+80>>1]=j;b[(c[f+176>>2]|0)+(e*764|0)+82>>1]=((b[h+6>>1]|0)/4|0)&65535;if((b[h+10>>1]|0)>(b[h+6>>1]|0)){m=(b[h+10>>1]|0)-(b[h+6>>1]|0)|0;n=256-(b[h+6>>1]|0)|0}else{m=(b[h+6>>1]|0)-(b[h+10>>1]|0)|0;n=b[h+6>>1]|0}if((n|0)==0){n=1}j=b[(c[f+176>>2]|0)+(e*764|0)+80>>1]|0;d=j+((m<<8|0)/(ac(b[h+12>>1]|0,n)|0)|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+84>>1]=d;b[(c[f+176>>2]|0)+(e*764|0)+86>>1]=((b[h+10>>1]|0)/4|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+88>>1]=(b[(c[f+176>>2]|0)+(e*764|0)+84>>1]|0)+(b[h+14>>1]|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+90>>1]=((b[h+10>>1]|0)/4|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+92>>1]=(b[(c[f+176>>2]|0)+(e*764|0)+88>>1]|0)+(256/(b[h+16>>1]|0)|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+94>>1]=0;if((b[h+20>>1]|0)==0){o=k;p=e;q=f;r=q+176|0;s=c[r>>2]|0;t=s+(p*764|0)|0;u=t+756|0;v=c[u>>2]|0;w=v|0;x=w+40|0;y=c[x>>2]|0;z=f;A=z+180|0;B=c[A>>2]|0;C=B+(y*52|0)|0;D=l;E=gQ(o,0,16,C,D)|0;i=g;return}c[(c[f+176>>2]|0)+(e*764|0)+360>>2]=2;c[(c[f+176>>2]|0)+(e*764|0)+356>>2]=1;b[(c[f+176>>2]|0)+(e*764|0)+384>>1]=0;b[(c[f+176>>2]|0)+(e*764|0)+386>>1]=0;d=(1024/(R(b[h+20>>1]|0)|0)|0)&65535;b[(c[f+176>>2]|0)+(e*764|0)+388>>1]=d;b[(c[f+176>>2]|0)+(e*764|0)+390>>1]=(((b[h+20>>1]|0)<0?-256:256)*10|0)&65535;o=k;p=e;q=f;r=q+176|0;s=c[r>>2]|0;t=s+(p*764|0)|0;u=t+756|0;v=c[u>>2]|0;w=v|0;x=w+40|0;y=c[x>>2]|0;z=f;A=z+180|0;B=c[A>>2]|0;C=B+(y*52|0)|0;D=l;E=gQ(o,0,16,C,D)|0;i=g;return}function d0(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;d=i;i=i+72|0;e=d|0;f=a;a=b;L2639:do{if((jl(f)|0)!=1182101099){g=-1}else{b=f;ja(b)|0;b=ja(f)|0;h=ja(f)|0;j=f;ja(j)|0;if(((b&255)>>1|0)<10){g=-1;break}do{if(((h&240)>>4|0)<=7){if((h&15|0)>9){break}b=jg(f)|0;if((b|0)<1024){g=-1;break L2639}j=aE(f|0)|0;aC(j|0,e|0)|0;if((b|0)!=(c[e+28>>2]|0)){g=-1;break L2639}else{di(f,a,0);g=0;break L2639}}}while(0);g=-1}}while(0);i=d;return g|0}function d1(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;g=i;i=i+2720|0;h=g|0;j=g+2712|0;k=b;b=e;e=k|0;bK(b|0,f|0,0)|0;bB(h|0,4,1,b|0)|0;bB(h+4|0,4,1,b|0)|0;c[h+8>>2]=jg(b)|0;bB(h+12|0,4,1,b|0)|0;a[h+16|0]=ja(b)|0;bB(h+17|0,256,1,b|0)|0;bB(h+273|0,128,1,b|0)|0;f=0;while(1){if((f|0)>=64){break}bB(h+404+(f*36|0)|0,19,1,b|0)|0;c[h+404+(f*36|0)+20>>2]=jg(b)|0;c[h+404+(f*36|0)+24>>2]=jg(b)|0;a[h+404+(f*36|0)+28|0]=ja(b)|0;a[h+404+(f*36|0)+29|0]=ja(b)|0;a[h+404+(f*36|0)+30|0]=ja(b)|0;a[h+404+(f*36|0)+31|0]=ja(b)|0;a[h+404+(f*36|0)+32|0]=ja(b)|0;f=f+1|0}c[e+140>>2]=64;c[e+144>>2]=64;f=0;while(1){if((f|0)<256){l=(d[h+17+f|0]|0)!=255}else{l=0}if(!l){break}if((d[h+17+f|0]|0)>(c[e+128>>2]|0)){c[e+128>>2]=d[h+17+f|0]|0}f=f+1|0}l=e+128|0;c[l>>2]=(c[l>>2]|0)+1;c[e+156>>2]=f;l=e+952|0;m=h+17|0;n=c[e+156>>2]|0;p3(l|0,m|0,n)|0;c[e+148>>2]=4;c[e+152>>2]=125;c[e+136>>2]=0;do{if((d[h+12|0]|0)==70){if((d[h+13|0]|0)!=50){o=2216;break}if((a[h+7|0]>>1&64|0)!=0){n=e+152|0;c[n>>2]=(c[n>>2]|0)-(d[h+7|0]>>1&63)}else{n=e+152|0;c[n>>2]=(c[n>>2]|0)+(d[h+7|0]>>1&63)}dm(k,30808,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0);i=p}else{o=2216}}while(0);if((o|0)==2216){do{if((d[h+12|0]|0)==70){if((d[h+13|0]|0)!=118){if((d[h+13|0]|0)!=107){o=2220;break}}dm(k,28408,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0);i=p}else{o=2220}}while(0);if((o|0)==2220){c[e+136>>2]=8;dm(k,27064,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0);i=p}}if((c[e+136>>2]|0)==0){do{if((d[h+14|0]|0)<48){o=2227}else{if((d[h+14|0]|0)>57){o=2227;break}if((d[h+15|0]|0)<48){o=2227;break}if((d[h+15|0]|0)>57){o=2227;break}q=(((d[h+14|0]|0)-48|0)*10|0)+(d[h+15|0]|0)-48|0}}while(0);if((o|0)==2227){q=8}c[e+136>>2]=q}c[e+152>>2]=(c[e+152>>2]<<2|0)/5|0;c[e+132>>2]=ac(c[e+136>>2]|0,c[e+128>>2]|0)|0;q=k+1280|0;c[q>>2]=c[q>>2]|4096;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}q=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=q;q=c[h+404+(f*36|0)+24>>2]|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=q;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=((q|0)!=0^1^1)&1;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=c[h+404+(f*36|0)+20>>2];if((c[(c[e+180>>2]|0)+(f*52|0)+36>>2]|0)==-1){c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=0}c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=c[h+404+(f*36|0)+24>>2];c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(c[h+404+(f*36|0)+20>>2]|0)!=-1?2:0;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=d[h+404+(f*36|0)+28|0]|0;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=d[h+404+(f*36|0)+29|0]|0;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=f;dh((c[e+176>>2]|0)+(f*764|0)|0,h+404+(f*36|0)|0,19)|0;f=f+1|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;f=0;while(1){if((f|0)>=(c[e+128>>2]|0)){break}q=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(f<<2)>>2]=q;c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]=64;q=0;while(1){if((q|0)>=(c[e+136>>2]|0)){break}o=(ac(f,c[e+136>>2]|0)|0)+q|0;c[(c[(c[e+168>>2]|0)+(f<<2)>>2]|0)+4+(q<<2)>>2]=o;p=pT(((c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(o<<2)>>2]=p;c[c[(c[e+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2];q=q+1|0}a[(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(f<<2)>>2]|0)+8>>2]<<2)>>2]|0)+4+(d[h+273+f|0]<<3)+5|0]=13;q=0;while(1){if((q|0)>=(c[e+136>>2]<<6|0)){break}o=(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(f<<2)>>2]|0)+4+(((q|0)%(c[e+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((q|0)/(c[e+136>>2]|0)|0)<<3)|0;bB(j|0,1,3,b|0)|0;p=d[j|0]>>2;if(!((p|0)==63|(p|0)==62|(p|0)==61)){a[o|0]=(d[j|0]>>2)+37&255;a[o+1|0]=((a[j+1|0]&240)>>4)+1+((a[j|0]&3)<<4)&255;a[o+2|0]=a[h+404+(((d[o+1|0]|0)-1|0)*36|0)+28|0]|0}p=a[j+1|0]&15;L2750:do{if((p|0)==0){a[o+3|0]=121;a[o+4|0]=a[j+2|0]|0}else if((p|0)==14){do{if((d[j+2|0]|0)!=10){if((d[j+2|0]|0)==11){break}if((d[j+2|0]|0)==12){break}n=(a[j+2|0]&240)>>4;if((n|0)==1){a[o+3|0]=14;a[o+4|0]=(192|a[j+2|0]&15)&255}else if((n|0)==2){a[o+3|0]=14;a[o+4|0]=(208|a[j+2|0]&15)&255}else if((n|0)==13){a[o+3|0]=14;a[o+4|0]=(144|a[j+2|0]&15)&255}else if((n|0)==14){a[o+3|0]=8;a[o+4|0]=((a[j+2|0]&15)<<4)+8&255}else if((n|0)==15){a[o+3|0]=15;a[o+4|0]=d[j+2|0]&15}break L2750}}while(0);a[o+3|0]=127}else if((p|0)==1){a[o+3|0]=120;a[o+4|0]=a[j+2|0]|0}else if((p|0)==2){a[o+3|0]=122;a[o+4|0]=a[j+2|0]|0}else if((p|0)==3){a[o+3|0]=123;a[o+4|0]=a[j+2|0]|0}else if((p|0)==6){a[o+3|0]=124;a[o+4|0]=d[j+2|0]<<1&255}else if((p|0)==7){a[o+3|0]=125;a[o+4|0]=d[j+2|0]<<1&255}else if((p|0)==11){a[o+3|0]=0;a[o+4|0]=a[j+2|0]|0}else if((p|0)==13){a[o+3|0]=12;a[o+4|0]=a[j+2|0]|0}}while(0);q=q+1|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}if((c[(c[e+180>>2]|0)+(f*52|0)+32>>2]|0)>2){j=k;h=b;q=(c[e+180>>2]|0)+(f*52|0)|0;gQ(j,h,0,q,0)|0}f=f+1|0}f=0;while(1){if((f|0)>=(c[e+136>>2]|0)){break}c[e+184+(f*12|0)>>2]=128;f=f+1|0}c[k+1268>>2]=255;c[k+1280>>2]=64;i=g;return 0}function d2(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;c=a;a=b;do{if((jl(c)|0)!=1380533830){d=-1}else{b=c;jl(b)|0;if((jl(c)|0)!=1095583302){d=-1;break}if((jl(c)|0)!=1296124238){d=-1;break}else{b=c;jl(b)|0;di(c,a,64);d=0;break}}}while(0);return d|0}function d3(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+8|0;f=e|0;g=a;a=b;b=d;d=g|0;bK(a|0,b|0,0)|0;jl(a)|0;jl(a)|0;jl(a)|0;h=aT(a|0)|0;c[d+140>>2]=0;c[d+144>>2]=0;j=es()|0;if((j|0)==0){k=-1;l=k;i=e;return l|0}ev(j,30776,114);ev(j,28384,56);ev(j,27056,94);ev(j,25952,48);ey(j,1);ey(j,32);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(j,g,a,f)}eB(j);c[d+132>>2]=ac(c[d+128>>2]|0,c[d+136>>2]|0)|0;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;bK(a|0,b+h|0,0)|0;c[f>>2]=0;j=es()|0;if((j|0)==0){k=-1;l=k;i=e;return l|0}ev(j,27056,58);ev(j,25952,128);ey(j,1);ey(j,32);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(j,g,a,f)}eB(j);j=0;while(1){if((j|0)>=(c[d+136>>2]|0)){break}c[d+184+(j*12|0)>>2]=128;j=j+1|0}j=g+1280|0;c[j>>2]=c[j>>2]|32;c[g+1284>>2]=1;k=0;l=k;i=e;return l|0}function d4(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;e=i;i=i+64|0;b=e|0;f=a;a=d;d=f|0;bB(b|0,1,64,a|0)|0;p5(d|0,b|0,64)|0;dm(f,28944,(b=i,i=i+1|0,i=i+7>>3<<3,c[b>>2]=0,b)|0);i=b;if((~((ja(a)|0)&255)&1|0)!=0){c[f+1280>>2]=4096}c[d+136>>2]=(ja(a)|0)&255;c[d+148>>2]=(ja(a)|0)&255;c[d+152>>2]=(ja(a)|0)&255;jc(a)|0;jc(a)|0;ja(a)|0;i=e;return}function d5(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;f=e;e=b|0;c[e+156>>2]=(ja(f)|0)&255;b=0;while(1){if((b|0)>=(c[e+156>>2]|0)){break}a[e+952+b|0]=ja(f)|0;b=b+1|0}return}function d6(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;a=((ja(d)|0)&255)+1|0;if((a|0)<=(c[e+128>>2]|0)){return}c[e+128>>2]=a;return}function d7(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=d;d=a|0;ja(e)|0;a=((ja(e)|0)&255)+1|0;if((a|0)>(c[d+140>>2]|0)){c[d+140>>2]=a}bK(e|0,28,1)|0;a=(ja(e)|0)&255;e=d+144|0;c[e>>2]=(c[e>>2]|0)+a;return}function d8(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;g=i;i=i+8|0;e=g|0;h=b;b=f;f=h|0;j=(ja(b)|0)&255;jg(b)|0;k=((ja(b)|0)&255)+1|0;l=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(j<<2)>>2]=l;c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]=k;l=0;while(1){if((l|0)>=(c[f+136>>2]|0)){break}m=(ac(j,c[f+136>>2]|0)|0)+l|0;c[(c[(c[f+168>>2]|0)+(j<<2)>>2]|0)+4+(l<<2)>>2]=m;n=pT(((c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(m<<2)>>2]=n;c[c[(c[f+172>>2]|0)+(m<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2];l=l+1|0}l=0;while(1){if((l|0)>=(k|0)){break}m=ja(b)|0;n=m;if((m&255|0)==0){l=l+1|0;continue}m=n&31;if((m|0)<(c[f+136>>2]|0)){o=(c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(j<<2)>>2]|0)+4+(m<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}else{o=e}m=o;if((n&128|0)!=0){p=ja(b)|0;q=ja(b)|0;if((q&255|0)==20){q=-93}else{if((q&255|0)>15){r=q&255;s=p&255;bf(24944,(t=i,i=i+16|0,c[t>>2]=r,c[t+8>>2]=s,t)|0)|0;i=t;p=0;q=0}}a[m+3|0]=q;a[m+4|0]=p}if((n&64|0)!=0){a[m+1|0]=ja(b)|0;a[m|0]=ja(b)|0;if((d[m|0]|0|0)==128){a[m|0]=-127}}if((n&32|0)!=0){a[m+2|0]=(((ja(b)|0)&255|0)/2|0)+1&255}}i=g;return}function d9(e,f,g,h){e=e|0;f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;f=i;i=i+32|0;j=f|0;k=e;e=g;g=k|0;l=h;ja(e)|0;h=(ja(e)|0)&255;bB((c[g+176>>2]|0)+(h*764|0)|0,1,28,e|0)|0;hy((c[g+176>>2]|0)+(h*764|0)|0)|0;m=(ja(e)|0)&255;c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=m;m=0;while(1){if((m|0)>=108){break}n=ja(e)|0;a[(c[g+176>>2]|0)+(h*764|0)+512+(m<<1)|0]=n;m=m+1|0}bK(e|0,11,1)|0;n=(ja(e)|0)&255;o=(ja(e)|0)&255;ja(e)|0;ja(e)|0;p=((ja(e)|0)&255|0)/4|0;q=((jc(e)|0)&65535|0)/16|0;ja(e)|0;r=(ja(e)|0)&255;if((r&1|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+44|0;c[s>>2]=c[s>>2]|1}if((r&2|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+44|0;c[s>>2]=c[s>>2]|2}if((r&4|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+44|0;c[s>>2]=c[s>>2]|4}if(((r&240)>>4&1|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+200|0;c[s>>2]=c[s>>2]|1}if(((r&240)>>4&2|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+200|0;c[s>>2]=c[s>>2]|2}if(((r&240)>>4&4|0)!=0){s=(c[g+176>>2]|0)+(h*764|0)+200|0;c[s>>2]=c[s>>2]|4}r=(ja(e)|0)&255;c[(c[g+176>>2]|0)+(h*764|0)+48>>2]=(r&15)+1;c[(c[g+176>>2]|0)+(h*764|0)+204>>2]=((r&240)>>4)+1;r=(ja(e)|0)&255;c[(c[g+176>>2]|0)+(h*764|0)+56>>2]=r&15;c[(c[g+176>>2]|0)+(h*764|0)+212>>2]=(r&240)>>4;r=(ja(e)|0)&255;c[(c[g+176>>2]|0)+(h*764|0)+64>>2]=r&15;c[(c[g+176>>2]|0)+(h*764|0)+220>>2]=(r&240)>>4;ja(e)|0;c[(c[g+176>>2]|0)+(h*764|0)+68>>2]=r&15;c[(c[g+176>>2]|0)+(h*764|0)+224>>2]=(r&240)>>4;if((c[(c[g+176>>2]|0)+(h*764|0)+48>>2]|0)<=0){t=2415}else{if((c[(c[g+176>>2]|0)+(h*764|0)+48>>2]|0)>=32){t=2415}}if((t|0)==2415){r=(c[g+176>>2]|0)+(h*764|0)+44|0;c[r>>2]=c[r>>2]&-2}if((c[(c[g+176>>2]|0)+(h*764|0)+204>>2]|0)<=0){t=2418}else{if((c[(c[g+176>>2]|0)+(h*764|0)+204>>2]|0)>=32){t=2418}}if((t|0)==2418){t=(c[g+176>>2]|0)+(h*764|0)+200|0;c[t>>2]=c[t>>2]&-2}bB(j|0,1,30,e|0)|0;m=0;while(1){if((m|0)>=(c[(c[g+176>>2]|0)+(h*764|0)+48>>2]|0)){break}t=(((jh(j+(m*3|0)|0)|0)&65535|0)/16|0)&65535;b[(c[g+176>>2]|0)+(h*764|0)+72+(m<<1<<1)>>1]=t;b[(c[g+176>>2]|0)+(h*764|0)+72+((m<<1)+1<<1)>>1]=d[j+((m*3|0)+2)|0]|0;m=m+1|0}bB(j|0,1,30,e|0)|0;m=0;while(1){if((m|0)>=(c[(c[g+176>>2]|0)+(h*764|0)+204>>2]|0)){break}t=(((jh(j+(m*3|0)|0)|0)&65535|0)/16|0)&65535;b[(c[g+176>>2]|0)+(h*764|0)+228+(m<<1<<1)>>1]=t;b[(c[g+176>>2]|0)+(h*764|0)+228+((m<<1)+1<<1)>>1]=d[j+((m*3|0)+2)|0]|0;m=m+1|0}ja(e)|0;ja(e)|0;if((c[(c[g+176>>2]|0)+(h*764|0)+36>>2]|0)==0){i=f;return}j=pT(64,c[(c[g+176>>2]|0)+(h*764|0)+36>>2]|0)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=j;m=0;while(1){if((m|0)>=(c[(c[g+176>>2]|0)+(h*764|0)+36>>2]|0)){break}jl(e)|0;jl(e)|0;bB((c[g+180>>2]|0)+((c[l>>2]|0)*52|0)|0,1,28,e|0)|0;hy((c[g+180>>2]|0)+((c[l>>2]|0)*52|0)|0)|0;j=((ja(e)|0)&255)<<2;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+8>>2]=j;if((c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+8>>2]|0)==0){c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+8>>2]=128}j=(ja(e)|0)&255;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)>>2]=j;j=(ja(e)|0)&255;ja(e)|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+20>>2]=n;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+24>>2]=p;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+28>>2]=q;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+32>>2]=o;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+40>>2]=c[l>>2];t=jg(e)|0;c[(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+32>>2]=t;t=jg(e)|0;c[(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+36>>2]=t;t=jg(e)|0;c[(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+40>>2]=t;c[(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+44>>2]=0;if((j&4|0)!=0){t=(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+44|0;c[t>>2]=c[t>>2]|1}if((j&8|0)!=0){t=(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+44|0;c[t>>2]=c[t>>2]|2}if((j&16|0)!=0){j=(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+44|0;c[j>>2]=c[j>>2]|4}j=jg(e)|0;hD(j,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+12|0,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+16|0);j=(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(m<<6)+16|0;c[j>>2]=(c[j>>2]|0)+0;jg(e)|0;jg(e)|0;if((c[(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)+32>>2]|0)>1){j=k;t=e;r=(c[g+180>>2]|0)+((c[l>>2]|0)*52|0)|0;gQ(j,t,0,r,0)|0}m=m+1|0;r=l|0;c[r>>2]=(c[r>>2]|0)+1}i=f;return}function ea(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;c=a;a=b;do{if((jl(c)|0)!=1380533830){d=-1}else{b=c;jl(b)|0;if((jl(c)|0)!=1095573536){d=-1;break}if((jl(c)|0)!=1229867348){d=-1;break}else{b=c;jl(b)|0;di(c,a,64);d=0;break}}}while(0);return d|0}function eb(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+64|0;g=f|0;h=a;a=b;b=e;e=h|0;bK(a|0,b|0,0)|0;jl(a)|0;jl(a)|0;jl(a)|0;j=aT(a|0)|0;c[e+140>>2]=0;c[e+144>>2]=0;k=es()|0;if((k|0)==0){l=-1;m=l;i=f;return m|0}ev(k,30752,30);ev(k,28376,44);ev(k,27048,124);ev(k,25944,28);ey(k,1);ey(k,16);ey(k,4);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(k,h,a,g)}eB(k);c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+144>>2]=c[e+140>>2];c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;bK(a|0,b+j|0,0)|0;k=es()|0;if((k|0)==0){l=-1;m=l;i=f;return m|0}ev(k,27048,76);ev(k,25944,34);ey(k,1);ey(k,16);ey(k,4);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(k,h,a,g)}eB(k);k=0;while(1){if((k|0)>=(c[e+136>>2]|0)){break}c[e+184+(k*12|0)>>2]=(d[g+k|0]|0)<<1;k=k+1|0}k=h+1280|0;c[k>>2]=c[k>>2]|32;c[h+1284>>2]=1;l=0;m=l;i=f;return m|0}function ec(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0;b=i;i=i+64|0;f=b|0;g=a;a=d;d=g|0;bB(f|0,1,64,a|0)|0;p5(d|0,f|0,64)|0;dm(g,24128,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0);i=f;if((~((ja(a)|0)&255)&1|0)!=0){f=g+1280|0;c[f>>2]=c[f>>2]|4096}c[d+136>>2]=(ja(a)|0)&255;c[d+148>>2]=(ja(a)|0)&255;c[d+152>>2]=(ja(a)|0)&255;jc(a)|0;jc(a)|0;ja(a)|0;bB(e|0,1,64,a|0)|0;i=b;return}function ed(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;f=e;e=b|0;c[e+156>>2]=((ja(f)|0)&255)+1;b=0;while(1){if((b|0)>=(c[e+156>>2]|0)){break}a[e+952+b|0]=ja(f)|0;b=b+1|0}return}function ee(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;a=((ja(d)|0)&255)+1|0;if((a|0)<=(c[e+128>>2]|0)){return}c[e+128>>2]=a;return}function ef(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=d;d=a|0;jl(e)|0;ja(e)|0;a=((ja(e)|0)&255)+1|0;if((a|0)<=(c[d+140>>2]|0)){return}c[d+140>>2]=a;return}function eg(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;g=i;i=i+8|0;e=g|0;h=b;b=f;f=h|0;j=(ja(b)|0)&255;jg(b)|0;k=((ja(b)|0)&255)+1|0;l=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(j<<2)>>2]=l;c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]=k;l=0;while(1){if((l|0)>=(c[f+136>>2]|0)){break}m=(ac(j,c[f+136>>2]|0)|0)+l|0;c[(c[(c[f+168>>2]|0)+(j<<2)>>2]|0)+4+(l<<2)>>2]=m;n=pT(((c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(m<<2)>>2]=n;c[c[(c[f+172>>2]|0)+(m<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(j<<2)>>2]>>2];l=l+1|0}l=0;while(1){if((l|0)>=(k|0)){break}m=ja(b)|0;n=m;if((m&255|0)==0){l=l+1|0;continue}m=n&31;if((m|0)<(c[f+136>>2]|0)){o=(c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(j<<2)>>2]|0)+4+(m<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}else{o=e}m=o;if((n&128|0)!=0){p=ja(b)|0;q=ja(b)|0;if((q&255|0)==20){q=-93}else{if((q&255|0)>15){r=q&255;s=p&255;bf(24912,(t=i,i=i+16|0,c[t>>2]=r,c[t+8>>2]=s,t)|0)|0;i=t;p=0;q=0}}a[m+3|0]=q;a[m+4|0]=p}if((n&64|0)!=0){a[m+1|0]=ja(b)|0;a[m|0]=ja(b)|0;if((d[m|0]|0|0)==128){a[m|0]=-127}}if((n&32|0)!=0){a[m+2|0]=(((ja(b)|0)&255|0)/2|0)+1&255}}i=g;return}function eh(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;e=a;a=d;d=e|0;jl(a)|0;ja(a)|0;b=(ja(a)|0)&255;bB((c[d+176>>2]|0)+(b*764|0)|0,1,28,a|0)|0;hy((c[d+176>>2]|0)+(b*764|0)|0)|0;bK(a|0,290,1)|0;f=(jc(a)|0)&65535;c[(c[d+176>>2]|0)+(b*764|0)+36>>2]=f;if((c[(c[d+176>>2]|0)+(b*764|0)+36>>2]|0)==0){return}f=pT(64,c[(c[d+176>>2]|0)+(b*764|0)+36>>2]|0)|0;c[(c[d+176>>2]|0)+(b*764|0)+756>>2]=f;jl(a)|0;jl(a)|0;jl(a)|0;jl(a)|0;jl(a)|0;jl(a)|0;bB((c[d+180>>2]|0)+(b*52|0)|0,1,28,a|0)|0;hy((c[d+180>>2]|0)+(b*52|0)|0)|0;jl(a)|0;ja(a)|0;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+40>>2]=b;f=(ja(a)|0)&255;c[(c[d+176>>2]|0)+(b*764|0)+32>>2]=f;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+8>>2]=128;f=(((jc(a)|0)&65535)+1|0)/512|0;c[c[(c[d+176>>2]|0)+(b*764|0)+756>>2]>>2]=f;f=(jc(a)|0)&65535;jc(a)|0;g=jg(a)|0;c[(c[d+180>>2]|0)+(b*52|0)+32>>2]=g;g=jg(a)|0;c[(c[d+180>>2]|0)+(b*52|0)+36>>2]=g;g=jg(a)|0;c[(c[d+180>>2]|0)+(b*52|0)+40>>2]=g;c[(c[d+180>>2]|0)+(b*52|0)+44>>2]=0;g=0;if((f&4|0)!=0){h=(c[d+180>>2]|0)+(b*52|0)+44|0;c[h>>2]=c[h>>2]|1}if((f&8|0)!=0){h=(c[d+180>>2]|0)+(b*52|0)+44|0;c[h>>2]=c[h>>2]|2}if((f&16|0)!=0){h=(c[d+180>>2]|0)+(b*52|0)+44|0;c[h>>2]=c[h>>2]|6}if((~f&128|0)!=0){g=1}f=jg(a)|0;hD(f,(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+16|0);f=(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+16|0;c[f>>2]=(c[f>>2]|0)+0;jg(a)|0;jg(a)|0;if((c[(c[d+180>>2]|0)+(b*52|0)+32>>2]|0)<=1){return}gQ(e,a,(g|0)!=0?2:0,(c[d+180>>2]|0)+(b*52|0)|0,0)|0;return}function ei(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;a=c;if((jl(d)|0)!=1195658750){e=-1;f=e;return f|0}bK(d|0,a+71|0,0)|0;if((jl(d)|0)!=1196246611){e=-1;f=e;return f|0}else{c=d;g=a+4|0;bK(c|0,g|0,0)|0;di(d,b,32);e=0;f=e;return f|0}return 0}function ej(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;g=i;i=i+64|0;h=g|0;j=g+32|0;k=b;b=e;e=f;f=k|0;bK(b|0,e|0,0)|0;jl(b)|0;bB(f|0,1,32,b|0)|0;bK(b|0,32,1)|0;bK(b|0,7,1)|0;l=(ja(b)|0)&255;m=(ja(b)|0)&255;n=(jc(b)|0)&65535;o=(ja(b)|0)&255;p=(ja(b)|0)&255;if((n|0)==0){dm(k,30704,(q=i,i=i+32|0,c[q>>2]=l,c[q+8>>2]=m,c[q+16>>2]=o,c[q+24>>2]=p,q)|0);i=q}else{dm(k,28328,(q=i,i=i+32|0,c[q>>2]=l,c[q+8>>2]=m,c[q+16>>2]=o,c[q+24>>2]=p,q)|0);i=q}bB(j|0,32,1,b|0)|0;q=0;while(1){if((q|0)>=32){break}if((d[j+q|0]|0|0)!=255){c[f+136>>2]=q+1}if((d[j+q|0]|0|0)==16){a[j+q|0]=8}c[f+184+(q*12|0)>>2]=((d[j+q|0]|0)-8<<4)+128;q=q+1|0}c[f+164>>2]=(ja(b)|0)&255;c[f+148>>2]=(ja(b)|0)&255;c[f+152>>2]=(ja(b)|0)&255;jc(b)|0;j=jg(b)|0;c[f+156>>2]=((ja(b)|0)&255)+1;p=jg(b)|0;c[f+128>>2]=((ja(b)|0)&255)+1;o=jg(b)|0;m=jg(b)|0;l=((ja(b)|0)&255)+1|0;c[f+144>>2]=l;c[f+140>>2]=l;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;bK(b|0,e+j|0,0)|0;q=0;while(1){if((q|0)>=(c[f+156>>2]|0)){break}a[f+952+q|0]=ja(b)|0;q=q+1|0}bK(b|0,e+o|0,0)|0;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}q=0;while(1){if((q|0)>=(c[f+140>>2]|0)){break}o=pT(64,1)|0;c[(c[f+176>>2]|0)+(q*764|0)+756>>2]=o;bB(h|0,32,1,b|0)|0;dh((c[f+176>>2]|0)+(q*764|0)|0,h|0,32)|0;bK(b|0,12,1)|0;ja(b)|0;o=jg(b)|0;c[(c[f+180>>2]|0)+(q*52|0)+32>>2]=o;o=jg(b)|0;c[(c[f+180>>2]|0)+(q*52|0)+36>>2]=o;o=jg(b)|0;c[(c[f+180>>2]|0)+(q*52|0)+40>>2]=o;o=(ja(b)|0)&255;j=(jc(b)|0)&65535;l=(ja(b)|0)&255;n=(ja(b)|0)&255;if((l|0)>64){r=64}else{r=l}c[c[(c[f+176>>2]|0)+(q*764|0)+756>>2]>>2]=r;if((n|0)>15){s=128}else{s=(n-8<<4)+128|0}c[(c[(c[f+176>>2]|0)+(q*764|0)+756>>2]|0)+8>>2]=s;hD(j,(c[(c[f+176>>2]|0)+(q*764|0)+756>>2]|0)+12|0,(c[(c[f+176>>2]|0)+(q*764|0)+756>>2]|0)+16|0);c[(c[f+176>>2]|0)+(q*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(q*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[f+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]=q;c[(c[f+180>>2]|0)+(q*52|0)+44>>2]=0;if((o&1|0)!=0){j=(c[f+180>>2]|0)+(q*52|0)+44|0;c[j>>2]=c[j>>2]|2}if((o&2|0)!=0){o=(c[f+180>>2]|0)+(q*52|0)+44|0;c[o>>2]=c[o>>2]|1;o=(c[f+180>>2]|0)+(q*52|0)+32|0;c[o>>2]=c[o>>2]>>1;o=(c[f+180>>2]|0)+(q*52|0)+36|0;c[o>>2]=c[o>>2]>>1;o=(c[f+180>>2]|0)+(q*52|0)+40|0;c[o>>2]=c[o>>2]>>1}q=q+1|0}bK(b|0,e+p|0,0)|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;q=0;while(1){if((q|0)>=(c[f+128>>2]|0)){break}p=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(q<<2)>>2]=p;c[c[(c[f+168>>2]|0)+(q<<2)>>2]>>2]=64;p=0;while(1){if((p|0)>=(c[f+136>>2]|0)){break}s=(ac(q,c[f+136>>2]|0)|0)+p|0;c[(c[(c[f+168>>2]|0)+(q<<2)>>2]|0)+4+(p<<2)>>2]=s;r=pT(((c[c[(c[f+168>>2]|0)+(q<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(s<<2)>>2]=r;c[c[(c[f+172>>2]|0)+(s<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(q<<2)>>2]>>2];p=p+1|0}p=(jc(b)|0)&65535;p=p-2|0;s=0;while(1){if((p|0)<=0){break}r=(ja(b)|0)&255;p=p-1|0;if((r|0)==0){s=s+1|0;continue}if((r&31|0)>=(c[f+136>>2]|0)){bV(27e3,239,31488,25920);return 0}h=(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(q<<2)>>2]|0)+4+((r&31)<<2)>>2]<<2)>>2]|0)+4+(s<<3)|0;if((r&32|0)!=0){t=(ja(b)|0)&255;a[h|0]=(((t&112)>>4)*12|0)+12+(t&15)&255;a[h+1|0]=ja(b)|0;p=p-2|0}if((r&64|0)!=0){do{t=(ja(b)|0)&255;p=p-1|0;r=(t&192)>>6;if((r|0)==0){a[h+3|0]=t&31;a[h+4|0]=ja(b)|0;p=p-1|0;ek(h+3|0,h+4|0)}else if((r|0)==1){a[h+5|0]=t&31;a[h+6|0]=ja(b)|0;p=p-1|0;ek(h+5|0,h+6|0)}else if((r|0)==2){ja(b)|0;p=p-1|0}}while((t&32|0)!=0)}}q=q+1|0}bK(b|0,e+m|0,0)|0;q=0;while(1){if((q|0)>=(c[f+140>>2]|0)){break}gQ(k,b,2,(c[f+180>>2]|0)+((c[(c[(c[f+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;q=q+1|0}i=g;return 0}function ek(b,c){b=b|0;c=c|0;var e=0;e=b;b=c;c=d[e]|0;if((c|0)==20){a[e]=-84}else if((c|0)==19){a[e]=16}else if((c|0)==0){a[b]=0}else if((c|0)==31){a[e]=-85}else if(!((c|0)==1|(c|0)==2|(c|0)==3|(c|0)==4|(c|0)==5|(c|0)==6|(c|0)==7))if((c|0)==17){a[b]=0;a[e]=0}else if((c|0)==30){a[b]=0;a[e]=0}else if((c|0)==16){a[e]=0}else if((c|0)==8){a[e]=29}else if((c|0)==18){a[e]=27}else if(!((c|0)==9|(c|0)==10|(c|0)==11|(c|0)==12|(c|0)==13|(c|0)==14|(c|0)==15)){a[b]=0;a[e]=0}return}function el(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0;d=i;i=i+8|0;e=d|0;f=b;b=c;if((bB(e|0,1,4,f|0)|0)>>>0<4){g=-1;h=g;i=d;return h|0}do{if((p6(e|0,28312,3)|0)==0){if((a[e+3|0]|0)>4){break}di(f,b,32);g=0;h=g;i=d;return h|0}}while(0);g=-1;h=g;i=d;return h|0}function em(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;g=i;i=i+40|0;h=g|0;j=b;b=e;e=j|0;bK(b|0,f|0,0)|0;bB(h|0,4,1,b|0)|0;f=d[h+3|0]|0;bB(e|0,32,1,b|0)|0;dm(j,30680,(k=i,i=i+8|0,c[k>>2]=f,k)|0);i=k;bK(b|0,160,1)|0;c[e+140>>2]=(jd(b)|0)&65535;c[e+144>>2]=c[e+140>>2];k=(jd(b)|0)&65535;c[e+136>>2]=(jd(b)|0)&65535;c[e+156>>2]=(jd(b)|0)&65535;c[e+160>>2]=(jd(b)|0)&65535;c[j+1268>>2]=256;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}l=0;while(1){if((l|0)>=(c[e+140>>2]|0)){break}m=pT(64,1)|0;c[(c[e+176>>2]|0)+(l*764|0)+756>>2]=m;bB(h|0,28,1,b|0)|0;dh((c[e+176>>2]|0)+(l*764|0)|0,h|0,28)|0;if((f|0)==1){m=b;jl(m)|0;m=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+32>>2]=m;m=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+36>>2]=m;n=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(l*52|0)+36>>2]|0)+n-1;m=b;jd(m)|0;m=b;jd(m)|0;c[c[(c[e+176>>2]|0)+(l*764|0)+756>>2]>>2]=255;c[(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+8>>2]=128;o=1;p=8363}else{m=b;bK(m|0,14,1)|0;m=b;jd(m)|0;o=(jd(b)|0)&65535;p=(jd(b)|0)&65535;hD(p,(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+12|0,(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+16|0);m=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+32>>2]=m;m=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+36>>2]=m;n=jl(b)|0;c[(c[e+180>>2]|0)+(l*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(l*52|0)+36>>2]|0)+n-1;m=(jd(b)|0)&65535;c[c[(c[e+176>>2]|0)+(l*764|0)+756>>2]>>2]=m;m=b;ja(m)|0;m=(jb(b)|0)<<24>>24;c[(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+16>>2]=m}c[(c[e+176>>2]|0)+(l*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(l*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]=l;c[(c[e+180>>2]|0)+(l*52|0)+44>>2]=(n|0)>2?2:0;if((o|0)>1){m=(c[e+180>>2]|0)+(l*52|0)+44|0;c[m>>2]=c[m>>2]|1;m=(c[e+180>>2]|0)+(l*52|0)+32|0;c[m>>2]=c[m>>2]>>1;m=(c[e+180>>2]|0)+(l*52|0)+36|0;c[m>>2]=c[m>>2]>>1;m=(c[e+180>>2]|0)+(l*52|0)+40|0;c[m>>2]=c[m>>2]>>1}l=l+1|0}l=0;while(1){if((l|0)>=256){break}a[e+952+l|0]=(jd(b)|0)&255;l=l+1|0}l=0;o=0;while(1){if((l|0)>=(c[e+156>>2]|0)){break}if((d[e+952+l|0]|0)>(o|0)){o=d[e+952+l|0]|0}l=l+1|0}c[e+128>>2]=o+1;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;l=0;while(1){if((l|0)>=(c[e+128>>2]|0)){break}o=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(l<<2)>>2]=o;c[c[(c[e+168>>2]|0)+(l<<2)>>2]>>2]=k;o=0;while(1){if((o|0)>=(c[e+136>>2]|0)){break}n=(ac(l,c[e+136>>2]|0)|0)+o|0;c[(c[(c[e+168>>2]|0)+(l<<2)>>2]|0)+4+(o<<2)>>2]=n;p=pT(((c[c[(c[e+168>>2]|0)+(l<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(n<<2)>>2]=p;c[c[(c[e+172>>2]|0)+(n<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(l<<2)>>2]>>2];o=o+1|0}o=0;while(1){if((o|0)>=(c[c[(c[e+168>>2]|0)+(l<<2)>>2]>>2]|0)){break}n=0;while(1){if((n|0)>=(c[e+136>>2]|0)){break}p=(c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(l<<2)>>2]|0)+4+(n<<2)>>2]<<2)>>2]|0)+4+(o<<3)|0;a[p|0]=ja(b)|0;if((a[p|0]|0)!=0){h=p|0;a[h]=(d[h]|0)+13&255}a[p+1|0]=ja(b)|0;a[p+3|0]=ja(b)|0;a[p+4|0]=ja(b)|0;if((f|0)>=4){a[p+2|0]=ja(b)|0}en(p);n=n+1|0}o=o+1|0}l=l+1|0}l=0;while(1){if((l|0)>=(c[e+140>>2]|0)){break}if((c[(c[e+180>>2]|0)+(l*52|0)+32>>2]|0)!=0){f=j;k=b;o=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(f,k,0,o,0)|0}l=l+1|0}i=g;return 0}function en(b){b=b|0;var c=0,e=0;c=b;if((d[c+3|0]|0|0)==14){e=2755}else{if((d[c+3|0]|0|0)==12){e=2755}}if((e|0)==2755){a[c+3|0]=0;a[c+4|0]=0}if((d[c+3|0]|0|0)<=15){return}e=d[c+3|0]|0;if((e|0)==16){a[c+3|0]=0}else if((e|0)==21){a[c+3|0]=-95}else if((e|0)==22){a[c+3|0]=-96}else if((e|0)==32){a[c+3|0]=12}else if((e|0)==33){a[c+3|0]=12;a[c+4|0]=-1}else if((e|0)==164){a[c+3|0]=-91;if((d[c+4|0]|0|0)>15){a[c+4|0]=15}b=c+4|0;a[b]=(d[b]|0)<<4&255}else if((e|0)==165){a[c+3|0]=-91;if((d[c+4|0]|0|0)>15){a[c+4|0]=15}}else if((e|0)==168){a[c+3|0]=-93}else{a[c+4|0]=0;a[c+3|0]=0}return}function eo(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;e=i;i=i+1200|0;f=e|0;g=b;b=c;bK(g|0,1536,1)|0;if((bB(f|0,1,51,g|0)|0)!=51){h=-1;j=h;i=e;return j|0}c=0;k=0;while(1){if((c|0)>=51){break}if((d[f+c|0]|0|0)==255){l=2780;break}if((d[f+c|0]|0|0)>(k|0)){k=d[f+c|0]|0}c=c+1|0}do{if((c|0)!=0){if((k|0)==0){break}if((c|0)>50){break}if((k|0)>50){break}c=0;L3391:while(1){if((c|0)>=(k|0)){l=2809;break}bB(f|0,1,1152,g|0)|0;m=0;while(1){if((m|0)>=64){break}n=0;while(1){if((n|0)>=9){break}o=a[f+(((m*9|0)<<1)+(n<<1))|0]|0;p=a[f+(((m*9|0)<<1)+(n<<1)+1)|0]|0;do{if((p&255|0)>6){if((p&255|0)>=16){break}if((o&255|0)!=128){l=2799;break L3391}}}while(0);if(((p&240)>>4|0)>6){if(((p&240)>>4|0)<10){l=2802;break L3391}}n=n+1|0}m=m+1|0}c=c+1|0}if((l|0)==2799){h=-1;j=h;i=e;return j|0}else if((l|0)==2809){di(g,b,0);h=0;j=h;i=e;return j|0}else if((l|0)==2802){h=-1;j=h;i=e;return j|0}}}while(0);h=-1;j=h;i=e;return j|0}function ep(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;g=i;i=i+1584|0;h=g|0;j=g+8|0;k=g+1544|0;l=b;b=e;e=f;f=l|0;bK(b|0,e|0,0)|0;bB(j|0,1,1536,b|0)|0;m=j|0;n=0;while(1){if((n|0)>=128){break}if((d[m+9|0]&-4|0)!=0){o=2822;break}if((d[m+10|0]&-4|0)!=0){o=2822;break}if((d[m+8|0]&-16|0)!=0){o=2824;break}n=n+1|0;m=m+12|0}c[f+140>>2]=n;bK(b|0,e|0,0)|0;c[f+136>>2]=9;c[f+152>>2]=135;c[f+148>>2]=6;c[f+144>>2]=c[f+140>>2];e=l+1280|0;c[e>>2]=c[e>>2]|4096;dm(l,28392,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0);i=e;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}bB(j|0,1,1536,b|0)|0;e=j|0;n=0;while(1){if((n|0)>=(c[f+140>>2]|0)){break}j=pT(64,1)|0;c[(c[f+176>>2]|0)+(n*764|0)+756>>2]=j;c[(c[f+176>>2]|0)+(n*764|0)+36>>2]=1;c[c[(c[f+176>>2]|0)+(n*764|0)+756>>2]>>2]=64;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+16>>2]=(a[e+11|0]|0)/4|0;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+12>>2]=0;c[(c[(c[f+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]=n;c[(c[f+176>>2]|0)+(n*764|0)+40>>2]=(a[e+7|0]&15)<<5;gQ(l,b,12288,(c[f+180>>2]|0)+(n*52|0)|0,e)|0;n=n+1|0;e=e+12|0}n=0;e=0;while(1){if((n|0)>=51){break}bB(f+952+n|0,1,1,b|0)|0;if((a[f+952+n|0]&128|0)!=0){o=2844;break}if((d[f+952+n|0]|0)>(e|0)){e=d[f+952+n|0]|0}n=n+1|0}bK(b|0,50-n|0,1)|0;c[f+156>>2]=n;c[f+128>>2]=e+1;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;n=0;while(1){if((n|0)>=(c[f+128>>2]|0)){break}e=k;p3(e|0,8432,36)|0;e=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(n<<2)>>2]=e;c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]=64;e=0;while(1){if((e|0)>=(c[f+136>>2]|0)){break}o=(ac(n,c[f+136>>2]|0)|0)+e|0;c[(c[(c[f+168>>2]|0)+(n<<2)>>2]|0)+4+(e<<2)>>2]=o;j=pT(((c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(o<<2)>>2]=j;c[c[(c[f+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2];e=e+1|0}e=0;while(1){if((e|0)>=(c[c[(c[f+168>>2]|0)+(n<<2)>>2]>>2]|0)){break}o=0;while(1){if((o|0)>=9){break}bB(h|0,1,2,b|0)|0;j=(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(n<<2)>>2]|0)+4+(o<<2)>>2]<<2)>>2]|0)+4+(e<<3)|0;if((a[h|0]&128|0)!=0){c[k+(o<<2)>>2]=(d[h+1|0]|0)+1}else{if((d[h|0]|0)==127){a[j|0]=-127}else{if((d[h|0]|0)>0){a[j|0]=(d[h|0]|0)+25&255;a[j+1|0]=c[k+(o<<2)>>2]&255}}}a[j+3|0]=0;a[j+4|0]=0;if((d[h+1|0]|0)==1){a[j+3|0]=13;a[j+4|0]=0}o=o+1|0}e=e+1|0}n=n+1|0}n=0;while(1){if((n|0)>=(c[f+136>>2]|0)){break}c[f+184+(n*12|0)>>2]=128;c[f+184+(n*12|0)+8>>2]=1;n=n+1|0}c[l+6560>>2]=2936;i=g;return 0}function eq(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=a;a=b;b=c;bK(d|0,b+1464|0,0)|0;c=jl(d)|0;do{if((c|0)!=1297370624){if((c|0)==1230254384){break}e=-1;f=e;return f|0}}while(0);bK(d|0,b|0,0)|0;di(d,a,28);e=0;f=e;return f|0}function er(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;j=i;i=i+1480|0;k=j|0;l=j+1472|0;m=f;f=g;g=m|0;bK(f|0,h|0,0)|0;bB(k|0,20,1,f|0)|0;h=0;while(1){if((h|0)>=31){break}bB(k+20+(h*30|0)|0,22,1,f|0)|0;b[k+20+(h*30|0)+22>>1]=jd(f)|0;a[k+20+(h*30|0)+24|0]=ja(f)|0;a[k+20+(h*30|0)+25|0]=ja(f)|0;b[k+20+(h*30|0)+26>>1]=jd(f)|0;b[k+20+(h*30|0)+28>>1]=jd(f)|0;h=h+1|0}a[k+950|0]=ja(f)|0;a[k+951|0]=ja(f)|0;bB(k+952|0,512,1,f|0)|0;c[k+1464>>2]=jl(f)|0;do{if((c[k+1464>>2]|0)==1230254384){dm(m,30632,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n}else{if((c[k+1464>>2]|0)==1297370624){dm(m,28248,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n;break}else{o=-1;p=o;i=j;return p|0}}}while(0);c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];c[g+128>>2]=d[k+950|0]|0;c[g+156>>2]=d[k+950|0]|0;c[g+132>>2]=d[k+951|0]|0;p5(g|0,k|0,20)|0;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}n=pT(64,1)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=n;n=(e[k+20+(h*30|0)+22>>1]|0)<<1;c[(c[g+180>>2]|0)+(h*52|0)+32>>2]=n;c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=((n|0)!=0^1^1)&1;c[(c[g+180>>2]|0)+(h*52|0)+36>>2]=(e[k+20+(h*30|0)+26>>1]|0)<<1;c[(c[g+180>>2]|0)+(h*52|0)+40>>2]=(c[(c[g+180>>2]|0)+(h*52|0)+36>>2]|0)+((e[k+20+(h*30|0)+28>>1]|0)<<1);c[(c[g+180>>2]|0)+(h*52|0)+44>>2]=(e[k+20+(h*30|0)+28>>1]|0|0)>1?2:0;c[c[(c[g+176>>2]|0)+(h*764|0)+756>>2]>>2]=d[k+20+(h*30|0)+25|0]|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+16>>2]=((d[k+20+(h*30|0)+24|0]|0|0)/72|0)<<4;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]=h;h=h+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[g+128>>2]|0)){break}n=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(h<<2)>>2]=n;c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]=64;q=0;while(1){if((q|0)>=(c[g+136>>2]|0)){break}c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(q<<2)>>2]=d[k+952+(h<<2)+q|0]|0;q=q+1|0}a[g+952+h|0]=h&255;h=h+1|0}h=0;while(1){if((h|0)>=(c[g+132>>2]|0)){break}k=pT(524,1)|0;c[(c[g+172>>2]|0)+(h<<2)>>2]=k;c[c[(c[g+172>>2]|0)+(h<<2)>>2]>>2]=64;q=0;while(1){if((q|0)>=(c[c[(c[g+172>>2]|0)+(h<<2)>>2]>>2]|0)){break}k=(c[(c[g+172>>2]|0)+(h<<2)>>2]|0)+4+(q<<3)|0;bB(l|0,1,4,f|0)|0;dj(k,l|0);q=q+1|0}h=h+1|0}q=m+1280|0;c[q>>2]=c[q>>2]|8192;h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}if((c[(c[g+180>>2]|0)+(h*52|0)+32>>2]|0)>4){q=m;l=f;k=(c[g+180>>2]|0)+(h*52|0)|0;gQ(q,l,0,k,0)|0}h=h+1|0}o=0;p=o;i=j;return p|0}function es(){var a=0,b=0,d=0;a=pS(16)|0;if((a|0)==0){b=0;d=b;return d|0}c[a>>2]=a;c[a+4>>2]=a;c[a+8>>2]=4;c[a+12>>2]=0;b=a;d=b;return d|0}function et(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+24|0;g=f|0;h=a;a=b;b=d;d=e;e=h;p7(g|0,0,17);j=bB(g|0,1,c[e+8>>2]|0,b|0)|0;if((j|0)!=(c[e+8>>2]|0)){i=f;return}if((c[e+12>>2]&16|0)!=0){if((aL(g|0,28216,4)|0)==0){j=b;jl(j)|0;j=b;jl(j)|0;j=g|0;k=c[e+8>>2]|0;l=b;bB(j|0,1,k|0,l|0)|0}}if((c[e+12>>2]&1|0)!=0){m=jg(b)|0}else{m=jl(b)|0}l=m;if((c[e+12>>2]&4|0)!=0){l=l+1&-2}if((c[e+12>>2]&8|0)!=0){l=l+3&-4}if((c[e+12>>2]&2|0)!=0){l=l-((c[e+8>>2]|0)+4)|0}eu(h,a,g|0,l,b,d)|0;i=f;return}function eu(a,b,d,e,f,g){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;h=b;b=d;d=e;e=f;f=g;g=a;a=aT(e|0)|0;i=c[g>>2]|0;while(1){if((i|0)==(g|0)){j=2998;break}k=i+(-12|0)|0;if((b|0)!=0){if((aL(b|0,k|0,c[g+8>>2]|0)|0)==0){break}}i=c[i>>2]|0}if((j|0)==2998){l=e;m=a;n=d;o=m+n|0;p=bK(l|0,o|0,0)|0;return 0}b7[c[k+8>>2]&255](h,d,e,f);l=e;m=a;n=d;o=m+n|0;p=bK(l|0,o|0,0)|0;return 0}function ev(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=pS(20)|0;p5(e|0,b|0,5)|0;c[e+8>>2]=d;ew(e+12|0,a|0);return}function ew(a,b){a=a|0;b=b|0;var d=0;d=b;eA(a,c[d+4>>2]|0,d);return}function ex(a,b){a=a|0;b=b|0;c[a+8>>2]=b;return}function ey(a,b){a=a|0;b=b|0;var d=0;d=a+12|0;c[d>>2]=c[d>>2]|b;return}function ez(a,b){a=a|0;b=b|0;var d=0;d=a;a=b;c[a+4>>2]=d;c[d>>2]=a;return}function eA(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=b;b=d;c[b+4>>2]=e;c[e>>2]=b;c[e+4>>2]=a;c[a>>2]=e;return}function eB(a){a=a|0;var b=0,d=0;b=a;a=c[b>>2]|0;while(1){if((a|0)==(b|0)){break}d=a+(-12|0)|0;eC(d+12|0);a=c[a>>2]|0;pU(d)}pU(b);return}function eC(a){a=a|0;var b=0;b=a;ez(c[b+4>>2]|0,c[b>>2]|0);return}function eD(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;a=c;bK(d|0,a+60|0,0)|0;if((jl(d)|0)!=1229795632){e=-1;f=e;return f|0}else{c=d;g=a;bK(c|0,g|0,0)|0;di(d,b,32);e=0;f=e;return f|0}return 0}function eE(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;j=i;i=i+1320|0;k=j|0;l=j+8|0;m=j+840|0;n=j+1224|0;o=j+1288|0;p=f;f=g;g=p|0;q=0;bK(f|0,h|0,0)|0;bB(l|0,32,1,f|0)|0;b[l+32>>1]=jc(f)|0;b[l+34>>1]=jc(f)|0;b[l+36>>1]=jc(f)|0;b[l+38>>1]=jc(f)|0;bB(l+40|0,8,1,f|0)|0;a[l+48|0]=ja(f)|0;a[l+49|0]=ja(f)|0;a[l+50|0]=ja(f)|0;a[l+51|0]=ja(f)|0;bB(l+52|0,8,1,f|0)|0;c[l+60>>2]=jl(f)|0;h=0;while(1){if((h|0)>=32){break}bB(l+64+(h<<4)|0,12,1,f|0)|0;a[l+64+(h<<4)+12|0]=ja(f)|0;a[l+64+(h<<4)+13|0]=ja(f)|0;a[l+64+(h<<4)+14|0]=ja(f)|0;a[l+64+(h<<4)+15|0]=ja(f)|0;h=h+1|0}bB(l+576|0,256,1,f|0)|0;dh(g|0,l|0,32)|0;c[g+156>>2]=e[l+32>>1]|0;c[g+140>>2]=e[l+36>>1]|0;c[g+144>>2]=1024;c[g+128>>2]=e[l+34>>1]|0;if((b[l+38>>1]&1|0)!=0){r=p+1280|0;c[r>>2]=c[r>>2]|4096}c[g+148>>2]=d[l+48|0]|0;c[g+152>>2]=d[l+49|0]|0;dm(p,30544,(r=i,i=i+1|0,i=i+7>>3<<3,c[r>>2]=0,r)|0);i=r;h=0;c[g+136>>2]=0;while(1){if((h|0)>=32){break}if((d[l+64+(h<<4)+12|0]|0)!=0){c[g+136>>2]=h+1;c[g+184+(h*12|0)>>2]=d[l+64+(h<<4)+13|0]|0}h=h+1|0}c[g+132>>2]=ac(c[g+128>>2]|0,c[g+136>>2]|0)|0;r=g+952|0;s=l+576|0;l=c[g+156>>2]|0;p3(r|0,s|0,l)|0;h=0;while(1){if((h|0)>=(c[g+156>>2]|0)){break}if((d[g+952+h|0]|0)==255){l=g+952+h|0;a[l]=(a[l]|0)-1&255}h=h+1|0}c[p+1264>>2]=8363;c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;p7(o|0,0,32);h=0;while(1){if((h|0)>=(c[g+128>>2]|0)){break}l=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(h<<2)>>2]=l;l=((jc(f)|0)&65535)-4|0;s=(jc(f)|0)&65535;c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]=s;s=0;while(1){if((s|0)>=(c[g+136>>2]|0)){break}r=(ac(h,c[g+136>>2]|0)|0)+s|0;c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(s<<2)>>2]=r;t=pT(((c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(r<<2)>>2]=t;c[c[(c[g+172>>2]|0)+(r<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2];s=s+1|0}s=0;while(1){r=l-1|0;l=r;if((r|0)<0){break}r=ja(f)|0;if((r&255|0)==0){s=s+1|0;continue}t=r&31;if((t|0)>=(c[g+136>>2]|0)){u=k}else{u=(c[(c[p+172>>2]|0)+(c[(c[(c[p+168>>2]|0)+(h<<2)>>2]|0)+4+(t<<2)>>2]<<2)>>2]|0)+4+(s<<3)|0}q=u;if((r&32|0)!=0){v=ja(f)|0;w=v&255;if((w|0)==255|(w|0)==160){v=-127}else{v=(((v&240)>>4)*12|0)+13+(v&15)&255}a[q|0]=v;a[q+1|0]=ja(f)|0;l=l-2|0}if((r&128|0)!=0){a[q+3|0]=ja(f)|0;a[q+4|0]=ja(f)|0;eF(t,q+3|0,q+4|0,o|0);l=l-2|0}if((r&64|0)!=0){a[q+5|0]=ja(f)|0;a[q+6|0]=ja(f)|0;eF(t,q+5|0,q+6|0,o|0);l=l-2|0}}h=h+1|0}c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}h=0;o=0;while(1){if((h|0)>=(c[g+140>>2]|0)){x=3127;break}bB(m|0,32,1,f|0)|0;bB(m+32|0,120,1,f|0)|0;bB(m+152|0,8,1,f|0)|0;q=0;while(1){if((q|0)>=32){break}b[m+160+(q<<1)>>1]=jc(f)|0;q=q+1|0}q=0;while(1){if((q|0)>=32){break}b[m+224+(q<<1)>>1]=jc(f)|0;q=q+1|0}q=0;while(1){if((q|0)>=32){break}b[m+288+(q<<1)>>1]=jc(f)|0;q=q+1|0}q=0;while(1){if((q|0)>=3){break}a[m+352+(q<<3)|0]=ja(f)|0;a[m+352+(q<<3)+1|0]=ja(f)|0;a[m+352+(q<<3)+2|0]=ja(f)|0;a[m+352+(q<<3)+3|0]=ja(f)|0;a[m+352+(q<<3)+4|0]=ja(f)|0;bB(m+352+(q<<3)+5|0,3,1,f|0)|0;q=q+1|0}b[m+376>>1]=jc(f)|0;b[m+378>>1]=jc(f)|0;c[m+380>>2]=jl(f)|0;if((c[m+380>>2]|0)!=1229533488){x=3102;break}if((b[m+378>>1]|0)!=0){u=pT(64,e[m+378>>1]|0)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=u}c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=e[m+378>>1]|0;hy(m|0)|0;p5((c[g+176>>2]|0)+(h*764|0)|0,m|0,24)|0;q=0;while(1){if((q|0)>=108){break}a[(c[g+176>>2]|0)+(h*764|0)+512+(q+12<<1)|0]=a[m+32+q|0]|0;q=q+1|0}c[(c[g+176>>2]|0)+(h*764|0)+48>>2]=d[m+352|0]|0;c[(c[g+176>>2]|0)+(h*764|0)+56>>2]=d[m+353|0]|0;c[(c[g+176>>2]|0)+(h*764|0)+64>>2]=d[m+354|0]|0;c[(c[g+176>>2]|0)+(h*764|0)+68>>2]=d[m+355|0]|0;c[(c[g+176>>2]|0)+(h*764|0)+44>>2]=(a[m+356|0]&1|0)!=0?1:0;u=(c[g+176>>2]|0)+(h*764|0)+44|0;c[u>>2]=c[u>>2]|((a[m+356|0]&2|0)!=0?2:0);u=(c[g+176>>2]|0)+(h*764|0)+44|0;c[u>>2]=c[u>>2]|((a[m+356|0]&4|0)!=0?4:0);q=0;while(1){if((q|0)>=(c[(c[g+176>>2]|0)+(h*764|0)+48>>2]|0)){break}b[(c[g+176>>2]|0)+(h*764|0)+72+(q<<1<<1)>>1]=b[m+160+(q<<1<<1)>>1]|0;b[(c[g+176>>2]|0)+(h*764|0)+72+((q<<1)+1<<1)>>1]=b[m+160+((q<<1)+1<<1)>>1]|0;q=q+1|0}q=0;while(1){if((q|0)>=(e[m+378>>1]|0)){break}bB(n|0,13,1,f|0)|0;bB(n+13|0,3,1,f|0)|0;c[n+16>>2]=jg(f)|0;c[n+20>>2]=jg(f)|0;c[n+24>>2]=jg(f)|0;c[n+28>>2]=jg(f)|0;a[n+32|0]=ja(f)|0;a[n+33|0]=ja(f)|0;bB(n+34|0,14,1,f|0)|0;a[n+48|0]=ja(f)|0;bB(n+49|0,5,1,f|0)|0;b[n+54>>1]=jc(f)|0;c[n+56>>2]=jg(f)|0;c[n+60>>2]=jl(f)|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)+40>>2]=o;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)>>2]=d[n+32|0]|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)+8>>2]=d[n+33|0]|0;c[(c[g+180>>2]|0)+(o*52|0)+32>>2]=c[n+16>>2];c[(c[g+180>>2]|0)+(o*52|0)+36>>2]=c[n+20>>2];c[(c[g+180>>2]|0)+(o*52|0)+40>>2]=c[n+24>>2];c[(c[g+180>>2]|0)+(o*52|0)+44>>2]=(a[n+48|0]&1|0)!=0?2:0;if((a[n+48|0]&4|0)!=0){u=(c[g+180>>2]|0)+(o*52|0)+44|0;c[u>>2]=c[u>>2]|1;u=(c[g+180>>2]|0)+(o*52|0)+32|0;c[u>>2]=c[u>>2]>>1;u=(c[g+180>>2]|0)+(o*52|0)+36|0;c[u>>2]=c[u>>2]>>1;u=(c[g+180>>2]|0)+(o*52|0)+40|0;c[u>>2]=c[u>>2]>>1}hD(c[n+28>>2]|0,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)+12|0,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)+16|0);if((c[(c[g+180>>2]|0)+(o*52|0)+32>>2]|0)!=0){u=p;k=f;l=(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+(q<<6)+40>>2]|0)*52|0)|0;gQ(u,k,0,l,0)|0}q=q+1|0;o=o+1|0}h=h+1|0}if((x|0)==3102){y=-2;z=y;i=j;return z|0}else if((x|0)==3127){c[g+144>>2]=o;c[g+180>>2]=pV(c[g+180>>2]|0,(c[g+144>>2]|0)*52|0)|0;g=p+1280|0;c[g>>2]=c[g>>2]|66081;c[p+1284>>2]=2;y=0;z=y;i=j;return z|0}return 0}function eF(b,c,e,f){b=b|0;c=c|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=b;b=c;c=e;e=f;f=(a[c]&240)>>4&255;h=d[c]&15;i=a[8816+(d[b]|0)|0]|0;a[b]=i;j=i&255;if((j|0)==254){a[b]=1;if((d[c]|0)<48){a[c]=(d[c]>>2&15|224)&255}else{a[c]=(d[c]>>4&15|240)&255}return}else if((j|0)==255){a[c]=0;a[b]=0;return}else if((j|0)==253){a[b]=2;if((d[c]|0)<48){a[c]=(d[c]>>2&15|224)&255}else{a[c]=(d[c]>>4&15|240)&255}return}else if((j|0)==14){i=f&255;if((i|0)==8){a[c]=(h&255|112)&255}else if((i|0)==10){a[c]=(h&255|96)&255}else if((i|0)==11){a[c]=(h&255|224)&255}else if((i|0)==12){if((h&255|0)==0){a[c]=0;a[b]=0}}else if((i|0)==5){a[c]=(h&255|64)&255}else if((i|0)==1|(i|0)==2|(i|0)==4|(i|0)==6|(i|0)==7|(i|0)==9|(i|0)==14|(i|0)==15){a[b]=0;a[c]=0}else if((i|0)==3){a[c]=(h&255|48)&255}return}else if((j|0)==0){if((a[c]|0)!=0){a[e+g|0]=a[c]|0}else{a[c]=a[e+g|0]|0}return}else{return}}function eG(c,f,g){c=c|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0;h=i;i=i+1088|0;j=h|0;k=c;c=f;f=g;g=0;bB(j|0,20,1,k|0)|0;l=0;while(1){if((l|0)>=31){m=3180;break}if((bB(j+20+(l*30|0)|0,1,20,k|0)|0)>>>0<20){m=3166;break}b[j+20+(l*30|0)+20>>1]=jd(k)|0;b[j+20+(l*30|0)+22>>1]=jd(k)|0;a[j+20+(l*30|0)+24|0]=ja(k)|0;a[j+20+(l*30|0)+25|0]=ja(k)|0;b[j+20+(l*30|0)+26>>1]=jd(k)|0;b[j+20+(l*30|0)+28>>1]=jd(k)|0;g=g+((e[j+20+(l*30|0)+22>>1]|0)<<1)|0;if((df(j+20+(l*30|0)|0,20)|0)<0){m=3168;break}if((d[j+20+(l*30|0)+25|0]|0|0)>64){m=3170;break}if((e[j+20+(l*30|0)+22>>1]|0|0)>32768){m=3172;break}if((e[j+20+(l*30|0)+26>>1]|0|0)>(e[j+20+(l*30|0)+22>>1]|0|0)){m=3174;break}if((e[j+20+(l*30|0)+22>>1]|0|0)!=0){if((e[j+20+(l*30|0)+28>>1]|0|0)>((e[j+20+(l*30|0)+22>>1]|0)<<1|0)){m=3177;break}}l=l+1|0}if((m|0)==3166){n=-1;o=n;i=h;return o|0}else if((m|0)==3168){n=-1;o=n;i=h;return o|0}else if((m|0)==3170){n=-1;o=n;i=h;return o|0}else if((m|0)==3172){n=-1;o=n;i=h;return o|0}else if((m|0)==3174){n=-1;o=n;i=h;return o|0}else if((m|0)==3177){n=-1;o=n;i=h;return o|0}else if((m|0)==3180){if((g|0)<8){n=-1;o=n;i=h;return o|0}a[j+950|0]=ja(k)|0;a[j+951|0]=ja(k)|0;bB(j+952|0,128,1,k|0)|0;bB(j+1080|0,4,1,k|0)|0;if((d[j+951|0]|0|0)>1){n=-1;o=n;i=h;return o|0}if((d[j+1083|0]|0|0)!=60){n=-1;o=n;i=h;return o|0}if((d[j+950|0]|0|0)>127){n=-1;o=n;i=h;return o|0}l=0;g=0;while(1){if((l|0)>=(d[j+950|0]|0|0)){break}if((d[j+952+l|0]|0|0)>(g|0)){g=d[j+952+l|0]|0}l=l+1|0}g=g+1|0;do{if((g|0)<=127){if((d[j+950|0]|0|0)==0){break}if((d[j+950|0]|0|0)>127){break}l=k;m=f|0;bK(l|0,m|0,0)|0;di(k,c,20);n=0;o=n;i=h;return o|0}}while(0);n=-1;o=n;i=h;return o|0}return 0}function eH(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;j=i;i=i+1096|0;k=j|0;l=j+1088|0;m=f;f=g;g=m|0;n=21;bK(f|0,h|0,0)|0;c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];h=0;bB(k|0,20,1,f|0)|0;o=0;while(1){if((o|0)>=31){break}bB(k+20+(o*30|0)|0,20,1,f|0)|0;b[k+20+(o*30|0)+20>>1]=jd(f)|0;b[k+20+(o*30|0)+22>>1]=jd(f)|0;a[k+20+(o*30|0)+24|0]=ja(f)|0;a[k+20+(o*30|0)+25|0]=ja(f)|0;b[k+20+(o*30|0)+26>>1]=jd(f)|0;b[k+20+(o*30|0)+28>>1]=jd(f)|0;h=h+((e[k+20+(o*30|0)+22>>1]|0)<<1)|0;o=o+1|0}a[k+950|0]=ja(f)|0;a[k+951|0]=ja(f)|0;bB(k+952|0,128,1,f|0)|0;bB(k+1080|0,4,1,f|0)|0;c[g+156>>2]=d[k+950|0]|0;h=g+952|0;p=k+952|0;q=c[g+156>>2]|0;p3(h|0,p|0,q)|0;o=0;while(1){if((o|0)>=(c[g+156>>2]|0)){break}if((d[g+952+o|0]|0|0)>(c[g+128>>2]|0)){c[g+128>>2]=d[g+952+o|0]|0}o=o+1|0}q=g+128|0;c[q>>2]=(c[q>>2]|0)+1;c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;p5(g|0,k|0,20)|0;dm(m,30496,(q=i,i=i+1|0,i=i+7>>3<<3,c[q>>2]=0,q)|0);i=q;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}o=0;while(1){if((o|0)>=(c[g+140>>2]|0)){break}q=pT(64,1)|0;c[(c[g+176>>2]|0)+(o*764|0)+756>>2]=q;c[(c[g+180>>2]|0)+(o*52|0)+32>>2]=(e[k+20+(o*30|0)+22>>1]|0)<<1;c[(c[g+180>>2]|0)+(o*52|0)+40>>2]=(c[(c[g+180>>2]|0)+(o*52|0)+36>>2]|0)+((e[k+20+(o*30|0)+28>>1]|0)<<1);c[(c[g+180>>2]|0)+(o*52|0)+44>>2]=(e[k+20+(o*30|0)+28>>1]|0|0)>1?2:0;c[(c[(c[g+176>>2]|0)+(o*764|0)+756>>2]|0)+16>>2]=0;c[c[(c[g+176>>2]|0)+(o*764|0)+756>>2]>>2]=d[k+20+(o*30|0)+25|0]|0;c[(c[(c[g+176>>2]|0)+(o*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]=o;c[(c[g+176>>2]|0)+(o*764|0)+36>>2]=((c[(c[g+180>>2]|0)+(o*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[g+176>>2]|0)+(o*764|0)+40>>2]=4095;dh((c[g+176>>2]|0)+(o*764|0)|0,k+20+(o*30|0)|0,20)|0;o=o+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;o=0;while(1){if((o|0)>=(c[g+128>>2]|0)){break}k=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(o<<2)>>2]=k;c[c[(c[g+168>>2]|0)+(o<<2)>>2]>>2]=64;k=0;while(1){if((k|0)>=(c[g+136>>2]|0)){break}q=(ac(o,c[g+136>>2]|0)|0)+k|0;c[(c[(c[g+168>>2]|0)+(o<<2)>>2]|0)+4+(k<<2)>>2]=q;p=pT(((c[c[(c[g+168>>2]|0)+(o<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(q<<2)>>2]=p;c[c[(c[g+172>>2]|0)+(q<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(o<<2)>>2]>>2];k=k+1|0}k=0;while(1){if((k|0)>=256){break}q=(c[(c[m+172>>2]|0)+(c[(c[(c[m+168>>2]|0)+(o<<2)>>2]|0)+4+((k&3)<<2)>>2]<<2)>>2]|0)+4+(k>>2<<3)|0;bB(l|0,1,3,f|0)|0;a[q|0]=(d[l|0]|0)&63;do{if((d[q|0]|0|0)!=0){if((d[q|0]|0|0)==63){r=3259;break}p=q|0;a[p]=(d[p]|0)+(n+12)&255}else{r=3259}}while(0);if((r|0)==3259){r=0;a[q|0]=0}a[q+1|0]=((a[l|0]&64)>>2|(a[l+1|0]&240)>>4)&255;a[q+3|0]=(d[l+1|0]|0)&15;a[q+4|0]=a[l+2|0]|0;dg(q);if((d[q+3|0]|0|0)==13){a[q+4|0]=(((d[q+4|0]|0|0)/10|0)<<4|((d[q+4|0]|0|0)%10|0))&255}k=k+1|0}o=o+1|0}l=m+1280|0;c[l>>2]=c[l>>2]|8192;o=0;while(1){if((o|0)>=(c[g+144>>2]|0)){break}if((c[(c[g+180>>2]|0)+(o*52|0)+32>>2]|0)!=0){l=m;r=f;n=(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(l,r,0,n,0)|0}o=o+1|0}i=j;return 0}function eI(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1229803597){d=-1;e=d;return e|0}else{di(c,b,26);d=0;e=d;return e|0}return 0}function eJ(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0,R=0,S=0,T=0,U=0,V=0;j=i;i=i+3040|0;k=j|0;l=j+8|0;m=j+520|0;n=j+712|0;o=j+1272|0;p=j+1576|0;q=j+1656|0;r=j+1768|0;s=j+1832|0;t=j+2312|0;u=j+2800|0;v=j+2840|0;w=j+2904|0;x=j+2968|0;y=j+2984|0;z=j+3032|0;A=f;f=g;g=h;h=A|0;B=x;c[B>>2]=c[1812];c[B+4>>2]=c[1813];c[B+8>>2]=c[1814];bK(f|0,g|0,0)|0;jl(f)|0;bB(m+4|0,26,1,f|0)|0;a[m+30|0]=ja(f)|0;a[m+31|0]=ja(f)|0;b[m+32>>1]=jc(f)|0;b[m+34>>1]=jc(f)|0;b[m+36>>1]=jc(f)|0;b[m+38>>1]=jc(f)|0;b[m+40>>1]=jc(f)|0;b[m+42>>1]=jc(f)|0;b[m+44>>1]=jc(f)|0;b[m+46>>1]=jc(f)|0;a[m+48|0]=ja(f)|0;a[m+49|0]=ja(f)|0;a[m+50|0]=ja(f)|0;a[m+51|0]=ja(f)|0;a[m+52|0]=ja(f)|0;a[m+53|0]=ja(f)|0;b[m+54>>1]=jc(f)|0;c[m+56>>2]=jg(f)|0;c[m+60>>2]=jg(f)|0;bB(m+64|0,64,1,f|0)|0;bB(m+128|0,64,1,f|0)|0;p5(h|0,m+4|0,64)|0;c[h+156>>2]=e[m+32>>1]|0;c[h+140>>2]=e[m+34>>1]|0;c[h+144>>2]=e[m+36>>1]|0;c[h+128>>2]=e[m+38>>1]|0;if((c[h+140>>2]|0)!=0){C=pT(4,c[h+140>>2]|0)|0}else{C=0}B=C;C=pT(4,c[h+144>>2]|0)|0;D=pT(4,c[h+128>>2]|0)|0;c[h+148>>2]=d[m+50|0]|0;c[h+152>>2]=d[m+51|0]|0;if((b[m+44>>1]&8|0)!=0){E=A+1280|0;c[E>>2]=c[E>>2]|4096}do{if((b[m+44>>1]&4|0)!=0){if((e[m+42>>1]|0)<512){break}E=A+1280|0;c[E>>2]=c[E>>2]|16384}}while(0);c[h+136>>2]=64;E=0;while(1){if((E|0)>=64){break}if((d[m+64+E|0]|0)==100){a[m+64+E|0]=32}if((a[m+64+E|0]&128|0)!=0){a[m+128+E|0]=0;F=h+184+(E*12|0)+8|0;c[F>>2]=c[F>>2]|2}if((b[m+44>>1]&1|0)!=0){c[h+184+(E*12|0)>>2]=d[m+64+E|0]<<7>>5;if((c[h+184+(E*12|0)>>2]|0)>255){c[h+184+(E*12|0)>>2]=255}}else{c[h+184+(E*12|0)>>2]=128}c[h+184+(E*12|0)+4>>2]=d[m+128+E|0]|0;E=E+1|0}bB(h+952|0,1,c[h+156>>2]|0,f|0)|0;F=(b[m+44>>1]&16|0)!=0?0:1;E=0;while(1){if((E|0)>=((c[h+156>>2]|0)-1|0)){break}if((d[h+952+E|0]|0)==254){qb(h+952+E|0,h+952+(E+1)|0,(c[h+156>>2]|0)-E-1|0);G=h+156|0;c[G>>2]=(c[G>>2]|0)-1}E=E+1|0}E=0;while(1){if((E|0)>=(c[h+140>>2]|0)){break}c[B+(E<<2)>>2]=jg(f)|0;E=E+1|0}E=0;while(1){if((E|0)>=(c[h+144>>2]|0)){break}c[C+(E<<2)>>2]=jg(f)|0;E=E+1|0}E=0;while(1){if((E|0)>=(c[h+128>>2]|0)){break}c[D+(E<<2)>>2]=jg(f)|0;E=E+1|0}c[A+1264>>2]=8363;G=e[m+40>>1]>>8;if((G|0)==1|(G|0)==2){do{if((e[m+42>>1]|0)==512){if((e[m+40>>1]|0)!=532){H=3341;break}if((e[m+44>>1]|0)!=9){H=3341;break}if((e[m+46>>1]|0)!=0){H=3341;break}if((d[m+31|0]|0)!=0){H=3341;break}if((d[m+30|0]|0)!=0){H=3341;break}if((e[m+34>>1]|0)!=0){H=3341;break}if(((e[m+38>>1]|0)+1|0)!=(e[m+32>>1]|0)){H=3341;break}if((d[m+48|0]|0)!=128){H=3341;break}if((d[m+49|0]|0)!=100){H=3341;break}if((d[m+50|0]|0)!=1){H=3341;break}if((d[m+52|0]|0)!=128){H=3341;break}if((d[m+53|0]|0)!=0){H=3341;break}if((e[m+54>>1]|0)!=0){H=3341;break}if((c[m+56>>2]|0)!=0){H=3341;break}if((c[m+60>>2]|0)!=0){H=3341;break}I=u|0;qc(I|0,28224)|0}else{H=3341}}while(0);if((H|0)==3341){do{if((e[m+42>>1]|0)==512){if((e[m+40>>1]|0)!=535){H=3344;break}I=u|0;qc(I|0,26944)|0;b[m+42>>1]=((b[m+44>>1]&4|0)!=0?532:256)&65535}else{H=3344}}while(0);if((H|0)==3344){if((e[m+40>>1]|0)==534){I=u|0;qc(I|0,25888)|0}else{if((e[m+40>>1]|0)==535){I=u|0;qc(I|0,24880)|0}else{do{if((e[m+40>>1]|0)==532){if((p6(m+60|0,24112,4)|0)!=0){H=3351;break}I=u|0;qc(I|0,23696)|0}else{H=3351}}while(0);if((H|0)==3351){I=u|0;J=(b[m+40>>1]&3840)>>8;K=b[m+40>>1]&255;aM(I|0,40,23152,(L=i,i=i+16|0,c[L>>2]=J,c[L+8>>2]=K,L)|0)|0;i=L}}}}}}else if((G|0)==0){qc(u|0,30488)|0}else if((G|0)==8|(G|0)==127){if((e[m+40>>1]|0)==2184){G=u|0;qc(G|0,22640)|0}else{if((e[m+40>>1]|0)==32767){G=u|0;qc(G|0,31232)|0}else{G=u|0;K=e[m+40>>1]|0;aM(G|0,40,30936,(L=i,i=i+8|0,c[L>>2]=K,L)|0)|0;i=L}}}else{K=e[m+40>>1]>>12;if((K|0)==1){G=e[m+40>>1]&4095;if((G&65535|0)>80){c[z>>2]=(((G&65535)-80|0)*86400|0)+1254355200;if((a_(z|0,y|0)|0)!=0){z=u|0;J=(c[y+20>>2]|0)+1900|0;I=(c[y+16>>2]|0)+1|0;M=c[y+12>>2]|0;aM(z|0,40,30600,(L=i,i=i+24|0,c[L>>2]=J,c[L+8>>2]=I,c[L+16>>2]=M,L)|0)|0;i=L}}else{M=u|0;I=G&65535;aM(M|0,40,30296,(L=i,i=i+8|0,c[L>>2]=I,L)|0)|0;i=L}}else if((K|0)==5){I=b[m+40>>1]&255;aM(u|0,40,29968,(L=i,i=i+16|0,c[L>>2]=(b[m+40>>1]&3840)>>8,c[L+8>>2]=I,L)|0)|0;i=L;if((p6(m+60|0,29648,4)|0)!=0){I=u|0;bI(I|0,29248,40)|0}}else if((K|0)==6){K=b[m+40>>1]&255;aM(u|0,40,29152,(L=i,i=i+16|0,c[L>>2]=(b[m+40>>1]&3840)>>8,c[L+8>>2]=K,L)|0)|0;i=L}else{aM(u|0,40,30936,(L=i,i=i+8|0,c[L>>2]=e[m+40>>1]|0,L)|0)|0;i=L}}K=e[m+42>>1]>>8;I=b[m+42>>1]&255;dm(A,28840,(L=i,i=i+24|0,c[L>>2]=u,c[L+8>>2]=K,c[L+16>>2]=I,L)|0);i=L;if((~e[m+44>>1]&4|0)!=0){c[h+140>>2]=c[h+144>>2]}if((b[m+46>>1]&1|0)!=0){L=pS((e[m+54>>1]|0)+1|0)|0;c[A+1220>>2]=L;if((L|0)==0){N=-1;O=N;i=j;return O|0}E=aT(f|0)|0;L=f;I=g+(c[m+56>>2]|0)|0;bK(L|0,I|0,0)|0;P=0;while(1){if((P|0)>=(e[m+54>>1]|0)){break}Q=ja(f)|0;if((Q&255|0)==13){Q=10}if((Q&255|0)<32){H=3398}else{if((Q&255|0)>127){H=3398}}do{if((H|0)==3398){H=0;if((Q&255|0)==10){break}if((Q&255|0)==9){break}Q=46}}while(0);a[(c[A+1220>>2]|0)+P|0]=Q;P=P+1|0}a[(c[A+1220>>2]|0)+P|0]=0;bK(f|0,E|0,0)|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}E=0;while(1){if((E|0)>=(c[h+140>>2]|0)){break}I=(c[h+176>>2]|0)+(E*764|0)|0;do{if((b[m+44>>1]&4|0)!=0){if((e[m+42>>1]|0)<512){H=3488;break}L=f;K=g+(c[B+(E<<2)>>2]|0)|0;bK(L|0,K|0,0)|0;c[o>>2]=jl(f)|0;K=o+4|0;L=f;bB(K|0,12,1,L|0)|0;a[o+16|0]=ja(f)|0;a[o+17|0]=ja(f)|0;a[o+18|0]=ja(f)|0;a[o+19|0]=ja(f)|0;b[o+20>>1]=jc(f)|0;a[o+22|0]=ja(f)|0;a[o+23|0]=ja(f)|0;a[o+24|0]=ja(f)|0;a[o+25|0]=ja(f)|0;a[o+26|0]=ja(f)|0;a[o+27|0]=ja(f)|0;b[o+28>>1]=jc(f)|0;a[o+30|0]=ja(f)|0;a[o+31|0]=ja(f)|0;L=o+32|0;K=f;bB(L|0,26,1,K|0)|0;eK(o+32|0,26);a[o+58|0]=ja(f)|0;a[o+59|0]=ja(f)|0;a[o+60|0]=ja(f)|0;a[o+61|0]=ja(f)|0;b[o+62>>1]=jc(f)|0;K=o+64|0;L=f;bB(K|0,240,1,L|0)|0;L=I|0;K=o+32|0;dh(L,K,25)|0;c[I+40>>2]=e[o+20>>1]<<5;a[q|0]=ja(f)|0;a[q+1|0]=ja(f)|0;a[q+2|0]=ja(f)|0;a[q+3|0]=ja(f)|0;a[q+4|0]=ja(f)|0;a[q+5|0]=ja(f)|0;P=0;while(1){if((P|0)>=25){break}a[q+6+(P<<2)|0]=ja(f)|0;b[q+6+(P<<2)+2>>1]=jc(f)|0;P=P+1|0}a[q+106|0]=ja(f)|0;c[I+44>>2]=(a[q|0]&1|0)!=0?1:0;K=I+44|0;c[K>>2]=c[K>>2]|((a[q|0]&2|0)!=0?4:0);K=I+44|0;c[K>>2]=c[K>>2]|((a[q|0]&4|0)!=0?18:0);K=I+44|0;c[K>>2]=c[K>>2]|((a[q|0]&8|0)!=0?32:0);c[I+48>>2]=d[q+1|0]|0;c[I+56>>2]=d[q+4|0]|0;c[I+60>>2]=d[q+5|0]|0;c[I+64>>2]=d[q+2|0]|0;c[I+68>>2]=d[q+3|0]|0;P=0;while(1){if((P|0)>=(d[q+1|0]|0)){break}b[I+72+(P<<1<<1)>>1]=b[q+6+(P<<2)+2>>1]|0;b[I+72+((P<<1)+1<<1)>>1]=a[q+6+(P<<2)|0]|0;P=P+1|0}a[q|0]=ja(f)|0;a[q+1|0]=ja(f)|0;a[q+2|0]=ja(f)|0;a[q+3|0]=ja(f)|0;a[q+4|0]=ja(f)|0;a[q+5|0]=ja(f)|0;P=0;while(1){if((P|0)>=25){break}a[q+6+(P<<2)|0]=ja(f)|0;b[q+6+(P<<2)+2>>1]=jc(f)|0;P=P+1|0}a[q+106|0]=ja(f)|0;c[I+200>>2]=(a[q|0]&1|0)!=0?1:0;K=I+200|0;c[K>>2]=c[K>>2]|((a[q|0]&2|0)!=0?4:0);K=I+200|0;c[K>>2]=c[K>>2]|((a[q|0]&4|0)!=0?18:0);K=I+200|0;c[K>>2]=c[K>>2]|((a[q|0]&8|0)!=0?32:0);c[I+204>>2]=d[q+1|0]|0;c[I+212>>2]=d[q+4|0]|0;c[I+216>>2]=d[q+5|0]|0;c[I+220>>2]=d[q+2|0]|0;c[I+224>>2]=d[q+3|0]|0;P=0;while(1){if((P|0)>=(d[q+1|0]|0)){break}b[I+228+(P<<1<<1)>>1]=b[q+6+(P<<2)+2>>1]|0;b[I+228+((P<<1)+1<<1)>>1]=a[q+6+(P<<2)|0]|0;P=P+1|0}a[q|0]=ja(f)|0;a[q+1|0]=ja(f)|0;a[q+2|0]=ja(f)|0;a[q+3|0]=ja(f)|0;a[q+4|0]=ja(f)|0;a[q+5|0]=ja(f)|0;P=0;while(1){if((P|0)>=25){break}a[q+6+(P<<2)|0]=ja(f)|0;b[q+6+(P<<2)+2>>1]=jc(f)|0;P=P+1|0}a[q+106|0]=ja(f)|0;c[I+356>>2]=(a[q|0]&1|0)!=0?1:0;K=I+356|0;c[K>>2]=c[K>>2]|((a[q|0]&2|0)!=0?4:0);K=I+356|0;c[K>>2]=c[K>>2]|((a[q|0]&4|0)!=0?18:0);K=I+356|0;c[K>>2]=c[K>>2]|((a[q|0]&8|0)!=0?32:0);c[I+360>>2]=d[q+1|0]|0;c[I+368>>2]=d[q+4|0]|0;c[I+372>>2]=d[q+5|0]|0;c[I+376>>2]=d[q+2|0]|0;c[I+380>>2]=d[q+3|0]|0;P=0;while(1){if((P|0)>=(d[q+1|0]|0)){break}b[I+384+(P<<1<<1)>>1]=b[q+6+(P<<2)+2>>1]|0;b[I+384+((P<<1)+1<<1)>>1]=a[q+6+(P<<2)|0]|0;P=P+1|0}if((c[I+200>>2]&1|0)!=0){P=0;while(1){if((P|0)>=(c[I+204>>2]|0)){break}K=I+228+((P<<1)+1<<1)|0;b[K>>1]=(b[K>>1]|0)+32&65535;P=P+1|0}}do{if((c[I+44>>2]&1|0)!=0){if((c[I+48>>2]|0)!=0){break}c[I+48>>2]=1}}while(0);do{if((c[I+200>>2]&1|0)!=0){if((c[I+204>>2]|0)!=0){break}c[I+204>>2]=1}}while(0);do{if((c[I+356>>2]&1|0)!=0){if((c[I+360>>2]|0)!=0){break}c[I+360>>2]=1}}while(0);if((a[q|0]&128|0)!=0){K=I+356|0;c[K>>2]=c[K>>2]|8;P=0;while(1){if((P|0)>=(d[q+1|0]|0)){break}K=I+384+((P<<1)+1<<1)|0;b[K>>1]=(b[K>>1]|0)+32&65535;K=I+384+((P<<1)+1<<1)|0;b[K>>1]=b[K>>1]<<2&65535;P=P+1|0}}else{P=0;while(1){if((P|0)>=(d[q+1|0]|0)){break}K=I+384+((P<<1)+1<<1)|0;b[K>>1]=((b[K>>1]|0)*50|0)&65535;P=P+1|0}}P=0;while(1){if((P|0)>=120){break}c[s+(P<<2)>>2]=-1;P=P+1|0}P=0;R=0;while(1){if((P|0)>=120){break}S=(d[o+64+((P<<1)+1)|0]|0)-1|0;if((S|0)<0){a[I+512+(P<<1)|0]=-1;a[I+512+(P<<1)+1|0]=0}else{if((c[s+(S<<2)>>2]|0)==-1){c[s+(S<<2)>>2]=R;c[t+(R<<2)>>2]=S;R=R+1|0}a[I+512+(P<<1)|0]=c[s+(S<<2)>>2]&255;a[I+512+(P<<1)+1|0]=(d[o+64+(P<<1)|0]|0)-P&255}P=P+1|0}c[I+36>>2]=R;c[I+32>>2]=d[o+24|0]>>1;if((R|0)!=0){c[I+756>>2]=pT(64,R)|0;P=0;while(1){if((P|0)>=(R|0)){break}c[(c[I+756>>2]|0)+(P<<6)+40>>2]=c[t+(P<<2)>>2];c[(c[I+756>>2]|0)+(P<<6)+44>>2]=d[o+17|0]|0;c[(c[I+756>>2]|0)+(P<<6)+48>>2]=d[o+18|0]|0;c[(c[I+756>>2]|0)+(P<<6)+52>>2]=c[x+((a[o+19|0]&3)<<2)>>2];if((a[o+25|0]&128|0)!=0){T=128}else{T=d[o+25|0]<<2}c[(c[I+756>>2]|0)+(P<<6)+8>>2]=T;c[(c[I+756>>2]|0)+(P<<6)+56>>2]=d[o+58|0]|0;c[(c[I+756>>2]|0)+(P<<6)+60>>2]=d[o+59|0]|0;P=P+1|0}}}else{H=3488}}while(0);if((H|0)==3488){H=0;if((b[m+44>>1]&4|0)!=0){K=f;L=g+(c[B+(E<<2)>>2]|0)|0;bK(K|0,L|0,0)|0;c[n>>2]=jl(f)|0;L=n+4|0;K=f;bB(L|0,12,1,K|0)|0;a[n+16|0]=ja(f)|0;a[n+17|0]=ja(f)|0;a[n+18|0]=ja(f)|0;a[n+19|0]=ja(f)|0;a[n+20|0]=ja(f)|0;a[n+21|0]=ja(f)|0;b[n+22>>1]=jc(f)|0;b[n+24>>1]=jc(f)|0;a[n+26|0]=ja(f)|0;a[n+27|0]=ja(f)|0;b[n+28>>1]=jc(f)|0;a[n+30|0]=ja(f)|0;a[n+31|0]=ja(f)|0;K=n+32|0;L=f;bB(K|0,26,1,L|0)|0;eK(n+32|0,26);L=n+58|0;K=f;bB(L|0,6,1,K|0)|0;K=n+64|0;L=f;bB(K|0,240,1,L|0)|0;L=n+304|0;K=f;bB(L|0,200,1,K|0)|0;K=n+504|0;L=f;bB(K|0,50,1,L|0)|0;L=I|0;K=n+32|0;dh(L,K,25)|0;c[I+40>>2]=e[n+24>>1]<<6;c[I+44>>2]=0;if((a[n+17|0]&1|0)!=0){K=I+44|0;c[K>>2]=c[K>>2]|1}if((a[n+17|0]&2|0)!=0){K=I+44|0;c[K>>2]=c[K>>2]|4}if((a[n+17|0]&4|0)!=0){K=I+44|0;c[K>>2]=c[K>>2]|18}if((a[n+17|0]&8|0)!=0){K=I+44|0;c[K>>2]=c[K>>2]|34}c[I+64>>2]=d[n+18|0]|0;c[I+68>>2]=d[n+19|0]|0;c[I+56>>2]=d[n+20|0]|0;c[I+60>>2]=d[n+21|0]|0;R=0;while(1){if((d[n+504+(R<<1)|0]|0)==255){break}R=R+1|0}c[I+48>>2]=R;while(1){K=R;R=K-1|0;if((K|0)==0){break}b[I+72+(R<<1<<1)>>1]=d[n+504+(R<<1)|0]|0;b[I+72+((R<<1)+1<<1)>>1]=d[n+504+((R<<1)+1)|0]|0}P=0;while(1){if((P|0)>=120){break}c[s+(P<<2)>>2]=-1;P=P+1|0}P=0;R=0;while(1){if((P|0)>=121){break}S=(d[n+64+((P<<1)+25)|0]|0)-1|0;if((S|0)<0){a[I+512+(P<<1)|0]=-1;a[I+512+(P<<1)+1|0]=0}else{if((c[s+(S<<2)>>2]|0)==-1){c[s+(S<<2)>>2]=R;c[t+(R<<2)>>2]=S;R=R+1|0}a[I+512+(P<<1)|0]=c[s+(S<<2)>>2]&255;a[I+512+(P<<1)+1|0]=(d[n+64+((P<<1)+24)|0]|0)-(P+12)&255}P=P+1|0}c[I+36>>2]=R;c[I+32>>2]=d[o+24|0]>>1;if((R|0)!=0){c[I+756>>2]=pT(64,R)|0;P=0;while(1){if((P|0)>=(R|0)){break}c[(c[I+756>>2]|0)+(P<<6)+40>>2]=c[t+(P<<2)>>2];c[(c[I+756>>2]|0)+(P<<6)+44>>2]=d[n+26|0]|0;c[(c[I+756>>2]|0)+(P<<6)+48>>2]=(d[n+27|0]|0)!=0?1:0;c[(c[I+756>>2]|0)+(P<<6)+52>>2]=0;c[(c[I+756>>2]|0)+(P<<6)+8>>2]=128;P=P+1|0}}}}E=E+1|0}E=0;while(1){if((E|0)>=(c[h+144>>2]|0)){break}n=(c[h+180>>2]|0)+(E*52|0)|0;if((~e[m+44>>1]&4|0)!=0){t=pT(64,1)|0;c[(c[h+176>>2]|0)+(E*764|0)+756>>2]=t}bK(f|0,g+(c[C+(E<<2)>>2]|0)|0,0)|0;c[p>>2]=jl(f)|0;bB(p+4|0,12,1,f|0)|0;a[p+16|0]=ja(f)|0;a[p+17|0]=ja(f)|0;a[p+18|0]=ja(f)|0;a[p+19|0]=ja(f)|0;bB(p+20|0,26,1,f|0)|0;eK(p+20|0,26);a[p+46|0]=ja(f)|0;a[p+47|0]=ja(f)|0;c[p+48>>2]=jg(f)|0;c[p+52>>2]=jg(f)|0;c[p+56>>2]=jg(f)|0;c[p+60>>2]=jg(f)|0;c[p+64>>2]=jg(f)|0;c[p+68>>2]=jg(f)|0;c[p+72>>2]=jg(f)|0;a[p+76|0]=ja(f)|0;a[p+77|0]=ja(f)|0;a[p+78|0]=ja(f)|0;a[p+79|0]=ja(f)|0;if((c[p>>2]|0)==1229803603){if((a[p+18|0]&2|0)!=0){c[n+44>>2]=1}c[n+32>>2]=c[p+48>>2];c[n+36>>2]=c[p+52>>2];c[n+40>>2]=c[p+56>>2];t=n+44|0;c[t>>2]=c[t>>2]|((a[p+18|0]&16|0)!=0?2:0);t=n+44|0;c[t>>2]=c[t>>2]|((a[p+18|0]&64|0)!=0?4:0);if((~e[m+44>>1]&4|0)!=0){c[c[(c[h+176>>2]|0)+(E*764|0)+756>>2]>>2]=d[p+19|0]|0;c[(c[(c[h+176>>2]|0)+(E*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(E*764|0)+756>>2]|0)+40>>2]=E;c[(c[h+176>>2]|0)+(E*764|0)+36>>2]=((c[n+32>>2]|0)!=0^1^1)&1;t=(c[h+176>>2]|0)+(E*764|0)|0;o=p+20|0;dh(t,o,25)|0}else{o=n|0;t=p+20|0;dh(o,t,25)|0}P=0;while(1){if((P|0)>=(c[h+140>>2]|0)){break}R=0;while(1){if((R|0)>=(c[(c[h+176>>2]|0)+(P*764|0)+36>>2]|0)){break}t=(c[(c[h+176>>2]|0)+(P*764|0)+756>>2]|0)+(R<<6)|0;if((c[t+40>>2]|0)==(E|0)){c[t>>2]=d[p+19|0]|0;c[t+4>>2]=d[p+17|0]|0;c[t+28>>2]=d[p+76|0]|0;c[t+24>>2]=d[p+77|0]>>1;c[t+20>>2]=d[p+79|0]|0;c[t+32>>2]=255-(d[p+78|0]|0)>>1;hD(c[p+60>>2]|0,(c[(c[h+176>>2]|0)+(P*764|0)+756>>2]|0)+(R<<6)+12|0,(c[(c[h+176>>2]|0)+(P*764|0)+756>>2]|0)+(R<<6)+16|0)}R=R+1|0}P=P+1|0}do{if((a[p+18|0]&1|0)!=0){if((c[n+32>>2]|0)<=1){break}t=0;o=f;s=g+(c[p+72>>2]|0)|0;bK(o|0,s|0,0)|0;if((~d[p+46|0]&1|0)!=0){t=t|2}if((a[p+18|0]&8|0)!=0){s=pT(1,c[n+32>>2]<<1)|0;if((a[p+18|0]&2|0)!=0){o=f;T=s;x=c[n+32>>2]|0;q=a[p+46|0]&4;eP(o,T,x,q)|0}else{q=f;x=s;T=c[n+32>>2]|0;o=a[p+46|0]&4;eN(q,x,T,o)|0}o=A;T=16|t;x=(c[h+180>>2]|0)+(E*52|0)|0;q=s;gQ(o,0,T,x,q)|0;pU(s)}else{s=A;q=f;x=t;t=(c[h+180>>2]|0)+(E*52|0)|0;gQ(s,q,x,t,0)|0}}}while(0)}E=E+1|0}c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;p7(v|0,0,64);p7(w|0,0,64);c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;E=0;p=0;while(1){if((E|0)>=(c[h+128>>2]|0)){break}R=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(E<<2)>>2]=R;R=0;if((c[D+(E<<2)>>2]|0)!=0){n=f;t=g+(c[D+(E<<2)>>2]|0)|0;bK(n|0,t|0,0)|0;t=(jc(f)|0)&65535;n=(jc(f)|0)&65535;c[c[(c[h+168>>2]|0)+(E<<2)>>2]>>2]=n;n=0;while(1){if((n|0)>=(c[h+136>>2]|0)){break}x=(ac(E,c[h+136>>2]|0)|0)+n|0;c[(c[(c[h+168>>2]|0)+(E<<2)>>2]|0)+4+(n<<2)>>2]=x;q=pT(((c[c[(c[h+168>>2]|0)+(E<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(x<<2)>>2]=q;c[c[(c[h+172>>2]|0)+(x<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(E<<2)>>2]>>2];n=n+1|0}p7(r|0,0,64);jc(f)|0;jc(f)|0;while(1){n=t-1|0;t=n;if((n|0)<0){break}Q=ja(f)|0;if(Q<<24>>24==0){R=R+1|0;continue}S=(Q&255)-1&63;if((Q&128|0)!=0){a[r+S|0]=ja(f)|0;t=t-1|0}if((S|0)>=(c[h+136>>2]|0)){U=k}else{U=(c[(c[A+172>>2]|0)+(c[(c[(c[A+168>>2]|0)+(E<<2)>>2]|0)+4+(S<<2)>>2]<<2)>>2]|0)+4+(R<<3)|0}V=U;if((a[r+S|0]&1|0)!=0){Q=ja(f)|0;n=Q&255;if((n|0)==255){Q=-127}else if((n|0)==254){Q=-126}else{if((Q&255|0)>119){Q=-125}else{Q=Q+1&255}}n=Q;a[V|0]=n;a[l+(S<<3)|0]=n;t=t-1|0}if((a[r+S|0]&2|0)!=0){Q=ja(f)|0;n=Q;a[V+1|0]=n;a[l+(S<<3)+1|0]=n;t=t-1|0}if((a[r+S|0]&4|0)!=0){Q=ja(f)|0;n=Q;a[V+2|0]=n;a[l+(S<<3)+2|0]=n;eL(V);t=t-1|0}if((a[r+S|0]&8|0)!=0){Q=ja(f)|0;a[V+3|0]=Q;a[V+4|0]=ja(f)|0;eM(S,V,v|0,w|0,F);a[l+(S<<3)+3|0]=a[V+3|0]|0;a[l+(S<<3)+4|0]=a[V+4|0]|0;t=t-2|0}if((a[r+S|0]&16|0)!=0){a[V|0]=a[l+(S<<3)|0]|0}if((a[r+S|0]&32|0)!=0){a[V+1|0]=a[l+(S<<3)+1|0]|0}if((a[r+S|0]&64|0)!=0){a[V+2|0]=a[l+(S<<3)+2|0]|0;eL(V)}if((a[r+S|0]&128|0)!=0){a[V+3|0]=a[l+(S<<3)+3|0]|0;a[V+4|0]=a[l+(S<<3)+4|0]|0}}S=(c[h+136>>2]|0)-1|0;while(1){if((S|0)<(p|0)){break}P=0;t=0;while(1){if((P|0)>=(c[c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(E<<2)>>2]|0)+4+(S<<2)>>2]<<2)>>2]>>2]|0)){break}V=(c[(c[A+172>>2]|0)+(c[(c[(c[A+168>>2]|0)+(E<<2)>>2]|0)+4+(S<<2)>>2]<<2)>>2]|0)+4+(P<<3)|0;if((d[V|0]|0)!=0){H=3630;break}if((d[V+2|0]|0)!=0){H=3630;break}if((d[V+1|0]|0)!=0){H=3630;break}if((d[V+3|0]|0)!=0){H=3630;break}if((d[V+4|0]|0)!=0){H=3630;break}if((d[V+5|0]|0)!=0){H=3630;break}if((d[V+6|0]|0)!=0){H=3630;break}P=P+1|0}if((H|0)==3630){H=0;t=1}do{if((t|0)!=0){if((S|0)<=(p|0)){break}p=S}}while(0);S=S-1|0}}else{c[c[(c[h+168>>2]|0)+(E<<2)>>2]>>2]=64;t=pT(524,1)|0;R=ac(E,c[h+136>>2]|0)|0;c[(c[h+172>>2]|0)+(R<<2)>>2]=t;t=ac(E,c[h+136>>2]|0)|0;c[c[(c[h+172>>2]|0)+(t<<2)>>2]>>2]=64;P=0;while(1){if((P|0)>=(c[h+136>>2]|0)){break}t=ac(E,c[h+136>>2]|0)|0;c[(c[(c[h+168>>2]|0)+(E<<2)>>2]|0)+4+(P<<2)>>2]=t;P=P+1|0}}E=E+1|0}pU(D);pU(C);if((B|0)!=0){pU(B)}c[h+136>>2]=p+1;p=A+1280|0;c[p>>2]=c[p>>2]|1542179;if((b[m+44>>1]&32|0)!=0){m=A+1280|0;c[m>>2]=c[m>>2]|2097152}else{m=A+1280|0;c[m>>2]=c[m>>2]|1024}if((F|0)==0){F=A+1280|0;c[F>>2]=c[F>>2]|524288;F=A+1280|0;c[F>>2]=c[F>>2]&-1048577}c[A+1272>>2]=128;c[A+1284>>2]=3;N=0;O=N;i=j;return O|0}function eK(b,c){b=b|0;c=c|0;var e=0,f=0;e=b;b=c;b=b-1|0;c=0;while(1){if((c|0)>=(b|0)){break}if((d[e+c|0]|0|0)==0){a[e+c|0]=32}c=c+1|0}c=c-1|0;while(1){if((c|0)>=0){f=(d[e+c|0]|0|0)==32}else{f=0}if(!f){break}if((d[e+c|0]|0|0)==32){a[e+c|0]=0}c=c-1|0}return}function eL(b){b=b|0;var c=0,e=0;c=b;b=d[c+2|0]|0;a[c+2|0]=0;if((b|0)<=64){a[c+2|0]=b+1&255;return}do{if((b|0)>=65){if((b|0)>74){e=3671;break}a[c+5|0]=14;a[c+6|0]=(160|b-65)&255}else{e=3671}}while(0);if((e|0)==3671){do{if((b|0)>=75){if((b|0)>84){e=3674;break}a[c+5|0]=14;a[c+6|0]=(176|b-75)&255}else{e=3674}}while(0);if((e|0)==3674){do{if((b|0)>=85){if((b|0)>94){e=3677;break}a[c+5|0]=-92;a[c+6|0]=b-85<<4&255}else{e=3677}}while(0);if((e|0)==3677){do{if((b|0)>=95){if((b|0)>104){e=3680;break}a[c+5|0]=-92;a[c+6|0]=b-95&255}else{e=3680}}while(0);if((e|0)==3680){do{if((b|0)>=105){if((b|0)>114){e=3683;break}a[c+5|0]=2;a[c+6|0]=b-105<<2&255}else{e=3683}}while(0);if((e|0)==3683){do{if((b|0)>=115){if((b|0)>124){e=3686;break}a[c+5|0]=1;a[c+6|0]=b-115<<2&255}else{e=3686}}while(0);if((e|0)==3686){do{if((b|0)>=128){if((b|0)>192){e=3691;break}if((b|0)==192){b=191}a[c+5|0]=22;a[c+6|0]=b-128<<2&255}else{e=3691}}while(0);if((e|0)==3691){do{if((b|0)>=193){if((b|0)>202){e=3694;break}a[c+5|0]=3;a[c+6|0]=b-193<<2&255}else{e=3694}}while(0);if((e|0)==3694){do{if((b|0)>=203){if((b|0)>212){break}a[c+5|0]=4;a[c+6|0]=b-203&255}}while(0)}}}}}}}}return}function eM(b,c,e,f,g){b=b|0;c=c|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0;h=b;b=c;c=e;e=f;f=g;g=(a[b+4|0]&240)>>4&255;i=d[b+4|0]&15;j=a[8784+(d[b+3|0]|0)|0]|0;a[b+3|0]=j;k=j&255;if((k|0)==255){a[b+4|0]=0;a[b+3|0]=0;return}else if((k|0)==254){a[b+3|0]=14;do{if((g&255|0)==0){if((d[b+4|0]|0)!=0){l=3717;break}a[b+4|0]=a[e+h|0]|0;g=(a[b+4|0]&240)>>4&255;i=d[b+4|0]&15}else{l=3717}}while(0);if((l|0)==3717){a[e+h|0]=a[b+4|0]|0}e=g&255;if((e|0)==1){a[b+4|0]=(48|i&255)&255}else if((e|0)==2){a[b+4|0]=(80|i&255)&255}else if((e|0)==3){a[b+4|0]=(64|i&255)&255}else if((e|0)==4){a[b+4|0]=(112|i&255)&255}else if((e|0)==5){if((i&255|0)<=3){a[b+3|0]=-117;a[b+4|0]=i}else{a[b+4|0]=0;a[b+3|0]=0}}else if((e|0)==6){a[b+4|0]=(224|i&255)&255}else if((e|0)==12|(e|0)==13){j=i;a[b+4|0]=j;if((j&255|0)==0){j=b+4|0;a[j]=(a[j]|0)+1&255}j=b+4|0;a[j]=(d[j]|(g&255)<<4)&255}else if((e|0)==14){a[b+3|0]=-120;a[b+4|0]=i}else if((e|0)==9){a[b+4|0]=0;a[b+3|0]=0}else if((e|0)==11){a[b+4|0]=(96|i&255)&255}else if((e|0)==7){a[b+3|0]=-125;g=b+4|0;a[g]=d[g]&15}else if((e|0)==8){a[b+3|0]=22;a[b+4|0]=(i&255)<<4&255}else{a[b+4|0]=0;a[b+3|0]=0}return}else if((k|0)==29){do{if((f|0)==0){if((d[b+4|0]|0)==0){break}a[b+4|0]=(((a[b+4|0]&240)>>4)+1<<4|(a[b+4|0]&15)+1)&255}}while(0);return}else if((k|0)==16){if((d[b+4|0]|0)>128){a[b+4|0]=0;a[b+3|0]=0}return}else if((k|0)==0){if((a[b+4|0]|0)!=0){a[c+h|0]=a[b+4|0]|0}else{a[b+4|0]=a[c+h|0]|0}return}else if((k|0)==132){do{if((d[b+4|0]|0)>127){if((d[b+4|0]|0)>=144){l=3741;break}a[b+3|0]=-123;a[b+4|0]=(d[b+4|0]|0)-128<<4&255}else{l=3741}}while(0);if((l|0)==3741){l=b+4|0;a[l]=d[l]<<1&255}return}else{return}}function eN(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;g=i;i=i+16|0;h=g|0;j=g+8|0;k=b;b=d;d=e;e=f;f=0;l=0;c[h>>2]=0;c[j>>2]=0;m=0;n=0;o=0;L4596:while(1){if((d|0)==0){p=3800;break}if((l|0)==0){l=32768;f=(jc(k)|0)&65535;m=9;o=0;n=0;c[j>>2]=0;c[h>>2]=0}q=l;if(q>>>0>d>>>0){q=d}r=0;do{s=(eO(k,h,j,m&255)|0)&65535;if((bH(k|0)|0)!=0){p=3767;break L4596}L4608:do{if((m&255|0)<7){if((1<<(m&255)-1|0)!=(s&65535|0)){p=3790;break}s=(eO(k,h,j,3)|0)+1&255;if((bH(k|0)|0)!=0){p=3772;break L4596}if((s&255|0)<(m&255|0)){t=s&255}else{t=(s&65535)+1&255&255}m=t&255}else{if((m&255|0)>=9){if((m&255|0)>=10){p=3796;break}if((s&65535|0)>=256){m=(s&65535)+1&255&255;break}else{p=3790;break}}u=(255>>9-(m&255))+4&65535;v=(u&65535)-8&65535;do{if((s&65535|0)>(v&65535|0)){if((s&65535|0)>(u&65535|0)){break}s=(s&65535)-(v&65535)&65535;if((s&255&255|0)<(m&255|0)){w=s&255&255}else{w=(s&65535)+1&255&255}m=w&255;break L4608}}while(0);p=3790}}while(0);if((p|0)==3790){p=0;if((m&255|0)<8){v=8-(m&255)&255;u=(s&65535)<<(v&255)&255;u=u<<24>>24>>(v&255)&255;s=u<<24>>24}s=(s&65535)+(n&255)&65535;n=s&255;o=(o&255)+(n&255)&255;if((e|0)!=0){x=o&255}else{x=n&255}a[b+r|0]=x&255;p=3796}if((p|0)==3796){p=0;r=r+1|0}}while(r>>>0<q>>>0);l=l-q|0;d=d-q|0;b=b+q|0}if((p|0)==3800){y=0;z=y;i=g;return z|0}else if((p|0)==3772){y=-1;z=y;i=g;return z|0}else if((p|0)==3767){y=-1;z=y;i=g;return z|0}return 0}function eO(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;f=a;a=b;b=d;d=e;e=0;g=d;h=c[b>>2]|0;i=c[a>>2]|0;if((d|0)<=0){j=e;k=g;l=32-k|0;m=j>>>(l>>>0);return m|0}do{if((h|0)==0){i=(ja(f)|0)&255;h=8}e=e>>>1;e=e|i<<31;i=i>>1;h=h-1|0;g=g-1|0;}while((g|0)!=0);g=d;c[b>>2]=h;c[a>>2]=i;j=e;k=g;l=32-k|0;m=j>>>(l>>>0);return m|0}function eP(a,d,e,f){a=a|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;g=i;i=i+16|0;h=g|0;j=g+8|0;k=a;a=d;d=e;e=f;f=0;l=0;c[h>>2]=0;c[j>>2]=0;m=0;n=0;o=0;L4670:while(1){if((d|0)==0){break}if((l|0)==0){l=16384;f=(jc(k)|0)&65535;m=17;o=0;n=0;c[j>>2]=0;c[h>>2]=0}p=l;if(p>>>0>d>>>0){p=d}q=0;do{r=eO(k,h,j,m&255)|0;if((bH(k|0)|0)!=0){s=3823;break L4670}L4682:do{if((m&255|0)<7){if((1<<(m&255)-1|0)!=(r|0)){s=3846;break}r=(eO(k,h,j,4)|0)+1|0;if((bH(k|0)|0)!=0){s=3828;break L4670}if((r&255&255|0)<(m&255|0)){t=r&255&255}else{t=r+1&255&255}m=t&255}else{if((m&255|0)>=17){if((m&255|0)>=18){s=3852;break}if(r>>>0>=65536){m=r+1&255&255;break}else{s=3846;break}}u=(65535>>17-(m&255))+8|0;v=u-16&65535;do{if(r>>>0>v>>>0){if(r>>>0>(u&65535)>>>0){break}r=r-v|0;if((r&255&255|0)<(m&255|0)){w=r&255&255}else{w=r+1&255&255}m=w&255;break L4682}}while(0);s=3846}}while(0);if((s|0)==3846){s=0;if((m&255|0)<16){v=16-(m&255)&255;u=r<<(v&255)&65535;u=u<<16>>16>>(v&255)&65535;r=u<<16>>16}r=r+(n<<16>>16)|0;n=r&65535;o=(o<<16>>16)+(n<<16>>16)&65535;if((e|0)!=0){x=o<<16>>16}else{x=n<<16>>16}b[a+(q<<1)>>1]=x&65535;s=3852}if((s|0)==3852){s=0;q=q+1|0}}while(q>>>0<p>>>0);l=l-p|0;d=d-p|0;a=a+(p<<1)|0;if((d|0)<=0){s=3856;break}}if((s|0)==3828){y=-1;z=y;i=g;return z|0}else if((s|0)!=3856)if((s|0)==3823){y=-1;z=y;i=g;return z|0}y=0;z=y;i=g;return z|0}function eQ(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+16|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,14,e|0)|0)>>>0<14){f=-1}else{if((p6(d|0,30584,14)|0)!=0){f=-1;break}else{di(e,a,30);f=0;break}}}while(0);i=c;return f|0}function eR(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;j=i;i=i+336|0;k=j|0;l=j+112|0;m=j+264|0;n=j+304|0;o=j+328|0;p=f;f=g;g=h;h=p|0;q=0;bK(f|0,g|0,0)|0;bB(k|0,14,1,f|0)|0;bB(k+14|0,30,1,f|0)|0;bB(k+44|0,20,1,f|0)|0;ja(f)|0;bB(k+65|0,20,1,f|0)|0;b[k+86>>1]=jc(f)|0;b[k+88>>1]=jc(f)|0;b[k+90>>1]=jc(f)|0;b[k+92>>1]=jc(f)|0;b[k+94>>1]=jc(f)|0;b[k+96>>1]=jc(f)|0;c[k+100>>2]=jg(f)|0;b[k+104>>1]=jc(f)|0;b[k+106>>1]=jc(f)|0;b[k+108>>1]=jc(f)|0;b[k+110>>1]=jc(f)|0;if((e[k+86>>1]>>8|0)==0){b[k+110>>1]=b[k+108>>1]|0;b[k+108>>1]=0;r=f;bK(r|0,-2|0,1)|0}c[h+148>>2]=e[k+88>>1]|0;c[h+152>>2]=e[k+90>>1]|0;c[h+136>>2]=e[k+96>>1]|0;c[h+128>>2]=e[k+104>>1]|0;r=e[k+106>>1]|0;c[h+144>>2]=r;c[h+140>>2]=r;c[h+156>>2]=e[k+108>>1]|0;c[h+132>>2]=ac(c[h+136>>2]|0,c[h+128>>2]|0)|0;r=p+1280|0;c[r>>2]=c[r>>2]|16384;p5(h|0,k+14|0,30)|0;p5(n|0,k+65|0,20)|0;a[n+20|0]=0;r=20;while(1){if((r|0)<0){break}if((a[n+r|0]|0)==32){a[n+r|0]=0}if((a[n+r|0]|0)!=0){s=3878;break}r=r-1|0}t=e[k+86>>1]>>8;u=b[k+86>>1]&255;aM(h+64|0,64,30456,(v=i,i=i+24|0,c[v>>2]=n,c[v+8>>2]=t,c[v+16>>2]=u,v)|0)|0;i=v;if((e[k+86>>1]|0)>0){r=0;while(1){if((r|0)>=(c[h+136>>2]|0)){break}c[h+184+(r*12|0)>>2]=((ja(f)|0)&255)<<2;r=r+1|0}r=0;while(1){if((r|0)>=(c[h+136>>2]|0)){break}c[h+184+(r*12|0)+4>>2]=(ja(f)|0)&255;r=r+1|0}bB(h+952|0,1,c[h+156>>2]|0,f|0)|0;bK(f|0,(e[k+110>>1]|0)-((c[h+136>>2]<<1)+109+(c[h+156>>2]|0))|0,1)|0}else{bK(f|0,g+240|0,0)|0;bB(h+952|0,1,256,f|0)|0;bK(f|0,g+(e[k+110>>1]|0)|0,0)|0;r=0;while(1){if((r|0)>=256){break}if((d[h+952+r|0]|0)==255){s=3894;break}r=r+1|0}c[h+156>>2]=r}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;k=0;g=0;r=0;while(1){if((r|0)>=(c[h+128>>2]|0)){break}v=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(r<<2)>>2]=v;v=jl(f)|0;if((v|0)!=555819297){if((v|0)!=1280311296){bV(28192,316,31360,26920);return 0}v=m|0;u=f;bB(v|0,30,1,u|0)|0;b[m+30>>1]=jc(f)|0;c[m+32>>2]=jg(f)|0;c[m+36>>2]=jg(f)|0;c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2]=e[m+30>>1]|0;u=0;while(1){if((u|0)>=(c[h+136>>2]|0)){break}v=(ac(r,c[h+136>>2]|0)|0)+u|0;c[(c[(c[h+168>>2]|0)+(r<<2)>>2]|0)+4+(u<<2)>>2]=v;t=pT(((c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(v<<2)>>2]=t;c[c[(c[h+172>>2]|0)+(v<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2];u=u+1|0}u=0;v=0;t=aT(f|0)|0;L4811:while(1){q=(c[(c[p+172>>2]|0)+(c[(c[(c[p+168>>2]|0)+(r<<2)>>2]|0)+4+(v<<2)>>2]<<2)>>2]|0)+4+(u<<3)|0;do{if(k<<24>>24!=0){e_(g,q,f);eT(v,q);k=k-1&255}else{g=ja(f)|0;while(1){q=(c[(c[p+172>>2]|0)+(c[(c[(c[p+168>>2]|0)+(r<<2)>>2]|0)+4+(v<<2)>>2]<<2)>>2]|0)+4+(u<<3)|0;n=g&255;if((n|0)==192){break L4811}else if((n|0)==160){break}else if((n|0)==225){s=3938;break}else if((n|0)==224){s=3948;break}else if((n|0)==128){s=3951;break}if((g&255|0)>192){if((g&255|0)<224){s=3956;break}}if((g&255|0)>160){if((g&255|0)<192){s=3961;break}}if((g&255|0)>128){if((g&255|0)<160){s=3966;break}}if((g&255|0)!=255){a[q|0]=(g&255)+37&255}else{if((g&255|0)==254){a[q|0]=-127}}g=ja(f)|0;if((g&255|0)<=100){s=3981;break}u=u+1|0}if((s|0)==3981){s=0;if((g&255|0)!=255){a[q+1|0]=(g&255)+1&255}g=ja(f)|0;if((g&255|0)!=255){a[q+2|0]=g}g=ja(f)|0;if((g&255|0)!=255){a[q+3|0]=(g&255)-65&255}g=ja(f)|0;a[q+4|0]=g;if((d[q+3|0]|0)>26){bV(28192,439,31360,24856);return 0}eT(v,q);do{if((d[q|0]|0)<=119){s=3994}else{if((d[q|0]|0)==129){s=3994;break}bV(28192,446,31360,24048);return 0}}while(0);if((s|0)==3994){s=0}if((d[q+1|0]|0)>100){bV(28192,447,31360,23672);return 0}if((d[q+2|0]|0)>65){bV(28192,448,31360,23088);return 0}break}else if((s|0)==3938){s=0;g=ja(f)|0;v=v+(g&255)|0}else if((s|0)==3948){s=0;g=ja(f)|0;u=u+(g&255)|0;s=3951}else if((s|0)==3961){s=0;k=ja(f)|0;e_(g,q,f);eT(v,q);break}else if((s|0)==3966){s=0;k=ja(f)|0;e_(g,q,f);eT(v,q);while(1){if(k<<24>>24==0){break}u=u+1|0;n=(c[(c[p+172>>2]|0)+(c[(c[(c[p+168>>2]|0)+(r<<2)>>2]|0)+4+(v<<2)>>2]<<2)>>2]|0)+4+(u<<3)|0;w=q;p3(n|0,w|0,8)|0;k=k-1&255}break}else if((s|0)==3956){s=0;e_(g,q,f);eT(v,q);break}if((s|0)==3951){s=0;break}v=v+1|0;if((v|0)>=(c[h+136>>2]|0)){v=v-1|0}u=-1}}while(0);u=u+1|0;if((u|0)>=(c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2]|0)){u=0;k=0;v=v+1|0;if((v|0)>=(c[h+136>>2]|0)){v=0}}}v=(aT(f|0)|0)-t|0;if((v|0)!=(c[m+32>>2]|0)){bV(28192,359,31360,25840);return 0}}r=r+1|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}r=0;while(1){if((r|0)>=(c[h+140>>2]|0)){break}m=pT(64,1)|0;c[(c[h+176>>2]|0)+(r*764|0)+756>>2]=m;bB(o|0,1,4,f|0)|0;do{if((d[o|0]|0)==63){if((d[o+1|0]|0)!=63){s=4023;break}if((d[o+2|0]|0)!=63){s=4023;break}if((d[o+3|0]|0)!=63){s=4023;break}}else{s=4023}}while(0);do{if((s|0)==4023){s=0;do{if((d[o|0]|0)==76){if((d[o+1|0]|0)!=68){s=4028;break}if((d[o+2|0]|0)!=83){s=4028;break}if((d[o+3|0]|0)!=83){s=4028;break}}else{s=4028}}while(0);if((s|0)==4028){s=0;bV(28192,483,31360,22552);return 0}b[l>>1]=jc(f)|0;t=l+2|0;m=f;bB(t|0,30,1,m|0)|0;m=l+32|0;t=f;bB(m|0,20,1,t|0)|0;t=l+52|0;m=f;bB(t|0,20,1,m|0)|0;a[l+72|0]=ja(f)|0;c[l+76>>2]=jg(f)|0;c[l+80>>2]=jg(f)|0;c[l+84>>2]=jg(f)|0;c[l+88>>2]=jg(f)|0;a[l+92|0]=ja(f)|0;a[l+93|0]=ja(f)|0;a[l+94|0]=ja(f)|0;a[l+95|0]=ja(f)|0;a[l+96|0]=ja(f)|0;a[l+97|0]=ja(f)|0;b[l+98>>1]=jc(f)|0;b[l+100>>1]=jc(f)|0;c[l+104>>2]=jg(f)|0;a[l+108|0]=ja(f)|0;m=l+109|0;t=f;bB(m|0,11,1,t|0)|0;t=l+120|0;m=f;bB(t|0,25,1,m|0)|0;c[(c[h+176>>2]|0)+(r*764|0)+36>>2]=((c[l+76>>2]|0)!=0^1^1)&1;c[(c[h+176>>2]|0)+(r*764|0)+32>>2]=64;c[(c[h+180>>2]|0)+(r*52|0)+32>>2]=c[l+76>>2];c[(c[h+180>>2]|0)+(r*52|0)+36>>2]=c[l+80>>2];c[(c[h+180>>2]|0)+(r*52|0)+40>>2]=c[l+84>>2];if((a[l+93|0]&1|0)!=0){c[(c[h+180>>2]|0)+(r*52|0)+44>>2]=1;m=(c[h+180>>2]|0)+(r*52|0)+32|0;c[m>>2]=c[m>>2]>>1;m=(c[h+180>>2]|0)+(r*52|0)+36|0;c[m>>2]=c[m>>2]>>1;m=(c[h+180>>2]|0)+(r*52|0)+40|0;c[m>>2]=c[m>>2]>>1}if((c[l+84>>2]|0)>>>0>0){c[(c[h+180>>2]|0)+(r*52|0)+44>>2]=2}a[l+96|0]=64;c[c[(c[h+176>>2]|0)+(r*764|0)+756>>2]>>2]=d[l+92|0]|0;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+4>>2]=d[l+96|0]|0;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+8>>2]=d[l+94|0]|0;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+40>>2]=r;m=(c[h+176>>2]|0)+(r*764|0)|0;t=l+2|0;dh(m,t,31)|0;hD(c[l+88>>2]|0,(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+16|0);t=f;m=(e[l+98>>1]|0)-144|0;bK(t|0,m|0,1)|0;if((c[(c[h+180>>2]|0)+(r*52|0)+32>>2]|0)!=0){m=p;t=f;k=(c[h+180>>2]|0)+(r*52|0)|0;gQ(m,t,0,k,0)|0;break}else{break}}}while(0);r=r+1|0}r=p+1280|0;c[r>>2]=c[r>>2]|545;c[p+1284>>2]=2;i=j;return 0}function eS(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;return}function eT(b,c){b=b|0;c=c|0;var e=0,f=0,g=0;b=c;c=(a[b+4|0]&240)>>4&255;e=(d[b+4|0]|0)&15;f=a[8752+(d[b+3|0]|0)|0]|0;a[b+3|0]=f;g=f&255;if((g|0)==14){f=c&255;if((f|0)==4){if((e&255|0)==3){e=e+1&255}a[b+4|0]=(e&255|64)&255}else if((f|0)==3){a[b+4|0]=(e&255|48)&255}else if((f|0)==5){a[b+4|0]=(e&255|80)&255}else if((f|0)==6){a[b+4|0]=(e&255|96)&255}else if((f|0)==7){if((e&255|0)==3){e=e+1&255}a[b+4|0]=(e&255|112)&255}else if((f|0)==12){a[b+4|0]=(e&255|192)&255}else if((f|0)==13){a[b+4|0]=(e&255|208)&255}else if((f|0)==14){a[b+4|0]=(e&255|224)&255}else{a[b+4|0]=0;a[b+3|0]=0}return}else if((g|0)==255){a[b+4|0]=0;a[b+3|0]=0;return}else{return}}function eU(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;a=e+140|0;c[a>>2]=(c[a>>2]|0)+1;c[e+144>>2]=c[e+140>>2];return}function eV(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;a=b;if((jl(c)|0)!=1347636512){d=-1;e=d;return e|0}ja(c)|0;ja(c)|0;ja(c)|0;if(((ja(c)|0)&255|0)!=0){d=-1;e=d;return e|0}if((jl(c)|0)!=1179208773){d=-1;e=d;return e|0}jl(c)|0;b=jg(c)|0;bK(c|0,b|0,1)|0;if((jl(c)|0)==1414091852){b=jg(c)|0;di(c,a,b)}else{di(c,a,0)}d=0;e=d;return e|0}function eW(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;f=i;i=i+24|0;g=f|0;h=b;b=d;d=e;e=h|0;bK(b|0,d|0,0)|0;jl(b)|0;c[g>>2]=0;a[e|0]=0;bK(b|0,8,1)|0;c[e+140>>2]=0;c[e+144>>2]=0;c[g+4>>2]=0;c[g+8>>2]=0;j=aT(b|0)|0;k=es()|0;if((k|0)==0){l=-1;m=l;i=f;return m|0}ev(k,30432,150);ev(k,28176,102);ev(k,26864,36);ev(k,25832,40);ev(k,24848,112);ey(k,1);while(1){if(!((bH(b|0)|0)!=0^1)){break}et(k,h,b,g)}eB(k);c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[g+12>>2]=pS(c[e+128>>2]<<3)|0;c[g+16>>2]=pS(2040)|0;dm(h,(c[g>>2]|0)!=0?24032:23648,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;bK(b|0,d+j|0,0)|0;c[e+156>>2]=0;k=es()|0;if((k|0)==0){l=-1;m=l;i=f;return m|0}ev(k,26864,148);ev(k,25832,92);ev(k,24848,70);ey(k,1);while(1){if(!((bH(b|0)|0)!=0^1)){break}et(k,h,b,g)}eB(k);k=0;while(1){if((k|0)>=(c[e+156>>2]|0)){break}b=0;while(1){if((b|0)>=(c[e+128>>2]|0)){break}if((p6((c[g+16>>2]|0)+(k<<3)|0,(c[g+12>>2]|0)+(b<<3)|0,((c[g>>2]|0)!=0?8:4)|0)|0)==0){o=4113;break}b=b+1|0}if((o|0)==4113){o=0;a[e+952+k|0]=b&255}if((b|0)==(c[e+128>>2]|0)){o=4117;break}k=k+1|0}pU(c[g+12>>2]|0);pU(c[g+16>>2]|0);l=0;m=l;i=f;return m|0}function eX(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0;d=i;i=i+40|0;e=d|0;f=b;bB(e|0,1,40,c|0)|0;if((f|0)>32){g=32}else{g=f}p5(a|0,e|0,g|0)|0;i=d;return}function eY(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=d;bK(e|0,10,1)|0;c[a+136>>2]=(ja(e)|0)&255;return}function eZ(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;d=i;i=i+24|0;g=d|0;h=b+128|0;c[h>>2]=(c[h>>2]|0)+1;bB(g|0,1,20,e|0)|0;if((a[g+9|0]|0)==0){i=d;return}if((a[g+13|0]|0)!=0){i=d;return}c[f>>2]=1;i=d;return}function e_(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0;f=b;b=c;c=e;p7(b|0,0,8);if((f&1|0)!=0){e=ja(c)|0;if((e&255|0)==254){a[b|0]=-127}else{a[b|0]=(e&255)+37&255}}if((f&2|0)!=0){a[b+1|0]=((ja(c)|0)&255)+1&255}if((f&4|0)!=0){a[b+2|0]=ja(c)|0}if((f&8|0)!=0){a[b+3|0]=((ja(c)|0)&255)-65&255}if((f&16|0)!=0){a[b+4|0]=ja(c)|0}do{if((d[b|0]|0|0)<=107){g=4154}else{if((d[b|0]|0|0)==129){g=4154;break}bV(28192,209,31520,31168)}}while(0);if((d[b+1|0]|0|0)>100){bV(28192,210,31520,23672)}if((d[b+2|0]|0|0)>64){bV(28192,211,31520,30904)}if((d[b+3|0]|0|0)<=26){return}else{bV(28192,212,31520,24856);return}}function e$(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0;b=i;i=i+24|0;f=d;d=a|0;a=e;bB(b|0,1,9,f|0)|0;jc(f)|0;e=jl(f)|0;while(1){if((e|0)==1330662472){break}bK(f|0,jg(f)|0,1)|0;e=jl(f)|0}jg(f)|0;bK(f|0,9,1)|0;e=ja(f)|0;g=0;while(1){if((e<<24>>24|0)==1){break}h=e<<24>>24;if((h|0)==7){c[d+148>>2]=(ja(f)|0)&255;j=f;ja(j)|0;c[d+152>>2]=(ja(f)|0)&255}else if((h|0)==13){ja(f)|0;c[d+184+(g*12|0)>>2]=(ja(f)|0)&255;ja(f)|0;g=g+1|0}else if((h|0)==14){ja(f)|0;ja(f)|0}else{h=(ja(f)|0)&255;bf(31144,(j=i,i=i+24|0,c[j>>2]=g,c[j+8>>2]=e<<24>>24,c[j+16>>2]=h,j)|0)|0;i=j}e=ja(f)|0}while(1){if((e<<24>>24|0)!=1){break}bB((c[a+16>>2]|0)+(c[d+156>>2]<<3)|0,1,((c[a>>2]|0)!=0?8:4)|0,f|0)|0;g=d+156|0;c[g>>2]=(c[g>>2]|0)+1;e=ja(f)|0}i=b;return}function e0(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;b=a;a=d;d=b|0;f=e;ja(a)|0;bK(a|0,8,1)|0;bK(a|0,((c[f>>2]|0)!=0?8:4)|0,1)|0;e=c[f+8>>2]|0;g=pT(64,1)|0;c[(c[d+176>>2]|0)+(e*764|0)+756>>2]=g;bB((c[d+176>>2]|0)+(e*764|0)|0,1,31,a|0)|0;hy((c[d+176>>2]|0)+(e*764|0)|0)|0;bK(a|0,8,1)|0;ja(a)|0;ja(a)|0;g=jg(a)|0;c[(c[d+180>>2]|0)+(e*52|0)+32>>2]=g;c[(c[d+176>>2]|0)+(e*764|0)+36>>2]=((c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0)!=0^1^1)&1;g=jg(a)|0;c[(c[d+180>>2]|0)+(e*52|0)+36>>2]=g;g=jg(a)|0;c[(c[d+180>>2]|0)+(e*52|0)+40>>2]=g;c[(c[d+180>>2]|0)+(e*52|0)+44>>2]=(c[(c[d+180>>2]|0)+(e*52|0)+40>>2]|0)>2?2:0;jc(a)|0;if((c[(c[d+180>>2]|0)+(e*52|0)+40>>2]|0)<0){c[(c[d+180>>2]|0)+(e*52|0)+40>>2]=0}g=0;if((c[f>>2]|0)!=0){if((c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0)>2){h=(c[d+180>>2]|0)+(e*52|0)+32|0;c[h>>2]=(c[h>>2]|0)-2}if((c[(c[d+180>>2]|0)+(e*52|0)+40>>2]|0)>2){h=(c[d+180>>2]|0)+(e*52|0)+40|0;c[h>>2]=(c[h>>2]|0)-2}g=((jb(a)|0)<<24>>24<<4&255)<<24>>24}h=(((ja(a)|0)&255|0)/2|0)+1|0;c[c[(c[d+176>>2]|0)+(e*764|0)+756>>2]>>2]=h;jg(a)|0;c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+40>>2]=e;h=jg(a)|0;h=(h*8363|0|0)/8448|0;hD(h,(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+16|0);h=(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+16|0;c[h>>2]=(c[h>>2]|0)+g;bK(a|0,16,1)|0;gQ(b,a,4,(c[d+180>>2]|0)+(e*52|0)|0,0)|0;e=f+8|0;c[e>>2]=(c[e>>2]|0)+1;return}function e1(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0;d=i;i=i+8|0;g=d|0;h=b;b=e;e=h|0;j=f;f=c[j+4>>2]|0;jg(b)|0;bB((c[j+12>>2]|0)+(f<<3)|0,1,((c[j>>2]|0)!=0?8:4)|0,b|0)|0;k=(jc(b)|0)&65535;l=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(f<<2)>>2]=l;c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]=k;l=0;while(1){if((l|0)>=(c[e+136>>2]|0)){break}m=(ac(f,c[e+136>>2]|0)|0)+l|0;c[(c[(c[e+168>>2]|0)+(f<<2)>>2]|0)+4+(l<<2)>>2]=m;n=pT(((c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(m<<2)>>2]=n;c[c[(c[e+172>>2]|0)+(m<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(f<<2)>>2]>>2];l=l+1|0}l=0;do{m=((jc(b)|0)&65535)-2|0;while(1){if((m|0)<=0){break}n=ja(b)|0;if((m|0)==1){o=4210;break}p=ja(b)|0;m=m-2|0;if((p&255|0)<(c[e+136>>2]|0)){q=(c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(f<<2)>>2]|0)+4+((p&255)<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}else{q=g}r=q;if((n&128|0)!=0){s=ja(b)|0;m=m-1|0;if((c[j>>2]|0)!=0){s=(s&255)+37&255}else{s=(((s&255)>>4)*12|0)+(s&15)+14&255}a[r|0]=s}if((n&64|0)!=0){a[r+1|0]=((ja(b)|0)&255)+1&255;m=m-1|0}if((n&32|0)!=0){a[r+2|0]=(((ja(b)|0)&255|0)/2|0)&255;m=m-1|0}if((n&16|0)!=0){n=ja(b)|0;s=ja(b)|0;m=m-2|0;if((n&255|0)>=64){if(((s&255)>>4|0)==0){a[r|0]=(((n&255)>>4)*12|0)+(n&15)+2&255;n=3;s=(s&255)+1<<1&255}else{t=f;u=l;v=p&255;w=n&255;x=s&255;bf(23040,(y=i,i=i+40|0,c[y>>2]=t,c[y+8>>2]=u,c[y+16>>2]=v,c[y+24>>2]=w,c[y+32>>2]=x,y)|0)|0;i=y}}else{x=n&255;if((x|0)==1){n=14;s=(160|((s&255|0)/2|0)&15)&255}else if((x|0)==2){n=10;s=((s&255|0)/2|0)<<4&255}else if((x|0)==3){n=14;s=(176|((s&255|0)/2|0)&15)&255}else if((x|0)==4){n=10;s=((s&255|0)/2|0)&255}else if((x|0)==12){n=1;s=(((s&255)-1|0)/2|0)&255}else if((x|0)==14){n=2;s=(((s&255)-1|0)/2|0)&255}else if((x|0)==15){n=3;s=((s&255|0)/4|0)&255}else if((x|0)==21){n=((c[j>>2]|0)!=0?4:172)&255}else if((x|0)==42){n=14;s=(144|s&15)&255}else if((x|0)==41){jc(b)|0;m=m-2|0}else if((x|0)==51){n=11}else if((x|0)==52){n=13}else if((x|0)==61){n=15}else if((x|0)==62){n=15}else{bf(22512,(y=i,i=i+40|0,c[y>>2]=f,c[y+8>>2]=l,c[y+16>>2]=p&255,c[y+24>>2]=n&255,c[y+32>>2]=s&255,y)|0)|0;i=y;s=0;n=0}}a[r+3|0]=n;a[r+4|0]=s}}if((o|0)==4210){o=0}l=l+1|0;}while((l|0)<(k|0));k=j+4|0;c[k>>2]=(c[k>>2]|0)+1;i=d;return}function e2(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;a=b;if((jl(c)|0)!=1145914444){d=-1;e=d;return e|0}ja(c)|0;if(((jd(c)|0)&65535|0)==18766){b=c;jl(b)|0;di(c,a,32)}else{di(c,a,0)}d=0;e=d;return e|0}function e3(e,f,g){e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;h=i;i=i+64|0;j=h|0;k=h+8|0;l=e;e=f;f=l|0;bK(e|0,g|0,0)|0;jl(e)|0;bB(j|0,1,1,e|0)|0;g=es()|0;if((g|0)==0){m=-1;n=m;i=h;return n|0}ev(g,30352,26);ev(g,28168,156);ev(g,26856,126);ev(g,25816,66);ev(g,24840,8);ev(g,24024,146);if(((a[j|0]&240)>>4|0)!=0){ev(g,23640,24);ev(g,23032,10);ev(g,22480,22)}else{ev(g,23032,4);ev(g,22480,16)}ex(g,2);ey(g,1);o=a[j|0]&15;dm(l,31120,(p=i,i=i+16|0,c[p>>2]=(a[j|0]&240)>>4,c[p+8>>2]=o,p)|0);i=p;c[l+1268>>2]=255;c[l+1264>>2]=8363;c[k+36>>2]=0;c[k+32>>2]=0;c[k+28>>2]=0;c[k+4>>2]=pT(256,4)|0;c[k>>2]=pT(256,4)|0;c[k+8>>2]=pS(1024)|0;c[k+12>>2]=pS(1024)|0;c[k+16>>2]=pS(1024)|0;c[k+20>>2]=pT(256,4)|0;p=0;while(1){if((p|0)>=256){break}c[(c[k+16>>2]|0)+(p<<2)>>2]=-1;c[(c[k+12>>2]|0)+(p<<2)>>2]=-1;c[(c[k+8>>2]|0)+(p<<2)>>2]=-1;p=p+1|0}while(1){if(!((bH(e|0)|0)!=0^1)){break}et(g,l,e,k)}eB(g);p=0;while(1){if((p|0)>=(c[f+128>>2]|0)){break}q=0;while(1){if((q|0)>=(c[c[(c[f+168>>2]|0)+(p<<2)>>2]>>2]|0)){break}r=0;while(1){if((r|0)>=(c[f+136>>2]|0)){break}g=0;while(1){if((g|0)>=(c[f+140>>2]|0)){break}if((q|0)<(c[c[(c[f+172>>2]|0)+(c[(c[(c[f+168>>2]|0)+(p<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]>>2]|0)){if((d[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(p<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(q<<3)+1|0]|0)!=0){if((d[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(p<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(q<<3)+1|0]|0)==(c[(c[k>>2]|0)+(g<<2)>>2]|0)){s=4286;break}}}g=g+1|0}if((s|0)==4286){s=0;a[(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(p<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(q<<3)+1|0]=g+1&255}r=r+1|0}q=q+1|0}p=p+1|0}p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}if((c[(c[k+8>>2]|0)+(p<<2)>>2]|0)>=0){c[(c[f+176>>2]|0)+(p*764|0)+44>>2]=1;c[(c[f+176>>2]|0)+(p*764|0)+48>>2]=16;q=0;while(1){if((q|0)>=(c[k+28>>2]|0)){break}if((c[(c[k+8>>2]|0)+(p<<2)>>2]|0)==(q|0)){s=4301;break}q=q+1|0}if((s|0)==4301){s=0;e=(c[f+176>>2]|0)+(p*764|0)+44|0;c[e>>2]=c[e>>2]|((a[(c[k+40>>2]|0)+(q*33|0)+31|0]&16|0)!=0?2:0);e=(c[f+176>>2]|0)+(p*764|0)+44|0;c[e>>2]=c[e>>2]|((a[(c[k+40>>2]|0)+(q*33|0)+31|0]&32|0)!=0?4:0);c[(c[f+176>>2]|0)+(p*764|0)+56>>2]=a[(c[k+40>>2]|0)+(q*33|0)+31|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+64>>2]=a[(c[k+40>>2]|0)+(q*33|0)+32|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+68>>2]=a[(c[k+40>>2]|0)+(q*33|0)+32|0]&240;b[(c[f+176>>2]|0)+(p*764|0)+72>>1]=0;r=1;while(1){if((r|0)>=(c[(c[f+176>>2]|0)+(p*764|0)+48>>2]|0)){break}b[(c[f+176>>2]|0)+(p*764|0)+72+(r<<1<<1)>>1]=(b[(c[f+176>>2]|0)+(p*764|0)+72+(r-1<<1<<1)>>1]|0)+(d[(c[k+40>>2]|0)+(q*33|0)+1+(r-1<<1)|0]|0)&65535;if((d[(c[k+40>>2]|0)+(q*33|0)+1+(r<<1)|0]|0)==0){s=4304;break}b[(c[f+176>>2]|0)+(p*764|0)+72+((r<<1)+1<<1)>>1]=d[(c[k+40>>2]|0)+(q*33|0)+1+((r-1<<1)+1)|0]|0;r=r+1|0}if((s|0)==4304){s=0}c[(c[f+176>>2]|0)+(p*764|0)+48>>2]=r}}if((c[(c[k+12>>2]|0)+(p<<2)>>2]|0)>=0){c[(c[f+176>>2]|0)+(p*764|0)+200>>2]=1;c[(c[f+176>>2]|0)+(p*764|0)+204>>2]=16;q=0;while(1){if((q|0)>=(c[k+32>>2]|0)){break}if((c[(c[k+12>>2]|0)+(p<<2)>>2]|0)==(q|0)){s=4315;break}q=q+1|0}if((s|0)==4315){s=0;e=(c[f+176>>2]|0)+(p*764|0)+200|0;c[e>>2]=c[e>>2]|((a[(c[k+44>>2]|0)+(q*33|0)+31|0]&16|0)!=0?2:0);e=(c[f+176>>2]|0)+(p*764|0)+200|0;c[e>>2]=c[e>>2]|((a[(c[k+44>>2]|0)+(q*33|0)+31|0]&32|0)!=0?4:0);c[(c[f+176>>2]|0)+(p*764|0)+212>>2]=a[(c[k+44>>2]|0)+(q*33|0)+31|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+220>>2]=a[(c[k+44>>2]|0)+(q*33|0)+32|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+224>>2]=a[(c[k+44>>2]|0)+(q*33|0)+32|0]&240;b[(c[f+176>>2]|0)+(p*764|0)+228>>1]=0;r=1;while(1){if((r|0)>=(c[(c[f+176>>2]|0)+(p*764|0)+204>>2]|0)){break}b[(c[f+176>>2]|0)+(p*764|0)+228+(r<<1<<1)>>1]=(b[(c[f+176>>2]|0)+(p*764|0)+228+(r-1<<1<<1)>>1]|0)+(d[(c[k+44>>2]|0)+(q*33|0)+1+(r-1<<1)|0]|0)&65535;if((d[(c[k+44>>2]|0)+(q*33|0)+1+(r<<1)|0]|0)==0){s=4318;break}b[(c[f+176>>2]|0)+(p*764|0)+228+((r<<1)+1<<1)>>1]=d[(c[k+44>>2]|0)+(q*33|0)+1+((r-1<<1)+1)|0]|0;r=r+1|0}if((s|0)==4318){s=0}c[(c[f+176>>2]|0)+(p*764|0)+204>>2]=r}}if((c[(c[k+16>>2]|0)+(p<<2)>>2]|0)>=0){c[(c[f+176>>2]|0)+(p*764|0)+356>>2]=1;c[(c[f+176>>2]|0)+(p*764|0)+360>>2]=16;q=0;while(1){if((q|0)>=(c[k+36>>2]|0)){break}if((c[(c[k+16>>2]|0)+(p<<2)>>2]|0)==(q|0)){s=4329;break}q=q+1|0}if((s|0)==4329){s=0;e=(c[f+176>>2]|0)+(p*764|0)+356|0;c[e>>2]=c[e>>2]|((a[(c[k+48>>2]|0)+(q*33|0)+31|0]&16|0)!=0?2:0);e=(c[f+176>>2]|0)+(p*764|0)+356|0;c[e>>2]=c[e>>2]|((a[(c[k+48>>2]|0)+(q*33|0)+31|0]&32|0)!=0?4:0);c[(c[f+176>>2]|0)+(p*764|0)+368>>2]=a[(c[k+48>>2]|0)+(q*33|0)+31|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+376>>2]=a[(c[k+48>>2]|0)+(q*33|0)+32|0]&15;c[(c[f+176>>2]|0)+(p*764|0)+380>>2]=a[(c[k+48>>2]|0)+(q*33|0)+32|0]&240;b[(c[f+176>>2]|0)+(p*764|0)+384>>1]=0;b[(c[f+176>>2]|0)+(p*764|0)+386>>1]=32;r=1;while(1){if((r|0)>=(c[(c[f+176>>2]|0)+(p*764|0)+360>>2]|0)){break}b[(c[f+176>>2]|0)+(p*764|0)+384+(r<<1<<1)>>1]=(b[(c[f+176>>2]|0)+(p*764|0)+384+(r-1<<1<<1)>>1]|0)+(d[(c[k+48>>2]|0)+(q*33|0)+1+(r-1<<1)|0]|0)&65535;if((d[(c[k+48>>2]|0)+(q*33|0)+1+(r<<1)|0]|0)==0){s=4332;break}b[(c[f+176>>2]|0)+(p*764|0)+384+((r<<1)+1<<1)>>1]=d[(c[k+48>>2]|0)+(q*33|0)+1+((r-1<<1)+1)|0]<<2&65535;r=r+1|0}if((s|0)==4332){s=0}c[(c[f+176>>2]|0)+(p*764|0)+360>>2]=r}}q=0;while(1){if((q|0)>=(c[(c[f+176>>2]|0)+(p*764|0)+36>>2]|0)){break}r=0;while(1){if((r|0)>=(c[f+144>>2]|0)){break}if((c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+(q<<6)+40>>2]|0)==(c[(c[k+4>>2]|0)+(r<<2)>>2]|0)){s=4344;break}r=r+1|0}if((s|0)==4344){s=0;c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+(q<<6)+40>>2]=r;hD(c[(c[k+20>>2]|0)+(r<<2)>>2]|0,(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+(q<<6)+12|0,(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+(q<<6)+16|0)}q=q+1|0}p=p+1|0}pU(c[k+20>>2]|0);pU(c[k+16>>2]|0);pU(c[k+12>>2]|0);pU(c[k+8>>2]|0);pU(c[k>>2]|0);pU(c[k+4>>2]|0);if((c[k+28>>2]|0)!=0){pU(c[k+40>>2]|0)}if((c[k+32>>2]|0)!=0){pU(c[k+44>>2]|0)}if((c[k+36>>2]|0)!=0){pU(c[k+48>>2]|0)}k=l+1280|0;c[k>>2]=c[k>>2]|32;c[l+1284>>2]=1;m=0;n=m;i=h;return n|0}function e4(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;e=d;d=a|0;bB(d|0,1,32,e|0)|0;bK(e|0,20,1)|0;c[d+156>>2]=(jc(e)|0)&65535;c[d+160>>2]=(jc(e)|0)&65535;ja(e)|0;c[d+148>>2]=(ja(e)|0)&255;c[d+152>>2]=(ja(e)|0)&255;a=0;while(1){if((a|0)>=32){break}b=ja(e)|0;if((b&128|0)!=0){f=4364;break}c[d+184+(a*12|0)>>2]=(b&255)<<1;a=a+1|0}c[d+136>>2]=a;bK(e|0,32-a-1|0,1)|0;bB(d+952|0,1,c[d+156>>2]|0,e|0)|0;return}function e5(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;f=e;e=b|0;c[e+132>>2]=((jc(f)|0)&65535)+1;c[e+172>>2]=pV(c[e+172>>2]|0,c[e+132>>2]<<2)|0;b=pT(1,2060)|0;d=pT(1,2060)|0;c[c[e+172>>2]>>2]=d;c[c[c[e+172>>2]>>2]>>2]=256;d=1;while(1){if((d|0)>=(c[e+132>>2]|0)){break}g=(jc(f)|0)&65535;p7(b|0,0,2060);h=0;while(1){if((g|0)==0){break}i=(ja(f)|0)&255;g=g-1|0;j=i&3;if((j|0)==0){h=h+(i>>2)|0}else if((j|0)==1){k=0;while(1){if((k|0)>(i>>2|0)){break}l=b+4+(h+k<<3)|0;m=b+4+(h-1<<3)|0;p3(l|0,m|0,8)|0;k=k+1|0}h=h+(k-1)|0}else if((j|0)==2){m=b+4+(h<<3)|0;l=b+4+(i>>2<<3)|0;p3(m|0,l|0,8)|0}else if((j|0)==3){if((i&4|0)!=0){l=ja(f)|0;g=g-1|0;if((l&255|0)==255){n=129}else{n=(l&255)+12|0}a[b+4+(h<<3)|0]=n&255}if((i&8|0)!=0){g=g-1|0;a[b+4+(h<<3)+1|0]=ja(f)|0}if((i&16|0)!=0){g=g-1|0;a[b+4+(h<<3)+2|0]=ja(f)|0}if((i&32|0)!=0){g=g-1|0;k=(ja(f)|0)&255;a[b+4+(h<<3)+3|0]=k&15;a[b+4+(h<<3)+5|0]=(k&240)>>4&255}if((i&64|0)!=0){g=g-1|0;a[b+4+(h<<3)+4|0]=ja(f)|0}if((i&128|0)!=0){g=g-1|0;a[b+4+(h<<3)+6|0]=ja(f)|0}}fj(b+4+(h<<3)+3|0,b+4+(h<<3)+4|0);fk(b+4+(h<<3)+5|0,b+4+(h<<3)+6|0);h=h+1|0}if((h|0)<=64){h=64}else{if((h|0)<=128){h=128}else{h=256}}g=pT(1,(h<<3)+12|0)|0;c[(c[e+172>>2]|0)+(d<<2)>>2]=g;g=c[(c[e+172>>2]|0)+(d<<2)>>2]|0;l=b;m=(h<<3)+12|0;p3(g|0,l|0,m)|0;c[c[(c[e+172>>2]|0)+(d<<2)>>2]>>2]=h;d=d+1|0}pU(b);return}function e6(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;b=a;a=d;d=b|0;f=e;e=0;while(1){if((e|0)>=(c[d+144>>2]|0)){break}if((c[(c[d+180>>2]|0)+(e*52|0)+44>>2]&1|0)!=0){g=c[(c[d+180>>2]|0)+(e*52|0)+32>>2]<<1}else{g=c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0}h=pT(1,g)|0;i=c[(c[f+24>>2]|0)+(e<<2)>>2]|0;if((i|0)==0){j=h;k=c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0;l=a;bB(j|0,1,k|0,l|0)|0}else if((i|0)==1){m=jg(a)|0;n=pS(m+4|0)|0;bB(n|0,1,m|0,a|0)|0;ff(h,n,m,c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0);pU(n)}else if((i|0)==2){m=jg(a)|0;n=pS(m+4|0)|0;bB(n|0,1,m|0,a|0)|0;fg(h,n,m,c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0);pU(n)}gQ(b,0,16,(c[d+180>>2]|0)+(e*52|0)|0,h)|0;pU(h);e=e+1|0}pU(c[f+24>>2]|0);return}function e7(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;d=e;e=f;f=(ja(d)|0)&255;c[e+28>>2]=f;if((f|0)==0){return}c[e+40>>2]=pT(c[e+28>>2]|0,33)|0;f=0;while(1){if((f|0)>=(c[e+28>>2]|0)){break}b=ja(d)|0;a[(c[e+40>>2]|0)+(f*33|0)|0]=b;bB((c[e+40>>2]|0)+(f*33|0)+1|0,1,30,d|0)|0;b=ja(d)|0;a[(c[e+40>>2]|0)+(f*33|0)+31|0]=b;b=ja(d)|0;a[(c[e+40>>2]|0)+(f*33|0)+32|0]=b;f=f+1|0}return}function e8(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;d=e;e=f;f=(ja(d)|0)&255;c[e+32>>2]=f;if((f|0)==0){return}c[e+44>>2]=pT(c[e+32>>2]|0,33)|0;f=0;while(1){if((f|0)>=(c[e+32>>2]|0)){break}b=ja(d)|0;a[(c[e+44>>2]|0)+(f*33|0)|0]=b;bB((c[e+44>>2]|0)+(f*33|0)+1|0,1,30,d|0)|0;b=ja(d)|0;a[(c[e+44>>2]|0)+(f*33|0)+31|0]=b;b=ja(d)|0;a[(c[e+44>>2]|0)+(f*33|0)+32|0]=b;f=f+1|0}return}function e9(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;d=e;e=f;f=(ja(d)|0)&255;c[e+36>>2]=f;if((f|0)==0){return}c[e+48>>2]=pT(c[e+36>>2]|0,33)|0;f=0;while(1){if((f|0)>=(c[e+36>>2]|0)){break}b=ja(d)|0;a[(c[e+48>>2]|0)+(f*33|0)|0]=b;bB((c[e+48>>2]|0)+(f*33|0)+1|0,1,30,d|0)|0;b=ja(d)|0;a[(c[e+48>>2]|0)+(f*33|0)+31|0]=b;b=ja(d)|0;a[(c[e+48>>2]|0)+(f*33|0)+32|0]=b;f=f+1|0}return}
function fa(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;d=i;i=i+40|0;g=d|0;h=e;e=b|0;b=f;c[e+140>>2]=(ja(h)|0)&255;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}j=(ja(h)|0)&255;c[(c[b>>2]|0)+(f<<2)>>2]=j;j=(ja(h)|0)&255;c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=j;bB(g|0,1,32,h|0)|0;a[g+32|0]=0;hy(g|0)|0;p5((c[e+176>>2]|0)+(f*764|0)|0,g|0,32)|0;j=pT(64,c[(c[e+176>>2]|0)+(f*764|0)+36>>2]|0)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=j;j=0;while(1){if((j|0)>=121){break}a[(c[e+176>>2]|0)+(f*764|0)+512+(j<<1)|0]=-1;j=j+1|0}j=0;k=0;while(1){if((j|0)>=(c[(c[e+176>>2]|0)+(f*764|0)+36>>2]|0)){break}l=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+40>>2]=l;l=((ja(h)|0)&255)+12|0;m=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)>>2]=m;m=k;while(1){if((m|0)>(l|0)){break}if((m|0)<121){a[(c[e+176>>2]|0)+(f*764|0)+512+(m<<1)|0]=j&255}m=m+1|0}k=l+1|0;m=(ja(h)|0)&255;if((j|0)==0){if((m&128|0)!=0){n=m&63}else{n=-1}c[(c[b+8>>2]|0)+(f<<2)>>2]=n}if((~m&64|0)!=0){c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)>>2]=255}o=((ja(h)|0)&255)<<1;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+8>>2]=o;m=(ja(h)|0)&255;if((j|0)==0){if((m&128|0)!=0){p=m&63}else{p=-1}c[(c[b+12>>2]|0)+(f<<2)>>2]=p}if((~m&64|0)!=0){c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+8>>2]=128}m=(jc(h)|0)&65535;if((j|0)==0){c[(c[e+176>>2]|0)+(f*764|0)+40>>2]=m}o=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+28>>2]=o;o=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+24>>2]=o;o=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+32>>2]=o;o=(ja(h)|0)&255;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+(j<<6)+20>>2]=o;ja(h)|0;m=(ja(h)|0)&255;if((j|0)==0){if((m&128|0)!=0){q=m&63}else{q=-1}c[(c[b+16>>2]|0)+(f<<2)>>2]=q}j=j+1|0}f=f+1|0}i=d;return}function fb(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0;e=d;d=a|0;c[d+128>>2]=(ja(e)|0)&255;c[d+132>>2]=(ac(c[d+128>>2]|0,c[d+136>>2]|0)|0)+1;c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;a=0;while(1){if((a|0)>=(c[d+128>>2]|0)){break}b=pT(1,((c[d+136>>2]|0)-1<<2)+8|0)|0;c[(c[d+168>>2]|0)+(a<<2)>>2]=b;b=(ja(e)|0)&255;f=((ja(e)|0)&255)+1|0;c[c[(c[d+168>>2]|0)+(a<<2)>>2]>>2]=f;bK(e|0,16,1)|0;f=0;while(1){if((f|0)>=(b|0)){break}g=(jc(e)|0)&65535;if((f|0)<(c[d+136>>2]|0)){c[(c[(c[d+168>>2]|0)+(a<<2)>>2]|0)+4+(f<<2)>>2]=g}f=f+1|0}a=a+1|0}return}function fc(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0;d=i;i=i+64|0;g=d|0;h=e;e=b|0;b=f;c[e+144>>2]=(ja(h)|0)&255;c[e+180>>2]=pT(52,c[e+144>>2]|0)|0;c[b+24>>2]=pT(4,c[e+144>>2]|0)|0;f=0;while(1){if((f|0)>=(c[e+144>>2]|0)){break}j=(ja(h)|0)&255;c[(c[b+4>>2]|0)+(f<<2)>>2]=j;bB(g|0,1,32,h|0)|0;a[g+32|0]=0;hy(g|0)|0;bK(h|0,8,1)|0;j=jg(h)|0;c[(c[b+20>>2]|0)+(f<<2)>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=j;c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0)>0?2:0;c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+36>>2]|0)+(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0);if((c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0)>0){j=(c[e+180>>2]|0)+(f*52|0)+40|0;c[j>>2]=(c[j>>2]|0)-1}ja(h)|0;j=ja(h)|0;if((j&1|0)!=0){k=(c[e+180>>2]|0)+(f*52|0)+44|0;c[k>>2]=c[k>>2]|1;k=(c[e+180>>2]|0)+(f*52|0)+32|0;c[k>>2]=c[k>>2]>>1;k=(c[e+180>>2]|0)+(f*52|0)+36|0;c[k>>2]=c[k>>2]>>1;k=(c[e+180>>2]|0)+(f*52|0)+40|0;c[k>>2]=c[k>>2]>>1}k=(c[e+180>>2]|0)+(f*52|0)+44|0;c[k>>2]=c[k>>2]|((j&2|0)!=0?4:0);c[(c[b+24>>2]|0)+(f<<2)>>2]=(j&12)>>2;f=f+1|0}i=d;return}function fd(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;e=d;d=a|0;c[d+128>>2]=(ja(e)|0)&255;c[d+132>>2]=(ac(c[d+128>>2]|0,c[d+136>>2]|0)|0)+1;c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;a=0;while(1){if((a|0)>=(c[d+128>>2]|0)){break}b=pT(1,((c[d+136>>2]|0)-1<<2)+8|0)|0;c[(c[d+168>>2]|0)+(a<<2)>>2]=b;c[c[(c[d+168>>2]|0)+(a<<2)>>2]>>2]=64;b=0;while(1){if((b|0)>=32){break}f=jc(e)|0;if((b|0)<(c[d+136>>2]|0)){c[(c[(c[d+168>>2]|0)+(a<<2)>>2]|0)+4+(b<<2)>>2]=f&65535}b=b+1|0}a=a+1|0}return}function fe(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0;d=i;i=i+64|0;g=d|0;h=e;e=b|0;b=f;f=(ja(h)|0)&255;c[e+144>>2]=f;c[e+140>>2]=f;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}c[b+24>>2]=pT(4,c[e+144>>2]|0)|0;f=0;while(1){if((f|0)>=(c[e+140>>2]|0)){break}c[(c[e+176>>2]|0)+(f*764|0)+36>>2]=1;j=pT(64,1)|0;c[(c[e+176>>2]|0)+(f*764|0)+756>>2]=j;j=(ja(h)|0)&255;c[(c[b+4>>2]|0)+(f<<2)>>2]=j;c[(c[b>>2]|0)+(f<<2)>>2]=j;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+40>>2]=j;bB(g|0,1,32,h|0)|0;a[g+32|0]=0;hy(g|0)|0;bK(h|0,8,1)|0;j=(jc(h)|0)&65535;c[(c[b+20>>2]|0)+(f<<2)>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+32>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+36>>2]=j;j=jg(h)|0;c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=j;c[(c[e+180>>2]|0)+(f*52|0)+44>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0)>0?2:0;c[(c[e+180>>2]|0)+(f*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(f*52|0)+36>>2]|0)+(c[(c[e+180>>2]|0)+(f*52|0)+40>>2]|0);j=(ja(h)|0)&255;c[c[(c[e+176>>2]|0)+(f*764|0)+756>>2]>>2]=j;c[(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+8>>2]=128;j=ja(h)|0;if((j&1|0)!=0){k=(c[e+180>>2]|0)+(f*52|0)+44|0;c[k>>2]=c[k>>2]|1;k=(c[e+180>>2]|0)+(f*52|0)+32|0;c[k>>2]=c[k>>2]>>1;k=(c[e+180>>2]|0)+(f*52|0)+36|0;c[k>>2]=c[k>>2]>>1;k=(c[e+180>>2]|0)+(f*52|0)+40|0;c[k>>2]=c[k>>2]>>1}k=(c[e+180>>2]|0)+(f*52|0)+44|0;c[k>>2]=c[k>>2]|((j&2|0)!=0?4:0);c[(c[b+24>>2]|0)+(f<<2)>>2]=(j&12)>>2;f=f+1|0}i=d;return}function ff(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;g=i;i=i+24|0;h=g|0;j=g+8|0;k=g+16|0;l=b;c[h>>2]=d;c[j>>2]=e;e=f;fi(0,h,j,k)|0;f=0;d=0;b=0;while(1){if((b|0)>=(e|0)){break}m=fi(1,h,j,k)|0;if((fi(1,h,j,k)|0)!=0){d=(fi(3,h,j,k)|0)&255}else{d=8;while(1){if((c[j>>2]|0)>=0){n=(fi(1,h,j,k)|0)!=0^1}else{n=0}if(!n){break}d=(d&255)+16&255}d=(d&255)+(fi(4,h,j,k)|0)&255}if((m|0)!=0){d=(d&255^255)&255}f=(f&255)+(d&255)&255;o=l;l=o+1|0;a[o]=f;b=b+1|0}i=g;return}function fg(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;g=i;i=i+24|0;h=g|0;j=g+8|0;k=g+16|0;l=b;c[h>>2]=d;c[j>>2]=e;e=f;fi(0,h,j,k)|0;f=0;d=0;b=0;m=0;while(1){if((m|0)>=(e|0)){break}b=fi(8,h,j,k)|0;n=fi(1,h,j,k)|0;if((fi(1,h,j,k)|0)!=0){d=(fi(3,h,j,k)|0)&255}else{d=8;while(1){if((c[j>>2]|0)>=0){o=(fi(1,h,j,k)|0)!=0^1}else{o=0}if(!o){break}d=(d&255)+16&255}d=(d&255)+(fi(4,h,j,k)|0)&255}if((n|0)!=0){d=(d&255^255)&255}f=(f&255)+(d&255)&255;p=l;l=p+1|0;a[p]=b&255;p=l;l=p+1|0;a[p]=f;m=m+1|0}i=g;return}function fh(b,c){b=b|0;c=c|0;var e=0,f=0,g=0;e=b;b=c;c=d[e]|0;if((c|0)==7){a[e]=-85;return}else if((c|0)==11|(c|0)==12|(c|0)==13){return}else if((c|0)==0){a[b]=0;return}else if((c|0)==8|(c|0)==9|(c|0)==10){a[b]=0;a[e]=0;return}else if((c|0)==14){f=(a[b]&240)>>4;if((f|0)==0|(f|0)==3|(f|0)==8){a[b]=0;a[e]=0}else if((f|0)==1){a[e]=25;g=b;a[g]=(d[g]|0)<<4&255}else if((f|0)==2){a[e]=25;f=b;a[f]=(d[f]|0)&15}return}else if((c|0)==15){a[e]=-93;return}else{return}}function fi(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0;f=a;a=b;b=d;d=e;if((f<<24>>24|0)==0){c[d>>2]=jj(c[a>>2]|0)|0;e=a;c[e>>2]=(c[e>>2]|0)+4;e=b;c[e>>2]=(c[e>>2]|0)-4;c[d+4>>2]=32;g=0;h=g;return h|0}e=c[d>>2]&(1<<(f<<24>>24))-1;i=d|0;c[i>>2]=(c[i>>2]|0)>>>(f<<24>>24>>>0);i=d+4|0;j=(c[i>>2]|0)-(f<<24>>24)|0;c[i>>2]=j;do{if(j>>>0<=24){if((c[b>>2]|0)!=0){i=a;f=c[i>>2]|0;c[i>>2]=f+1;i=jj(f)|0;f=d|0;c[f>>2]=c[f>>2]|i<<c[d+4>>2];i=d+4|0;c[i>>2]=(c[i>>2]|0)+8;i=b;c[i>>2]=(c[i>>2]|0)-1;break}g=e;h=g;return h|0}}while(0);g=e;h=g;return h|0}function fj(b,c){b=b|0;c=c|0;var e=0;e=b;b=c;c=d[e]|0;if((c|0)==6){a[b]=0;a[e]=0}else if((c|0)==5){a[e]=0}fh(e,b);return}function fk(b,c){b=b|0;c=c|0;var e=0;e=b;b=c;c=d[e]|0;if((c|0)==2){a[e]=-95}else if((c|0)==4){a[e]=7}else if((c|0)==3){a[e]=27}else if((c|0)==1){a[e]=-96}else if((c|0)==5){a[e]=29}else if((c|0)==6){a[b]=0;a[e]=0}fh(e,b);return}function fl(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1296385026){d=-1;e=d;return e|0}else{di(c,b,0);d=0;e=d;return e|0}return 0}function fm(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;g=i;i=i+4720|0;h=g|0;j=g+40|0;k=g+4136|0;l=g+4392|0;m=g+4648|0;n=b;b=e;e=n|0;bK(b|0,f|0,0)|0;if((jl(b)|0)!=1296385026){o=-1;p=o;i=g;return p|0}dm(n,30320,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0);i=f;c[e+144>>2]=32;c[e+140>>2]=32;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}bB(h|0,1,40,b|0)|0;q=0;while(1){if((q|0)>=31){break}bB(h|0,1,40,b|0)|0;dh((c[e+176>>2]|0)+(q*764|0)|0,h|0,32)|0;r=pT(64,1)|0;c[(c[e+176>>2]|0)+(q*764|0)+756>>2]=r;q=q+1|0}ja(b)|0;q=0;while(1){if((q|0)>=31){break}h=(ja(b)|0)&255;c[c[(c[e+176>>2]|0)+(q*764|0)+756>>2]>>2]=h;c[(c[(c[e+176>>2]|0)+(q*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(q*764|0)+756>>2]|0)+16>>2]=0;c[(c[(c[e+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]=q;q=q+1|0}jd(b)|0;q=0;while(1){if((q|0)>=31){break}h=(jd(b)|0)&65535;c[(c[e+180>>2]|0)+(q*52|0)+36>>2]=h;q=q+1|0}jd(b)|0;q=0;while(1){if((q|0)>=31){break}h=(jd(b)|0)&65535;c[(c[e+180>>2]|0)+(q*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(q*52|0)+36>>2]|0)+h;c[(c[e+180>>2]|0)+(q*52|0)+44>>2]=h>>>0>1?2:0;q=q+1|0}c[e+136>>2]=4;c[e+128>>2]=(jd(b)|0)&65535;c[e+132>>2]=ac(c[e+136>>2]|0,c[e+128>>2]|0)|0;bB(e+952|0,1,100,b|0)|0;c[e+156>>2]=(jd(b)|0)&65535;c[e+148>>2]=192/((jd(b)|0)&65535|0)|0;jd(b)|0;h=(jd(b)|0)&65535;jl(b)|0;bK(b|0,16,1)|0;if((h|0)==6){h=n+1280|0;c[h>>2]=c[h>>2]|192}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;q=0;while(1){if((q|0)>=(c[e+128>>2]|0)){break}h=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(q<<2)>>2]=h;c[c[(c[e+168>>2]|0)+(q<<2)>>2]>>2]=64;h=0;while(1){if((h|0)>=(c[e+136>>2]|0)){break}r=(ac(q,c[e+136>>2]|0)|0)+h|0;c[(c[(c[e+168>>2]|0)+(q<<2)>>2]|0)+4+(h<<2)>>2]=r;s=pT(((c[c[(c[e+168>>2]|0)+(q<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(r<<2)>>2]=s;c[c[(c[e+172>>2]|0)+(r<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(q<<2)>>2]>>2];h=h+1|0}jl(b)|0;h=0;while(1){if((h|0)>=64){break}r=0;while(1){if((r|0)>=4){break}s=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(q<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(h<<3)|0;a[s|0]=(ht((jd(b)|0)&65535)|0)&255;t=ja(b)|0;a[s+1|0]=(t&255)>>4&255;a[s+3|0]=t&255&15;a[s+4|0]=ja(b)|0;t=d[s+3|0]|0;if(!((t|0)==0|(t|0)==1|(t|0)==2|(t|0)==3|(t|0)==4|(t|0)==12))if((t|0)==13|(t|0)==14){a[s+3|0]=10}else if((t|0)==15){a[s+3|0]=(192/(d[s+3|0]|0|0)|0)&255}r=r+1|0}h=h+1|0}q=q+1|0}q=0;while(1){if((q|0)>=31){break}b=0;dl(n,k|0,256);h=dk(k|0,(c[e+176>>2]|0)+(q*764|0)|0,l|0,256)|0;if((h|0)!=0){r=j|0;s=k|0;t=l|0;aM(r|0,4096,28160,(f=i,i=i+16|0,c[f>>2]=s,c[f+8>>2]=t,f)|0)|0;i=f;t=bh(j|0,26848)|0;b=t;if((t|0)!=0){t=aE(b|0)|0;aC(t|0,m|0)|0;c[(c[e+180>>2]|0)+(q*52|0)+32>>2]=c[m+28>>2]}}c[(c[e+176>>2]|0)+(q*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(q*52|0)+32>>2]|0)!=0^1^1)&1;do{if((p4((c[e+176>>2]|0)+(q*764|0)|0)|0)!=0){u=4724}else{if((c[(c[e+180>>2]|0)+(q*52|0)+32>>2]|0)!=0){u=4724;break}}}while(0);if((u|0)==4724){u=0;if((h|0)!=0){t=n;s=b;r=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(t,s,0,r,0)|0;r=b;au(r|0)|0}}q=q+1|0}o=0;p=o;i=g;return p|0}function fn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1296385027){d=-1;e=d;return e|0}else{di(c,b,0);d=0;e=d;return e|0}return 0}function fo(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;f=i;i=i+40|0;g=f|0;h=b;b=d;d=h|0;bK(b|0,e|0,0)|0;jl(b)|0;dm(h,30272,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0);i=e;c[d+144>>2]=32;c[d+140>>2]=32;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}e=0;while(1){if((e|0)>=32){break}j=0;while(1){if((j|0)>=40){break}k=ja(b)|0;a[g+j|0]=k;if((k&255|0)==0){l=4751;break}j=j+1|0}if((l|0)==4751){l=0}dh((c[d+176>>2]|0)+(e*764|0)|0,g|0,32)|0;j=pT(64,1)|0;c[(c[d+176>>2]|0)+(e*764|0)+756>>2]=j;e=e+1|0}g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}if((g&-2147483648|0)!=0){m=(ja(b)|0)&255}else{m=0}c[c[(c[d+176>>2]|0)+(e*764|0)+756>>2]>>2]=m;c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+16>>2]=0;c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+40>>2]=e;e=e+1|0;g=g<<1}g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}if((g&-2147483648|0)!=0){n=(jd(b)|0)&65535}else{n=0}c[(c[d+180>>2]|0)+(e*52|0)+36>>2]=n;e=e+1|0;g=g<<1}g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}if((g&-2147483648|0)!=0){o=(jd(b)|0)&65535}else{o=0}n=o;c[(c[d+180>>2]|0)+(e*52|0)+32>>2]=(c[(c[d+180>>2]|0)+(e*52|0)+36>>2]|0)+n;c[(c[d+180>>2]|0)+(e*52|0)+40>>2]=(c[(c[d+180>>2]|0)+(e*52|0)+36>>2]|0)+n;c[(c[d+180>>2]|0)+(e*52|0)+44>>2]=n>>>0>1?2:0;e=e+1|0;g=g<<1}c[d+136>>2]=4;c[d+128>>2]=(jd(b)|0)&65535;c[d+132>>2]=ac(c[d+136>>2]|0,c[d+128>>2]|0)|0;c[d+156>>2]=(jd(b)|0)&65535;bB(d+952|0,1,c[d+156>>2]|0,b|0)|0;c[d+148>>2]=(jd(b)|0)&65535;if((c[d+148>>2]|0)>10){c[d+152>>2]=((c[d+148>>2]|0)*125|0|0)/33|0;c[d+148>>2]=6}o=(jb(b)|0)<<24>>24;ja(b)|0;n=(jd(b)|0)&65535;jl(b)|0;bK(b|0,16,1)|0;g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}if((g&-2147483648|0)!=0){m=b;ja(m)|0}e=e+1|0;g=g<<1}g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}if((g&-2147483648|0)!=0){m=b;ja(m)|0}e=e+1|0;g=g<<1}if((n|0)==6){n=h+1280|0;c[n>>2]=c[n>>2]|192}e=0;while(1){if((e|0)>=32){break}c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+12>>2]=o;e=e+1|0}c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;e=0;while(1){if((e|0)>=(c[d+128>>2]|0)){break}o=pT(1,((c[d+136>>2]|0)-1<<2)+8|0)|0;c[(c[d+168>>2]|0)+(e<<2)>>2]=o;c[c[(c[d+168>>2]|0)+(e<<2)>>2]>>2]=64;o=0;while(1){if((o|0)>=(c[d+136>>2]|0)){break}n=(ac(e,c[d+136>>2]|0)|0)+o|0;c[(c[(c[d+168>>2]|0)+(e<<2)>>2]|0)+4+(o<<2)>>2]=n;m=pT(((c[c[(c[d+168>>2]|0)+(e<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[d+172>>2]|0)+(n<<2)>>2]=m;c[c[(c[d+172>>2]|0)+(n<<2)>>2]>>2]=c[c[(c[d+168>>2]|0)+(e<<2)>>2]>>2];o=o+1|0}ja(b)|0;o=ja(b)|0;n=jd(b)|0;m=pT(1,(n&65535)+16|0)|0;if((m|0)==0){bV(28136,317,31328,26840);return 0}if((o&16|0)!=0){c[m>>2]=0}else{if((o&1|0)!=0){c[m>>2]=-1}else{c[m>>2]=jl(b)|0}}if((o&32|0)!=0){c[m+4>>2]=0}else{if((o&2|0)!=0){c[m+4>>2]=-1}else{c[m+4>>2]=jl(b)|0}}if((o&64|0)!=0){c[m+8>>2]=0}else{if((o&4|0)!=0){c[m+8>>2]=-1}else{c[m+8>>2]=jl(b)|0}}if((o&128|0)!=0){c[m+12>>2]=0}else{if((o&8|0)!=0){c[m+12>>2]=-1}else{c[m+12>>2]=jl(b)|0}}bB(m+16|0,1,n&65535|0,b|0)|0;fs(h,e&65535,m);pU(m);e=e+1|0}g=jl(b)|0;e=0;while(1){if((e|0)>=32){break}do{if((~g&-2147483648|0)==0){m=jl(b)|0;c[(c[d+180>>2]|0)+(e*52|0)+32>>2]=m;if((jd(b)|0)<<16>>16!=0){break}c[(c[d+176>>2]|0)+(e*764|0)+36>>2]=((c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0)!=0^1^1)&1;m=h;n=b;o=(c[d+180>>2]|0)+((c[(c[(c[d+176>>2]|0)+(e*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(m,n,0,o,0)|0}}while(0);e=e+1|0;g=g<<1}i=f;return 0}function fp(a,c){a=a|0;c=c|0;var f=0,g=0;f=c;c=a+((e[f>>1]|0|0)/2|0)|0;if((b[f>>1]&1|0)!=0){g=(d[c]|0)&15}else{g=(d[c]|0)>>4&255}c=f;b[c>>1]=(b[c>>1]|0)+1&65535;return g|0}function fq(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;d=a;a=b;b=c;c=0;while(1){e=b;b=e-1&255;if(e<<24>>24==0){break}c=(c&65535)<<4&65535;c=(c&65535|(fp(d,a)|0)&255)&65535}return c|0}function fr(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1296385028){d=-1;e=d;return e|0}else{di(c,b,0);d=0;e=d;return e|0}return 0}function fs(e,f,g){e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;h=i;i=i+40|0;j=h|0;k=h+8|0;l=h+16|0;m=h+24|0;n=h+32|0;o=e;e=f;f=g;c[j>>2]=c[f>>2];c[k>>2]=c[f+4>>2];c[l>>2]=c[f+8>>2];c[m>>2]=c[f+12>>2];g=j;j=l;b[n>>1]=0;l=f+16|0;p=c[o+136>>2]|0;f=f+16|0;f=pT(3,256)|0;q=f;r=f;if((q|0)==0){bV(28136,98,31280,25808)}f=0;while(1){if((f|0)>=64){break}if((f|0)==32){g=k;j=m}if((c[g>>2]&-2147483648|0)!=0){s=fq(l,n,((p|0)/4|0)&255)|0;s=(s&65535)<<16-p&65535;t=q;u=0;while(1){if((u&255|0)>=(p|0)){break}if((s&32768|0)!=0){a[t]=(fq(l,n,2)|0)&255;a[t+1|0]=((fp(l,n)|0)&255)<<4&255}s=(s&65535)<<1&65535;t=t+3|0;u=u+1&255}}if((c[j>>2]&-2147483648|0)!=0){s=fq(l,n,((p|0)/4|0)&255)|0;s=(s&65535)<<16-p&65535;t=q;u=0;while(1){if((u&255|0)>=(p|0)){break}if((s&32768|0)!=0){v=(fp(l,n)|0)&255;w=t+1|0;a[w]=(d[w]|v)&255;a[t+2|0]=(fq(l,n,2)|0)&255}s=(s&65535)<<1&65535;t=t+3|0;u=u+1&255}}q=q+(p*3|0)|0;v=g;c[v>>2]=c[v>>2]<<1;v=j;c[v>>2]=c[v>>2]<<1;f=f+1|0}f=0;while(1){if((f|0)>=64){break}j=0;while(1){if((j|0)>=4){break}g=(c[(c[o+172>>2]|0)+(c[(c[(c[o+168>>2]|0)+((e&65535)<<2)>>2]|0)+4+(j<<2)>>2]<<2)>>2]|0)+4+(f<<3)|0;a[g|0]=a[r+((f*12|0)+(j*3|0))|0]|0;if((a[g|0]|0)!=0){p=g|0;a[p]=(d[p]|0)+48&255}a[g+1|0]=d[r+((f*12|0)+(j*3|0)+1)|0]>>4&255;if((a[g+1|0]|0)!=0){p=g+1|0;a[p]=(a[p]|0)+1&255}a[g+3|0]=d[r+((f*12|0)+(j*3|0)+1)|0]&15;a[g+4|0]=a[r+((f*12|0)+(j*3|0)+2)|0]|0;p=d[g+3|0]|0;if((p|0)==12){a[g+4|0]=(((a[g+4|0]&240)>>4)*10|0)+(a[g+4|0]&15)&255}else if((p|0)==13){a[g+3|0]=10}else if((p|0)==15){if((d[g+4|0]|0)==0){a[g+3|0]=13}if((d[g+4|0]|0)==255){a[g+3|0]=0;a[g+4|0]=0;a[g+2|0]=1}else{if((d[g+4|0]|0)==254){a[g+3|0]=0;a[g+4|0]=0}else{if((d[g+4|0]|0)==241){a[g+3|0]=14;a[g+4|0]=-109}else{if((d[g+4|0]|0)==242){a[g+3|0]=14;a[g+4|0]=-61}else{if((d[g+4|0]|0)==243){a[g+3|0]=14;a[g+4|0]=-45}else{if((d[g+4|0]|0)>10){a[g+3|0]=-85;a[g+4|0]=(((d[g+4|0]|0)*125|0|0)/33|0)&255}}}}}}}else if(!((p|0)==0|(p|0)==1|(p|0)==2|(p|0)==3|(p|0)==4)){a[g+3|0]=0;a[g+4|0]=0}j=j+1|0}f=f+1|0}pU(r);i=h;return}function ft(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0;j=i;i=i+1696|0;k=j|0;l=j+16|0;m=j+1040|0;n=j+1048|0;o=j+1088|0;p=j+1096|0;q=j+1128|0;r=j+1160|0;s=f;f=g;g=h;h=s|0;t=0;bK(f|0,g|0,0)|0;jl(f)|0;u=2;v=10;w=aT(f|0)|0;bK(f|0,0,2)|0;if((aT(f|0)|0)>2e3){x=f;bK(x|0,-1023|0,1)|0;x=l|0;y=f;bB(x|0,1,1024,y|0)|0;z=0;while(1){if((z|0)>=1012){break}if((p6(l+z|0,30240,8)|0)==0){A=4946;break}z=z+1|0}if((A|0)==4946){u=d[l+z+10|0]|0;v=d[l+z+11|0]|0}}bK(f|0,g+w|0,0)|0;aM(h+64|0,64,28112,(g=i,i=i+16|0,c[g>>2]=u,c[g+8>>2]=v,g)|0)|0;i=g;g=(ja(f)|0)&255;v=0;y=0;z=0;while(1){if((z|0)>=8){break}if((g&128|0)!=0){y=y<<8;y=y|(ja(f)|0)&255;v=v+1|0}z=z+1|0;g=g<<1}y=y<<32-(v<<3);v=0;z=0;while(1){if((z|0)>=32){break}g=0;p7(34120+(z*48|0)|0,0,48);if((~y&-2147483648|0)==0){x=ja(f)|0;B=ja(f)|0;C=0;while(1){if((C|0)>=(B&255|0)){break}a[n+C|0]=ja(f)|0;C=C+1|0}a[n+C|0]=0;c[34160+(z*48|0)>>2]=64;if((x&1|0)==0){c[34152+(z*48|0)>>2]=((jd(f)|0)&65535)<<1}if((x&2|0)==0){g=((jd(f)|0)&65535)<<1&65535}if((x&4|0)==0){B=f;ja(B)|0}if((x&8|0)==0){B=f;ja(B)|0}if((x&32|0)==0){c[34160+(z*48|0)>>2]=(ja(f)|0)&255}if((x&64|0)==0){c[34164+(z*48|0)>>2]=(jb(f)|0)<<24>>24}c[34156+(z*48|0)>>2]=(c[34152+(z*48|0)>>2]|0)+(g&65535);dh(34120+(z*48|0)|0,n|0,32)|0;v=v+1|0}z=z+1|0;y=y<<1}c[h+128>>2]=(jd(f)|0)&65535;c[h+156>>2]=(jd(f)|0)&65535;bB(h+952|0,1,c[h+156>>2]|0,f|0)|0;n=(jd(f)|0)&65535;if((n|0)<=10){c[h+148>>2]=n;c[h+152>>2]=125}else{c[h+152>>2]=(n*125|0|0)/33|0}n=(jb(f)|0)<<24>>24;jb(f)|0;B=(jb(f)|0)<<24>>24;c[h+148>>2]=(ja(f)|0)&255;if((~B&32|0)!=0){D=s+1280|0;c[D>>2]=c[D>>2]|192}if((B&16|0)!=0){t=1}if((u|0)==2){c[h+148>>2]=(B&32|0)!=0?5:6}bK(f|0,20,1)|0;bB(k|0,1,16,f|0)|0;ja(f)|0;z=0;while(1){if((z|0)>=32){break}k=34164+(z*48|0)|0;c[k>>2]=(c[k>>2]|0)+n;z=z+1|0}ja(f)|0;c[h+136>>2]=(ja(f)|0)&255;bK(f|0,-2|0,1)|0;c[h+132>>2]=ac(c[h+136>>2]|0,c[h+128>>2]|0)|0;c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;z=0;while(1){if((z|0)>=(c[h+128>>2]|0)){break}ja(f)|0;ja(f)|0;n=((ja(f)|0)&255)+1|0;jd(f)|0;C=0;while(1){if((C|0)>=4){break}if((n|0)<=(C<<6|0)){A=5009;break}a[o+C|0]=ja(f)|0;C=C+1|0}if((A|0)==5009){A=0}g=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(z<<2)>>2]=g;c[c[(c[h+168>>2]|0)+(z<<2)>>2]>>2]=n;g=0;while(1){if((g|0)>=(c[h+136>>2]|0)){break}x=(ac(z,c[h+136>>2]|0)|0)+g|0;c[(c[(c[h+168>>2]|0)+(z<<2)>>2]|0)+4+(g<<2)>>2]=x;k=pT(((c[c[(c[h+168>>2]|0)+(z<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(x<<2)>>2]=k;c[c[(c[h+172>>2]|0)+(x<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(z<<2)>>2]>>2];g=g+1|0}g=0;while(1){if((g|0)>=8){break}c[p+(g<<2)>>2]=0;c[q+(g<<2)>>2]=0;g=g+1|0}x=0;g=0;while(1){if((g|0)>=8){break}if((n|0)<=(g<<5|0)){A=5040;break}k=d[o+((g|0)/2|0)|0]|0;B=((g|0)%2|0)<<2;if((k&128>>B|0)!=0){E=-1}else{if((k&64>>B|0)!=0){F=0}else{F=jl(f)|0}E=F}c[p+(g<<2)>>2]=E;if((k&32>>B|0)!=0){G=-1}else{if((k&16>>B|0)!=0){H=0}else{H=jl(f)|0}G=H}c[q+(g<<2)>>2]=G;x=x+1|0;g=g+1|0}if((A|0)==5040){A=0}if(((ja(f)|0)&255|0)!=255){A=5044;break}c[m>>2]=0;g=0;while(1){if((g|0)>=(x|0)){break}C=0;while(1){if((C|0)>=32){break}n=(g<<5)+C|0;if((c[p+(g<<2)>>2]&-2147483648|0)!=0){I=fv(f,m)|0;J=0;while(1){if((J|0)>=4){break}K=(c[(c[s+172>>2]|0)+(c[(c[(c[s+168>>2]|0)+(z<<2)>>2]|0)+4+(J<<2)>>2]<<2)>>2]|0)+4+(n<<3)|0;if((I&8|0)!=0){L=(fw(f,m)|0)&65535;a[K|0]=L>>>4&255;if((a[K|0]|0)!=0){B=K|0;a[B]=(d[B]|0)+48&255}a[K+1|0]=L&15}J=J+1|0;I=(I&255)<<1&255}}if((c[q+(g<<2)>>2]&-2147483648|0)!=0){I=fv(f,m)|0;J=0;while(1){if((J|0)>=4){break}K=(c[(c[s+172>>2]|0)+(c[(c[(c[s+168>>2]|0)+(z<<2)>>2]|0)+4+(J<<2)>>2]<<2)>>2]|0)+4+(n<<3)|0;if((I&8|0)!=0){L=(fw(f,m)|0)&65535;a[K+3|0]=L>>>8&255;a[K+4|0]=L&255;fu(K)}J=J+1|0;I=(I&255)<<1&255}}n=p+(g<<2)|0;c[n>>2]=c[n>>2]<<1;n=q+(g<<2)|0;c[n>>2]=c[n>>2]<<1;C=C+1|0}g=g+1|0}z=z+1|0}if((A|0)==5044){M=-1;N=M;i=j;return N|0}c[h+140>>2]=v;c[s+6548>>2]=pT(4,c[h+140>>2]|0)|0;c[s+6552>>2]=pT(4,c[h+140>>2]|0)|0;y=jl(f)|0;jd(f)|0;jd(f)|0;y=y<<1;w=aT(f|0)|0;q=0;p=y;z=0;while(1){if((z|0)>=32){break}if((~p&-2147483648|0)==0){I=f;jd(I)|0;I=(jd(f)|0)&65535;J=(jd(f)|0)<<16>>16;K=aT(f|0)|0;do{if((J|0)==0){A=5081}else{if((J|0)==-2){A=5081;break}if((J|0)==-1){L=f;bK(L|0,20,1)|0;q=q+((jd(f)|0)&65535)|0}}}while(0);if((A|0)==5081){A=0;q=q+1|0}J=f;L=K+I|0;bK(J|0,L|0,0)|0}z=z+1|0;p=p<<1}bK(f|0,w|0,0)|0;c[h+144>>2]=q;c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}q=0;z=0;L6272:while(1){if((z|0)>=(v|0)){A=5131;break}do{if((~y&-2147483648|0)==0){w=f;jd(w)|0;w=(jd(f)|0)&65535;p=(jd(f)|0)<<16>>16;L=(c[h+176>>2]|0)+(z*764|0)|0;J=34120+(z*48|0)|0;p5(L|0,J|0,32)|0;if((p|0)==-2){J=aT(f|0)|0;L=f;jl(L)|0;L=f;jd(L)|0;L=f;jd(L)|0;L=f;jd(L)|0;b[r+10>>1]=jd(f)|0;b[r+12>>1]=jd(f)|0;b[r+14>>1]=jd(f)|0;b[r+16>>1]=jd(f)|0;a[r+18|0]=ja(f)|0;a[r+19|0]=ja(f)|0;b[r+20>>1]=jd(f)|0;L=r+22|0;m=e[r+14>>1]|0;G=f;bB(L|0,1,m|0,G|0)|0;G=r+150|0;m=e[r+16>>1]|0;L=f;bB(G|0,1,m|0,L|0)|0;L=f;m=J+(jl(f)|0)|0;bK(L|0,m|0,0)|0;m=jl(f)|0;L=f;jd(L)|0;L=pS(8)|0;c[(c[h+176>>2]|0)+(z*764|0)+760>>2]=L;if((c[(c[h+176>>2]|0)+(z*764|0)+760>>2]|0)==0){A=5101;break L6272}L=pT(64,1)|0;c[(c[h+176>>2]|0)+(z*764|0)+756>>2]=L;if((c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)==0){A=5103;break L6272}c[(c[h+176>>2]|0)+(z*764|0)+36>>2]=1;c[c[(c[h+176>>2]|0)+(z*764|0)+760>>2]>>2]=d[r+18|0]|0;c[(c[(c[h+176>>2]|0)+(z*764|0)+760>>2]|0)+4>>2]=d[r+19|0]|0;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+8>>2]=128;c[c[(c[h+176>>2]|0)+(z*764|0)+756>>2]>>2]=c[34160+(z*48|0)>>2];c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+12>>2]=c[34164+(z*48|0)>>2];c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+40>>2]=q;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+16>>2]=0;c[(c[h+180>>2]|0)+(q*52|0)+32>>2]=m;c[(c[h+180>>2]|0)+(q*52|0)+36>>2]=c[34152+(z*48|0)>>2];c[(c[h+180>>2]|0)+(q*52|0)+40>>2]=c[34156+(z*48|0)>>2];c[(c[h+180>>2]|0)+(q*52|0)+44>>2]=(c[34156+(z*48|0)>>2]|0)>1?2:0;m=s;L=f;J=(c[h+180>>2]|0)+(q*52|0)|0;gQ(m,L,0,J,0)|0;q=q+1|0;J=pT(1,e[r+14>>1]|0)|0;c[(c[s+6548>>2]|0)+(z<<2)>>2]=J;J=c[(c[s+6548>>2]|0)+(z<<2)>>2]|0;L=r+22|0;m=e[r+14>>1]|0;p3(J|0,L|0,m)|0;m=pT(1,e[r+16>>1]|0)|0;c[(c[s+6552>>2]|0)+(z<<2)>>2]=m;m=c[(c[s+6552>>2]|0)+(z<<2)>>2]|0;L=r+150|0;J=e[r+16>>1]|0;p3(m|0,L|0,J)|0;break}if((p|0)!=-1){if((p|0)!=0){p=f;J=w;bK(p|0,J|0,1)|0;break}J=pT(64,1)|0;c[(c[h+176>>2]|0)+(z*764|0)+756>>2]=J;c[(c[h+176>>2]|0)+(z*764|0)+36>>2]=1;c[c[(c[h+176>>2]|0)+(z*764|0)+756>>2]>>2]=c[34160+(z*48|0)>>2];c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+12>>2]=c[34164+(z*48|0)>>2];c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+40>>2]=q;c[(c[h+180>>2]|0)+(q*52|0)+32>>2]=w;c[(c[h+180>>2]|0)+(q*52|0)+36>>2]=c[34152+(z*48|0)>>2];c[(c[h+180>>2]|0)+(q*52|0)+40>>2]=c[34156+(z*48|0)>>2];c[(c[h+180>>2]|0)+(q*52|0)+44>>2]=(c[34156+(z*48|0)>>2]|0)>1?2:0;J=s;p=f;L=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(J,p,0,L,0)|0;q=q+1|0;break}L=aT(f|0)|0;p=f;jl(p)|0;p=f;jd(p)|0;p=f;jd(p)|0;p=f;jd(p)|0;b[r+10>>1]=jd(f)|0;b[r+12>>1]=jd(f)|0;b[r+14>>1]=jd(f)|0;b[r+16>>1]=jd(f)|0;a[r+18|0]=ja(f)|0;a[r+19|0]=ja(f)|0;b[r+20>>1]=jd(f)|0;p=r+22|0;J=e[r+14>>1]|0;m=f;bB(p|0,1,J|0,m|0)|0;m=r+150|0;J=e[r+16>>1]|0;p=f;bB(m|0,1,J|0,p|0)|0;C=0;while(1){if((C|0)>=(e[r+20>>1]|0)){break}c[r+280+(C<<2)>>2]=jl(f)|0;C=C+1|0}if((e[r+20>>1]|0)==65535){break}p=pS(8)|0;c[(c[h+176>>2]|0)+(z*764|0)+760>>2]=p;if((c[(c[h+176>>2]|0)+(z*764|0)+760>>2]|0)==0){A=5117;break L6272}p=pT(64,e[r+20>>1]|0)|0;c[(c[h+176>>2]|0)+(z*764|0)+756>>2]=p;if((c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)==0){A=5119;break L6272}c[(c[h+176>>2]|0)+(z*764|0)+36>>2]=e[r+20>>1]|0;c[c[(c[h+176>>2]|0)+(z*764|0)+760>>2]>>2]=d[r+18|0]|0;c[(c[(c[h+176>>2]|0)+(z*764|0)+760>>2]|0)+4>>2]=d[r+19|0]|0;C=0;while(1){if((C|0)>=(e[r+20>>1]|0)){break}c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+(C<<6)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+(C<<6)>>2]=c[34160+(z*48|0)>>2];c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+(C<<6)+12>>2]=(c[34164+(z*48|0)>>2]|0)-24;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+(C<<6)+40>>2]=q;c[(c[(c[h+176>>2]|0)+(z*764|0)+756>>2]|0)+(C<<6)+16>>2]=0;bK(f|0,L+(c[r+280+(C<<2)>>2]|0)|0,0)|0;p=((jd(f)|0)&65535)<<1;c[(c[h+180>>2]|0)+(q*52|0)+32>>2]=p;c[(c[h+180>>2]|0)+(q*52|0)+36>>2]=0;c[(c[h+180>>2]|0)+(q*52|0)+40>>2]=c[(c[h+180>>2]|0)+(q*52|0)+32>>2];c[(c[h+180>>2]|0)+(q*52|0)+44>>2]=2;gQ(s,f,0,(c[h+180>>2]|0)+(q*52|0)|0,0)|0;q=q+1|0;C=C+1|0}p=pT(1,e[r+14>>1]|0)|0;c[(c[s+6548>>2]|0)+(z<<2)>>2]=p;p=c[(c[s+6548>>2]|0)+(z<<2)>>2]|0;J=r+22|0;m=e[r+14>>1]|0;p3(p|0,J|0,m)|0;m=pT(1,e[r+16>>1]|0)|0;c[(c[s+6552>>2]|0)+(z<<2)>>2]=m;m=c[(c[s+6552>>2]|0)+(z<<2)>>2]|0;J=r+150|0;p=e[r+16>>1]|0;p3(m|0,J|0,p)|0;bK(f|0,L+w|0,0)|0}}while(0);z=z+1|0;y=y<<1}if((A|0)==5101){M=-1;N=M;i=j;return N|0}else if((A|0)==5103){M=-1;N=M;i=j;return N|0}else if((A|0)==5131){jd(f)|0;while(1){if(!((bH(f|0)|0)!=0^1)){break}y=jl(f)|0;if((y|0)<0){A=5134;break}z=jl(f)|0;r=z;if((z|0)<0){A=5136;break}z=aT(f|0)|0;s=y;if((s|0)!=1212957763)if((s|0)==1296385110){jl(f)|0;}else if((s|0)==1095650895){if((r|0)<1023){O=r}else{O=1023}s=O;bB(l|0,1,s|0,f|0)|0;a[l+s|0]=0}bK(f|0,z+r|0,0)|0}M=0;N=M;i=j;return N|0}else if((A|0)==5117){M=-1;N=M;i=j;return N|0}else if((A|0)==5119){M=-1;N=M;i=j;return N|0}return 0}function fu(b){b=b|0;var c=0;c=b;b=d[c+3|0]|0;if((b|0)==12){a[c+4|0]=(((a[c+4|0]&240)>>4)*10|0)+(a[c+4|0]&15)&255;return}else if((b|0)==13){a[c+3|0]=10;return}else if((b|0)==15){if((d[c+4|0]|0|0)==0){a[c+3|0]=13}if((d[c+4|0]|0|0)==255){a[c+3|0]=0;a[c+4|0]=0;a[c+2|0]=1}else{if((d[c+4|0]|0|0)==254){a[c+3|0]=0;a[c+4|0]=0}else{if((d[c+4|0]|0|0)==241){a[c+3|0]=14;a[c+4|0]=-109}else{if((d[c+4|0]|0|0)==242){a[c+3|0]=14;a[c+4|0]=-61}else{if((d[c+4|0]|0|0)==243){a[c+3|0]=14;a[c+4|0]=-45}else{if((d[c+4|0]|0|0)>10){a[c+3|0]=-85;a[c+4|0]=(((d[c+4|0]|0)*125|0|0)/33|0)&255}}}}}}return}else if((b|0)==0|(b|0)==1|(b|0)==2|(b|0)==3|(b|0)==4){return}else{a[c+3|0]=0;a[c+4|0]=0;return}}function fv(b,e){b=b|0;e=e|0;var f=0,g=0;f=e;if((c[f>>2]&1|0)!=0){g=(d[35656]|0)&15}else{a[35656]=ja(b)|0;g=(d[35656]|0)>>4&255}b=f;c[b>>2]=c[b>>2]^1;return g|0}function fw(a,b){a=a|0;b=b|0;var c=0,d=0;c=a;a=b;b=(fv(c,a)|0)&255;d=(fv(c,a)|0)&255;return(b<<8|d<<4|(fv(c,a)|0)&255)&65535|0}function fx(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;e=i;i=i+384|0;f=e|0;g=b;b=c;do{if((bB(f|0,1,384,g|0)|0)>>>0<384){h=-1}else{if((d[f+249|0]|0|0)!=127){h=-1;break}c=0;while(1){if((c|0)>=31){j=5214;break}k=(ji(f+(c<<3)|0)|0)&65535;if((k|0)>32767){j=5200;break}if((a[f+((c<<3)+2)|0]&240|0)!=0){j=5202;break}if((d[f+((c<<3)+3)|0]|0|0)>64){j=5204;break}l=(ji(f+(c<<3)+4|0)|0)&65535;if((l|0)>(k|0)){j=5206;break}m=(ji(f+(c<<3)+6|0)|0)&65535;if((l+m-1|0)>(k|0)){j=5208;break}if((k|0)>0){if((m|0)==0){j=5211;break}}c=c+1|0}if((j|0)==5200){h=-1;break}else if((j|0)==5202){h=-1;break}else if((j|0)==5204){h=-1;break}else if((j|0)==5206){h=-1;break}else if((j|0)==5208){h=-1;break}else if((j|0)==5211){h=-1;break}else if((j|0)==5214){c=d[f+248|0]|0;if((c|0)!=((ji(f+378|0)|0)&65535|0)){h=-1;break}c=(ji(f+378|0)|0)&65535;if((c|0)!=((ji(f+380|0)|0)&65535|0)){h=-1;break}else{di(g,b,0);h=0;break}}}}while(0);i=e;return h|0}function fy(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0;g=i;i=i+7248|0;h=g|0;j=g+72|0;k=g+4168|0;l=g+6216|0;m=g+7240|0;n=b;b=e;e=n|0;bK(b|0,f|0,0)|0;dm(n,26712,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0);i=f;c[e+136>>2]=4;c[e+144>>2]=31;c[e+140>>2]=31;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}o=0;while(1){if((o|0)>=31){break}p=pT(64,1)|0;c[(c[e+176>>2]|0)+(o*764|0)+756>>2]=p;p=((jd(b)|0)&65535)<<1;c[(c[e+180>>2]|0)+(o*52|0)+32>>2]=p;p=(((ja(b)|0)&255)<<4&255)<<24>>24;c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+16>>2]=p;p=(ja(b)|0)&255;c[c[(c[e+176>>2]|0)+(o*764|0)+756>>2]>>2]=p;p=((jd(b)|0)&65535)<<1;c[(c[e+180>>2]|0)+(o*52|0)+36>>2]=p;p=(jd(b)|0)&65535;c[(c[e+180>>2]|0)+(o*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(o*52|0)+36>>2]|0)+(p<<1);c[(c[e+180>>2]|0)+(o*52|0)+44>>2]=(p|0)>1?2:0;c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]=o;c[(c[e+176>>2]|0)+(o*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(o*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[e+176>>2]|0)+(o*764|0)+40>>2]=4095;o=o+1|0}p=(ja(b)|0)&255;c[e+128>>2]=p;c[e+156>>2]=p;ja(b)|0;o=0;while(1){if((o|0)>=128){break}a[e+952+o|0]=ja(b)|0;o=o+1|0}c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;p=(jd(b)|0)&65535;jd(b)|0;o=0;while(1){if((o|0)>=(p|0)){break}r=0;while(1){if((r|0)>=4){break}c[k+(o<<4)+(r<<2)>>2]=(jd(b)|0)&65535;r=r+1|0}o=o+1|0}p=aT(b|0)|0;o=0;while(1){if((o|0)>=(c[e+128>>2]|0)){break}s=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(o<<2)>>2]=s;c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2]=64;s=0;while(1){if((s|0)>=(c[e+136>>2]|0)){break}t=(ac(o,c[e+136>>2]|0)|0)+s|0;c[(c[(c[e+168>>2]|0)+(o<<2)>>2]|0)+4+(s<<2)>>2]=t;u=pT(((c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(t<<2)>>2]=u;c[c[(c[e+172>>2]|0)+(t<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(o<<2)>>2]>>2];s=s+1|0}r=0;while(1){if((r|0)>=4){break}bK(b|0,p+(c[k+(o<<4)+(r<<2)>>2]|0)|0,0)|0;bB(l|0,1,1024,b|0)|0;s=0;t=0;while(1){if((s|0)>=4){break}u=0;while(1){if((u|0)>=4){break}v=0;while(1){if((v|0)>=4){break}w=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(o<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(t<<3)|0;x=m;y=l+(d[l+((d[l+((d[l+s|0]|0)+u)|0]|0)+v)|0]<<1)|0;a[x]=a[y]|0;a[x+1|0]=a[y+1|0]|0;a[x+2|0]=a[y+2|0]|0;a[x+3|0]=a[y+3|0]|0;dj(w,m|0);v=v+1|0;t=t+1|0}u=u+1|0}s=s+1|0}r=r+1|0}o=o+1|0}L6500:do{if((p4(c[n+1212>>2]|0)|0)>>>0<5){z=5286}else{if((a[(c[n+1212>>2]|0)+3|0]|0)!=46){z=5286;break}a[c[n+1212>>2]|0]=115;a[(c[n+1212>>2]|0)+1|0]=109;a[(c[n+1212>>2]|0)+2|0]=112;r=j|0;m=c[n+1208>>2]|0;l=c[n+1212>>2]|0;aM(r|0,4096,28104,(f=i,i=i+16|0,c[f>>2]=m,c[f+8>>2]=l,f)|0)|0;i=f;do{if((bP(j|0,h|0)|0)<0){if((aR(c[n+1212>>2]|0,45)|0)!=0){l=aW(j|0,45)|0;if((l|0)!=0){m=l;qc(m|0,26832)|0}}if((bP(j|0,h|0)|0)<0){m=c[q>>2]|0;l=j|0;bE(m|0,25760,(f=i,i=i+8|0,c[f>>2]=l,f)|0)|0;i=f;break L6500}else{break}}}while(0);l=bh(j|0,24824)|0;m=l;if((l|0)==0){l=c[q>>2]|0;r=j|0;bE(l|0,23984,(f=i,i=i+8|0,c[f>>2]=r,f)|0)|0;i=f;break}o=0;while(1){if((o|0)>=(c[e+140>>2]|0)){break}gQ(n,m,512,(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;o=o+1|0}au(m|0)|0;r=n+1280|0;c[r>>2]=c[r>>2]|8192;A=0;B=A;i=g;return B|0}}while(0);if((z|0)==5286){z=c[q>>2]|0;j=c[n+1212>>2]|0;bE(z|0,30208,(f=i,i=i+8|0,c[f>>2]=j,f)|0)|0;i=f}o=0;while(1){if((o|0)>=(c[e+140>>2]|0)){break}c[(c[e+176>>2]|0)+(o*764|0)+36>>2]=0;p7((c[e+180>>2]|0)+(o*52|0)|0,0,52);o=o+1|0}A=0;B=A;i=g;return B|0}function fz(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;if((jf(d)|0)!=5064532){e=-1;f=e;return f|0}ja(d)|0;if((jl(d)|0)!=-1119009965){e=-1;f=e;return f|0}else{a=d;bK(a|0,18,1)|0;a=d;g=c+(jl(d)|0)|0;bK(a|0,g|0,0)|0;di(d,b,32);e=0;f=e;return f|0}return 0}function fA(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;g=i;i=i+256|0;h=g|0;j=b;b=e;e=f;f=j|0;bK(b|0,e|0,0)|0;jf(b)|0;k=(ja(b)|0)&255;jl(b)|0;dm(j,30184,(l=i,i=i+16|0,c[l>>2]=(k&240)>>4,c[l+8>>2]=k&15,l)|0);i=l;c[f+136>>2]=(jd(b)|0)&65535;jd(b)|0;c[f+156>>2]=(jd(b)|0)&65535;c[f+128>>2]=(jd(b)|0)&65535;c[f+132>>2]=(jd(b)|0)&65535;l=(jd(b)|0)&65535;c[f+144>>2]=l;c[f+140>>2]=l;jd(b)|0;jl(b)|0;l=jl(b)|0;k=jl(b)|0;m=jl(b)|0;n=jl(b)|0;o=jl(b)|0;jl(b)|0;jl(b)|0;jl(b)|0;bK(b|0,e+l|0,0)|0;bB(f|0,1,32,b|0)|0;k=jl(b)|0;c[f+156>>2]=(jd(b)|0)&65535;c[f+160>>2]=(jd(b)|0)&65535;c[f+152>>2]=(ja(b)|0)&255;c[f+148>>2]=(ja(b)|0)&255;jd(b)|0;ja(b)|0;ja(b)|0;l=0;while(1){if((l|0)>=(c[f+136>>2]|0)){break}jd(b)|0;l=l+1|0}bK(b|0,e+k|0,0)|0;l=0;while(1){if((l|0)>=(c[f+156>>2]|0)){break}a[f+952+l|0]=(jd(b)|0)&255;l=l+1|0}c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}bK(b|0,e+m|0,0)|0;l=0;while(1){if((l|0)>=(c[f+140>>2]|0)){break}m=pT(64,1)|0;c[(c[f+176>>2]|0)+(l*764|0)+756>>2]=m;bB((c[f+176>>2]|0)+(l*764|0)|0,1,32,b|0)|0;c[h+(l<<2)>>2]=jl(b)|0;m=jl(b)|0;c[(c[f+180>>2]|0)+(l*52|0)+32>>2]=m;m=jl(b)|0;c[(c[f+180>>2]|0)+(l*52|0)+36>>2]=m;m=c[(c[f+180>>2]|0)+(l*52|0)+36>>2]|0;k=m+(jl(b)|0)|0;c[(c[f+180>>2]|0)+(l*52|0)+40>>2]=k;jl(b)|0;jl(b)|0;k=jl(b)|0;hD(k,(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+12|0,(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+16|0);k=((jd(b)|0)&65535)>>4;c[c[(c[f+176>>2]|0)+(l*764|0)+756>>2]>>2]=k;ja(b)|0;ja(b)|0;c[(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+8>>2]=128;k=(ja(b)|0)&255;c[(c[f+180>>2]|0)+(l*52|0)+44>>2]=(k&3|0)!=0?2:0;m=(c[f+180>>2]|0)+(l*52|0)+44|0;c[m>>2]=c[m>>2]|((k&2|0)!=0?4:0);ja(b)|0;k=(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+16|0;c[k>>2]=(c[k>>2]|0)+0;ja(b)|0;ja(b)|0;ja(b)|0;ja(b)|0;jd(b)|0;jl(b)|0;jl(b)|0;c[(c[f+176>>2]|0)+(l*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(l*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]=l;l=l+1|0}c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;l=1;while(1){if((l|0)>=(c[f+132>>2]|0)){break}bK(b|0,e+o+(l<<2)|0,0)|0;bK(b|0,e+(jl(b)|0)|0,0)|0;k=(jd(b)|0)&65535;m=pT((k<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(l<<2)>>2]=m;c[c[(c[f+172>>2]|0)+(l<<2)>>2]>>2]=k;p=0;while(1){if((p|0)>=(k|0)){break}m=ja(b)|0;p=p+(m&3)|0;q=0;r=(c[(c[f+172>>2]|0)+(l<<2)>>2]|0)+4+(p<<3)|0;if((m&4|0)!=0){q=ja(b)|0}if((m&8|0)!=0){a[r+1|0]=ja(b)|0}if((m&16|0)!=0){a[r+2|0]=ja(b)|0}if((m&32|0)!=0){a[r+3|0]=ja(b)|0}if((m&64|0)!=0){a[r+4|0]=ja(b)|0}if((m&128|0)!=0){m=b;ja(m)|0;}if((q&255|0)==1){a[r|0]=-127}else{if((q&255|0)>11){a[r|0]=(q&255)+1&255}}if((d[r+3|0]|0|0)>=16){q=d[r+3|0]|0;if((q|0)==19|(q|0)==20|(q|0)==21|(q|0)==23|(q|0)==28|(q|0)==29|(q|0)==30){a[r+3|0]=14;a[r+4|0]=((a[r+3|0]&15)<<4|a[r+4|0]&15)&255}else{a[r+4|0]=0;a[r+3|0]=0}}do{if((d[r+2|0]|0|0)>=16){if((d[r+2|0]|0|0)>80){s=5379;break}q=r+2|0;a[q]=(d[q]|0)-15&255}else{s=5379}}while(0);if((s|0)==5379){s=0;q=(d[r+2|0]|0)>>4;if((q|0)==12){a[r+5|0]=8;a[r+6|0]=((d[r+2|0]|0)-192<<4)+8&255}else if((q|0)==6){a[r+5|0]=-92;a[r+6|0]=(d[r+2|0]|0)-96&255}else if((q|0)==11){a[r+5|0]=4;a[r+6|0]=(d[r+2|0]|0)-176&255}else if((q|0)==7){a[r+5|0]=-92;a[r+6|0]=(d[r+2|0]|0)-112<<4&255}else if((q|0)==15){a[r+5|0]=3;a[r+6|0]=(d[r+2|0]|0)-240<<4&255}else if((q|0)==14){a[r+5|0]=25;a[r+6|0]=(d[r+2|0]|0)-224&255}else if((q|0)==13){a[r+5|0]=25;a[r+6|0]=(d[r+2|0]|0)-208<<4&255}else if((q|0)==9){a[r+5|0]=14;a[r+6|0]=(160|(d[r+2|0]|0)-144)&255}else if((q|0)==10){a[r+5|0]=4;a[r+6|0]=(d[r+2|0]|0)-160<<4&255}else if((q|0)==8){a[r+5|0]=14;a[r+6|0]=(176|(d[r+2|0]|0)-128)&255}a[r+2|0]=0}p=p+1|0}l=l+1|0}s=pT(523,1)|0;c[c[f+172>>2]>>2]=s;c[c[c[f+172>>2]>>2]>>2]=64;bK(b|0,e+n|0,0)|0;l=0;while(1){if((l|0)>=(c[f+128>>2]|0)){break}n=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(l<<2)>>2]=n;n=(jd(b)|0)&65535;c[c[(c[f+168>>2]|0)+(l<<2)>>2]>>2]=n;p=0;while(1){if((p|0)>=(c[f+136>>2]|0)){break}n=((jd(b)|0)&65535)-1|0;c[(c[(c[f+168>>2]|0)+(l<<2)>>2]|0)+4+(p<<2)>>2]=n;p=p+1|0}l=l+1|0}l=0;while(1){if((l|0)>=(c[f+140>>2]|0)){break}if((c[(c[f+176>>2]|0)+(l*764|0)+36>>2]|0)!=0){p=b;n=e+(c[h+(l<<2)>>2]|0)|0;bK(p|0,n|0,0)|0;n=j;p=b;s=(c[f+180>>2]|0)+((c[(c[(c[f+176>>2]|0)+(l*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(n,p,0,s,0)|0}l=l+1|0}i=g;return 0}function fB(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0;d=i;i=i+8|0;e=d|0;f=a;a=b;b=c;if((bB(e|0,1,4,f|0)|0)>>>0<4){g=-1;h=g;i=d;return h|0}do{if((p6(e|0,25752,4)|0)!=0){if((p6(e|0,24808,4)|0)==0){break}g=-1;h=g;i=d;return h|0}}while(0);bK(f|0,28,1)|0;e=jl(f)|0;if((e|0)!=0){c=f;j=b+e+44|0;bK(c|0,j|0,0)|0;e=jl(f)|0;j=jl(f)|0;c=f;k=b+e|0;bK(c|0,k|0,0)|0;di(f,a,j)}else{di(f,a,0)}g=0;h=g;i=d;return h|0}function fC(f,g,j){f=f|0;g=g|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0;k=i;i=i+1560|0;l=k|0;m=k+56|0;n=k+848|0;o=k+856|0;p=k+864|0;q=k+1400|0;r=k+1424|0;s=k+1512|0;t=k+1520|0;u=f;f=g;g=j;j=u|0;v=0;w=0;bK(f|0,g|0,0)|0;bB(l|0,4,1,f|0)|0;v=(a[l+3|0]|0)-49+1|0;c[l+4>>2]=jl(f)|0;x=jl(f)|0;jd(f)|0;jd(f)|0;y=jl(f)|0;jl(f)|0;z=jl(f)|0;jl(f)|0;A=jl(f)|0;jl(f)|0;b[l+40>>1]=jd(f)|0;b[l+42>>1]=jd(f)|0;b[l+44>>1]=jd(f)|0;b[l+46>>1]=jd(f)|0;b[l+48>>1]=jd(f)|0;a[l+50|0]=ja(f)|0;a[l+51|0]=ja(f)|0;bK(f|0,g+x|0,0)|0;x=0;while(1){if((x|0)>=63){break}b[m+(x<<3)>>1]=jd(f)|0;b[m+(x<<3)+2>>1]=jd(f)|0;a[m+(x<<3)+4|0]=ja(f)|0;a[m+(x<<3)+5|0]=ja(f)|0;a[m+(x<<3)+6|0]=ja(f)|0;a[m+(x<<3)+7|0]=jb(f)|0;x=x+1|0}b[m+504>>1]=jd(f)|0;b[m+506>>1]=jd(f)|0;x=0;while(1){if((x|0)>=256){break}a[m+508+x|0]=ja(f)|0;x=x+1|0}b[m+764>>1]=jd(f)|0;a[m+766|0]=ja(f)|0;a[m+767|0]=ja(f)|0;a[m+768|0]=ja(f)|0;a[m+769|0]=ja(f)|0;x=0;while(1){if((x|0)>=16){break}a[m+770+x|0]=ja(f)|0;x=x+1|0}a[m+786|0]=ja(f)|0;a[m+787|0]=ja(f)|0;c[u+1264>>2]=8363;l=u+1280|0;c[l>>2]=c[l>>2]|((a[m+767|0]&32|0)!=0?0:192);l=a[m+767|0]&64;B=a[m+768|0]&32;C=(a[m+768|0]&31)+1|0;h[u+1256>>3]=2.64;c[j+148>>2]=d[m+769|0]|0;if((l|0)!=0){D=fF(e[m+764>>1]|0)|0}else{if((B|0)!=0){E=(ac(e[m+764>>1]|0,C)|0)/16|0}else{E=e[m+764>>1]|0}D=E}c[j+152>>2]=D;c[j+128>>2]=e[m+504>>1]|0;c[j+140>>2]=d[m+787|0]|0;c[j+156>>2]=e[m+506>>1]|0;c[j+160>>2]=0;c[j+136>>2]=0;D=j+952|0;E=m+508|0;F=c[j+156>>2]|0;p3(D|0,E|0,F)|0;a[j|0]=0;c[j+144>>2]=0;x=0;while(1){if((x|0)>=(c[j+140>>2]|0)){break}bK(f|0,g+z+(x<<2)|0,0)|0;F=jl(f)|0;if((F|0)!=0){E=f;D=g+F|0;bK(E|0,D|0,0)|0;D=f;jl(D)|0;if(((jd(f)|0)<<16>>16|0)==-1){D=f;bK(D|0,14,1)|0;D=(jd(f)|0)&65535;E=j+144|0;c[E>>2]=(c[E>>2]|0)+D}else{D=j+144|0;c[D>>2]=(c[D>>2]|0)+1}}x=x+1|0}b[r+8>>1]=0;b[r+10>>1]=0;b[r+24>>1]=0;b[r+26>>1]=0;D=0;E=0;if((A|0)!=0){F=f;G=g+A|0;bK(F|0,G|0,0)|0;G=f;jl(G)|0;D=jl(f)|0;b[r+8>>1]=jd(f)|0;b[r+10>>1]=jd(f)|0;G=f;jl(G)|0;G=f;jl(G)|0;E=jl(f)|0;b[r+24>>1]=jd(f)|0;b[r+26>>1]=jd(f)|0;G=f;jl(G)|0;G=f;jl(G)|0;G=f;jl(G)|0;G=f;jl(G)|0;G=jl(f)|0;c[r+48>>2]=jl(f)|0;F=f;H=g+G|0;bK(F|0,H|0,0)|0;x=0;while(1){if(x>>>0>=(c[r+48>>2]|0)>>>0){break}if((x|0)>=64){I=5485;break}a[j+x|0]=ja(f)|0;x=x+1|0}}x=0;while(1){if((x|0)>=(c[j+128>>2]|0)){break}bK(f|0,g+y+(x<<2)|0,0)|0;H=jl(f)|0;if((H|0)!=0){F=f;G=g+H|0;bK(F|0,G|0,0)|0;if((v|0)>0){b[n>>1]=jd(f)|0;b[n+2>>1]=jd(f)|0}else{b[n>>1]=(ja(f)|0)&255;b[n+2>>1]=(ja(f)|0)&255}if((e[n>>1]|0)>(c[j+136>>2]|0)){c[j+136>>2]=e[n>>1]|0}}x=x+1|0}c[j+132>>2]=ac(c[j+128>>2]|0,c[j+136>>2]|0)|0;if((v|0)==0){J=(c[j+136>>2]|0)>4?30160:28088}else{J=26808}dm(u,J,(J=i,i=i+1|0,i=i+7>>3<<3,c[J>>2]=0,J)|0);i=J;c[j+172>>2]=pT(4,c[j+132>>2]|0)|0;c[j+168>>2]=pT(4,(c[j+128>>2]|0)+1|0)|0;x=0;while(1){if((x|0)>=(c[j+128>>2]|0)){break}bK(f|0,g+y+(x<<2)|0,0)|0;J=jl(f)|0;if((J|0)!=0){G=f;F=g+J|0;bK(G|0,F|0,0)|0;if((v|0)>0){b[n>>1]=jd(f)|0;b[n+2>>1]=jd(f)|0;F=f;jl(F)|0}else{b[n>>1]=(ja(f)|0)&255;b[n+2>>1]=(ja(f)|0)&255}F=pT(1,((c[j+136>>2]|0)-1<<2)+8|0)|0;c[(c[j+168>>2]|0)+(x<<2)>>2]=F;c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]=(e[n+2>>1]|0)+1;F=0;while(1){if((F|0)>=(c[j+136>>2]|0)){break}G=(ac(x,c[j+136>>2]|0)|0)+F|0;c[(c[(c[j+168>>2]|0)+(x<<2)>>2]|0)+4+(F<<2)>>2]=G;J=pT(((c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[j+172>>2]|0)+(G<<2)>>2]=J;c[c[(c[j+172>>2]|0)+(G<<2)>>2]>>2]=c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2];F=F+1|0}if((v|0)>0){K=0;while(1){if((K|0)>=(c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)){break}L=0;while(1){if((L|0)>=(e[n>>1]|0)){break}a[s|0]=ja(f)|0;a[s+1|0]=ja(f)|0;a[s+2|0]=ja(f)|0;a[s+3|0]=ja(f)|0;M=(c[(c[u+172>>2]|0)+(c[(c[(c[u+168>>2]|0)+(x<<2)>>2]|0)+4+(L<<2)>>2]<<2)>>2]|0)+4+(K<<3)|0;a[M|0]=d[s|0]&127;if((a[M|0]|0)!=0){F=M|0;a[F]=(d[F]|0)+((a[m+766|0]|0)+48)&255}a[M+1|0]=d[s+1|0]&63;do{if((d[M+1|0]|0)!=0){if((a[M|0]|0)!=0){break}a[M+5|0]=-79;a[M+6|0]=a[M+1|0]|0;a[M+1|0]=0}}while(0);a[M+3|0]=a[s+2|0]|0;a[M+4|0]=a[s+3|0]|0;fG(M,B,C,l);L=L+1|0}K=K+1|0}}else{K=0;while(1){if((K|0)>=(c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)){break}L=0;while(1){if((L|0)>=(e[n>>1]|0)){break}a[s|0]=ja(f)|0;a[s+1|0]=ja(f)|0;a[s+2|0]=ja(f)|0;M=(c[(c[u+172>>2]|0)+(c[(c[(c[u+168>>2]|0)+(x<<2)>>2]|0)+4+(L<<2)>>2]<<2)>>2]|0)+4+(K<<3)|0;a[M|0]=d[s|0]&63;if((a[M|0]|0)!=0){F=M|0;a[F]=(d[F]|0)+48&255}a[M+1|0]=(d[s+1|0]>>4|(a[s|0]&128)>>3|(a[s|0]&64)>>1)&255;do{if((d[M+1|0]|0)!=0){if((a[M|0]|0)!=0){break}a[M+5|0]=-79;a[M+6|0]=a[M+1|0]|0;a[M+1|0]=0}}while(0);a[M+3|0]=d[s+1|0]&15;a[M+4|0]=a[s+2|0]|0;fG(M,B,C,l);L=L+1|0}K=K+1|0}}}x=x+1|0}c[u+6548>>2]=pT(4,c[j+140>>2]|0)|0;c[u+6552>>2]=pT(4,c[j+140>>2]|0)|0;c[j+176>>2]=pT(764,c[j+140>>2]|0)|0;if((c[j+144>>2]|0)!=0){c[j+180>>2]=pT(52,c[j+144>>2]|0)|0}x=0;w=0;L6857:while(1){if((x|0)>=(c[j+140>>2]|0)){I=5625;break}p7(t|0,0,40);bK(f|0,g+z+(x<<2)|0,0)|0;l=jl(f)|0;do{if((l|0)!=0){C=f;B=g+l|0;bK(C|0,B|0,0)|0;c[o>>2]=jl(f)|0;b[o+4>>1]=jd(f)|0;B=aT(f|0)|0;do{if((A|0)!=0){if((x|0)>=(e[r+24>>1]|0)){break}C=f;s=E+(ac(x,e[r+26>>1]|0)|0)|0;bK(C|0,s|0,0)|0;s=t|0;C=f;bB(s|0,40,1,C|0)|0}}while(0);a[q+3|0]=0;do{if((A|0)!=0){if((x|0)>=(e[r+8>>1]|0)){break}C=f;s=D+(ac(x,e[r+10>>1]|0)|0)|0;bK(C|0,s|0,0)|0;a[q|0]=ja(f)|0;a[q+1|0]=ja(f)|0;a[q+2|0]=ja(f)|0;a[q+3|0]=ja(f)|0}}while(0);s=f;C=B;bK(s|0,C|0,0)|0;if((b[o+4>>1]|0)==-2){C=aT(f|0)|0;a[p+6|0]=ja(f)|0;s=f;bK(s|0,3,1)|0;b[p+10>>1]=jd(f)|0;b[p+12>>1]=jd(f)|0;b[p+14>>1]=jd(f)|0;b[p+16>>1]=jd(f)|0;a[p+18|0]=ja(f)|0;a[p+19|0]=ja(f)|0;b[p+20>>1]=jd(f)|0;s=p+22|0;n=f;bB(s|0,1,128,n|0)|0;n=p+150|0;s=f;bB(n|0,1,128,s|0)|0;s=f;n=C-6+(jl(f)|0)|0;bK(s|0,n|0,0)|0;n=jl(f)|0;s=f;jd(s)|0;s=pS(8)|0;c[(c[j+176>>2]|0)+(x*764|0)+760>>2]=s;if((c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)==0){I=5593;break L6857}s=pT(64,1)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=s;if((c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)==0){I=5595;break L6857}c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=1;c[c[(c[j+176>>2]|0)+(x*764|0)+760>>2]>>2]=d[p+18|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)+4>>2]=d[p+19|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+8>>2]=128;c[c[(c[j+176>>2]|0)+(x*764|0)+756>>2]>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+12>>2]=a[m+(x<<3)+7|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+16>>2]=a[q+3|0]|0;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=n;c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=e[m+(x<<3)>>1]<<1;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=(c[(c[j+180>>2]|0)+(w*52|0)+36>>2]|0)+(e[m+(x<<3)+2>>1]<<1);c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=(e[m+(x<<3)+2>>1]|0)>1?2:0;n=u;s=f;C=(c[j+180>>2]|0)+(w*52|0)|0;gQ(n,s,0,C,0)|0;w=w+1|0;C=pT(1,e[p+14>>1]|0)|0;c[(c[u+6548>>2]|0)+(x<<2)>>2]=C;C=c[(c[u+6548>>2]|0)+(x<<2)>>2]|0;s=p+22|0;n=e[p+14>>1]|0;p3(C|0,s|0,n)|0;n=pT(1,e[p+16>>1]|0)|0;c[(c[u+6552>>2]|0)+(x<<2)>>2]=n;n=c[(c[u+6552>>2]|0)+(x<<2)>>2]|0;s=p+150|0;C=e[p+16>>1]|0;p3(n|0,s|0,C)|0;break}if((b[o+4>>1]|0)!=-1){if((b[o+4>>1]|0)!=0){break}C=pT(64,1)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=C;c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=1;c[c[(c[j+176>>2]|0)+(x*764|0)+756>>2]>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+12>>2]=a[m+(x<<3)+7|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+16>>2]=a[q+3|0]<<4;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=c[o>>2];c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=e[m+(x<<3)>>1]<<1;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=(c[(c[j+180>>2]|0)+(w*52|0)+36>>2]|0)+(e[m+(x<<3)+2>>1]<<1);c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=0;if((e[m+(x<<3)+2>>1]|0)>1){C=(c[j+180>>2]|0)+(w*52|0)+44|0;c[C>>2]=c[C>>2]|2}C=f;s=g+l+6|0;bK(C|0,s|0,0)|0;s=u;C=f;n=(c[j+180>>2]|0)+(w*52|0)|0;gQ(s,C,0,n,0)|0;w=w+1|0;break}n=aT(f|0)|0;a[p+6|0]=ja(f)|0;C=f;bK(C|0,3,1)|0;b[p+10>>1]=jd(f)|0;b[p+12>>1]=jd(f)|0;b[p+14>>1]=jd(f)|0;b[p+16>>1]=jd(f)|0;a[p+18|0]=ja(f)|0;a[p+19|0]=ja(f)|0;b[p+20>>1]=jd(f)|0;C=p+22|0;s=f;bB(C|0,1,128,s|0)|0;s=p+150|0;C=f;bB(s|0,1,128,C|0)|0;K=0;while(1){if((K|0)>=64){break}c[p+280+(K<<2)>>2]=jl(f)|0;K=K+1|0}if((e[p+20>>1]|0)==65535){break}B=pS(8)|0;c[(c[j+176>>2]|0)+(x*764|0)+760>>2]=B;if((c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)==0){I=5609;break L6857}B=pT(64,e[p+20>>1]|0)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=B;if((c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)==0){I=5611;break L6857}c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=e[p+20>>1]|0;c[c[(c[j+176>>2]|0)+(x*764|0)+760>>2]>>2]=d[p+18|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)+4>>2]=d[p+19|0]|0;K=0;while(1){if((K|0)>=(e[p+20>>1]|0)){break}c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(K<<6)+8>>2]=128;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(K<<6)>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(K<<6)+12>>2]=(a[m+(x<<3)+7|0]|0)-24;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(K<<6)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(K<<6)+16>>2]=a[q+3|0]|0;bK(f|0,n-6+(c[p+280+(K<<2)>>2]|0)|0,0)|0;B=((jd(f)|0)&65535)<<1;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=B;c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=0;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=c[(c[j+180>>2]|0)+(w*52|0)+32>>2];c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=2;gQ(u,f,0,(c[j+180>>2]|0)+(w*52|0)|0,0)|0;w=w+1|0;K=K+1|0}n=pT(1,e[p+14>>1]|0)|0;c[(c[u+6548>>2]|0)+(x<<2)>>2]=n;n=c[(c[u+6548>>2]|0)+(x<<2)>>2]|0;B=p+22|0;C=e[p+14>>1]|0;p3(n|0,B|0,C)|0;C=pT(1,e[p+16>>1]|0)|0;c[(c[u+6552>>2]|0)+(x<<2)>>2]=C;C=c[(c[u+6552>>2]|0)+(x<<2)>>2]|0;B=p+150|0;n=e[p+16>>1]|0;p3(C|0,B|0,n)|0}}while(0);x=x+1|0}if((I|0)==5593){N=-1;O=N;i=k;return O|0}else if((I|0)==5609){N=-1;O=N;i=k;return O|0}else if((I|0)==5611){N=-1;O=N;i=k;return O|0}else if((I|0)==5625){x=0;while(1){if((x|0)>=(c[j+128>>2]|0)){break}K=0;while(1){if((K|0)>=(c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)){break}L=0;while(1){if((L|0)>=(c[j+136>>2]|0)){break}M=(c[(c[u+172>>2]|0)+(c[(c[(c[u+168>>2]|0)+(x<<2)>>2]|0)+4+(L<<2)>>2]<<2)>>2]|0)+4+(K<<3)|0;do{if((a[M|0]|0)!=0){if((a[M+1|0]|0)==0){I=5633;break}if((c[(c[u+6552>>2]|0)+((d[M+1|0]|0)-1<<2)>>2]|0)==0){while(1){if((d[M|0]|0)<=84){break}p=M|0;a[p]=(d[p]|0)-12&255}}}else{I=5633}}while(0);if((I|0)==5633){I=0}L=L+1|0}K=K+1|0}x=x+1|0}x=0;while(1){if((x|0)>=(c[j+136>>2]|0)){break}c[j+184+(x*12|0)+4>>2]=d[m+770+x|0]|0;c[j+184+(x*12|0)>>2]=(((x+1|0)/2|0|0)%2|0)*255|0;x=x+1|0}N=0;O=N;i=k;return O|0}else if((I|0)==5595){N=-1;O=N;i=k;return O|0}return 0}function fD(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0;d=i;i=i+8|0;e=d|0;f=a;a=b;b=c;if((bB(e|0,1,4,f|0)|0)>>>0<4){g=-1;h=g;i=d;return h|0}do{if((p6(e|0,26800,4)|0)!=0){if((p6(e|0,25728,4)|0)==0){break}g=-1;h=g;i=d;return h|0}}while(0);bK(f|0,28,1)|0;e=jl(f)|0;if((e|0)!=0){c=f;j=b+e+44|0;bK(c|0,j|0,0)|0;e=jl(f)|0;j=jl(f)|0;c=f;k=b+e|0;bK(c|0,k|0,0)|0;di(f,a,j)}else{di(f,a,0)}g=0;h=g;i=d;return h|0}function fE(f,g,j){f=f|0;g=g|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0;k=i;i=i+1560|0;l=k|0;m=k+56|0;n=k+848|0;o=k+856|0;p=k+864|0;q=k+1400|0;r=k+1424|0;s=k+1512|0;t=k+1520|0;u=f;f=g;g=j;j=u|0;v=0;w=0;bK(f|0,g|0,0)|0;bB(l|0,4,1,f|0)|0;v=(a[l+3|0]|0)-49+1|0;c[l+4>>2]=jl(f)|0;x=jl(f)|0;jd(f)|0;jd(f)|0;y=jl(f)|0;jl(f)|0;z=jl(f)|0;jl(f)|0;A=jl(f)|0;jl(f)|0;b[l+40>>1]=jd(f)|0;b[l+42>>1]=jd(f)|0;b[l+44>>1]=jd(f)|0;b[l+46>>1]=jd(f)|0;b[l+48>>1]=jd(f)|0;a[l+50|0]=ja(f)|0;a[l+51|0]=ja(f)|0;bK(f|0,g+x|0,0)|0;x=0;while(1){if((x|0)>=63){break}b[m+(x<<3)>>1]=jd(f)|0;b[m+(x<<3)+2>>1]=jd(f)|0;a[m+(x<<3)+4|0]=ja(f)|0;a[m+(x<<3)+5|0]=ja(f)|0;a[m+(x<<3)+6|0]=ja(f)|0;a[m+(x<<3)+7|0]=jb(f)|0;x=x+1|0}b[m+504>>1]=jd(f)|0;b[m+506>>1]=jd(f)|0;l=jl(f)|0;jl(f)|0;B=jl(f)|0;b[m+520>>1]=jd(f)|0;b[m+522>>1]=jd(f)|0;C=jl(f)|0;c[m+528>>2]=jl(f)|0;b[m+532>>1]=jd(f)|0;b[m+534>>1]=jd(f)|0;a[m+536|0]=ja(f)|0;a[m+537|0]=ja(f)|0;b[m+538>>1]=jd(f)|0;a[m+540|0]=ja(f)|0;bK(f|0,223,1)|0;b[m+764>>1]=jd(f)|0;a[m+766|0]=ja(f)|0;a[m+767|0]=ja(f)|0;a[m+768|0]=ja(f)|0;a[m+769|0]=ja(f)|0;x=0;while(1){if((x|0)>=16){break}ja(f)|0;x=x+1|0}a[m+786|0]=ja(f)|0;a[m+787|0]=ja(f)|0;bK(f|0,g+l|0,0)|0;bK(f|0,g+(jl(f)|0)|0,0)|0;bK(f|0,32,1)|0;jl(f)|0;jl(f)|0;c[j+156>>2]=(jd(f)|0)&65535;x=0;while(1){if((x|0)>=(c[j+156>>2]|0)){break}a[j+952+x|0]=(jd(f)|0)&255;x=x+1|0}c[u+1264>>2]=8363;l=u+1280|0;c[l>>2]=c[l>>2]|((a[m+767|0]&32|0)!=0?0:192);l=a[m+767|0]&64;D=a[m+768|0]&32;E=(a[m+768|0]&31)+1|0;h[u+1256>>3]=2.64;c[j+148>>2]=d[m+769|0]|0;if((l|0)!=0){F=fF(e[m+764>>1]|0)|0}else{if((D|0)!=0){G=(ac(e[m+764>>1]|0,E)|0)/16|0}else{G=e[m+764>>1]|0}F=G}c[j+152>>2]=F;c[j+128>>2]=e[m+504>>1]|0;c[j+140>>2]=d[m+787|0]|0;c[j+160>>2]=0;c[j+136>>2]=0;a[j|0]=0;c[j+144>>2]=0;x=0;while(1){if((x|0)>=(c[j+140>>2]|0)){break}bK(f|0,g+z+(x<<2)|0,0)|0;F=jl(f)|0;if((F|0)!=0){G=f;H=g+F|0;bK(G|0,H|0,0)|0;H=f;jl(H)|0;if(((jd(f)|0)<<16>>16|0)==-1){H=f;bK(H|0,14,1)|0;H=(jd(f)|0)&65535;G=j+144|0;c[G>>2]=(c[G>>2]|0)+H}else{H=j+144|0;c[H>>2]=(c[H>>2]|0)+1}}x=x+1|0}b[r+8>>1]=0;b[r+10>>1]=0;b[r+24>>1]=0;b[r+26>>1]=0;H=0;G=0;if((A|0)!=0){F=f;I=g+A|0;bK(F|0,I|0,0)|0;I=f;jl(I)|0;H=jl(f)|0;b[r+8>>1]=jd(f)|0;b[r+10>>1]=jd(f)|0;I=f;jl(I)|0;I=f;jl(I)|0;G=jl(f)|0;b[r+24>>1]=jd(f)|0;b[r+26>>1]=jd(f)|0;I=f;jl(I)|0;I=f;jl(I)|0;I=f;jl(I)|0;I=f;jl(I)|0;I=jl(f)|0;c[r+48>>2]=jl(f)|0;F=f;J=g+I|0;bK(F|0,J|0,0)|0;x=0;while(1){if(x>>>0>=(c[r+48>>2]|0)>>>0){break}if((x|0)>=64){K=5726;break}a[j+x|0]=ja(f)|0;x=x+1|0}}x=0;while(1){if((x|0)>=(c[j+128>>2]|0)){break}bK(f|0,g+y+(x<<2)|0,0)|0;J=jl(f)|0;if((J|0)!=0){F=f;I=g+J|0;bK(F|0,I|0,0)|0;b[n>>1]=jd(f)|0;b[n+2>>1]=jd(f)|0;if((e[n>>1]|0)>(c[j+136>>2]|0)){c[j+136>>2]=e[n>>1]|0}}x=x+1|0}c[j+132>>2]=ac(c[j+128>>2]|0,c[j+136>>2]|0)|0;if((v|0)==2){dm(u,30144,(L=i,i=i+1|0,i=i+7>>3<<3,c[L>>2]=0,L)|0);i=L}else{dm(u,28056,(L=i,i=i+8|0,c[L>>2]=v+48,L)|0);i=L}c[j+172>>2]=pT(4,c[j+132>>2]|0)|0;c[j+168>>2]=pT(4,(c[j+128>>2]|0)+1|0)|0;x=0;while(1){if((x|0)>=(c[j+128>>2]|0)){break}bK(f|0,g+y+(x<<2)|0,0)|0;L=jl(f)|0;if((L|0)!=0){v=f;I=g+L|0;bK(v|0,I|0,0)|0;b[n>>1]=jd(f)|0;b[n+2>>1]=jd(f)|0;I=f;jl(I)|0;I=pT(1,((c[j+136>>2]|0)-1<<2)+8|0)|0;c[(c[j+168>>2]|0)+(x<<2)>>2]=I;c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]=(e[n+2>>1]|0)+1;I=0;while(1){if((I|0)>=(c[j+136>>2]|0)){break}v=(ac(x,c[j+136>>2]|0)|0)+I|0;c[(c[(c[j+168>>2]|0)+(x<<2)>>2]|0)+4+(I<<2)>>2]=v;L=pT(((c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[j+172>>2]|0)+(v<<2)>>2]=L;c[c[(c[j+172>>2]|0)+(v<<2)>>2]>>2]=c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2];I=I+1|0}M=0;while(1){if((M|0)>=(c[c[(c[j+168>>2]|0)+(x<<2)>>2]>>2]|0)){break}I=0;while(1){if((I|0)>=(e[n>>1]|0)){break}a[s|0]=ja(f)|0;a[s+1|0]=ja(f)|0;a[s+2|0]=ja(f)|0;a[s+3|0]=ja(f)|0;v=(c[(c[u+172>>2]|0)+(c[(c[(c[u+168>>2]|0)+(x<<2)>>2]|0)+4+(I<<2)>>2]<<2)>>2]|0)+4+(M<<3)|0;a[v|0]=d[s|0]&127;if((a[v|0]|0)!=0){L=v|0;a[L]=(d[L]|0)+((a[m+766|0]|0)+24)&255}a[v+1|0]=d[s+1|0]&63;do{if((d[v+1|0]|0)!=0){if((a[v|0]|0)!=0){break}a[v+5|0]=-79;a[v+6|0]=a[v+1|0]|0;a[v+1|0]=0}}while(0);a[v+3|0]=a[s+2|0]|0;a[v+4|0]=a[s+3|0]|0;fG(v,D,E,l);I=I+1|0}M=M+1|0}}x=x+1|0}c[u+6548>>2]=pT(4,c[j+140>>2]|0)|0;c[u+6552>>2]=pT(4,c[j+140>>2]|0)|0;c[j+176>>2]=pT(764,c[j+140>>2]|0)|0;if((c[j+144>>2]|0)!=0){c[j+180>>2]=pT(52,c[j+144>>2]|0)|0}x=0;w=0;L7120:while(1){if((x|0)>=(c[j+140>>2]|0)){K=5848;break}p7(t|0,0,40);bK(f|0,g+z+(x<<2)|0,0)|0;l=jl(f)|0;do{if((l|0)!=0){E=f;D=g+l|0;bK(E|0,D|0,0)|0;c[o>>2]=jl(f)|0;b[o+4>>1]=jd(f)|0;D=aT(f|0)|0;do{if((A|0)!=0){if((x|0)>=(e[r+24>>1]|0)){break}E=f;s=G+(ac(x,e[r+26>>1]|0)|0)|0;bK(E|0,s|0,0)|0;s=t|0;E=f;bB(s|0,40,1,E|0)|0}}while(0);a[q+3|0]=0;do{if((A|0)!=0){if((x|0)>=(e[r+8>>1]|0)){break}E=f;s=H+(ac(x,e[r+10>>1]|0)|0)|0;bK(E|0,s|0,0)|0;a[q|0]=ja(f)|0;a[q+1|0]=ja(f)|0;a[q+2|0]=ja(f)|0;a[q+3|0]=ja(f)|0;if((e[r+10>>1]|0)>4){a[q+4|0]=ja(f)|0;a[q+5|0]=ja(f)|0}}}while(0);s=f;E=D;bK(s|0,E|0,0)|0;if((b[o+4>>1]|0)==-2){E=aT(f|0)|0;a[p+6|0]=ja(f)|0;s=f;bK(s|0,3,1)|0;b[p+10>>1]=jd(f)|0;b[p+12>>1]=jd(f)|0;b[p+14>>1]=jd(f)|0;b[p+16>>1]=jd(f)|0;a[p+18|0]=ja(f)|0;a[p+19|0]=ja(f)|0;b[p+20>>1]=jd(f)|0;s=p+22|0;n=f;bB(s|0,1,128,n|0)|0;n=p+150|0;s=f;bB(n|0,1,128,s|0)|0;s=f;n=E-6+(jl(f)|0)|0;bK(s|0,n|0,0)|0;n=jl(f)|0;s=f;jd(s)|0;s=pS(8)|0;c[(c[j+176>>2]|0)+(x*764|0)+760>>2]=s;if((c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)==0){K=5814;break L7120}s=pT(64,1)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=s;if((c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)==0){K=5816;break L7120}c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=1;c[c[(c[j+176>>2]|0)+(x*764|0)+760>>2]>>2]=d[p+18|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)+4>>2]=d[p+19|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+8>>2]=128;c[c[(c[j+176>>2]|0)+(x*764|0)+756>>2]>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+12>>2]=a[m+(x<<3)+7|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+16>>2]=a[q+3|0]|0;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=n;c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=e[m+(x<<3)>>1]<<1;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=(c[(c[j+180>>2]|0)+(w*52|0)+36>>2]|0)+(e[m+(x<<3)+2>>1]<<1);c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=(e[m+(x<<3)+2>>1]|0)>1?2:0;n=u;s=f;E=(c[j+180>>2]|0)+(w*52|0)|0;gQ(n,s,0,E,0)|0;w=w+1|0;E=pT(1,e[p+14>>1]|0)|0;c[(c[u+6548>>2]|0)+(x<<2)>>2]=E;E=c[(c[u+6548>>2]|0)+(x<<2)>>2]|0;s=p+22|0;n=e[p+14>>1]|0;p3(E|0,s|0,n)|0;n=pT(1,e[p+16>>1]|0)|0;c[(c[u+6552>>2]|0)+(x<<2)>>2]=n;n=c[(c[u+6552>>2]|0)+(x<<2)>>2]|0;s=p+150|0;E=e[p+16>>1]|0;p3(n|0,s|0,E)|0;break}if((b[o+4>>1]|0)!=-1){if((b[o+4>>1]&-49|0)!=0){break}E=pT(64,1)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=E;c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=1;c[c[(c[j+176>>2]|0)+(x*764|0)+756>>2]>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+12>>2]=a[m+(x<<3)+7|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+16>>2]=a[q+3|0]<<4;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=c[o>>2];c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=e[m+(x<<3)>>1]<<1;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=(c[(c[j+180>>2]|0)+(w*52|0)+36>>2]|0)+(e[m+(x<<3)+2>>1]<<1);c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=0;if((e[m+(x<<3)+2>>1]|0)>1){E=(c[j+180>>2]|0)+(w*52|0)+44|0;c[E>>2]=c[E>>2]|2}if((b[o+4>>1]&16|0)!=0){E=(c[j+180>>2]|0)+(w*52|0)+44|0;c[E>>2]=c[E>>2]|1;E=(c[j+180>>2]|0)+(w*52|0)+32|0;c[E>>2]=c[E>>2]>>1;E=(c[j+180>>2]|0)+(w*52|0)+36|0;c[E>>2]=c[E>>2]>>1;E=(c[j+180>>2]|0)+(w*52|0)+40|0;c[E>>2]=c[E>>2]>>1}E=f;s=g+l+6|0;bK(E|0,s|0,0)|0;s=u;E=f;n=(c[j+180>>2]|0)+(w*52|0)|0;gQ(s,E,64,n,0)|0;w=w+1|0;break}n=aT(f|0)|0;a[p+6|0]=ja(f)|0;E=f;bK(E|0,3,1)|0;b[p+10>>1]=jd(f)|0;b[p+12>>1]=jd(f)|0;b[p+14>>1]=jd(f)|0;b[p+16>>1]=jd(f)|0;a[p+18|0]=ja(f)|0;a[p+19|0]=ja(f)|0;b[p+20>>1]=jd(f)|0;E=p+22|0;s=f;bB(E|0,1,128,s|0)|0;s=p+150|0;E=f;bB(s|0,1,128,E|0)|0;M=0;while(1){if((M|0)>=64){break}c[p+280+(M<<2)>>2]=jl(f)|0;M=M+1|0}if((e[p+20>>1]|0)==65535){break}D=pS(8)|0;c[(c[j+176>>2]|0)+(x*764|0)+760>>2]=D;if((c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)==0){K=5830;break L7120}D=pT(64,e[p+20>>1]|0)|0;c[(c[j+176>>2]|0)+(x*764|0)+756>>2]=D;if((c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)==0){K=5832;break L7120}c[(c[j+176>>2]|0)+(x*764|0)+36>>2]=e[p+20>>1]|0;c[c[(c[j+176>>2]|0)+(x*764|0)+760>>2]>>2]=d[p+18|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+760>>2]|0)+4>>2]=d[p+19|0]|0;M=0;while(1){if((M|0)>=(e[p+20>>1]|0)){break}c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(M<<6)+8>>2]=128;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(M<<6)>>2]=d[m+(x<<3)+6|0]|0;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(M<<6)+12>>2]=(a[m+(x<<3)+7|0]|0)-24;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(M<<6)+40>>2]=w;c[(c[(c[j+176>>2]|0)+(x*764|0)+756>>2]|0)+(M<<6)+16>>2]=a[q+3|0]|0;bK(f|0,n-6+(c[p+280+(M<<2)>>2]|0)|0,0)|0;D=((jd(f)|0)&65535)<<1;c[(c[j+180>>2]|0)+(w*52|0)+32>>2]=D;c[(c[j+180>>2]|0)+(w*52|0)+36>>2]=0;c[(c[j+180>>2]|0)+(w*52|0)+40>>2]=c[(c[j+180>>2]|0)+(w*52|0)+32>>2];c[(c[j+180>>2]|0)+(w*52|0)+44>>2]=2;gQ(u,f,0,(c[j+180>>2]|0)+(w*52|0)|0,0)|0;w=w+1|0;M=M+1|0}n=pT(1,e[p+14>>1]|0)|0;c[(c[u+6548>>2]|0)+(x<<2)>>2]=n;n=c[(c[u+6548>>2]|0)+(x<<2)>>2]|0;D=p+22|0;E=e[p+14>>1]|0;p3(n|0,D|0,E)|0;E=pT(1,e[p+16>>1]|0)|0;c[(c[u+6552>>2]|0)+(x<<2)>>2]=E;E=c[(c[u+6552>>2]|0)+(x<<2)>>2]|0;D=p+150|0;n=e[p+16>>1]|0;p3(E|0,D|0,n)|0}}while(0);x=x+1|0}if((K|0)==5816){N=-1;O=N;i=k;return O|0}else if((K|0)==5814){N=-1;O=N;i=k;return O|0}else if((K|0)==5830){N=-1;O=N;i=k;return O|0}else if((K|0)==5832){N=-1;O=N;i=k;return O|0}else if((K|0)==5848){bK(f|0,g+B|0,0)|0;x=0;while(1){if((x|0)>=(c[j+136>>2]|0)){break}c[j+184+(x*12|0)+4>>2]=(ja(f)|0)&255;x=x+1|0}if((C|0)!=0){B=f;K=g+C|0;bK(B|0,K|0,0)|0;x=0;while(1){if((x|0)>=(c[j+136>>2]|0)){break}K=(jb(f)|0)<<24>>24<<3;if((K|0)>127){P=127}else{P=K}c[j+184+(x*12|0)>>2]=P+128;x=x+1|0}}else{x=0;while(1){if((x|0)>=(c[j+136>>2]|0)){break}c[j+184+(x*12|0)>>2]=128;x=x+1|0}}N=0;O=N;i=k;return O|0}return 0}function fF(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;if((b|0)<=0){d=b;e=d;return e|0}if((b|0)>10){f=10}else{f=b}b=f;d=c[6632+(b-1<<2)>>2]|0;e=d;return e|0}function fG(b,c,e,f){b=b|0;c=c|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=b;b=c;c=e;e=f;f=d[g+3|0]|0;if((f|0)==0|(f|0)==1|(f|0)==2|(f|0)==3|(f|0)==4|(f|0)==5|(f|0)==6|(f|0)==7){return}else if((f|0)==25){a[g+3|0]=9;return}else if((f|0)==9){a[g+3|0]=15;return}else if((f|0)==21){a[g+3|0]=-90;return}else if((f|0)==13){a[g+3|0]=10;return}else if((f|0)==11|(f|0)==12){return}else if((f|0)==24){a[g+3|0]=14;if((d[g+4|0]|0|0)>15){a[g+4|0]=15}h=g+4|0;a[h]=(d[h]|0|192)&255;return}else if((f|0)==14){a[g+4|0]=0;a[g+3|0]=0;return}else if((f|0)==15){if((d[g+4|0]|0|0)==0){a[g+3|0]=13;return}if((d[g+4|0]|0|0)<=240){a[g+3|0]=-85;if((e|0)!=0){i=fF(d[g+4|0]|0)|0}else{if((b|0)!=0){j=(d[g+4|0]|0|0)/(c|0)|0}else{j=d[g+4|0]|0}i=j}a[g+4|0]=i&255;return}i=d[g+4|0]|0;if((i|0)==241){a[g+3|0]=14;a[g+4|0]=-109}else if((i|0)==242){a[g+3|0]=14;a[g+4|0]=-45}else if((i|0)==243){a[g+3|0]=14;a[g+4|0]=-110}else if((i|0)==248|(i|0)==249|(i|0)==250|(i|0)==251|(i|0)==253|(i|0)==254){a[g+4|0]=0;a[g+3|0]=0}else if((i|0)==255){a[g+3|0]=14;a[g+4|0]=-61}else{a[g+4|0]=0;a[g+3|0]=0}return}else if((f|0)==17){a[g+3|0]=-81;return}else if((f|0)==18){a[g+3|0]=-80;return}else if((f|0)==20){a[g+3|0]=-122;return}else if((f|0)==8){a[g+4|0]=0;a[g+3|0]=0;return}else if((f|0)==22){a[g+3|0]=14;if((d[g+4|0]|0|0)>15){a[g+4|0]=15}i=g+4|0;a[i]=(d[i]|0|96)&255;return}else if((f|0)==26){a[g+3|0]=-83;return}else if((f|0)==27){a[g+3|0]=-82;return}else{a[g+4|0]=0;a[g+3|0]=0;return}}function fH(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;f=i;i=i+80|0;g=f|0;h=f+8|0;j=b;b=d;d=e;bK(j|0,d+1080|0,0)|0;if((bB(g|0,1,4,j|0)|0)>>>0<4){k=-1;l=k;i=f;return l|0}L7297:do{if((aL(g+2|0,28368,2)|0)!=0){m=5952}else{if((bF(a[g|0]|0)|0)==0){m=5952;break}if((bF(a[g+1|0]|0)|0)==0){m=5952;break}n=(((a[g|0]|0)-48|0)*10|0)+(a[g+1|0]|0)-48|0;do{if((n|0)>0){if((n|0)>32){break}break L7297}}while(0);m=5952}}while(0);L7305:do{if((m|0)==5952){do{if((aL(g+1|0,28184,3)|0)==0){if((bF(a[g|0]|0)|0)==0){break}if(((a[g|0]|0)-48|0)!=0){break L7305}else{break}}}while(0);n=0;while(1){if((c[6352+(n*20|0)>>2]|0)==0){break}if((p6(g|0,c[6336+(n*20|0)>>2]|0,4)|0)==0){m=5960;break}n=n+1|0}if((c[6352+(n*20|0)>>2]|0)==0){k=-1;l=k;i=f;return l|0}bK(j|0,d+20|0,0)|0;n=0;while(1){if((n|0)>=31){m=5979;break}bK(j|0,22,1)|0;if(((jd(j)|0)&32768|0)!=0){m=5968;break}if(((ja(j)|0)&240|0)!=0){m=5970;break}if(((ja(j)|0)&255|0)>64){m=5972;break}if(((jd(j)|0)&32768|0)!=0){m=5974;break}if(((jd(j)|0)&32768|0)!=0){m=5976;break}n=n+1|0}if((m|0)==5968){k=-1;l=k;i=f;return l|0}else if((m|0)==5974){k=-1;l=k;i=f;return l|0}else if((m|0)==5970){k=-1;l=k;i=f;return l|0}else if((m|0)==5972){k=-1;l=k;i=f;return l|0}else if((m|0)==5979){aC(aE(j|0)|0,h|0)|0;e=0;bK(j|0,d+20|0,0)|0;n=0;while(1){if((n|0)>=31){break}bK(j|0,22,1)|0;e=e+(((jd(j)|0)&65535)<<1)|0;bK(j|0,6,1)|0;n=n+1|0}o=0;bK(j|0,d+952|0,0)|0;n=0;while(1){if((n|0)>=128){break}p=ja(j)|0;if((p&255|0)>127){m=5986;break}if((p&255|0)>(o|0)){o=p&255}n=n+1|0}o=o+1|0;if((d+1084+(o*768|0)+e|0)!=(c[h+28>>2]|0)){break}k=-1;l=k;i=f;return l|0}else if((m|0)==5976){k=-1;l=k;i=f;return l|0}}}while(0);bK(j|0,d|0,0)|0;di(j,b,20);k=0;l=k;i=f;return l|0}function fI(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0;j=i;i=i+5488|0;k=j|0;l=j+1088|0;m=j+1096|0;n=j+5192|0;o=j+5200|0;p=j+5232|0;q=f;f=g;g=h;h=q|0;r=0;p7(m|0,0,4096);s=36776;t=0;u=0;bK(f|0,g|0,0)|0;c[h+140>>2]=31;c[h+144>>2]=c[h+140>>2];c[h+136>>2]=0;v=0;w=0;x=q+1280|0;c[x>>2]=c[x>>2]|8192;bB(k|0,20,1,f|0)|0;x=0;while(1){if((x|0)>=31){break}bB(k+20+(x*30|0)|0,22,1,f|0)|0;b[k+20+(x*30|0)+22>>1]=jd(f)|0;a[k+20+(x*30|0)+24|0]=ja(f)|0;a[k+20+(x*30|0)+25|0]=ja(f)|0;b[k+20+(x*30|0)+26>>1]=jd(f)|0;b[k+20+(x*30|0)+28>>1]=jd(f)|0;v=v+(e[k+20+(x*30|0)+22>>1]<<1)|0;x=x+1|0}a[k+950|0]=ja(f)|0;a[k+951|0]=ja(f)|0;bB(k+952|0,128,1,f|0)|0;p7(n|0,0,8);bB(n|0,4,1,f|0)|0;x=0;while(1){if((c[6352+(x*20|0)>>2]|0)==0){break}if((aL(n|0,c[6336+(x*20|0)>>2]|0,4)|0)==0){y=6014;break}x=x+1|0}if((y|0)==6014){c[h+136>>2]=c[6352+(x*20|0)>>2];s=c[6348+(x*20|0)>>2]|0;t=c[6340+(x*20|0)>>2]|0;u=c[6344+(x*20|0)>>2]|0}if((c[h+136>>2]|0)==0){do{if((aL(n+2|0,28368,2)|0)!=0){y=6022}else{if((bF(a[n|0]|0)|0)==0){y=6022;break}if((bF(a[n+1|0]|0)|0)==0){y=6022;break}c[h+136>>2]=(((a[n|0]|0)-48|0)*10|0)+(a[n+1|0]|0)-48}}while(0);L7391:do{if((y|0)==6022){do{if((aL(n+1|0,28184,3)|0)==0){if((bF(a[n|0]|0)|0)==0){break}c[h+136>>2]=(a[n|0]|0)-48;break L7391}}while(0);z=-1;A=z;i=j;return A|0}}while(0);s=28016;t=1;B=q+1280|0;c[B>>2]=c[B>>2]&-8193}p5(h|0,k|0,20)|0;c[h+156>>2]=d[k+950|0]|0;if((c[h+160>>2]|0)>=(c[h+156>>2]|0)){c[h+160>>2]=0}B=h+952|0;C=k+952|0;p3(B|0,C|0,128)|0;x=0;while(1){if((x|0)>=128){break}if((d[h+952+x|0]|0)>127){y=6033;break}if((d[h+952+x|0]|0)>(c[h+128>>2]|0)){c[h+128>>2]=d[h+952+x|0]|0}x=x+1|0}C=h+128|0;c[C>>2]=(c[C>>2]|0)+1;w=ac(c[h+136>>2]<<8,c[h+128>>2]|0)|0;c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}x=0;while(1){if((x|0)>=(c[h+140>>2]|0)){break}w=pT(64,1)|0;c[(c[h+176>>2]|0)+(x*764|0)+756>>2]=w;c[(c[h+180>>2]|0)+(x*52|0)+32>>2]=e[k+20+(x*30|0)+22>>1]<<1;c[(c[h+180>>2]|0)+(x*52|0)+36>>2]=e[k+20+(x*30|0)+26>>1]<<1;c[(c[h+180>>2]|0)+(x*52|0)+40>>2]=(c[(c[h+180>>2]|0)+(x*52|0)+36>>2]|0)+(e[k+20+(x*30|0)+28>>1]<<1);if((c[(c[h+180>>2]|0)+(x*52|0)+40>>2]|0)>(c[(c[h+180>>2]|0)+(x*52|0)+32>>2]|0)){c[(c[h+180>>2]|0)+(x*52|0)+40>>2]=c[(c[h+180>>2]|0)+(x*52|0)+32>>2]}if((e[k+20+(x*30|0)+28>>1]|0)>1){D=(c[(c[h+180>>2]|0)+(x*52|0)+40>>2]|0)>8}else{D=0}c[(c[h+180>>2]|0)+(x*52|0)+44>>2]=D?2:0;c[(c[(c[h+176>>2]|0)+(x*764|0)+756>>2]|0)+16>>2]=(a[k+20+(x*30|0)+24|0]<<4&255)<<24>>24;c[c[(c[h+176>>2]|0)+(x*764|0)+756>>2]>>2]=a[k+20+(x*30|0)+25|0]|0;c[(c[(c[h+176>>2]|0)+(x*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]=x;c[(c[h+176>>2]|0)+(x*764|0)+36>>2]=((c[(c[h+180>>2]|0)+(x*52|0)+32>>2]|0)!=0^1^1)&1;dh((c[h+176>>2]|0)+(x*764|0)|0,k+20+(x*30|0)|0,22)|0;x=x+1|0}L7431:do{if((t|0)==0){D=((ac(c[h+128>>2]<<2,c[h+136>>2]|0)|0)<<6)+1084+v|0;do{if((D|0)<(c[q+1240>>2]|0)){w=aT(f|0)|0;C=f;B=g+1084+((ac(c[h+128>>2]<<2,c[h+136>>2]|0)|0)<<6)+v|0;bK(C|0,B|0,0)|0;B=o|0;C=f;bB(B|0,1,4,C|0)|0;C=f;B=g+w|0;bK(C|0,B|0,0)|0;if((p6(o|0,27848,4)|0)!=0){break}else{s=27624;u=0;break L7431}}}while(0);if((aL(n|0,26352,4)|0)!=0){E=0}else{E=((c[h+128>>2]<<5<<6)+1084+v|0)==(c[q+1240>>2]|0)}if(E){c[h+136>>2]=8;s=27584;u=0;break}if((aL(n|0,26352,4)|0)!=0){F=0}else{F=((c[h+128>>2]<<10)+1084|0)==(c[q+1240>>2]|0)}r=F&1;if(F){s=30104;break}do{if((c[h+136>>2]|0)==4){if((d[k+951|0]|0)!=(c[h+128>>2]|0)){y=6068;break}s=27440}else{y=6068}}while(0);if((y|0)==6068){do{if((c[h+136>>2]|0)==4){if((d[k+951|0]|0)!=120){y=6071;break}s=25712;u=1}else{y=6071}}while(0);if((y|0)==6071){if((d[k+951|0]|0)<127){if((c[h+136>>2]|0)==4){s=25712;u=1}else{s=27288;u=0}c[h+160>>2]=d[k+951|0]|0}}}do{if((c[h+136>>2]|0)!=4){if((d[k+951|0]|0)!=127){break}s=27192;u=0;D=q+1280|0;c[D>>2]=c[D>>2]&-8193;c[q+1284>>2]=2}}while(0);do{if((c[h+136>>2]|0)==4){if((d[k+951|0]|0)!=127){break}x=0;while(1){if((x|0)>=31){break}if((e[k+20+(x*30|0)+28>>1]|0)==0){y=6086;break}x=x+1|0}if((x|0)<31){s=27168;u=0}}}while(0);do{if((d[k+951|0]|0)!=120){if((d[k+951|0]|0)>=127){break}x=0;while(1){if((x|0)>=31){break}if((e[k+20+(x*30|0)+28>>1]|0)==0){y=6097;break}x=x+1|0}L7501:do{if((x|0)==31){x=0;while(1){if((x|0)>=31){break}if((e[k+20+(x*30|0)+22>>1]|0)==1){if((a[k+20+(x*30|0)+25|0]|0)==0){y=6105;break}}x=x+1|0}if((y|0)==6105){s=27024;u=0;break L7431}x=0;while(1){if((x|0)>=31){break}if((fJ(k+20+(x*30|0)|0)|0)!=0){y=6111;break}x=x+1|0}if((x|0)==31){x=0;while(1){if((x|0)>=31){break}if((e[k+20+(x*30|0)+22>>1]|0)==0){if((e[k+20+(x*30|0)+28>>1]|0)==1){y=6119;break}}x=x+1|0}if((y|0)==6119){D=c[h+136>>2]|0;if((D|0)==4){s=25712}else if((D|0)==6|(D|0)==8){s=31104;u=0}else{s=27288;u=0}break L7431}if((c[h+136>>2]|0)==4){s=30104}else{do{if((c[h+136>>2]|0)==6){y=6130}else{if((c[h+136>>2]|0)==8){y=6130;break}s=27288;u=0}}while(0);if((y|0)==6130){s=26872;u=0;D=q+1280|0;c[D>>2]=c[D>>2]&-8193}}}}else{x=15;while(1){if((x|0)>=31){break}if((p4(k+20+(x*30|0)|0)|0)!=0){y=6139;break}if((e[k+20+(x*30|0)+22>>1]|0)>0){y=6139;break}x=x+1|0}do{if((x|0)==31){if((fJ(k+440|0)|0)==0){break}s=26744;u=0;break L7431}}while(0);x=0;while(1){if((x|0)>=31){break}if((fJ(k+20+(x*30|0)|0)|0)!=0){y=6148;break}x=x+1|0}if((x|0)<31){s=26680;u=0;break L7431}do{if((c[h+136>>2]|0)!=4){if((c[h+136>>2]|0)==6){break}if((c[h+136>>2]|0)==8){break}s=27288;u=0;break L7501}}while(0);s=23584;u=0;D=q+1280|0;c[D>>2]=c[D>>2]&-8193;break L7431}}while(0)}}while(0)}}while(0);c[h+132>>2]=ac(c[h+136>>2]|0,c[h+128>>2]|0)|0;aM(h+64|0,64,26464,(y=i,i=i+16|0,c[y>>2]=s,c[y+8>>2]=n,y)|0)|0;i=y;x=0;while(1){if((x|0)>=(c[h+140>>2]|0)){break}x=x+1|0}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;x=0;while(1){if((x|0)>=(c[h+128>>2]|0)){break}n=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(x<<2)>>2]=n;c[c[(c[h+168>>2]|0)+(x<<2)>>2]>>2]=64;n=0;while(1){if((n|0)>=(c[h+136>>2]|0)){break}k=(ac(x,c[h+136>>2]|0)|0)+n|0;c[(c[(c[h+168>>2]|0)+(x<<2)>>2]|0)+4+(n<<2)>>2]=k;F=pT(((c[c[(c[h+168>>2]|0)+(x<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(k<<2)>>2]=F;c[c[(c[h+172>>2]|0)+(k<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(x<<2)>>2]>>2];n=n+1|0}n=0;while(1){if((n|0)>=(c[h+136>>2]<<6|0)){break}k=(c[(c[q+172>>2]|0)+(c[(c[(c[q+168>>2]|0)+(x<<2)>>2]|0)+4+(((n|0)%(c[h+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((n|0)/(c[h+136>>2]|0)|0)<<3)|0;bB(l|0,1,4,f|0)|0;dj(k,l|0);n=n+1|0}x=x+1|0}l=aW(c[q+1216>>2]|0,47)|0;if((l|0)!=0){n=m|0;k=c[q+1216>>2]|0;F=l-(c[q+1216>>2]|0)|0;p5(n|0,k|0,F|0)|0}x=0;while(1){if((x|0)>=(c[h+144>>2]|0)){break}if((c[(c[h+180>>2]|0)+(x*52|0)+32>>2]|0)!=0){F=(u|0)!=0?512:0;if((r|0)!=0){k=p|0;n=m|0;l=(c[h+176>>2]|0)+(x*764|0)|0;aM(k|0,64,26416,(y=i,i=i+16|0,c[y>>2]=n,c[y+8>>2]=l,y)|0)|0;i=y;l=bh(p|0,26264)|0;if((l|0)!=0){n=q;k=l;l=F;E=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(n,k,l,E,0)|0}}else{E=q;l=f;k=F;F=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(x*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(E,l,k,F,0)|0}}x=x+1|0}if((c[h+136>>2]|0)>4){h=q+1280|0;c[h>>2]=c[h>>2]&-8193;h=q+1280|0;c[h>>2]=c[h>>2]|32;c[q+1284>>2]=1}else{if((a7(s|0,30104)|0)==0){s=q+1280|0;c[s>>2]=c[s>>2]|4}}z=0;A=z;i=j;return A|0}function fJ(b){b=b|0;var c=0,d=0,e=0;c=b;do{if((a[c|0]|0)!=115){if((a[c|0]|0)==83){break}d=0;e=d;return e|0}}while(0);do{if((a[c+1|0]|0)!=116){if((a[c+1|0]|0)==84){break}d=0;e=d;return e|0}}while(0);do{if((a[c+2|0]|0)==45){if((a[c+5|0]|0)!=58){break}do{if((bF(a[c+3|0]|0)|0)!=0){if((bF(a[c+4|0]|0)|0)==0){break}d=1;e=d;return e|0}}while(0);d=0;e=d;return e|0}}while(0);d=0;e=d;return e|0}function fK(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0;c=i;i=i+8|0;e=c|0;f=a;a=b;do{if((bB(e|0,1,4,f|0)|0)>>>0<4){g=-1}else{if((p6(e|0,28008,3)|0)!=0){g=-1;break}if((d[e+3|0]|0|0)!=16){g=-1;break}else{di(f,a,20);g=0;break}}}while(0);i=c;return g|0}function fL(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;j=i;i=i+368|0;k=j|0;l=j+72|0;m=j+112|0;n=j+304|0;o=f;f=g;g=o|0;bK(f|0,h|0,0)|0;bB(k|0,3,1,f|0)|0;a[k+3|0]=ja(f)|0;bB(k+4|0,20,1,f|0)|0;b[k+24>>1]=jc(f)|0;a[k+26|0]=ja(f)|0;a[k+27|0]=ja(f)|0;b[k+28>>1]=jc(f)|0;a[k+30|0]=ja(f)|0;a[k+31|0]=ja(f)|0;a[k+32|0]=ja(f)|0;a[k+33|0]=ja(f)|0;bB(k+34|0,32,1,f|0)|0;c[g+132>>2]=(e[k+24>>1]|0)+1;c[g+128>>2]=(d[k+26|0]|0)+1;c[g+156>>2]=(d[k+27|0]|0)+1;c[g+140>>2]=d[k+30|0]|0;c[g+144>>2]=c[g+140>>2];c[g+136>>2]=d[k+33|0]|0;c[g+148>>2]=6;c[g+152>>2]=125;p5(g|0,k+4|0,20)|0;h=a[k+3|0]&15;dm(o,30040,(p=i,i=i+16|0,c[p>>2]=(a[k+3|0]&240)>>4,c[p+8>>2]=h,p)|0);i=p;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}p=0;while(1){if((p|0)>=(c[g+140>>2]|0)){break}h=pT(64,1)|0;c[(c[g+176>>2]|0)+(p*764|0)+756>>2]=h;bB(l|0,22,1,f|0)|0;c[l+24>>2]=jg(f)|0;c[l+28>>2]=jg(f)|0;c[l+32>>2]=jg(f)|0;a[l+36|0]=ja(f)|0;a[l+37|0]=ja(f)|0;a[l+38|0]=ja(f)|0;c[(c[g+180>>2]|0)+(p*52|0)+32>>2]=c[l+24>>2];c[(c[g+176>>2]|0)+(p*764|0)+36>>2]=(c[l+24>>2]|0)>>>0>0?1:0;c[(c[g+180>>2]|0)+(p*52|0)+36>>2]=c[l+28>>2];c[(c[g+180>>2]|0)+(p*52|0)+40>>2]=c[l+32>>2];c[(c[g+180>>2]|0)+(p*52|0)+44>>2]=(c[(c[g+180>>2]|0)+(p*52|0)+40>>2]|0)!=0?2:0;if((a[k+31|0]&1|0)!=0){h=(c[g+180>>2]|0)+(p*52|0)+44|0;c[h>>2]=c[h>>2]|1;h=(c[g+180>>2]|0)+(p*52|0)+32|0;c[h>>2]=c[h>>2]>>1;h=(c[g+180>>2]|0)+(p*52|0)+36|0;c[h>>2]=c[h>>2]>>1;h=(c[g+180>>2]|0)+(p*52|0)+40|0;c[h>>2]=c[h>>2]>>1}c[c[(c[g+176>>2]|0)+(p*764|0)+756>>2]>>2]=d[l+37|0]|0;c[(c[(c[g+176>>2]|0)+(p*764|0)+756>>2]|0)+16>>2]=d[l+36|0]|0;c[(c[(c[g+176>>2]|0)+(p*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]=p;dh((c[g+176>>2]|0)+(p*764|0)|0,l|0,22)|0;p=p+1|0}bB(g+952|0,1,128,f|0)|0;c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;p=0;while(1){if((p|0)>=(c[g+132>>2]|0)){break}l=pT(((d[k+32|0]|0)<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(p<<2)>>2]=l;c[c[(c[g+172>>2]|0)+(p<<2)>>2]>>2]=d[k+32|0]|0;if((p|0)!=0){l=m;h=f;bB(l|0,3,64,h|0)|0;q=0;while(1){if((q|0)>=64){break}h=(d[m+(q*3|0)|0]|0)>>2&255;a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)|0]=h;if(h<<24>>24!=0){h=(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)|0;a[h]=(d[h]|0)+37&255}a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+1|0]=((a[m+(q*3|0)|0]&3)<<4)+((a[m+((q*3|0)+1)|0]&240)>>4)&255;a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+3|0]=(d[m+((q*3|0)+1)|0]|0)&15;a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+4|0]=a[m+((q*3|0)+2)|0]|0;if((d[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+3|0]|0|0)>15){a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+4|0]=0;a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+3|0]=0}do{if((d[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+3|0]|0|0)==14){if(((a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+4|0]&240)>>4|0)!=8){break}a[(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+3|0]=8;h=(c[(c[g+172>>2]|0)+(p<<2)>>2]|0)+4+(q<<3)+4|0;a[h]=(d[h]|0)<<4&255}}while(0);q=q+1|0}}p=p+1|0}p=0;while(1){if((p|0)>=(c[g+128>>2]|0)){break}m=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(p<<2)>>2]=m;c[c[(c[g+168>>2]|0)+(p<<2)>>2]>>2]=64;q=0;while(1){if((q|0)>=32){break}b[n+(q<<1)>>1]=jc(f)|0;q=q+1|0}q=0;while(1){if((q|0)>=(c[g+136>>2]|0)){break}c[(c[(c[g+168>>2]|0)+(p<<2)>>2]|0)+4+(q<<2)>>2]=e[n+(q<<1)>>1]|0;q=q+1|0}p=p+1|0}bK(f|0,e[k+28>>1]|0|0,1)|0;p=0;while(1){if((p|0)>=(c[g+140>>2]|0)){break}gQ(o,f,2,(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;p=p+1|0}p=0;while(1){if((p|0)>=(c[g+136>>2]|0)){break}c[g+184+(p*12|0)>>2]=(d[k+34+p|0]|0)<<4;p=p+1|0}i=j;return 0}function fM(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1313800192){d=-1;e=d;return e|0}else{di(c,b,(ja(c)|0)&255);d=0;e=d;return e|0}return 0}function fN(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;f=i;g=b;b=d;d=g|0;bK(b|0,e|0,0)|0;jl(b)|0;dm(g,30024,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0);i=e;e=(ja(b)|0)&255;h=0;while(1){if((h|0)>=(e|0)){break}j=ja(b)|0;if((h|0)<64){a[d+h|0]=j}h=h+1|0}jc(b)|0;jc(b)|0;jc(b)|0;jc(b)|0;ja(b)|0;c[d+128>>2]=(ja(b)|0)&255;ja(b)|0;c[d+136>>2]=(ja(b)|0)&255;c[d+132>>2]=ac(c[d+128>>2]|0,c[d+136>>2]|0)|0;ja(b)|0;jc(b)|0;jc(b)|0;ja(b)|0;c[d+144>>2]=63;c[d+140>>2]=63;h=0;while(1){if((h|0)>=256){break}j=ja(b)|0;if((j&255|0)==255){k=6328;break}a[d+952+h|0]=j;h=h+1|0}bK(b|0,255-h|0,1)|0;c[d+156>>2]=h;c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}h=0;while(1){if((h|0)>=(c[d+140>>2]|0)){break}j=pT(64,1)|0;c[(c[d+176>>2]|0)+(h*764|0)+756>>2]=j;e=(ja(b)|0)&255;j=0;l=0;while(1){if((l|0)>=(e|0)){break}k=ja(b)|0;if((k&255|0)!=32){j=1}if((l|0)<32){a[(c[d+176>>2]|0)+(h*764|0)+l|0]=k}l=l+1|0}if((j|0)==0){a[(c[d+176>>2]|0)+(h*764|0)|0]=0}jg(b)|0;jg(b)|0;k=(ja(b)|0)&255;c[c[(c[d+176>>2]|0)+(h*764|0)+756>>2]>>2]=k;k=(jc(b)|0)&65535;m=(jc(b)|0)&65535;c[(c[d+180>>2]|0)+(h*52|0)+32>>2]=m;m=(jc(b)|0)&65535;c[(c[d+180>>2]|0)+(h*52|0)+36>>2]=m;m=(jc(b)|0)&65535;c[(c[d+180>>2]|0)+(h*52|0)+40>>2]=m;jg(b)|0;jc(b)|0;c[(c[d+176>>2]|0)+(h*764|0)+36>>2]=((c[(c[d+180>>2]|0)+(h*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[d+180>>2]|0)+(h*52|0)+36>>2]=0;c[(c[d+180>>2]|0)+(h*52|0)+40>>2]=0;c[(c[d+180>>2]|0)+(h*52|0)+44>>2]=(c[(c[d+180>>2]|0)+(h*52|0)+40>>2]|0)>0?2:0;c[(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+16>>2]=0;c[(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]=h;k=(k*8363|0|0)/8448|0;hD(k,(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+12|0,(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+16|0);h=h+1|0}c[d+172>>2]=pT(4,c[d+132>>2]|0)|0;c[d+168>>2]=pT(4,(c[d+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[d+128>>2]|0)){break}e=pT(1,((c[d+136>>2]|0)-1<<2)+8|0)|0;c[(c[d+168>>2]|0)+(h<<2)>>2]=e;c[c[(c[d+168>>2]|0)+(h<<2)>>2]>>2]=64;e=0;while(1){if((e|0)>=(c[d+136>>2]|0)){break}k=(ac(h,c[d+136>>2]|0)|0)+e|0;c[(c[(c[d+168>>2]|0)+(h<<2)>>2]|0)+4+(e<<2)>>2]=k;m=pT(((c[c[(c[d+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[d+172>>2]|0)+(k<<2)>>2]=m;c[c[(c[d+172>>2]|0)+(k<<2)>>2]>>2]=c[c[(c[d+168>>2]|0)+(h<<2)>>2]>>2];e=e+1|0}l=0;while(1){if((l|0)>=(c[c[(c[d+168>>2]|0)+(h<<2)>>2]>>2]|0)){break}e=0;while(1){if((e|0)>=(c[d+136>>2]|0)){break}j=(c[(c[g+172>>2]|0)+(c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(e<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0;k=jg(b)|0;m=k&63;n=(k&8128)>>>6;o=(k&1040384)>>>13;p=(k&15728640)>>>20;if((m|0)!=63){a[j|0]=m+36&255}if((n|0)!=127){a[j+1|0]=n+1&255}if((o|0)!=127){a[j+2|0]=o&255}if((p|0)!=15){a[j+3|0]=a[8736+p|0]|0;a[j+4|0]=(k&-16777216)>>>24&255}e=e+1|0}l=l+1|0}h=h+1|0}h=0;while(1){if((h|0)>=(c[d+140>>2]|0)){break}if((c[(c[d+180>>2]|0)+(h*52|0)+32>>2]|0)!=0){l=g;e=b;k=(c[d+180>>2]|0)+((c[(c[(c[d+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(l,e,2,k,0)|0}h=h+1|0}h=g+1280|0;c[h>>2]=c[h>>2]|545;c[g+1284>>2]=2;i=f;return 0}function fO(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+8|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,8,e|0)|0)>>>0<8){f=-1}else{if((aL(d|0,22416,8)|0)!=0){f=-1;break}else{di(e,a,0);f=0;break}}}while(0);i=c;return f|0}function fP(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+296|0;f=e|0;g=a;a=b;bK(a|0,d|0,0)|0;bK(a|0,8,1)|0;d=es()|0;if((d|0)==0){h=-1;j=h;i=e;return j|0}p7(f|0,0,296);ev(d,29992,38);ev(d,28e3,52);ev(d,26736,68);ev(d,25704,86);ev(d,24760,50);ev(d,23968,130);ev(d,23576,118);ev(d,22976,106);dm(g,26040,(b=i,i=i+1|0,i=i+7>>3<<3,c[b>>2]=0,b)|0);i=b;while(1){if(!((bH(a|0)|0)!=0^1)){break}et(d,g,a,f)}eB(d);h=0;j=h;i=e;return j|0}function fQ(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;e=d;d=a|0;c[d+136>>2]=0;a=0;while(1){if((a|0)>=4){break}b=(((jd(e)|0)&65535|0)!=0^1^1)&1;while(1){if((b|0)<0){break}c[d+184+((c[d+136>>2]|0)*12|0)>>2]=(((a+1|0)/2|0|0)%2|0)*255|0;f=d+136|0;c[f>>2]=(c[f>>2]|0)+1;b=b-1|0}a=a+1|0}return}function fR(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0;f=d;d=a|0;a=e;c[d+140>>2]=(b|0)/32|0;c[d+144>>2]=c[d+140>>2];c[d+176>>2]=pT(764,c[d+140>>2]|0)|0;if((c[d+144>>2]|0)!=0){c[d+180>>2]=pT(52,c[d+144>>2]|0)|0}b=0;e=0;while(1){if((b|0)>=(c[d+140>>2]|0)){break}g=pT(64,1)|0;c[(c[d+176>>2]|0)+(b*764|0)+756>>2]=g;bB((c[d+176>>2]|0)+(b*764|0)|0,1,20,f|0)|0;hy((c[d+176>>2]|0)+(b*764|0)|0)|0;g=(jl(f)|0)&-2;c[(c[d+180>>2]|0)+(e*52|0)+32>>2]=g;g=(jd(f)|0)&65535;c[(c[d+180>>2]|0)+(e*52|0)+36>>2]=g;g=(jd(f)|0)&65535;c[(c[d+180>>2]|0)+(e*52|0)+40>>2]=(c[(c[d+180>>2]|0)+(e*52|0)+36>>2]|0)+g;c[(c[d+180>>2]|0)+(e*52|0)+44>>2]=(g|0)>2?2:0;g=(jd(f)|0)&65535;c[c[(c[d+176>>2]|0)+(b*764|0)+756>>2]>>2]=g;c[a+(b<<2)>>2]=(jd(f)|0)&65535;c[(c[d+176>>2]|0)+(b*764|0)+36>>2]=((c[(c[d+180>>2]|0)+(e*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+40>>2]=e;c[a+144+(e<<2)>>2]=b;if((c[(c[d+176>>2]|0)+(b*764|0)+36>>2]|0)!=0){e=e+1|0}b=b+1|0}return}function fS(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;c[e+148>>2]=(jd(d)|0)&65535;c[e+152>>2]=125;return}function fT(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;c[e+128>>2]=(jd(d)|0)&65535;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;return}function fU(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;c[a+156>>2]=(jd(d)|0)&65535;return}function fV(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=a|0;bB(e+952|0,1,c[e+156>>2]|0,d|0)|0;return}function fW(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0;e=b;b=f;f=e|0;h=g;if((c[h+288>>2]|0)>=(c[f+128>>2]|0)){return}if((c[h+288>>2]|0)==0){c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0}g=jd(b)|0;i=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(c[h+288>>2]<<2)>>2]=i;c[c[(c[f+168>>2]|0)+(c[h+288>>2]<<2)>>2]>>2]=g&65535;i=0;while(1){if((i|0)>=(c[f+136>>2]|0)){break}j=(ac(c[h+288>>2]|0,c[f+136>>2]|0)|0)+i|0;c[(c[(c[f+168>>2]|0)+(c[h+288>>2]<<2)>>2]|0)+4+(i<<2)>>2]=j;k=pT(((c[c[(c[f+168>>2]|0)+(c[h+288>>2]<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(j<<2)>>2]=k;c[c[(c[f+172>>2]|0)+(j<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(c[h+288>>2]<<2)>>2]>>2];i=i+1|0}i=0;while(1){if((i|0)>=(ac(g&65535,c[f+136>>2]|0)|0)){break}j=(c[(c[e+172>>2]|0)+(c[(c[(c[e+168>>2]|0)+(c[h+288>>2]<<2)>>2]|0)+4+(((i|0)%(c[f+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((i|0)/(c[f+136>>2]|0)|0)<<3)|0;p7(j|0,0,8);k=ja(b)|0;l=ja(b)|0;if(k<<24>>24!=0){a[j|0]=(k&255)+48&255;a[j+1|0]=(l&255)+1&255}a[j+3|0]=c[8608+(((ja(b)|0)&255)<<2)>>2]&255;a[j+4|0]=ja(b)|0;do{if((d[j+3|0]|0|0)==12){if((d[j+4|0]|0|0)<=64){break}if((d[j+4|0]|0|0)<=80){a[j+3|0]=10;l=j+4|0;a[l]=(d[l]|0)-64&255}else{if((d[j+4|0]|0|0)<=96){a[j+3|0]=10;a[j+4|0]=(d[j+4|0]|0)-80<<4&255}else{if((d[j+4|0]|0|0)<=112){a[j+3|0]=-82;a[j+4|0]=(d[j+4|0]|0)-96&255}else{if((d[j+4|0]|0|0)<=128){a[j+3|0]=-83;a[j+4|0]=(d[j+4|0]|0)-112&255}}}}}}while(0);if((d[j+3|0]|0|0)==0){a[j+4|0]=(((24-((a[j+4|0]&240)>>4)|0)%12|0)<<4|a[j+4|0]&15)&255}if((d[j+3|0]|0|0)==255){a[j+4|0]=0;a[j+3|0]=0}i=i+1|0}i=h+288|0;c[i>>2]=(c[i>>2]|0)+1;return}function fX(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;b=a;a=d;d=b|0;f=e;e=0;if((c[f+292>>2]|0)>=(c[d+140>>2]|0)){return}g=c[f+144+(c[f+292>>2]<<2)>>2]|0;if((c[f+(g<<2)>>2]|0)==0){h=70}else{if((c[f+(g<<2)>>2]|0)==2){h=70}}if((h|0)==70){e=8}gQ(b,a,e,(c[d+180>>2]|0)+((c[(c[(c[d+176>>2]|0)+(g*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;g=f+292|0;c[g>>2]=(c[g>>2]|0)+1;return}function fY(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0;e=b;b=c;if(((ja(e)|0)&255|0)!=174){f=-1;g=f;return g|0}c=pS(65536)|0;h=c;if((c|0)==0){f=-1;g=f;return g|0}f_(h,e,65536);e=0;while(1){if((e|0)>=128){break}if((d[h+(e+7936)|0]|0)!=0){if((d[h+7936|0]|0)<224){i=83;break}}e=e+1|0}if((i|0)==83){pU(h);f=-1;g=f;return g|0}if((b|0)!=0){c=b;j=h+8096|0;p3(c|0,j|0,16)|0;a[b+16|0]=0;e=15;while(1){if((e|0)<0){break}if((a[b+e|0]|0)!=32){i=91;break}a[b+e|0]=0;e=e-1|0}}pU(h);f=0;g=f;return g|0}function fZ(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;g=i;h=b;b=e;e=h|0;bK(b|0,f|0,0)|0;ja(b)|0;f=pT(1,65536)|0;j=f;if((f|0)==0){k=-1;l=k;i=g;return l|0}f_(j,b,65536);b=0;while(1){if((d[j+(b+7936)|0]|0)!=0){m=(b|0)<128}else{m=0}if(!m){break}a[e+952+b|0]=(d[j+(b+7936)|0]|0)-224&255;b=b+1|0}c[e+156>>2]=b;m=e|0;f=j+8096|0;p3(m|0,f|0,16)|0;dm(h,25872,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0);i=f;c[e+148>>2]=3;c[e+152>>2]=((d[j+8129|0]|0)*125|0|0)/136|0;c[e+128>>2]=0;b=0;while(1){if((b|0)>=(c[e+156>>2]|0)){break}if((d[e+952+b|0]|0)>(c[e+128>>2]|0)){c[e+128>>2]=d[e+952+b|0]|0}b=b+1|0}f=e+128|0;c[f>>2]=(c[f>>2]|0)+1;c[e+136>>2]=4;c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;b=0;while(1){if((b|0)>=(c[e+128>>2]|0)){break}f=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(b<<2)>>2]=f;c[c[(c[e+168>>2]|0)+(b<<2)>>2]>>2]=64;f=0;while(1){if((f|0)>=(c[e+136>>2]|0)){break}m=(ac(b,c[e+136>>2]|0)|0)+f|0;c[(c[(c[e+168>>2]|0)+(b<<2)>>2]|0)+4+(f<<2)>>2]=m;n=pT(((c[c[(c[e+168>>2]|0)+(b<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(m<<2)>>2]=n;c[c[(c[e+172>>2]|0)+(m<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(b<<2)>>2]>>2];f=f+1|0}f=0;while(1){if((f|0)>=64){break}m=0;while(1){if((m|0)>=4){break}n=a[j+((b<<8)+(f<<2)+m)|0]|0;o=(c[(c[h+172>>2]|0)+(c[(c[(c[h+168>>2]|0)+(b<<2)>>2]|0)+4+(m<<2)>>2]<<2)>>2]|0)+4+(f<<3)|0;if((n&255|0)==240){a[o+3|0]=13;a[o+4|0]=0}else{a[o|0]=n&255&15;if((a[o|0]|0)!=0){p=o|0;a[p]=(d[p]|0)+48&255}a[o+1|0]=(n&240)>>4&255}m=m+1|0}f=f+1|0}b=b+1|0}c[e+144>>2]=15;c[e+140>>2]=15;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}b=0;while(1){if((b|0)>=15){break}f=pT(64,1)|0;c[(c[e+176>>2]|0)+(b*764|0)+756>>2]=f;if((d[j+(b+8065)|0]|0)<16){q=0}else{q=d[j+(b+8081)|0]<<8}c[(c[e+180>>2]|0)+(b*52|0)+32>>2]=q;c[(c[(c[e+176>>2]|0)+(b*764|0)+756>>2]|0)+16>>2]=0;c[c[(c[e+176>>2]|0)+(b*764|0)+756>>2]>>2]=64;c[(c[e+180>>2]|0)+(b*52|0)+36>>2]=0;c[(c[e+180>>2]|0)+(b*52|0)+40>>2]=0;c[(c[e+180>>2]|0)+(b*52|0)+44>>2]=0;c[(c[(c[e+176>>2]|0)+(b*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(b*764|0)+756>>2]|0)+40>>2]=b;c[(c[e+176>>2]|0)+(b*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(b*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[e+176>>2]|0)+(b*764|0)+40>>2]=4095;b=b+1|0}b=8192;while(1){if((b|0)>=65536){break}a[j+b|0]=d[j+b|0]<<2&255;b=b+1|0}b=0;while(1){if((b|0)>=(c[e+140>>2]|0)){break}if((c[(c[e+180>>2]|0)+(b*52|0)+32>>2]|0)!=0){q=h;f=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(b*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;m=j+8192+((d[j+(b+8065)|0]|0)-16<<8)|0;gQ(q,0,18,f,m)|0}b=b+1|0}pU(j);b=0;while(1){if((b|0)>=(c[e+136>>2]|0)){break}c[e+184+(b*12|0)>>2]=128;b=b+1|0}b=h+1280|0;c[b>>2]=c[b>>2]|8192;k=0;l=k;i=g;return l|0}function f_(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=b;b=c;c=d;d=0;while(1){if((d|0)>=(c|0)){f=201;break}g=(ja(b)|0)&255;if((bH(b|0)|0)!=0){break}if((g|0)==174){h=(ja(b)|0)&255;i=h;if((h|0)==1){h=d;d=h+1|0;a[e+h|0]=-82}else{h=(ja(b)|0)&255;while(1){j=i;i=j-1|0;if((j|0)!=0){k=(d|0)<(c|0)}else{k=0}if(!k){break}j=d;d=j+1|0;a[e+j|0]=h&255}}}else{h=d;d=h+1|0;a[e+h|0]=g&255}}if((f|0)==201){return}return}function f$(a,b,c){a=a|0;b=b|0;c=c|0;return jr(a,b,c)|0}function f0(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;c=i;i=i+1024|0;d=c|0;e=a;a=b;do{if((a|0)>1024){f=1024}else{f=a}b=f;p7(d|0,0,b|0);ay(d|0,1,b|0,e|0)|0;a=a-b|0;if((b|0)>0){g=(a|0)>0}else{g=0}}while(g);i=c;return 0}function f1(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;e=i;i=i+96|0;f=e|0;g=e+72|0;h=d;d=-1;j=bl(bn(a|0)|0,25824)|0;if((j|0)==0){k=-1;l=k;i=e;return l|0}a=bl(bn(b|0)|0,29960)|0;if((aC(aE(j|0)|0,f|0)|0)<0){m=-1}else{m=c[f+28>>2]|0}if((m|0)<2048){k=-2;l=k;i=e;return l|0}f=pS(m+4096|0)|0;if((f|0)==0){aF(27944);k=-1;l=k;i=e;return l|0}bB(f|0,m|0,1,j|0)|0;b=0;while(1){if((c[4848+(b<<2)>>2]|0)==0){break}if((b2[c[(c[4848+(b<<2)>>2]|0)+4>>2]&511](f,g|0,m)|0)>=0){n=227;break}b=b+1|0}if((c[4848+(b<<2)>>2]|0)==0){k=-1;l=k;i=e;return l|0}bK(j|0,0,0)|0;d=b6[c[(c[4848+(b<<2)>>2]|0)+8>>2]&127](j,a)|0;if((d|0)<0){k=-1;l=k;i=e;return l|0}au(a|0)|0;au(j|0)|0;pU(f);if((h|0)!=0){c[h>>2]=c[c[4848+(b<<2)>>2]>>2]}k=0;l=k;i=e;return l|0}function f2(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+24|0;f=e|0;g=a;a=b;b=d;d=0;while(1){if((c[4848+(d<<2)>>2]|0)==0){h=259;break}j=b2[c[(c[4848+(d<<2)>>2]|0)+4>>2]&511](g,f|0,a)|0;if((j|0)>0){h=249;break}if((j|0)==0){h=251;break}d=d+1|0}if((h|0)==249){k=j;l=k;i=e;return l|0}else if((h|0)==251){if((b|0)!=0){j=b|0;a=f;p3(j|0,a|0,21)|0;a=b+64|0;b=c[c[4848+(d<<2)>>2]>>2]|0;p5(a|0,b|0,64)|0}k=0;l=k;i=e;return l|0}else if((h|0)==259){k=-1;l=k;i=e;return l|0}return 0}function f3(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0;e=b;b=c;c=d;if((b|0)==0){return}if((e|0)==0){a[b]=0;return}if((c|0)>20){c=20}d=b;f=e;e=c;p3(d|0,f|0,e)|0;a[b+c|0]=0;return}function f4(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0;e=a;a=b;b=c;c=0;if((b|0)<1080){f=1080-b|0;g=f;return g|0}do{if((d[e+1080|0]|0|0)==83){if((d[e+1081|0]|0|0)!=78){break}if((d[e+1082|0]|0|0)!=84){break}if((d[e+1083|0]|0|0)!=46){break}if((d[e+(c+951)|0]|0|0)!=127){f=-1;g=f;return g|0}if((d[e+(c+950)|0]|0|0)>127){f=-1;g=f;return g|0}else{f3(e,a,20);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function f5(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;e=i;i=i+2176|0;f=e|0;g=b;b=c;c=0;p7(f|0,0,2048);p7(e+2048|0,0,128);bB(f|0,950,1,g|0)|0;ay(f|0,950,1,b|0)|0;h=0;while(1){if((h|0)>=31){break}c=c+(((ji(f+(h*30|0)+42|0)|0)&65535)<<1)|0;h=h+1|0}j=ja(g)|0;jm(b,j);p7(f|0,0,2048);bB(f|0,129,1,g|0)|0;ay(f|0,129,1,b|0)|0;jq(b,1294879534);j=0;h=1;while(1){if((h|0)>=129){break}if((d[f+h|0]|0|0)>(j&255|0)){j=a[f+h|0]|0}h=h+1|0}bK(g|0,1084,0)|0;h=0;while(1){if((h|0)>(j&255|0)){break}k=0;while(1){if((k|0)>=256){break}a[f|0]=ja(g)|0;a[f+1|0]=ja(g)|0;a[f+2|0]=ja(g)|0;a[f+3|0]=ja(g)|0;l=(d[f|0]|0)&240;m=(a[f|0]&15)<<4&255;m=(m&255|(d[f+2|0]|0))&255;n=a[f+3|0]|0;l=(l&255|(d[5440+((d[f+1|0]|0)<<1)|0]|0))&255;o=a[5441+((d[f+1|0]|0)<<1)|0]|0;jm(b,l);jm(b,o);jm(b,m);jm(b,n);k=k+1|0}h=h+1|0}f$(b,g,c)|0;i=e;return 0}function f6(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=c;c=0;if((b|0)<260){f=260-b|0;g=f;return g|0}do{if((d[e|0]|0|0)==83){if((d[e+1|0]|0|0)!=78){break}if((d[e+2|0]|0|0)!=84){break}if((d[e+3|0]|0|0)!=33){break}b=0;while(1){if((b|0)>=31){break}if((d[e+(c+11+(b<<3))|0]|0|0)>64){h=326;break}b=b+1|0}if((h|0)==326){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=31){h=335;break}if((d[e+(c+10+(b<<3))|0]|0|0)>15){h=332;break}b=b+1|0}if((h|0)==332){f=-1;g=f;return g|0}else if((h|0)==335){f3(0,a,0);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function f7(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;e=i;i=i+2192|0;f=e|0;g=e+2176|0;h=b;b=c;c=0;j=0;p7(f|0,0,2048);p7(e+2048|0,0,128);f0(b,20)|0;bK(h|0,8,0)|0;k=0;while(1){if((k|0)>=31){break}f0(b,22)|0;l=(jd(h)|0)&65535;jo(b,l&65535);j=j+(l<<1)|0;jm(b,ja(h)|0);jm(b,ja(h)|0);jo(b,jd(h)|0);jo(b,jd(h)|0);k=k+1|0}l=ja(h)|0;jm(b,l);jm(b,ja(h)|0);k=0;while(1){if((k|0)>=128){break}l=ja(h)|0;m=l;jm(b,l);if((m&255|0)>(c&255|0)){n=m&255}else{n=c&255}c=n&255;k=k+1|0}jq(b,1294879534);bK(h|0,770,0)|0;k=0;while(1){if((k|0)>(c&255|0)){break}n=0;while(1){if((n|0)>=256){break}l=0;o=0;p=0;m=0;a[f|0]=ja(h)|0;if((d[f|0]|0|0)==128){jq(b,0)}else{if((d[f|0]|0|0)==192){q=g|0;r=b;ay(q|0,4,1,r|0)|0;m=a[g|0]|0;p=a[g+1|0]|0;o=a[g+2|0]|0;l=a[g+3|0]|0}else{do{if((d[f|0]|0|0)!=192){if((d[f|0]|0|0)==192){break}a[f+1|0]=ja(h)|0;a[f+2|0]=ja(h)|0;m=(a[f+1|0]&128)>>3&255;m=(m&255|(d[5440+((d[f|0]|0)>>1<<1)|0]|0))&255;p=a[5441+((d[f|0]|0)>>1<<1)|0]|0;o=(a[f+1|0]&112)<<1&255;o=(o&255|(a[f|0]&1)<<4)&255;o=(o&255|a[f+1|0]&15)&255;l=a[f+2|0]|0;jm(b,m);jm(b,p);jm(b,o);jm(b,l)}}while(0)}}a[g|0]=a[g+4|0]|0;a[g+1|0]=a[g+5|0]|0;a[g+2|0]=a[g+6|0]|0;a[g+3|0]=a[g+7|0]|0;a[g+4|0]=a[g+8|0]|0;a[g+5|0]=a[g+9|0]|0;a[g+6|0]=a[g+10|0]|0;a[g+7|0]=a[g+11|0]|0;a[g+8|0]=a[g+12|0]|0;a[g+9|0]=a[g+13|0]|0;a[g+10|0]=a[g+14|0]|0;a[g+11|0]=a[g+15|0]|0;a[g+12|0]=m;a[g+13|0]=p;a[g+14|0]=o;a[g+15|0]=l;n=n+1|0}k=k+1|0}f$(b,h,j)|0;i=e;return 0}function f8(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=c;c=0;if((b|0)<260){f=260-b|0;g=f;return g|0}b=0;while(1){if((b|0)>=31){break}if((d[e+(c+(b<<3)+4)|0]|0|0)>64){h=378;break}b=b+1|0}if((h|0)==378){f=-1;g=f;return g|0}if((jk(e+c+256|0)|0)!=1397446996){f=-1;g=f;return g|0}else{f3(0,a,0);f=0;g=f;return g|0}return 0}function f9(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;f=i;i=i+3200|0;g=f+128|0;h=f+1152|0;j=b;b=e;e=0;k=0;l=0;m=0;p7(f|0,0,128);p7(h|0,0,512);f0(b,20)|0;e=0;while(1){if((e|0)>=31){break}f0(b,22)|0;n=(jd(j)|0)&65535;jo(b,n&65535);m=m+(n<<1)|0;jm(b,ja(j)|0);jm(b,ja(j)|0);jo(b,jd(j)|0);jo(b,jd(j)|0);e=e+1|0}jl(j)|0;jl(j)|0;jl(j)|0;n=((ja(j)|0)&255)+1&255;o=n;jm(b,n);jm(b,127);e=0;while(1){if((e|0)>=(o&255|0)){break}k=0;while(1){if((k|0)>=4){break}c[h+(e<<4)+(k<<2)>>2]=(jd(j)|0)&65535;k=k+1|0}e=e+1|0}e=0;while(1){if((e|0)>=128){break}if((e|0)<(o&255|0)){p=e}else{p=0}jm(b,p&255);e=e+1|0}jq(b,1294879534);ja(j)|0;p=aT(j|0)|0;e=0;while(1){if((e|0)>=(o&255|0)){break}p7(g|0,0,1024);k=0;while(1){if((k|0)>=4){break}bK(j|0,p+((c[h+(e<<4)+(k<<2)>>2]|0)-1<<8)|0,0)|0;l=0;while(1){if((l|0)>=64){break}n=(l<<4)+(k<<2)|0;q=ja(j)|0;r=ja(j)|0;s=ja(j)|0;t=ja(j)|0;a[g+n|0]=(r&240|(d[5440+((q&255)<<1)|0]|0))&255;a[g+(n+1)|0]=a[5441+((q&255)<<1)|0]|0;a[g+(n+2)|0]=((r&255)<<4&240|s&255)&255;a[g+(n+3)|0]=t;l=l+1|0}k=k+1|0}ay(g|0,1024,1,b|0)|0;e=e+1|0}f$(b,j,m)|0;i=f;return 0}function ga(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;c=a;a=b;b=0;e=((d[c+(b+268)|0]|0)<<8)+(d[c+(b+269)|0]|0)|0;if((e&3|0)!=0){f=-1;g=f;return g|0}h=(e|0)/4|0;do{if((h|0)!=0){if((h|0)>127){break}if((d[c+(b+784)|0]|0|0)!=0){f=-1;g=f;return g|0}h=0;while(1){if((h|0)>=31){break}i=((d[c+(b+20+(h<<3))|0]|0)<<8)+(d[c+(b+21+(h<<3))|0]|0)<<1;if((i+2|0)<((((d[c+(b+24+(h<<3))|0]|0)<<8)+(d[c+(b+25+(h<<3))|0]|0)<<1)+(((d[c+(b+26+(h<<3))|0]|0)<<8)+(d[c+(b+27+(h<<3))|0]|0)<<1)|0)){j=431;break}h=h+1|0}if((j|0)==431){f=-1;g=f;return g|0}h=0;while(1){if((h|0)>=31){break}if((d[c+(b+22+(h<<3))|0]|0|0)>15){j=438;break}if((d[c+(b+23+(h<<3))|0]|0|0)>64){j=438;break}h=h+1|0}if((j|0)==438){f=-1;g=f;return g|0}h=((d[c+(b+784)|0]|0)<<24)+((d[c+(b+785)|0]|0)<<16)+((d[c+(b+786)|0]|0)<<8)+(d[c+(b+787)|0]|0)|0;if((h|0)<788){f=-1;g=f;return g|0}i=0;while(1){if((i|0)>=(e|0)){break}k=((d[c+(b+272+i)|0]|0)<<24)+((d[c+(b+273+i)|0]|0)<<16)+((d[c+(b+274+i)|0]|0)<<8)+(d[c+(b+275+i)|0]|0)|0;if((k|0)>(h|0)){j=446;break}i=i+4|0}if((j|0)==446){f=-1;g=f;return g|0}i=i+2|0;while(1){if((i|0)>=128){break}k=((d[c+(b+272+(i<<2))|0]|0)<<24)+((d[c+(b+273+(i<<2))|0]|0)<<16)+((d[c+(b+274+(i<<2))|0]|0)<<8)+(d[c+(b+275+(i<<2))|0]|0)|0;if((k|0)!=0){j=452;break}i=i+1|0}if((j|0)==452){f=-1;g=f;return g|0}i=b+788|0;L565:while(1){if((i|0)>=(h+b-4|0)){j=473;break}if((d[c+i|0]|0|0)==128){i=i+1|0;continue}if((d[c+i|0]|0|0)>128){j=460;break}do{if((d[c+i|0]|0|0)==0){if((d[c+(i+1)|0]|0|0)!=0){break}if((d[c+(i+2)|0]|0|0)!=0){break}if((d[c+(i+3)|0]|0|0)==0){j=465;break L565}}}while(0);if(((d[c+(i+2)|0]|0)*15|0|0)==12){if((d[c+(i+3)|0]|0|0)>64){j=468;break}}if(((d[c+(i+2)|0]|0)*15|0|0)==13){if((d[c+(i+3)|0]|0|0)>64){j=471;break}}i=i+4|0}if((j|0)==460){f=-1;g=f;return g|0}else if((j|0)==465){f=-1;g=f;return g|0}else if((j|0)==468){f=-1;g=f;return g|0}else if((j|0)==471){f=-1;g=f;return g|0}else if((j|0)==473){f3(c,a,20);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function gb(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;f=i;i=i+2816|0;g=f|0;h=f+128|0;j=f+256|0;k=f+1280|0;l=f+1792|0;m=f+2304|0;n=b;b=e;e=0;o=0;p=0;q=0;r=0;s=0;p7(g|0,0,128);p7(h|0,0,128);p7(k|0,0,512);p7(l|0,0,512);p7(m|0,0,512);f$(b,n,20)|0;o=0;while(1){if((o|0)>=31){break}f0(b,22)|0;t=(jd(n)|0)&65535;jo(b,t&65535);r=r+(t<<1)|0;jm(b,ja(n)|0);jm(b,ja(n)|0);jo(b,jd(n)|0);jo(b,jd(n)|0);o=o+1|0}t=(jd(n)|0)&255;bK(n|0,2,1)|0;o=0;while(1){if((o|0)>=128){break}c[k+(o<<2)>>2]=jl(n)|0;o=o+1|0}u=0;o=0;while(1){if((o|0)>=(t&255|0)){break}if((o|0)==0){a[g|0]=0;u=u+1|0}else{p=0;while(1){if((p|0)>=(o|0)){break}if((c[k+(o<<2)>>2]|0)==(c[k+(p<<2)>>2]|0)){v=503;break}p=p+1|0}if((v|0)==503){v=0;a[g+o|0]=a[g+p|0]|0}if((p|0)==(o|0)){w=u;u=w+1|0;a[g+o|0]=w&255}}o=o+1|0}o=0;while(1){if((o|0)>=128){break}c[l+(o<<2)>>2]=c[k+(o<<2)>>2];o=o+1|0}L631:while(1){o=0;L633:while(1){if((o|0)>=(t&255|0)){break L631}p=0;while(1){if((p|0)>=(o|0)){break}if((c[l+(o<<2)>>2]|0)<(c[l+(p<<2)>>2]|0)){break L633}p=p+1|0}o=o+1|0}u=d[g+p|0]|0;a[g+p|0]=a[g+o|0]|0;a[g+o|0]=u&255;u=c[l+(p<<2)>>2]|0;c[l+(p<<2)>>2]=c[l+(o<<2)>>2];c[l+(o<<2)>>2]=u}p=0;o=0;while(1){if((o|0)>=128){break}do{if((o|0)==0){c[m+(p<<2)>>2]=c[l+(o<<2)>>2]}else{if((c[l+(o<<2)>>2]|0)==(c[m+(p<<2)>>2]|0)){break}else{u=p+1|0;p=u;c[m+(u<<2)>>2]=c[l+(o<<2)>>2];break}}}while(0);o=o+1|0}p=0;o=0;while(1){if((o|0)>=((t&255)-1|0)){break}c[l+(p<<2)>>2]=c[m+(o<<2)>>2];p=p+1|0;if(((c[m+(o+1<<2)>>2]|0)-(c[m+(o<<2)>>2]|0)|0)>1024){c[l+(p<<2)>>2]=(c[m+(o<<2)>>2]|0)+1024;p=p+1|0}o=o+1|0}o=0;while(1){if((o|0)>=128){break}p=0;while(1){if((p|0)>=128){break}if((c[k+(o<<2)>>2]|0)==(c[l+(p<<2)>>2]|0)){v=544;break}p=p+1|0}if((v|0)==544){v=0;a[h+o|0]=p&255}o=o+1|0}p7(g|0,0,128);o=0;while(1){if((o|0)>=(t&255|0)){break}a[g+o|0]=a[h+o|0]|0;o=o+1|0}jm(b,t);o=0;while(1){if((o|0)>=(t&255|0)){break}if((d[g+o|0]|0|0)>(e&255|0)){e=a[g+o|0]|0}o=o+1|0}jm(b,127);ay(g|0,128,1,b|0)|0;jq(b,1294879534);bK(n|0,784,0)|0;s=(jl(n)|0)+788|0;e=(e&255)+1&255;o=0;while(1){if((o|0)>=(e&255|0)){break}p7(j|0,0,1024);p=0;while(1){if((p|0)>=64){break}q=0;while(1){if((q|0)>=4){break}g=(p<<4)+(q<<2)|0;t=ja(n)|0;if((t&255|0)!=128){h=ja(n)|0;v=ja(n)|0;l=ja(n)|0;a[j+g|0]=t&255&15;a[j+(g+1)|0]=h;a[j+(g+2)|0]=v&255&15;a[j+(g+3)|0]=l;l=(t&240|(v&255)>>4&15)>>2&255;v=j+g|0;a[v]=(d[v]|0|l&240)&255;v=j+(g+2)|0;a[v]=(d[v]|0|(l&255)<<4&240)&255}q=q+1|0}p=p+1|0}ay(j|0,1024,1,b|0)|0;o=o+1|0}bK(n|0,s|0,0)|0;f$(b,n,r)|0;i=f;return 0}function gc(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;f=b;b=c;c=e;e=0;if((c|0)<564){g=564-c|0;h=g;return h|0}c=0;i=0;while(1){if((i|0)>=31){j=593;break}k=f+e+(i*14|0)|0;l=jk(k+130|0)|0;m=(ji(k+134|0)|0)&65535;n=jk(k+138|0)|0;o=(ji(k+142|0)|0)&65535;if((d[f+(e+(i*14|0)+137)|0]|0|0)>64){j=581;break}if((n|0)<(l|0)){j=583;break}if((l|0)<564){j=586;break}if((n|0)<564){j=586;break}if((n-l|0)>(m|0)){j=588;break}if((n-l+o|0)>(m+2|0)){j=590;break}c=c+m|0;i=i+1|0}if((j|0)==586){g=-1;h=g;return h|0}else if((j|0)==590){g=-1;h=g;return h|0}else if((j|0)==581){g=-1;h=g;return h|0}else if((j|0)==588){g=-1;h=g;return h|0}else if((j|0)==583){g=-1;h=g;return h|0}else if((j|0)==593){do{if((c|0)>2){if((c|0)>2031585){break}do{if((d[f+e|0]|0|0)<=127){if((d[f+e|0]|0|0)==0){break}l=0;i=0;while(1){if((i|0)>=128){break}if((d[f+(e+i+2)|0]|0|0)>127){j=602;break}if((d[f+(e+i+2)|0]|0|0)>(l|0)){l=d[f+(e+i+2)|0]|0}i=i+1|0}if((j|0)==602){g=-1;h=g;return h|0}l=l+1|0;l=l<<10;i=(d[f+e|0]|0)+2|0;while(1){if((i|0)>=128){break}if((d[f+(e+i+2)|0]|0|0)!=0){j=610;break}i=i+1|0}if((j|0)==610){g=-1;h=g;return h|0}m=e+564+c|0;i=0;while(1){if((i|0)>=(l|0)){j=630;break}if((d[f+(m+i)|0]|0|0)>31){j=616;break}if((d[f+(m+i+1)|0]|0|0)>72){j=619;break}if((a[f+(m+i+1)|0]&1|0)==1){j=619;break}if((a[f+(m+i+2)|0]&15|0)==12){if((d[f+(m+i+3)|0]|0|0)>64){j=622;break}}if((a[f+(m+i+2)|0]&15|0)==13){if((d[f+(m+i+3)|0]|0|0)>64){j=625;break}}if((a[f+(m+i+2)|0]&15|0)==11){j=627;break}i=i+4|0}if((j|0)==616){g=-1;h=g;return h|0}else if((j|0)==619){g=-1;h=g;return h|0}else if((j|0)==622){g=-1;h=g;return h|0}else if((j|0)==625){g=-1;h=g;return h|0}else if((j|0)==627){g=-1;h=g;return h|0}else if((j|0)==630){f3(0,b,0);g=-1;h=g;return h|0}}}while(0);g=-1;h=g;return h|0}}while(0);g=-1;h=g;return h|0}return 0}function gd(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;f=i;i=i+1280|0;g=f|0;h=f+1024|0;j=f+1152|0;k=b;b=e;e=0;p7(h|0,0,124);p7(j|0,0,124);f0(b,1080)|0;l=pS(130)|0;p7(l|0,0,130);bK(b|0,950,0)|0;bB(l|0,130,1,k|0)|0;ay(l|0,130,1,b|0)|0;m=0;n=0;while(1){if((m|0)>=128){break}if((d[l+(m+2)|0]|0|0)>(n&255|0)){n=a[l+(m+2)|0]|0}m=m+1|0}pU(l);m=0;while(1){if((m|0)>=31){break}bK(b|0,(m*30|0)+42|0,0)|0;c[h+(m<<2)>>2]=jl(k)|0;o=(jd(k)|0)&65535;p=o;jo(b,o&65535);e=e+p|0;c[j+(m<<2)>>2]=p;jm(b,ja(k)|0);jm(b,ja(k)|0);p=jl(k)|0;jo(b,(((p-(c[h+(m<<2)>>2]|0)|0)>>>0)/2|0)&65535);jo(b,jd(k)|0);m=m+1|0}bK(k|0,e|0,1)|0;bK(b|0,0,2)|0;jq(b,1294879534);l=pS(1024)|0;m=0;while(1){if((m|0)>(n&255|0)){break}p7(l|0,0,1024);p7(g|0,0,1024);bB(l|0,1024,1,k|0)|0;e=0;while(1){if((e|0)>=64){break}p=0;while(1){if((p|0)>=4){break}o=(e<<4)+(p<<2)|0;a[g+(o+3)|0]=a[l+(o+3)|0]|0;a[g+(o+2)|0]=(d[l+(o+2)|0]|0)&15;a[g+o|0]=(d[l+o|0]|0)&240;q=g+(o+2)|0;a[q]=(d[q]|0|(d[l+o|0]|0)<<4&240)&255;q=g+o|0;a[q]=(d[q]|0|(d[5440+(((d[l+(o+1)|0]|0|0)/2|0)<<1)|0]|0))&255;a[g+(o+1)|0]=a[5441+(((d[l+(o+1)|0]|0|0)/2|0)<<1)|0]|0;p=p+1|0}e=e+1|0}ay(g|0,1024,1,b|0)|0;m=m+1|0}pU(l);m=0;while(1){if((m|0)>=31){break}if((c[j+(m<<2)>>2]|0)!=0){l=k;g=c[h+(m<<2)>>2]|0;bK(l|0,g|0,0)|0;g=b;l=k;n=c[j+(m<<2)>>2]|0;f$(g,l,n)|0}m=m+1|0}i=f;return 0}function ge(a,b){a=a|0;b=b|0;var d=0,e=0;d=a;a=b;if((c[d>>2]|0)==(c[a>>2]|0)){e=0;return e|0}else{e=(c[d>>2]|0)>(c[a>>2]|0)?1:-1;return e|0}return 0}function gf(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;e=a;a=b;b=c;c=0;if((b|0)<182){f=182-b|0;g=f;return g|0}b=0;h=0;while(1){if((h|0)>=15){i=708;break}if((d[e+(c+7+(h*12|0))|0]|0|0)>64){i=690;break}if((d[e+(c+6+(h*12|0))|0]|0|0)!=0){i=692;break}j=jk(e+c+(h*12|0)|0)|0;if((j|0)<180){if((j|0)!=0){i=695;break}}k=(ji(e+c+(h*12|0)+4|0)|0)&65535;l=(ji(e+c+(h*12|0)+8|0)|0)&65535;m=(ji(e+c+(h*12|0)+10|0)|0)&65535;if((l|0)>(k|0)){i=699;break}if((m|0)>(k+1|0)){i=699;break}if((k|0)>32768){i=699;break}if((m|0)==0){i=701;break}if((k|0)==0){if((l|0)!=0){i=705;break}if((m|0)!=1){i=705;break}}b=b+k|0;h=h+1|0}if((i|0)==708){if((b|0)<2){f=-1;g=f;return g|0}j=-1;h=0;l=0;while(1){if((h|0)>=256){break}if(((ji(e+c+h+180|0)|0)&65535|0)==65535){i=713;break}k=(ji(e+c+h+180|0)|0)&65535;if((k|0)<180){i=715;break}if((k|0)>(l|0)){l=k}h=h+2|0}if((i|0)==713){j=0}else if((i|0)==715){f=-1;g=f;return g|0}if((j|0)==-1){f=-1;g=f;return g|0}else{f3(0,a,0);f=0;g=f;return g|0}}else if((i|0)==692){f=-1;g=f;return g|0}else if((i|0)==695){f=-1;g=f;return g|0}else if((i|0)==705){f=-1;g=f;return g|0}else if((i|0)==690){f=-1;g=f;return g|0}else if((i|0)==699){f=-1;g=f;return g|0}else if((i|0)==701){f=-1;g=f;return g|0}return 0}function gg(f,g){f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;h=i;i=i+2656|0;j=h|0;k=h+1024|0;l=h+1536|0;m=h+2048|0;n=h+2560|0;o=h+2624|0;p=f;f=g;g=0;q=0;while(1){if((q|0)>=128){break}c[m+(q<<2)>>2]=0;c[l+(q<<2)>>2]=0;c[k+(q<<2)>>2]=0;q=q+1|0}f0(f,20)|0;q=0;while(1){if((q|0)>=15){break}c[n+(q<<2)>>2]=jl(p)|0;f0(f,22)|0;r=jd(p)|0;b[o+(q<<1)>>1]=r;jo(f,r);r=o+(q<<1)|0;b[r>>1]=(e[r>>1]|0)<<1&65535;jm(f,ja(p)|0);jm(f,ja(p)|0);jo(f,jd(p)|0);jo(f,jd(p)|0);q=q+1|0}q=15;while(1){if((q|0)>=31){break}f0(f,22)|0;jo(f,0);jm(f,0);jm(f,64);jo(f,0);jo(f,1);q=q+1|0}bB(j|0,2,128,p|0)|0;r=0;while(1){if((r&255|0)>=128){break}if((d[j+((r&255)<<1)|0]|0|0)==255){s=750;break}t=(ji(j+((r&255)<<1)|0)|0)&65535;c[k+((r&255)<<2)>>2]=t;c[l+((r&255)<<2)>>2]=t;r=r+1&255}jm(f,r);jm(f,127);bm(l|0,r&255|0,4,24);q=0;t=0;while(1){if((q|0)>=(r&255|0)){break}u=t;t=u+1|0;c[m+(u<<2)>>2]=c[l+(q<<2)>>2];while(1){if((c[l+(q+1<<2)>>2]|0)==(c[l+(q<<2)>>2]|0)){v=(q|0)<(r&255|0)}else{v=0}if(!v){break}q=q+1|0}q=q+1|0}p7(j|0,0,128);q=0;while(1){if((q|0)>=(r&255|0)){break}t=0;while(1){if((c[k+(q<<2)>>2]|0)==(c[m+(t<<2)>>2]|0)){break}t=t+1|0}a[j+q|0]=t&255;if((t|0)>(g|0)){g=t}q=q+1|0}ay(j|0,128,1,f|0)|0;jq(f,1294879534);q=0;while(1){if((q|0)>(g|0)){break}bK(p|0,c[m+(q<<2)>>2]|0,0)|0;p7(j|0,0,1024);t=ja(p)|0;k=0;while(1){if((k|0)>=64){break}r=ja(p)|0;v=(r&255)>>6<<2&255;l=r&63;if((l|0)<=36){a[j+((k<<4)+(v&255))|0]=a[5440+(l<<1)|0]|0;a[j+((k<<4)+(v&255)+1)|0]=a[5441+(l<<1)|0]|0}a[j+((k<<4)+(v&255)+2)|0]=ja(p)|0;a[j+((k<<4)+(v&255)+3)|0]=ja(p)|0;if((t&128|0)!=0){s=779;break}t=ja(p)|0;k=k+(t&127)|0}if((s|0)==779){s=0}ay(j|0,1024,1,f|0)|0;q=q+1|0}q=0;while(1){if((q|0)>=15){break}if((c[n+(q<<2)>>2]|0)!=0){j=p;s=c[n+(q<<2)>>2]|0;bK(j|0,s|0,0)|0;s=f;j=p;m=e[o+(q<<1)>>1]|0;f$(s,j,m)|0}q=q+1|0}i=h;return 0}function gh(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;e=a;a=b;b=c;c=0;if((b|0)<1024){f=1024-b|0;g=f;return g|0}if((p6(e|0,29920,8)|0)!=0){f=-1;g=f;return g|0}b=(ji(e+c+28|0)|0)&65535;do{if((b&7|0)==0){if((b|0)==0){break}b=(b|0)/8|0;h=0;while(1){if((h|0)>=(b|0)){break}if((d[e+(c+30+(h<<3))|0]|0|0)>15){i=802;break}h=h+1|0}if((i|0)==802){f=-1;g=f;return g|0}h=0;while(1){if((h|0)>=(b|0)){break}if((d[e+(c+31+(h<<3))|0]|0|0)>64){i=808;break}h=h+1|0}if((i|0)==808){f=-1;g=f;return g|0}j=0;h=0;while(1){if((h|0)>=(b|0)){i=824;break}k=c+(h<<3)|0;l=((ji(e+k+32|0)|0)&65535)<<1;m=((ji(e+k+34|0)|0)&65535)<<1;n=((ji(e+k+36|0)|0)&65535)<<1;if((l|0)>65535){i=816;break}if((m|0)>65535){i=816;break}if((n|0)>65535){i=816;break}if((m+n|0)>(l+2|0)){i=818;break}if((m|0)!=0){if((n|0)==0){i=821;break}}j=j+l|0;h=h+1|0}if((i|0)==821){f=-1;g=f;return g|0}else if((i|0)==816){f=-1;g=f;return g|0}else if((i|0)==818){f=-1;g=f;return g|0}else if((i|0)==824){if((j|0)<=4){f=-1;g=f;return g|0}l=d[e+(c+(b<<3)+31)|0]|0;do{if((b|0)!=0){if((b|0)>128){break}f3(e+8|0,a,20);f=0;g=f;return g|0}}while(0);f=-1;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function gi(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0;f=i;i=i+3256|0;g=f|0;h=f+128|0;j=f+1152|0;k=f+1208|0;l=b;b=e;e=0;m=0;n=999999;o=0;p=0;p7(k|0,0,2048);p7(g|0,0,128);bK(l|0,8,1)|0;f$(b,l,20)|0;q=(((jd(l)|0)&65535|0)/8|0)&255;e=0;while(1){if((e|0)>=(q&255|0)){break}f0(b,22)|0;r=ja(l)|0;s=ja(l)|0;t=(jd(l)|0)&65535;jo(b,t&65535);o=o+(t<<1)|0;jm(b,r);jm(b,s);jo(b,jd(l)|0);jo(b,jd(l)|0);e=e+1|0}p7(j|0,0,30);a[j+29|0]=1;while(1){if((e|0)>=31){break}ay(j|0,30,1,b|0)|0;e=e+1|0}ja(l)|0;j=ja(l)|0;q=j;jm(b,j);jm(b,127);e=0;j=0;while(1){if((e|0)>=(q&255|0)){break}a[g+e|0]=(((jd(l)|0)&65535|0)/8|0)&255;if((d[g+e|0]|0|0)>(j&255|0)){j=a[g+e|0]|0}e=e+1|0}e=0;while(1){if((e|0)>(j&255|0)){break}m=0;while(1){if((m|0)>=4){break}c[k+(e<<4)+(m<<2)>>2]=(jd(l)|0)&65535;if((c[k+(e<<4)+(m<<2)>>2]|0)>(p|0)){p=c[k+(e<<4)+(m<<2)>>2]|0}m=m+1|0}e=e+1|0}ay(g|0,128,1,b|0)|0;jq(b,1294879534);n=(aT(l|0)|0)+2|0;e=0;while(1){if((e|0)>(j&255|0)){break}p7(h|0,0,1024);m=0;while(1){if((m|0)>=4){break}bK(l|0,n+(c[k+(e<<4)+(m<<2)>>2]|0)|0,0)|0;g=0;while(1){if((g|0)>=64){break}q=(g<<4)+(m<<2)|0;s=ja(l)|0;do{if((s&192|0)==192){g=g+(256-(s&255))|0;g=g-1|0}else{if((s&192|0)==128){u=ja(l)|0;v=(s&255)>>1&15;w=u;do{if((v&255|0)==5){x=878}else{if((v&255|0)==6){x=878;break}if((v&255|0)==10){x=878}}}while(0);if((x|0)==878){x=0;if((w&255|0)>128){w=256-(w&255)&255}else{if((w&255|0)<=128){w=(w&255)<<4&240}}}if((v&255|0)==8){v=0}a[h+(q+2)|0]=v;a[h+(q+3)|0]=w;break}u=ja(l)|0;t=((u&255)>>4&15|(s&255)>>2&16)&255;if((s&64|0)==64){y=127-(s&255)&255}else{y=s&255&63}v=u&255&15;if((v&255|0)==0){a[h+q|0]=t&255&240;z=h+q|0;a[z]=(d[z]|0|(d[5440+((y&255)<<1)|0]|0))&255;a[h+(q+1)|0]=a[5441+((y&255)<<1)|0]|0;a[h+(q+2)|0]=(t&255)<<4&240;break}r=ja(l)|0;if((v&255|0)==8){v=0}w=r;do{if((v&255|0)==5){x=897}else{if((v&255|0)==6){x=897;break}if((v&255|0)==10){x=897}}}while(0);if((x|0)==897){x=0;if((w&255|0)>128){w=256-(w&255)&255}else{if((w&255|0)<=128){w=(w&255)<<4&240}}}a[h+q|0]=t&255&240;z=h+q|0;a[z]=(d[z]|0|(d[5440+((y&255)<<1)|0]|0))&255;a[h+(q+1)|0]=a[5441+((y&255)<<1)|0]|0;a[h+(q+2)|0]=(t&255)<<4&240;z=h+(q+2)|0;a[z]=(d[z]|0|v&255)&255;a[h+(q+3)|0]=w}}while(0);g=g+1|0}g=aT(l|0)|0;if((g|0)>(p|0)){p=g}m=m+1|0}ay(h|0,1024,1,b|0)|0;e=e+1|0}if((p&1|0)==0){A=l;B=p;C=bK(A|0,B|0,0)|0;D=b;E=l;F=o;G=f$(D,E,F)|0;i=f;return 0}p=p+1|0;A=l;B=p;C=bK(A|0,B|0,0)|0;D=b;E=l;F=o;G=f$(D,E,F)|0;i=f;return 0}function gj(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;e=a;a=b;b=c;c=0;if((b|0)<1084){f=1084-b|0;g=f;return g|0}if((jk(e+c+1080|0)|0)!=1294879534){f=-1;g=f;return g|0}h=0;i=0;while(1){if((i|0)>=31){break}j=c+(i*30|0)|0;k=((ji(e+j+42|0)|0)&65535)<<1;h=h+k|0;l=(ji(e+j+46|0)|0)&65535;if((k+2|0)<(l+((ji(e+j+48|0)|0)&65535)<<1|0)){m=925;break}i=i+1|0}if((m|0)==925){f=-1;g=f;return g|0}if((h|0)<=2){f=-1;g=f;return g|0}i=0;while(1){if((i|0)>=31){break}h=c+(i*30|0)|0;if((d[e+(h+40)|0]|0|0)>15){m=935;break}if((d[e+(h+44)|0]|0|0)!=0){m=935;break}if((d[e+(h+45)|0]|0|0)>64){m=935;break}i=i+1|0}if((m|0)==935){f=-1;g=f;return g|0}h=d[e+(c+950)|0]|0;do{if((h|0)<=127){if((h|0)==0){break}i=0;k=0;while(1){if((k|0)>=(h|0)){break}if((d[e+(c+952+k)|0]|0|0)>(i|0)){i=d[e+(c+952+k)|0]|0}if((d[e+(c+952+k)|0]|0|0)>127){m=946;break}k=k+1|0}if((m|0)==946){f=-1;g=f;return g|0}while(1){if((k|0)==128){break}if((d[e+(c+952+k)|0]|0|0)!=0){m=952;break}k=k+1|0}if((m|0)==952){f=-1;g=f;return g|0}i=i+1|0;if((b|0)<(((i<<8)*3|0)+1084|0)){f=((i<<8)*3|0)+1084-b|0;g=f;return g|0}k=0;while(1){if((k|0)>=(i<<8|0)){m=964;break}if((d[e+(c+1084+(k*3|0))|0]|0|0)>116){m=961;break}k=k+1|0}if((m|0)==964){f3(e,a,20);f=0;g=f;return g|0}else if((m|0)==961){f=-1;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function gk(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;e=i;i=i+1032|0;f=e|0;g=b;b=c;c=0;h=2;j=0;k=0;l=0;m=0;n=0;f$(b,g,20)|0;j=0;while(1){if((j|0)>=31){break}f$(b,g,20)|0;jm(b,0);jm(b,0);o=ja(g)|0;p=ja(g)|0;k=((o&255)<<8)+(p&255)|0;if((k|0)!=0){if((k|0)<256){q=16-(p&255)&255}else{q=256-(p&255)&255}}else{q=0}r=(jd(g)|0)&65535;m=r;jo(b,r&65535);n=n+(m<<1)|0;ja(g)|0;jm(b,q);jm(b,ja(g)|0);k=(jd(g)|0)&65535;l=(jd(g)|0)&65535;do{if(((k<<1)+l|0)<=(m|0)){if((k|0)==0){break}h=1;k=k<<1}}while(0);jo(b,k&65535);jo(b,l&65535);j=j+1|0}l=ja(g)|0;jm(b,l);jm(b,127);ja(g)|0;bB(f|0,128,1,g|0)|0;ay(f|0,128,1,b|0)|0;j=0;while(1){if((j|0)>=128){break}if((d[f+j|0]|0|0)>(c&255|0)){c=a[f+j|0]|0}j=j+1|0}c=c+1&255;jq(b,1294879534);bK(g|0,1080,0)|0;l=jl(g)|0;do{if((l|0)!=0){if((l|0)==1294879534){break}if((l|0)==1431193923){break}h=g;bK(h|0,-4|0,1)|0}}while(0);j=0;while(1){if((j|0)>=(c&255|0)){break}k=0;while(1){if((k|0)>=256){break}o=ja(g)|0;p=ja(g)|0;l=ja(g)|0;h=((o&255)>>2&16|(p&255)>>4&15)&255;m=o&255&63;q=p&255&15;r=l;if((q&255|0)==13){l=((r&255|0)/10|0)&255;r=((l&255)<<4)+(((r&255|0)%10|0)&255)&255}a[f+(k<<2)|0]=(h&240|(d[5440+((m&255)<<1)|0]|0))&255;a[f+((k<<2)+1)|0]=a[5441+((m&255)<<1)|0]|0;a[f+((k<<2)+2)|0]=((h&255)<<4&240|q&255)&255;a[f+((k<<2)+3)|0]=r;k=k+1|0}ay(f|0,1024,1,b|0)|0;j=j+1|0}f$(b,g,n)|0;i=e;return 0}function gl(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;f=b;b=c;c=e;e=0;if((c|0)<1084){g=1084-c|0;h=g;return h|0}if((jk(f+e+1080|0)|0)==0){g=-1;h=g;return h|0}i=0;j=0;k=0;while(1){if((k|0)>=31){l=1043;break}m=e+(k*30|0)|0;n=((ji(f+m+42|0)|0)&65535)<<1;o=((ji(f+m+46|0)|0)&65535)<<1;p=((ji(f+m+48|0)|0)&65535)<<1;i=i+n|0;if((p|0)!=0){if((n+2|0)<(o+p|0)){l=1021;break}}if((n|0)>65535){l=1025;break}if((o|0)>65535){l=1025;break}if((p|0)>65535){l=1025;break}if((d[f+(m+45)|0]|0|0)>64){l=1027;break}q=(ji(f+m+40|0)|0)&65535;if((q|0)!=0){if((n|0)==0){l=1032;break}}if((q|0)>8){if((q|0)<247){l=1032;break}}if((o|0)!=0){if((p|0)<=2){l=1035;break}}if((d[f+(m+45)|0]|0|0)!=0){if((n|0)==0){l=1038;break}}if((n|0)!=0){j=n+1|0}k=k+1|0}if((l|0)==1025){g=-1;h=g;return h|0}else if((l|0)==1035){g=-1;h=g;return h|0}else if((l|0)==1038){g=-1;h=g;return h|0}else if((l|0)==1021){g=-1;h=g;return h|0}else if((l|0)==1043){if((i|0)<=2){g=-1;h=g;return h|0}i=d[f+(e+950)|0]|0;do{if((i|0)<=127){if((i|0)==0){break}k=0;n=0;while(1){if((n|0)>=(i|0)){break}if((d[f+(e+952+n)|0]|0|0)>(k|0)){k=d[f+(e+952+n)|0]|0}if((d[f+(e+952+n)|0]|0|0)>127){l=1053;break}n=n+1|0}if((l|0)==1053){g=-1;h=g;return h|0}while(1){if((n|0)==128){break}if((d[f+(e+952+n)|0]|0|0)!=0){l=1059;break}n=n+1|0}if((l|0)==1059){g=-1;h=g;return h|0}k=k+1|0;if((c|0)<(((k<<8)*3|0)+1082|0)){g=((k<<8)*3|0)+1082-c|0;h=g;return h|0}n=0;while(1){if((n|0)>=(k<<8|0)){l=1084;break}m=e+1080+(n*3|0)|0;if((d[f+m|0]|0|0)>116){l=1068;break}if((a[f+m|0]&63|0)>36){l=1070;break}if((a[f+(m+1)|0]&15|0)==12){if((d[f+(m+2)|0]|0|0)>64){l=1073;break}}if((a[f+(m+1)|0]&15|0)==11){if((d[f+(m+2)|0]|0|0)>127){l=1076;break}}if((a[f+(m+1)|0]&15|0)==13){if((d[f+(m+2)|0]|0|0)>64){l=1079;break}}p=(d[f+m|0]|0)>>2&48|(d[f+(e+1081+(n*3|0)+1)|0]|0)>>4&15;if((p|0)>(j|0)){l=1081;break}n=n+1|0}if((l|0)==1070){g=-1;h=g;return h|0}else if((l|0)==1079){g=-1;h=g;return h|0}else if((l|0)==1081){g=-1;h=g;return h|0}else if((l|0)==1084){n=0;while(1){if((n|0)>=20){l=1092;break}if((d[f+(e+n)|0]|0|0)!=0){if((d[f+(e+n)|0]|0|0)<32){l=1089;break}}if((d[f+(e+n)|0]|0|0)>180){l=1089;break}n=n+1|0}if((l|0)==1092){f3(f,b,20);g=0;h=g;return h|0}else if((l|0)==1089){g=-1;h=g;return h|0}}else if((l|0)==1073){g=-1;h=g;return h|0}else if((l|0)==1076){g=-1;h=g;return h|0}else if((l|0)==1068){g=-1;h=g;return h|0}}}while(0);g=-1;h=g;return h|0}else if((l|0)==1032){g=-1;h=g;return h|0}else if((l|0)==1027){g=-1;h=g;return h|0}return 0}function gm(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;f=b;b=c;c=e;e=0;if((c|0)<1084){g=1084-c|0;h=g;return h|0}if((jk(f+e+1080|0)|0)!=0){g=-1;h=g;return h|0}i=0;j=0;k=0;while(1){if((k|0)>=31){l=1147;break}m=e+(k*30|0)|0;n=((ji(f+m+42|0)|0)&65535)<<1;o=((ji(f+m+46|0)|0)&65535)<<1;p=((ji(f+m+48|0)|0)&65535)<<1;i=i+n|0;if((p|0)!=0){if((n+2|0)<(o+p|0)){l=1125;break}}if((n|0)>65535){l=1129;break}if((o|0)>65535){l=1129;break}if((p|0)>65535){l=1129;break}if((d[f+(m+45)|0]|0|0)>64){l=1131;break}q=(ji(f+m+40|0)|0)&65535;if((q|0)!=0){if((n|0)==0){l=1136;break}}if((q|0)>8){if((q|0)<247){l=1136;break}}if((o|0)!=0){if((p|0)<=2){l=1139;break}}if((d[f+(m+45)|0]|0|0)!=0){if((n|0)==0){l=1142;break}}if((n|0)!=0){j=n+1|0}k=k+1|0}if((l|0)==1147){if((i|0)<=2){g=-1;h=g;return h|0}i=d[f+(e+950)|0]|0;do{if((i|0)<=127){if((i|0)==0){break}k=0;n=0;while(1){if((n|0)>=(i|0)){break}if((d[f+(e+952+n)|0]|0|0)>(k|0)){k=d[f+(e+952+n)|0]|0}if((d[f+(e+952+n)|0]|0|0)>127){l=1157;break}n=n+1|0}if((l|0)==1157){g=-1;h=g;return h|0}while(1){if((n|0)==128){break}if((d[f+(e+952+n)|0]|0|0)!=0){l=1163;break}n=n+1|0}if((l|0)==1163){g=-1;h=g;return h|0}k=k+1|0;if((c|0)<(((k<<8)*3|0)+1086|0)){g=((k<<8)*3|0)+1086-c|0;h=g;return h|0}n=0;while(1){if((n|0)>=(k<<8|0)){l=1188;break}m=e+1084+(n*3|0)|0;if((d[f+m|0]|0|0)>116){l=1172;break}if((a[f+m|0]&63|0)>36){l=1174;break}if((a[f+(m+1)|0]&15|0)==12){if((d[f+(m+2)|0]|0|0)>64){l=1177;break}}if((a[f+(m+1)|0]&15|0)==11){if((d[f+(m+2)|0]|0|0)>127){l=1180;break}}if((a[f+(m+1)|0]&15|0)==13){if((d[f+(m+2)|0]|0|0)>64){l=1183;break}}p=(d[f+m|0]|0)>>2&48|(d[f+(e+1085+(n*3|0)+1)|0]|0)>>4&15;if((p|0)>(j|0)){l=1185;break}n=n+1|0}if((l|0)==1188){f3(f,b,20);g=0;h=g;return h|0}else if((l|0)==1183){g=-1;h=g;return h|0}else if((l|0)==1180){g=-1;h=g;return h|0}else if((l|0)==1172){g=-1;h=g;return h|0}else if((l|0)==1177){g=-1;h=g;return h|0}else if((l|0)==1185){g=-1;h=g;return h|0}else if((l|0)==1174){g=-1;h=g;return h|0}}}while(0);g=-1;h=g;return h|0}else if((l|0)==1136){g=-1;h=g;return h|0}else if((l|0)==1142){g=-1;h=g;return h|0}else if((l|0)==1125){g=-1;h=g;return h|0}else if((l|0)==1129){g=-1;h=g;return h|0}else if((l|0)==1139){g=-1;h=g;return h|0}else if((l|0)==1131){g=-1;h=g;return h|0}return 0}function gn(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;f=b;b=c;c=e;e=0;if((c|0)<1084){g=1084-c|0;h=g;return h|0}if((jk(f+e+1080|0)|0)==0){g=-1;h=g;return h|0}i=0;j=0;k=0;while(1){if((k|0)>=31){l=1236;break}m=e+(k*30|0)|0;n=((ji(f+m+22|0)|0)&65535)<<1;o=((ji(f+m+26|0)|0)&65535)<<1;p=((ji(f+m+28|0)|0)&65535)<<1;j=j+n|0;if((n+2|0)<(o+p|0)){l=1219;break}if((n|0)>65535){l=1223;break}if((o|0)>65535){l=1223;break}if((p|0)>65535){l=1223;break}if((d[f+(m+25)|0]|0|0)>64){l=1225;break}if(((ji(f+m+20|0)|0)&65535|0)!=0){if((n|0)==0){l=1228;break}}if((d[f+(m+25)|0]|0|0)!=0){if((n|0)==0){l=1231;break}}if((n|0)!=0){i=n+1|0}k=k+1|0}if((l|0)==1225){g=-1;h=g;return h|0}else if((l|0)==1219){g=-1;h=g;return h|0}else if((l|0)==1223){g=-1;h=g;return h|0}else if((l|0)==1231){g=-1;h=g;return h|0}else if((l|0)==1228){g=-1;h=g;return h|0}else if((l|0)==1236){if((j|0)<=2){g=-1;h=g;return h|0}j=d[f+(e+930)|0]|0;do{if((j|0)<=127){if((j|0)==0){break}k=0;n=0;while(1){if((n|0)>=(j|0)){break}if((d[f+(e+932+n)|0]|0|0)>(k|0)){k=d[f+(e+932+n)|0]|0}if((d[f+(e+932+n)|0]|0|0)>127){l=1246;break}n=n+1|0}if((l|0)==1246){g=-1;h=g;return h|0}n=n+2|0;while(1){if((n|0)==128){break}if((d[f+(e+932+n)|0]|0|0)!=0){l=1252;break}n=n+1|0}if((l|0)==1252){g=-1;h=g;return h|0}k=k+1|0;if((c|0)<(((k<<8)*3|0)+1062|0)){g=((k<<8)*3|0)+1062-c|0;h=g;return h|0}n=0;while(1){if((n|0)>=(k<<8|0)){l=1277;break}m=e+1060+(n*3|0)|0;if((d[f+m|0]|0|0)>116){l=1261;break}if((a[f+m|0]&63|0)>36){l=1263;break}if((a[f+(m+1)|0]&15|0)==12){if((d[f+(m+2)|0]|0|0)>64){l=1266;break}}if((a[f+(m+1)|0]&15|0)==11){if((d[f+(m+2)|0]|0|0)>127){l=1269;break}}if((a[f+(m+1)|0]&15|0)==13){if((d[f+(m+2)|0]|0|0)>64){l=1272;break}}p=(d[f+m|0]|0)>>2&48|(d[f+(m+2)|0]|0)>>4&15;if((p|0)>(i|0)){l=1274;break}n=n+1|0}if((l|0)==1269){g=-1;h=g;return h|0}else if((l|0)==1274){g=-1;h=g;return h|0}else if((l|0)==1263){g=-1;h=g;return h|0}else if((l|0)==1272){g=-1;h=g;return h|0}else if((l|0)==1266){g=-1;h=g;return h|0}else if((l|0)==1277){f3(0,b,0);g=0;h=g;return h|0}else if((l|0)==1261){g=-1;h=g;return h|0}}}while(0);g=-1;h=g;return h|0}return 0}function go(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;e=i;i=i+1032|0;f=e|0;g=b;b=c;c=0;h=2;j=0;f0(b,20)|0;k=0;while(1){if((k|0)>=31){break}f$(b,g,20)|0;jm(b,0);jm(b,0);l=ja(g)|0;m=ja(g)|0;n=((l&255)<<8)+(m&255)|0;if((n|0)!=0){if((n|0)<256){c=16-(m&255)&255}else{c=256-(m&255)&255}}else{c=0}o=(jd(g)|0)&65535;p=o;jo(b,o&65535);j=j+(p<<1)|0;ja(g)|0;jm(b,c);jm(b,ja(g)|0);n=(jd(g)|0)&65535;o=(jd(g)|0)&65535;do{if(((n<<1)+o|0)<=(p|0)){if((n|0)==0){break}h=1;n=n<<1}}while(0);jo(b,n&65535);jo(b,o&65535);k=k+1|0}h=ja(g)|0;jm(b,h);jm(b,127);ja(g)|0;bB(f|0,128,1,g|0)|0;ay(f|0,128,1,b|0)|0;k=0;h=0;while(1){if((k|0)>=128){break}if((d[f+k|0]|0|0)>(h&255|0)){h=a[f+k|0]|0}k=k+1|0}h=h+1&255;jq(b,1294879534);k=0;while(1){if((k|0)>=(h&255|0)){break}n=0;while(1){if((n|0)>=256){break}l=ja(g)|0;m=ja(g)|0;c=ja(g)|0;p=((l&255)>>2&16|(m&255)>>4&15)&255;q=l&255&63;r=m&255&15;s=c;if((r&255|0)==13){c=((s&255|0)/10|0)&255;s=((c&255)<<4)+(((s&255|0)%10|0)&255)&255}a[f+(n<<2)|0]=p&255&240;c=f+(n<<2)|0;a[c]=(d[c]|0|(d[5440+((q&255)<<1)|0]|0))&255;a[f+((n<<2)+1)|0]=a[5441+((q&255)<<1)|0]|0;a[f+((n<<2)+2)|0]=((p&255)<<4&240|r&255)&255;a[f+((n<<2)+3)|0]=s;n=n+1|0}ay(f|0,1024,1,b|0)|0;k=k+1|0}f$(b,g,j)|0;i=e;return 0}function gp(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0;e=a;a=b;b=c;c=0;if((b|0)<1082){f=1082-b|0;g=f;return g|0}do{if((d[e+1080|0]|0|0)==87){if((d[e+1081|0]|0|0)!=78){break}if((d[e+(c+951)|0]|0|0)!=127){f=-1;g=f;return g|0}if((d[e+(c+950)|0]|0|0)>127){f=-1;g=f;return g|0}else{f3(e,a,20);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function gq(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;e=i;i=i+1024|0;f=e|0;g=b;b=c;c=0;f$(b,g,950)|0;h=0;while(1){if((h|0)>=31){break}bK(g|0,(h*30|0)+42|0,0)|0;c=c+(((jd(g)|0)&65535)<<1)|0;h=h+1|0}bK(g|0,950,0)|0;j=ja(g)|0;jm(b,j);bB(f|0,129,1,g|0)|0;ay(f|0,129,1,b|0)|0;jq(b,1294879534);h=0;j=0;while(1){if((h|0)>=128){break}if((d[f+(h+1)|0]|0|0)>(j&255|0)){j=a[f+(h+1)|0]|0}h=h+1|0}j=j+1&255;bK(g|0,1084,0)|0;h=0;while(1){if((h|0)>=(j&255|0)){break}f=0;while(1){if((f|0)>=256){break}k=ja(g)|0;l=ja(g)|0;m=ja(g)|0;n=ja(g)|0;jm(b,((k&255)*240|0|(d[5440+(((k&255|0)/2|0)<<1)|0]|0))&255);jm(b,a[5441+(((k&255|0)/2|0)<<1)|0]|0);jm(b,((l&255)<<4&240|m&255)&255);jm(b,n);f=f+1|0}h=h+1|0}f$(b,g,c)|0;i=e;return 0}function gr(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=a;a=b;b=c;c=0;if((b|0)<2048){f=2048-b|0;g=f;return g|0}if((d[e+3|0]|0|0)!=60){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=128){break}h=((d[e+(c+(b<<2))|0]|0)<<24)+((d[e+(c+(b<<2)+1)|0]|0)<<16)+((d[e+(c+(b<<2)+2)|0]|0)<<8)+(d[e+(c+(b<<2)+3)|0]|0)|0;i=((h|0)/4|0)<<2;if((i|0)!=(h|0)){j=1375;break}if((h|0)>132156){j=1375;break}b=b+1|0}if((j|0)==1375){f=-1;g=f;return g|0}h=0;while(1){if((h|0)>=64){break}if((d[e+(c+3+(h<<2))|0]|0|0)!=60){if((d[e+(c+3+(h<<2))|0]|0|0)!=0){j=1382;break}}h=h+1|0}if((j|0)==1382){f=-1;g=f;return g|0}h=0;while(1){if((h|0)>=31){break}if((d[e+(c+519+(h<<4))|0]|0|0)>64){j=1388;break}h=h+1|0}if((j|0)==1388){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=30){j=1400;break}i=jk(e+c+526+(b<<4)|0)|0;h=((ji(e+c+524+(b<<4)|0)|0)&65535)<<1;k=jk(e+c+520+(b+1<<4)|0)|0;if((i|0)<2108){j=1395;break}if((k|0)<2108){j=1395;break}if((i|0)>(k|0)){j=1397;break}b=b+1|0}if((j|0)==1397){f=-1;g=f;return g|0}else if((j|0)==1395){f=-1;g=f;return g|0}else if((j|0)==1400){f3(0,a,0);f=0;g=f;return g|0}return 0}function gs(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;e=i;i=i+1160|0;f=e|0;g=e+128|0;h=b;b=c;c=0;j=0;p7(f|0,0,128);p7(g|0,0,1025);f0(b,20)|0;bK(h|0,518,0)|0;k=0;while(1){if((k|0)>=31){break}f0(b,22)|0;l=ja(h)|0;m=ja(h)|0;n=jl(h)|0;o=(jd(h)|0)&65535;p=jl(h)|0;q=(jd(h)|0)&65535;jo(b,q&65535);j=j+(q<<1)|0;n=n-p|0;jm(b,l);jm(b,m);jo(b,((n|0)/2|0)&65535);jo(b,o&65535);jd(h)|0;k=k+1|0}bK(h|0,0,0)|0;o=0;c=0;while(1){if((o&255|0)>=128){break}p=jl(h)|0;if((p|0)==0){m=1417;break}a[f+(o&255)|0]=((p-60|0)/1024|0)-1&255;if((d[f+(o&255)|0]|0|0)>(c&255|0)){c=a[f+(o&255)|0]|0}o=o+1&255}c=c+1&255;jm(b,o);jm(b,127);ay(f|0,128,1,b|0)|0;jq(b,1294879534);bK(h|0,1084,0)|0;k=0;while(1){if((k|0)>=(c&255|0)){break}n=0;while(1){if((n|0)>=256){break}f=((ja(h)|0)&255)>>3&31;o=ja(h)|0;p=ja(h)|0;m=ja(h)|0;l=p&255;if((l|0)==136){p=14;m=(m&255|160)&255}else if((l|0)==140){p=14;m=(m&255|176)&255}else if((l|0)==8){p=1}else if((l|0)==80){p=15}else if((l|0)==92){p=14;m=(m&255|16)&255}else if((l|0)==0){p=0}else if((l|0)==24){p=4}else if((l|0)==28){p=4}else if((l|0)==16){p=3}else if((l|0)==44){p=6}else if((l|0)==132){p=14;m=(m&255|144)&255}else if((l|0)==60){p=10;r=(m&255)<<4&240;s=(m&255)>>4&15;m=(r&255|s&255)&255}else if((l|0)==40){p=6;r=(m&255)<<4&240;s=(m&255)>>4&15;m=(r&255|s&255)&255}else if((l|0)==36){p=5}else if((l|0)==88){p=14;m=1}else if((l|0)==72){p=12}else if((l|0)==56){p=9}else if((l|0)==20){p=3}else if((l|0)==68){p=11}else if((l|0)==4){p=0}else if((l|0)==12){p=2}else if((l|0)==64){p=10}else if((l|0)==96){p=14;m=(m&255|32)&255}else if((l|0)==76){p=13}else if((l|0)==148){p=14;m=(m&255|208)&255}else if((l|0)==152){p=14;m=(m&255|224)&255}else{m=0;p=0}a[g+(n<<2)|0]=f&255&240;l=g+(n<<2)|0;a[l]=(d[l]|0|(d[5440+(((o&255|0)/2|0)<<1)|0]|0))&255;a[g+((n<<2)+1)|0]=a[5441+(((o&255|0)/2|0)<<1)|0]|0;a[g+((n<<2)+2)|0]=(f&255)<<4&240;f=g+((n<<2)+2)|0;a[f]=(d[f]|0|p&255)&255;a[g+((n<<2)+3)|0]=m;n=n+1|0}ay(g|0,1024,1,b|0)|0;k=k+1|0}f$(b,h,j)|0;i=e;return 0}function gt(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;e=a;a=b;b=c;c=0;if((b|0)<505){f=505-b|0;g=f;return g|0}h=jk(e+c|0)|0;do{if((h|0)>=502){if((h|0)>2163190){break}i=0;while(1){if((i|0)>=31){j=1474;break}if((d[e+(c+9+(i<<4))|0]|0|0)>64){j=1469;break}if((((ji(e+c+6+(i<<4)|0)|0)&65535|0)%72|0|0)!=0){j=1471;break}i=i+1|0}if((j|0)==1471){f=-1;g=f;return g|0}else if((j|0)==1474){k=0;i=0;while(1){if((i|0)>=31){j=1485;break}l=((ji(e+c+10+(i<<4)|0)|0)&65535)<<1;m=((ji(e+c+12+(i<<4)|0)|0)&65535)<<1;n=jk(e+c+14+(i<<4)|0)|0;if((l|0)>65535){j=1478;break}if((m|0)>65535){j=1478;break}if((n|0)<(h|0)){j=1480;break}if((n|0)>(k|0)){k=n;}i=i+1|0}if((j|0)==1478){f=-1;g=f;return g|0}else if((j|0)==1485){n=d[e+(c+5)|0]|0;do{if((n|0)<=127){if((n|0)==0){break}if((b|0)<(c+h+(n<<2)+4|0)){f=c+h+(n<<2)+4-b|0;g=f;return g|0}if((jk(e+c+h+(n<<2)|0)|0)!=-1){f=-1;g=f;return g|0}else{f3(0,a,0);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}else if((j|0)==1480){f=-1;g=f;return g|0}}else if((j|0)==1469){f=-1;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function gu(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;f=i;i=i+2176|0;g=f|0;h=f+1024|0;j=f+1152|0;k=f+1664|0;l=b;b=e;e=0;m=999999;p7(j|0,0,512);p7(k|0,0,512);p7(h|0,0,128);n=jl(l)|0;o=ja(l)|0;p=ja(l)|0;f0(b,20)|0;q=0;while(1){if((q|0)>=31){break}f0(b,22)|0;r=(((jd(l)|0)&65535|0)/72|0)&255;ja(l)|0;s=ja(l)|0;t=(jd(l)|0)&65535;u=t;jo(b,t&65535);e=e+(u<<1)|0;jm(b,r);jm(b,s);u=(jd(l)|0)&65535;v=jl(l)|0;if((v|0)<(m|0)){m=v}w=(((jl(l)|0)-v|0)>>>0)/2|0;jo(b,w&65535);jo(b,u&65535);q=q+1|0}jm(b,p);jm(b,127);bK(l|0,n|0,0)|0;q=0;while(1){if((q|0)>=(p&255|0)){break}c[j+(q<<2)>>2]=jl(l)|0;q=q+1|0}n=0;q=0;while(1){if((q|0)>=(p&255|0)){break}if((q|0)==0){a[h|0]=0;c[k>>2]=c[j>>2];n=n+1&255}else{w=0;while(1){if((w|0)>=(q|0)){break}if((c[j+(q<<2)>>2]|0)==(c[j+(w<<2)>>2]|0)){x=1523;break}w=w+1|0}if((x|0)==1523){x=0;a[h+q|0]=a[h+w|0]|0}if((w|0)==(q|0)){c[k+((n&255)<<2)>>2]=c[j+(q<<2)>>2];a[h+q|0]=n;n=n+1&255}}q=q+1|0}ay(h|0,128,1,b|0)|0;jq(b,1294879534);q=0;while(1){if((q|0)>(o&255|0)){break}p7(g|0,0,1024);bK(l|0,c[k+(q<<2)>>2]|0,0)|0;w=0;while(1){if((w|0)>=256){break}h=ja(l)|0;j=ja(l)|0;x=ja(l)|0;n=ja(l)|0;p=((j&127|0)/2|0)&255;u=((j&255)<<4&16|(x&255)>>4&15)&255;v=h&255;a[g+(v<<2)|0]=u&255&240;j=g+(v<<2)|0;a[j]=(d[j]|0|(d[5440+((p&255)<<1)|0]|0))&255;a[g+((v<<2)+1)|0]=a[5441+((p&255)<<1)|0]|0;a[g+((v<<2)+2)|0]=(x&15&255|(u&255)<<4&240)&255;a[g+((v<<2)+3)|0]=n;w=h&255;w=w+1|0}ay(g|0,1024,1,b|0)|0;q=q+1|0}bK(l|0,m|0,0)|0;f$(b,l,e)|0;i=f;return 0}function gv(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1347636734){d=-1;e=d;return e|0}else{di(c,b,60);d=0;e=d;return e|0}return 0}function gw(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;f=i;i=i+1280|0;g=f|0;h=f+1024|0;j=b;b=d;d=e;e=j|0;bK(b|0,d|0,0)|0;jl(b)|0;bB(g|0,1,60,b|0)|0;p5(e|0,g|0,64)|0;k=(ja(b)|0)&255;l=(ja(b)|0)&255;ja(b)|0;if((k&1|0)!=0){m=-1;n=m;i=f;return n|0}dm(j,29864,(k=i,i=i+16|0,c[k>>2]=(l&240)>>4,c[k+8>>2]=l&15,k)|0);i=k;c[e+148>>2]=(ja(b)|0)&255;c[e+152>>2]=(ja(b)|0)&255;ja(b)|0;jc(b)|0;c[e+156>>2]=(jc(b)|0)&65535;c[e+128>>2]=(jc(b)|0)&65535;c[e+140>>2]=(jc(b)|0)&65535;c[e+136>>2]=(jc(b)|0)&65535;jc(b)|0;c[e+144>>2]=c[e+140>>2];c[e+132>>2]=ac(c[e+128>>2]|0,c[e+136>>2]|0)|0;k=jg(b)|0;l=jg(b)|0;o=jg(b)|0;p=jg(b)|0;bK(b|0,d+k|0,0)|0;bB(e+952|0,1,c[e+156>>2]|0,b|0)|0;bK(b|0,d+l|0,0)|0;bB(g|0,1,16,b|0)|0;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}bK(b|0,d+p|0,0)|0;p=0;while(1){if((p|0)>=(c[e+140>>2]|0)){break}l=pT(64,1)|0;c[(c[e+176>>2]|0)+(p*764|0)+756>>2]=l;bB(g|0,1,13,b|0)|0;bB(g|0,1,24,b|0)|0;p5((c[e+176>>2]|0)+(p*764|0)|0,g|0,24)|0;hy((c[e+176>>2]|0)+(p*764|0)|0)|0;c[h+(p<<2)>>2]=jg(b)|0;jg(b)|0;jc(b)|0;l=(ja(b)|0)&255;k=jg(b)|0;c[(c[e+180>>2]|0)+(p*52|0)+32>>2]=k;k=jg(b)|0;c[(c[e+180>>2]|0)+(p*52|0)+36>>2]=k;k=jg(b)|0;c[(c[e+180>>2]|0)+(p*52|0)+40>>2]=k;k=(((ja(b)|0)&255)<<4&255)<<24>>24;q=(ja(b)|0)&255;c[c[(c[e+176>>2]|0)+(p*764|0)+756>>2]>>2]=q;q=((((jc(b)|0)&65535)*8363|0|0)/8448|0)&65535;c[(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]=p;c[(c[e+176>>2]|0)+(p*764|0)+36>>2]=((c[(c[e+180>>2]|0)+(p*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[e+180>>2]|0)+(p*52|0)+44>>2]=(l&128|0)!=0?2:0;r=(c[e+180>>2]|0)+(p*52|0)+44|0;c[r>>2]=c[r>>2]|((l&32|0)!=0?4:0);hD(q&65535,(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+12|0,(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+16|0);q=(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+16|0;c[q>>2]=(c[q>>2]|0)+k;p=p+1|0}c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;bK(b|0,d+o|0,0)|0;p=0;while(1){if((p|0)>=(c[e+128>>2]|0)){break}o=((jc(b)|0)&65535)-4|0;g=ja(b)|0;ja(b)|0;k=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(p<<2)>>2]=k;c[c[(c[e+168>>2]|0)+(p<<2)>>2]>>2]=g&255;k=0;while(1){if((k|0)>=(c[e+136>>2]|0)){break}q=(ac(p,c[e+136>>2]|0)|0)+k|0;c[(c[(c[e+168>>2]|0)+(p<<2)>>2]|0)+4+(k<<2)>>2]=q;l=pT(((c[c[(c[e+168>>2]|0)+(p<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[e+172>>2]|0)+(q<<2)>>2]=l;c[c[(c[e+172>>2]|0)+(q<<2)>>2]>>2]=c[c[(c[e+168>>2]|0)+(p<<2)>>2]>>2];k=k+1|0}k=0;while(1){if((k|0)>=(g&255|0)){break}while(1){if((o|0)<=0){break}q=ja(b)|0;o=o-1|0;if((q&255|0)==0){s=1584;break}l=(c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(p<<2)>>2]|0)+4+((q&15)<<2)>>2]<<2)>>2]|0)+4+(k<<3)|0;if((q&128|0)!=0){a[l|0]=((ja(b)|0)&255)+37&255;a[l+1|0]=ja(b)|0;o=o-2|0}if((q&64|0)!=0){a[l+2|0]=((ja(b)|0)&255)+1&255;o=o-1|0}if((q&32|0)!=0){a[l+3|0]=ja(b)|0;a[l+4|0]=ja(b)|0;o=o-2|0}}if((s|0)==1584){s=0}k=k+1|0}if((o|0)>0){k=b;g=o;bK(k|0,g|0,1)|0}p=p+1|0}p=0;while(1){if((p|0)>=(c[e+140>>2]|0)){break}bK(b|0,d+(c[h+(p<<2)>>2]|0)|0,0)|0;gQ(j,b,1,(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;p=p+1|0}m=0;n=m;i=f;return n|0}function gx(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;a=b;if((jl(c)|0)!=1179603533){d=-1;e=d;return e|0}jl(c)|0;if((jl(c)|0)!=1297040460){d=-1;e=d;return e|0}if((jl(c)|0)!=1447383635){d=-1;e=d;return e|0}jl(c)|0;bK(c|0,10,1)|0;if((jl(c)|0)==1229866575){b=c;jl(b)|0;di(c,a,32)}else{di(c,a,0)}d=0;e=d;return e|0}function gy(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+24|0;f=e|0;g=a;a=b;bK(a|0,d|0,0)|0;jl(a)|0;jl(a)|0;jl(a)|0;jl(a)|0;jl(a)|0;bB(f|0,1,10,a|0)|0;dm(g,29808,(d=i,i=i+8|0,c[d>>2]=f+4,d)|0);i=d;d=es()|0;if((d|0)==0){h=-1;j=h;i=e;return j|0}ev(d,27864,138);ev(d,26704,140);ev(d,25640,144);ey(d,2);while(1){if(!((bH(a|0)|0)!=0^1)){break}et(d,g,a,0)}eB(d);h=0;j=h;i=e;return j|0}function gz(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;return}function gA(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;e=d;d=a|0;bB(d|0,1,32,e|0)|0;c[d+140>>2]=(jd(e)|0)&65535;c[d+156>>2]=(jd(e)|0)&65535;c[d+128>>2]=(jd(e)|0)&65535;c[d+164>>2]=(jd(e)|0)&65535;c[d+152>>2]=(jd(e)|0)&65535;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;jd(e)|0;return}function gB(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;gC(a,c,0)|0;return}function gC(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0;h=i;i=i+1096|0;j=h|0;k=h+1088|0;l=f;f=g;g=l|0;bB(j|0,20,1,f|0)|0;m=0;while(1){if((m|0)>=31){break}bB(j+20+(m*30|0)|0,22,1,f|0)|0;b[j+20+(m*30|0)+22>>1]=jd(f)|0;a[j+20+(m*30|0)+24|0]=ja(f)|0;a[j+20+(m*30|0)+25|0]=ja(f)|0;b[j+20+(m*30|0)+26>>1]=jd(f)|0;b[j+20+(m*30|0)+28>>1]=jd(f)|0;m=m+1|0}a[j+950|0]=ja(f)|0;a[j+951|0]=ja(f)|0;bB(j+952|0,128,1,f|0)|0;bB(j+1080|0,4,1,f|0)|0;c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];c[g+136>>2]=4;c[g+156>>2]=d[j+950|0]|0;c[g+160>>2]=d[j+951|0]|0;n=g+952|0;o=j+952|0;p3(n|0,o|0,128)|0;m=0;while(1){if((m|0)>=128){break}if((d[g+952+m|0]|0)>(c[g+128>>2]|0)){c[g+128>>2]=d[g+952+m|0]|0}m=m+1|0}o=g+128|0;c[o>>2]=(c[o>>2]|0)+1;c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}m=0;while(1){if((m|0)>=(c[g+140>>2]|0)){break}o=pT(64,1)|0;c[(c[g+176>>2]|0)+(m*764|0)+756>>2]=o;c[(c[g+180>>2]|0)+(m*52|0)+32>>2]=e[j+20+(m*30|0)+22>>1]<<1;c[(c[g+180>>2]|0)+(m*52|0)+36>>2]=e[j+20+(m*30|0)+26>>1]<<1;c[(c[g+180>>2]|0)+(m*52|0)+40>>2]=(c[(c[g+180>>2]|0)+(m*52|0)+36>>2]|0)+(e[j+20+(m*30|0)+28>>1]<<1);c[(c[g+180>>2]|0)+(m*52|0)+44>>2]=(e[j+20+(m*30|0)+28>>1]|0)>1?2:0;c[(c[(c[g+176>>2]|0)+(m*764|0)+756>>2]|0)+16>>2]=(a[j+20+(m*30|0)+24|0]<<4&255)<<24>>24;c[c[(c[g+176>>2]|0)+(m*764|0)+756>>2]>>2]=a[j+20+(m*30|0)+25|0]|0;c[(c[(c[g+176>>2]|0)+(m*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(m*764|0)+756>>2]|0)+40>>2]=m;c[(c[g+176>>2]|0)+(m*764|0)+36>>2]=((c[(c[g+180>>2]|0)+(m*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[g+176>>2]|0)+(m*764|0)+40>>2]=4095;dh((c[g+176>>2]|0)+(m*764|0)|0,j+20+(m*30|0)|0,22)|0;m=m+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;m=0;while(1){if((m|0)>=(c[g+128>>2]|0)){break}j=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(m<<2)>>2]=j;c[c[(c[g+168>>2]|0)+(m<<2)>>2]>>2]=64;j=0;while(1){if((j|0)>=(c[g+136>>2]|0)){break}o=(ac(m,c[g+136>>2]|0)|0)+j|0;c[(c[(c[g+168>>2]|0)+(m<<2)>>2]|0)+4+(j<<2)>>2]=o;n=pT(((c[c[(c[g+168>>2]|0)+(m<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(o<<2)>>2]=n;c[c[(c[g+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(m<<2)>>2]>>2];j=j+1|0}j=0;while(1){if((j|0)>=256){break}o=(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(m<<2)>>2]|0)+4+(((j|0)%4|0)<<2)>>2]<<2)>>2]|0)+4+(((j|0)/4|0)<<3)|0;bB(k|0,1,4,f|0)|0;dj(o,k|0);j=j+1|0}m=m+1|0}k=l+1280|0;c[k>>2]=c[k>>2]|8192;m=0;while(1){if((m|0)>=(c[g+144>>2]|0)){break}if((c[(c[g+180>>2]|0)+(m*52|0)+32>>2]|0)!=0){k=l;j=f;o=(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(m*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(k,j,0,o,0)|0}m=m+1|0}i=h;return 0}function gD(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;a=c;bK(d|0,a+44|0,0)|0;if((jl(d)|0)!=1347702086){e=-1;f=e;return f|0}else{c=d;g=a|0;bK(c|0,g|0,0)|0;di(d,b,28);e=0;f=e;return f|0}return 0}function gE(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;j=i;i=i+1720|0;k=j|0;l=j+1024|0;m=j+1632|0;n=f;f=g;g=h;h=n|0;bK(f|0,g|0,0)|0;bB(l|0,28,1,f|0)|0;a[l+28|0]=ja(f)|0;a[l+29|0]=ja(f)|0;a[l+30|0]=ja(f)|0;a[l+31|0]=ja(f)|0;b[l+32>>1]=jc(f)|0;b[l+34>>1]=jc(f)|0;b[l+36>>1]=jc(f)|0;b[l+38>>1]=jc(f)|0;b[l+40>>1]=jc(f)|0;b[l+42>>1]=jc(f)|0;c[l+44>>2]=jl(f)|0;bB(l+48|0,16,1,f|0)|0;bB(l+64|0,32,1,f|0)|0;bB(l+96|0,256,1,f|0)|0;o=0;while(1){if((o|0)>=128){break}b[l+352+(o<<1)>>1]=jc(f)|0;o=o+1|0}c[h+156>>2]=e[l+32>>1]|0;c[h+140>>2]=e[l+34>>1]|0;c[h+128>>2]=e[l+36>>1]|0;c[h+136>>2]=e[l+38>>1]|0;c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;c[h+144>>2]=c[h+140>>2];c[h+148>>2]=6;c[h+152>>2]=125;p=h+952|0;q=l+96|0;p3(p|0,q|0,256)|0;c[n+1264>>2]=8363;dh(h|0,l|0,28)|0;q=d[l+29|0]|0;dm(n,29768,(p=i,i=i+16|0,c[p>>2]=d[l+30|0]|0,c[p+8>>2]=q,p)|0);i=p;c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}o=0;while(1){if((o|0)>=(c[h+140>>2]|0)){break}p=pT(64,1)|0;c[(c[h+176>>2]|0)+(o*764|0)+756>>2]=p;a[m|0]=ja(f)|0;bB(m+1|0,12,1,f|0)|0;a[m+13|0]=ja(f)|0;b[m+14>>1]=jc(f)|0;b[m+16>>1]=jc(f)|0;c[m+20>>2]=jg(f)|0;c[m+24>>2]=jg(f)|0;c[m+28>>2]=jg(f)|0;c[m+32>>2]=jg(f)|0;c[m+36>>2]=jg(f)|0;c[m+40>>2]=jg(f)|0;c[m+44>>2]=jg(f)|0;a[m+48|0]=ja(f)|0;a[m+49|0]=ja(f)|0;bB(m+50|0,28,1,f|0)|0;c[m+80>>2]=jl(f)|0;if((a[m|0]&3|0)==1){c[k+(o<<2)>>2]=c[m+20>>2];c[(c[h+180>>2]|0)+(o*52|0)+32>>2]=c[m+24>>2];c[(c[h+176>>2]|0)+(o*764|0)+36>>2]=(c[m+24>>2]|0)>>>0>0?1:0;c[(c[h+180>>2]|0)+(o*52|0)+36>>2]=c[m+28>>2];c[(c[h+180>>2]|0)+(o*52|0)+40>>2]=c[m+32>>2];c[(c[h+180>>2]|0)+(o*52|0)+44>>2]=0;if((a[m|0]&4|0)!=0){p=(c[h+180>>2]|0)+(o*52|0)+44|0;c[p>>2]=c[p>>2]|2}if((a[m|0]&8|0)!=0){p=(c[h+180>>2]|0)+(o*52|0)+44|0;c[p>>2]=c[p>>2]|6}if((a[m|0]&16|0)!=0){p=(c[h+180>>2]|0)+(o*52|0)+44|0;c[p>>2]=c[p>>2]|1;p=(c[h+180>>2]|0)+(o*52|0)+32|0;c[p>>2]=c[p>>2]>>1;p=(c[h+180>>2]|0)+(o*52|0)+36|0;c[p>>2]=c[p>>2]>>1;p=(c[h+180>>2]|0)+(o*52|0)+40|0;c[p>>2]=c[p>>2]>>1}c[c[(c[h+176>>2]|0)+(o*764|0)+756>>2]>>2]=d[m+13|0]|0;c[(c[(c[h+176>>2]|0)+(o*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]=o;c[m+80>>2]=0;p=(c[h+176>>2]|0)+(o*764|0)|0;q=m+50|0;dh(p,q,28)|0;hD(e[m+14>>1]|0,(c[(c[h+176>>2]|0)+(o*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(o*764|0)+756>>2]|0)+16|0)}o=o+1|0}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;o=0;while(1){if((o|0)>=(c[h+128>>2]|0)){break}if((b[l+352+(o<<1)>>1]|0)!=0){m=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(o<<2)>>2]=m;c[c[(c[h+168>>2]|0)+(o<<2)>>2]>>2]=64;m=0;while(1){if((m|0)>=(c[h+136>>2]|0)){break}q=(ac(o,c[h+136>>2]|0)|0)+m|0;c[(c[(c[h+168>>2]|0)+(o<<2)>>2]|0)+4+(m<<2)>>2]=q;p=pT(((c[c[(c[h+168>>2]|0)+(o<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(q<<2)>>2]=p;c[c[(c[h+172>>2]|0)+(q<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(o<<2)>>2]>>2];m=m+1|0}bK(f|0,g+(e[l+352+(o<<1)>>1]<<4)|0,0)|0;m=0;while(1){if((m|0)>=64){break}q=ja(f)|0;if(q<<24>>24==0){m=m+1|0;continue}p=q&31;if((p|0)>=(c[h+136>>2]|0)){continue}r=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(o<<2)>>2]|0)+4+(p<<2)>>2]<<2)>>2]|0)+4+(m<<3)|0;if((q&32|0)!=0){p=ja(f)|0;s=p&255;if((s|0)==255){p=0}else if((s|0)==254){p=-127}else{p=(p&255)+12&255}a[r|0]=p;a[r+1|0]=ja(f)|0}if((q&64|0)!=0){a[r+3|0]=ja(f)|0;a[r+4|0]=ja(f)|0;if((d[r+3|0]|0)>23){a[r+4|0]=0;a[r+3|0]=0}p=d[r+3|0]|0;if((p|0)==14){if(((a[r+4|0]&240)>>4|0)==8){a[r+3|0]=8;a[r+4|0]=(a[r+4|0]&15)<<4&255}}else if((p|0)==16){a[r+3|0]=16}else if((p|0)==17){a[r+3|0]=27}else if((p|0)==18){a[r+3|0]=-84}else if((p|0)==19){a[r+3|0]=-100}else if((p|0)==20){a[r+3|0]=-99}else if((p|0)==21){a[r+3|0]=-97}else if((p|0)==22){a[r+3|0]=-98}else if((p|0)==23){a[r+4|0]=0;a[r+3|0]=0}}if((q&128|0)!=0){a[r+2|0]=((ja(f)|0)&255)+1&255}}}o=o+1|0}o=0;while(1){if((o|0)>=(c[h+144>>2]|0)){break}do{if((c[(c[h+176>>2]|0)+(o*764|0)+36>>2]|0)!=0){m=c[(c[(c[h+176>>2]|0)+(o*764|0)+756>>2]|0)+40>>2]|0;r=(c[h+180>>2]|0)+(m*52|0)|0;if((c[r+32>>2]|0)==0){break}else{q=f;p=g+(c[k+(m<<2)>>2]|0)|0;bK(q|0,p|0,0)|0;p=n;q=f;m=r;gQ(p,q,4,m,0)|0;break}}}while(0);o=o+1|0}c[n+1276>>2]=5160;o=0;while(1){if((o|0)>=(c[h+136>>2]|0)){break}c[h+184+(o*12|0)>>2]=d[l+64+o|0]<<4;o=o+1|0}o=n+1280|0;c[o>>2]=c[o>>2]|545;c[n+1284>>2]=2;i=j;return 0}function gF(a,b,c){a=a|0;b=b|0;c=c|0;return gH(a,b,c,0)|0}function gG(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;j=i;i=i+5200|0;k=j|0;l=j+1088|0;m=j+1096|0;n=j+1104|0;o=f;f=g;g=o|0;if((hz(n|0,4096)|0)<0){p=-1;q=p;i=j;return q|0}bI(n|0,29752,4096)|0;r=bM(n|0)|0;s=r;if((r|0)<0){p=-1;q=p;i=j;return q|0}if((f1(aE(f|0)|0,s,m)|0)<0){r=s;aQ(r|0)|0;r=n|0;bU(r|0)|0;p=-1;q=p;i=j;return q|0}r=bl(s|0,27856)|0;f=r;if((r|0)==0){r=s;aQ(r|0)|0;r=n|0;bU(r|0)|0;p=-1;q=p;i=j;return q|0}bK(f|0,h|0,0)|0;bB(k|0,20,1,f|0)|0;h=0;while(1){if((h|0)>=31){break}bB(k+20+(h*30|0)|0,22,1,f|0)|0;b[k+20+(h*30|0)+22>>1]=jd(f)|0;a[k+20+(h*30|0)+24|0]=ja(f)|0;a[k+20+(h*30|0)+25|0]=ja(f)|0;b[k+20+(h*30|0)+26>>1]=jd(f)|0;b[k+20+(h*30|0)+28>>1]=jd(f)|0;h=h+1|0}a[k+950|0]=ja(f)|0;a[k+951|0]=ja(f)|0;bB(k+952|0,128,1,f|0)|0;bB(k+1080|0,4,1,f|0)|0;if((p6(k+1080|0,26672,4)|0)!=0){r=f;au(r|0)|0;r=n|0;bU(r|0)|0;p=-1;q=p;i=j;return q|0}c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];c[g+136>>2]=4;c[g+156>>2]=d[k+950|0]|0;c[g+160>>2]=d[k+951|0]|0;r=g+952|0;s=k+952|0;p3(r|0,s|0,128)|0;h=0;while(1){if((h|0)>=128){break}if((c[g+136>>2]|0)>4){s=g+952+h|0;a[s]=d[s]>>1&255}if((d[g+952+h|0]|0)>(c[g+128>>2]|0)){c[g+128>>2]=d[g+952+h|0]|0}h=h+1|0}s=g+128|0;c[s>>2]=(c[s>>2]|0)+1;c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;aM(g|0,64,25632,(s=i,i=i+8|0,c[s>>2]=k,s)|0)|0;i=s;aM(g+64|0,64,25632,(s=i,i=i+8|0,c[s>>2]=c[m>>2],s)|0)|0;i=s;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}s=pT(64,1)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=s;c[(c[g+180>>2]|0)+(h*52|0)+32>>2]=e[k+20+(h*30|0)+22>>1]<<1;c[(c[g+180>>2]|0)+(h*52|0)+36>>2]=e[k+20+(h*30|0)+26>>1]<<1;c[(c[g+180>>2]|0)+(h*52|0)+40>>2]=(c[(c[g+180>>2]|0)+(h*52|0)+36>>2]|0)+(e[k+20+(h*30|0)+28>>1]<<1);c[(c[g+180>>2]|0)+(h*52|0)+44>>2]=(e[k+20+(h*30|0)+28>>1]|0)>1?2:0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+16>>2]=(a[k+20+(h*30|0)+24|0]<<4&255)<<24>>24;c[c[(c[g+176>>2]|0)+(h*764|0)+756>>2]>>2]=a[k+20+(h*30|0)+25|0]|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]=h;c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=((c[(c[g+180>>2]|0)+(h*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[g+176>>2]|0)+(h*764|0)+40>>2]=4095;dh((c[g+176>>2]|0)+(h*764|0)|0,k+20+(h*30|0)|0,22)|0;h=h+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[g+128>>2]|0)){break}k=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(h<<2)>>2]=k;c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]=64;k=0;while(1){if((k|0)>=(c[g+136>>2]|0)){break}s=(ac(h,c[g+136>>2]|0)|0)+k|0;c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(k<<2)>>2]=s;m=pT(((c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(s<<2)>>2]=m;c[c[(c[g+172>>2]|0)+(s<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2];k=k+1|0}k=0;while(1){if((k|0)>=256){break}s=(c[(c[o+172>>2]|0)+(c[(c[(c[o+168>>2]|0)+(h<<2)>>2]|0)+4+(((k|0)%4|0)<<2)>>2]<<2)>>2]|0)+4+(((k|0)/4|0)<<3)|0;bB(l|0,1,4,f|0)|0;dj(s,l|0);k=k+1|0}h=h+1|0}l=o+1280|0;c[l>>2]=c[l>>2]|8192;h=0;while(1){if((h|0)>=(c[g+144>>2]|0)){break}gQ(o,f,0,(c[g+180>>2]|0)+((c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;h=h+1|0}au(f|0)|0;bU(n|0)|0;p=0;q=p;i=j;return q|0}function gH(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0;c=a;a=d;d=65536;b=pT(1,65536)|0;bB(b|0,d|0,1,c|0)|0;while(1){e=f2(b,d,a)|0;f=e;if((e|0)<=0){g=1879;break}e=pV(b,d+f|0)|0;if((e|0)==0){g=1877;break}b=e;bB(b+d|0,f|0,1,c|0)|0;d=d+f|0}if((g|0)==1879){pU(b);h=(f|0)==0?0:-1;i=h;return i|0}else if((g|0)==1877){pU(b);h=-1;i=h;return i|0}return 0}function gI(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+16|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,16,e|0)|0)>>>0<16){f=-1}else{if((p6(d|0,27816,16)|0)!=0){f=-1;break}else{di(e,a,0);f=0;break}}}while(0);i=c;return f|0}function gJ(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;j=i;i=i+80|0;k=j|0;l=j+16|0;m=f;f=g;g=h;h=m|0;bK(f|0,g|0,0)|0;bK(f|0,16,0)|0;n=ja(f)|0;o=ja(f)|0;c[h+136>>2]=9;c[h+152>>2]=125;c[h+148>>2]=o&31;if((c[h+148>>2]|0)<=2){c[h+148>>2]=6}c[h+144>>2]=0;dm(m,29696,(p=i,i=i+16|0,c[p>>2]=(n&240)>>4,c[p+8>>2]=n&15,p)|0);i=p;if((o&128|0)!=0){do{o=ja(f)|0;q=o;}while((o&255|0)!=0)}o=aT(f|0)|0;c[h+140>>2]=0;while(1){p=ja(f)|0;q=p;if((p&255|0)==0){break}c[h+140>>2]=q&255;bB(k|0,1,11,f|0)|0}bK(f|0,o|0,0)|0;c[h+144>>2]=c[h+140>>2];c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}while(1){o=ja(f)|0;q=o;if((o&255|0)==0){break}bB(k|0,1,11,f|0)|0;gQ(m,f,12288,(c[h+180>>2]|0)+(((q&255)-1|0)*52|0)|0,k|0)|0}k=0;while(1){if((k|0)>=(c[h+140>>2]|0)){break}o=pT(64,1)|0;c[(c[h+176>>2]|0)+(k*764|0)+756>>2]=o;c[(c[h+176>>2]|0)+(k*764|0)+36>>2]=1;c[c[(c[h+176>>2]|0)+(k*764|0)+756>>2]>>2]=64;c[(c[(c[h+176>>2]|0)+(k*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(k*764|0)+756>>2]|0)+12>>2]=-1;c[(c[(c[h+176>>2]|0)+(k*764|0)+756>>2]|0)+40>>2]=k;k=k+1|0}c[h+156>>2]=(ja(f)|0)&255;k=0;o=0;while(1){if((k|0)>=(c[h+156>>2]|0)){break}q=ja(f)|0;if((q&255|0)<128){p=o;o=p+1|0;a[h+952+p|0]=q}k=k+1|0}k=0;c[h+128>>2]=0;while(1){if((k|0)>=32){break}b[l+(k<<1)>>1]=jc(f)|0;if((b[l+(k<<1)>>1]|0)!=0){o=h+128|0;c[o>>2]=(c[o>>2]|0)+1}k=k+1|0}c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;k=0;while(1){if((k|0)>=(c[h+128>>2]|0)){break}o=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(k<<2)>>2]=o;c[c[(c[h+168>>2]|0)+(k<<2)>>2]>>2]=64;o=0;while(1){if((o|0)>=(c[h+136>>2]|0)){break}p=(ac(k,c[h+136>>2]|0)|0)+o|0;c[(c[(c[h+168>>2]|0)+(k<<2)>>2]|0)+4+(o<<2)>>2]=p;n=pT(((c[c[(c[h+168>>2]|0)+(k<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(p<<2)>>2]=n;c[c[(c[h+172>>2]|0)+(p<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(k<<2)>>2]>>2];o=o+1|0}if((e[l+(k<<1)>>1]|0)!=0){o=f;p=g+(e[l+(k<<1)>>1]|0)|0;bK(o|0,p|0,0)|0;do{p=ja(f)|0;do{o=ja(f)|0;(o&127|0)>=(c[h+136>>2]|0);n=(c[(c[m+172>>2]|0)+(c[(c[(c[m+168>>2]|0)+(k<<2)>>2]|0)+4+((o&127)<<2)>>2]<<2)>>2]|0)+4+((p&127)<<3)|0;q=ja(f)|0;a[n+1|0]=(q&128)>>3&255;a[n|0]=q&255&15;if((d[n|0]|0)==15){a[n|0]=-127}else{if((a[n|0]|0)!=0){r=n|0;a[r]=(d[r]|0)+((((q&112)>>4)*12|0)+26)&255}}q=ja(f)|0;r=n+1|0;a[r]=(d[r]|(q&240)>>4)&255;a[n+3|0]=q&255&15;if((a[n+3|0]|0)!=0){q=ja(f)|0;a[n+4|0]=q;do{if((d[n+3|0]|0)==15){if((d[n+4|0]|0)>2){break}a[n+4|0]=6}}while(0)}}while((~(o&255)&128|0)!=0);}while((~(p&255)&128|0)!=0)}k=k+1|0}k=0;while(1){if((k|0)>=(c[h+136>>2]|0)){break}c[h+184+(k*12|0)>>2]=128;c[h+184+(k*12|0)+8>>2]=1;k=k+1|0}c[m+6560>>2]=2936;k=m+1280|0;c[k>>2]=c[k>>2]|4096;i=j;return 0}function gK(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+8|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,4,e|0)|0)>>>0<4){f=-1}else{if((p6(d|0,29688,4)|0)!=0){f=-1;break}if(((ja(e)|0)&255|0)!=32){f=-1;break}else{di(e,a,32);f=0;break}}}while(0);i=c;return f|0}function gL(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;j=i;i=i+656|0;k=j|0;l=j+48|0;m=j+184|0;n=j+200|0;o=j+560|0;p=j+592|0;q=j+616|0;r=f;f=g;g=h;h=r|0;bK(f|0,g|0,0)|0;if((gM(f,k,29688)|0)<0){s=-1;t=s;i=j;return t|0}u=e[k+38>>1]|0;bB(p|0,1,20,f|0)|0;a[p+20|0]=0;bB(q|0,1,32,f|0)|0;a[q+32|0]=0;b[l+52>>1]=jc(f)|0;a[l+54|0]=ja(f)|0;a[l+55|0]=ja(f)|0;b[l+56>>1]=jc(f)|0;b[l+58>>1]=jc(f)|0;a[l+60|0]=ja(f)|0;a[l+61|0]=ja(f)|0;bB(l+62|0,32,1,f|0)|0;c[l+96>>2]=jg(f)|0;if((u|0)>=274){q=f;bK(q|0,32,1)|0}q=0;while(1){if((q|0)>=(e[l+56>>1]|0)){break}a[h+952+q|0]=(jc(f)|0)&255;q=q+1|0}p5(h|0,k+5|0,20)|0;aM(h+64|0,64,27776,(v=i,i=i+24|0,c[v>>2]=p,c[v+8>>2]=u>>8,c[v+16>>2]=u&255,v)|0)|0;i=v;c[h+156>>2]=e[l+56>>1]|0;c[h+128>>2]=e[l+58>>1]|0;c[h+136>>2]=d[l+54|0]|0;c[h+132>>2]=(ac(c[h+136>>2]|0,c[h+128>>2]|0)|0)+1;c[h+140>>2]=d[l+55|0]|0;c[h+148>>2]=d[l+60|0]|0;c[h+152>>2]=d[l+61|0]|0;v=r+1280|0;c[v>>2]=c[v>>2]|((b[l+52>>1]&1|0)!=0?4096:0);q=0;while(1){if((q|0)>=(c[h+136>>2]|0)){break}c[h+184+(q*12|0)>>2]=a[l+62+q|0]&255;q=q+1|0}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;v=(e[k+40>>1]|0)+42+(c[l+96>>2]|0)|0;q=0;while(1){if((q|0)>=(c[h+128>>2]|0)){break}bK(f|0,g+v|0,0)|0;if((gM(f,k,26648)|0)<0){w=2018;break}b[m>>1]=jc(f)|0;a[m+2|0]=ja(f)|0;b[m+4>>1]=jc(f)|0;c[m+8>>2]=jg(f)|0;v=v+((e[k+40>>1]|0)+42+(c[m+8>>2]|0))|0;l=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(q<<2)>>2]=l;c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2]=e[m+4>>1]|0;l=0;while(1){if((l|0)>=(c[h+136>>2]|0)){break}p=(ac(q,c[h+136>>2]|0)|0)+l|0;c[(c[(c[h+168>>2]|0)+(q<<2)>>2]|0)+4+(l<<2)>>2]=p;x=pT(((c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(p<<2)>>2]=x;c[c[(c[h+172>>2]|0)+(p<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2];l=l+1|0}l=0;while(1){if((l|0)>=(e[m+4>>1]|0)){break}y=0;while(1){p=ja(f)|0;if((p&255|0)==0){w=2033;break}if((y|0)>=(d[m+2|0]|0)){w=2035;break}x=(c[(c[r+172>>2]|0)+(c[(c[(c[r+168>>2]|0)+(q<<2)>>2]|0)+4+(y<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0;if((p&1|0)!=0){y=(ja(f)|0)&255;x=(c[(c[r+172>>2]|0)+(c[(c[(c[r+168>>2]|0)+(q<<2)>>2]|0)+4+(y<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}if((p&2|0)!=0){a[x|0]=((ja(f)|0)&255)+1&255;if((d[x|0]|0)==255){a[x|0]=-127}else{z=x|0;a[z]=(d[z]|0)+12&255}}if((p&4|0)!=0){a[x+1|0]=ja(f)|0}if((p&8|0)!=0){a[x+3|0]=ja(f)|0}if((p&16|0)!=0){a[x+4|0]=ja(f)|0}if((p&32|0)!=0){a[x+5|0]=ja(f)|0}if((p&64|0)!=0){a[x+6|0]=ja(f)|0}y=y+1|0}if((w|0)==2033){w=0}else if((w|0)==2035){w=0}l=l+1|0}q=q+1|0}if((w|0)==2018){s=-1;t=s;i=j;return t|0}bK(f|0,g+v|0,0)|0;c[h+144>>2]=1024;c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}v=0;q=0;L2543:while(1){if((q|0)>=(c[h+140>>2]|0)){w=2125;break}if((gM(f,k,25608)|0)<0){w=2070;break}dh((c[h+176>>2]|0)+(q*764|0)|0,k+5|0,32)|0;if((e[k+40>>1]|0)==0){a[n|0]=0}else{a[n|0]=ja(f)|0;b[n+2>>1]=jc(f)|0;g=n+4|0;m=f;bB(g|0,120,1,m|0)|0;a[n+124|0]=ja(f)|0;y=0;while(1){if((y|0)>=12){break}c[n+128+(y<<3)>>2]=jg(f)|0;c[n+128+(y<<3)+4>>2]=jg(f)|0;y=y+1|0}a[n+224|0]=ja(f)|0;a[n+225|0]=ja(f)|0;a[n+226|0]=ja(f)|0;b[n+228>>1]=jc(f)|0;a[n+232|0]=ja(f)|0;y=0;while(1){if((y|0)>=12){break}c[n+236+(y<<3)>>2]=jg(f)|0;c[n+236+(y<<3)+4>>2]=jg(f)|0;y=y+1|0}a[n+332|0]=ja(f)|0;a[n+333|0]=ja(f)|0;a[n+334|0]=ja(f)|0;b[n+336>>1]=jc(f)|0;a[n+340|0]=ja(f)|0;a[n+341|0]=ja(f)|0;a[n+342|0]=ja(f)|0;a[n+343|0]=ja(f)|0;b[n+344>>1]=jc(f)|0;if((u|0)>=272){a[n+346|0]=ja(f)|0;a[n+347|0]=ja(f)|0;a[n+348|0]=ja(f)|0;a[n+349|0]=ja(f)|0}if((u|0)>=274){a[n+350|0]=ja(f)|0;a[n+351|0]=ja(f)|0;a[n+352|0]=ja(f)|0;a[n+353|0]=ja(f)|0}c[(c[h+176>>2]|0)+(q*764|0)+36>>2]=d[n|0]|0;if((c[(c[h+176>>2]|0)+(q*764|0)+36>>2]|0)>16){c[(c[h+176>>2]|0)+(q*764|0)+36>>2]=16}m=pT(64,c[(c[h+176>>2]|0)+(q*764|0)+36>>2]|0)|0;c[(c[h+176>>2]|0)+(q*764|0)+756>>2]=m;y=0;while(1){if((y|0)>=120){break}a[(c[h+176>>2]|0)+(q*764|0)+512+(y<<1)|0]=a[n+4+y|0]|0;y=y+1|0}c[(c[h+176>>2]|0)+(q*764|0)+40>>2]=e[n+344>>1]|0;c[(c[h+176>>2]|0)+(q*764|0)+48>>2]=d[n+124|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+56>>2]=d[n+224|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+64>>2]=d[n+225|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+68>>2]=d[n+226|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+44>>2]=e[n+228>>1]|0;c[(c[h+176>>2]|0)+(q*764|0)+204>>2]=d[n+232|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+212>>2]=d[n+332|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+220>>2]=d[n+333|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+224>>2]=d[n+334|0]|0;c[(c[h+176>>2]|0)+(q*764|0)+200>>2]=e[n+336>>1]|0;if((c[(c[h+176>>2]|0)+(q*764|0)+48>>2]|0)<=0){m=(c[h+176>>2]|0)+(q*764|0)+44|0;c[m>>2]=c[m>>2]&-2}if((c[(c[h+176>>2]|0)+(q*764|0)+204>>2]|0)<=0){m=(c[h+176>>2]|0)+(q*764|0)+200|0;c[m>>2]=c[m>>2]&-2}y=0;while(1){if((y|0)>=(c[(c[h+176>>2]|0)+(q*764|0)+48>>2]|0)){break}b[(c[h+176>>2]|0)+(q*764|0)+72+(y<<1<<1)>>1]=c[n+128+(y<<3)>>2]&65535;b[(c[h+176>>2]|0)+(q*764|0)+72+((y<<1)+1<<1)>>1]=((c[n+128+(y<<3)+4>>2]|0)/2|0)&65535;y=y+1|0}y=0;while(1){if((y|0)>=(c[(c[h+176>>2]|0)+(q*764|0)+204>>2]|0)){break}b[(c[h+176>>2]|0)+(q*764|0)+228+(y<<1<<1)>>1]=c[n+236+(y<<3)>>2]&65535;b[(c[h+176>>2]|0)+(q*764|0)+228+((y<<1)+1<<1)>>1]=((c[n+236+(y<<3)+4>>2]|0)/2|0)+32&65535;y=y+1|0}y=0;while(1){if((y|0)>=(c[(c[h+176>>2]|0)+(q*764|0)+36>>2]|0)){break}if((gM(f,k,24696)|0)<0){w=2112;break L2543}b[o>>1]=jc(f)|0;a[o+2|0]=ja(f)|0;a[o+3|0]=ja(f)|0;c[o+4>>2]=jg(f)|0;a[o+8|0]=(jg(f)|0)&255;c[o+12>>2]=jg(f)|0;c[o+16>>2]=jg(f)|0;c[o+20>>2]=jg(f)|0;a[o+24|0]=ja(f)|0;a[o+25|0]=ja(f)|0;hD(c[o+20>>2]|0,(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+16|0);m=(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+12|0;c[m>>2]=(c[m>>2]|0)+(48-(d[o+24|0]|0));m=(ac(d[o+3|0]|0,d[o+2|0]|0)|0)/64|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)>>2]=m;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+8>>2]=(a[o+25|0]<<1)+128;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+20>>2]=a[n+340|0]|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+24>>2]=a[n+342|0]|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+28>>2]=a[n+343|0]|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+32>>2]=a[n+341|0]|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+40>>2]=v;if((v|0)>=1024){m=f;g=c[o+4>>2]|0;bK(m|0,g|0,1)|0}else{g=(c[h+180>>2]|0)+(v*52|0)|0;m=k+5|0;dh(g,m,32)|0;c[(c[h+180>>2]|0)+(v*52|0)+32>>2]=c[o+4>>2];c[(c[h+180>>2]|0)+(v*52|0)+36>>2]=c[o+12>>2];c[(c[h+180>>2]|0)+(v*52|0)+40>>2]=c[o+16>>2];c[(c[h+180>>2]|0)+(v*52|0)+44>>2]=0;if((b[o>>1]&2|0)!=0){m=(c[h+180>>2]|0)+(v*52|0)+44|0;c[m>>2]=c[m>>2]|1;m=(c[h+180>>2]|0)+(v*52|0)+32|0;c[m>>2]=c[m>>2]>>1;m=(c[h+180>>2]|0)+(v*52|0)+36|0;c[m>>2]=c[m>>2]>>1;m=(c[h+180>>2]|0)+(v*52|0)+40|0;c[m>>2]=c[m>>2]>>1}m=(c[h+180>>2]|0)+(v*52|0)+44|0;c[m>>2]=c[m>>2]|((a[o+8|0]&3|0)!=0?2:0);m=(c[h+180>>2]|0)+(v*52|0)+44|0;c[m>>2]=c[m>>2]|((d[o+8|0]|0)==2?4:0);m=r;g=f;l=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+(y<<6)+40>>2]|0)*52|0)|0;gQ(m,g,1,l,0)|0}y=y+1|0;v=v+1|0}}q=q+1|0}if((w|0)==2070){s=-1;t=s;i=j;return t|0}else if((w|0)==2112){s=-1;t=s;i=j;return t|0}else if((w|0)==2125){c[h+144>>2]=v;c[h+180>>2]=pV(c[h+180>>2]|0,(c[h+144>>2]|0)*52|0)|0;h=r+1280|0;c[h>>2]=c[h>>2]|32;c[r+1284>>2]=1;s=0;t=s;i=j;return t|0}return 0}function gM(c,d,e){c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0;f=c;c=d;bB(c|0,4,1,f|0)|0;if((p6(e|0,c|0,4)|0)!=0){g=-1;h=g;return h|0}a[c+4|0]=ja(f)|0;if((a[c+4|0]|0)!=32){g=-1;h=g;return h|0}bB(c+5|0,32,1,f|0)|0;a[c+37|0]=ja(f)|0;b[c+38>>1]=jc(f)|0;b[c+40>>1]=jc(f)|0;g=0;h=g;return h|0}function gN(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;d=a;a=c;bK(d|0,a+44|0,0)|0;if((jl(d)|0)!=1396920909){e=-1;f=e;return f|0}else{c=d;g=a|0;bK(c|0,g|0,0)|0;di(d,b,28);e=0;f=e;return f|0}return 0}function gO(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0;j=i;i=i+336|0;k=j|0;l=j+80|0;m=j+88|0;n=j+184|0;o=j+264|0;p=j+304|0;q=f;f=g;g=h;h=q|0;r=0;s=0;bK(f|0,g|0,0)|0;bB(m|0,28,1,f|0)|0;ja(f)|0;a[m+29|0]=ja(f)|0;jc(f)|0;b[m+32>>1]=jc(f)|0;b[m+34>>1]=jc(f)|0;b[m+36>>1]=jc(f)|0;b[m+38>>1]=jc(f)|0;b[m+40>>1]=jc(f)|0;b[m+42>>1]=jc(f)|0;c[m+44>>2]=jl(f)|0;a[m+48|0]=ja(f)|0;a[m+49|0]=ja(f)|0;a[m+50|0]=ja(f)|0;a[m+51|0]=ja(f)|0;a[m+52|0]=ja(f)|0;a[m+53|0]=ja(f)|0;jg(f)|0;jg(f)|0;b[m+62>>1]=jc(f)|0;bB(m+64|0,32,1,f|0)|0;do{if((e[m+40>>1]|0)==4865){if((d[m+27|0]|0)!=135){break}s=1}}while(0);if((s|0)!=0){t=0;while(1){if(t>>>0>=28){break}if((d[m+t|0]|0)==135){a[m+t|0]=0}t=t+1|0}t=0;while(1){if(t>>>0>=2){break}if((d[m+36+t|0]|0)==135){a[m+36+t|0]=0}t=t+1|0}t=0;while(1){if(t>>>0>=2){break}if((d[m+38+t|0]|0)==135){a[m+38+t|0]=0}t=t+1|0}}dh(h|0,m|0,28)|0;c[h+156>>2]=e[m+32>>1]|0;c[h+140>>2]=e[m+34>>1]|0;c[h+144>>2]=c[h+140>>2];c[h+128>>2]=e[m+36>>1]|0;t=pT(2,c[h+140>>2]|0)|0;u=pT(2,c[h+128>>2]|0)|0;if((b[m+38>>1]&16|0)!=0){v=q+1280|0;c[v>>2]=c[v>>2]|8192}if((b[m+38>>1]&64|0)!=0){v=q+1280|0;c[v>>2]=c[v>>2]|64}c[h+148>>2]=d[m+49|0]|0;c[h+152>>2]=d[m+50|0]|0;v=0;while(1){if((v|0)>=32){break}if((d[m+64+v|0]|0)!=255){c[h+136>>2]=v+1;if((a[m+51|0]&128|0)!=0){c[h+184+(v*12|0)>>2]=(d[m+64+v|0]&15|0)<8?0:255}else{c[h+184+(v*12|0)>>2]=128}}v=v+1|0}c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;bB(h+952|0,1,c[h+156>>2]|0,f|0)|0;v=0;while(1){if((v|0)>=(c[h+140>>2]|0)){break}b[t+(v<<1)>>1]=jc(f)|0;v=v+1|0}v=0;while(1){if((v|0)>=(c[h+128>>2]|0)){break}b[u+(v<<1)>>1]=jc(f)|0;v=v+1|0}v=0;w=m+53|0;a[w]=(d[w]|0)-252&255;while(1){if((a[m+53|0]|0)!=0){x=0}else{x=(v|0)<32}if(!x){break}w=ja(f)|0;if((w&32|0)!=0){c[h+184+(v*12|0)>>2]=(w&255)<<4&255}else{if(((d[m+51|0]|0)%128|0|0)!=0){y=((v&1)*160|0)+48|0}else{y=128}c[h+184+(v*12|0)>>2]=y}v=v+1|0}c[q+1264>>2]=8363;if((e[m+40>>1]|0)==4864){y=q+1280|0;c[y>>2]=c[y>>2]|64}y=e[m+40>>1]>>12;do{if((y|0)==6){z=2231}else if((y|0)==1){x=b[m+40>>1]&255;aM(o|0,40,29656,(A=i,i=i+16|0,c[A>>2]=(b[m+40>>1]&3840)>>8,c[A+8>>2]=x,A)|0)|0;i=A;x=q+1280|0;c[x>>2]=c[x>>2]|16}else if((y|0)==2){x=b[m+40>>1]&255;aM(o|0,40,27752,(A=i,i=i+16|0,c[A>>2]=(b[m+40>>1]&3840)>>8,c[A+8>>2]=x,A)|0)|0;i=A}else if((y|0)==5){x=b[m+40>>1]&255;aM(o|0,40,23928,(A=i,i=i+16|0,c[A>>2]=(b[m+40>>1]&3840)>>8,c[A+8>>2]=x,A)|0)|0;i=A}else if((y|0)==4){if((e[m+40>>1]|0)!=16640){x=o|0;w=(b[m+40>>1]&3840)>>8;B=b[m+40>>1]&255;aM(x|0,40,23520,(A=i,i=i+16|0,c[A>>2]=w,c[A+8>>2]=B,A)|0)|0;i=A;break}else{z=2231;break}}else if((y|0)==3){if((e[m+40>>1]|0)==12822){B=o|0;qc(B|0,26600)|0}else{if((e[m+40>>1]|0)==12823){B=o|0;qc(B|0,25576)|0}else{B=o|0;w=(b[m+40>>1]&3840)>>8;x=b[m+40>>1]&255;aM(B|0,40,24672,(A=i,i=i+16|0,c[A>>2]=w,c[A+8>>2]=x,A)|0)|0;i=A}}}else{aM(o|0,40,22344,(A=i,i=i+8|0,c[A>>2]=e[m+40>>1]|0,A)|0)|0;i=A}}while(0);if((z|0)==2231){y=o|0;x=(b[m+40>>1]&3840)>>8;w=b[m+40>>1]&255;aM(y|0,40,22952,(A=i,i=i+16|0,c[A>>2]=x,c[A+8>>2]=w,A)|0)|0;i=A}aM(h+64|0,64,31096,(A=i,i=i+8|0,c[A>>2]=o,A)|0)|0;i=A;c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;p7(p|0,0,32);v=0;while(1){if((v|0)>=(c[h+128>>2]|0)){break}A=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(v<<2)>>2]=A;c[c[(c[h+168>>2]|0)+(v<<2)>>2]>>2]=64;A=0;while(1){if((A|0)>=(c[h+136>>2]|0)){break}o=(ac(v,c[h+136>>2]|0)|0)+A|0;c[(c[(c[h+168>>2]|0)+(v<<2)>>2]|0)+4+(A<<2)>>2]=o;w=pT(((c[c[(c[h+168>>2]|0)+(v<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(o<<2)>>2]=w;c[c[(c[h+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(v<<2)>>2]>>2];A=A+1|0}if((b[u+(v<<1)>>1]|0)!=0){A=f;o=g+(e[u+(v<<1)>>1]<<4)|0;bK(A|0,o|0,0)|0;o=0;A=((jc(f)|0)&65535)-2|0;while(1){if((o|0)>=(c[c[(c[h+168>>2]|0)+(v<<2)>>2]>>2]|0)){break}w=ja(f)|0;if((w&255|0)==0){o=o+1|0;continue}x=w&31;if((x|0)>=(c[h+136>>2]|0)){C=l}else{C=(c[(c[q+172>>2]|0)+(c[(c[(c[q+168>>2]|0)+(v<<2)>>2]|0)+4+(x<<2)>>2]<<2)>>2]|0)+4+(o<<3)|0}r=C;if((w&32|0)!=0){y=ja(f)|0;B=y;D=y&255;if((D|0)==255){B=0}else if((D|0)==254){B=-127}else{B=(((B&240)>>4)*12|0)+13+(B&15)&255}a[r|0]=B;a[r+1|0]=ja(f)|0;A=A-2|0}if((w&64|0)!=0){a[r+2|0]=((ja(f)|0)&255)+1&255;A=A-1|0}if((w&128|0)!=0){a[r+3|0]=ja(f)|0;a[r+4|0]=ja(f)|0;gP(x,r,p|0);A=A-2|0}}}v=v+1|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}v=0;while(1){if((v|0)>=(c[h+140>>2]|0)){z=2340;break}p=pT(64,1)|0;c[(c[h+176>>2]|0)+(v*764|0)+756>>2]=p;bK(f|0,g+(e[t+(v<<1)>>1]<<4)|0,0)|0;p=ja(f)|0;c[(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+40>>2]=v;if((p&255|0)>=2){r=k|0;C=f;bB(r|0,12,1,C|0)|0;C=k+12|0;r=f;bB(C|0,3,1,r|0)|0;r=k+15|0;C=f;bB(r|0,12,1,C|0)|0;a[k+27|0]=ja(f)|0;a[k+28|0]=ja(f)|0;C=f;jc(C)|0;b[k+32>>1]=jc(f)|0;C=f;jc(C)|0;C=k+36|0;r=f;bB(C|0,12,1,r|0)|0;r=k+48|0;C=f;bB(r|0,28,1,C|0)|0;c[k+76>>2]=jl(f)|0;if((c[k+76>>2]|0)!=1396920905){z=2289;break}c[k+76>>2]=0;C=(c[h+176>>2]|0)+(v*764|0)|0;r=k+48|0;dh(C,r,28)|0;c[(c[h+176>>2]|0)+(v*764|0)+36>>2]=1;c[c[(c[h+176>>2]|0)+(v*764|0)+756>>2]>>2]=d[k+27|0]|0;hD(e[k+32>>1]|0,(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+16|0);r=(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+12|0;c[r>>2]=(c[r>>2]|0)+12;r=q;C=f;l=(c[h+180>>2]|0)+(v*52|0)|0;A=k+15|0;gQ(r,C,4096,l,A)|0}else{A=n|0;l=f;bB(A|0,13,1,l|0)|0;b[n+14>>1]=jc(f)|0;c[n+16>>2]=jg(f)|0;c[n+20>>2]=jg(f)|0;c[n+24>>2]=jg(f)|0;a[n+28|0]=ja(f)|0;a[n+29|0]=ja(f)|0;a[n+30|0]=ja(f)|0;a[n+31|0]=ja(f)|0;b[n+32>>1]=jc(f)|0;b[n+34>>1]=jc(f)|0;l=n+36|0;A=f;bB(l|0,4,1,A|0)|0;b[n+40>>1]=jc(f)|0;b[n+42>>1]=jc(f)|0;c[n+44>>2]=jg(f)|0;A=n+48|0;l=f;bB(A|0,28,1,l|0)|0;c[n+76>>2]=jl(f)|0;if((p&255|0)==1){if((c[n+76>>2]|0)!=1396920915){z=2297;break}}if((s|0)!=0){p=0;while(1){if(p>>>0>=4){break}if((d[n+16+p|0]|0)==135){a[n+16+p|0]=0}p=p+1|0}p=0;while(1){if(p>>>0>=4){break}if((d[n+20+p|0]|0)==135){a[n+20+p|0]=0}p=p+1|0}p=0;while(1){if(p>>>0>=4){break}if((d[n+24+p|0]|0)==135){a[n+24+p|0]=0}p=p+1|0}p=0;while(1){if(p>>>0>=1){break}if((d[n+31+p|0]|0)==135){a[n+31+p|0]=0}p=p+1|0}}c[(c[h+180>>2]|0)+(v*52|0)+32>>2]=c[n+16>>2];c[(c[h+176>>2]|0)+(v*764|0)+36>>2]=(c[n+16>>2]|0)>>>0>0?1:0;c[(c[h+180>>2]|0)+(v*52|0)+36>>2]=c[n+20>>2];c[(c[h+180>>2]|0)+(v*52|0)+40>>2]=c[n+24>>2];c[(c[h+180>>2]|0)+(v*52|0)+44>>2]=(a[n+31|0]&1|0)!=0?2:0;if((a[n+31|0]&4|0)!=0){p=(c[h+180>>2]|0)+(v*52|0)+44|0;c[p>>2]=c[p>>2]|1;p=(c[h+180>>2]|0)+(v*52|0)+32|0;c[p>>2]=c[p>>2]>>1;p=(c[h+180>>2]|0)+(v*52|0)+36|0;c[p>>2]=c[p>>2]>>1;p=(c[h+180>>2]|0)+(v*52|0)+40|0;c[p>>2]=c[p>>2]>>1}c[c[(c[h+176>>2]|0)+(v*764|0)+756>>2]>>2]=d[n+28|0]|0;c[n+76>>2]=0;p=(c[h+176>>2]|0)+(v*764|0)|0;l=n+48|0;dh(p,l,28)|0;hD(e[n+32>>1]|0,(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(v*764|0)+756>>2]|0)+16|0);l=f;p=g+(e[n+14>>1]<<4)|0;bK(l|0,p|0,0)|0;p=q;l=f;A=(e[m+42>>1]|0)-1<<1;C=(c[h+180>>2]|0)+(v*52|0)|0;gQ(p,l,A,C,0)|0}v=v+1|0}if((z|0)!=2289)if((z|0)!=2297)if((z|0)==2340){pU(u);pU(t);c[q+6560>>2]=2936;z=q+1280|0;c[z>>2]=c[z>>2]|545;c[q+1284>>2]=2;E=0;F=E;i=j;return F|0}pU(u);pU(t);E=-1;F=E;i=j;return F|0}function gP(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0;f=b;b=c;c=e;e=(a[b+4|0]&240)>>4&255;g=d[b+4|0]&15;h=a[8576+(d[b+3|0]|0)|0]|0;a[b+3|0]=h;i=h&255;if((i|0)==255){a[b+4|0]=0;a[b+3|0]=0;return}else if((i|0)==254){a[b+3|0]=14;h=e&255;if((h|0)==8){a[b+3|0]=22;a[b+4|0]=(g&255)<<4&255}else if((h|0)==1){a[b+4|0]=(a[b+4|0]&15|48)&255}else if((h|0)==3){a[b+4|0]=(a[b+4|0]&15|64)&255}else if((h|0)==11){a[b+4|0]=(a[b+4|0]&15|96)&255}else if((h|0)==2){a[b+4|0]=(a[b+4|0]&15|80)&255}else if((h|0)==4){a[b+4|0]=(a[b+4|0]&15|112)&255}else if((h|0)==5|(h|0)==6|(h|0)==7|(h|0)==9|(h|0)==10){a[b+4|0]=0;a[b+3|0]=0}else if((h|0)==12){if(g<<24>>24==0){a[b+4|0]=0;a[b+3|0]=0}}return}else if((i|0)==0){if((a[b+4|0]|0)!=0){a[c+f|0]=a[b+4|0]|0}else{a[b+4|0]=a[c+f|0]|0}return}else{return}}function gQ(b,d,e,f,g){b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0;h=i;i=i+24|0;j=h|0;k=h+8|0;l=b;b=d;d=e;e=f;f=g;if((d&20480|0)!=0){g=11;if((d&16384|0)!=0){g=392}else{if((d&8192|0)!=0){gR(f)}}m=pS(g+4|0)|0;c[e+48>>2]=m;if((m|0)==0){n=-1;o=n;i=h;return o|0}else{c[c[e+48>>2]>>2]=0;m=e+48|0;c[m>>2]=(c[m>>2]|0)+4;m=c[e+48>>2]|0;p=f;r=g;p3(m|0,p|0,r)|0;r=e+44|0;c[r>>2]=c[r>>2]|32768;c[e+32>>2]=g;n=0;o=n;i=h;return o|0}}if((c[e+32>>2]|0)==0){n=0;o=n;i=h;return o|0}if((c[l+1288>>2]&1|0)!=0){if((~d&16|0)!=0){l=b;g=c[e+32>>2]|0;bK(l|0,g|0,1)|0}n=0;o=n;i=h;return o|0}if((c[e+40>>2]|0)>(c[e+32>>2]|0)){c[e+40>>2]=c[e+32>>2]}if((c[e+36>>2]|0)>=(c[e+32>>2]|0)){s=2387}else{if((c[e+36>>2]|0)>=(c[e+40>>2]|0)){s=2387}}if((s|0)==2387){c[e+40>>2]=0;c[e+36>>2]=0;g=e+44|0;c[g>>2]=c[g>>2]&-7}g=c[e+32>>2]|0;l=4;r=0;if((c[e+44>>2]&4|0)!=0){if((~c[e+44>>2]&2|0)!=0){p=e+44|0;c[p>>2]=c[p>>2]&-5}}if((c[e+44>>2]&4|0)!=0){r=(c[e+40>>2]|0)-(c[e+36>>2]|0)-((c[e+32>>2]|0)-(c[e+40>>2]|0))|0;if((r|0)<0){r=0}}if((c[e+44>>2]&1|0)!=0){g=g<<1;l=l<<1;r=r<<1}p=pS(g+l+r+4|0)|0;c[e+48>>2]=p;if((p|0)==0){n=-1;o=n;i=h;return o|0}c[c[e+48>>2]>>2]=0;p=e+48|0;c[p>>2]=(c[p>>2]|0)+4;if((d&16|0)!=0){p=c[e+48>>2]|0;l=f;f=g;p3(p|0,l|0,f)|0}else{f=aT(b|0)|0;l=bB(j|0,1,5,b|0)|0;p=b;m=f;bK(p|0,m|0,0)|0;do{if((l|0)==5){if((p6(j|0,24832,5)|0)!=0){s=2405;break}m=g>>1;p=b;bK(p|0,5,1)|0;p=k|0;f=b;bB(p|0,1,16,f|0)|0;f=(c[e+48>>2]|0)+m|0;p=m;t=b;bB(f|0,1,p|0,t|0)|0;gS((c[e+48>>2]|0)+m|0,c[e+48>>2]|0,k|0,g)}else{s=2405}}while(0);if((s|0)==2405){s=bB(c[e+48>>2]|0,1,g|0,b|0)|0;if((s|0)!=(g|0)){b=c[q>>2]|0;k=s-g|0;bE(b|0,29608,(b=i,i=i+8|0,c[b>>2]=k,b)|0)|0;i=b;p7((c[e+48>>2]|0)+s|0,0,g-s|0)}}}if((d&8|0)!=0){gT(c[e+48>>2]|0,c[e+32>>2]|0)}if((c[e+44>>2]&1|0)!=0){if((d&64|0)!=0){gU(c[e+48>>2]|0,c[e+32>>2]|0)}}if((d&1|0)!=0){gV(c[e+48>>2]|0,c[e+32>>2]|0,c[e+44>>2]&1)}else{if((d&4|0)!=0){s=c[e+32>>2]|0;if((c[e+44>>2]&1|0)!=0){s=s<<1}gV(c[e+48>>2]|0,s,0)}}if((d&2|0)!=0){gW(c[e+48>>2]|0,c[e+32>>2]|0,c[e+44>>2]&1)}if((d&256|0)!=0){gX(c[e+48>>2]|0,c[e+32>>2]|0,c[e+44>>2]&1);s=e+32|0;c[s>>2]=(c[s>>2]|0)/2|0}if((d&128|0)!=0){gY(c[e+48>>2]|0,c[e+32>>2]|0)}if((d&512|0)!=0){do{if((c[e+36>>2]|0)==0){if((c[e+32>>2]|0)<=(c[e+40>>2]|0)){break}d=e+44|0;c[d>>2]=c[d>>2]|16}}while(0)}if((c[e+44>>2]&4|0)!=0){gZ(e);g=g+r|0}if((c[e+44>>2]&1|0)!=0){u=0;while(1){if((u|0)>=8){break}a[(c[e+48>>2]|0)+(g+u)|0]=a[(c[e+48>>2]|0)+(g-2+u)|0]|0;u=u+1|0}}else{u=0;while(1){if((u|0)>=4){break}a[(c[e+48>>2]|0)+(g+u)|0]=a[(c[e+48>>2]|0)+(g-1+u)|0]|0;u=u+1|0}}if((c[e+44>>2]&1|0)!=0){a[(c[e+48>>2]|0)-2|0]=a[c[e+48>>2]|0]|0;a[(c[e+48>>2]|0)-1|0]=a[(c[e+48>>2]|0)+1|0]|0}else{a[(c[e+48>>2]|0)-1|0]=a[c[e+48>>2]|0]|0}if((c[e+44>>2]&2|0)!=0){if((c[e+44>>2]&1|0)!=0){g=c[e+40>>2]<<1;d=c[e+36>>2]<<1;if((c[e+44>>2]&4|0)!=0){g=g+((c[e+40>>2]|0)-(c[e+36>>2]|0)<<1)|0}a[(c[e+48>>2]|0)+g|0]=a[(c[e+48>>2]|0)+(g-2)|0]|0;a[(c[e+48>>2]|0)+(g+1)|0]=a[(c[e+48>>2]|0)+(g-1)|0]|0;u=0;while(1){if((u|0)>=6){break}a[(c[e+48>>2]|0)+(g+2+u)|0]=a[(c[e+48>>2]|0)+(d+u)|0]|0;u=u+1|0}}else{d=(c[e+40>>2]|0)+r|0;r=c[e+36>>2]|0;a[(c[e+48>>2]|0)+d|0]=a[(c[e+48>>2]|0)+(d-1)|0]|0;u=0;while(1){if((u|0)>=3){break}a[(c[e+48>>2]|0)+(d+1+u)|0]=a[(c[e+48>>2]|0)+(r+u)|0]|0;u=u+1|0}}}n=0;o=n;i=h;return o|0}function gR(b){b=b|0;var c=0,d=0,e=0,f=0;c=i;i=i+16|0;d=c|0;e=b;b=0;while(1){if((b|0)>=10){break}f=a[e+b|0]|0;a[e+b|0]=a[e+(b+1)|0]|0;a[e+(b+1)|0]=f;b=b+2|0}b=d;f=e;p3(b|0,f|0,11)|0;a[e+8|0]=a[d+10|0]|0;a[e+10|0]=a[d+9|0]|0;a[e+9|0]=a[d+8|0]|0;i=c;return}function gS(b,c,e,f){b=b|0;c=c|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0;g=b;b=c;c=e;e=f;f=0;e=(e+1|0)/2|0;h=0;while(1){if((h|0)>=(e|0)){break}i=a[g]|0;j=g;g=j+1|0;k=d[j]>>4&255;f=(f<<24>>24)+(a[c+(i&15)|0]|0)&255;i=b;b=i+1|0;a[i]=f;f=(f<<24>>24)+(a[c+(k&15)|0]|0)&255;k=b;b=k+1|0;a[k]=f;h=h+1|0}return}function gT(b,c){b=b|0;c=c|0;var e=0;e=b;b=c;while(1){c=b;b=c-1|0;if((c|0)==0){break}c=e;a[c]=(d[c]|0)<<1&255;e=e+1|0}return}function gU(b,c){b=b|0;c=c|0;var d=0,e=0;d=b;b=c;c=0;while(1){if((c|0)>=(b|0)){break}e=a[d|0]|0;a[d|0]=a[d+1|0]|0;a[d+1|0]=e;d=d+2|0;c=c+1|0}return}function gV(c,f,g){c=c|0;f=f|0;g=g|0;var h=0,i=0;h=c;c=f;f=h;i=0;if((g|0)!=0){while(1){g=c;c=g-1|0;if((g|0)==0){break}i=(e[f>>1]|0)+(i&65535)&65535;g=f;f=g+2|0;b[g>>1]=i}return}else{while(1){f=c;c=f-1|0;if((f|0)==0){break}i=(d[h]|0)+(i&65535)&65535;f=h;h=f+1|0;a[f]=i&255}return}}function gW(c,f,g){c=c|0;f=f|0;g=g|0;var h=0;h=c;c=f;f=h;if((g|0)!=0){while(1){g=c;c=g-1|0;if((g|0)==0){break}g=f;b[g>>1]=(e[g>>1]|0)+32768&65535;f=f+2|0}return}else{while(1){f=c;c=f-1|0;if((f|0)==0){break}f=h;a[f]=(d[f]|0)-128&255;h=h+1|0}return}}function gX(c,e,f){c=c|0;e=e|0;f=f|0;var g=0,h=0;g=c;c=e;e=g;if((f|0)!=0){c=(c|0)/2|0;h=0;while(1){if((h|0)>=(c|0)){break}b[e+(h<<1)>>1]=(((b[e+(h<<1<<1)>>1]|0)+(b[e+((h<<1)+1<<1)>>1]|0)|0)/2|0)&65535;h=h+1|0}return}else{h=0;while(1){if((h|0)>=(c|0)){break}a[g+h|0]=(((d[g+(h<<1)|0]|0)+(d[g+((h<<1)+1)|0]|0)|0)/2|0)&255;h=h+1|0}return}}function gY(b,c){b=b|0;c=c|0;var e=0,f=0;e=b;b=c;c=0;while(1){if((c|0)>=(b|0)){break}f=a[e+c|0]|0;a[e+c|0]=a[1192+((f&255)>>1)|0]|0;if((f&1|0)!=0){f=e+c|0;a[f]=((d[f]|0)*-1|0)&255}c=c+1|0}return}function gZ(d){d=d|0;var e=0,f=0,g=0,h=0,i=0;e=d;d=c[e+48>>2]|0;f=c[e+48>>2]|0;if((c[e+32>>2]|0)>(c[e+40>>2]|0)){g=c[e+40>>2]|0}else{g=c[e+32>>2]|0}h=(c[e+40>>2]|0)-(c[e+36>>2]|0)|0;if((c[e+44>>2]&1|0)!=0){d=d+(g<<1)|0;i=0;while(1){if((i|0)>=(h|0)){break}b[d+(i<<1)>>1]=b[d+(-i<<1)-2>>1]|0;i=i+1|0}return}else{f=f+g|0;i=0;while(1){if((i|0)>=(h|0)){break}a[f+i|0]=a[f+(-i|0)-1|0]|0;i=i+1|0}return}}function g_(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=a;a=b;bK(c|0,60,1)|0;b=jl(c)|0;bK(c|0,60,1)|0;d=jl(c)|0;do{if((b|0)!=1397706311){if((d|0)==1397706311){break}e=-1;f=e;return f|0}}while(0);di(c,a,0);e=0;f=e;return f|0}function g$(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=a;a=b;b=c;if((g0(d,a,15,b)|0)<0){e=g0(d,a,31,b)|0;f=e;return f|0}else{e=0;f=e;return f|0}return 0}function g0(f,g,h,j){f=f|0;g=g|0;h=h|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;k=i;i=i+1232|0;l=k|0;m=k+24|0;n=k+160|0;o=k+168|0;p=k+296|0;q=f;f=g;g=h;h=q|0;bK(f|0,j|0,0)|0;j=0;while(1){if((j|0)>=(g|0)){break}c[o+(j<<2)>>2]=jl(f)|0;j=j+1|0}c[l>>2]=jl(f)|0;b[l+4>>1]=jd(f)|0;bB(l+6|0,14,1,f|0)|0;if((c[l>>2]|0)!=1397706311){r=-1;s=r;i=k;return s|0}c[h+140>>2]=g;c[h+144>>2]=c[h+140>>2];c[h+152>>2]=1776930/(e[l+4>>1]|0|0)|0;j=0;while(1){if((j|0)>=(c[h+140>>2]|0)){break}bB(p+(j*30|0)|0,22,1,f|0)|0;b[p+(j*30|0)+22>>1]=jd(f)|0;a[p+(j*30|0)+24|0]=ja(f)|0;a[p+(j*30|0)+25|0]=ja(f)|0;b[p+(j*30|0)+26>>1]=jd(f)|0;b[p+(j*30|0)+28>>1]=jd(f)|0;j=j+1|0}a[m|0]=ja(f)|0;a[m+1|0]=ja(f)|0;bB(m+2|0,128,1,f|0)|0;c[h+156>>2]=d[m|0]|0;if((c[h+156>>2]|0)>127){r=-1;s=r;i=k;return s|0}l=h+952|0;g=m+2|0;m=c[h+156>>2]|0;p3(l|0,g|0,m)|0;j=0;c[h+128>>2]=0;while(1){if((j|0)>=(c[h+156>>2]|0)){break}if((d[h+952+j|0]|0|0)>(c[h+128>>2]|0)){c[h+128>>2]=d[h+952+j|0]|0}j=j+1|0}m=h+128|0;c[m>>2]=(c[m>>2]|0)+1;c[h+132>>2]=ac(c[h+136>>2]|0,c[h+128>>2]|0)|0;if((c[h+140>>2]|0)==15){dm(q,29584,(t=i,i=i+1|0,i=i+7>>3<<3,c[t>>2]=0,t)|0);i=t}else{dm(q,27736,(t=i,i=i+1|0,i=i+7>>3<<3,c[t>>2]=0,t)|0);i=t}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}j=0;while(1){if((j|0)>=(c[h+140>>2]|0)){break}t=pT(64,1)|0;c[(c[h+176>>2]|0)+(j*764|0)+756>>2]=t;t=c[o+(j<<2)>>2]|0;c[(c[h+180>>2]|0)+(j*52|0)+32>>2]=t;c[(c[h+176>>2]|0)+(j*764|0)+36>>2]=((t|0)!=0^1^1)&1;c[(c[h+180>>2]|0)+(j*52|0)+36>>2]=e[p+(j*30|0)+26>>1]|0;c[(c[h+180>>2]|0)+(j*52|0)+40>>2]=(c[(c[h+180>>2]|0)+(j*52|0)+36>>2]|0)+((e[p+(j*30|0)+28>>1]|0)<<1);c[(c[h+180>>2]|0)+(j*52|0)+44>>2]=(e[p+(j*30|0)+28>>1]|0|0)>1?2:0;c[c[(c[h+176>>2]|0)+(j*764|0)+756>>2]>>2]=d[p+(j*30|0)+25|0]|0;c[(c[(c[h+176>>2]|0)+(j*764|0)+756>>2]|0)+16>>2]=((d[p+(j*30|0)+24|0]|0)<<4&255)<<24>>24;c[(c[(c[h+176>>2]|0)+(j*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(j*764|0)+756>>2]|0)+40>>2]=j;dh((c[h+176>>2]|0)+(j*764|0)|0,p+(j*30|0)|0,22)|0;j=j+1|0}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;j=0;while(1){if((j|0)>=(c[h+128>>2]|0)){break}p=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(j<<2)>>2]=p;c[c[(c[h+168>>2]|0)+(j<<2)>>2]>>2]=64;p=0;while(1){if((p|0)>=(c[h+136>>2]|0)){break}o=(ac(j,c[h+136>>2]|0)|0)+p|0;c[(c[(c[h+168>>2]|0)+(j<<2)>>2]|0)+4+(p<<2)>>2]=o;t=pT(((c[c[(c[h+168>>2]|0)+(j<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(o<<2)>>2]=t;c[c[(c[h+172>>2]|0)+(o<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(j<<2)>>2]>>2];p=p+1|0}p=0;while(1){if((p|0)>=(c[h+136>>2]<<6|0)){break}o=(c[(c[q+172>>2]|0)+(c[(c[(c[q+168>>2]|0)+(j<<2)>>2]|0)+4+(((p|0)%(c[h+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((p|0)/(c[h+136>>2]|0)|0)<<3)|0;bB(n|0,1,4,f|0)|0;a[o|0]=(ht((a[n|0]&15)<<8|(d[n+1|0]|0))|0)&255;a[o+1|0]=((a[n|0]&240)>>4<<4|(a[n+2|0]&240)>>4)&255;a[o+4|0]=a[n+3|0]|0;t=a[n+2|0]&15;if((t|0)==1){a[o+3|0]=0}else if((t|0)==2){if(((d[o+4|0]|0)>>4|0)!=0){a[o+3|0]=2;m=o+4|0;a[m]=(d[m]|0)>>4&255}else{if((a[o+4|0]&15|0)!=0){a[o+3|0]=1;m=o+4|0;a[m]=(d[m]|0)&15}}}else if((t|0)==5){a[o+3|0]=-95}else if((t|0)==6){a[o+3|0]=12;a[o+4|0]=64-(d[n+3|0]|0)&255}else if((t|0)==3|(t|0)==4|(t|0)==7|(t|0)==8){u=2635}else{u=2635}if((u|0)==2635){u=0;a[o+4|0]=0;a[o+3|0]=0}p=p+1|0}j=j+1|0}u=q+1280|0;c[u>>2]=c[u>>2]|8192;j=0;while(1){if((j|0)>=(c[h+140>>2]|0)){break}if((c[(c[h+180>>2]|0)+(j*52|0)+32>>2]|0)>2){u=q;n=f;p=(c[h+180>>2]|0)+(j*52|0)|0;gQ(u,n,0,p,0)|0}j=j+1|0}r=0;s=r;i=k;return s|0}function g1(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;c=i;i=i+8|0;d=c|0;e=a;a=b;if((bB(d|0,1,6,e|0)|0)>>>0<6){f=-1;g=f;i=c;return g|0}do{if((p6(d|0,29560,6)|0)!=0){if((p6(d|0,26592,6)|0)==0){break}f=-1;g=f;i=c;return g|0}}while(0);di(e,a,0);f=0;g=f;i=c;return g|0}function g2(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;g=i;i=i+1056|0;h=g|0;j=g+32|0;k=b;b=e;e=f;f=k|0;bK(b|0,e|0,0)|0;bB(h|0,6,1,b|0)|0;do{if((p6(h|0,29560,6)|0)!=0){if((p6(h|0,26592,8)|0)!=0){l=-1;m=l;i=g;return m|0}else{dm(k,25528,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n;break}}else{dm(k,27720,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0);i=n}}while(0);n=(jc(b)|0)&65535;c[f+148>>2]=(jc(b)|0)&65535;bK(b|0,10,1)|0;c[f+144>>2]=15;c[f+140>>2]=15;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}o=0;while(1){if((o|0)>=(c[f+140>>2]|0)){break}p=pT(64,1)|0;c[(c[f+176>>2]|0)+(o*764|0)+756>>2]=p;bB(h|0,1,22,b|0)|0;if((a[h|0]|0)!=0){a[h+((d[h|0]|0)+1)|0]=0;p=(c[f+176>>2]|0)+(o*764|0)|0;q=h+1|0;dh(p,q,22)|0}jc(b)|0;q=((ja(b)|0)&255)>>2;c[c[(c[f+176>>2]|0)+(o*764|0)+756>>2]>>2]=q;c[(c[(c[f+176>>2]|0)+(o*764|0)+756>>2]|0)+8>>2]=128;bK(b|0,5,1)|0;o=o+1|0}c[f+156>>2]=(ja(b)|0)&127;ja(b)|0;bB(f+952|0,1,128,b|0)|0;bK(b|0,e+600|0,0)|0;c[f+136>>2]=14;c[f+128>>2]=(n|0)/896|0;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;o=0;while(1){if((o|0)>=(c[f+128>>2]|0)){break}n=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(o<<2)>>2]=n;c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2]=64;n=0;while(1){if((n|0)>=(c[f+136>>2]|0)){break}e=(ac(o,c[f+136>>2]|0)|0)+n|0;c[(c[(c[f+168>>2]|0)+(o<<2)>>2]|0)+4+(n<<2)>>2]=e;h=pT(((c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(e<<2)>>2]=h;c[c[(c[f+172>>2]|0)+(e<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2];n=n+1|0}r=0;while(1){if((r|0)>=(c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2]|0)){break}s=0;while(1){if((s|0)>=(c[f+136>>2]|0)){break}t=(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(o<<2)>>2]|0)+4+(s<<2)>>2]<<2)>>2]|0)+4+(r<<3)|0;a[t|0]=ja(b)|0;if((a[t|0]|0)!=0){n=t|0;a[n]=(d[n]|0)+24&255}s=s+1|0}r=r+1|0}o=o+1|0}o=0;while(1){if((o|0)>=(c[f+128>>2]|0)){break}r=0;while(1){if((r|0)>=(c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2]|0)){break}s=0;while(1){if((s|0)>=(c[f+136>>2]|0)){break}t=(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(o<<2)>>2]|0)+4+(s<<2)>>2]<<2)>>2]|0)+4+(r<<3)|0;n=ja(b)|0;a[t+1|0]=(n&255)>>4&255;e=n&15;if((e|0)==0){a[t+3|0]=0}else if((e|0)==6){a[t+3|0]=-96}else if((e|0)==15){a[t+3|0]=15}else if((e|0)==3){a[t+3|0]=12}else if((e|0)==5){a[t+3|0]=-95}s=s+1|0}r=r+1|0}o=o+1|0}o=0;while(1){if((o|0)>=(c[f+128>>2]|0)){break}r=0;while(1){if((r|0)>=(c[c[(c[f+168>>2]|0)+(o<<2)>>2]>>2]|0)){break}s=0;while(1){if((s|0)>=(c[f+136>>2]|0)){break}t=(c[(c[k+172>>2]|0)+(c[(c[(c[k+168>>2]|0)+(o<<2)>>2]|0)+4+(s<<2)>>2]<<2)>>2]|0)+4+(r<<3)|0;a[t+4|0]=ja(b)|0;e=d[t+3|0]|0;if((e|0)==12|(e|0)==161|(e|0)==160){e=t+4|0;a[e]=d[e]>>2&255}s=s+1|0}r=r+1|0}o=o+1|0}o=0;while(1){if((o|0)>=(c[f+140>>2]|0)){break}if((a[(c[f+176>>2]|0)+(o*764|0)|0]|0)!=0){r=j|0;s=c[k+1208>>2]|0;p5(r|0,s|0,255)|0;if((a[j|0]|0)!=0){s=j|0;bI(s|0,24656,255)|0}s=j|0;r=(c[f+176>>2]|0)+(o*764|0)|0;bI(s|0,r|0,255)|0;r=bh(j|0,23912)|0;s=r;if((r|0)!=0){r=k;t=s;b=o;nC(r,t,b)|0;b=s;au(b|0)|0}}o=o+1|0}l=0;m=l;i=g;return m|0}
function g3(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;j=i;i=i+680|0;k=j|0;l=j+600|0;m=j+608|0;n=f;f=g;g=h;aC(aE(n|0)|0,m|0)|0;if((c[m+28>>2]|0)<600){o=-1;p=o;i=j;return p|0}h=0;bK(n|0,g|0,0)|0;bB(k|0,1,20,n|0)|0;if((df(k|0,20)|0)<0){o=-1;p=o;i=j;return p|0}q=0;while(1){if((q|0)>=15){break}bB(k+20+(q*30|0)|0,1,22,n|0)|0;b[k+20+(q*30|0)+22>>1]=jd(n)|0;a[k+20+(q*30|0)+24|0]=ja(n)|0;a[k+20+(q*30|0)+25|0]=ja(n)|0;b[k+20+(q*30|0)+26>>1]=jd(n)|0;b[k+20+(q*30|0)+28>>1]=jd(n)|0;q=q+1|0}a[k+470|0]=ja(n)|0;a[k+471|0]=ja(n)|0;bB(k+472|0,1,128,n|0)|0;q=0;r=0;while(1){if((q|0)>=128){break}if((d[k+472+q|0]|0)>127){s=2774;break}if((d[k+472+q|0]|0)>(r|0)){r=d[k+472+q|0]|0}q=q+1|0}if((s|0)==2774){o=-1;p=o;i=j;return p|0}r=r+1|0;do{if((r|0)<=127){if((d[k+470|0]|0)==0){break}if((d[k+470|0]|0)>127){break}q=0;while(1){if((q|0)>=15){s=2807;break}if((df(k+20+(q*30|0)|0,22)|0)<0){s=2786;break}if((a[k+20+(q*30|0)+25|0]|0)>64){s=2788;break}if((a[k+20+(q*30|0)+24|0]|0)>15){s=2790;break}if((e[k+20+(q*30|0)+22>>1]|0)>32768){s=2792;break}if((e[k+20+(q*30|0)+26>>1]>>1|0)>32768){s=2794;break}if((e[k+20+(q*30|0)+28>>1]|0)>32768){s=2796;break}if((e[k+20+(q*30|0)+26>>1]>>1|0)>(e[k+20+(q*30|0)+22>>1]|0)){s=2798;break}if((e[k+20+(q*30|0)+22>>1]|0)!=0){if((e[k+20+(q*30|0)+26>>1]>>1|0)==(e[k+20+(q*30|0)+22>>1]|0)){s=2801;break}}if((e[k+20+(q*30|0)+22>>1]|0)==0){if((e[k+20+(q*30|0)+26>>1]|0)>0){s=2804;break}}h=h+(e[k+20+(q*30|0)+22>>1]<<1)|0;q=q+1|0}if((s|0)==2786){o=-1;p=o;i=j;return p|0}else if((s|0)==2788){o=-1;p=o;i=j;return p|0}else if((s|0)==2790){o=-1;p=o;i=j;return p|0}else if((s|0)==2792){o=-1;p=o;i=j;return p|0}else if((s|0)==2794){o=-1;p=o;i=j;return p|0}else if((s|0)==2796){o=-1;p=o;i=j;return p|0}else if((s|0)==2798){o=-1;p=o;i=j;return p|0}else if((s|0)==2801){o=-1;p=o;i=j;return p|0}else if((s|0)==2804){o=-1;p=o;i=j;return p|0}else if((s|0)==2807){if((h|0)<8){o=-1;p=o;i=j;return p|0}if((c[m+28>>2]|0)<((r<<10)+600+h|0)){o=-1;p=o;i=j;return p|0}q=0;L3473:while(1){if((q|0)>=(r|0)){s=2833;break}t=0;while(1){if((t|0)>=256){break}bB(l|0,1,4,n|0)|0;if(((a[l|0]&240)>>4|0)!=0){s=2816;break L3473}u=((a[l|0]&15)<<8)+(d[l+1|0]|0)|0;do{if((u|0)!=0){if((u|0)==162){break}v=0;while(1){if((c[5984+(v<<2)>>2]|0)<0){break}if((u|0)==(c[5984+(v<<2)>>2]|0)){s=2824;break}v=v+1|0}if((s|0)==2824){s=0}if((c[5984+(v<<2)>>2]|0)<0){s=2828;break L3473}}}while(0);t=t+1|0}q=q+1|0}if((s|0)==2816){o=-1;p=o;i=j;return p|0}else if((s|0)==2828){o=-1;p=o;i=j;return p|0}else if((s|0)==2833){bK(n|0,g|0,0)|0;di(n,f,20);o=0;p=o;i=j;return p|0}}}}while(0);o=-1;p=o;i=j;return p|0}function g4(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;j=i;i=i+616|0;k=j|0;l=j+8|0;m=j+608|0;n=f;f=g;g=h;h=n|0;o=1;p=0;q=0;bK(f|0,g|0,0)|0;c[h+140>>2]=15;c[h+144>>2]=c[h+140>>2];q=0;bB(l|0,1,20,f|0)|0;r=0;while(1){if((r|0)>=15){break}bB(l+20+(r*30|0)|0,1,22,f|0)|0;b[l+20+(r*30|0)+22>>1]=jd(f)|0;a[l+20+(r*30|0)+24|0]=ja(f)|0;a[l+20+(r*30|0)+25|0]=ja(f)|0;b[l+20+(r*30|0)+26>>1]=jd(f)|0;b[l+20+(r*30|0)+28>>1]=jd(f)|0;r=r+1|0}a[l+470|0]=ja(f)|0;a[l+471|0]=ja(f)|0;bB(l+472|0,1,128,f|0)|0;c[h+156>>2]=d[l+470|0]|0;c[h+160>>2]=d[l+471|0]|0;if((c[h+160>>2]|0)<64){o=0}s=h+952|0;t=l+472|0;p3(s|0,t|0,128)|0;r=0;while(1){if((r|0)>=128){break}if((d[h+952+r|0]|0)>(c[h+128>>2]|0)){c[h+128>>2]=d[h+952+r|0]|0}r=r+1|0}t=h+128|0;c[t>>2]=(c[t>>2]|0)+1;r=0;while(1){if((r|0)>=(c[h+140>>2]|0)){break}if((a[l+20+(r*30|0)+24|0]|0)!=0){o=0}do{if((e[l+20+(r*30|0)+22>>1]|0)==0){if((e[l+20+(r*30|0)+28>>1]|0)!=1){break}p=1}}while(0);do{if((e[l+20+(r*30|0)+22>>1]|0)>4999){u=2877}else{if((e[l+20+(r*30|0)+26>>1]|0)>9999){u=2877;break}if((e[l+20+(r*30|0)+28>>1]|0)>4999){u=2877}}}while(0);if((u|0)==2877){u=0;o=0}q=q+(e[l+20+(r*30|0)+22>>1]<<1)|0;r=r+1|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}r=0;while(1){if((r|0)>=(c[h+140>>2]|0)){break}q=pT(64,1)|0;c[(c[h+176>>2]|0)+(r*764|0)+756>>2]=q;c[(c[h+180>>2]|0)+(r*52|0)+32>>2]=e[l+20+(r*30|0)+22>>1]<<1;c[(c[h+180>>2]|0)+(r*52|0)+36>>2]=e[l+20+(r*30|0)+26>>1]|0;c[(c[h+180>>2]|0)+(r*52|0)+40>>2]=(c[(c[h+180>>2]|0)+(r*52|0)+36>>2]|0)+(e[l+20+(r*30|0)+28>>1]<<1);c[(c[h+180>>2]|0)+(r*52|0)+44>>2]=(e[l+20+(r*30|0)+28>>1]|0)>1?2:0;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+16>>2]=(a[l+20+(r*30|0)+24|0]<<4&255)<<24>>24;c[c[(c[h+176>>2]|0)+(r*764|0)+756>>2]>>2]=a[l+20+(r*30|0)+25|0]|0;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+40>>2]=r;c[(c[h+176>>2]|0)+(r*764|0)+36>>2]=((c[(c[h+180>>2]|0)+(r*52|0)+32>>2]|0)!=0^1^1)&1;p5((c[h+176>>2]|0)+(r*764|0)|0,l+20+(r*30|0)|0,22)|0;hy((c[h+176>>2]|0)+(r*764|0)|0)|0;r=r+1|0}c[h+132>>2]=ac(c[h+136>>2]|0,c[h+128>>2]|0)|0;p5(h|0,l|0,20)|0;l=0;q=aT(f|0)|0;r=0;while(1){if((r|0)>=(c[h+128>>2]|0)){break}v=0;while(1){if((v|0)>=(c[h+136>>2]<<6|0)){break}bB(m|0,1,4,f|0)|0;dj(k,m|0);if((a[k+3|0]|0)!=0){l=l|1<<d[k+3|0]}else{if((a[k+4|0]|0)!=0){l=l|1}}do{if((d[k+3|0]|0)!=0){if((d[k+3|0]|0)==1){break}if((d[k+3|0]|0)==2){break}o=0}}while(0);if((d[k+3|0]|0)==1){if((d[k+4|0]|0)==0){o=0}}if((d[k+3|0]|0)==2){do{if((a[k+4|0]&15|0)!=0){if((a[k+4|0]&240|0)==0){break}o=0}}while(0)}v=v+1|0}r=r+1|0}if((l&-7|0)!=0){o=0}if((o|0)!=0){w=29512}else{if((l&-53256|0)==0){w=27704}else{if((l&-63496|0)==0){w=26568}else{w=25512}}}aM(h+64|0,64,24648,(l=i,i=i+8|0,c[l>>2]=w,l)|0)|0;i=l;bK(f|0,g+q|0,0)|0;c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;r=0;while(1){if((r|0)>=(c[h+128>>2]|0)){break}q=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(r<<2)>>2]=q;c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2]=64;q=0;while(1){if((q|0)>=(c[h+136>>2]|0)){break}g=(ac(r,c[h+136>>2]|0)|0)+q|0;c[(c[(c[h+168>>2]|0)+(r<<2)>>2]|0)+4+(q<<2)>>2]=g;l=pT(((c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(g<<2)>>2]=l;c[c[(c[h+172>>2]|0)+(g<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(r<<2)>>2]>>2];q=q+1|0}v=0;while(1){if((v|0)>=(c[h+136>>2]<<6|0)){break}x=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(r<<2)>>2]|0)+4+(((v|0)%(c[h+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((v|0)/(c[h+136>>2]|0)|0)<<3)|0;bB(m|0,1,4,f|0)|0;dj(x,m|0);v=v+1|0}r=r+1|0}r=0;while(1){if((r|0)>=(c[h+140>>2]|0)){break}r=r+1|0}m=n+1280|0;c[m>>2]=c[m>>2]|8192;if((o|0)!=0){c[h+152>>2]=c[h+160>>2];c[h+160>>2]=0;r=0;while(1){if((r|0)>=(c[h+140>>2]|0)){break}r=r+1|0}r=0;while(1){if((r|0)>=(c[h+128>>2]|0)){break}v=0;while(1){if((v|0)>=(c[h+136>>2]<<6|0)){break}x=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(r<<2)>>2]|0)+4+(((v|0)%(c[h+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((v|0)/(c[h+136>>2]|0)|0)<<3)|0;if((d[x+3|0]|0)==1){a[x+3|0]=0}else{do{if((d[x+3|0]|0)==2){if((a[x+4|0]&240|0)!=0){u=2975;break}a[x+3|0]=1}else{u=2975}}while(0);if((u|0)==2975){u=0;do{if((d[x+3|0]|0)==2){if((a[x+4|0]&15|0)!=0){break}o=x+4|0;a[o]=d[o]>>4&255}}while(0)}}v=v+1|0}r=r+1|0}}else{if((c[h+160>>2]|0)>=(c[h+156>>2]|0)){c[h+160>>2]=0}}r=0;while(1){if((r|0)>=(c[h+144>>2]|0)){break}if((c[(c[h+180>>2]|0)+(r*52|0)+32>>2]|0)!=0){v=n;x=f;u=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(r*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(v,x,512,u,0)|0}r=r+1|0}i=j;return 0}function g5(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;d=a;a=b;b=c;bK(d|0,b|0,0)|0;if(((ja(d)|0)&255|0)>32){e=-1;f=e;return f|0}c=(jc(d)|0)&65535;g=(jc(d)|0)&65535;h=(jc(d)|0)&65535;do{if((c|0)>=138){if((g|0)<138){break}if((h|0)<138){break}i=d;j=b+c|0;bK(i|0,j|0,0)|0;j=((ja(d)|0)&255)+1|0;i=0;k=0;while(1){if((i|0)>=(j|0)){break}l=(ja(d)|0)&255;if((l|0)>32){m=3006;break}if((l|0)>(k|0)){k=l}ja(d)|0;i=i+1|0}if((m|0)==3006){e=-1;f=e;return f|0}bK(d|0,h|0,0)|0;i=0;while(1){if((i|0)>=(k|0)){break}if(((ja(d)|0)&255|0)!=(i+1|0)){m=3014;break}jc(d)|0;jc(d)|0;jc(d)|0;i=i+1|0}if((m|0)==3014){e=-1;f=e;return f|0}if(((ja(d)|0)&255|0)!=255){e=-1;f=e;return f|0}else{i=d;k=b+7|0;bK(i|0,k|0,0)|0;di(d,a,18);e=0;f=e;return f|0}}}while(0);e=-1;f=e;return f|0}function g6(e,f,g){e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0;h=i;i=i+2928|0;j=h|0;k=h+104|0;l=h+2152|0;m=h+2536|0;n=e;e=f;f=n|0;bK(e|0,g|0,0)|0;c[f+148>>2]=(ja(e)|0)&255;g=(jc(e)|0)&65535;o=(jc(e)|0)&65535;p=(jc(e)|0)&65535;bB(j|0,18,1,e|0)|0;dh(f|0,j|0,18)|0;dm(n,29480,(q=i,i=i+1|0,i=i+7>>3<<3,c[q>>2]=0,q)|0);i=q;jc(e)|0;bK(e|0,g|0,0)|0;c[f+156>>2]=((ja(e)|0)&255)+1;g=0;q=0;while(1){if((g|0)>=(c[f+156>>2]|0)){break}c[k+(g<<3)>>2]=(ja(e)|0)&255;c[k+(g<<3)+4>>2]=(jb(e)|0)<<24>>24;r=0;s=0;while(1){if((r|0)>=(g|0)){break}if((c[k+(g<<3)>>2]|0)==(c[k+(r<<3)>>2]|0)){if((c[k+(g<<3)+4>>2]|0)==(c[k+(r<<3)+4>>2]|0)){t=3035;break}}r=r+1|0}if((t|0)==3035){t=0;a[f+952+g|0]=a[f+952+r|0]|0;s=1}if((s|0)==0){u=q;q=u+1|0;a[f+952+g|0]=u&255}g=g+1|0}c[f+136>>2]=3;c[f+128>>2]=q;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;c[f+140>>2]=15;c[f+144>>2]=c[f+140>>2];q=(p-o|0)/33|0;c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;bK(e|0,p|0,0)|0;p=pT(c[f+128>>2]|0,4)|0;g=0;while(1){if((g|0)>=32){break}if(((ja(e)|0)&255|0)==255){t=3055;break}c[l+(g*12|0)>>2]=(jc(e)|0)&65535;c[l+(g*12|0)+4>>2]=(jc(e)|0)&65535;c[l+(g*12|0)+8>>2]=(jc(e)|0)&65535;g=g+1|0}g=0;while(1){if((g|0)>=(c[f+156>>2]|0)){break}u=(c[k+(g<<3)>>2]|0)-1|0;v=d[f+952+g|0]|0;w=c[k+(g<<3)+4>>2]|0;if((c[p+(v<<2)>>2]|0)==0){x=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(v<<2)>>2]=x;c[c[(c[f+168>>2]|0)+(v<<2)>>2]>>2]=64;x=0;while(1){if((x|0)>=(c[f+136>>2]|0)){break}y=(ac(v,c[f+136>>2]|0)|0)+x|0;c[(c[(c[f+168>>2]|0)+(v<<2)>>2]|0)+4+(x<<2)>>2]=y;z=pT(((c[c[(c[f+168>>2]|0)+(v<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(y<<2)>>2]=z;c[c[(c[f+172>>2]|0)+(y<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(v<<2)>>2]>>2];x=x+1|0}r=0;while(1){if((r|0)>=3){break}x=0;s=0;bK(e|0,c[l+(u*12|0)+(r<<2)>>2]|0,0)|0;do{while(1){A=(ja(e)|0)&255;if((A|0)==255){t=3075;break}B=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(v<<2)>>2]|0)+4+(r<<2)>>2]<<2)>>2]|0)+4+(x<<3)|0;if((A|0)<=95){t=3077;break}if((A|0)<=111){a[B+1|0]=A-95-1&255}else{if((A|0)<=127){a[B+3|0]=-16;a[B+4|0]=A-112&255}else{if((A|0)==128){t=3083;break}if((A|0)==129){x=x+(s+1)|0}else{if((A|0)==130){a[B+3|0]=-16;a[B+4|0]=0;a[B+5|0]=-14;a[B+6|0]=0}else{if((A|0)<=142){a[B+3|0]=A+240-128&255;a[B+4|0]=ja(e)|0;a[B+5|0]=-15;a[B+6|0]=ja(e)|0}else{s=A-161|0}}}}}}if((t|0)==3075){t=0}else if((t|0)==3077){t=0;a[B|0]=A+18+w&255;x=x+(s+1)|0}else if((t|0)==3083){t=0;a[B|0]=-127;x=x+(s+1)|0}}while((A|0)!=255);r=r+1|0}c[p+(v<<2)>>2]=1}g=g+1|0}pU(p);c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}bK(e|0,27,0)|0;g=0;while(1){if((g|0)>=(c[f+140>>2]|0)){break}p7(m|0,0,392);p=pT(64,1)|0;c[(c[f+176>>2]|0)+(g*764|0)+756>>2]=p;c[(c[f+176>>2]|0)+(g*764|0)+36>>2]=1;c[c[(c[f+176>>2]|0)+(g*764|0)+756>>2]>>2]=64;c[(c[(c[f+176>>2]|0)+(g*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[f+176>>2]|0)+(g*764|0)+756>>2]|0)+12>>2]=-1;c[(c[(c[f+176>>2]|0)+(g*764|0)+756>>2]|0)+40>>2]=g;bB(j|0,1,99,e|0)|0;if((d[j+97|0]|0)==0){c[m>>2]=32;c[m+4>>2]=33}else{c[m>>2]=(d[j+97|0]|0)-1;if((c[m>>2]|0)>31){c[m>>2]=31}c[m+4>>2]=(d[j+97|0]|0)+(d[j+98|0]|0);if((c[m+4>>2]|0)>32){c[m+4>>2]=32}if((c[m+4>>2]|0)==0){c[m+4>>2]=1}if((c[m>>2]|0)>=(c[m+4>>2]|0)){c[m>>2]=(c[m+4>>2]|0)-1}if((c[m+4>>2]|0)<32){p=m+4|0;c[p>>2]=(c[p>>2]|0)+(33-((c[m>>2]|0)+1));c[m>>2]=32}}r=0;while(1){if((r|0)>=31){break}p=m+8+(r<<3)|0;A=j+((r*3|0)+1)|0;p7(p|0,0,8);if((~d[A+1|0]&128|0)!=0){B=p+4|0;c[B>>2]=c[B>>2]|64;a[p+3|0]=d[A+1|0]&31;if((a[p+3|0]&16|0)!=0){B=p+3|0;a[B]=(a[B]|240)&255}}if((~d[A+1|0]&64|0)!=0){B=p+4|0;c[B>>2]=c[B>>2]|32}a[p+2|0]=d[A|0]&15;b[p>>1]=((a[A|0]&240)<<4|d[A+2|0])&65535;if((~d[A+1|0]&32|0)!=0){b[p>>1]=-(b[p>>1]|0)&65535}A=p+4|0;c[A>>2]=c[A>>2]|8;r=r+1|0}gQ(n,e,16384,(c[f+180>>2]|0)+(g*52|0)|0,m)|0;g=g+1|0}bK(e|0,o|0,0)|0;c[n+6556>>2]=pT(1,1320)|0;o=c[n+6556>>2]|0;g=0;while(1){if((g|0)>=(q|0)){break}m=o+(((ja(e)|0)&255)*88|0)|0;c[m>>2]=32;c[m+4>>2]=31;r=0;while(1){if((r|0)>=32){break}a[m+8+r|0]=jb(e)|0;r=r+1|0}g=g+1|0}g=0;while(1){if((g|0)>=4){break}c[f+184+(g*12|0)>>2]=128;c[f+184+(g*12|0)+8>>2]=1;g=g+1|0}c[n+6560>>2]=2872;i=h;return 0}function g7(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;c=a;if((jl(c)|0)!=1398032717){d=-1;e=d;return e|0}else{di(c,b,0);d=0;e=d;return e|0}return 0}function g8(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;j=i;i=i+416|0;k=j|0;l=j+408|0;m=f;f=g;g=h;h=m|0;bK(f|0,g|0,0)|0;c[k>>2]=jl(f)|0;c[k+4>>2]=jl(f)|0;jl(f)|0;jl(f)|0;b[k+16>>1]=jd(f)|0;b[k+18>>1]=jd(f)|0;b[k+20>>1]=jd(f)|0;bB(k+22|0,128,1,f|0)|0;n=0;while(1){if((n|0)>=64){break}c[k+152+(n<<2)>>2]=(jl(f)|0)+12;n=n+1|0}c[h+156>>2]=e[k+18>>1]|0;c[h+128>>2]=e[k+20>>1]|0;c[h+140>>2]=e[k+16>>1]|0;c[h+144>>2]=c[h+140>>2];c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;n=0;while(1){if((n|0)>=(c[h+156>>2]|0)){break}a[h+952+n|0]=a[k+22+n|0]|0;n=n+1|0}dm(m,24536,(o=i,i=i+1|0,i=i+7>>3<<3,c[o>>2]=0,o)|0);i=o;c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;n=0;while(1){if((n|0)>=(c[h+128>>2]|0)){break}o=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(n<<2)>>2]=o;c[c[(c[h+168>>2]|0)+(n<<2)>>2]>>2]=64;o=0;while(1){if((o|0)>=(c[h+136>>2]|0)){break}p=(ac(n,c[h+136>>2]|0)|0)+o|0;c[(c[(c[h+168>>2]|0)+(n<<2)>>2]|0)+4+(o<<2)>>2]=p;q=pT(((c[c[(c[h+168>>2]|0)+(n<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(p<<2)>>2]=q;c[c[(c[h+172>>2]|0)+(p<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(n<<2)>>2]>>2];o=o+1|0}bK(f|0,g+(c[k+152+(n<<2)>>2]|0)+8|0,0)|0;o=0;while(1){if((o|0)>=4){break}p=0;while(1){if((p|0)>=64){break}q=(c[(c[m+172>>2]|0)+(c[(c[(c[m+168>>2]|0)+(n<<2)>>2]|0)+4+(o<<2)>>2]<<2)>>2]|0)+4+(p<<3)|0;r=ja(f)|0;if((r&128|0)!=0){p=p+(r&127)|0}else{s=ja(f)|0;t=ja(f)|0;u=s&255&63;a[q|0]=u;if((u&255|0)!=0){u=q|0;a[u]=(d[u]|0)+47&255}a[q+1|0]=r&255&31;a[q+3|0]=((s&255)>>4&12|(r&255)>>5)&255;a[q+4|0]=t;dg(q)}p=p+1|0}o=o+1|0}n=n+1|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}bK(f|0,g+(c[k+4>>2]|0)+(c[h+144>>2]<<2)|0,0)|0;n=0;while(1){if((n|0)>=(c[h+144>>2]|0)){break}b[l>>1]=jd(f)|0;a[l+2|0]=ja(f)|0;a[l+3|0]=ja(f)|0;b[l+4>>1]=jd(f)|0;b[l+6>>1]=jd(f)|0;k=pT(64,1)|0;c[(c[h+176>>2]|0)+(n*764|0)+756>>2]=k;c[(c[h+180>>2]|0)+(n*52|0)+32>>2]=(e[l>>1]|0)<<1;c[(c[h+180>>2]|0)+(n*52|0)+36>>2]=(e[l+4>>1]|0)<<1;c[(c[h+180>>2]|0)+(n*52|0)+40>>2]=(c[(c[h+180>>2]|0)+(n*52|0)+36>>2]|0)+((e[l+6>>1]|0)<<1);c[(c[h+180>>2]|0)+(n*52|0)+44>>2]=(e[l+6>>1]|0|0)>1?2:0;c[(c[(c[h+176>>2]|0)+(n*764|0)+756>>2]|0)+16>>2]=((d[l+2|0]|0)<<4&255)<<24>>24;c[c[(c[h+176>>2]|0)+(n*764|0)+756>>2]>>2]=d[l+3|0]|0;c[(c[(c[h+176>>2]|0)+(n*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]=n;c[(c[h+176>>2]|0)+(n*764|0)+36>>2]=((c[(c[h+180>>2]|0)+(n*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[h+176>>2]|0)+(n*764|0)+40>>2]=4095;if((c[(c[h+180>>2]|0)+(n*52|0)+32>>2]|0)!=0){k=m;g=f;o=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(n*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(k,g,0,o,0)|0}n=n+1|0}n=m+1280|0;c[n>>2]=c[n>>2]|8192;i=j;return 0}function g9(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;d=i;i=i+8|0;e=d|0;f=a;a=b;b=c;bK(f|0,b+20|0,0)|0;L3966:do{if((bB(e|0,1,8,f|0)|0)>>>0<8){g=-1}else{do{if((p6(e|0,25496,8)|0)!=0){if((p6(e|0,29464,8)|0)==0){break}g=-1;break L3966}}while(0);c=f;ja(c)|0;if(((ja(f)|0)&255|0)!=2){g=-1;break}if(((ja(f)|0)&255|0)<1){g=-1;break}else{c=f;h=b|0;bK(c|0,h|0,0)|0;di(f,a,20);g=0;break}}}while(0);i=d;return g|0}function ha(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;j=i;i=i+1168|0;k=j|0;l=f;f=g;g=l|0;m=0;bK(f|0,h|0,0)|0;bB(k|0,20,1,f|0)|0;bB(k+20|0,8,1,f|0)|0;a[k+28|0]=ja(f)|0;a[k+29|0]=ja(f)|0;a[k+30|0]=ja(f)|0;a[k+31|0]=ja(f)|0;a[k+32|0]=ja(f)|0;a[k+33|0]=ja(f)|0;a[k+34|0]=ja(f)|0;bB(k+35|0,13,1,f|0)|0;h=0;while(1){if((h|0)>=31){break}bB(k+48+(h*36|0)|0,12,1,f|0)|0;a[k+48+(h*36|0)+12|0]=ja(f)|0;a[k+48+(h*36|0)+13|0]=ja(f)|0;b[k+48+(h*36|0)+14>>1]=jc(f)|0;b[k+48+(h*36|0)+16>>1]=jc(f)|0;b[k+48+(h*36|0)+18>>1]=jc(f)|0;b[k+48+(h*36|0)+20>>1]=jc(f)|0;a[k+48+(h*36|0)+22|0]=ja(f)|0;a[k+48+(h*36|0)+23|0]=ja(f)|0;b[k+48+(h*36|0)+24>>1]=jc(f)|0;c[k+48+(h*36|0)+28>>2]=jg(f)|0;b[k+48+(h*36|0)+32>>1]=jc(f)|0;h=h+1|0}if((aL(k+20|0,29464,8)|0)==0){m=1}c[g+128>>2]=d[k+33|0]|0;c[g+132>>2]=ac(c[g+128>>2]|0,c[g+136>>2]|0)|0;c[g+148>>2]=(a[k+32|0]&240)>>4;c[g+140>>2]=31;c[g+144>>2]=c[g+140>>2];c[l+1264>>2]=8363;dh(g|0,k|0,20)|0;if((m|0)!=0){m=g+64|0;aM(m|0,64,27688,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0)|0;i=n}else{m=g+64|0;o=d[k+30|0]|0;p=d[k+31|0]|0;aM(m|0,64,26512,(n=i,i=i+16|0,c[n>>2]=o,c[n+8>>2]=p,n)|0)|0;i=n}c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}n=pT(64,1)|0;c[(c[g+176>>2]|0)+(h*764|0)+756>>2]=n;n=e[k+48+(h*36|0)+16>>1]|0;c[(c[g+180>>2]|0)+(h*52|0)+32>>2]=n;c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=((n|0)!=0^1^1)&1;c[(c[g+180>>2]|0)+(h*52|0)+36>>2]=e[k+48+(h*36|0)+18>>1]|0;c[(c[g+180>>2]|0)+(h*52|0)+40>>2]=e[k+48+(h*36|0)+20>>1]|0;if((c[(c[g+180>>2]|0)+(h*52|0)+40>>2]|0)==65535){c[(c[g+180>>2]|0)+(h*52|0)+40>>2]=0}c[(c[g+180>>2]|0)+(h*52|0)+44>>2]=(c[(c[g+180>>2]|0)+(h*52|0)+40>>2]|0)>0?2:0;c[c[(c[g+176>>2]|0)+(h*764|0)+756>>2]>>2]=d[k+48+(h*36|0)+22|0]|0;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+40>>2]=h;dh((c[g+176>>2]|0)+(h*764|0)|0,k+48+(h*36|0)|0,12)|0;b[k+48+(h*36|0)+24>>1]=(((e[k+48+(h*36|0)+24>>1]|0)*8363|0|0)/8448|0)&65535;hD(e[k+48+(h*36|0)+24>>1]|0,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+12|0,(c[(c[g+176>>2]|0)+(h*764|0)+756>>2]|0)+16|0);h=h+1|0}bB(g+952|0,1,128,f|0)|0;h=0;while(1){if((h|0)>=128){break}if((d[g+952+h|0]|0|0)>=(c[g+128>>2]|0)){q=3261;break}h=h+1|0}c[g+156>>2]=h;c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;h=0;while(1){if((h|0)>=(c[g+128>>2]|0)){break}k=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(h<<2)>>2]=k;c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]=64;k=0;while(1){if((k|0)>=(c[g+136>>2]|0)){break}n=(ac(h,c[g+136>>2]|0)|0)+k|0;c[(c[(c[g+168>>2]|0)+(h<<2)>>2]|0)+4+(k<<2)>>2]=n;p=pT(((c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(n<<2)>>2]=p;c[c[(c[g+172>>2]|0)+(n<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(h<<2)>>2]>>2];k=k+1|0}k=0;while(1){if((k|0)>=(c[g+136>>2]<<6|0)){break}n=(c[(c[l+172>>2]|0)+(c[(c[(c[l+168>>2]|0)+(h<<2)>>2]|0)+4+(((k|0)%(c[g+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((k|0)/(c[g+136>>2]|0)|0)<<3)|0;p=ja(f)|0;p7(n|0,0,8);o=p&255;if((o|0)==255){p=0;q=3285}else if(!((o|0)==251|(o|0)==252|(o|0)==253)){q=3285}if((q|0)==3285){q=0;if((p&255|0)!=0){r=(p&15)+13+((((p&240)>>4)+2|0)*12|0)|0}else{r=0}a[n|0]=r&255;p=ja(f)|0;a[n+2|0]=p&255&7;a[n+1|0]=(p&248)>>3&255;p=ja(f)|0;o=n+2|0;a[o]=(d[o]|0)+((p&240)>>1)&255;if((d[n+2|0]|0|0)>64){a[n+2|0]=0}else{o=n+2|0;a[o]=(a[o]|0)+1&255}a[n+3|0]=a[8560+(p&15)|0]|0;a[n+4|0]=ja(f)|0;p=d[n+3|0]|0;if((p|0)==15){a[n+4|0]=(a[n+4|0]&240)>>4&255}else if((p|0)==255){a[n+3|0]=0;a[n+4|0]=0}}k=k+1|0}h=h+1|0}h=0;while(1){if((h|0)>=(c[g+140>>2]|0)){break}if((c[(c[g+180>>2]|0)+(h*52|0)+32>>2]|0)>1){r=l;q=f;k=(c[g+180>>2]|0)+(h*52|0)|0;gQ(r,q,0,k,0)|0}else{c[(c[g+176>>2]|0)+(h*764|0)+36>>2]=0}h=h+1|0}h=l+1280|0;c[h>>2]=c[h>>2]|609;c[l+1284>>2]=2;i=j;return 0}function hb(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;d=i;i=i+8|0;e=d|0;f=a;a=b;b=c;bK(f|0,b+20|0,0)|0;if((bB(e|0,1,8,f|0)|0)>>>0<8){g=-1;h=g;i=d;return h|0}do{if((p6(e|0,25464,8)|0)!=0){if((p6(e|0,29448,8)|0)==0){break}g=-1;h=g;i=d;return h|0}}while(0);bK(f|0,b+60|0,0)|0;if((bB(e|0,1,4,f|0)|0)>>>0<4){g=-1;h=g;i=d;return h|0}if((p6(e|0,24608,4)|0)!=0){g=-1;h=g;i=d;return h|0}else{e=f;c=b|0;bK(e|0,c|0,0)|0;di(f,a,20);g=0;h=g;i=d;return h|0}return 0}function hc(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;j=i;i=i+152|0;k=j|0;l=j+8|0;m=j+72|0;n=f;f=g;g=h;h=n|0;o=0;p=0;q=0;bK(f|0,g|0,0)|0;bB(l|0,20,1,f|0)|0;bB(l+20|0,8,1,f|0)|0;b[l+28>>1]=jc(f)|0;b[l+30>>1]=jc(f)|0;b[l+32>>1]=jc(f)|0;b[l+34>>1]=jc(f)|0;b[l+36>>1]=jc(f)|0;b[l+38>>1]=jc(f)|0;b[l+40>>1]=jc(f)|0;a[l+42|0]=ja(f)|0;a[l+43|0]=ja(f)|0;b[l+44>>1]=jc(f)|0;b[l+46>>1]=jc(f)|0;b[l+48>>1]=jc(f)|0;b[l+50>>1]=jc(f)|0;b[l+52>>1]=jc(f)|0;b[l+54>>1]=jc(f)|0;b[l+56>>1]=jc(f)|0;b[l+58>>1]=jc(f)|0;bB(l+60|0,4,1,f|0)|0;if((aL(l+20|0,29448,8)|0)==0){q=1}c[h+140>>2]=e[l+50>>1]|0;c[h+128>>2]=e[l+48>>1]|0;c[h+132>>2]=ac(c[h+128>>2]|0,c[h+136>>2]|0)|0;c[h+156>>2]=e[l+52>>1]|0;c[h+148>>2]=(a[l+43|0]&240)>>4;c[h+144>>2]=c[h+140>>2];c[n+1264>>2]=8363;bK(f|0,g+(e[l+32>>1]<<4)|0,0)|0;r=jc(f)|0;bK(f|0,g+((r&65535)<<4)|0,0)|0;r=jc(f)|0;if((r&65535|0)==(e[l+28>>1]|0)){o=1}p5(h|0,l|0,20)|0;if((q|0)!=0){dm(n,27672,(s=i,i=i+1|0,i=i+7>>3<<3,c[s>>2]=0,s)|0);i=s}else{q=h+64|0;r=(o|0)!=0?0:1;aM(q|0,64,26496,(s=i,i=i+8|0,c[s>>2]=r,s)|0)|0;i=s}s=pT(2,c[h+128>>2]|0)|0;r=pT(2,c[h+140>>2]|0)|0;bK(f|0,g+(e[l+32>>1]<<4)|0,0)|0;q=0;while(1){if((q|0)>=(c[h+128>>2]|0)){break}b[s+(q<<1)>>1]=jc(f)|0;q=q+1|0}bK(f|0,g+(e[l+34>>1]<<4)|0,0)|0;q=0;while(1){if((q|0)>=(c[h+140>>2]|0)){break}b[r+(q<<1)>>1]=jc(f)|0;q=q+1|0}bK(f|0,g+(e[l+36>>1]<<4)+32|0,0)|0;q=0;while(1){if((q|0)>=(c[h+156>>2]|0)){break}a[h+952+q|0]=ja(f)|0;bK(f|0,4,1)|0;q=q+1|0}c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}q=0;while(1){if((q|0)>=(c[h+140>>2]|0)){break}l=pT(64,1)|0;c[(c[h+176>>2]|0)+(q*764|0)+756>>2]=l;bK(f|0,g+(e[r+(q<<1)>>1]<<4)|0,0)|0;a[m|0]=ja(f)|0;bB(m+1|0,13,1,f|0)|0;b[m+14>>1]=jc(f)|0;c[m+16>>2]=jg(f)|0;c[m+20>>2]=jg(f)|0;c[m+24>>2]=jg(f)|0;a[m+28|0]=ja(f)|0;a[m+29|0]=ja(f)|0;a[m+30|0]=ja(f)|0;a[m+31|0]=ja(f)|0;b[m+32>>1]=jc(f)|0;b[m+34>>1]=jc(f)|0;bB(m+36|0,4,1,f|0)|0;b[m+40>>1]=jc(f)|0;b[m+42>>1]=jc(f)|0;c[m+44>>2]=jg(f)|0;bB(m+48|0,28,1,f|0)|0;bB(m+76|0,4,1,f|0)|0;l=c[m+16>>2]|0;c[(c[h+180>>2]|0)+(q*52|0)+32>>2]=l;c[(c[h+176>>2]|0)+(q*764|0)+36>>2]=((l|0)!=0^1^1)&1;c[(c[h+180>>2]|0)+(q*52|0)+36>>2]=c[m+20>>2];c[(c[h+180>>2]|0)+(q*52|0)+40>>2]=c[m+24>>2];if((c[(c[h+180>>2]|0)+(q*52|0)+40>>2]|0)==65535){c[(c[h+180>>2]|0)+(q*52|0)+40>>2]=0}c[(c[h+180>>2]|0)+(q*52|0)+44>>2]=(c[(c[h+180>>2]|0)+(q*52|0)+40>>2]|0)>0?2:0;c[c[(c[h+176>>2]|0)+(q*764|0)+756>>2]>>2]=d[m+28|0]|0;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]=q;dh((c[h+176>>2]|0)+(q*764|0)|0,m+48|0,12)|0;b[m+32>>1]=(((e[m+32>>1]|0)*8363|0|0)/8448|0)&65535;hD(e[m+32>>1]|0,(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+12|0,(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+16|0);q=q+1|0}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;q=0;while(1){if((q|0)>=(c[h+128>>2]|0)){break}m=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(q<<2)>>2]=m;c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2]=64;m=0;while(1){if((m|0)>=(c[h+136>>2]|0)){break}l=(ac(q,c[h+136>>2]|0)|0)+m|0;c[(c[(c[h+168>>2]|0)+(q<<2)>>2]|0)+4+(m<<2)>>2]=l;t=pT(((c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(l<<2)>>2]=t;c[c[(c[h+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(q<<2)>>2]>>2];m=m+1|0}if((b[s+(q<<1)>>1]|0)!=0){m=f;l=g+(e[s+(q<<1)>>1]<<4)|0;bK(m|0,l|0,0)|0;if((o|0)!=0){l=f;bK(l|0,2,1)|0}l=0;while(1){if((l|0)>=64){break}m=ja(f)|0;if((m&255|0)==0){l=l+1|0;continue}t=m&31;if((t|0)>=(c[h+136>>2]|0)){u=k}else{u=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(q<<2)>>2]|0)+4+(t<<2)>>2]<<2)>>2]|0)+4+(l<<3)|0}p=u;if((m&32|0)!=0){t=ja(f)|0;v=t&255;if((v|0)==255){t=0}else if((v|0)==254){t=-127}else{t=(((t&240)>>4)*12|0)+37+(t&15)&255}a[p|0]=t;a[p+1|0]=ja(f)|0}if((m&64|0)!=0){a[p+2|0]=((ja(f)|0)&255)+1&255}if((m&128|0)!=0){a[p+3|0]=a[8544+((ja(f)|0)&255)|0]|0;a[p+4|0]=ja(f)|0;m=d[p+3|0]|0;if((m|0)==255){a[p+3|0]=0;a[p+4|0]=0}else if((m|0)==15){a[p+4|0]=(a[p+4|0]&240)>>4&255}}}}q=q+1|0}pU(s);pU(r);q=0;while(1){if((q|0)>=(c[h+140>>2]|0)){break}gQ(n,f,0,(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(q*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;q=q+1|0}q=n+1280|0;c[q>>2]=c[q>>2]|609;c[n+1284>>2]=2;i=j;return 0}function hd(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;c=a;a=b;b=jl(c)|0;d=jl(c)|0;do{if((b|0)==33624851){if((d|0)!=336724235){break}if(((ja(c)|0)&255|0)>1){e=-1;f=e;return f|0}g=c;ja(g)|0;g=c;jc(g)|0;g=c;jc(g)|0;g=c;je(g)|0;g=0;while(1){if((g|0)>=63){break}if((~((ja(c)|0)&255)&128|0)!=0){h=c;je(h)|0}g=g+1|0}di(c,a,(ja(c)|0)&255);e=0;f=e;return f|0}}while(0);e=-1;f=e;return f|0}function he(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;g=i;i=i+392|0;h=g|0;j=g+256|0;k=g+264|0;l=b;b=e;e=l|0;bK(b|0,f|0,0)|0;bK(b|0,8,1)|0;ja(b)|0;dm(l,24368,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0);i=f;c[e+136>>2]=(ja(b)|0)&255;m=(jc(b)|0)&65535;c[e+128>>2]=m;c[e+156>>2]=m;c[e+132>>2]=(jc(b)|0)&65535;je(b)|0;c[e+144>>2]=63;c[e+140>>2]=63;c[e+176>>2]=pT(764,c[e+140>>2]|0)|0;if((c[e+144>>2]|0)!=0){c[e+180>>2]=pT(52,c[e+144>>2]|0)|0}m=0;while(1){if((m|0)>=(c[e+140>>2]|0)){break}n=pT(64,1)|0;c[(c[e+176>>2]|0)+(m*764|0)+756>>2]=n;c[h+(m<<2)>>2]=(ja(b)|0)&255;if((~c[h+(m<<2)>>2]&128|0)!=0){n=(je(b)|0)<<1;c[(c[e+180>>2]|0)+(m*52|0)+32>>2]=n}m=m+1|0}n=(ja(b)|0)&255;bB(e|0,1,n|0,b|0)|0;bB(j|0,1,8,b|0)|0;o=e+132|0;c[o>>2]=(c[o>>2]|0)+1;c[e+172>>2]=pT(4,c[e+132>>2]|0)|0;c[e+168>>2]=pT(4,(c[e+128>>2]|0)+1|0)|0;n=(ja(b)|0)&255;do{if((n|0)!=0){if((n|0)==1){break}p=-1;r=p;i=g;return r|0}}while(0);o=(ac(c[e+156>>2]|0,c[e+136>>2]|0)|0)<<1;s=pS(o)|0;t=s;if((s|0)==0){p=-1;r=p;i=g;return r|0}do{if((n|0)!=0){if((j4(b,t,13,0,o,o,31)|0)!=0){break}pU(t);p=-1;r=p;i=g;return r|0}else{s=t;u=o;v=b;bB(s|0,1,u|0,v|0)|0}}while(0);m=0;while(1){if((m|0)>=(c[e+156>>2]|0)){break}v=pT(1,((c[e+136>>2]|0)-1<<2)+8|0)|0;c[(c[e+168>>2]|0)+(m<<2)>>2]=v;c[c[(c[e+168>>2]|0)+(m<<2)>>2]>>2]=64;w=0;while(1){if((w|0)>=(c[e+136>>2]|0)){break}v=hf(t+((ac(m,c[e+136>>2]|0)|0)+w<<1)|0)|0;c[(c[(c[e+168>>2]|0)+(m<<2)>>2]|0)+4+(w<<2)>>2]=v;if((c[(c[(c[e+168>>2]|0)+(m<<2)>>2]|0)+4+(w<<2)>>2]|0)==4096){c[(c[(c[e+168>>2]|0)+(m<<2)>>2]|0)+4+(w<<2)>>2]=(c[e+132>>2]|0)-1}w=w+1|0}a[e+952+m|0]=m&255;m=m+1|0}pU(t);n=(ja(b)|0)&255;do{if((n|0)!=0){if((n|0)==1){break}p=-1;r=p;i=g;return r|0}}while(0);o=(c[e+132>>2]|0)-1<<6<<2;v=pS(o)|0;t=v;if((v|0)==0){p=-1;r=p;i=g;return r|0}do{if((n|0)!=0){if((j4(b,t,13,0,o,o,31)|0)!=0){break}pU(t);p=-1;r=p;i=g;return r|0}else{v=t;u=o;s=b;bB(v|0,1,u|0,s|0)|0}}while(0);m=0;while(1){if((m|0)>=((c[e+132>>2]|0)-1|0)){break}o=pT(523,1)|0;c[(c[e+172>>2]|0)+(m<<2)>>2]=o;c[c[(c[e+172>>2]|0)+(m<<2)>>2]>>2]=64;w=0;while(1){if((w|0)>=(c[c[(c[e+172>>2]|0)+(m<<2)>>2]>>2]|0)){break}o=(c[(c[e+172>>2]|0)+(m<<2)>>2]|0)+4+(w<<3)|0;s=hg(t+((m<<6)+w<<2)|0)|0;a[o|0]=s&63;if((a[o|0]|0)!=0){u=o|0;a[u]=(d[u]|0)+48&255}a[o+1|0]=(s&8128)>>>6&255;a[o+3|0]=(s&1032192)>>>14&255;if((d[j+(d[o+3|0]>>3)|0]&1<<(a[o+3|0]&7)|0)!=0){hh(o,(s&-1048576)>>>20)}else{a[o+3|0]=0}w=w+1|0}m=m+1|0}pU(t);t=pT(523,1)|0;c[(c[e+172>>2]|0)+(m<<2)>>2]=t;c[c[(c[e+172>>2]|0)+(m<<2)>>2]>>2]=64;m=0;while(1){if((m|0)>=(c[e+140>>2]|0)){break}p7(k|0,0,128);bB(k|0,1,c[h+(m<<2)>>2]&127|0,b|0)|0;dh((c[e+176>>2]|0)+(m*764|0)|0,k|0,32)|0;if((~c[h+(m<<2)>>2]&128|0)!=0){t=(je(b)|0)<<1;c[(c[e+180>>2]|0)+(m*52|0)+36>>2]=t;t=(je(b)|0)<<1;if((t|0)>2){w=(c[e+180>>2]|0)+(m*52|0)+44|0;c[w>>2]=c[w>>2]|2}c[(c[e+180>>2]|0)+(m*52|0)+40>>2]=(c[(c[e+180>>2]|0)+(m*52|0)+36>>2]|0)+t;c[(c[e+176>>2]|0)+(m*764|0)+36>>2]=1;t=(ja(b)|0)&255;c[c[(c[e+176>>2]|0)+(m*764|0)+756>>2]>>2]=t;c[(c[(c[e+176>>2]|0)+(m*764|0)+756>>2]|0)+8>>2]=128;t=(((ja(b)|0)&255)<<4&255)<<24>>24;c[(c[(c[e+176>>2]|0)+(m*764|0)+756>>2]|0)+16>>2]=t;c[(c[(c[e+176>>2]|0)+(m*764|0)+756>>2]|0)+40>>2]=m}do{if((c[h+(m<<2)>>2]&128|0)!=0){x=3511}else{if((c[(c[e+180>>2]|0)+(m*52|0)+32>>2]|0)==0){x=3511;break}n=(ja(b)|0)&255;do{if((n|0)!=0){if((n|0)==1){break}t=c[q>>2]|0;bE(t|0,29384,(f=i,i=i+1|0,i=i+7>>3<<3,c[f>>2]=0,f)|0)|0;i=f}}while(0);if((n|0)==1){t=pS(c[(c[e+180>>2]|0)+(m*52|0)+32>>2]|0)|0;w=b;j=t;o=c[(c[e+180>>2]|0)+(m*52|0)+32>>2]|0;s=c[(c[e+180>>2]|0)+(m*52|0)+32>>2]|0;j4(w,j,13,0,o,s,31)|0;s=l;o=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(m*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;j=t;gQ(s,0,17,o,j)|0;pU(t)}else{t=l;j=b;o=(c[e+180>>2]|0)+((c[(c[(c[e+176>>2]|0)+(m*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0;gQ(t,j,128,o,0)|0}}}while(0);if((x|0)==3511){x=0}m=m+1|0}m=0;while(1){if((m|0)>=(c[e+136>>2]|0)){break}c[e+184+(m*12|0)>>2]=(((m+3|0)/2|0|0)%2|0)*255|0;m=m+1|0}p=0;r=p;i=g;return r|0}function hf(a){a=a|0;var b=0,c=0;b=a;a=b;b=a+1|0;c=b;b=c+1|0;return(d[c]|0)<<8|(d[a]|0)|0}function hg(a){a=a|0;var b=0,c=0,e=0,f=0;b=a;a=b;b=a+1|0;c=b;b=c+1|0;e=b;b=e+1|0;f=b;b=f+1|0;return(d[f]|0)<<24|(d[e]|0)<<16|(d[c]|0)<<8|(d[a]|0)|0}function hh(b,c){b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0;e=b;b=c;c=d[e+3|0]|0;if((c|0)==47){do{if((b|0)>=256){if((b|0)>2048){f=3585;break}a[e+3|0]=15;a[e+4|0]=b+4>>3&255}else{f=3585}}while(0);return}else if((c|0)==0|(c|0)==1|(c|0)==2){a[e+4|0]=b&255;if((b>>8|0)!=0){a[e+5|0]=-96;a[e+6|0]=b>>8&255}return}else if((c|0)==11|(c|0)==12|(c|0)==13|(c|0)==15){a[e+4|0]=b&255;return}else if((c|0)==3|(c|0)==4|(c|0)==7){a[e+4|0]=b&255;return}else if((c|0)==10){if((b&255|0)!=0){a[e+4|0]=b&255}else{a[e+3|0]=0}a[e+5|0]=14;a[e+6|0]=(16|(b&3840)>>8)&255;return}else if((c|0)==9){a[e+4|0]=b>>1&255;return}else if((c|0)==5|(c|0)==6){a[e+4|0]=b&255;if((b|0)==0){g=e+3|0;a[g]=(d[g]|0)-2&255}return}else if((c|0)==34){a[e+3|0]=2;a[e+4|0]=b&255;if((b>>8|0)!=0){a[e+5|0]=-95;a[e+6|0]=b>>8&255}return}else if(!((c|0)==42|(c|0)==43|(c|0)==48|(c|0)==49|(c|0)==50))if((c|0)==19){a[e+3|0]=14;a[e+4|0]=(48|b&15)&255;return}else if((c|0)==20){a[e+3|0]=14;a[e+4|0]=(64|b&15)&255;return}else if((c|0)==21){a[e+3|0]=14;a[e+4|0]=(80|b&15)&255;return}else if((c|0)==22){a[e+3|0]=14;a[e+4|0]=(96|b&15)&255;return}else if((c|0)==23){a[e+3|0]=14;a[e+4|0]=(112|b&15)&255;return}else if((c|0)==25){if((b|0)<16){a[e+3|0]=14;a[e+4|0]=(144|b&15)&255}else{a[e+3|0]=0}return}else if((c|0)==17|(c|0)==18|(c|0)==26|(c|0)==27){if((d[e+3|0]|0|0)==17){h=1}else{h=(d[e+3|0]|0|0)==26}g=(h?1:2)&255;if((d[e+3|0]|0|0)==17){i=1}else{i=(d[e+3|0]|0|0)==18}h=(i?10:11)&255;do{if((b&255|0)!=0){if((b&255|0)>=16){f=3565;break}a[e+3|0]=14;a[e+4|0]=((g&255)<<4|b&15)&255}else{f=3565}}while(0);if((f|0)==3565){a[e+3|0]=0}if((b>>8|0)!=0){a[e+5|0]=14;a[e+6|0]=((h&255)<<4|b>>8)&255}return}else if((c|0)==28){a[e+3|0]=14;a[e+4|0]=(192|b&15)&255;return}else if((c|0)==29){a[e+3|0]=14;a[e+4|0]=(208|b&15)&255;return}else if((c|0)==30){a[e+3|0]=14;a[e+4|0]=(224|b&15)&255;return}else if((c|0)==31){a[e+3|0]=0;return}else if((c|0)==32){a[e+3|0]=0;a[e+4|0]=b&255;if((b>>8|0)!=0){a[e+5|0]=-95;a[e+6|0]=b>>8&255}return}else if((c|0)==33){a[e+3|0]=1;a[e+4|0]=b&255;if((b>>8|0)!=0){a[e+5|0]=-95;a[e+6|0]=b>>8&255}return}a[e+3|0]=0;return}function hi(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;c=i;i=i+16|0;d=c|0;e=a;a=b;if((bB(d|0,1,8,e|0)|0)>>>0<8){f=-1;g=f;i=c;return g|0}do{if((p6(d|0,27648,8)|0)!=0){if((p6(d|0,26480,8)|0)==0){break}f=-1;g=f;i=c;return g|0}}while(0);di(e,a,0);f=0;g=f;i=c;return g|0}function hj(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;g=i;i=i+128|0;h=g|0;j=g+16|0;k=g+80|0;l=g+96|0;m=g+112|0;n=b;b=e;e=f;f=n|0;bK(b|0,e|0,0)|0;bB(h|0,8,1,b|0)|0;dm(n,24320,(o=i,i=i+8|0,c[o>>2]=h,o)|0);i=o;jd(b)|0;c[f+128>>2]=(jd(b)|0)&65535;c[f+140>>2]=16;c[f+144>>2]=c[f+140>>2];c[f+136>>2]=4;c[f+132>>2]=ac(c[f+128>>2]|0,c[f+136>>2]|0)|0;p=n+1280|0;c[p>>2]=c[p>>2]|8192;jd(b)|0;p=0;while(1){if((p|0)>=128){break}a[f+952+p|0]=ja(b)|0;p=p+1|0}c[f+156>>2]=(ja(b)|0)&255;ja(b)|0;jd(b)|0;c[f+176>>2]=pT(764,c[f+140>>2]|0)|0;if((c[f+144>>2]|0)!=0){c[f+180>>2]=pT(52,c[f+144>>2]|0)|0}p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}q=pT(64,1)|0;c[(c[f+176>>2]|0)+(p*764|0)+756>>2]=q;bB(h|0,8,1,b|0)|0;dh((c[f+176>>2]|0)+(p*764|0)|0,h|0,8)|0;p=p+1|0}jd(b)|0;p=0;while(1){if((p|0)>=5){break}jd(b)|0;p=p+1|0}p=0;while(1){if((p|0)>=5){break}jd(b)|0;p=p+1|0}p=0;while(1){if((p|0)>=5){break}jd(b)|0;p=p+1|0}c[f+172>>2]=pT(4,c[f+132>>2]|0)|0;c[f+168>>2]=pT(4,(c[f+128>>2]|0)+1|0)|0;p=0;while(1){if((p|0)>=(c[f+128>>2]|0)){break}h=pT(1,((c[f+136>>2]|0)-1<<2)+8|0)|0;c[(c[f+168>>2]|0)+(p<<2)>>2]=h;c[c[(c[f+168>>2]|0)+(p<<2)>>2]>>2]=64;h=0;while(1){if((h|0)>=(c[f+136>>2]|0)){break}q=(ac(p,c[f+136>>2]|0)|0)+h|0;c[(c[(c[f+168>>2]|0)+(p<<2)>>2]|0)+4+(h<<2)>>2]=q;r=pT(((c[c[(c[f+168>>2]|0)+(p<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[f+172>>2]|0)+(q<<2)>>2]=r;c[c[(c[f+172>>2]|0)+(q<<2)>>2]>>2]=c[c[(c[f+168>>2]|0)+(p<<2)>>2]>>2];h=h+1|0}h=0;while(1){if((h|0)>=(c[c[(c[f+168>>2]|0)+(p<<2)>>2]>>2]|0)){break}q=0;while(1){if((q|0)>=(c[f+136>>2]|0)){break}r=(c[(c[n+172>>2]|0)+(c[(c[(c[n+168>>2]|0)+(p<<2)>>2]|0)+4+(q<<2)>>2]<<2)>>2]|0)+4+(h<<3)|0;s=(ja(b)|0)&255;if((s|0)!=0){a[r|0]=((s>>4)*12|0)&255;t=r|0;a[t]=(d[t]|0)+((s&15)+36)&255}s=(ja(b)|0)&255;a[r+1|0]=s>>4&255;if((a[r+1|0]|0)!=0){t=r+1|0;a[t]=(d[t]|0)+1&255}t=s&15;s=t;if((t|0)!=0){if((s|0)==13){a[r+3|0]=13;a[r+4|0]=0}else{r=s;bf(29328,(o=i,i=i+8|0,c[o>>2]=r,o)|0)|0;i=o}}q=q+1|0}h=h+1|0}p=p+1|0}o=aT(b|0)|0;jl(b)|0;p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}h=((ja(b)|0)&255|0)/2|0;c[c[(c[f+176>>2]|0)+(p*764|0)+756>>2]>>2]=h;c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+8>>2]=128;a[k+p|0]=ja(b)|0;a[l+p|0]=ja(b)|0;a[m+p|0]=ja(b)|0;p=p+1|0}p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}c[j+(p<<2)>>2]=jl(b)|0;m=jl(b)|0;c[(c[f+180>>2]|0)+(p*52|0)+32>>2]=m;p=p+1|0}jl(b)|0;jl(b)|0;jl(b)|0;jl(b)|0;p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}c[(c[f+176>>2]|0)+(p*764|0)+36>>2]=((c[(c[f+180>>2]|0)+(p*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[f+180>>2]|0)+(p*52|0)+36>>2]=0;c[(c[f+180>>2]|0)+(p*52|0)+40>>2]=0;c[(c[f+180>>2]|0)+(p*52|0)+44>>2]=(c[(c[f+180>>2]|0)+(p*52|0)+40>>2]|0)>0?2:0;c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+16>>2]=0;c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]=p;p=p+1|0}p=0;while(1){if((p|0)>=(c[f+140>>2]|0)){break}bK(b|0,e+o+(c[j+(p<<2)>>2]|0)|0,0)|0;gQ(n,b,2,(c[f+180>>2]|0)+((c[(c[(c[f+176>>2]|0)+(p*764|0)+756>>2]|0)+40>>2]|0)*52|0)|0,0)|0;p=p+1|0}i=g;return 0}function hk(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0;d=i;i=i+16|0;e=d|0;f=b;b=c;L4564:do{if((bB(e|0,1,15,f|0)|0)>>>0<15){g=-1}else{if((p6(e|0,23896,14)|0)!=0){g=-1;break}do{if((a[e+14|0]|0)>=48){if((a[e+14|0]|0)>52){break}di(f,b,32);g=0;break L4564}}while(0);g=-1}}while(0);i=d;return g|0}function hl(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;j=i;i=i+408|0;k=j|0;l=j+48|0;m=j+312|0;n=j+384|0;o=j+392|0;p=f;f=g;g=p|0;q=o;c[q>>2]=c[352];c[q+4>>2]=c[353];c[q+8>>2]=c[354];c[q+12>>2]=c[355];q=0;r=0;bK(f|0,h|0,0)|0;bB(k|0,15,1,f|0)|0;bB(k+15|0,32,1,f|0)|0;a[k+47|0]=ja(f)|0;h=(d[k+14|0]|0)-48|0;p5(g|0,k+15|0,32)|0;a[k+15|0]=0;dm(p,24576,(s=i,i=i+16|0,c[s>>2]=c[o+(h-1<<2)>>2],c[s+8>>2]=h,s)|0);i=s;bK(f|0,d[k+47|0]<<5|0,1)|0;k=(ja(f)|0)&255;c[g+144>>2]=k;c[g+140>>2]=k;c[g+176>>2]=pT(764,c[g+140>>2]|0)|0;if((c[g+144>>2]|0)!=0){c[g+180>>2]=pT(52,c[g+144>>2]|0)|0}k=0;while(1){if((k|0)>=(c[g+140>>2]|0)){break}s=pT(64,1)|0;c[(c[g+176>>2]|0)+(k*764|0)+756>>2]=s;bB(m|0,32,1,f|0)|0;bB(m+32|0,12,1,f|0)|0;c[m+44>>2]=jg(f)|0;c[m+48>>2]=jg(f)|0;c[m+52>>2]=jg(f)|0;c[m+56>>2]=jg(f)|0;a[m+60|0]=ja(f)|0;a[m+61|0]=ja(f)|0;b[m+62>>1]=jc(f)|0;if((h|0)<4){t=0}else{t=(jc(f)|0)&65535}b[m+64>>1]=t&65535;if((h|0)>3){s=m+64|0;b[s>>1]=(e[s>>1]^e[m+62>>1])&65535;s=m+62|0;b[s>>1]=(e[s>>1]^e[m+64>>1])&65535;s=m+64|0;b[s>>1]=(e[s>>1]^e[m+62>>1])&65535}c[(c[g+180>>2]|0)+(k*52|0)+32>>2]=(c[m+56>>2]|0)-(c[m+52>>2]|0);c[(c[g+176>>2]|0)+(k*764|0)+36>>2]=((c[(c[g+180>>2]|0)+(k*52|0)+32>>2]|0)!=0^1^1)&1;c[(c[g+180>>2]|0)+(k*52|0)+36>>2]=c[m+44>>2];c[(c[g+180>>2]|0)+(k*52|0)+40>>2]=c[m+48>>2];s=d[m+61|0]|0;if((s|0)==20|(s|0)==4){c[(c[g+180>>2]|0)+(k*52|0)+44>>2]=1}else if((s|0)==8){c[(c[g+180>>2]|0)+(k*52|0)+44>>2]=2}else if((s|0)==12){c[(c[g+180>>2]|0)+(k*52|0)+44>>2]=3}else if((s|0)==24){c[(c[g+180>>2]|0)+(k*52|0)+44>>2]=10}else if((s|0)==28){c[(c[g+180>>2]|0)+(k*52|0)+44>>2]=11}c[c[(c[g+176>>2]|0)+(k*764|0)+756>>2]>>2]=d[m+60|0]|0;c[(c[(c[g+176>>2]|0)+(k*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[g+176>>2]|0)+(k*764|0)+756>>2]|0)+40>>2]=k;dh((c[g+176>>2]|0)+(k*764|0)|0,m|0,24)|0;if((h|0)>3){hD(e[m+64>>1]|0,(c[(c[g+176>>2]|0)+(k*764|0)+756>>2]|0)+12|0,(c[(c[g+176>>2]|0)+(k*764|0)+756>>2]|0)+16|0)}k=k+1|0}bB(l|0,256,1,f|0)|0;a[l+256|0]=ja(f)|0;a[l+257|0]=ja(f)|0;k=0;while(1){if((k|0)>=256){break}if((d[l+k|0]|0)==255){m=3753;break}a[g+952+k|0]=a[l+k|0]|0;k=k+1|0}c[g+156>>2]=k;c[g+136>>2]=(d[l+256|0]|0)+1;c[g+128>>2]=(d[l+257|0]|0)+1;c[g+148>>2]=6;c[g+152>>2]=125;c[g+132>>2]=ac(c[g+136>>2]|0,c[g+128>>2]|0)|0;k=0;while(1){if((k|0)>=(c[g+136>>2]|0)){break}if((h|0)>=3){u=ja(f)|0;c[g+184+(k*12|0)>>2]=((u&255)*255|0|0)/15|0}else{c[g+184+(k*12|0)>>2]=(((k+1|0)/2|0|0)%2|0)*255|0}k=k+1|0}c[g+172>>2]=pT(4,c[g+132>>2]|0)|0;c[g+168>>2]=pT(4,(c[g+128>>2]|0)+1|0)|0;k=0;while(1){if((k|0)>=(c[g+128>>2]|0)){break}h=pT(1,((c[g+136>>2]|0)-1<<2)+8|0)|0;c[(c[g+168>>2]|0)+(k<<2)>>2]=h;c[c[(c[g+168>>2]|0)+(k<<2)>>2]>>2]=64;h=0;while(1){if((h|0)>=(c[g+136>>2]|0)){break}l=(ac(k,c[g+136>>2]|0)|0)+h|0;c[(c[(c[g+168>>2]|0)+(k<<2)>>2]|0)+4+(h<<2)>>2]=l;m=pT(((c[c[(c[g+168>>2]|0)+(k<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[g+172>>2]|0)+(l<<2)>>2]=m;c[c[(c[g+172>>2]|0)+(l<<2)>>2]>>2]=c[c[(c[g+168>>2]|0)+(k<<2)>>2]>>2];h=h+1|0}k=k+1|0}k=0;while(1){if((k|0)>=(c[g+136>>2]|0)){break}h=0;while(1){if((h|0)>=(c[g+128>>2]<<6|0)){break}l=1;u=ja(f)|0;if((u&255|0)==252){l=(ja(f)|0)&255;u=ja(f)|0}bB(n|0,4,1,f|0)|0;if((l|0)==0){l=l+1|0}m=0;while(1){if((m|0)>=(l|0)){break}t=(c[(c[p+172>>2]|0)+(c[(c[(c[p+168>>2]|0)+(h>>6<<2)>>2]|0)+4+(k<<2)>>2]<<2)>>2]|0)+4+((h&63)<<3)|0;p7(t|0,0,8);if(u<<24>>24!=0){a[t|0]=(u&255)+36&255}a[t+1|0]=a[n|0]|0;a[t+3|0]=(a[n+1|0]&240)>>4&255;a[t+5|0]=d[n+1|0]&15;a[t+4|0]=a[n+3|0]|0;a[t+6|0]=a[n+2|0]|0;s=d[t+3|0]|0;if((s|0)==5|(s|0)==6){a[t+4|0]=0;a[t+3|0]=0}else if((s|0)==3){q=32}else if((s|0)==11){a[t+3|0]=8;o=t+4|0;a[o]=d[o]<<4&255}else if((s|0)==9){o=t+4|0;a[o]=d[o]<<2&255}else if((s|0)==0){if((a[t+4|0]|0)!=0){q=0}if((q|0)!=0){a[t+3|0]=3;q=q-1|0}}s=d[t+5|0]|0;if((s|0)==3){r=32}else if((s|0)==5|(s|0)==6){a[t+6|0]=0;a[t+5|0]=0}else if((s|0)==11){a[t+5|0]=8;o=t+6|0;a[o]=d[o]<<4&255}else if((s|0)==9){o=t+6|0;a[o]=d[o]<<2&255}else if((s|0)==0){if((a[t+6|0]|0)!=0){r=0}if((r|0)!=0){a[t+5|0]=3;r=r-1|0}}m=m+1|0;h=h+1|0}}k=k+1|0}k=0;while(1){if((k|0)>=(c[g+140>>2]|0)){break}if((c[(c[g+180>>2]|0)+(k*52|0)+32>>2]|0)!=0){r=p;q=f;n=(c[g+180>>2]|0)+(k*52|0)|0;gQ(r,q,0,n,0)|0}k=k+1|0}c[p+1268>>2]=256;i=j;return 0}function hm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;c=i;i=i+1504|0;b=c|0;d=-1;e=b|0;if((bB(b|0,1,1500,a|0)|0)>>>0<1500){f=-1;g=f;i=c;return g|0}a=jk(e)|0;if((a|0)!=-1048368482){f=-1;g=f;i=c;return g|0}b=0;while(1){if((b|0)>=1500){break}a=jk(e)|0;if((p6(e|0,29288,16)|0)==0){h=3832;break}if((a|0)==1229803597){h=3834;break}if((b|0)>44){if((a|0)==1396920909){h=3837;break}}if((b|0)>1080){if((a|0)==1294879534){h=3840;break}}b=b+1|0;e=e+1|0}if((h|0)==3834){d=b}else if((h|0)==3837){d=b-44|0}else if((h|0)==3832){d=b}else if((h|0)==3840){d=b-1080|0}if((d|0)<0){f=-1;g=f;i=c;return g|0}else{f=0;g=f;i=c;return g|0}return 0}function hn(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0;e=i;i=i+1504|0;f=e|0;g=a;a=b;b=f|0;bK(a|0,d|0,0)|0;bB(f|0,1,1500,a|0)|0;f=0;while(1){if((f|0)>=1500){h=3869;break}d=jk(b)|0;if((p6(b|0,29288,16)|0)==0){h=3858;break}if((d|0)==1229803597){h=3860;break}if((f|0)>44){if((d|0)==1396920909){h=3863;break}}if((f|0)>1080){if((d|0)==1294879534){h=3866;break}}f=f+1|0;b=b+1|0}if((h|0)==3866){j=b2[c[1656]&511](g,a,f-1080|0)|0;k=j;i=e;return k|0}else if((h|0)==3863){j=b2[c[848]&511](g,a,f-44|0)|0;k=j;i=e;return k|0}else if((h|0)==3860){j=b2[c[1810]&511](g,a,f)|0;k=j;i=e;return k|0}else if((h|0)==3858){j=b2[c[34]&511](g,a,f)|0;k=j;i=e;return k|0}else if((h|0)==3869){j=-1;k=j;i=e;return k|0}return 0}function ho(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;c=i;i=i+24|0;d=c|0;e=a;a=b;do{if((bB(d|0,1,17,e|0)|0)>>>0<17){f=-1}else{if((p6(d|0,22920,17)|0)!=0){f=-1;break}else{di(e,a,20);f=0;break}}}while(0);i=c;return f|0}function hp(f,g,h){f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;j=i;i=i+1264|0;k=j|0;l=j+336|0;m=j+352|0;n=j+392|0;o=j+600|0;p=j+1240|0;q=f;f=g;g=h;h=q|0;r=0;bK(f|0,g|0,0)|0;bB(k|0,17,1,f|0)|0;bB(k+17|0,20,1,f|0)|0;ja(f)|0;bB(k+38|0,20,1,f|0)|0;b[k+58>>1]=jc(f)|0;c[k+60>>2]=jg(f)|0;b[k+64>>1]=jc(f)|0;b[k+66>>1]=jc(f)|0;b[k+68>>1]=jc(f)|0;b[k+70>>1]=jc(f)|0;b[k+72>>1]=jc(f)|0;b[k+74>>1]=jc(f)|0;b[k+76>>1]=jc(f)|0;b[k+78>>1]=jc(f)|0;bB(k+80|0,(c[k+60>>2]|0)-20|0,1,f|0)|0;p5(h|0,k+17|0,20)|0;c[h+156>>2]=e[k+64>>1]|0;c[h+160>>2]=e[k+66>>1]|0;c[h+136>>2]=e[k+68>>1]|0;c[h+128>>2]=e[k+70>>1]|0;c[h+132>>2]=(ac(c[h+136>>2]|0,c[h+128>>2]|0)|0)+1;c[h+140>>2]=e[k+72>>1]|0;c[h+148>>2]=e[k+76>>1]|0;c[h+152>>2]=e[k+78>>1]|0;s=q+1280|0;c[s>>2]=c[s>>2]|((b[k+74>>1]&1|0)!=0?4096:0);s=h+952|0;t=k+80|0;u=c[h+156>>2]|0;p3(s|0,t|0,u)|0;a[p+20|0]=0;aM(p|0,20,29272,(u=i,i=i+8|0,c[u>>2]=k+38,u)|0)|0;i=u;t=20;while(1){if((t|0)<0){break}if((a[p+t|0]|0)==32){a[p+t|0]=0}if((a[p+t|0]|0)!=0){v=3889;break}t=t-1|0}if((c[k+60>>2]|0)==275){s=p|0;qc(s|0,27608)|0}else{if((a[p|0]|0)==0){s=p|0;qc(s|0,26440)|0}}if((aL(p|0,25408,19)|0)==0){if((c[h+152>>2]|0)<=10){c[h+152>>2]=((53-(c[h+152>>2]<<1)|0)*125|0|0)/33|0}}if((aL(p|0,24552,18)|0)==0){s=p|0;qc(s|0,23872)|0}s=e[k+58>>1]>>8;w=b[k+58>>1]&255;aM(h+64|0,64,23480,(u=i,i=i+24|0,c[u>>2]=p,c[u+8>>2]=s,c[u+16>>2]=w,u)|0)|0;i=u;bK(f|0,g+(c[k+60>>2]|0)+60|0,0)|0;if((e[k+58>>1]|0)<=259){v=3984}while(1){if((v|0)==3984){v=0;c[h+144>>2]=1024;c[h+176>>2]=pT(764,c[h+140>>2]|0)|0;if((c[h+144>>2]|0)!=0){c[h+180>>2]=pT(52,c[h+144>>2]|0)|0}t=0;while(1){if((t|0)>=(c[h+140>>2]|0)){break}c[m>>2]=jg(f)|0;if((bH(f|0)|0)!=0){v=3993;break}bB(m+4|0,22,1,f|0)|0;a[m+26|0]=ja(f)|0;b[m+28>>1]=jc(f)|0;c[m+32>>2]=jg(f)|0;if((e[m+28>>1]|0)>16){v=3997;break}if((e[m+28>>1]|0)>0){if((c[m+32>>2]|0)>>>0>256){v=3997;break}}dh((c[h+176>>2]|0)+(t*764|0)|0,m+4|0,22)|0;c[(c[h+176>>2]|0)+(t*764|0)+36>>2]=e[m+28>>1]|0;if((c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)>16){c[(c[h+176>>2]|0)+(t*764|0)+36>>2]=16}if((c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)!=0){g=pT(64,c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)|0;c[(c[h+176>>2]|0)+(t*764|0)+756>>2]=g;if((c[m>>2]|0)==38){g=f;ja(g)|0;g=f;jg(g)|0;p7(n|0,0,208)}else{g=n|0;u=f;bB(g|0,96,1,u|0)|0;x=0;while(1){if((x|0)>=24){break}b[n+96+(x<<1)>>1]=jc(f)|0;x=x+1|0}x=0;while(1){if((x|0)>=24){break}b[n+144+(x<<1)>>1]=jc(f)|0;x=x+1|0}a[n+192|0]=ja(f)|0;a[n+193|0]=ja(f)|0;a[n+194|0]=ja(f)|0;a[n+195|0]=ja(f)|0;a[n+196|0]=ja(f)|0;a[n+197|0]=ja(f)|0;a[n+198|0]=ja(f)|0;a[n+199|0]=ja(f)|0;a[n+200|0]=ja(f)|0;a[n+201|0]=ja(f)|0;a[n+202|0]=ja(f)|0;a[n+203|0]=ja(f)|0;a[n+204|0]=ja(f)|0;a[n+205|0]=ja(f)|0;b[n+206>>1]=jc(f)|0;bK(f|0,(c[m>>2]|0)-33-208|0,1)|0;c[(c[h+176>>2]|0)+(t*764|0)+40>>2]=e[n+206>>1]|0;c[(c[h+176>>2]|0)+(t*764|0)+48>>2]=d[n+192|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+56>>2]=d[n+194|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+64>>2]=d[n+195|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+68>>2]=d[n+196|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+44>>2]=d[n+200|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+204>>2]=d[n+193|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+212>>2]=d[n+197|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+220>>2]=d[n+198|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+224>>2]=d[n+199|0]|0;c[(c[h+176>>2]|0)+(t*764|0)+200>>2]=d[n+201|0]|0;if((c[(c[h+176>>2]|0)+(t*764|0)+48>>2]|0)<=0){v=4015}else{if((c[(c[h+176>>2]|0)+(t*764|0)+48>>2]|0)>32){v=4015}}if((v|0)==4015){v=0;u=(c[h+176>>2]|0)+(t*764|0)+44|0;c[u>>2]=c[u>>2]&-2}if((c[(c[h+176>>2]|0)+(t*764|0)+204>>2]|0)<=0){v=4018}else{if((c[(c[h+176>>2]|0)+(t*764|0)+204>>2]|0)>32){v=4018}}if((v|0)==4018){v=0;u=(c[h+176>>2]|0)+(t*764|0)+200|0;c[u>>2]=c[u>>2]&-2}u=(c[h+176>>2]|0)+(t*764|0)+72|0;g=n+96|0;w=c[(c[h+176>>2]|0)+(t*764|0)+48>>2]<<2;p3(u|0,g|0,w)|0;w=(c[h+176>>2]|0)+(t*764|0)+228|0;g=n+144|0;u=c[(c[h+176>>2]|0)+(t*764|0)+204>>2]<<2;p3(w|0,g|0,u)|0;x=12;while(1){if((x|0)>=108){break}a[(c[h+176>>2]|0)+(t*764|0)+512+(x<<1)|0]=a[n+(x-12)|0]|0;if((d[(c[h+176>>2]|0)+(t*764|0)+512+(x<<1)|0]|0)>=(c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)){a[(c[h+176>>2]|0)+(t*764|0)+512+(x<<1)|0]=-1}x=x+1|0}}x=0;while(1){if((x|0)>=(c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)){break}c[o+(x*40|0)>>2]=jg(f)|0;c[o+(x*40|0)+4>>2]=jg(f)|0;c[o+(x*40|0)+8>>2]=jg(f)|0;a[o+(x*40|0)+12|0]=ja(f)|0;a[o+(x*40|0)+13|0]=jb(f)|0;a[o+(x*40|0)+14|0]=ja(f)|0;a[o+(x*40|0)+15|0]=ja(f)|0;a[o+(x*40|0)+16|0]=jb(f)|0;a[o+(x*40|0)+17|0]=ja(f)|0;bB(o+(x*40|0)+18|0,22,1,f|0)|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)>>2]=d[o+(x*40|0)+12|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+8>>2]=d[o+(x*40|0)+15|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+12>>2]=a[o+(x*40|0)+16|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+16>>2]=a[o+(x*40|0)+13|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+20>>2]=d[n+202|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+24>>2]=d[n+204|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+28>>2]=d[n+205|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+32>>2]=d[n+203|0]|0;c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+40>>2]=r;if((r|0)<1024){u=(c[h+180>>2]|0)+(r*52|0)|0;g=o+(x*40|0)+18|0;dh(u,g,22)|0;c[(c[h+180>>2]|0)+(r*52|0)+32>>2]=c[o+(x*40|0)>>2];c[(c[h+180>>2]|0)+(r*52|0)+36>>2]=c[o+(x*40|0)+4>>2];c[(c[h+180>>2]|0)+(r*52|0)+40>>2]=(c[o+(x*40|0)+4>>2]|0)+(c[o+(x*40|0)+8>>2]|0);c[(c[h+180>>2]|0)+(r*52|0)+44>>2]=0;if((a[o+(x*40|0)+14|0]&16|0)!=0){g=(c[h+180>>2]|0)+(r*52|0)+44|0;c[g>>2]=c[g>>2]|1;g=(c[h+180>>2]|0)+(r*52|0)+32|0;c[g>>2]=c[g>>2]>>1;g=(c[h+180>>2]|0)+(r*52|0)+36|0;c[g>>2]=c[g>>2]>>1;g=(c[h+180>>2]|0)+(r*52|0)+40|0;c[g>>2]=c[g>>2]>>1}g=(c[h+180>>2]|0)+(r*52|0)+44|0;c[g>>2]=c[g>>2]|((a[o+(x*40|0)+14|0]&1|0)!=0?2:0);g=(c[h+180>>2]|0)+(r*52|0)+44|0;c[g>>2]=c[g>>2]|((a[o+(x*40|0)+14|0]&2|0)!=0?6:0)}x=x+1|0;r=r+1|0}x=0;while(1){if((x|0)>=(c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)){break}if((r|0)<1024){if((e[k+58>>1]|0)>259){g=q;u=f;w=(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+40>>2]|0)*52|0)|0;gQ(g,u,1,w,0)|0}}x=x+1|0}}else{bK(f|0,(c[m>>2]|0)-33|0,1)|0}t=t+1|0}if((v|0)==3997){v=0;c[h+140>>2]=t}else if((v|0)==3993){v=0;c[h+140>>2]=t}c[h+144>>2]=r;c[h+180>>2]=pV(c[h+180>>2]|0,(c[h+144>>2]|0)*52|0)|0;if((e[k+58>>1]|0)>259){v=4050;break}}c[h+172>>2]=pT(4,c[h+132>>2]|0)|0;c[h+168>>2]=pT(4,(c[h+128>>2]|0)+1|0)|0;t=0;while(1){if((t|0)>=(c[h+128>>2]|0)){break}c[l>>2]=jg(f)|0;a[l+4|0]=ja(f)|0;if((e[k+58>>1]|0)>258){y=(jc(f)|0)&65535}else{y=((ja(f)|0)&255)+1|0}b[l+6>>1]=y&65535;b[l+8>>1]=jc(f)|0;w=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(t<<2)>>2]=w;w=e[l+6>>1]|0;c[c[(c[h+168>>2]|0)+(t<<2)>>2]>>2]=w;u=w;if((w|0)==0){c[c[(c[h+168>>2]|0)+(t<<2)>>2]>>2]=256;u=256}w=0;while(1){if((w|0)>=(c[h+136>>2]|0)){break}g=(ac(t,c[h+136>>2]|0)|0)+w|0;c[(c[(c[h+168>>2]|0)+(t<<2)>>2]|0)+4+(w<<2)>>2]=g;s=pT(((c[c[(c[h+168>>2]|0)+(t<<2)>>2]>>2]|0)-1<<3)+12|0,1)|0;c[(c[h+172>>2]|0)+(g<<2)>>2]=s;c[c[(c[h+172>>2]|0)+(g<<2)>>2]>>2]=c[c[(c[h+168>>2]|0)+(t<<2)>>2]>>2];w=w+1|0}if((b[l+8>>1]|0)!=0){w=pT(1,e[l+8>>1]|0)|0;g=w;s=w;w=g;p=e[l+8>>1]|0;z=f;bB(w|0,1,p|0,z|0)|0;x=0;while(1){if((x|0)>=(ac(c[h+136>>2]|0,u)|0)){break}if((s-g|0)>=(e[l+8>>1]|0)){v=3935;break}z=(c[(c[q+172>>2]|0)+(c[(c[(c[q+168>>2]|0)+(t<<2)>>2]|0)+4+(((x|0)%(c[h+136>>2]|0)|0)<<2)>>2]<<2)>>2]|0)+4+(((x|0)/(c[h+136>>2]|0)|0)<<3)|0;p=s;s=p+1|0;w=a[p]|0;p=w;if((w&128|0)!=0){if((p&1|0)!=0){w=s;s=w+1|0;a[z|0]=a[w]|0}if((p&2|0)!=0){w=s;s=w+1|0;a[z+1|0]=a[w]|0}if((p&4|0)!=0){w=s;s=w+1|0;a[z+2|0]=a[w]|0}if((p&8|0)!=0){w=s;s=w+1|0;a[z+3|0]=a[w]|0}if((p&16|0)!=0){w=s;s=w+1|0;a[z+4|0]=a[w]|0}}else{a[z|0]=p;p=s;s=p+1|0;a[z+1|0]=a[p]|0;p=s;s=p+1|0;a[z+2|0]=a[p]|0;p=s;s=p+1|0;a[z+3|0]=a[p]|0;p=s;s=p+1|0;a[z+4|0]=a[p]|0}if((d[z|0]|0)==97){a[z|0]=-127}else{if((d[z|0]|0)>0){p=z|0;a[p]=(d[p]|0)+12&255}}L4961:do{if((a[z+2|0]|0)!=0){do{if((d[z+2|0]|0)>=16){if((d[z+2|0]|0)>80){break}p=z+2|0;a[p]=(d[p]|0)-15&255;break L4961}}while(0);p=d[z+2|0]>>4;if((p|0)==14){a[z+5|0]=25;a[z+6|0]=(d[z+2|0]|0)-224&255}else if((p|0)==8){a[z+5|0]=14;a[z+6|0]=(176|(d[z+2|0]|0)-128)&255}else if((p|0)==9){a[z+5|0]=14;a[z+6|0]=(160|(d[z+2|0]|0)-144)&255}else if((p|0)==15){a[z+5|0]=3;a[z+6|0]=(d[z+2|0]|0)-240<<4&255}else if((p|0)==7){a[z+5|0]=-92;a[z+6|0]=(d[z+2|0]|0)-112<<4&255}else if((p|0)==10){a[z+5|0]=4;a[z+6|0]=(d[z+2|0]|0)-160<<4&255}else if((p|0)==6){a[z+5|0]=-92;a[z+6|0]=(d[z+2|0]|0)-96&255}else if((p|0)==13){a[z+5|0]=25;a[z+6|0]=(d[z+2|0]|0)-208<<4&255}else if((p|0)==11){a[z+5|0]=4;a[z+6|0]=(d[z+2|0]|0)-176&255}else if((p|0)==12){a[z+5|0]=8;a[z+6|0]=((d[z+2|0]|0)-192<<4)+8&255}a[z+2|0]=0}}while(0);x=x+1|0}if((v|0)==3935){v=0}pU(g)}t=t+1|0}s=pT(1,((c[h+136>>2]|0)-1<<2)+8|0)|0;c[(c[h+168>>2]|0)+(t<<2)>>2]=s;c[c[(c[h+168>>2]|0)+(t<<2)>>2]>>2]=64;s=pT(1,524)|0;u=ac(t,c[h+136>>2]|0)|0;c[(c[h+172>>2]|0)+(u<<2)>>2]=s;s=ac(t,c[h+136>>2]|0)|0;c[c[(c[h+172>>2]|0)+(s<<2)>>2]>>2]=64;x=0;while(1){if((x|0)>=(c[h+136>>2]|0)){break}s=ac(t,c[h+136>>2]|0)|0;c[(c[(c[h+168>>2]|0)+(t<<2)>>2]|0)+4+(x<<2)>>2]=s;x=x+1|0}s=h+128|0;c[s>>2]=(c[s>>2]|0)+1;if((e[k+58>>1]|0)<=259){v=3982;break}v=3984}if((e[k+58>>1]|0)<=259){t=0;while(1){if((t|0)>=(c[h+140>>2]|0)){break}x=0;while(1){if((x|0)>=(c[(c[h+176>>2]|0)+(t*764|0)+36>>2]|0)){break}gQ(q,f,1,(c[h+180>>2]|0)+((c[(c[(c[h+176>>2]|0)+(t*764|0)+756>>2]|0)+(x<<6)+40>>2]|0)*52|0)|0,0)|0;x=x+1|0}t=t+1|0}}t=0;while(1){if((t|0)>=(c[h+136>>2]|0)){break}c[h+184+(t*12|0)>>2]=128;t=t+1|0}t=q+1280|0;c[t>>2]=c[t>>2]|32;c[q+1284>>2]=1;i=j;return 0}function hq(a){a=a|0;var b=0;b=a;a=b+16|0;c[a>>2]=0;c[a+4>>2]=0;c[b>>2]=1732584193;c[b+4>>2]=-271733879;c[b+8>>2]=-1732584194;c[b+12>>2]=271733878;return}function hr(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0;e=a;a=b;b=d;d=e+16|0;f=((c[d>>2]|0)>>>3|c[d+4>>2]<<29)&63;d=64-f|0;g=b;h=e+16|0;i=qf(c[h>>2]|0,c[h+4>>2]|0,g<<3|0>>>29,0<<3|g>>>29)|0;c[h>>2]=i;c[h+4>>2]=G;if(b>>>0>=d>>>0){if((f|0)!=0){h=e+24+f|0;i=a;g=d;p3(h|0,i|0,g)|0;hs(e|0,e+24|0);a=a+d|0;b=b-d|0;f=0}while(1){if(b>>>0<64){break}hs(e|0,a);a=a+64|0;b=b-64|0}}if((b|0)==0){return}d=e+24+f|0;f=a;a=b;p3(d|0,f|0,a)|0;return}function hs(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0;d=i;i=i+64|0;e=d|0;f=a;a=e;g=b;p3(a|0,g|0,64)|0;g=c[f>>2]|0;a=c[f+4>>2]|0;b=c[f+8>>2]|0;h=c[f+12>>2]|0;g=g+((h^a&(b^h))+(c[e>>2]|0)-680876936)|0;g=g<<7|g>>>25;g=g+a|0;h=h+((b^g&(a^b))+(c[e+4>>2]|0)-389564586)|0;h=h<<12|h>>>20;h=h+g|0;b=b+((a^h&(g^a))+(c[e+8>>2]|0)+606105819)|0;b=b<<17|b>>>15;b=b+h|0;a=a+((g^b&(h^g))+(c[e+12>>2]|0)-1044525330)|0;a=a<<22|a>>>10;a=a+b|0;g=g+((h^a&(b^h))+(c[e+16>>2]|0)-176418897)|0;g=g<<7|g>>>25;g=g+a|0;h=h+((b^g&(a^b))+(c[e+20>>2]|0)+1200080426)|0;h=h<<12|h>>>20;h=h+g|0;b=b+((a^h&(g^a))+(c[e+24>>2]|0)-1473231341)|0;b=b<<17|b>>>15;b=b+h|0;a=a+((g^b&(h^g))+(c[e+28>>2]|0)-45705983)|0;a=a<<22|a>>>10;a=a+b|0;g=g+((h^a&(b^h))+(c[e+32>>2]|0)+1770035416)|0;g=g<<7|g>>>25;g=g+a|0;h=h+((b^g&(a^b))+(c[e+36>>2]|0)-1958414417)|0;h=h<<12|h>>>20;h=h+g|0;b=b+((a^h&(g^a))+(c[e+40>>2]|0)-42063)|0;b=b<<17|b>>>15;b=b+h|0;a=a+((g^b&(h^g))+(c[e+44>>2]|0)-1990404162)|0;a=a<<22|a>>>10;a=a+b|0;g=g+((h^a&(b^h))+(c[e+48>>2]|0)+1804603682)|0;g=g<<7|g>>>25;g=g+a|0;h=h+((b^g&(a^b))+(c[e+52>>2]|0)-40341101)|0;h=h<<12|h>>>20;h=h+g|0;b=b+((a^h&(g^a))+(c[e+56>>2]|0)-1502002290)|0;b=b<<17|b>>>15;b=b+h|0;a=a+((g^b&(h^g))+(c[e+60>>2]|0)+1236535329)|0;a=a<<22|a>>>10;a=a+b|0;g=g+((b^h&(a^b))+(c[e+4>>2]|0)-165796510)|0;g=g<<5|g>>>27;g=g+a|0;h=h+((a^b&(g^a))+(c[e+24>>2]|0)-1069501632)|0;h=h<<9|h>>>23;h=h+g|0;b=b+((g^a&(h^g))+(c[e+44>>2]|0)+643717713)|0;b=b<<14|b>>>18;b=b+h|0;a=a+((h^g&(b^h))+(c[e>>2]|0)-373897302)|0;a=a<<20|a>>>12;a=a+b|0;g=g+((b^h&(a^b))+(c[e+20>>2]|0)-701558691)|0;g=g<<5|g>>>27;g=g+a|0;h=h+((a^b&(g^a))+(c[e+40>>2]|0)+38016083)|0;h=h<<9|h>>>23;h=h+g|0;b=b+((g^a&(h^g))+(c[e+60>>2]|0)-660478335)|0;b=b<<14|b>>>18;b=b+h|0;a=a+((h^g&(b^h))+(c[e+16>>2]|0)-405537848)|0;a=a<<20|a>>>12;a=a+b|0;g=g+((b^h&(a^b))+(c[e+36>>2]|0)+568446438)|0;g=g<<5|g>>>27;g=g+a|0;h=h+((a^b&(g^a))+(c[e+56>>2]|0)-1019803690)|0;h=h<<9|h>>>23;h=h+g|0;b=b+((g^a&(h^g))+(c[e+12>>2]|0)-187363961)|0;b=b<<14|b>>>18;b=b+h|0;a=a+((h^g&(b^h))+(c[e+32>>2]|0)+1163531501)|0;a=a<<20|a>>>12;a=a+b|0;g=g+((b^h&(a^b))+(c[e+52>>2]|0)-1444681467)|0;g=g<<5|g>>>27;g=g+a|0;h=h+((a^b&(g^a))+(c[e+8>>2]|0)-51403784)|0;h=h<<9|h>>>23;h=h+g|0;b=b+((g^a&(h^g))+(c[e+28>>2]|0)+1735328473)|0;b=b<<14|b>>>18;b=b+h|0;a=a+((h^g&(b^h))+(c[e+48>>2]|0)-1926607734)|0;a=a<<20|a>>>12;a=a+b|0;g=g+((a^b^h)+(c[e+20>>2]|0)-378558)|0;g=g<<4|g>>>28;g=g+a|0;h=h+((g^a^b)+(c[e+32>>2]|0)-2022574463)|0;h=h<<11|h>>>21;h=h+g|0;b=b+((h^g^a)+(c[e+44>>2]|0)+1839030562)|0;b=b<<16|b>>>16;b=b+h|0;a=a+((b^h^g)+(c[e+56>>2]|0)-35309556)|0;a=a<<23|a>>>9;a=a+b|0;g=g+((a^b^h)+(c[e+4>>2]|0)-1530992060)|0;g=g<<4|g>>>28;g=g+a|0;h=h+((g^a^b)+(c[e+16>>2]|0)+1272893353)|0;h=h<<11|h>>>21;h=h+g|0;b=b+((h^g^a)+(c[e+28>>2]|0)-155497632)|0;b=b<<16|b>>>16;b=b+h|0;a=a+((b^h^g)+(c[e+40>>2]|0)-1094730640)|0;a=a<<23|a>>>9;a=a+b|0;g=g+((a^b^h)+(c[e+52>>2]|0)+681279174)|0;g=g<<4|g>>>28;g=g+a|0;h=h+((g^a^b)+(c[e>>2]|0)-358537222)|0;h=h<<11|h>>>21;h=h+g|0;b=b+((h^g^a)+(c[e+12>>2]|0)-722521979)|0;b=b<<16|b>>>16;b=b+h|0;a=a+((b^h^g)+(c[e+24>>2]|0)+76029189)|0;a=a<<23|a>>>9;a=a+b|0;g=g+((a^b^h)+(c[e+36>>2]|0)-640364487)|0;g=g<<4|g>>>28;g=g+a|0;h=h+((g^a^b)+(c[e+48>>2]|0)-421815835)|0;h=h<<11|h>>>21;h=h+g|0;b=b+((h^g^a)+(c[e+60>>2]|0)+530742520)|0;b=b<<16|b>>>16;b=b+h|0;a=a+((b^h^g)+(c[e+8>>2]|0)-995338651)|0;a=a<<23|a>>>9;a=a+b|0;g=g+((b^(a|~h))+(c[e>>2]|0)-198630844)|0;g=g<<6|g>>>26;g=g+a|0;h=h+((a^(g|~b))+(c[e+28>>2]|0)+1126891415)|0;h=h<<10|h>>>22;h=h+g|0;b=b+((g^(h|~a))+(c[e+56>>2]|0)-1416354905)|0;b=b<<15|b>>>17;b=b+h|0;a=a+((h^(b|~g))+(c[e+20>>2]|0)-57434055)|0;a=a<<21|a>>>11;a=a+b|0;g=g+((b^(a|~h))+(c[e+48>>2]|0)+1700485571)|0;g=g<<6|g>>>26;g=g+a|0;h=h+((a^(g|~b))+(c[e+12>>2]|0)-1894986606)|0;h=h<<10|h>>>22;h=h+g|0;b=b+((g^(h|~a))+(c[e+40>>2]|0)-1051523)|0;b=b<<15|b>>>17;b=b+h|0;a=a+((h^(b|~g))+(c[e+4>>2]|0)-2054922799)|0;a=a<<21|a>>>11;a=a+b|0;g=g+((b^(a|~h))+(c[e+32>>2]|0)+1873313359)|0;g=g<<6|g>>>26;g=g+a|0;h=h+((a^(g|~b))+(c[e+60>>2]|0)-30611744)|0;h=h<<10|h>>>22;h=h+g|0;b=b+((g^(h|~a))+(c[e+24>>2]|0)-1560198380)|0;b=b<<15|b>>>17;b=b+h|0;a=a+((h^(b|~g))+(c[e+52>>2]|0)+1309151649)|0;a=a<<21|a>>>11;a=a+b|0;g=g+((b^(a|~h))+(c[e+16>>2]|0)-145523070)|0;g=g<<6|g>>>26;g=g+a|0;h=h+((a^(g|~b))+(c[e+44>>2]|0)-1120210379)|0;h=h<<10|h>>>22;h=h+g|0;b=b+((g^(h|~a))+(c[e+8>>2]|0)+718787259)|0;b=b<<15|b>>>17;b=b+h|0;a=a+((h^(b|~g))+(c[e+36>>2]|0)-343485551)|0;a=a<<21|a>>>11;a=a+b|0;e=f|0;c[e>>2]=(c[e>>2]|0)+g;g=f+4|0;c[g>>2]=(c[g>>2]|0)+a;a=f+8|0;c[a>>2]=(c[a>>2]|0)+b;b=f+12|0;c[b>>2]=(c[b>>2]|0)+h;i=d;return}function ht(a){a=a|0;var b=0,d=0,e=0,f=0,g=0,h=0;b=a;a=5920;if((b|0)==0){d=0;e=d;return e|0}f=24;while(1){if((b|0)>3627){break}f=f+12|0;b=b<<1}while(1){if((b|0)<=(c[a>>2]|0)){break}a=a-32|0;f=f-1|0}g=7;while(1){if((g|0)!=0){h=(c[a>>2]|0)>(b|0)}else{h=0}if(!h){break}a=a+4|0;g=g-1|0}d=f-(g>>2)|0;e=d;return e|0}function hu(a,b){a=a|0;b=b|0;var d=0;d=(c[a+616>>2]|0)+(b*448|0)+8|0;c[d>>2]=c[d>>2]|2;return}function hv(a,b){a=a|0;b=b|0;var d=0;d=(c[a+616>>2]|0)+(b*448|0)+8|0;c[d>>2]=c[d>>2]|1;return}function hw(b,d){b=b|0;d=d|0;var e=0;e=b;b=d;hx(b);if((e|0)==0){return}d=0;while(1){if((d|0)>=4){break}a[e+(d<<2)+3|0]=(c[b+(d<<2)>>2]|0)>>>24&255;a[e+(d<<2)+2|0]=(c[b+(d<<2)>>2]|0)>>>16&255;a[e+(d<<2)+1|0]=(c[b+(d<<2)>>2]|0)>>>8&255;a[e+(d<<2)|0]=c[b+(d<<2)>>2]&255;d=d+1|0}p7(b|0,0,88);return}function hx(b){b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;d=i;i=i+8|0;e=d|0;f=b;a[e+7|0]=((c[f+20>>2]|0)>>>24|0<<8)&255;a[e+6|0]=((c[f+20>>2]|0)>>>16|0<<16)&255;a[e+5|0]=((c[f+20>>2]|0)>>>8|0<<24)&255;a[e+4|0]=c[f+20>>2]&255;b=f+16|0;a[e+3|0]=((c[b>>2]|0)>>>24|c[b+4>>2]<<8)&255;b=f+16|0;a[e+2|0]=((c[b>>2]|0)>>>16|c[b+4>>2]<<16)&255;b=f+16|0;a[e+1|0]=((c[b>>2]|0)>>>8|c[b+4>>2]<<24)&255;a[e|0]=c[f+16>>2]&255;b=f+16|0;g=c[b+4>>2]|0;h=qg(64,0,((c[b>>2]|0)>>>3|g<<29)&63,(g>>>3|0<<29)&0)|0;g=h;if(g>>>0>=9){j=f;k=g;l=k-8|0;hr(j,32064,l);m=f;n=e|0;hr(m,n,8);i=d;return}g=g+64|0;j=f;k=g;l=k-8|0;hr(j,32064,l);m=f;n=e|0;hr(m,n,8);i=d;return}function hy(b){b=b|0;var c=0,e=0,f=0;c=b;b=0;while(1){if(b>>>0>=(p4(c|0)|0)>>>0){break}if((bd(a[c+b|0]|0)|0)!=0){if((d[c+b|0]|0)>127){e=4126}}else{e=4126}if((e|0)==4126){e=0;a[c+b|0]=32}b=b+1|0}while(1){if((a[c]|0)!=0){f=(a[c+((p4(c|0)|0)-1)|0]|0)==32}else{f=0}if(!f){break}a[c+((p4(c|0)|0)-1)|0]=0}return c|0}function hz(a,b){a=a|0;b=b|0;var c=0,d=0;c=a;a=b;b=bt(24104|0)|0;if((b|0)!=0){d=b}else{d=8488}p5(c|0,d|0,a|0)|0;bI(c|0,29240|0,a|0)|0;return 0}function hA(a,b,c){a=a|0;b=b|0;c=c|0;var d=0.0,e=0.0;d=+(a|0)+ +(b|0)/128.0;if((c|0)!=0){e=(240.0-d)*16.0;return+e}else{e=13694.0/+T(2.0,+(d/12.0));return+e}return 0.0}function hB(a,b){a=a|0;b=b|0;return~~(56090624.0/+T(2.0,+((+(a|0)+ +(b|0)/12800.0)/12.0)))|0}function hC(a,b,c,d,e){a=+a;b=b|0;c=c|0;d=d|0;e=e|0;var f=0.0,g=0,h=0,i=0,j=0,k=0,l=0;f=a;g=b;b=d;d=e;if((g|0)==0){h=0;i=h;return i|0}if((c|0)!=0){if(f>907.0){f=907.0}if(f<108.0){f=108.0}}if((d|0)!=0){j=~~((+(240-g<<4|0)-f)*8.0*100.0);if((b|0)!=0){k=((j|0)/12800|0)*12800|0}else{k=j}h=k;i=h;return i|0}if(f<8.0){f=8.0}j=~~+aK(+(+aa(+(+hA(g,0,0)/f))*1536.0/.6931471805599453*100.0));if((b|0)!=0){l=((j|0)/12800|0)*12800|0}else{l=j}h=l;i=h;return i|0}function hD(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=b;b=d;if((e|0)==0){c[b>>2]=0;c[a>>2]=0;return}else{d=~~(+aa(+(+(e|0)/8363.0))*1536.0/.6931471805599453);c[a>>2]=(d|0)/128|0;c[b>>2]=(d|0)%128|0;return}}function hE(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;g=e;e=f;f=b;b=f|0;i=f+1480|0;j=f+1536|0;k=j|0;l=b+316|0;m=0;do{if((g|0)>=4e3){if((g|0)>49170){break}if((oL(f,g,e,c[j+1264>>2]|0)|0)<0){n=-2;o=n;return o|0}c[b+312>>2]=c[j+1268>>2];c[b>>2]=0;c[b+4>>2]=0;c[b+12>>2]=-1;c[b+8>>2]=0;h[b+32>>3]=0.0;c[b+48>>2]=0;p=0;while(1){if((p|0)>=64){break}a[b+876+p|0]=0;c[b+620+(p<<2)>>2]=100;p=p+1|0}while(1){if((c[b>>2]|0)<(c[k+156>>2]|0)){q=(d[k+952+(c[b>>2]|0)|0]|0|0)>=(c[k+128>>2]|0)}else{q=0}if(!q){break}p=b|0;c[p>>2]=(c[p>>2]|0)+1}if((c[b>>2]|0)>=(c[k+156>>2]|0)){c[k+156>>2]=0}do{if((c[k+156>>2]|0)==0){r=4193}else{if((c[k+136>>2]|0)==0){r=4193;break}c[l+28>>2]=c[c[(c[k+168>>2]|0)+((d[k+952+(c[b>>2]|0)|0]|0)<<2)>>2]>>2];c[l+32>>2]=c[b+372>>2]}}while(0);if((r|0)==4193){c[b+364>>2]=0;c[b>>2]=0;c[b+368>>2]=0;c[b+8>>2]=0;c[l+32>>2]=0;c[l+28>>2]=0}c[b+312>>2]=c[j+1292+((c[b>>2]|0)*20|0)+8>>2];c[b+20>>2]=c[j+1292+((c[b>>2]|0)*20|0)+4>>2];c[b+16>>2]=c[j+1292+((c[b>>2]|0)*20|0)>>2];h[b+40>>3]=+h[j+1256>>3]*+h[j+1248>>3]/+(c[b+20>>2]|0);do{if((ip(f,c[k+136>>2]|0)|0)!=0){m=-2}else{c[l+16>>2]=0;c[l+4>>2]=-1;c[l>>2]=0;c[l+40>>2]=0;c[l+24>>2]=pT(c[b+944>>2]|0,8)|0;if((c[l+24>>2]|0)==0){m=-6;break}c[b+616>>2]=pT(c[b+944>>2]|0,448)|0;do{if((c[b+616>>2]|0)==0){m=-6}else{if((b6[c[c[j+6560>>2]>>2]&127](f,c[i>>2]|0)|0)<0){m=-2;pU(c[b+616>>2]|0);break}else{p=c[(c[j+6560>>2]|0)+8>>2]|0;s=f;b_[p&31](s)|0;hF(f);n=0;o=n;return o|0}}}while(0);pU(c[l+24>>2]|0)}}while(0);n=m;o=n;return o|0}}while(0);n=-7;o=n;return o|0}function hF(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;a=b|0;d=b+1536|0;e=d|0;b_[c[(c[d+6560>>2]|0)+8>>2]&31](b)|0;p7(c[a+616>>2]|0,0,(c[a+944>>2]|0)*448|0|0);b=0;while(1){if((b|0)>=(c[a+944>>2]|0)){break}f=(c[a+616>>2]|0)+(b*448|0)|0;c[f+16>>2]=-1;c[f+36>>2]=-1;b=b+1|0}b=0;while(1){if((b|0)>=(c[a+940>>2]|0)){break}f=(c[a+616>>2]|0)+(b*448|0)|0;c[f+48>>2]=c[e+184+(b*12|0)>>2];c[f+52>>2]=c[e+184+(b*12|0)+4>>2];c[f+332>>2]=255;b=b+1|0}return}function hG(a){a=a|0;var b=0,e=0,f=0,g=0,i=0,j=0,k=0;b=a;a=b|0;e=b+1536|0;f=e|0;g=a+316|0;do{if((c[f+156>>2]|0)>0){if((d[f+952+(c[a>>2]|0)|0]|0|0)==255){break}if((c[a>>2]|0)!=(c[a+4>>2]|0)){i=c[e+6416+(c[a+52>>2]<<3)>>2]|0;if((c[a+4>>2]|0)==-2){j=-1;k=j;return k|0}if((c[a+4>>2]|0)==-1){c[a+4>>2]=i}if((c[a+4>>2]|0)==(i|0)){c[g+32>>2]=c[a+360+(c[a+52>>2]<<4)+12>>2]}if((c[a+4>>2]|0)>(c[a+360+(c[a+52>>2]<<4)+4>>2]|0)){c[g+32>>2]=0}c[g+16>>2]=0;c[g+4>>2]=-1;c[a>>2]=(c[a+4>>2]|0)-1;if((c[a>>2]|0)<(i|0)){c[a>>2]=i-1}hH(b);if((c[e+1292+((c[a>>2]|0)*20|0)>>2]|0)!=0){c[a+16>>2]=c[e+1292+((c[a>>2]|0)*20|0)>>2]}c[a+20>>2]=c[e+1292+((c[a>>2]|0)*20|0)+4>>2];c[a+312>>2]=c[e+1292+((c[a>>2]|0)*20|0)+8>>2];h[a+32>>3]=+(c[e+1292+((c[a>>2]|0)*20|0)+12>>2]|0);is(b);hF(b)}else{i=a+12|0;c[i>>2]=(c[i>>2]|0)+1;i=c[a+12>>2]|0;if((i|0)>=(ac(c[a+16>>2]|0,(c[g+8>>2]|0)+1|0)|0)){hI(b)}}if((c[a+12>>2]|0)==0){do{if((c[a>>2]|0)==(c[a+360+(c[a+52>>2]<<4)+4>>2]|0)){if((c[a+8>>2]|0)!=(c[a+360+(c[a+52>>2]<<4)+8>>2]|0)){break}if((c[g+32>>2]|0)==0){i=a+48|0;c[i>>2]=(c[i>>2]|0)+1;c[g+32>>2]=c[a+360+(c[a+52>>2]<<4)+12>>2]}i=g+32|0;c[i>>2]=(c[i>>2]|0)-1}}while(0);if((c[g+12>>2]|0)!=0){c[g+12>>2]=0}else{hJ(b,d[f+952+(c[a>>2]|0)|0]|0,c[a+8>>2]|0)}}hK(b);i=0;while(1){if((i|0)>=(c[a+944>>2]|0)){break}hL(b,i,c[a+12>>2]|0);i=i+1|0}h[a+40>>3]=+h[e+1256>>3]*+h[e+1248>>3]/+(c[a+20>>2]|0);i=a+32|0;h[i>>3]=+h[i>>3]+ +h[a+40>>3];ox(b);j=0;k=j;return k|0}}while(0);j=-1;k=j;return k|0}function hH(a){a=a|0;var b=0,e=0,f=0,g=0,i=0,j=0;b=a;a=b|0;e=a+316|0;f=b+1536|0;g=f|0;do{i=a|0;c[i>>2]=(c[i>>2]|0)+1;if((c[a>>2]|0)>=(c[g+156>>2]|0)){j=4263}else{if((d[g+952+(c[a>>2]|0)|0]|0|0)==255){j=4263}}if((j|0)==4263){j=0;do{if((c[g+160>>2]|0)>(c[g+156>>2]|0)){j=4266}else{if((d[g+952+(c[g+160>>2]|0)|0]|0|0)>=(c[g+128>>2]|0)){j=4266;break}if((c[a>>2]|0)<(c[f+6416+(c[a+52>>2]<<3)>>2]|0)){j=4266;break}i=h2(b,c[g+160>>2]|0)|0;if((i|0)==(c[a+52>>2]|0)){c[a>>2]=c[g+160>>2]}else{c[a>>2]=c[f+6416+(c[a+52>>2]<<3)>>2]}}}while(0);if((j|0)==4266){j=0;c[a>>2]=c[f+6416+(c[a+52>>2]<<3)>>2]}c[a+312>>2]=c[f+1292+((c[a>>2]|0)*20|0)+8>>2]}}while((d[g+952+(c[a>>2]|0)|0]|0|0)>=(c[g+128>>2]|0));h[a+32>>3]=+(c[f+1292+((c[a>>2]|0)*20|0)+12>>2]|0);c[e+28>>2]=c[c[(c[g+168>>2]|0)+((d[g+952+(c[a>>2]|0)|0]|0)<<2)>>2]>>2];if((c[e+16>>2]|0)>=(c[e+28>>2]|0)){c[e+16>>2]=0}c[a+8>>2]=c[e+16>>2];c[e+16>>2]=0;c[a+4>>2]=c[a>>2];c[a+12>>2]=0;if((c[f+1280>>2]&256|0)==0){return}f=0;while(1){if((f|0)>=(c[g+136>>2]|0)){break}c[(c[a+616>>2]|0)+(f*448|0)+4>>2]=0;f=f+1|0}return}function hI(a){a=a|0;var b=0,d=0,e=0;b=a;a=b|0;d=a+316|0;do{if((c[d+8>>2]|0)!=0){if((c[d>>2]|0)==0){break}c[d+12>>2]=1}}while(0);c[a+12>>2]=0;c[d+8>>2]=0;if((c[d>>2]|0)!=0){c[d>>2]=0;if((c[d+4>>2]|0)!=-1){c[a>>2]=(c[d+4>>2]|0)-1;c[d+4>>2]=-1}hH(b);return}if((c[d+20>>2]|0)!=0){c[a+8>>2]=(c[(c[d+24>>2]|0)+((c[d+20>>2]|0)-1<<3)>>2]|0)-1;c[d+20>>2]=0}if((c[d+36>>2]|0)==0){e=a+8|0;c[e>>2]=(c[e>>2]|0)+1;c[d+40>>2]=0}else{e=d+36|0;c[e>>2]=(c[e>>2]|0)-1}if((c[a+8>>2]|0)>=(c[d+28>>2]|0)){hH(b)}return}function hJ(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;e=a;a=b;b=d;d=e+1536|0;f=d|0;g=e+316|0;h=0;while(1){if((h|0)>=(c[f+136>>2]|0)){break}if((b|0)<(c[c[(c[f+172>>2]|0)+(c[(c[(c[d+168>>2]|0)+(a<<2)>>2]|0)+4+(h<<2)>>2]<<2)>>2]>>2]|0)){i=(c[(c[d+172>>2]|0)+(c[(c[(c[d+168>>2]|0)+(a<<2)>>2]|0)+4+(h<<2)>>2]<<2)>>2]|0)+4+(b<<3)|0}else{i=35728}if((hW(e,i,h)|0)==0){if((c[g+40>>2]|0)!=0){if((c[g+36>>2]|0)>0){j=4311}}else{j=4311}if((j|0)==4311){j=0;k=e;l=i;m=h;hX(k,l,m)|0}}h=h+1|0}return}function hK(b){b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=b;b=e|0;f=e+1536|0;g=0;while(1){if((g|0)>=(c[f+136>>2]|0)){break}h=b+964+(g<<3)|0;if((d[h+7|0]|0|0)>0){i=e;j=h;k=g;hX(i,j,k)|0;a[h+7|0]=0}g=g+1|0}return}function hL(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=a;a=b;b=d;d=e+1536|0;f=d|0;g=(c[e+616>>2]|0)+(a*448|0)|0;c[g+436>>2]=0;if((c[g+56>>2]|0)>0){h=g+56|0;i=(c[h>>2]|0)-1|0;c[h>>2]=i;if((i|0)==0){i=e;h=c[g+416>>2]|0;j=a;hX(i,h,j)|0}}j=iI(e,a)|0;if((j|0)==-1){hO(e,a,b);return}L5352:do{if((b|0)==0){if((j|0)==256){break}do{if((c[g+36>>2]|0)>>>0<(c[f+140>>2]|0)>>>0){if((c[(c[f+176>>2]|0)+((c[g+36>>2]|0)*764|0)+36>>2]|0)<=0){break}if((j|0)==0){break}break L5352}}while(0);it(e,a);return}}while(0);do{if((c[g+36>>2]|0)>>>0<(c[f+140>>2]|0)>>>0){if((c[(c[f+176>>2]|0)+((c[g+36>>2]|0)*764|0)+36>>2]|0)<=0){break}if((b|0)==0){k=(c[g>>2]&67584|0)!=0^1^1}else{k=0}ou(e,a,g,k&1);if((c[g>>2]&262144|0)!=0){h=g+192|0;i=(c[h>>2]|0)-1|0;c[h>>2]=i;if((i|0)<=0){if((c[g+196>>2]|0)<16){iG(e,a,0)}i=g+72|0;c[i>>2]=(c[i>>2]|0)+(c[3400+((c[g+196>>2]|0)*12|0)>>2]|0);i=g+72|0;c[i>>2]=ac(c[i>>2]|0,c[3404+((c[g+196>>2]|0)*12|0)>>2]|0)|0;i=g+72|0;c[i>>2]=(c[i>>2]|0)/(c[3408+((c[g+196>>2]|0)*12|0)>>2]|0)|0;c[g+192>>2]=c[g+188>>2]&15}}hP(e,a,b,j);hQ(e,a,b,j);hS(e,a,b,j);hO(e,a,b);hT(e,a,b);hU(e,a,b);if((c[g+60>>2]|0)!=0){i=g+60|0;h=(c[i>>2]|0)-1|0;c[i>>2]=h;if((h|0)==0){h=g+8|0;c[h>>2]=c[h>>2]|2}}if((c[d+1280>>2]&4|0)!=0){hR(d,g)}c[g+432>>2]=ix(e,a)|0;return}}while(0);return}function hM(e,f){e=e|0;f=f|0;var g=0,i=0,j=0,k=0,l=0,m=0,n=0;g=f;f=e;e=f|0;i=f+1480|0;j=f+1536|0;f=c[j+136>>2]|0;do{if((c[e+4>>2]|0)>=0){if((c[e+4>>2]|0)>=(c[j+156>>2]|0)){k=4362;break}c[g>>2]=c[e+4>>2]}else{k=4362}}while(0);if((k|0)==4362){c[g>>2]=0}c[g+4>>2]=d[j+952+(c[g>>2]|0)|0]|0;if((c[g+4>>2]|0)<(c[j+128>>2]|0)){c[g+12>>2]=c[c[(c[j+168>>2]|0)+(c[g+4>>2]<<2)>>2]>>2]}else{c[g+12>>2]=0}c[g+8>>2]=c[e+8>>2];c[g+16>>2]=c[e+12>>2];c[g+20>>2]=c[e+16>>2];c[g+24>>2]=c[e+20>>2];c[g+32>>2]=c[e+360+(c[e+52>>2]<<4)>>2];c[g+36>>2]=~~(+h[e+40>>3]*1.0e3);c[g+28>>2]=~~+h[e+32>>3];c[g+40>>2]=c[i+24>>2];c[g+48>>2]=24585;c[g+44>>2]=c[i+36>>2];if((~c[i+4>>2]&4|0)!=0){k=g+44|0;c[k>>2]=c[k>>2]<<1}if((~c[i+4>>2]&1|0)!=0){i=g+44|0;c[i>>2]=c[i>>2]<<1}c[g+52>>2]=c[e+312>>2];c[g+56>>2]=c[e+48>>2];c[g+60>>2]=c[e+944>>2];c[g+64>>2]=c[e+948>>2];c[g+68>>2]=c[e+52>>2];if((c[e+616>>2]|0)==0){return}i=0;while(1){if((i|0)>=(f|0)){break}k=(c[e+616>>2]|0)+(i*448|0)|0;l=g+72+(i*24|0)|0;a[l+10|0]=c[k+16>>2]&255;b[l+8>>1]=c[k+428>>2]&65535;c[l>>2]=c[k+424>>2];c[l+4>>2]=c[k+432>>2];a[l+11|0]=c[k+36>>2]&255;a[l+12|0]=c[k+44>>2]&255;a[l+13|0]=c[k+436>>2]>>4&255;a[l+14|0]=c[k+440>>2]&255;a[l+15|0]=0;p7(l+16|0,0,8);do{if((c[g+4>>2]|0)<(c[j+128>>2]|0)){if((c[g+8>>2]|0)>=(c[g+12>>2]|0)){break}k=c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(c[g+4>>2]<<2)>>2]|0)+4+(i<<2)>>2]<<2)>>2]|0;if((c[g+8>>2]|0)<(c[k>>2]|0)){m=l+16|0;n=k+4+(c[g+8>>2]<<3)|0;p3(m|0,n|0,8)|0}}}while(0);i=i+1|0}return}function hN(a){a=a|0;var b=0,d=0;b=a;a=b|0;d=a+316|0;ir(b);b_[c[(c[b+8096>>2]|0)+4>>2]&31](b)|0;pU(c[a+616>>2]|0);pU(c[d+24>>2]|0);c[a+616>>2]=0;c[d+24>>2]=0;oM(b);return}function hO(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0;e=a;a=d;d=e|0;f=e+1536|0;e=(c[d+616>>2]|0)+(b*448|0)|0;if(((a|0)%(c[d+16>>2]|0)|0|0)!=0){g=4387}else{if((c[f+1280>>2]&64|0)!=0){g=4387}}if((g|0)==4387){if((c[e>>2]&1048576|0)!=0){b=d+312|0;c[b>>2]=(c[b>>2]|0)+(c[e+228>>2]|0)}if((c[e>>2]&1|0)!=0){g=4391}else{if((c[e+4>>2]&1|0)!=0){g=4391}}if((g|0)==4391){g=e+72|0;c[g>>2]=(c[g>>2]|0)+(c[e+212>>2]|0)}if((c[e+4>>2]&1|0)!=0){do{if((c[e+212>>2]|0)>0){if((c[e+72>>2]|0)<=(c[f+1268>>2]|0)){break}c[e+72>>2]=c[f+1268>>2];g=e+4|0;c[g>>2]=c[g>>2]&-2}}while(0);do{if((c[e+212>>2]|0)<0){if((c[e+72>>2]|0)>=0){break}c[e+72>>2]=0;g=e+4|0;c[g>>2]=c[g>>2]&-2}}while(0)}if((c[e>>2]&8192|0)!=0){g=e+72|0;c[g>>2]=(c[g>>2]|0)+(c[e+220>>2]|0)}if((c[e>>2]&512|0)!=0){g=e+52|0;c[g>>2]=(c[g>>2]|0)+(c[e+240>>2]|0)}}if(((a|0)%(c[d+16>>2]|0)|0|0)==0){if((c[e>>2]&64|0)!=0){a=e+72|0;c[a>>2]=(c[a>>2]|0)+(c[e+216>>2]|0)}if((c[e>>2]&1024|0)!=0){a=e+52|0;c[a>>2]=(c[a>>2]|0)+(c[e+244>>2]|0)}if((c[e>>2]&1048576|0)!=0){a=d+312|0;c[a>>2]=(c[a>>2]|0)+(c[e+232>>2]|0)}}if((c[e+72>>2]|0)<0){c[e+72>>2]=0}else{if((c[e+72>>2]|0)>(c[f+1268>>2]|0)){c[e+72>>2]=c[f+1268>>2]}}if((c[d+312>>2]|0)<0){c[d+312>>2]=0}else{if((c[d+312>>2]|0)>(c[f+1272>>2]|0)){c[d+312>>2]=c[f+1272>>2]}}if((c[e+52>>2]|0)<0){c[e+52>>2]=0}else{if((c[e+52>>2]|0)>(c[f+1268>>2]|0)){c[e+52>>2]=c[f+1268>>2]}}nw(e+112|0);return}function hP(a,d,f,g){a=a|0;d=d|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;f=i;i=i+8|0;h=f|0;j=a;a=d;d=g;g=j|0;k=j+1536|0;l=(c[g+616>>2]|0)+(a*448|0)|0;m=(c[k+176>>2]|0)+((c[l+36>>2]|0)*764|0)|0;if((c[k+1280>>2]&262144|0)==0){do{if((c[l+8>>2]&2|0)!=0){if((c[m+44>>2]&1|0)!=0){break}c[l+64>>2]=0}}while(0)}do{if((c[l+8>>2]&3|0)!=0){n=4443}else{if((d|0)==3){n=4443;break}if((d|0)==2){n=4443}}}while(0);if((n|0)==4443){if((c[l+64>>2]|0)>(c[m+40>>2]|0)){o=l+64|0;c[o>>2]=(c[o>>2]|0)-(c[m+40>>2]|0)}else{c[l+64>>2]=0;o=l+8|0;c[o>>2]=c[o>>2]|4}do{if((c[l+64>>2]|0)==0){if((c[k+1280>>2]&32768|0)!=0){it(j,a);i=f;return}else{c[l+72>>2]=0;break}}}while(0)}o=m5(m+44|0,e[l+88>>1]|0)|0;if((o|0)==(-1|0)){p=l+8|0;c[p>>2]=c[p>>2]|4}else if((o|0)!=0){if((c[k+1280>>2]&2|0)!=0){o=l+8|0;c[o>>2]=c[o>>2]|1}}o=(m3(m+44|0,e[l+88>>1]|0,64,h)|0)&65535;if((c[l+8>>2]&2|0)!=0){q=1}else{q=(d|0)==2}b[l+88>>1]=(m4(m+44|0,e[l+88>>1]|0,q&1)|0)&65535;do{if((c[h>>2]|0)!=0){if((o&65535|0)!=0){break}q=l+8|0;c[q>>2]=c[q>>2]|4}}while(0);do{if((c[l+8>>2]&4|0)!=0){if((a|0)<(c[g+940>>2]|0)){break}it(j,a);i=f;return}}while(0);do{if((c[k+6548>>2]|0)!=0){if((c[(c[k+6548>>2]|0)+(c[l+36>>2]<<2)>>2]|0)==0){n=4469;break}r=(ac(c[l+412>>2]|0,c[l+72>>2]|0)|0)/64|0}else{n=4469}}while(0);if((n|0)==4469){r=c[l+72>>2]|0}if((c[l>>2]&32|0)!=0){r=r+((nB(l+112|0)|0)/128|0)|0}if((r|0)<0){r=0}else{if((r|0)>(c[k+1268>>2]|0)){r=c[k+1268>>2]|0}}r=(ac(r,c[l+64>>2]|0)|0)>>5;if((c[k+1280>>2]&16|0)!=0){s=64}else{s=c[g+312>>2]|0}g=ac(o&65535,s)|0;s=ac(g,c[l+52>>2]|0)|0;r=(ac((s|0)/(c[k+1272>>2]|0)|0,(r<<6|0)/(c[k+1268>>2]|0)|0)|0)>>>18;r=(ac(r,hV(j,a)|0)|0)/100|0;if((c[k+1276>>2]|0)!=0){if((c[k+1268>>2]|0)==255){t=c[(c[k+1276>>2]|0)+(r>>2<<2)>>2]<<2}else{t=c[(c[k+1276>>2]|0)+(r>>4<<2)>>2]<<4}r=t}if((c[k+1280>>2]&16384|0)!=0){k=ac(r,c[m+32>>2]|0)|0;r=(ac(k,c[l+76>>2]|0)|0)>>12}if((c[l+200>>2]|0)!=0){if((c[l+204>>2]|0)==0){c[l+204>>2]=(c[l+200>>2]&240)>>4|128}else{if((c[l+204>>2]|0)==128){c[l+204>>2]=c[l+200>>2]&15}}k=l+204|0;c[k>>2]=(c[k>>2]|0)-1;if((~c[l+204>>2]&128|0)!=0){r=0}}c[l+436>>2]=r;iu(j,a,r);i=f;return}function hQ(a,d,f,g){a=a|0;d=d|0;f=f|0;g=g|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0.0,u=0.0,v=0;f=i;i=i+32|0;j=f+8|0;k=f+16|0;l=f+24|0;m=a;a=d;d=m+1480|0;n=m+1536|0;o=(c[m+616>>2]|0)+(a*448|0)|0;p=(c[n+176>>2]|0)+((c[o+36>>2]|0)*764|0)|0;q=m3(p+356|0,e[o+92>>1]|0,0,f|0)|0;if((c[o+8>>2]&2|0)!=0){r=1}else{r=(g|0)==2}b[o+92>>1]=(m4(p+356|0,e[o+92>>1]|0,r&1)|0)&65535;if((c[o>>2]&16384|0)!=0){if((c[o+316>>2]|0)==0){r=o+12|0;c[r>>2]=(c[r>>2]|0)+(c[o+304>>2]|0);h[o+24>>3]=+hA(c[o+12>>2]|0,c[o+32>>2]|0,c[n+1280>>2]&4096);c[o+316>>2]=c[o+312>>2]}r=o+316|0;c[r>>2]=(c[r>>2]|0)-1}r=nB(o+168|0)|0;g=(r|0)/((c[o+184>>2]|0)+1<<10|0)|0;if((c[o>>2]&16|0)!=0){s=4511}else{if((c[o+4>>2]&16|0)!=0){s=4511}}if((s|0)==4511){if((c[n+1280>>2]&524288|0)!=0){g=g+((nB(o+96|0)|0)>>9)|0}else{g=g+((nB(o+96|0)|0)>>10)|0}}t=+h[o+24>>3]+ +(g|0);u=t+ +(oq(o)|0);g=hC(u,c[o+12>>2]|0,c[n+1280>>2]&8192,c[o+68>>2]|0,c[n+1280>>2]&4096)|0;if((~c[p+356>>2]&8|0)!=0){g=g+(q<<7)|0}s=c[o+144+(c[o+164>>2]<<2)>>2]|0;g=g+(s+(or(n,o)|0)<<7)|0;c[o+428>>2]=g>>7;c[o+424>>2]=hB(c[o+12>>2]|0,g)|0;iF(m,a,g);if((c[n+1280>>2]&65536|0)==0){i=f;return}if((c[p+356>>2]&8|0)!=0){v=(ac(c[o+332>>2]|0,q)|0)>>8}else{v=c[o+332>>2]|0}q=c[o+336>>2]|0;if((v|0)>255){v=255}else{if((v|0)<255){m6(c[d>>2]|0,v,q,j,k,l);iw(m,a,176,c[j>>2]|0);iw(m,a,177,c[k>>2]|0);iw(m,a,178,c[l>>2]|0);iw(m,a,3,q)}}iw(m,a,2,v);i=f;return}function hR(b,e){b=b|0;e=e|0;var f=0,g=0,h=0;f=b;b=e;e=(c[f+180>>2]|0)+((c[b+44>>2]|0)*52|0)|0;g=b+324|0;c[g>>2]=(c[g>>2]|0)+(c[7264+(c[b+320>>2]<<2)>>2]|0);if((c[e+44>>2]&2|0)==0){return}if((c[b+324>>2]|0)<128){return}c[b+324>>2]=0;g=(c[e+40>>2]|0)-(c[e+36>>2]|0)|0;if((c[f+1280>>2]&8|0)==0){f=b+328|0;h=(c[f>>2]|0)+1|0;c[f>>2]=h;if((h|0)>(g|0)){c[b+328>>2]=0}if((~c[e+44>>2]&1|0)!=0){g=(c[e+48>>2]|0)+((c[e+36>>2]|0)+(c[b+328>>2]|0))|0;a[g]=((d[g]|0)^255)&255}}return}function hS(a,d,f,g){a=a|0;d=d|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0;f=i;i=i+8|0;h=a;a=d;d=h+1480|0;j=(c[h+616>>2]|0)+(a*448|0)|0;k=(c[h+1712>>2]|0)+((c[j+36>>2]|0)*764|0)|0;l=0;m=m3(k+200|0,e[j+90>>1]|0,32,f|0)|0;if((c[j+8>>2]&2|0)!=0){n=1}else{n=(g|0)==2}b[j+90>>1]=(m4(k+200|0,e[j+90>>1]|0,n&1)|0)&65535;if((c[j>>2]&524288|0)!=0){l=(nB(j+128|0)|0)/512|0}n=(c[j+288>>2]|0)+l|0;l=n+((ac(m-32|0,128-(R((c[j+288>>2]|0)-128|0)|0)|0)|0)/32|0)|0;m=c[j+48>>2]|0;l=m+((ac(l-128|0,128-(R((c[j+48>>2]|0)-128|0)|0)|0)|0)/128|0)|0;if((c[d+4>>2]&4|0)!=0){l=0}else{l=(ac(l-128|0,c[d+12>>2]|0)|0)/100|0}if((l|0)<-128){l=-128}else{if((l|0)>127){l=127}}c[j+440>>2]=l+128;iv(h,a,l);i=f;return}function hT(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0;e=a;a=d;d=e|0;f=e+1536|0;e=(c[d+616>>2]|0)+(b*448|0)|0;if(((a|0)%(c[d+16>>2]|0)|0|0)!=0){g=4562}else{if((c[f+1280>>2]&128|0)!=0){g=4562}}if((g|0)==4562){if((c[e>>2]&8|0)!=0){g=4564}else{if((c[e+4>>2]&8|0)!=0){g=4564}}if((g|0)==4564){b=e+24|0;h[b>>3]=+h[b>>3]+ +(c[e+252>>2]|0)}if((c[e>>2]&4|0)!=0){g=4567}else{if((c[e+4>>2]&4|0)!=0){g=4567}}if((g|0)==4567){g=0;if((c[e+272>>2]|0)>0){b=e+24|0;h[b>>3]=+h[b>>3]+ +(c[e+276>>2]|0);if(+h[e+24>>3]>=+h[e+264>>3]){g=1}}else{b=e+24|0;h[b>>3]=+h[b>>3]- +(c[e+276>>2]|0);if(+h[e+24>>3]<=+h[e+264>>3]){g=1}}if((g|0)!=0){h[e+24>>3]=+h[e+264>>3];c[e+272>>2]=0;g=e|0;c[g>>2]=c[g>>2]&-5;g=e+4|0;c[g>>2]=c[g>>2]&-5}}}if(((a|0)%(c[d+16>>2]|0)|0|0)==0){if((c[e>>2]&128|0)!=0){h[e+24>>3]=(+h[e+24>>3]*4.0+ +(c[e+256>>2]|0))/4.0}if((c[e>>2]&32768|0)!=0){g=e+12|0;c[g>>2]=(c[g>>2]|0)+(c[e+308>>2]|0);h[e+24>>3]=+hA(c[e+12>>2]|0,c[e+32>>2]|0,c[f+1280>>2]&4096)}}if(+h[e+24>>3]<1.0){c[e+72>>2]=0}if((c[f+1280>>2]&4096|0)!=0){if(+h[e+24>>3]<0.0){h[e+24>>3]=0.0}else{if(+h[e+24>>3]>7680.0){h[e+24>>3]=7680.0}}}else{if(+h[e+24>>3]<8.0){h[e+24>>3]=8.0}else{if(+h[e+24>>3]>6847.0){h[e+24>>3]=6847.0}}}g=e+164|0;c[g>>2]=(c[g>>2]|0)+1;g=e+164|0;c[g>>2]=(c[g>>2]|0)%(c[e+160>>2]|0)|0;do{if(((a|0)%(c[d+16>>2]|0)|0|0)==0){if((c[f+1280>>2]&1048576|0)!=0){break}return}}while(0);nw(e+96|0);nw(e+168|0);if((c[e+184>>2]|0)>1){f=e+184|0;c[f>>2]=(c[f>>2]|0)-2}else{c[e+184>>2]=0}return}function hU(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0;e=a|0;a=(c[e+616>>2]|0)+(b*448|0)|0;if((c[a>>2]&2|0)==0){f=a;g=f+128|0;nw(g);return}if(((d|0)%(c[e+16>>2]|0)|0|0)==0){e=a+288|0;c[e>>2]=(c[e>>2]|0)+(c[a+296>>2]|0)}else{e=a+288|0;c[e>>2]=(c[e>>2]|0)+(c[a+292>>2]|0)}if((c[a+288>>2]|0)<0){c[a+288>>2]=0}else{if((c[a+288>>2]|0)>255){c[a+288>>2]=255}}f=a;g=f+128|0;nw(g);return}function hV(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=a;a=b;b=d|0;if((a|0)<(c[b+940>>2]|0)){e=c[b+620+(a<<2)>>2]|0;f=e;return f|0}if((a|0)>=(c[b+944>>2]|0)){e=0;f=e;return f|0}g=ic(d,a)|0;if((g|0)<0){e=0;f=e;return f|0}else{e=c[b+620+(g<<2)>>2]|0;f=e;return f|0}return 0}function hW(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=e;e=b|0;b=(c[e+616>>2]|0)+(f*448|0)|0;if((d[g+3|0]|0)==15){if((d[g+4|0]|0)<32){h=4640}else{h=4639}}else{h=4639}if((h|0)==4639){if((d[g+3|0]|0)==163){h=4640}}if((h|0)==4640){if((a[g+4|0]|0)!=0){c[e+16>>2]=d[g+4|0]|0}}if((d[g+5|0]|0)==15){if((d[g+6|0]|0)<32){h=4646}else{h=4645}}else{h=4645}if((h|0)==4645){if((d[g+5|0]|0)==163){h=4646}}if((h|0)==4646){if((a[g+6|0]|0)!=0){c[e+16>>2]=d[g+6|0]|0}}do{if((d[g+3|0]|0)==14){if(((a[g+4|0]&240)>>4|0)!=13){break}c[b+56>>2]=(a[g+4|0]&15)+1;c[b+416>>2]=g;if((a[g+1|0]|0)!=0){c[b+420>>2]=d[g+1|0]|0}i=1;j=i;return j|0}}while(0);do{if((d[g+5|0]|0)==14){if(((a[g+6|0]&240)>>4|0)!=13){break}c[b+56>>2]=(a[g+6|0]&15)+1;c[b+416>>2]=g;if((a[g+1|0]|0)!=0){c[b+420>>2]=d[g+1|0]|0}i=1;j=i;return j|0}}while(0);i=0;j=i;return j|0}function hX(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=a;a=b;b=d;d=c[e+2820>>2]|0;if((d|0)==2){f=h_(e,a,b)|0}else if((d|0)==1){f=hZ(e,a,b)|0}else if((d|0)==3){f=h1(e,a,b)|0}else if((d|0)==0){f=hY(e,a,b)|0}else{f=hY(e,a,b)|0}return f|0}function hY(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;g=b;b=e;e=f;f=g|0;h=g+1536|0;i=(c[f+616>>2]|0)+(e*448|0)|0;j=0;c[i>>2]=0;k=-1;l=0;m=0;do{if((d[b+3|0]|0)==3){n=4677}else{if((d[b+3|0]|0)==5){n=4677;break}if((d[b+3|0]|0)==122){n=4677;break}if((d[b+5|0]|0)==3){n=4677;break}if((d[b+5|0]|0)==5){n=4677;break}if((d[b+5|0]|0)==122){n=4677}}}while(0);if((n|0)==4677){l=1}if((a[b+1|0]|0)!=0){o=(d[b+1|0]|0)-1|0;p=i|0;c[p>>2]=c[p>>2]|2048;m=1;c[i+64>>2]=32768;c[i+4>>2]=0;c[i+84>>2]=0;p=i+8|0;c[p>>2]=c[p>>2]&-3;do{if(o>>>0<(c[h+140>>2]|0)>>>0){if((c[(c[h+176>>2]|0)+(o*764|0)+36>>2]|0)<=0){n=4687;break}if((l|0)!=0){q=h$(g,o,d[b|0]|0)|0;if((q|0)!=0){c[i+72>>2]=c[q>>2];m=0}}else{c[i+36>>2]=o}}else{n=4687}}while(0);if((n|0)==4687){j=1;it(g,e)}c[i+388>>2]=0;c[i+384>>2]=0}if((a[b|0]|0)!=0){o=i|0;c[o>>2]=c[o>>2]|65536;if((d[b|0]|0)==129){o=i+8|0;c[o>>2]=c[o>>2]|2;m=0}else{if((l|0)==0){c[i+16>>2]=(d[b|0]|0)-1;o=i+8|0;c[o>>2]=c[o>>2]&-5;q=h$(g,c[i+36>>2]|0,c[i+16>>2]|0)|0;do{if((j|0)!=0){n=4701}else{if((q|0)==0){n=4701;break}k=(c[i+16>>2]|0)+(c[q+12>>2]|0)+(a[(c[h+176>>2]|0)+((c[i+36>>2]|0)*764|0)+512+(c[i+16>>2]<<1)+1|0]|0)|0;o=c[q+40>>2]|0;if((c[(c[h+180>>2]|0)+(o*52|0)+32>>2]|0)==0){o=-1}do{if((o|0)>=0){if((o|0)>=(c[h+144>>2]|0)){break}p=g;r=e;s=c[i+36>>2]|0;t=o;u=k;iz(p,r,s,t,u,0,0,0)|0;c[i+44>>2]=o}}while(0)}}while(0);if((n|0)==4701){c[i>>2]=0;m=0}}}}q=h$(g,c[i+36>>2]|0,c[i+16>>2]|0)|0;h4(g,k,q,i,l);do{if((d[b+1|0]|0)!=0){if((q|0)==0){break}h0(h,i)}}while(0);if((a[b+2|0]|0)!=0){c[i+72>>2]=(d[b+2|0]|0)-1;h=i|0;c[h>>2]=c[h>>2]|4096}m2(g,e,a[b|0]|0,a[b+5|0]|0,a[b+6|0]|0,i,1);m2(g,e,a[b|0]|0,a[b+3|0]|0,a[b+4|0]|0,i,0);if((c[i>>2]&4096|0)!=0){m=0}if((q|0)==0){v=0;w=v;return w|0}if((k|0)>=0){c[i+12>>2]=k;iG(g,e,c[i+84>>2]|0);do{if((c[i>>2]&256|0)!=0){if((c[f+28>>2]&2|0)==0){break}e=i+84|0;c[e>>2]=c[e>>2]<<1}}while(0);f=i|0;c[f>>2]=c[f>>2]&-257}if((m|0)!=0){c[i+72>>2]=c[q>>2];q=i|0;c[q>>2]=c[q>>2]|4096}v=0;w=v;return w|0}function hZ(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;g=b;b=e;e=f;f=g+1536|0;h=(c[g+616>>2]|0)+(e*448|0)|0;c[h>>2]=0;i=-1;j=d[b|0]|0;k=0;l=0;m=0;do{if((d[b+3|0]|0)==3){n=4731}else{if((d[b+3|0]|0)==5){n=4731;break}if((d[b+3|0]|0)==122){n=4731;break}if((d[b+5|0]|0)==3){n=4731;break}if((d[b+5|0]|0)==5){n=4731;break}if((d[b+5|0]|0)==122){n=4731}}}while(0);if((n|0)==4731){l=1}if((a[b+1|0]|0)!=0){o=(d[b+1|0]|0)-1|0;p=h|0;c[p>>2]=c[p>>2]|2048;m=1;c[h+64>>2]=32768;c[h+4>>2]=0;p=h+8|0;c[p>>2]=c[p>>2]&-3;do{if(o>>>0<(c[f+140>>2]|0)>>>0){if((c[(c[f+176>>2]|0)+(o*764|0)+36>>2]|0)<=0){n=4738;break}if((l|0)==0){c[h+36>>2]=o}}else{n=4738}}while(0);if((n|0)==4738){k=1;c[h>>2]=0;if((l|0)!=0){j=0}}c[h+388>>2]=0;c[h+384>>2]=0}if((j|0)!=0){o=h|0;c[o>>2]=c[o>>2]|65536;if((j|0)==129){o=h+8|0;c[o>>2]=c[o>>2]|2;m=0}else{if((l|0)!=0){j=0}}if((k|0)!=0){it(g,e)}}if((a[b+1|0]|0)!=0){do{if((j|0)==0){n=4754}else{if((j|0)>=129){n=4754;break}q=h$(g,c[h+36>>2]|0,j-1|0)|0;if((q|0)!=0){c[h+72>>2]=c[q>>2]}else{c[h+72>>2]=0}c[h+40>>2]=c[h+36>>2];o=h|0;c[o>>2]=c[o>>2]|4096}}while(0);if((n|0)==4754){q=h$(g,c[h+40>>2]|0,c[h+16>>2]|0)|0;if((q|0)!=0){c[h+72>>2]=c[q>>2];q=h|0;c[q>>2]=c[q>>2]|4096}}}do{if(j>>>0<=121){if((j|0)<=0){break}q=j-1|0;j=q;c[h+16>>2]=q;c[h+64>>2]=32768;q=h+8|0;c[q>>2]=c[q>>2]&-7;r=h$(g,c[h+36>>2]|0,j)|0;do{if((k|0)!=0){n=4772}else{if((r|0)==0){n=4772;break}i=j+(c[r+12>>2]|0)+(a[(c[f+176>>2]|0)+((c[h+36>>2]|0)*764|0)+512+(j<<1)+1|0]|0)|0;q=c[r+40>>2]|0;if((c[(c[f+180>>2]|0)+(q*52|0)+32>>2]|0)==0){q=-1}do{if((q|0)>=0){if((q|0)>=(c[f+144>>2]|0)){break}o=g;p=e;s=c[h+36>>2]|0;t=q;u=i;iz(o,p,s,t,u,0,0,0)|0;c[h+44>>2]=q}}while(0)}}while(0);if((n|0)==4772){c[h>>2]=0;m=0}}}while(0);r=h$(g,c[h+36>>2]|0,c[h+16>>2]|0)|0;h4(g,i,r,h,l);do{if((d[b+1|0]|0)!=0){if((r|0)==0){break}h0(f,h)}}while(0);if((a[b+2|0]|0)!=0){c[h+72>>2]=(d[b+2|0]|0)-1;f=h|0;c[f>>2]=c[f>>2]|4096}c[h+84>>2]=0;m2(g,e,a[b|0]|0,a[b+5|0]|0,a[b+6|0]|0,h,1);m2(g,e,a[b|0]|0,a[b+3|0]|0,a[b+4|0]|0,h,0);if((c[h>>2]&4096|0)!=0){m=0}if((r|0)==0){v=0;w=v;return w|0}if((i|0)>=0){c[h+12>>2]=i;iG(g,e,c[h+84>>2]|0);if((c[h>>2]&256|0)!=0){e=h+84|0;c[e>>2]=c[e>>2]<<1}e=h|0;c[e>>2]=c[e>>2]&-257}if((m|0)!=0){c[h+72>>2]=c[r>>2];r=h|0;c[r>>2]=c[r>>2]|4096}v=0;w=v;return w|0}function h_(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;g=b;b=e;e=f;f=g|0;h=g+1536|0;i=h|0;j=(c[f+616>>2]|0)+(e*448|0)|0;c[j>>2]=0;k=-1;l=0;m=0;n=0;do{if((d[b+3|0]|0)==3){o=4799}else{if((d[b+3|0]|0)==5){o=4799;break}if((d[b+3|0]|0)==122){o=4799;break}if((d[b+5|0]|0)==3){o=4799;break}if((d[b+5|0]|0)==5){o=4799;break}if((d[b+5|0]|0)==122){o=4799}}}while(0);if((o|0)==4799){m=1}if((a[b+1|0]|0)!=0){p=(d[b+1|0]|0)-1|0;q=j|0;c[q>>2]=c[q>>2]|2048;n=1;c[j+64>>2]=32768;c[j+4>>2]=0;c[j+84>>2]=0;q=j+8|0;c[q>>2]=c[q>>2]&-3;do{if(p>>>0<(c[i+140>>2]|0)>>>0){if((c[(c[i+176>>2]|0)+(p*764|0)+36>>2]|0)<=0){o=4811;break}if((c[j+36>>2]|0)!=(p|0)){l=1;if((m|0)!=0){r=h$(g,p,d[b|0]|0)|0;if((r|0)!=0){c[j+72>>2]=c[r>>2];n=0}}else{c[j+36>>2]=p}}}else{o=4811}}while(0);if((o|0)==4811){c[j>>2]=0;n=0}c[j+388>>2]=0;c[j+384>>2]=0}if((a[b|0]|0)!=0){o=j|0;c[o>>2]=c[o>>2]|65536;if((d[b|0]|0)==129){o=j+8|0;c[o>>2]=c[o>>2]|2;n=0}else{if((m|0)!=0){if((l|0)!=0){c[j+84>>2]=0}}else{c[j+16>>2]=(d[b|0]|0)-1;l=j+8|0;c[l>>2]=c[l>>2]&-5;r=h$(g,c[j+36>>2]|0,c[j+16>>2]|0)|0;if((r|0)!=0){k=(c[j+16>>2]|0)+(c[r+12>>2]|0)+(a[(c[i+176>>2]|0)+((c[j+36>>2]|0)*764|0)+512+(c[j+16>>2]<<1)+1|0]|0)|0;l=c[r+40>>2]|0;if((c[(c[i+180>>2]|0)+(l*52|0)+32>>2]|0)==0){l=-1}do{if((l|0)>=0){if((l|0)>=(c[i+144>>2]|0)){break}o=g;p=e;q=c[j+36>>2]|0;s=l;t=k;iz(o,p,q,s,t,0,0,0)|0;c[j+44>>2]=l}}while(0)}else{c[j>>2]=0;n=0}}}}r=h$(g,c[j+36>>2]|0,c[j+16>>2]|0)|0;h4(g,k,r,j,m);do{if((d[b+1|0]|0)!=0){if((r|0)==0){break}h0(i,j)}}while(0);if((a[b+2|0]|0)!=0){c[j+72>>2]=(d[b+2|0]|0)-1;i=j|0;c[i>>2]=c[i>>2]|4096}m2(g,e,a[b|0]|0,a[b+5|0]|0,a[b+6|0]|0,j,1);m2(g,e,a[b|0]|0,a[b+3|0]|0,a[b+4|0]|0,j,0);if((c[j>>2]&4096|0)!=0){n=0}if((r|0)==0){u=0;v=u;return v|0}if((k|0)>=0){c[j+12>>2]=k;iG(g,e,c[j+84>>2]|0);if((c[j>>2]&256|0)!=0){e=j+84|0;c[e>>2]=c[e>>2]<<1}e=j|0;c[e>>2]=c[e>>2]&-257}if((n|0)!=0){c[j+72>>2]=c[r>>2];r=j|0;c[r>>2]=c[r>>2]|4096}do{if((c[h+1280>>2]&16|0)!=0){if((c[j>>2]&4096|0)==0){break}r=ac(c[j+72>>2]|0,c[f+312>>2]|0)|0;c[j+72>>2]=(r|0)/(c[h+1268>>2]|0)|0}}while(0);u=0;v=u;return v|0}function h$(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0,h=0;f=b;b=e;e=a+1536|0;do{if((f|0)>=0){if((b|0)<0){break}if((d[(c[e+176>>2]|0)+(f*764|0)+512+(b<<1)|0]|0|0)==255){break}g=(c[(c[e+176>>2]|0)+(f*764|0)+756>>2]|0)+((d[(c[e+176>>2]|0)+(f*764|0)+512+(b<<1)|0]|0)<<6)|0;h=g;return h|0}}while(0);g=0;h=g;return h|0}function h0(a,d){a=a|0;d=d|0;var e=0;e=a;a=d;if((~c[(c[e+176>>2]|0)+((c[a+36>>2]|0)*764|0)+44>>2]&32|0)!=0){b[a+88>>1]=0}if((~c[(c[e+176>>2]|0)+((c[a+36>>2]|0)*764|0)+200>>2]&32|0)!=0){b[a+90>>1]=0}if((~c[(c[e+176>>2]|0)+((c[a+36>>2]|0)*764|0)+356>>2]&32|0)==0){return}b[a+92>>1]=0;return}function h1(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;g=b;b=e;e=f;f=g|0;i=g+1536|0;j=i|0;k=(c[f+616>>2]|0)+(e*448|0)|0;l=a[b+1|0]|0;if(l<<24>>24!=0){c[k+420>>2]=0}else{do{if((d[b|0]|0)!=0){if((c[k+420>>2]|0)==0){break}l=c[k+420>>2]&255;c[k+420>>2]=0}}while(0)}c[k>>2]=0;m=-1;n=d[b|0]|0;o=0;p=0;q=0;r=0;s=0;t=0;u=c[k+36>>2]|0;do{if((d[b+3|0]|0)==3){v=4883}else{if((d[b+3|0]|0)==5){v=4883;break}if((d[b+3|0]|0)==122){v=4883;break}if((d[b+5|0]|0)==3){v=4883;break}if((d[b+5|0]|0)==5){v=4883;break}if((d[b+5|0]|0)==122){v=4883}}}while(0);if((v|0)==4883){q=1}if((c[k+8>>2]&3|0)!=0){r=1}if(l<<24>>24!=0){w=(l&255)-1|0;do{if((r|0)!=0){if((q|0)==0){v=4890;break}if((c[k+36>>2]|0)!=(w|0)){v=4890}}else{v=4890}}while(0);if((v|0)==4890){r=k|0;c[r>>2]=c[r>>2]|2048;t=1;s=1;c[k+64>>2]=32768}c[k+4>>2]=0;do{if(w>>>0<(c[j+140>>2]|0)>>>0){if((c[(c[j+176>>2]|0)+(w*764|0)+36>>2]|0)<=0){v=4905;break}if((n|0)==0){if((c[k+36>>2]|0)==(w|0)){r=k|0;c[r>>2]=c[r>>2]|2048;t=1}else{n=(c[k+16>>2]|0)+1|0}}if((c[k+36>>2]|0)!=(w|0)){o=1;u=w;if((q|0)!=0){x=h$(g,w,n)|0;if((x|0)!=0){c[k+72>>2]=c[x>>2];t=0}}}}else{v=4905}}while(0);if((v|0)==4905){p=1;c[k>>2]=0;t=0}c[k+388>>2]=0;c[k+384>>2]=0}do{if((n|0)!=0){if((p|0)!=0){break}w=k|0;c[w>>2]=c[w>>2]|65536;if((n|0)==131){w=k+8|0;c[w>>2]=c[w>>2]|1;s=0;t=0}else{if((n|0)==130){w=k+8|0;c[w>>2]=c[w>>2]|4;h[k+24>>3]=0.0;it(g,e)}else{if((n|0)==129){w=k+8|0;c[w>>2]=c[w>>2]|2;s=0;t=0;if((c[i+1280>>2]&2097152|0)!=0){w=k+8|0;c[w>>2]=c[w>>2]|4}}else{if((q|0)!=0){do{if((o|0)!=0){v=4920}else{if((c[k+8>>2]&4|0)!=0){v=4920;break}n=0}}while(0);if((v|0)==4920){w=k|0;c[w>>2]=c[w>>2]|2048;w=k+8|0;c[w>>2]=c[w>>2]&-3}do{if((c[i+1280>>2]&2097152|0)!=0){if((d[b+1|0]|0)==0){break}h0(j,k)}}while(0)}}}}}}while(0);do{if(n>>>0<=121){if((n|0)<=0){break}if((p|0)!=0){break}i=n-1|0;n=i;c[k+16>>2]=i;i=k+8|0;c[i>>2]=c[i>>2]&-5;x=h$(g,u,n)|0;if((x|0)!=0){m=n+(c[x+12>>2]|0)+(a[(c[j+176>>2]|0)+(u*764|0)+512+(n<<1)+1|0]|0)|0;i=c[x+40>>2]|0;if((i|0)>=(c[j+144>>2]|0)){v=4936}else{if((c[(c[j+180>>2]|0)+(i*52|0)+32>>2]|0)==0){v=4936}}if((v|0)==4936){i=-1}do{if((i|0)>=0){if((i|0)>=(c[j+144>>2]|0)){break}o=iz(g,e,u,i,m,c[x+44>>2]|0,c[x+48>>2]|0,c[x+52>>2]|0)|0;if((o|0)<0){y=-1;z=y;return z|0}if((o|0)!=(e|0)){h3(f,o,e);c[(c[f+616>>2]|0)+(o*448|0)>>2]=0}c[k+44>>2]=i}}while(0)}else{c[k>>2]=0;t=0}}}while(0);do{if(u>>>0<(c[j+140>>2]|0)>>>0){if((c[(c[j+176>>2]|0)+(u*764|0)+36>>2]|0)<=0){break}c[k+36>>2]=u}}while(0);x=h$(g,c[k+36>>2]|0,c[k+16>>2]|0)|0;h4(g,m,x,k,q);do{if((m|0)>=0){if((x|0)==0){break}h0(j,k)}}while(0);if((a[b+2|0]|0)!=0){c[k+72>>2]=(d[b+2|0]|0)-1;j=k|0;c[j>>2]=c[j>>2]|4096}c[k+84>>2]=0;m2(g,e,a[b|0]|0,a[b+3|0]|0,a[b+4|0]|0,k,0);m2(g,e,a[b|0]|0,a[b+5|0]|0,a[b+6|0]|0,k,1);if((c[k>>2]&4096|0)!=0){t=0}if((x|0)==0){y=0;z=y;return z|0}if((m|0)>=0){c[k+12>>2]=m;iG(g,e,c[k+84>>2]|0);if((c[k>>2]&256|0)!=0){e=k+84|0;c[e>>2]=c[e>>2]<<1}e=k|0;c[e>>2]=c[e>>2]&-257}if((s|0)!=0){s=k+8|0;c[s>>2]=c[s>>2]&-4}if((t|0)!=0){c[k+72>>2]=c[x>>2];x=k|0;c[x>>2]=c[x>>2]|4096}y=0;z=y;return z|0}function h2(a,b){a=a|0;b=b|0;return d[a+56+b|0]|0|0}function h3(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=b;b=d;if((a|0)<=0){return}if((a|0)==(b|0)){return}d=(c[e+616>>2]|0)+(a*448|0)|0;a=(c[e+616>>2]|0)+(b*448|0)|0;p3(d|0,a|0,448)|0;return}function h4(a,b,d,e,f){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;var g=0,i=0;g=b;b=d;d=e;e=f;f=a+1536|0;do{if((b|0)!=0){if((g|0)<0){break}c[d+288>>2]=c[b+8>>2];c[d+32>>2]=c[b+16>>2];c[d+76>>2]=c[b+4>>2];if((c[b+56>>2]&128|0)!=0){c[d+332>>2]=(c[b+56>>2]|0)-128<<1}else{c[d+332>>2]=255}if((c[b+60>>2]&128|0)!=0){c[d+336>>2]=(c[b+60>>2]|0)-128<<1}else{c[d+336>>2]=0}ny(d+168|0,c[b+24>>2]|0);nz(d+168|0,c[b+28>>2]>>2);nA(d+168|0,c[b+20>>2]|0);c[d+184>>2]=c[b+32>>2];nx(d+96|0,0);nx(d+112|0,0);h[d+264>>3]=+hA(g,c[d+32>>2]|0,c[f+1280>>2]&4096);if(+h[d+24>>3]<1.0){i=4990}else{if((e|0)==0){i=4990}}if((i|0)==4990){h[d+24>>3]=+h[d+264>>3]}}}while(0);c[d+56>>2]=0;c[d+200>>2]=0;c[d+144>>2]=0;c[d+164>>2]=0;c[d+160>>2]=1;return}function h5(b){b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;e=i;i=i+256|0;f=e|0;g=b;b=g|0;h=g+1536|0;j=h|0;k=0;p7(b+56|0,-1|0,256);a[f|0]=0;c[b+360>>2]=h6(g,k,0)|0;l=1;while(1){m=0;while(1){if((m|0)>=(c[j+156>>2]|0)){break}if((d[b+56+m|0]|0|0)==255){n=4997;break}m=m+1|0}if((n|0)==4997){n=0}if((m|0)==(c[j+156>>2]|0)){break}if((l|0)>=16){break}k=m;a[f+l|0]=k&255;c[b+360+(l<<4)>>2]=h6(g,k,l)|0;if((c[b+360+(l<<4)>>2]|0)>0){l=l+1|0}}c[h+6412>>2]=l;m=0;while(1){if((m|0)>=(c[h+6412>>2]|0)){break}c[h+6416+(m<<3)>>2]=d[f+m|0]|0;c[h+6416+(m<<3)+4>>2]=c[b+360+(m<<4)>>2];m=m+1|0}i=e;return 0}function h6(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0.0,D=0.0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0,R=0,S=0,T=0,U=0,V=0,W=0,X=0,Y=0,Z=0,_=0,$=0;g=b;b=e;e=f;f=g|0;i=g+1536|0;j=i|0;k=0;l=0;while(1){if((l|0)<(c[j+156>>2]|0)){m=(d[j+952+l|0]|0)>=(c[j+128>>2]|0)}else{m=0}if(!m){break}l=l+1|0}if((l|0)>=(c[j+156>>2]|0)){c[j+156>>2]=0}if((c[j+156>>2]|0)==0){n=0;o=n;return o|0}m=pT(4,c[j+156>>2]|0)|0;l=0;while(1){if((l|0)>=(c[j+156>>2]|0)){break}if((d[j+952+l|0]|0)>=(c[j+128>>2]|0)){p=1}else{if((c[c[(c[j+168>>2]|0)+(d[j+952+l|0]<<2)>>2]>>2]|0)!=0){q=c[c[(c[j+168>>2]|0)+(d[j+952+l|0]<<2)>>2]>>2]|0}else{q=1}p=q}c[m+(l<<2)>>2]=pT(1,p)|0;l=l+1|0}p=pT(4,c[j+136>>2]|0)|0;q=pT(4,c[j+136>>2]|0)|0;r=0;s=0;t=c[j+164>>2]|0;u=c[j+152>>2]|0;v=c[j+148>>2]|0;w=~~+h[i+1248>>3];x=-1;l=b-1|0;y=0;z=0;A=0;B=0;C=0.0;D=0.0;E=0;L6300:while(1){F=l+1|0;l=F;if(F>>>0>=(c[j+156>>2]|0)>>>0){do{if((c[j+160>>2]|0)>(c[j+156>>2]|0)){G=5035}else{if((d[j+952+(c[j+160>>2]|0)|0]|0)>=(c[j+128>>2]|0)){G=5035;break}if((h2(g,c[j+160>>2]|0)|0)==(e|0)){l=c[j+160>>2]|0}else{l=b}}}while(0);if((G|0)==5035){G=0;l=c[i+6416+(e<<3)>>2]|0}H=d[j+952+l|0]|0;if((H|0)==255){G=5041;break}}H=d[j+952+l|0]|0;F=i+1292+(l*20|0)|0;if((b|0)!=0){if((d[f+56+l|0]|0)!=255){G=5045;break}}a[f+56+l|0]=e&255;if((H|0)>=(c[j+128>>2]|0)){if((H|0)==255){l=c[j+156>>2]|0;continue}else{continue}}if((A|0)<(c[c[(c[j+168>>2]|0)+(H<<2)>>2]>>2]|0)){if((a[(c[m+(l<<2)>>2]|0)+A|0]|0)!=0){G=5052;break}}c[F+8>>2]=t;c[F+4>>2]=u;c[F>>2]=v;c[F+12>>2]=~~(C+ +h[i+1256>>3]*+(y|0)/+(u|0));do{if((c[F+16>>2]|0)==0){if((l|0)==0){break}if((l|0)==(b|0)){D=C+ +h[i+1256>>3]*+(y|0)/+(u|0)}c[F+16>>2]=A}}while(0);F=c[c[(c[j+168>>2]|0)+(H<<2)>>2]>>2]|0;I=A;A=0;while(1){if((I|0)>=(F|0)){break}if((u|0)<20){u=20}if((z|0)>512){G=5063;break L6300}if((r|0)==0){if((a[(c[m+(l<<2)>>2]|0)+I|0]|0)!=0){G=5066;break L6300}}J=(c[m+(l<<2)>>2]|0)+I|0;a[J]=(a[J]|0)+1&255;k=0;J=0;while(1){if((J|0)>=(c[j+136>>2]|0)){break}if((I|0)<(c[c[(c[j+172>>2]|0)+(c[(c[(c[j+168>>2]|0)+(H<<2)>>2]|0)+4+(J<<2)>>2]<<2)>>2]>>2]|0)){K=(c[(c[i+172>>2]|0)+(c[(c[(c[i+168>>2]|0)+(d[j+952+l|0]<<2)>>2]|0)+4+(J<<2)>>2]<<2)>>2]|0)+4+(I<<3)|0;if((E|0)!=0){L=0;M=0;N=0;O=0}else{O=d[K+3|0]|0;N=d[K+4|0]|0;M=d[K+5|0]|0;L=d[K+6|0]|0}if((O|0)==16){G=5076}else{if((M|0)==16){G=5076}}if((G|0)==5076){G=0;if((O|0)==16){P=N}else{P=L}t=P;if((t|0)>(c[i+1272>>2]|0)){Q=c[i+1272>>2]|0}else{if((t|0)<0){R=0}else{R=t}Q=R}t=Q}if((O|0)==17){G=5088}else{if((M|0)==17){G=5088}}if((G|0)==5088){G=0;if((O|0)==17){S=N}else{S=L}T=S;while(1){if((T|0)!=0){G=5093;break}K=B;T=K;if((K|0)==0){G=5113;break}}if((G|0)==5093){G=0;B=T;K=(T&240)>>4;U=T&15;if((c[i+1280>>2]&32|0)!=0){do{if((U|0)==15){if((K|0)==0){G=5097;break}t=t+K|0}else{G=5097}}while(0);if((G|0)==5097){G=0;do{if((K|0)==15){if((U|0)==0){G=5100;break}t=t-U|0}else{G=5100}}while(0);if((G|0)==5100){G=0;if((c[i+1280>>2]&64|0)!=0){t=t+(ac(K-U|0,v)|0)|0}else{t=t+(ac(K-U|0,v-1|0)|0)|0}}}}else{if((c[i+1280>>2]&64|0)!=0){t=t+(ac(K-U|0,v)|0)|0}else{t=t+(ac(K-U|0,v-1|0)|0)|0}}}else if((G|0)==5113){G=0}}if((O|0)==15){if((N|0)!=0){G=5119}else{G=5117}}else{G=5117}do{if((G|0)==5117){G=0;if((M|0)!=15){break}if((L|0)!=0){G=5119}}}while(0);if((G|0)==5119){G=0;if((O|0)==15){V=N}else{V=L}T=V;y=y+(ac(ac(z,v)|0,w)|0)|0;z=0;if((T|0)!=0){do{if((c[f+28>>2]&1|0)!=0){G=5125}else{if((T|0)<32){G=5125;break}C=C+ +h[i+1256>>3]*+(y|0)/+(u|0);y=0;u=T}}while(0);if((G|0)==5125){G=0;v=T}}}if((O|0)==126){O=163}if((M|0)==126){M=163}if((O|0)==163){if((N|0)!=0){G=5137}else{G=5135}}else{G=5135}do{if((G|0)==5135){G=0;if((M|0)!=163){break}if((L|0)!=0){G=5137}}}while(0);if((G|0)==5137){G=0;if((O|0)==163){W=N}else{W=L}T=W;y=y+(ac(ac(z,v)|0,w)|0)|0;z=0;v=T}if((O|0)==171){if((N|0)!=0){G=5145}else{G=5143}}else{G=5143}do{if((G|0)==5143){G=0;if((M|0)!=171){break}if((L|0)!=0){G=5145}}}while(0);if((G|0)==5145){G=0;if((O|0)==171){X=N}else{X=L}T=X;y=y+(ac(ac(z,v)|0,w)|0)|0;z=0;C=C+ +h[i+1256>>3]*+(y|0)/+(u|0);y=0;u=T}if((O|0)==135){if((N|0)!=0){G=5153}else{G=5151}}else{G=5151}do{if((G|0)==5151){G=0;if((M|0)!=135){break}if((L|0)!=0){G=5153}}}while(0);if((G|0)==5153){G=0;if((O|0)==135){Y=N}else{Y=L}T=Y;y=y+(ac(ac(z,v)|0,w)|0)|0;z=0;C=C+ +h[i+1256>>3]*+(y|0)/+(u|0);y=0;if(((T&240)>>4|0)==0){u=u-(T&15)|0;if((u|0)<32){u=32}}else{if(((T&240)>>4|0)==1){u=u+(T&15)|0;if((u|0)>255){u=255}}else{u=T}}}if((O|0)==11){G=5169}else{if((M|0)==11){G=5169}}if((G|0)==5169){G=0;if((O|0)==11){Z=N}else{Z=L}x=Z;F=0}if((O|0)==13){G=5175}else{if((M|0)==13){G=5175}}if((G|0)==5175){G=0;if((O|0)==13){_=N}else{_=L}T=_;A=(((T&240)>>4)*10|0)+(T&15)|0;F=0}if((O|0)==14){G=5181}else{if((M|0)==14){G=5181}}if((G|0)==5181){G=0;if((O|0)==14){$=N}else{$=L}T=$;if((T>>4|0)==14){k=T&15;y=y+(ac(ac(k,v)|0,w)|0)|0}if((T>>4|0)==6){U=T&15;T=U;if((U|0)!=0){if((c[p+(J<<2)>>2]|0)!=0){U=p+(J<<2)|0;K=(c[U>>2]|0)-1|0;c[U>>2]=K;if((K|0)!=0){s=J+1|0}else{r=r-1|0;if((c[i+1280>>2]&1|0)!=0){c[q+(J<<2)>>2]=I+1}}}else{if((c[q+(J<<2)>>2]|0)<=(I|0)){c[p+(J<<2)>>2]=T;s=J+1|0;r=r+1|0}}}else{c[q+(J<<2)>>2]=I}}}}J=J+1|0}E=0;if((s|0)!=0){J=s-1|0;s=J;I=(c[q+(J<<2)>>2]|0)-1|0;s=0}I=I+1|0;z=z+1|0}do{if((A|0)!=0){if((k|0)==0){break}E=1}}while(0);if((x|0)>=0){l=x-1|0;x=-1}y=y+(ac(ac(z,v)|0,w)|0)|0;z=0}if((G|0)==5052){G=5214}else if((G|0)!=5063)if((G|0)==5045){G=5214}else if((G|0)==5041){G=5214}else if((G|0)==5066){z=z-1|0}if((G|0)==5214){I=A}c[f+360+(e<<4)+12>>2]=a[(c[m+(l<<2)>>2]|0)+I|0]|0;c[f+360+(e<<4)+8>>2]=I;c[f+360+(e<<4)+4>>2]=l;pU(q);pU(p);l=c[j+156>>2]|0;while(1){j=l;l=j-1|0;if((j|0)==0){break}pU(c[m+(l<<2)>>2]|0)}pU(m);C=C-D;y=y+(ac(ac(z,v)|0,w)|0)|0;n=~~(C+ +h[i+1256>>3]*+(y|0)/+(u|0));o=n;return o|0}function h7(a,b,c){a=a|0;b=b|0;c=c|0;return}function h8(a,b){a=a|0;b=b|0;return 0}function h9(a){a=a|0;return 0}function ia(a){a=a|0;return 0}function ib(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;f=d;d=e;e=c[a+8100>>2]|0;a=e+(b*408|0)|0;b=f;if((b|0)==242){c[e+1228>>2]=15;c[e+1232>>2]=0;c[a+12>>2]=0;return}else if((b|0)==243|(b|0)==244|(b|0)==245|(b|0)==246|(b|0)==247|(b|0)==248|(b|0)==249|(b|0)==250|(b|0)==251|(b|0)==252|(b|0)==253|(b|0)==254){c[e+1228>>2]=f-240;c[e+1232>>2]=c[e+1232>>2]&65280|d;c[a+12>>2]=0;return}else if((b|0)==240){if((d|0)<15){c[a+12>>2]=d}c[e+1228>>2]=15;c[e+1232>>2]=0;return}else if((b|0)==241){c[e+1232>>2]=c[e+1232>>2]&255|d<<8;return}else{return}}function ic(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a|0;a=c[(c[d+956>>2]|0)+(b<<3)+4>>2]|0;if((a|0)<0){e=-1;f=e;return f|0}else{e=c[(c[d+960>>2]|0)+(a*116|0)+4>>2]|0;f=e;return f|0}return 0}function id(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;c[d+8100>>2]=im(b)|0;if((c[d+8100>>2]|0)==0){e=-1;f=e;return f|0}else{e=0;f=e;return f|0}return 0}function ie(a){a=a|0;il(c[a+8100>>2]|0);return 0}function ig(a){a=a|0;iM(c[(c[a+8100>>2]|0)+1236>>2]|0);return 0}function ih(b,d,e){b=b|0;d=d|0;e=e|0;var f=0;f=(c[b+8100>>2]|0)+(d*408|0)|0;d=f+16|0;b=e;p3(d|0,b|0,392)|0;c[f+8>>2]=0;c[f>>2]=a[f+26|0]|0;return}function ii(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=b;b=d;d=e;e=c[g+8100>>2]|0;h=e+(b*408|0)|0;d=d-12|0;d=d+(a[(c[g+8092>>2]|0)+((c[h+12>>2]|0)*88|0)+8+(c[h+8>>2]|0)|0]|0)|0;c[e+(b*408|0)+4>>2]=~~(4095.0/+T(2.0,+((+(d|0)+ +(f|0)/100.0)/12.0)));return}function ij(a,b,d,e,f,g){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0;h=b;b=c[a+8100>>2]|0;if((h|0)!=0){ik(b);iT(c[b+1236>>2]|0,h,d,e,f,g);return}else{return}}function ik(d){d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=d;d=127;f=0;g=0;while(1){if((g|0)>=3){break}h=e+(g*408|0)|0;i=h+24+(c[h+8>>2]<<3)|0;j=(c[h+4>>2]|0)+(b[i>>1]|0)|0;iS(c[e+1236>>2]|0,g<<1|0,j&255);iS(c[e+1236>>2]|0,(g<<1)+1|0,j>>8);iS(c[e+1236>>2]|0,g+8|0,a[i+2|0]|0);f=f+(a[i+3|0]|0)|0;if((c[i+4>>2]&32|0)!=0){d=d&~(1<<g)}if((c[i+4>>2]&64|0)!=0){d=d&~(8<<g)}i=h+8|0;c[i>>2]=(c[i>>2]|0)+1;if((c[h+8>>2]|0)>=(c[h+20>>2]|0)){c[h+8>>2]=c[h+16>>2]}g=g+1|0}iS(c[e+1236>>2]|0,11,c[e+1232>>2]&255);iS(c[e+1236>>2]|0,12,(c[e+1232>>2]&65280)>>8);iS(c[e+1236>>2]|0,13,c[e+1228>>2]|0);iS(c[e+1236>>2]|0,6,f);iS(c[e+1236>>2]|0,7,d);return}function il(a){a=a|0;var b=0;b=a;iM(c[b+1236>>2]|0);iN(c[b+1236>>2]|0);pU(b);return}function im(a){a=a|0;var b=0,d=0,e=0;b=pT(1,1240)|0;do{if((b|0)!=0){c[b+1236>>2]=iL(1773400,1,a)|0;if((c[b+1236>>2]|0)==0){pU(b);break}else{d=b;e=d;return e|0}}}while(0);d=0;e=d;return e|0}function io(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=a;a=b;b=e|0;f=(c[b+960>>2]|0)+(a*116|0)|0;if(a>>>0>=(c[b+952>>2]|0)>>>0){return}if((d|0)!=0){oE(e,a,0)}a=b+948|0;c[a>>2]=(c[a>>2]|0)-1;a=(c[b+956>>2]|0)+(c[f+4>>2]<<3)|0;c[a>>2]=(c[a>>2]|0)-1;c[(c[b+956>>2]|0)+(c[f>>2]<<3)+4>>2]=-1;p7(f|0,0,116);c[f+4>>2]=-1;c[f>>2]=-1;return}function ip(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=a;a=b;b=d|0;c[b+940>>2]=a;a=oI(d,-1)|0;c[b+944>>2]=c[b+940>>2];if((c[d+2816>>2]&32768|0)!=0){e=b+944|0;c[e>>2]=(c[e>>2]|0)+a}else{if((a|0)>(c[b+944>>2]|0)){a=c[b+944>>2]|0}}e=oI(d,a)|0;c[b+952>>2]=e;a=e;c[b+960>>2]=pT(c[b+952>>2]|0,116)|0;do{if((c[b+960>>2]|0)!=0){e=0;while(1){if((e|0)>=(c[b+952>>2]|0)){break}c[(c[b+960>>2]|0)+(e*116|0)>>2]=-1;c[(c[b+960>>2]|0)+(e*116|0)+4>>2]=-1;e=e+1|0}c[b+956>>2]=pS(c[b+944>>2]<<3)|0;if((c[b+956>>2]|0)==0){pU(c[b+960>>2]|0);break}e=0;while(1){if((e|0)>=(c[b+944>>2]|0)){break}c[(c[b+956>>2]|0)+(e<<3)+4>>2]=-1;c[(c[b+956>>2]|0)+(e<<3)>>2]=0;e=e+1|0}c[b+948>>2]=0;f=0;g=f;return g|0}}while(0);f=-1;g=f;return g|0}function iq(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;if(a>>>0>=(c[d+944>>2]|0)>>>0){e=-1;f=e;return f|0}b=c[(c[d+956>>2]|0)+(a<<3)+4>>2]|0;if(b>>>0>=(c[d+952>>2]|0)>>>0){e=-1;f=e;return f|0}else{e=b;f=e;return f|0}return 0}function ir(a){a=a|0;var b=0;b=a|0;if((c[b+944>>2]|0)<1){return}else{c[b+952>>2]=0;c[b+948>>2]=0;c[b+944>>2]=0;c[b+940>>2]=0;pU(c[b+960>>2]|0);pU(c[b+956>>2]|0);return}}function is(a){a=a|0;var b=0;b=a;a=b|0;if((c[a+944>>2]|0)<1){return}oI(b,c[a+952>>2]|0)|0;p7(c[a+960>>2]|0,0,(c[a+952>>2]|0)*116|0|0);b=0;while(1){if((b|0)>=(c[a+952>>2]|0)){break}c[(c[a+960>>2]|0)+(b*116|0)>>2]=-1;c[(c[a+960>>2]|0)+(b*116|0)+4>>2]=-1;b=b+1|0}b=0;while(1){if((b|0)>=(c[a+944>>2]|0)){break}c[(c[a+956>>2]|0)+(b<<3)+4>>2]=-1;c[(c[a+956>>2]|0)+(b<<3)>>2]=0;b=b+1|0}c[a+948>>2]=0;return}function it(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;b=d|0;e=iq(b,a)|0;f=e;if((e|0)<0){return}else{oE(d,f,0);d=b+948|0;c[d>>2]=(c[d>>2]|0)-1;d=(c[b+956>>2]|0)+(c[(c[b+960>>2]|0)+(f*116|0)+4>>2]<<3)|0;c[d>>2]=(c[d>>2]|0)-1;c[(c[b+956>>2]|0)+(a<<3)+4>>2]=-1;p7((c[b+960>>2]|0)+(f*116|0)|0,0,116);c[(c[b+960>>2]|0)+(f*116|0)+4>>2]=-1;c[(c[b+960>>2]|0)+(f*116|0)>>2]=-1;return}}function iu(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;f=b;b=d;d=e;e=f|0;g=iq(e,b)|0;h=g;if((g|0)<0){return}g=c[(c[e+960>>2]|0)+(h*116|0)+4>>2]|0;do{if((g|0)<64){if((a[e+876+g|0]|0)==0){break}d=0}}while(0);oE(f,h,d);if((d|0)!=0){return}if((b|0)<(c[e+940>>2]|0)){return}io(f,h,1);return}function iv(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=a;a=iq(d|0,b)|0;if((a|0)<0){return}else{oH(d,a,c);return}}function iw(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0;e=a;a=iq(e|0,b)|0;if((a|0)<0){return}else{oK(e,a,c,d);return}}function ix(a,b){a=a|0;b=b|0;var c=0,d=0,e=0;c=a;a=iq(c|0,b)|0;if((a|0)<0){d=-1;e=d;return e|0}else{d=oB(c,a)|0;e=d;return e|0}return 0}function iy(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0;e=a;a=d;d=e|0;f=iq(d,b)|0;b=f;if((f|0)<0){return}f=(c[d+960>>2]|0)+(b*116|0)|0;if((c[f+48>>2]|0)==(a|0)){return}else{d=c[f+28>>2]|0;g=c[f+36>>2]|0;oD(e,b,a);oA(e,b,d,g);return}}function iz(a,b,d,e,f,g,h,i){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0;j=a;a=b;b=d;d=e;e=f;f=g;g=h;h=i;i=j|0;if(a>>>0>=(c[i+944>>2]|0)>>>0){k=-1;l=k;return l|0}if((b|0)<0){d=-1}if((g|0)!=0){m=0;while(1){if((m|0)>=(c[i+952>>2]|0)){break}iA(j,m,a,b,d,e,g,h);m=m+1|0}}m=c[(c[i+956>>2]|0)+(a<<3)+4>>2]|0;if((m|0)>-1){if((c[(c[i+960>>2]|0)+(m*116|0)+56>>2]|0)!=0){h=iD(j,a)|0;a=c[i+940>>2]|0;do{g=a;a=g+1|0;}while((c[(c[i+956>>2]|0)+(g<<3)+4>>2]|0)>-1);g=a-1|0;a=g;c[(c[i+960>>2]|0)+(m*116|0)>>2]=g;c[(c[i+956>>2]|0)+(a<<3)+4>>2]=m;m=h}}else{m=iD(j,a)|0}if((d|0)<0){io(j,m,1);k=a;l=k;return l|0}else{oD(j,m,d);oF(j,m,e);c[(c[i+960>>2]|0)+(m*116|0)+44>>2]=b;c[(c[i+960>>2]|0)+(m*116|0)+56>>2]=f;k=a;l=k;return l|0}return 0}function iA(a,b,d,e,f,g,h,i){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0;j=a;a=b;b=d;d=f;f=g;g=h;h=i;i=j|0;k=(c[i+960>>2]|0)+(a*116|0)|0;l=c[(c[i+956>>2]|0)+(b<<3)+4>>2]|0;if((c[k+4>>2]|0)!=(b|0)){return}if((c[k+44>>2]|0)!=(e|0)){return}do{if((g|0)==3){m=5421}else{if((g|0)==2){if((c[k+48>>2]|0)==(d|0)){m=5421;break}}if((g|0)!=1){break}if((c[k+12>>2]|0)==(f|0)){m=5421}}}while(0);if((m|0)==5421){if((h|0)!=0){if((a|0)!=(l|0)){m=5424}else{if((c[k+56>>2]|0)!=0){m=5424}}if((m|0)==5424){c[k+56>>2]=h}}else{io(j,a,1)}}return}function iB(a){a=a|0;var b=0,d=0,e=0,f=0;b=a|0;a=-1;d=2147483647;e=0;while(1){if((e|0)>=(c[b+952>>2]|0)){break}f=(c[b+960>>2]|0)+(e*116|0)|0;do{if((c[f>>2]|0)>=(c[b+940>>2]|0)){if((c[f+20>>2]|0)>=(d|0)){break}a=e;d=c[f+20>>2]|0}}while(0);e=e+1|0}c[(c[b+956>>2]|0)+(c[(c[b+960>>2]|0)+(a*116|0)>>2]<<3)+4>>2]=-1;e=(c[b+956>>2]|0)+(c[(c[b+960>>2]|0)+(a*116|0)+4>>2]<<3)|0;c[e>>2]=(c[e>>2]|0)-1;e=b+948|0;c[e>>2]=(c[e>>2]|0)-1;return a|0}function iC(b,c,d){b=b|0;c=c|0;d=d|0;var e=0,f=0;e=b;b=c;c=d-b|0;b=b*15|0;d=0;while(1){if((d|0)>=16){break}f=e;e=f+1|0;a[f]=b&255;b=b+c|0;d=d+1|0}return e|0}function iD(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;b=d|0;e=0;while(1){if((e|0)>=(c[b+952>>2]|0)){break}if((c[(c[b+960>>2]|0)+(e*116|0)>>2]|0)==-1){f=5449;break}e=e+1|0}if((e|0)==(c[b+952>>2]|0)){e=iB(d)|0}d=(c[b+956>>2]|0)+(a<<3)|0;c[d>>2]=(c[d>>2]|0)+1;d=b+948|0;c[d>>2]=(c[d>>2]|0)+1;c[(c[b+960>>2]|0)+(e*116|0)>>2]=a;c[(c[b+960>>2]|0)+(e*116|0)+4>>2]=a;c[(c[b+956>>2]|0)+(a<<3)+4>>2]=e;return e|0}function iE(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a|0;a=iq(e,b)|0;if((a|0)<0){return}else{c[(c[e+960>>2]|0)+(a*116|0)+56>>2]=d;return}}function iF(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=a;a=iq(d|0,b)|0;if((a|0)<0){return}else{oG(d,a,c);return}}function iG(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=a;a=iq(d|0,b)|0;if((a|0)<0){return}else{oA(d,a,c,0);return}}function iH(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=d;d=e|0;f=c[d+940>>2]|0;while(1){if((f|0)>=(c[d+944>>2]|0)){break}g=iq(d,f)|0;h=g;if((g|0)>=0){if((c[(c[d+960>>2]|0)+(h*116|0)+4>>2]|0)==(a|0)){g=b;if((g|0)==3){hv(e,f)}else if((g|0)==0){io(e,h,1)}else if((g|0)==2){hu(e,f)}}}f=f+1|0}return}function iI(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=b;b=a|0;a=iq(b,d)|0;if((a|0)<0){e=-1;f=e;return f|0}if((d|0)<(c[b+940>>2]|0)){e=256;f=e;return f|0}else{e=c[(c[b+960>>2]|0)+(a*116|0)+56>>2]|0;f=e;return f|0}return 0}function iJ(){var a=0,b=0,c=0;a=pS(2056)|0;if((a|0)==0){b=a;c=b;return c|0}else{iK(a);b=a;c=b;return c|0}return 0}function iK(a){a=a|0;var b=0;b=a;p7(b|0,0,512);c[b+2048>>2]=0;c[b+2052>>2]=0;return}function iL(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=a;a=b;b=d;d=pT(1,1236)|0;do{if((d|0)!=0){c[d>>2]=iJ()|0;if((c[d>>2]|0)==0){pU(d);break}c[d+4>>2]=0;f=d+192|0;g=0;while(1){if((g|0)>=16){break}h=c[32704+(g<<2)>>2]|0;i=0;while(1){if((i|0)>=4){break}f=iC(f,c[h+(i<<1<<2)>>2]|0,c[h+((i<<1)+1<<2)>>2]|0)|0;i=i+1|0}g=g+1|0}c[d+16>>2]=(e|0)/(a|0)|0;c[d+12>>2]=b;c[d+76>>2]=0;c[d+144>>2]=d+104;c[d+148>>2]=d+108;c[d+152>>2]=d+112;iM(d);j=d;k=j;return k|0}}while(0);j=0;k=j;return k|0}function iM(a){a=a|0;var b=0;b=a;iS(b,7,63);iS(b,8,0);iS(b,9,0);iS(b,10,0);c[b+168>>2]=65535;c[b+164>>2]=1;c[b+188>>2]=0;c[b+184>>2]=0;c[b+180>>2]=0;iK(c[b>>2]|0);return}function iN(a){a=a|0;var b=0;b=a;iO(c[b>>2]|0);pU(c[b+1220>>2]|0);pU(b);return}function iO(a){a=a|0;pU(a);return}function iP(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0.0;e=a;a=b&15;a=(a<<8)+d|0;if((a|0)<=5){f=0;g=f;return g|0}else{h=+((c[e+16>>2]|0)>>>0>>>0);h=h/(+(a|0)*8.0*+(c[e+12>>2]|0));h=h*2147483648.0;f=~~h;g=f;return g|0}return 0}function iQ(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0.0;d=a;a=b&31;if((a|0)<3){e=0;f=e;return f|0}else{g=+((c[d+16>>2]|0)>>>0>>>0);g=g/(+(a|0)*8.0*+(c[d+12>>2]|0));g=g*32768.0;e=~~g;f=e;return f|0}return 0}function iR(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0.0;e=a;a=b;a=(a<<8)+d|0;if((a|0)<3){f=0;g=f;return g|0}else{h=+((c[e+16>>2]|0)>>>0>>>0);h=h/(+(a|0)*512.0*+(c[e+12>>2]|0));h=h*4294967296.0;f=~~h;g=f;return g|0}return 0}function iS(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=d;d=b;if((d|0)==8){c[e+52>>2]=a&31;c[e+104>>2]=c[8+((a&15)<<2)>>2];if((a&16|0)!=0){c[e+144>>2]=e+116}else{c[e+144>>2]=e+104}return}else if((d|0)==10){c[e+60>>2]=a&31;c[e+112>>2]=c[8+((a&15)<<2)>>2];if((a&16|0)!=0){c[e+152>>2]=e+116}else{c[e+152>>2]=e+112}return}else if((d|0)==2){c[e+28>>2]=a&255;c[e+84>>2]=iP(e,c[e+32>>2]|0,c[e+28>>2]|0)|0;if((c[e+84>>2]|0)==0){c[e+96>>2]=-2147483648}return}else if((d|0)==0){c[e+20>>2]=a&255;c[e+80>>2]=iP(e,c[e+24>>2]|0,c[e+20>>2]|0)|0;if((c[e+80>>2]|0)==0){c[e+92>>2]=-2147483648}return}else if((d|0)==6){c[e+44>>2]=a&31;c[e+156>>2]=iQ(e,c[e+44>>2]|0)|0;if((c[e+156>>2]|0)==0){c[e+160>>2]=0;c[e+168>>2]=65535}return}else if((d|0)==5){c[e+40>>2]=a&15;c[e+88>>2]=iP(e,c[e+40>>2]|0,c[e+36>>2]|0)|0;if((c[e+88>>2]|0)==0){c[e+100>>2]=-2147483648}return}else if((d|0)==7){c[e+48>>2]=a&255;c[e+120>>2]=(a&1|0)!=0?65535:0;c[e+124>>2]=(a&2|0)!=0?65535:0;c[e+128>>2]=(a&4|0)!=0?65535:0;c[e+132>>2]=(a&8|0)!=0?65535:0;c[e+136>>2]=(a&16|0)!=0?65535:0;c[e+140>>2]=(a&32|0)!=0?65535:0;return}else if((d|0)==3){c[e+32>>2]=a&15;c[e+84>>2]=iP(e,c[e+32>>2]|0,c[e+28>>2]|0)|0;if((c[e+84>>2]|0)==0){c[e+96>>2]=-2147483648}return}else if((d|0)==1){c[e+24>>2]=a&15;c[e+80>>2]=iP(e,c[e+24>>2]|0,c[e+20>>2]|0)|0;if((c[e+80>>2]|0)==0){c[e+92>>2]=-2147483648}return}else if((d|0)==9){c[e+56>>2]=a&31;c[e+108>>2]=c[8+((a&15)<<2)>>2];if((a&16|0)!=0){c[e+148>>2]=e+116}else{c[e+148>>2]=e+108}return}else if((d|0)==13){c[e+72>>2]=a&15;c[e+180>>2]=0;c[e+184>>2]=0;c[e+188>>2]=a&15;return}else if((d|0)==4){c[e+36>>2]=a&255;c[e+88>>2]=iP(e,c[e+40>>2]|0,c[e+36>>2]|0)|0;if((c[e+88>>2]|0)==0){c[e+100>>2]=-2147483648}return}else if((d|0)==12){c[e+68>>2]=a&255;c[e+176>>2]=iR(e,c[e+68>>2]|0,c[e+64>>2]|0)|0;return}else if((d|0)==11){c[e+64>>2]=a&255;c[e+176>>2]=iR(e,c[e+68>>2]|0,c[e+64>>2]|0)|0;return}else{return}}function iT(a,b,d,e,f,g){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;h=a;a=b;b=d;d=e;e=f;f=g;g=a;i=b;while(1){if((b|0)<=0){break}j=iU(h)|0;if((f|0)!=0){k=a;a=k+4|0;c[k>>2]=ac(j,e)|0}k=a;a=k+4|0;c[k>>2]=ac(j,d)|0;b=b-1|0}iZ(h,g,i);return}function iU(a){a=a|0;var b=0,e=0,f=0,g=0;b=a;if((c[b+160>>2]&-65536|0)!=0){a=iW(b)|0;e=b+168|0;c[e>>2]=c[e>>2]^a;a=b+160|0;c[a>>2]=c[a>>2]&65535}a=c[b+168>>2]|0;c[b+116>>2]=c[8+((d[b+192+(c[b+188>>2]<<6)+(c[b+184>>2]<<5)+((c[b+180>>2]|0)>>>27)|0]|0)<<2)>>2];e=(c[b+92>>2]>>31|c[b+120>>2])&(a|c[b+132>>2]);f=c[c[b+144>>2]>>2]&e;e=(c[b+96>>2]>>31|c[b+124>>2])&(a|c[b+136>>2]);f=f+(c[c[b+148>>2]>>2]&e)|0;e=(c[b+100>>2]>>31|c[b+128>>2])&(a|c[b+140>>2]);f=f+(c[c[b+152>>2]>>2]&e)|0;e=b+92|0;c[e>>2]=(c[e>>2]|0)+(c[b+80>>2]|0);e=b+96|0;c[e>>2]=(c[e>>2]|0)+(c[b+84>>2]|0);e=b+100|0;c[e>>2]=(c[e>>2]|0)+(c[b+88>>2]|0);e=b+160|0;c[e>>2]=(c[e>>2]|0)+(c[b+156>>2]|0);e=b+180|0;c[e>>2]=(c[e>>2]|0)+(c[b+176>>2]|0);if(0!=(c[b+184>>2]|0)){g=f;return g|0}if((c[b+180>>2]|0)>>>0<(c[b+176>>2]|0)>>>0){c[b+184>>2]=1}g=f;return g|0}function iV(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;return}function iW(a){a=a|0;var b=0;b=a;a=c[b+164>>2]&1^(c[b+164>>2]|0)>>>2&1;c[b+164>>2]=(c[b+164>>2]|0)>>>1|a<<16;return((a|0)!=0?0:65535)|0}function iX(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0;d=b;b=c[a+8100>>2]|0;a=-1;d=d+1|0;e=0;while(1){if((e|0)>=9){break}if((c[b+4+(e<<2)>>2]|0)==(d|0)){f=5616;break}if((c[b+4+(e<<2)>>2]|0)<0){f=5618;break}e=e+1|0}if((f|0)==5618){a=e}else if((f|0)==5616){g=e;h=g;return h|0}if((a|0)!=-1){c[b+4+(a<<2)>>2]=d}g=a;h=g;return h|0}function iY(a){a=a|0;var b=0;b=c[a+8100>>2]|0;a=0;while(1){if((a|0)>=9){break}c[b+4+(a<<2)>>2]=-1;a=a+1|0}return}function iZ(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=a;a=b;b=d;d=i_(e,a,b)|0;if((b|0)>0){f=a;a=f+4|0;c[f>>2]=(c[e+1228>>2]|0)+((c[e+1232>>2]|0)+(c[e+1232>>2]|0))+(c[d>>2]|0)>>2;if((b|0)>1){f=a;a=f+4|0;c[f>>2]=(c[e+1232>>2]|0)+((c[d>>2]|0)+(c[d>>2]|0))+(c[d+4>>2]|0)>>2}}c[e+1228>>2]=c[d+(b-2<<2)>>2];c[e+1232>>2]=c[d+(b-1<<2)>>2];e=2;while(1){if((e|0)>=(b|0)){break}f=a;a=f+4|0;c[f>>2]=(c[d>>2]|0)+((c[d+4>>2]|0)+(c[d+4>>2]|0))+(c[d+8>>2]|0)>>2;d=d+4|0;e=e+1|0}return}function i_(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=a;a=d;if((a|0)>(c[e+1224>>2]|0)){c[e+1220>>2]=pS(a<<2)|0;c[e+1224>>2]=a}d=c[e+1220>>2]|0;f=b;b=a<<2;p3(d|0,f|0,b)|0;return c[e+1220>>2]|0}function i$(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;c[d+8100>>2]=i9()|0;if((c[d+8100>>2]|0)==0){e=-1;f=e;return f|0}else{iY(d);a=ne(8,3579545,b)|0;c[c[d+8100>>2]>>2]=a;e=(a|0)!=0|0;f=e;return f|0}return 0}function i0(a){a=a|0;var b=0;b=a;a=c[b+8100>>2]|0;i1(b)|0;nh(c[a>>2]|0);i8(a);return 0}function i1(a){a=a|0;var b=0;b=a;nb(c[c[b+8100>>2]>>2]|0);iY(b);return 0}function i2(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0;f=a;a=b;b=e;e=c[f+8100>>2]|0;g=iX(f,a)|0;a=g;if((g|0)<0){return}g=0;while(1){if((g|0)>=10){break}i7(e,(c[3952+(g<<2)>>2]|0)+(c[3880+(((g|0)%2|0)*36|0)+(a<<2)>>2]|0)|0,d[b+g|0]|0)|0;g=g+1|0}i7(e,(c[998]|0)+a|0,d[b+10|0]|0)|0;i7(e,a+176|0,(i6(e,a+176|0)|0)&255&-33)|0;return}function i3(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;f=a;a=b;b=d;d=c[f+8100>>2]|0;g=iX(f,a)|0;a=g;if((g|0)<0){return}g=(b|0)%12|0;f=c[72+(g<<2)>>2]|0;h=f+((ac((c[72+(g+1<<2)>>2]|0)-(c[72+(g<<2)>>2]|0)|0,e)|0)/100|0)|0;e=((b|0)/12|0)-2|0;if((e|0)<0){e=0}i7(d,a+160|0,h&255)|0;i7(d,a+176|0,32|e<<2&28|h>>8&3)|0;return}function i4(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0;e=a;a=b;b=d;d=c[e+8100>>2]|0;f=iX(e,a)|0;a=f;if((f|0)<0){return}if((b|0)>63){b=63}if((i6(d,a+200|0)|0)<<24>>24!=0){g=c[3880+(a<<2)>>2]|0;h=(i6(d,g+64|0)|0)&255;f=d;e=g+64|0;i=h&192|63-b;i7(f,e,i)|0}g=c[3916+(a<<2)>>2]|0;h=(i6(d,g+64|0)|0)&255;i7(d,g+64|0,h&192|63-b)|0;return}function i5(a,b,d,e,f,g){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0;h=b;if((h|0)!=0){m9(c[c[a+8100>>2]>>2]|0,h,d,e,f,g);return}else{return}}function i6(a,b){a=a|0;b=b|0;var d=0;d=a;nj(c[d>>2]|0,0,b)|0;return nc(c[d>>2]|0,1)|0}function i7(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;nj(c[e>>2]|0,0,b)|0;return nj(c[e>>2]|0,1,d)|0}function i8(a){a=a|0;pU(a);return}function i9(){var a=0,b=0,d=0,e=0;a=pS(40)|0;if((a|0)==0){b=0;d=b;return d|0}e=0;while(1){if((e|0)>=9){break}c[a+4+(e<<2)>>2]=-1;e=e+1|0}b=a;d=b;return d|0}function ja(a){a=a|0;return(aN(a|0)|0)&255|0}function jb(a){a=a|0;return(aN(a|0)|0)&255|0}function jc(a){a=a|0;var b=0;b=a;a=(ja(b)|0)&255;return(((ja(b)|0)&255)<<8|a)&65535|0}function jd(a){a=a|0;var b=0;b=a;a=(ja(b)|0)&255;return(a<<8|(ja(b)|0)&255)&65535|0}function je(a){a=a|0;var b=0,c=0;b=a;a=(ja(b)|0)&255;c=(ja(b)|0)&255;return((ja(b)|0)&255)<<16|c<<8|a|0}function jf(a){a=a|0;var b=0,c=0;b=a;a=(ja(b)|0)&255;c=(ja(b)|0)&255;return a<<16|c<<8|(ja(b)|0)&255|0}function jg(a){a=a|0;var b=0,c=0,d=0;b=a;a=(ja(b)|0)&255;c=(ja(b)|0)&255;d=(ja(b)|0)&255;return((ja(b)|0)&255)<<24|d<<16|c<<8|a|0}function jh(a){a=a|0;var b=0;b=a;return((d[b+1|0]|0)<<8|(d[b|0]|0))&65535|0}function ji(a){a=a|0;var b=0;b=a;return((d[b|0]|0)<<8|(d[b+1|0]|0))&65535|0}function jj(a){a=a|0;var b=0;b=a;return(d[b+3|0]|0)<<24|(d[b+2|0]|0)<<16|(d[b+1|0]|0)<<8|(d[b|0]|0)|0}function jk(a){a=a|0;var b=0;b=a;return(d[b|0]|0)<<24|(d[b+1|0]|0)<<16|(d[b+2|0]|0)<<8|(d[b+3|0]|0)|0}function jl(a){a=a|0;var b=0,c=0,d=0;b=a;a=(ja(b)|0)&255;c=(ja(b)|0)&255;d=(ja(b)|0)&255;return a<<24|c<<16|d<<8|(ja(b)|0)&255|0}function jm(a,b){a=a|0;b=b|0;aw(b&255|0,a|0)|0;return}function jn(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;jm(c,a&65535&255);jm(c,(a&65280)>>8&255);return}function jo(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;jm(c,(a&65280)>>8&255);jm(c,a&65535&255);return}function jp(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;jm(c,a&255);jm(c,(a&65280)>>>8&255);jm(c,(a&16711680)>>>16&255);jm(c,(a&-16777216)>>>24&255);return}function jq(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;jm(c,(a&-16777216)>>>24&255);jm(c,(a&16711680)>>>16&255);jm(c,(a&65280)>>>8&255);jm(c,a&255);return}function jr(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;d=i;i=i+1024|0;e=d|0;f=a;a=b;b=c;do{if((b|0)>1024){g=1024}else{g=b}c=bB(e|0,1,g|0,a|0)|0;ay(e|0,1,c|0,f|0)|0;b=b-c|0;if((c|0)>0){h=(b|0)>0}else{h=0}}while(h);i=d;return 0}function js(a,b){a=a|0;b=b|0;var c=0,d=0;c=b;do{if((c|0)==0){d=-1}else{if((jt(a,c)|0)<0){d=-1;break}else{d=0;break}}}while(0);return d|0}function jt(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;f=i;i=i+40|0;g=f|0;h=b;b=e;e=0;if((ju(h,g)|0)==0){j=-1;k=j;i=f;return k|0}if((d[g|0]|0|0)==0){j=-1;k=j;i=f;return k|0}l=jv(h,g)|0;h=l;if((l|0)==0){l=c[q>>2]|0;bE(l|0,23600,(m=i,i=i+1|0,i=i+7>>3<<3,c[m>>2]=0,m)|0)|0;i=m;j=-1;k=j;i=f;return k|0}l=0;n=0;o=d[g|0]|0;if((o|0)==127){n=1;l=j2(h,d[g+1|0]|0,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((o|0)==6){n=1;l=j2(h,0,1,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((o|0)==5){n=1;l=j2(h,0,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((o|0)!=7)if((o|0)==9){n=1;l=j2(h,d[g+1|0]|0,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((o|0)==4){n=1;l=j5(h,c[g+16>>2]|0,c[g+32>>2]|0)|0}else if((o|0)==1|(o|0)==2){n=1;l=h}else if((o|0)==8){n=1;l=j2(h,d[g+1|0]|0,1,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((o|0)==3){n=1;l=kk(h,c[g+16>>2]|0,c[g+32>>2]|0)|0}if((l|0)==0){n=c[q>>2]|0;bE(n|0,29120,(m=i,i=i+1|0,i=i+7>>3<<3,c[m>>2]=0,m)|0)|0;i=m;e=1}else{while(1){n=aR(g+2|0,47)|0;if((n|0)==0){break}a[n]=95}n=ay(l|0,1,c[g+32>>2]|0,b|0)|0;if((n|0)!=(c[g+32>>2]|0)){g=c[q>>2]|0;n=bx(c[(bw()|0)>>2]|0)|0;bE(g|0,27568,(m=i,i=i+8|0,c[m>>2]=n,m)|0)|0;i=m;e=1}if((l|0)!=(h|0)){pU(l)}}pU(h);j=e;k=j;i=f;return k|0}function ju(b,d){b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0;e=b;b=d;bK(e|0,8,1)|0;d=((jg(e)|0)>>>0)/36|0;f=jg(e)|0;g=jg(e)|0;jg(e)|0;g=jg(e)|0;bK(e|0,68,1)|0;g=0;while(1){if((g|0)>=(d|0)){h=5776;break}i=(ja(e)|0)&255;if((i|0)==0){h=5766;break}a[b|0]=i&127;bB(b+2|0,1,11,e|0)|0;a[b+14|0]=0;c[b+32>>2]=jg(e)|0;jg(e)|0;jg(e)|0;i=jg(e)|0;c[b+16>>2]=jg(e)|0;c[b+36>>2]=jg(e)|0;if((i|0)!=1){if((c[b+36>>2]&-2147483648|0)==0){h=5771;break}}g=g+1|0}if((h|0)==5776){return 1}else if((h|0)==5766){return 1}else if((h|0)==5771){c[b+28>>2]=i>>16;a[b+1|0]=(i&65280)>>8&255;i=b+36|0;c[i>>2]=c[i>>2]&2147483647;i=b+36|0;c[i>>2]=(c[i>>2]|0)+f;return 1}return 0}function jv(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0;d=a;a=b;b=c[a+16>>2]|0;e=pS(b)|0;f=e;if((e|0)==0){g=0;h=g;return h|0}bK(d|0,c[a+36>>2]|0,0)|0;if((bB(f|0,1,b|0,d|0)|0)!=(b|0)){pU(f);f=0}g=f;h=g;return h|0}function jw(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0;d=i;i=i+8|0;e=d|0;f=b;b=pS(4096)|0;g=b;if((b|0)==0){h=-6;j=h;i=d;return j|0}b=jx(e,a,0,0)|0;a=b;if((b|0)==0){while(1){b=jy(c[e>>2]|0,g,4096)|0;a=b;if((b|0)<=0){k=5790;break}if((a|0)!=(ay(g|0,1,a|0,f|0)|0)){k=5792;break}}if((k|0)==5792){a=-4}}do{if((a|0)==-1){if((c[(c[e>>2]|0)+1072>>2]|0)!=(c[(c[e>>2]|0)+1076>>2]|0)){break}a=0}}while(0);if((c[(c[e>>2]|0)+1084>>2]|0)!=0){pU(c[(c[e>>2]|0)+1084>>2]|0)}pU(c[e>>2]|0);pU(g);h=(a|0)==0?0:-1;j=h;i=d;return j|0}function jx(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,v=0,w=0,x=0;f=1;g=0;h=i;i=i+168|0;c[h>>2]=0;while(1)switch(f|0){case 1:j=a;k=b;l=d;m=e;n=41124;if((k|0)!=0){f=2;break}else{f=3;break};case 2:n=n+4096|0;f=3;break;case 3:o=ai(6,n|0)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;p=o;c[j>>2]=p;q=p;if((p|0)!=0){f=5;break}else{f=4;break};case 4:r=-6;f=26;break;case 5:p7(q|0,0,41124);p=k;c[q+20>>2]=p;if(0==(p|0)){f=6;break}else{f=7;break};case 6:c[q+36>>2]=l;c[q+28>>2]=m;f=8;break;case 7:c[q+36>>2]=q+41124;f=8;break;case 8:n=0;f=9;break;case 9:if(n>>>0<256){f=10;break}else{f=19;break};case 10:s=n<<24;v=8;f=11;break;case 11:if((v|0)!=0){f=12;break}else{f=17;break};case 12:if((s&-2147483648|0)!=0){f=13;break}else{f=14;break};case 13:w=s<<1^79764919;f=15;break;case 14:w=s<<1;f=15;break;case 15:s=w;f=16;break;case 16:v=v-1|0;f=11;break;case 17:c[q+48+(n<<2)>>2]=s;f=18;break;case 18:n=n+1|0;f=9;break;case 19:x=qd(q+41084|0,f,h)|0;f=27;break;case 27:n=x;if((n|0)!=0){f=20;break}else{f=21;break};case 20:r=n;f=26;break;case 21:p=aq(52,q|0,32)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;n=p;if((n-1113221168-1|0)>>>0>=9){f=22;break}else{f=23;break};case 22:r=-2;f=26;break;case 23:c[q+1088>>2]=(n-1113221168|0)*1e5|0;p=ai(6,c[q+1088>>2]<<2|0)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;o=p;c[q+1084>>2]=o;if((o|0)!=0){f=25;break}else{f=24;break};case 24:r=-6;f=26;break;case 25:r=0;f=26;break;case 26:return r|0;case-1:if((g|0)==19){x=u;f=27}t=u=0;break}return 0}function jy(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;f=b;b=d;d=e;if((c[f+12>>2]|0)<0){g=c[f+12>>2]|0;h=g;return h|0}e=0;i=c[f+1084>>2]|0;j=c[f+4>>2]|0;k=c[f+16>>2]|0;if((c[f>>2]|0)!=0){l=f|0;c[l>>2]=(c[l>>2]|0)-1;m=5808}else{m=5826}L7251:while(1){do{if((m|0)==5808){m=0;if((e|0)>=(d|0)){m=5809;break L7251}l=e;e=l+1|0;a[b+l|0]=k&255;c[f+1080>>2]=c[f+1080>>2]<<8^c[f+48+(((c[f+1080>>2]|0)>>>24^k)<<2)>>2];if((c[f>>2]|0)!=0){l=f|0;c[l>>2]=(c[l>>2]|0)-1;m=5808;continue L7251}else{break}}else if((m|0)==5826){m=0;n=jz(f)|0;if((n|0)!=0){m=5827;break L7251}c[f+1080>>2]=-1;j=c[f+4>>2]|0;k=c[f+16>>2]|0}}while(0);while(1){l=f+12|0;o=c[l>>2]|0;c[l>>2]=o-1;if((o|0)==0){m=5814;break}n=k;j=c[i+(j<<2)>>2]|0;k=j&255;j=j>>8;o=f+8|0;l=(c[o>>2]|0)-1|0;c[o>>2]=l;if((l|0)!=0){m=5816;break}c[f>>2]=k;k=n;c[f+8>>2]=5;if((c[f>>2]|0)!=0){m=5821;break}}if((m|0)==5821){m=0;l=f|0;c[l>>2]=(c[l>>2]|0)-1}else if((m|0)==5816){m=0;if((k|0)!=(n|0)){c[f+8>>2]=4}}else if((m|0)==5814){m=0;c[f+1080>>2]=~c[f+1080>>2];c[f+1076>>2]=(c[f+1076>>2]<<1|(c[f+1076>>2]|0)>>>31)^c[f+1080>>2];if((c[f+1080>>2]|0)!=(c[f+1072>>2]|0)){m=5824;break}m=5826;continue}m=5808}if((m|0)==5827){c[f+12>>2]=n;if((n|0)!=-1){p=n}else{p=e}g=p;h=g;return h|0}else if((m|0)==5809){c[f+4>>2]=j;c[f+16>>2]=k;k=f|0;c[k>>2]=(c[k>>2]|0)+1;g=d;h=g;return h|0}else if((m|0)==5824){c[f+1076>>2]=(c[f+1072>>2]|0)+1;g=-1;h=g;return h|0}return 0}function jz(b){b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0,R=0,S=0;e=i;i=i+1824|0;f=1;g=0;h=i;i=i+168|0;c[h>>2]=0;while(1)switch(f|0){case 1:j=e|0;k=e+1024|0;l=e+1280|0;m=e+1536|0;n=e+1800|0;o=b;p=0;q=0;r=0;s=c[o+1084>>2]|0;v=c[o+1088>>2]|0;w=o+1092|0;x=qd(o+41084|0,f,h)|0;f=143;break;case 143:y=x;if((y|0)!=0){f=2;break}else{f=3;break};case 2:z=y;f=142;break;case 3:A=aq(52,o|0,24)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;y=A;A=aq(52,o|0,24)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;B=A;A=aq(52,o|0,32)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;c[o+1072>>2]=A;if((y|0)==1536581){f=4;break}else{f=6;break};case 4:if((B|0)==3690640){f=5;break}else{f=6;break};case 5:z=-1;f=142;break;case 6:if((y|0)!=3227993){f=8;break}else{f=7;break};case 7:if((B|0)!=2511705){f=8;break}else{f=9;break};case 8:z=-2;f=142;break;case 9:A=aq(52,o|0,1)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;if((A|0)!=0){f=10;break}else{f=11;break};case 10:z=-7;f=142;break;case 11:A=aq(52,o|0,24)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;C=A;if(A>>>0>v>>>0){f=12;break}else{f=13;break};case 12:z=-5;f=142;break;case 13:A=aq(52,o|0,16)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;D=A;E=0;y=0;f=14;break;case 14:if((y|0)<16){f=15;break}else{f=25;break};case 15:if((D&1<<15-y|0)!=0){f=16;break}else{f=23;break};case 16:A=aq(52,o|0,16)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;F=A;B=0;f=17;break;case 17:if((B|0)<16){f=18;break}else{f=22;break};case 18:if((F&1<<15-B|0)!=0){f=19;break}else{f=20;break};case 19:A=E;E=A+1|0;a[k+A|0]=(y<<4)+B&255;f=20;break;case 20:f=21;break;case 21:B=B+1|0;f=17;break;case 22:f=23;break;case 23:f=24;break;case 24:y=y+1|0;f=14;break;case 25:A=aq(52,o|0,3)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;G=A;if((G|0)<2){f=27;break}else{f=26;break};case 26:if((G|0)>6){f=27;break}else{f=28;break};case 27:z=-5;f=142;break;case 28:A=aq(52,o|0,15)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;H=A;if((A|0)!=0){f=30;break}else{f=29;break};case 29:z=-5;f=142;break;case 30:y=0;f=31;break;case 31:if((y|0)<(G|0)){f=32;break}else{f=34;break};case 32:a[l+y|0]=y&255;f=33;break;case 33:y=y+1|0;f=31;break;case 34:y=0;f=35;break;case 35:if((y|0)<(H|0)){f=36;break}else{f=48;break};case 36:B=0;f=37;break;case 37:A=aq(52,o|0,1)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;if((A|0)!=0){f=38;break}else{f=42;break};case 38:if((B|0)>=(G|0)){f=39;break}else{f=40;break};case 39:z=-5;f=142;break;case 40:f=41;break;case 41:B=B+1|0;f=37;break;case 42:I=a[l+B|0]|0;f=43;break;case 43:if((B|0)!=0){f=44;break}else{f=46;break};case 44:a[l+B|0]=a[l+(B-1)|0]|0;f=45;break;case 45:B=B-1|0;f=43;break;case 46:A=I;a[w+y|0]=A;a[l|0]=A;f=47;break;case 47:y=y+1|0;f=35;break;case 48:J=E+2|0;B=0;f=49;break;case 49:if((B|0)<(G|0)){f=50;break}else{f=89;break};case 50:A=aq(52,o|0,5)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;D=A-1|0;y=0;f=51;break;case 51:if((y|0)<(J|0)){f=52;break}else{f=60;break};case 52:f=53;break;case 53:if(D>>>0>19){f=54;break}else{f=55;break};case 54:z=-5;f=142;break;case 55:A=aq(52,o|0,2)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;F=A;if((F|0)<2){f=56;break}else{f=57;break};case 56:A=o+40|0;c[A>>2]=(c[A>>2]|0)+1;f=58;break;case 57:D=D+((F+1&2)-1)|0;f=53;break;case 58:a[m+y|0]=D+1&255;f=59;break;case 59:y=y+1|0;f=51;break;case 60:A=d[m|0]|0;K=A;L=A;y=1;f=61;break;case 61:if((y|0)<(J|0)){f=62;break}else{f=69;break};case 62:if((d[m+y|0]|0|0)>(K|0)){f=63;break}else{f=64;break};case 63:K=d[m+y|0]|0;f=67;break;case 64:if((d[m+y|0]|0|0)<(L|0)){f=65;break}else{f=66;break};case 65:L=d[m+y|0]|0;f=66;break;case 66:f=67;break;case 67:f=68;break;case 68:y=y+1|0;f=61;break;case 69:r=o+33860+(B*1204|0)|0;c[r+1196>>2]=L;c[r+1200>>2]=K;p=r+84-4|0;q=r-4|0;M=0;y=L;f=70;break;case 70:if((y|0)<=(K|0)){f=71;break}else{f=79;break};case 71:c[q+(y<<2)>>2]=0;a[n+y|0]=0;D=0;f=72;break;case 72:if((D|0)<(J|0)){f=73;break}else{f=77;break};case 73:if((d[m+D|0]|0|0)==(y|0)){f=74;break}else{f=75;break};case 74:A=M;M=A+1|0;c[r+164+(A<<2)>>2]=D;f=75;break;case 75:f=76;break;case 76:D=D+1|0;f=72;break;case 77:f=78;break;case 78:y=y+1|0;f=70;break;case 79:y=0;f=80;break;case 80:if((y|0)<(J|0)){f=81;break}else{f=83;break};case 81:A=n+(d[m+y|0]|0)|0;a[A]=(a[A]|0)+1&255;f=82;break;case 82:y=y+1|0;f=80;break;case 83:D=0;M=0;y=L;f=84;break;case 84:if((y|0)<(K|0)){f=85;break}else{f=87;break};case 85:M=M+(d[n+y|0]|0)|0;c[q+(y<<2)>>2]=(M<<K-y)-1;M=M<<1;A=D+(d[n+y|0]|0)|0;D=A;c[p+(y+1<<2)>>2]=M-A;f=86;break;case 86:y=y+1|0;f=84;break;case 87:c[q+(K+1<<2)>>2]=2147483647;c[q+(K<<2)>>2]=M+(d[n+K|0]|0)-1;c[p+(L<<2)>>2]=0;f=88;break;case 88:B=B+1|0;f=49;break;case 89:y=0;f=90;break;case 90:if((y|0)<256){f=91;break}else{f=93;break};case 91:c[j+(y<<2)>>2]=0;a[l+y|0]=y&255;f=92;break;case 92:y=y+1|0;f=90;break;case 93:N=0;J=0;O=0;P=0;f=94;break;case 94:A=J;J=A-1|0;if((A|0)!=0){f=98;break}else{f=95;break};case 95:J=49;if((N|0)>=(H|0)){f=96;break}else{f=97;break};case 96:z=-5;f=142;break;case 97:A=N;N=A+1|0;r=o+33860+((d[w+A|0]|0)*1204|0)|0;p=r+84-4|0;q=r-4|0;f=98;break;case 98:f=99;break;case 99:if((c[o+40>>2]|0)>>>0<(c[r+1200>>2]|0)>>>0){f=100;break}else{f=103;break};case 100:if((c[o+32>>2]|0)==(c[o+28>>2]|0)){f=101;break}else{f=102;break};case 101:A=aq(52,o|0,c[r+1200>>2]&255|0)|0;if((t|0)!=0&(u|0)!=0){g=qe(c[t>>2]|0,h)|0;if((g|0)>0){f=-1;break}else return 0}t=u=0;B=A;f=104;break;case 102:A=c[o+44>>2]<<8;Q=o+32|0;R=c[Q>>2]|0;c[Q>>2]=R+1;c[o+44>>2]=A|(d[(c[o+36>>2]|0)+R|0]|0);R=o+40|0;c[R>>2]=(c[R>>2]|0)+8;f=99;break;case 103:R=o+40|0;c[R>>2]=(c[R>>2]|0)-(c[r+1200>>2]|0);B=(c[o+44>>2]|0)>>>((c[o+40>>2]|0)>>>0)&(1<<c[r+1200>>2])-1;f=104;break;case 104:y=c[r+1196>>2]|0;f=105;break;case 105:if((B|0)>(c[q+(y<<2)>>2]|0)){f=106;break}else{f=107;break};case 106:y=y+1|0;f=105;break;case 107:R=o+40|0;c[R>>2]=(c[R>>2]|0)+((c[r+1200>>2]|0)-y);if((y|0)>(c[r+1200>>2]|0)){f=109;break}else{f=108;break};case 108:R=(B>>(c[r+1200>>2]|0)-y)-(c[p+(y<<2)>>2]|0)|0;B=R;if(R>>>0>=258){f=109;break}else{f=110;break};case 109:z=-5;f=142;break;case 110:S=c[r+164+(B<<2)>>2]|0;if(S>>>0<=1){f=111;break}else{f=114;break};case 111:if((P|0)!=0){f=113;break}else{f=112;break};case 112:P=1;D=0;f=113;break;case 113:D=D+(P<<S)|0;P=P<<1;f=94;break;case 114:if((P|0)!=0){f=115;break}else{f=121;break};case 115:P=0;if((O+D|0)>=(v|0)){f=116;break}else{f=117;break};case 116:z=-5;f=142;break;case 117:I=a[k+(d[l|0]|0)|0]|0;R=j+((I&255)<<2)|0;c[R>>2]=(c[R>>2]|0)+D;f=118;break;case 118:R=D;D=R-1|0;if((R|0)!=0){f=119;break}else{f=120;break};case 119:R=O;O=R+1|0;c[s+(R<<2)>>2]=I&255;f=118;break;case 120:f=121;break;case 121:if((S|0)>(E|0)){f=122;break}else{f=123;break};case 122:f=129;break;case 123:if((O|0)>=(v|0)){f=124;break}else{f=125;break};case 124:z=-5;f=142;break;case 125:y=S-1|0;I=a[l+y|0]|0;f=126;break;case 126:a[l+y|0]=a[l+(y-1)|0]|0;f=127;break;case 127:R=y-1|0;y=R;if((R|0)!=0){f=126;break}else{f=128;break};case 128:a[l|0]=I;I=a[k+(I&255)|0]|0;R=j+((I&255)<<2)|0;c[R>>2]=(c[R>>2]|0)+1;R=O;O=R+1|0;c[s+(R<<2)>>2]=I&255;f=94;break;case 129:B=0;y=0;f=130;break;case 130:if((y|0)<256){f=131;break}else{f=133;break};case 131:F=B+(c[j+(y<<2)>>2]|0)|0;c[j+(y<<2)>>2]=B;B=F;f=132;break;case 132:y=y+1|0;f=130;break;case 133:y=0;f=134;break;case 134:if((y|0)<(O|0)){f=135;break}else{f=137;break};case 135:I=c[s+(y<<2)>>2]&255;R=s+(c[j+((I&255)<<2)>>2]<<2)|0;c[R>>2]=c[R>>2]|y<<8;R=j+((I&255)<<2)|0;c[R>>2]=(c[R>>2]|0)+1;f=136;break;case 136:y=y+1|0;f=134;break;case 137:if((O|0)!=0){f=138;break}else{f=141;break};case 138:if(C>>>0>=O>>>0){f=139;break}else{f=140;break};case 139:z=-5;f=142;break;case 140:c[o+4>>2]=c[s+(C<<2)>>2];c[o+16>>2]=c[o+4>>2]&255&255;R=o+4|0;c[R>>2]=c[R>>2]>>8;c[o+8>>2]=5;f=141;break;case 141:c[o+12>>2]=O;z=0;f=142;break;case 142:i=e;return z|0;case-1:if((g|0)==1){x=u;f=143}t=u=0;break}return 0}function jA(a){a=a|0;var b=0,d=0,e=0;b=a;a=0;while(1){if((a|0)>=256){break}d=a;e=0;while(1){if((e|0)>=8){break}if((d&1|0)!=0){d=d>>>1^-306674912}else{d=d>>>1}e=e+1|0}c[b+4+(a<<2)>>2]=d;a=a+1|0}return 0}function jB(a,b,e,f){a=a|0;b=b|0;e=e|0;f=f|0;var g=0;g=a;a=b;b=e;e=f;f=0;while(1){if((f|0)>=(a|0)){break}b=c[e+4+(((b^(d[g+f|0]|0))&255)<<2)>>2]^b>>>8;f=f+1|0}return b|0}function jC(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;f=i;i=i+24|0;g=f|0;h=f+16|0;j=b;b=e;a[g|0]=ja(j)|0;a[g+1|0]=ja(j)|0;a[g+2|0]=ja(j)|0;a[g+3|0]=ja(j)|0;c[g+4>>2]=jg(j)|0;a[g+8|0]=ja(j)|0;a[g+9|0]=ja(j)|0;if((d[g+2|0]|0|0)!=8){k=-1;l=k;i=f;return l|0}if((a[g+3|0]&4|0)!=0){e=j;m=(jc(j)|0)&65535;bK(e|0,m|0,1)|0}if((a[g+3|0]&8|0)!=0){do{n=(ja(j)|0)&255;}while((n|0)!=0)}if((a[g+3|0]&16|0)!=0){do{n=(ja(j)|0)&255;}while((n|0)!=0)}if((a[g+3|0]&2|0)!=0){g=j;jc(g)|0}g=jG(j,b,h,1)|0;if((g|0)!=0){k=-1;l=k;i=f;return l|0}g=jg(j)|0;if((g|0)!=(c[h>>2]|0)){k=-1;l=k;i=f;return l|0}g=jg(j)|0;if((g|0)!=(aT(b|0)|0)){k=-1;l=k;i=f;return l|0}else{k=0;l=k;i=f;return l|0}return 0}function jD(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0;e=a;a=b;b=0;while(1){if((c[e+40>>2]|0)>>>0>=a<<24>>24>>>0){f=5892;break}if((c[e+32>>2]|0)==(c[e+28>>2]|0)){g=bB(c[e+36>>2]|0,1,4096,c[e+20>>2]|0)|0;c[e+28>>2]=g;if((g|0)<=0){f=5887;break}c[e+32>>2]=0}if((c[e+40>>2]|0)>>>0>=24){b=c[e+44>>2]&(1<<c[e+40>>2])-1;a=(a<<24>>24)-(c[e+40>>2]|0)&255;b=b<<(a<<24>>24);c[e+40>>2]=0}g=c[e+44>>2]<<8;h=e+32|0;i=c[h>>2]|0;c[h>>2]=i+1;c[e+44>>2]=g|(d[(c[e+36>>2]|0)+i|0]|0);i=e+40|0;c[i>>2]=(c[i>>2]|0)+8}if((f|0)==5892){i=e+40|0;c[i>>2]=(c[i>>2]|0)-(a<<24>>24);b=b|(c[e+44>>2]|0)>>>((c[e+40>>2]|0)>>>0)&(1<<(a<<24>>24))-1;return b|0}else if((f|0)==5887){bQ(e+41084|0,-3|0);return 0}return 0}function jE(a){a=a|0;var b=0;b=a;a=d[3624+((c[b>>2]|0)>>>24&255)|0]|0|(d[3624+((c[b>>2]|0)>>>16&255)|0]|0)<<8|(d[3624+((c[b>>2]|0)>>>8&255)|0]|0)<<16|(d[3624+(c[b>>2]&255)|0]|0)<<24;a=a>>>((32-(c[b+4>>2]|0)|0)>>>0);c[b>>2]=a;return 0}function jF(f,g,h,i,j,k,l){f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;var m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;m=f;f=g;g=h;h=i;i=j;j=k;k=l;l=0;jE(g)|0;n=c[f+32768>>2]|0;o=f|0;while(1){p=h;q=0;while(1){if((c[g+4>>2]|0)<=0){c[g>>2]=aN(m|0)|0;c[g+4>>2]=8}if((c[g>>2]&1|0)==0){if((b[h+(q*6|0)+2>>1]|0)==0){r=5900;break}q=b[h+(q*6|0)+2>>1]|0}else{if((b[h+(q*6|0)+4>>1]|0)==0){r=5903;break}q=b[h+(q*6|0)+4>>1]|0}s=g+4|0;c[s>>2]=(c[s>>2]|0)-1;s=g|0;c[s>>2]=(c[s>>2]|0)>>>1}if((r|0)==5903){r=0;l=e[h+(q*6|0)>>1]|0}else if((r|0)==5900){r=0;l=e[h+(q*6|0)>>1]|0}if((l|0)<256){s=n;n=s+1|0;a[o+s|0]=l&255;if((n|0)>=32768){s=o;t=j;ay(s|0,1,32768,t|0)|0;c[f+32772>>2]=jB(f|0,32768,c[f+32772>>2]|0,k)|0;n=0}}else{if((l|0)==256){break}l=l-257|0;t=c[7112+(l<<2)>>2]|0;if((c[6992+(l<<2)>>2]|0)!=0){while(1){if((c[g+4>>2]|0)>=(c[6992+(l<<2)>>2]|0)){break}s=aN(m|0)|0;u=g|0;c[u>>2]=(c[u>>2]|0)+(s<<c[g+4>>2]);s=g+4|0;c[s>>2]=(c[s>>2]|0)+8}t=t+(c[g>>2]&(1<<c[6992+(l<<2)>>2])-1)|0;s=g+4|0;c[s>>2]=(c[s>>2]|0)-(c[6992+(l<<2)>>2]|0);s=g|0;c[s>>2]=(c[s>>2]|0)>>>((c[6992+(l<<2)>>2]|0)>>>0)}if((c[f+34060>>2]|0)==0){if((c[g+4>>2]|0)<5){s=aN(m|0)|0;u=g|0;c[u>>2]=(c[u>>2]|0)+(s<<c[g+4>>2]);s=g+4|0;c[s>>2]=(c[s>>2]|0)+8}l=c[g>>2]&31;l=d[3624+(l&255)|0]>>3;s=g+4|0;c[s>>2]=(c[s>>2]|0)-5;s=g|0;c[s>>2]=(c[s>>2]|0)>>>5}else{p=h;q=0;while(1){if((c[g+4>>2]|0)<=0){c[g>>2]=aN(m|0)|0;c[g+4>>2]=8}if((c[g>>2]&1|0)==0){if((b[i+(q*6|0)+2>>1]|0)==0){r=5926;break}q=b[i+(q*6|0)+2>>1]|0}else{if((b[i+(q*6|0)+4>>1]|0)==0){r=5929;break}q=b[i+(q*6|0)+4>>1]|0}s=g+4|0;c[s>>2]=(c[s>>2]|0)-1;s=g|0;c[s>>2]=(c[s>>2]|0)>>>1}if((r|0)==5926){r=0;l=e[i+(q*6|0)>>1]|0}else if((r|0)==5929){r=0;l=e[i+(q*6|0)>>1]|0}}p=c[11352+(l<<2)>>2]|0;if((c[11232+(l<<2)>>2]|0)!=0){while(1){if((c[g+4>>2]|0)>=(c[11232+(l<<2)>>2]|0)){break}s=aN(m|0)|0;u=g|0;c[u>>2]=(c[u>>2]|0)+(s<<c[g+4>>2]);s=g+4|0;c[s>>2]=(c[s>>2]|0)+8}p=p+(c[g>>2]&(1<<c[11232+(l<<2)>>2])-1)|0;q=g+4|0;c[q>>2]=(c[q>>2]|0)-(c[11232+(l<<2)>>2]|0);q=g|0;c[q>>2]=(c[q>>2]|0)>>>((c[11232+(l<<2)>>2]|0)>>>0)}q=n-p|0;do{if((q|0)>=0){if((n+t|0)>=32768){r=5942;break}if((q+t|0)>=(n|0)){r=5942;break}s=o+n|0;u=o+q|0;v=t;p3(s|0,u|0,v)|0;n=n+t|0}else{r=5942}}while(0);if((r|0)==5942){r=0;if((q|0)<0){q=q+32768|0}p=0;while(1){if((p|0)>=(t|0)){break}v=q;q=v+1|0;u=n;n=u+1|0;a[o+u|0]=a[o+v|0]|0;if((q|0)>=32768){q=0}if((n|0)>=32768){v=o;u=j;ay(v|0,1,32768,u|0)|0;c[f+32772>>2]=jB(f|0,32768,c[f+32772>>2]|0,k)|0;n=0}p=p+1|0}}}}c[f+32768>>2]=n;jE(g)|0;return 0}function jG(b,e,f,g){b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0;h=i;i=i+35104|0;j=h|0;k=h+8|0;l=h+34072|0;m=b;b=e;e=f;p7(k|0,0,34064);c[k+32772>>2]=-1;c[l>>2]=0;f=pS(6144)|0;n=pS(6144)|0;c[k+32768>>2]=0;do{if((g|0)!=0){o=5965}else{p=(aN(m|0)|0)&255;q=(aN(m|0)|0)&255;if((p&15|0)!=8){bf(23544,(r=i,i=i+1|0,i=i+7>>3<<3,c[r>>2]=0,r)|0)|0;i=r;break}if((q&32|0)!=0){s=m;jl(s)|0;}if(((((p&255)<<8)+(q&255)|0)%31|0|0)!=0){bf(29072,(r=i,i=i+1|0,i=i+7>>3<<3,c[r>>2]=0,r)|0)|0;i=r}o=5965}}while(0);do{if((o|0)==5965){jH(l)|0;c[j>>2]=0;c[j+4>>2]=0;while(1){if((c[j+4>>2]|0)<3){g=d[3624+(aN(m|0)|0)|0]|0;c[j>>2]=g+(c[j>>2]<<8);g=j+4|0;c[g>>2]=(c[g>>2]|0)+8}g=(c[j>>2]|0)>>>(((c[j+4>>2]|0)-1|0)>>>0);q=(c[j>>2]|0)>>>(((c[j+4>>2]|0)-3|0)>>>0)&3;p=j+4|0;c[p>>2]=(c[p>>2]|0)-3;c[j>>2]=c[j>>2]&(1<<c[j+4>>2])-1;if((q|0)==0){c[j>>2]=0;c[j+4>>2]=0;t=(jc(m)|0)&65535;u=(jc(m)|0)&65535^65535;if((t|0)!=(u|0)){o=5970;break}u=0;while(1){if((u|0)>=(t|0)){break}p=(aN(m|0)|0)&255;s=k+32768|0;v=c[s>>2]|0;c[s>>2]=v+1;a[k+v|0]=p;if((c[k+32768>>2]|0)>=32768){p=k|0;v=b;ay(p|0,1,32768,v|0)|0;c[k+32772>>2]=jB(k|0,32768,c[k+32772>>2]|0,l)|0;c[k+32768>>2]=0}u=u+1|0}}else{if((q|0)==2){if((c[l>>2]|0)==0){v=l|0;jI(k,v)|0}v=m;p=c[l>>2]|0;s=b;jF(v,k,j,p,0,s,l)|0}else{if((q|0)==1){if((jJ(m,k,j,f,n)|0)<0){o=5984;break}s=m;p=f;v=n;w=b;jF(s,k,j,p,v,w,l)|0}else{if((q|0)==3){o=5987;break}}}}if((g|0)==1){o=5992;break}}if((o|0)!=5992)if((o|0)==5987){bf(26360,(r=i,i=i+1|0,i=i+7>>3<<3,c[r>>2]=0,r)|0)|0;i=r}else if((o|0)==5984){jK(l)|0;break}else if((o|0)==5970){bf(27520,(r=i,i=i+16|0,c[r>>2]=t,c[r+8>>2]=u,r)|0)|0;i=r}if((c[k+32768>>2]|0)!=0){w=k|0;v=c[k+32768>>2]|0;p=b;ay(w|0,1,v|0,p|0)|0;c[k+32772>>2]=jB(k|0,c[k+32768>>2]|0,c[k+32772>>2]|0,l)|0}if((f|0)!=0){pU(f)}if((n|0)!=0){pU(n)}c[e>>2]=~c[k+32772>>2];jK(l)|0;if((c[j+4>>2]|0)==8){jE(j)|0;p=c[j>>2]|0;v=m;a$(p|0,v|0)|0}x=0;y=x;i=h;return y|0}}while(0);pU(n);pU(f);x=-1;y=x;i=h;return y|0}function jH(a){a=a|0;jA(a)|0;return 0}function jI(a,d){a=a|0;d=d|0;c[a+34060>>2]=0;a=pS(3600)|0;c[d>>2]=a;b[a+2>>1]=0;b[a+4>>1]=0;d=0;d=jL(a,8,144,48,0,d)|0;d=jL(a,9,112,400,144,d)|0;d=jL(a,7,24,0,256,d)|0;d=jL(a,8,8,192,280,d)|0;return 0}function jJ(a,b,e,f,g){a=a|0;b=b|0;e=e|0;f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;h=i;i=i+320|0;j=h|0;k=h+80|0;l=h+160|0;m=h+240|0;n=a;a=b;b=e;e=f;f=g;while(1){if((c[b+4>>2]|0)>=14){break}g=d[3624+(aN(n|0)|0)|0]|0;c[b>>2]=g+(c[b>>2]<<8);g=b+4|0;c[g>>2]=(c[g>>2]|0)+8}g=(c[b>>2]|0)>>>(((c[b+4>>2]|0)-5|0)>>>0);o=b+4|0;c[o>>2]=(c[o>>2]|0)-5;c[b>>2]=c[b>>2]&(1<<c[b+4>>2])-1;o=(c[b>>2]|0)>>>(((c[b+4>>2]|0)-5|0)>>>0);p=b+4|0;c[p>>2]=(c[p>>2]|0)-5;c[b>>2]=c[b>>2]&(1<<c[b+4>>2])-1;p=(c[b>>2]|0)>>>(((c[b+4>>2]|0)-4|0)>>>0);q=b+4|0;c[q>>2]=(c[q>>2]|0)-4;c[b>>2]=c[b>>2]&(1<<c[b+4>>2])-1;g=((d[3624+g|0]|0)>>3)+257|0;o=((d[3624+o|0]|0)>>3)+1|0;p=((d[3624+p|0]|0)>>4)+4|0;p7(j|0,0,76);p7(k|0,0,76);p7(l|0,0,76);q=0;while(1){if((q|0)>=(p|0)){break}if((c[b+4>>2]|0)<3){r=d[3624+(aN(n|0)|0)|0]|0;c[b>>2]=r+(c[b>>2]<<8);r=b+4|0;c[r>>2]=(c[r>>2]|0)+8}c[j+(c[10592+(q<<2)>>2]<<2)>>2]=(c[b>>2]|0)>>>(((c[b+4>>2]|0)-3|0)>>>0);c[j+(c[10592+(q<<2)>>2]<<2)>>2]=(d[3624+(c[j+(c[10592+(q<<2)>>2]<<2)>>2]|0)|0]|0)>>5;r=b+4|0;c[r>>2]=(c[r>>2]|0)-3;c[b>>2]=c[b>>2]&(1<<c[b+4>>2])-1;q=q+1|0}q=0;while(1){if((q|0)>=19){break}p=l+(c[j+(q<<2)>>2]<<2)|0;c[p>>2]=(c[p>>2]|0)+1;q=q+1|0}p=0;c[l>>2]=0;r=1;while(1){if((r|0)>7){break}p=p+(c[l+(r-1<<2)>>2]|0)<<1;c[m+(r<<2)>>2]=p;r=r+1|0}q=0;while(1){if((q|0)>=19){break}if((c[j+(q<<2)>>2]|0)!=0){c[k+(q<<2)>>2]=c[m+(c[j+(q<<2)>>2]<<2)>>2];r=m+(c[j+(q<<2)>>2]<<2)|0;c[r>>2]=(c[r>>2]|0)+1}q=q+1|0}p7(a+32776|0,0,1152);q=jM(n,b,a+32776|0,g,j|0,k|0,e)|0;if((q|0)<0){s=-1;t=s;i=h;return t|0}do{if((o|0)==0){c[a+34060>>2]=0}else{c[a+34060>>2]=o;p7(a+33928|0,0,132);q=jM(n,b,a+33928|0,o,j|0,k|0,f)|0;if((q|0)>=0){break}s=-1;t=s;i=h;return t|0}}while(0);s=0;t=s;i=h;return t|0}function jK(a){a=a|0;var b=0;b=a;if((c[b>>2]|0)==0){return 0}pU(c[b>>2]|0);return 0}function jL(a,c,d,e,f,g){a=a|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;h=a;a=c;c=d;d=e;e=f;f=g;g=0;while(1){if((g|0)>=(c|0)){break}i=h;j=1<<a-1;k=0;while(1){if((k|0)>=(a|0)){break}if((d&j|0)==0){if((b[i+2>>1]|0)==0){f=f+1|0;b[i+2>>1]=f&65535;b[h+(f*6|0)+2>>1]=0;b[h+(f*6|0)+4>>1]=0}i=h+((b[i+2>>1]|0)*6|0)|0}else{if((b[i+4>>1]|0)==0){f=f+1|0;b[i+4>>1]=f&65535;b[h+(f*6|0)+2>>1]=0;b[h+(f*6|0)+4>>1]=0}i=h+((b[i+4>>1]|0)*6|0)|0}j=j>>1;k=k+1|0}k=e;e=k+1|0;b[i>>1]=k&65535;d=d+1|0;g=g+1|0}return f|0}function jM(a,e,f,g,h,j,k){a=a|0;e=e|0;f=f|0;g=g|0;h=h|0;j=j|0;k=k|0;var l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0;l=i;i=i+4096|0;m=l|0;n=l+2048|0;o=a;a=e;e=f;f=g;g=h;h=j;j=k;k=0;while(1){if((k|0)>=(f|0)){break}p=0;while(1){if((p|0)>=19){break}if((c[g+(p<<2)>>2]|0)!=0){while(1){if((c[a+4>>2]|0)>=(c[g+(p<<2)>>2]|0)){break}r=d[3624+(aN(o|0)|0)|0]|0;c[a>>2]=r+(c[a>>2]<<8);r=a+4|0;c[r>>2]=(c[r>>2]|0)+8}if(((c[a>>2]|0)>>>(((c[a+4>>2]|0)-(c[g+(p<<2)>>2]|0)|0)>>>0)|0)==(c[h+(p<<2)>>2]|0)){s=6076;break}}p=p+1|0}if((s|0)==6076){s=0;r=a+4|0;c[r>>2]=(c[r>>2]|0)-(c[g+(p<<2)>>2]|0);c[a>>2]=c[a>>2]&(1<<c[a+4>>2])-1}if((p|0)<=15){r=k;k=r+1|0;c[e+(r<<2)>>2]=p}else{if((p|0)==16){if((k|0)!=0){t=c[e+(k-1<<2)>>2]|0}else{t=0}if((c[a+4>>2]|0)<2){r=d[3624+(aN(o|0)|0)|0]|0;c[a>>2]=r+(c[a>>2]<<8);r=a+4|0;c[r>>2]=(c[r>>2]|0)+8}u=d[3624+((c[a>>2]|0)>>>(((c[a+4>>2]|0)-2|0)>>>0))|0]>>6;r=a+4|0;c[r>>2]=(c[r>>2]|0)-2;c[a>>2]=c[a>>2]&(1<<c[a+4>>2])-1;v=0;while(1){if((v|0)>=(u+3|0)){break}r=k;k=r+1|0;c[e+(r<<2)>>2]=t;v=v+1|0}}else{if((p|0)==17){if((c[a+4>>2]|0)<3){r=d[3624+(aN(o|0)|0)|0]|0;c[a>>2]=r+(c[a>>2]<<8);r=a+4|0;c[r>>2]=(c[r>>2]|0)+8}u=d[3624+((c[a>>2]|0)>>>(((c[a+4>>2]|0)-3|0)>>>0))|0]>>5;r=a+4|0;c[r>>2]=(c[r>>2]|0)-3;c[a>>2]=c[a>>2]&(1<<c[a+4>>2])-1;v=u+3|0;p7(e+(k<<2)|0,0,v<<2|0);k=k+v|0}else{if((p|0)!=18){s=6100;break}if((c[a+4>>2]|0)<7){r=d[3624+(aN(o|0)|0)|0]|0;c[a>>2]=r+(c[a>>2]<<8);r=a+4|0;c[r>>2]=(c[r>>2]|0)+8}u=d[3624+((c[a>>2]|0)>>>(((c[a+4>>2]|0)-7|0)>>>0))|0]>>1;r=a+4|0;c[r>>2]=(c[r>>2]|0)-7;c[a>>2]=c[a>>2]&(1<<c[a+4>>2])-1;v=u+11|0;p7(e+(k<<2)|0,0,v<<2|0);k=k+v|0}}}}if((s|0)==6100){v=c[q>>2]|0;a=p;bE(v|0,25288,(w=i,i=i+8|0,c[w>>2]=a,w)|0)|0;i=w;x=-1;y=x;i=l;return y|0}p7(m|0,0,f<<2|0);a=0;p=0;while(1){if((p|0)>=(f|0)){break}if((c[e+(p<<2)>>2]|0)>=(f|0)){s=6108;break}v=m+(c[e+(p<<2)>>2]<<2)|0;c[v>>2]=(c[v>>2]|0)+1;if((a|0)<(c[e+(p<<2)>>2]|0)){a=c[e+(p<<2)>>2]|0}p=p+1|0}if((s|0)==6108){s=c[q>>2]|0;bE(s|0,24456,(w=i,i=i+1|0,i=i+7>>3<<3,c[w>>2]=0,w)|0)|0;i=w;x=-1;y=x;i=l;return y|0}t=0;c[m>>2]=0;w=1;while(1){if((w|0)>(a|0)){break}t=t+(c[m+(w-1<<2)>>2]|0)<<1;c[n+(w<<2)>>2]=t;w=w+1|0}b[j+2>>1]=0;b[j+4>>1]=0;w=0;p=0;while(1){if((p|0)>=(f|0)){break}if((c[e+(p<<2)>>2]|0)!=0){t=c[n+(c[e+(p<<2)>>2]<<2)>>2]|0;m=0;u=1<<(c[e+(p<<2)>>2]|0)-1;k=0;while(1){if((k|0)>=(c[e+(p<<2)>>2]|0)){break}if((t&u|0)==0){if((b[j+(m*6|0)+2>>1]|0)==0){w=w+1|0;b[j+(m*6|0)+2>>1]=w&65535;b[j+(w*6|0)+2>>1]=0;b[j+(w*6|0)+4>>1]=0}m=b[j+(m*6|0)+2>>1]|0}else{if((b[j+(m*6|0)+4>>1]|0)==0){w=w+1|0;b[j+(m*6|0)+4>>1]=w&65535;b[j+(w*6|0)+2>>1]=0;b[j+(w*6|0)+4>>1]=0}m=b[j+(m*6|0)+4>>1]|0}u=u>>1;k=k+1|0}b[j+(m*6|0)>>1]=p&65535;a=n+(c[e+(p<<2)>>2]<<2)|0;c[a>>2]=(c[a>>2]|0)+1}p=p+1|0}x=0;y=x;i=l;return y|0}function jN(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;d=i;i=i+56|0;e=d|0;f=d+24|0;g=a;a=b;if((jg(g)|0)!=1129474426){h=-1;j=h;i=d;return j|0}if((jg(g)|0)!=1634291279){h=-1;j=h;i=d;return j|0}if(((jc(g)|0)&65535|0)<14){h=-1;j=h;i=d;return j|0}c[e>>2]=(jc(g)|0)&65535;c[e+4>>2]=(jc(g)|0)&65535;c[e+8>>2]=jg(g)|0;c[e+12>>2]=jg(g)|0;c[e+16>>2]=(ja(g)|0)&255;c[e+20>>2]=(ja(g)|0)&255;if((c[e+4>>2]|0)==0){h=-1;j=h;i=d;return j|0}bK(g|0,c[e+12>>2]|0,0)|0;b=pS(c[e+4>>2]<<2)|0;if((b|0)==0){h=-1;j=h;i=d;return j|0}k=0;while(1){if(k>>>0>=(c[e+4>>2]|0)>>>0){break}c[b+(k<<2)>>2]=jg(g)|0;k=k+1|0}k=0;while(1){if(k>>>0>=(c[e+4>>2]|0)>>>0){l=6169;break}bK(g|0,c[b+(k<<2)>>2]|0,0)|0;c[f>>2]=jg(g)|0;c[f+4>>2]=jg(g)|0;c[f+8>>2]=jg(g)|0;c[f+12>>2]=(jc(g)|0)&65535;c[f+16>>2]=(jc(g)|0)&65535;c[f+20>>2]=(jc(g)|0)&65535;c[f+24>>2]=(jc(g)|0)&65535;m=pS(c[f+12>>2]<<3)|0;if((m|0)==0){break}n=0;while(1){if(n>>>0>=(c[f+12>>2]|0)>>>0){break}c[m+(n<<3)>>2]=jg(g)|0;c[m+(n<<3)+4>>2]=jg(g)|0;n=n+1|0}n=aT(g|0)|0;o=f+20|0;c[o>>2]=(c[o>>2]|0)+n;if((~c[f+16>>2]&1|0)!=0){jO(f,m,g,a)}else{if((c[f+16>>2]&4|0)!=0){jP(f,m,g,a)}else{jQ(f,m,g,a)}}pU(m);k=k+1|0}if((l|0)==6169){pU(b);h=0;j=h;i=d;return j|0}pU(b);h=-1;j=h;i=d;return j|0}function jO(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;f=a;a=b;b=d;d=e;e=0;while(1){if((e|0)>=(c[f+12>>2]|0)){break}jr(d,b,c[a+4>>2]|0)|0;e=e+1|0;a=a+8|0}return}function jP(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;f=i;i=i+8|0;g=f|0;h=a;a=b;b=d;d=e;e=0;j=c[h+24>>2]|0;k=0;c[g>>2]=0;c[g+4>>2]=0;bK(d|0,c[a>>2]|0,0)|0;bK(b|0,c[h+20>>2]|0,0)|0;l=0;while(1){if(l>>>0>=(c[h+12>>2]|0)>>>0){m=6215;break}n=c[a+(l<<3)+4>>2]>>1;o=65536;p=jR(b,j+1|0,g)|0;if(p>>>0>=(c[20768+(j<<2)>>2]|0)>>>0){q=c[10400+(j<<2)>>2]|0;r=jR(b,q,g)|0;s=r+(p-(c[20768+(j<<2)>>2]|0)<<q)|0;if((s|0)!=(j|0)){j=s&15}else{s=jR(b,4,g)|0;p=s;if((s|0)==15){if((jR(b,1,g)|0)!=0){m=6191;break}o=65535}else{o=p+65520|0}}}else{o=p}if(o>>>0<65536){if((o&1|0)!=0){o=-((o+1|0)>>>1)|0}else{o=o>>>1}if((c[h+16>>2]&2|0)!=0){o=o+k|0;k=o}else{if((c[h+16>>2]&512|0)==0){o=o^32768}}e=e+1|0;jn(d,o&65535)}if(e>>>0>=n>>>0){n=l+1|0;l=n;if(n>>>0>=(c[h+12>>2]|0)>>>0){m=6209;break}e=0;n=d;o=c[a+(l<<3)>>2]|0;bK(n|0,o|0,0)|0}}if((m|0)==6209){i=f;return}else if((m|0)==6191){i=f;return}else if((m|0)==6215){i=f;return}}function jQ(a,b,e,f){a=a|0;b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;g=i;i=i+264|0;h=g|0;j=g+8|0;k=a;a=b;b=e;e=f;f=0;l=c[k+24>>2]|0;m=0;bB(j|0,1,256,b|0)|0;c[h>>2]=0;c[h+4>>2]=0;bK(e|0,c[a>>2]|0,0)|0;bK(b|0,c[k+20>>2]|0,0)|0;n=0;while(1){if(n>>>0>=(c[k+12>>2]|0)>>>0){o=6240;break}p=c[a+(n<<3)+4>>2]|0;q=256;r=jR(b,l+1|0,h)|0;if(r>>>0>=(c[20736+(l<<2)>>2]|0)>>>0){s=c[10368+(l<<2)>>2]|0;t=jR(b,s,h)|0;u=t+(r-(c[20736+(l<<2)>>2]|0)<<s)|0;if((u|0)!=(l|0)){l=u&7}else{u=jR(b,3,h)|0;r=u;if((u|0)==7){if((jR(b,1,h)|0)!=0){o=6223;break}q=255}else{q=r+248|0}}}else{q=r}if(q>>>0<256){r=d[j+q|0]|0;if((c[k+16>>2]&2|0)!=0){r=r+m|0;m=r}f=f+1|0;jm(e,r&255)}if(f>>>0>=p>>>0){p=n+1|0;n=p;if(p>>>0>=(c[k+12>>2]|0)>>>0){o=6235;break}f=0;p=e;r=c[a+(n<<3)>>2]|0;bK(p|0,r|0,0)|0}}if((o|0)==6235){i=g;return}else if((o|0)==6223){i=g;return}else if((o|0)==6240){i=g;return}}function jR(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=d;if((a|0)==0){f=0;g=f;return g|0}while(1){if((c[b>>2]|0)>>>0>=24){break}d=(ja(e)|0)&255;h=b+4|0;c[h>>2]=c[h>>2]|d<<c[b>>2];d=b|0;c[d>>2]=(c[d>>2]|0)+8}e=c[b+4>>2]&(1<<a)-1;d=b+4|0;c[d>>2]=(c[d>>2]|0)>>>(a>>>0);d=b|0;c[d>>2]=(c[d>>2]|0)-a;f=e;g=f;return g|0}function jS(a,b){a=a|0;b=b|0;var c=0,d=0;c=i;i=i+8|0;d=a;bK(d|0,24,0)|0;jG(d,b,c|0,0)|0;i=c;return 0}function jT(a){a=a|0;var b=0,d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;b=i;i=i+2048|0;d=b|0;e=b+1024|0;f=a;bK(f|0,0,0)|0;if((bB(e|0,1,16,f|0)|0)>>>0<16){g=-1;h=g;i=b;return h|0}if((p6(e|0,23496,16)|0)!=0){g=-1;h=g;i=b;return h|0}bK(f|0,60,0)|0;a=jg(f)|0;bK(f|0,6,1)|0;j=(jc(f)|0)&65535;k=(jc(f)|0)&65535;do{if((j|0)<=256){if((k|0)>128){break}l=f;m=a+60|0;bK(l|0,m|0,0)|0;m=0;while(1){if((m|0)>=(j|0)){break}l=jg(f)|0;bK(f|0,3,1)|0;bK(f|0,l-9+((jc(f)|0)&65535)|0,1)|0;m=m+1|0}m=0;L7840:while(1){if((m|0)>=(k|0)){n=6283;break}l=jg(f)|0;if(l>>>0>263){n=6266;break}bK(f|0,-4|0,1)|0;bB(e|0,l|0,1,f|0)|0;l=(jh(e+27|0)|0)&65535;if((l|0)>255){n=6268;break}if((l|0)!=0){o=0;while(1){if((o|0)>=(l|0)){break}c[d+(o<<2)>>2]=jg(f)|0;bK(f|0,36,1)|0;o=o+1|0}o=0;while(1){if((o|0)>=(l|0)){break}jl(f)|0;if((jl(f)|0)==1332176723){n=6278;break L7840}bK(f|0,(c[d+(o<<2)>>2]|0)-8|0,1)|0;o=o+1|0}}m=m+1|0}if((n|0)==6278){g=0;h=g;i=b;return h|0}else if((n|0)==6266){g=-1;h=g;i=b;return h|0}else if((n|0)==6283){g=-1;h=g;i=b;return h|0}else if((n|0)==6268){g=-1;h=g;i=b;return h|0}}}while(0);g=-1;h=g;i=b;return h|0}function jU(b,d){b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;e=i;i=i+12296|0;f=e|0;g=e+1024|0;h=e+11264|0;j=e+12288|0;k=b;b=d;c[j>>2]=0;bK(k|0,60,0)|0;d=jg(k)|0;bK(k|0,6,1)|0;l=(jc(k)|0)&65535;m=(jc(k)|0)&65535;bK(k|0,d+60|0,0)|0;d=0;while(1){if((d|0)>=(l|0)){break}n=jg(k)|0;bK(k|0,3,1)|0;bK(k|0,n-9+((jc(k)|0)&65535)|0,1)|0;d=d+1|0}l=aT(k|0)|0;bK(k|0,0,0)|0;jr(b,k,l)|0;d=0;L7877:while(1){if((d|0)>=(m|0)){o=6328;break}l=jg(k)|0;if(l>>>0>1024){o=6299;break}bK(k|0,-4|0,1)|0;bB(f|0,l|0,1,k|0)|0;a[f+26|0]=0;ay(f|0,l|0,1,b|0)|0;l=(jh(f+27|0)|0)&65535;if((l|0)!=0){n=0;while(1){if((n|0)>=(l|0)){break}c[g+(n*40|0)>>2]=jg(k)|0;bB(g+(n*40|0)+4|0,1,36,k|0)|0;n=n+1|0}n=0;while(1){if((n|0)>=(l|0)){break}if((c[g+(n*40|0)>>2]|0)>>>0>0){p=8;if((a[g+(n*40|0)+14|0]&16|0)!=0){p=16}c[h+(n<<2)>>2]=jV(k,c[g+(n*40|0)>>2]|0,p,j)|0;c[g+(n*40|0)>>2]=c[j>>2];if((c[h+(n<<2)>>2]|0)==0){o=6312;break L7877}}n=n+1|0}n=0;while(1){if((n|0)>=(l|0)){break}jp(b,c[g+(n*40|0)>>2]|0);ay(g+(n*40|0)+4|0,1,36,b|0)|0;n=n+1|0}n=0;while(1){if((n|0)>=(l|0)){break}if((c[g+(n*40|0)>>2]|0)>>>0>0){p=c[h+(n<<2)>>2]|0;q=c[g+(n*40|0)>>2]|0;r=b;ay(p|0,1,q|0,r|0)|0;pU(c[h+(n<<2)>>2]|0)}n=n+1|0}}d=d+1|0}if((o|0)==6328){s=0;t=s;i=e;return t|0}else if((o|0)==6312){s=-1;t=s;i=e;return t|0}else if((o|0)==6299){s=-1;t=s;i=e;return t|0}return 0}function jV(e,f,g,h){e=e|0;f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;j=i;i=i+16|0;k=j+8|0;l=e;e=f;f=g;g=h;jg(l)|0;h=jl(l)|0;bK(l|0,-8|0,1)|0;m=pT(1,e)|0;n=m;if((m|0)==0){o=0;p=o;i=j;return p|0}jl(l)|0;bB(n|0,1,e-4|0,l|0)|0;if((h|0)!=1332176723){c[g>>2]=e;o=n;p=o;i=j;return p|0}h=lU(n,e,j|0,k)|0;pU(n);if((h|0)<0){o=0;p=o;i=j;return p|0}n=c[k>>2]|0;do{if((f|0)==8){q=0;while(1){if((q|0)>=(h|0)){break}a[n+q|0]=b[(c[k>>2]|0)+(q<<1)>>1]>>8&255;q=q+1|0}n=pV(c[k>>2]|0,h)|0;if((n|0)!=0){c[k>>2]=n;break}pU(c[k>>2]|0);o=0;p=o;i=j;return p|0}}while(0);if((f|0)==8){q=h-1|0;while(1){if((q|0)<=0){break}f=n+q|0;a[f]=(d[f]|0)-(d[n+(q-1)|0]|0)&255;q=q-1|0}c[g>>2]=h}else{q=h-1|0;while(1){if((q|0)<=0){break}f=(c[k>>2]|0)+(q<<1)|0;b[f>>1]=(b[f>>1]|0)-(b[(c[k>>2]|0)+(q-1<<1)>>1]|0)&65535;q=q-1|0}c[g>>2]=h<<1}o=n;p=o;i=j;return p|0}function jW(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;e=i;i=i+72|0;f=e|0;g=a;a=b;do{if((a|0)!=0){b=aE(g|0)|0;aC(b|0,f|0)|0;b=c[f+28>>2]|0;if((b|0)!=(((b|0)/2|0)<<1|0)){break}h=pS(b)|0;if((h|0)==0){break}j=h;k=b;l=g;bB(j|0,k|0,1,l|0)|0;do{if((d[h+4|0]|0|0)<9){m=6375}else{if((d[h+5|0]|0|0)<9){m=6375;break}if((d[h+6|0]|0|0)<9){m=6375;break}if((d[h+7|0]|0|0)<9){m=6375;break}if(((((d[h+4|0]|0)<<16|(d[h+5|0]|0)<<8|(d[h+6|0]|0))<<8)+(d[h+7|0]|0)&-252645136|0)!=0){break}if(((d[h+b-4|0]|0)<<16|(d[h+b-4+1|0]|0)<<8|(d[h+b-4+2|0]|0)|0)==0){break}if((jX(h,b,a)|0)==-1){break}pU(h);n=0;o=n;i=e;return o|0}}while(0);pU(h)}}while(0);n=-1;o=n;i=e;return o|0}function jX(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;f=b;b=c;c=e;e=0;if(b>>>0<16){g=-1;h=g;return h|0}do{if((d[f|0]|0|0)==80){if((d[f+1|0]|0|0)!=80){i=6397;break}if((d[f+2|0]|0|0)!=50){i=6397;break}if((d[f+3|0]|0|0)!=48){i=6397;break}if((b&3|0)==0){j=0;break}g=-1;h=g;return h|0}else{i=6397}}while(0);L8006:do{if((i|0)==6397){do{if((d[f|0]|0|0)==80){if((d[f+1|0]|0|0)!=88){break}if((d[f+2|0]|0|0)!=50){break}if((d[f+3|0]|0|0)!=48){break}if((b-2&3|0)!=0){g=-1;h=g;return h|0}else{j=1;break L8006}}}while(0);g=-1;h=g;return h|0}}while(0);i=(d[f+(b-4)|0]|0)<<16|(d[f+(b-3)|0]|0)<<8|(d[f+(b-2)|0]|0);k=pS(i)|0;if((k|0)==0){g=-1;h=g;return h|0}if((j&255|0)==0){if((jY(f+8|0,k,f+4|0,b-12|0,i,a[f+(b-1)|0]|0)|0)!=0){b=c;c=k;f=i;jZ(b,c,f)|0}else{e=-1}}else{e=-1}pU(k);g=e;h=g;return h|0}function jY(b,c,e,f,g,h){b=b|0;c=c|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;i=b;b=c;c=e;e=f;f=g;g=h;h=0;j=0;k=0;do{if((i|0)!=0){if((b|0)==0){break}if((c|0)==0){break}l=i+e|0;m=b+f|0;n=m;o=m;m=g;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6430;break}q=l-1|0;l=q;j=j|(d[q]|0)<<(h&255);h=(h&255)+8&255}if((p|0)==6430){r=0;s=r;return s|0}q=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}L8053:while(1){if(k>>>0>=f>>>0){break}m=1;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6442;break L8053}t=l-1|0;l=t;j=j|(d[t]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}if((q|0)==0){u=1;do{m=2;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6454;break L8053}t=l-1|0;l=t;j=j|(d[t]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}u=u+q|0;}while((q|0)==3);while(1){t=u;u=t-1|0;if((t|0)==0){break}m=8;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6468;break L8053}t=l-1|0;l=t;j=j|(d[t]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}if(o>>>0<=b>>>0){p=6476;break L8053}t=o-1|0;o=t;a[t]=q&255;k=k+1|0}if((k|0)==(f|0)){p=6480;break}}m=2;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6486;break L8053}t=l-1|0;l=t;j=j|(d[t]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}t=d[c+q|0]|0;u=q+2|0;if((q|0)==3){m=1;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6497;break L8053}v=l-1|0;l=v;j=j|(d[v]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){v=m;m=v-1&255;if(v<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}if((q|0)==0){t=7}m=t&255;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6509;break L8053}v=l-1|0;l=v;j=j|(d[v]|0)<<(h&255);h=(h&255)+8&255}w=0;h=(h&255)-(m&255)&255;while(1){v=m;m=v-1&255;if(v<<24>>24==0){break}w=w<<1|j&1;j=j>>>1}do{m=3;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6520;break L8053}v=l-1|0;l=v;j=j|(d[v]|0)<<(h&255);h=(h&255)+8&255}q=0;h=(h&255)-(m&255)&255;while(1){v=m;m=v-1&255;if(v<<24>>24==0){break}q=q<<1|j&1;j=j>>>1}u=u+q|0;}while((q|0)==7)}else{m=t&255;while(1){if((h&255|0)>=(m&255|0)){break}if(l>>>0<i>>>0){p=6533;break L8053}v=l-1|0;l=v;j=j|(d[v]|0)<<(h&255);h=(h&255)+8&255}w=0;h=(h&255)-(m&255)&255;while(1){t=m;m=t-1&255;if(t<<24>>24==0){break}w=w<<1|j&1;j=j>>>1}}if((o+w|0)>>>0>=n>>>0){p=6541;break}while(1){t=u;u=t-1|0;if((t|0)==0){break}q=d[o+w|0]|0;if(o>>>0<=b>>>0){p=6546;break L8053}t=o-1|0;o=t;a[t]=q&255;k=k+1|0}}if((p|0)==6468){r=0;s=r;return s|0}else if((p|0)==6454){r=0;s=r;return s|0}else if((p|0)==6509){r=0;s=r;return s|0}else if((p|0)==6486){r=0;s=r;return s|0}else if((p|0)==6541){r=0;s=r;return s|0}else if((p|0)==6546){r=0;s=r;return s|0}else if((p|0)==6520){r=0;s=r;return s|0}else if((p|0)==6497){r=0;s=r;return s|0}else if((p|0)==6442){r=0;s=r;return s|0}else if((p|0)==6476){r=0;s=r;return s|0}else if((p|0)!=6480)if((p|0)==6533){r=0;s=r;return s|0}r=1;s=r;return s|0}}while(0);r=0;s=r;return s|0}function jZ(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;d=a;a=c;if((d|0)!=0){e=(ay(b|0,1,a|0,d|0)|0)==(a|0)}else{e=0}return e&1|0}function j_(a){a=a|0;var b=0,e=0,f=0;b=a;if((c[b>>2]|0)>>>0<(c[b+4>>2]|0)>>>0){a=b|0;b=c[a>>2]|0;c[a>>2]=b+1;e=d[b]|0;f=e;return f|0}else{e=-1;f=e;return f|0}return 0}function j$(a){a=a|0;var b=0;b=a;c[b>>2]=0;c[b+4>>2]=0;return}function j0(b,d){b=b|0;d=d|0;var e=0;e=d;if((c[e+8>>2]|0)>>>0>=(c[e+12>>2]|0)>>>0){return}d=e+8|0;e=c[d>>2]|0;c[d>>2]=e+1;a[e]=b&255;return}function j1(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;do{if((c[d+4>>2]|0)==0){c[d>>2]=j_(b)|0;if((c[d>>2]|0)!=-1){c[d+4>>2]=8;break}e=-1;f=e;return f|0}}while(0);b=d+4|0;c[b>>2]=(c[b>>2]|0)-1;e=(c[d>>2]&1<<7-(c[d+4>>2]|0)|0)!=0?1:0;f=e;return f|0}function j2(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=pS(1065020)|0;h=g;if((g|0)==0){i=0;j=i;return j|0}else{g=j3(a,b,c,d,e,f,h)|0;pU(h);i=g;j=i;return j|0}return 0}function j3(a,b,e,f,g,h,j){a=a|0;b=b|0;e=e|0;f=f|0;g=g|0;h=h|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0;k=i;i=i+8|0;l=k|0;m=a;a=b;b=g;g=j;j=0;n=1;c[g+786480>>2]=h;c[g+786448>>2]=e;c[g+786484>>2]=1<<a;e=pS(b)|0;h=e;if((e|0)==0){e=c[q>>2]|0;bE(e|0,29040,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0)|0;i=e;o=0;p=o;i=k;return p|0}c[g+786464>>2]=m;c[g+786468>>2]=m+f;c[g+786472>>2]=h;c[g+786476>>2]=h+b;c[g+786440>>2]=0;c[g+786436>>2]=0;c[g+786444>>2]=0;kf(-1,0,g+786456|0,g+786464|0);c[g+786452>>2]=0;b=9;if((a|0)==0){c[g+786452>>2]=1;b=12;c[g+786484>>2]=4096}f=b;ka(f,g);c[l>>2]=0;m=0;if((c[g+786480>>2]&32|0)!=0){e=g+786464|0;c[e>>2]=(c[e>>2]|0)+1}if((a|0)==16){e=g+786464|0;r=c[e>>2]|0;c[e>>2]=r+1;c[g+786484>>2]=1<<(d[r]|0)}if((c[g+786484>>2]|0)>(1<<a|0)){o=0;p=o;i=k;return p|0}c[g+1065016>>2]=0;L8255:while(1){if((j6(l,b,g)|0)==0){s=6607;break}if((c[g+786480>>2]&1|0)!=0){if((c[l>>2]|0)==257){s=6610;break}}a=0;if((n|0)!=0){j=c[l>>2]|0;n=0;if((c[g+786452>>2]|0)!=0){a=1}}do{if((c[l>>2]|0)==256){if((c[g+786452>>2]|0)!=0){break}c[g+524288>>2]=255;if((c[g+786480>>2]&8|0)!=0){r=g+524288|0;c[r>>2]=(c[r>>2]|0)+1}kb(b,g);b=f;if((j6(l,b,g)|0)==0){s=6621;break L8255}}}while(0);if((c[g+786452>>2]|0)!=0){s=6625}else{if((c[l>>2]|0)<=(c[g+524288>>2]|0)){s=6627}else{s=6625}}do{if((s|0)==6625){s=0;if((c[g+786452>>2]|0)!=0){if((c[g+262144+(c[l>>2]<<2)>>2]|0)!=-1){s=6627;break}}kc(m,g);kd(j,g)}}while(0);if((s|0)==6627){s=0;kc(c[l>>2]|0,g);j=j7(c[l>>2]|0,g)|0}if((c[g+524288>>2]|0)!=((c[g+786484>>2]|0)-1|0)){if((a|0)==0){(ke(m,j,g)|0)==0;do{if((c[g+524288>>2]|0)!=((c[g+786484>>2]|0)-1|0)){if((c[g+524288>>2]|0)!=((1<<b)-1|0)){break}b=b+1|0;kb(b-1|0,g)}}while(0)}}m=c[l>>2]|0}do{if((~c[g+786480>>2]&2|0)!=0){if((c[g+786464>>2]|0)==(c[g+786468>>2]|0)){break}pU(h);o=0;p=o;i=k;return p|0}}while(0);o=h;p=o;i=k;return p|0}function j4(a,b,d,e,f,g,h){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0;i=a;a=b;b=f;f=g;g=h;h=pS(1065020)|0;j=h;if((h|0)==0){k=0;l=k;return l|0}h=pS(b)|0;m=h;if((h|0)==0){aF(23424);k=0;l=k;return l|0}h=aT(i|0)|0;bB(m|0,1,b|0,i|0)|0;n=j3(m,d,e,b,f,g,j)|0;b=a;e=n;d=f;p3(b|0,e|0,d)|0;if((g&16|0)!=0){o=(c[j+1065016>>2]|0)+3&-4}else{o=c[j+1065016>>2]|0}bK(i|0,h+o|0,0)|0;pU(n);pU(m);pU(j);k=a;l=k;return l|0}function j5(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0;e=i;i=i+32|0;f=e|0;g=e+8|0;h=e+16|0;j=a;a=d;d=pS(a)|0;k=d;if((d|0)==0){d=c[q>>2]|0;bE(d|0,23448,(l=i,i=i+1|0,i=i+7>>3<<3,c[l>>2]=0,l)|0)|0;i=l;aV(1);return 0}c[h>>2]=j;c[h+4>>2]=j+b;c[h+8>>2]=k;c[h+12>>2]=k+a;a=j_(h)|0;a=a|(j_(h)|0)<<8;if((a|0)==0){pU(k);m=0;n=m;i=e;return n|0}b=pS(a<<3)|0;j=b;if((b|0)==0){b=c[q>>2]|0;bE(b|0,23448,(l=i,i=i+1|0,i=i+7>>3<<3,c[l>>2]=0,l)|0)|0;i=l;aV(1);return 0}c[j+4>>2]=65279;c[j>>2]=65279;l=0;while(1){if((l|0)>=(a|0)){break}c[j+(l<<3)>>2]=j_(h)|0;b=(j_(h)|0)<<8;d=j+(l<<3)|0;c[d>>2]=c[d>>2]|b;c[j+(l<<3)+4>>2]=j_(h)|0;b=(j_(h)|0)<<8;d=j+(l<<3)+4|0;c[d>>2]=c[d>>2]|b;l=l+1|0}j$(f);kf(-1,0,g,h);L8342:while(1){l=0;while(1){if((l&32768|0)!=0){break}if((l|0)>=(a|0)){o=6675;break L8342}b=j1(f,h)|0;if((b|0)==-1){o=6677;break}l=c[j+(l<<3)+(b<<2)>>2]|0}if((o|0)==6677){o=0;l=65279}l=l^65535;if((l|0)!=256){kf(l,6,g,h)}if((l|0)==256){o=6683;break}}if((o|0)==6683){pU(j);m=k;n=m;i=e;return n|0}else if((o|0)==6675){pU(j);pU(k);m=0;n=m;i=e;return n|0}return 0}function j6(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0;f=a;a=b;b=e;e=0;g=0;c[f>>2]=0;while(1){if((g|0)>=(a|0)){break}if((c[b+786440>>2]|0)==0){if((c[b+786464>>2]|0)>>>0>=(c[b+786468>>2]|0)>>>0){h=6692;break}i=b+786464|0;j=c[i>>2]|0;c[i>>2]=j+1;c[b+786436>>2]=d[j]|0;c[b+786440>>2]=8;j=b+1065016|0;c[j>>2]=(c[j>>2]|0)+1}if((c[b+786440>>2]|0)<(a-g|0)){e=c[b+786440>>2]|0}else{e=a-g|0}if((c[b+786452>>2]|0)!=0){j=b+786436|0;c[j>>2]=c[j>>2]&255;j=b+786436|0;c[j>>2]=c[j>>2]<<e;g=g+e|0;j=f;c[j>>2]=c[j>>2]|c[b+786436>>2]>>8<<a-g;j=b+786440|0;c[j>>2]=(c[j>>2]|0)-e}else{j=f;c[j>>2]=c[j>>2]|(c[b+786436>>2]&(1<<e)-1)<<g;j=b+786436|0;c[j>>2]=c[j>>2]>>e;j=b+786440|0;c[j>>2]=(c[j>>2]|0)-e;g=g+e|0}}if((h|0)==6692){k=0;l=k;return l|0}do{if((c[f>>2]|0)>=0){if((c[f>>2]|0)>((c[b+786484>>2]|0)-1|0)){break}h=b+786444|0;c[h>>2]=(c[h>>2]|0)+1;h=b+786444|0;c[h>>2]=c[h>>2]&7;k=1;l=k;return l|0}}while(0);k=0;l=k;return l|0}function j7(a,b){a=a|0;b=b|0;var d=0;d=a;a=b;if((c[a+(d<<2)>>2]|0)!=-1){d=c[a+524292+(d<<2)>>2]|0}return c[a+262144+(d<<2)>>2]|0}function j8(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=d;d=(a+b|2048)&65535;b=(ac(d,d)|0)>>6&4095;while(1){if((c[e+262144+(b<<2)>>2]|0)!=-1){f=(c[e+1048632+(b<<2)>>2]|0)!=-1}else{f=0}if(!f){break}b=c[e+1048632+(b<<2)>>2]|0}if((c[e+262144+(b<<2)>>2]|0)==-1){g=b;h=g;return h|0}f=b;b=b+101|0;b=b&4095;do{if((c[e+262144+(b<<2)>>2]|0)!=-1){d=0;while(1){if((d|0)>=(c[e+786484>>2]|0)){break}if((c[e+262144+(b<<2)>>2]|0)==-1){a=6723;break}d=d+1|0;b=b+1|0;b=b&4095}if((b|0)!=(c[e+786484>>2]|0)){break}g=-1;h=g;return h|0}}while(0);c[e+1048632+(f<<2)>>2]=b;g=b;h=g;return h|0}function j9(b,d){b=b|0;d=d|0;var e=0;e=d;if((c[e+8>>2]|0)>>>0>=(c[e+12>>2]|0)>>>0){return}d=e+8|0;e=c[d>>2]|0;c[d>>2]=e+1;a[e]=b&255;return}function ka(a,b){a=a|0;b=b|0;var d=0;d=b;b=1<<a-1;a=0;while(1){if((a|0)>=65536){break}c[d+262144+(a<<2)>>2]=-1;c[d+(a<<2)>>2]=-1;c[d+524292+(a<<2)>>2]=-1;a=a+1|0}a=0;while(1){if((a|0)>=4096){break}c[d+1048632+(a<<2)>>2]=-1;a=a+1|0}if((c[d+786452>>2]|0)!=0){c[d+524288>>2]=-1;a=0;while(1){if((a|0)>=256){break}ke(65535,a,d)|0;a=a+1|0}return}a=0;while(1){if((a|0)>=(b|0)){break}c[d+262144+(a<<2)>>2]=a;a=a+1|0}c[d+524288>>2]=b-1;if((c[d+786480>>2]&8|0)!=0){b=d+524288|0;c[b>>2]=(c[b>>2]|0)+1}return}function kb(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=i;i=i+8|0;e=d|0;f=a;a=b;if((c[a+786480>>2]&4|0)!=0){i=d;return}do{if((c[a+786444>>2]|0)==0){g=6772;break}}while((j6(e,f,a)|0)!=0);if((g|0)==6772){i=d;return}i=d;return}function kc(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;b=a+786488|0;while(1){if((c[a+(d<<2)>>2]|0)!=-1){e=b>>>0<(a+786488+(c[a+786484>>2]<<2)|0)>>>0}else{e=0}if(!e){break}f=b;b=f+4|0;c[f>>2]=c[a+262144+(d<<2)>>2];d=c[a+(d<<2)>>2]|0}kd(c[a+262144+(d<<2)>>2]|0,a);while(1){if(b>>>0<=(a+786488|0)>>>0){break}d=b-4|0;b=d;kd(c[d>>2]|0,a)}return}function kd(a,b){a=a|0;b=b|0;var d=0;d=a;a=b;if((c[a+786448>>2]|0)!=0){kf(d,2,a+786456|0,a+786464|0);return}else{j9(d,a+786464|0);return}}function ke(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=d;d=b+524288|0;c[d>>2]=(c[d>>2]|0)+1;if((c[b+524288>>2]&c[b+786484>>2]|0)!=0){c[b+524288>>2]=(c[b+786484>>2]|0)-1;f=1;g=f;return g|0}d=c[b+524288>>2]|0;do{if((c[b+786452>>2]|0)!=0){h=j8(e,a,b)|0;d=h;if((h|0)!=-1){break}f=0;g=f;return g|0}}while(0);c[b+262144+(d<<2)>>2]=a;if((e|0)>=(c[b+786484>>2]|0)){f=1;g=f;return g|0}c[b+(d<<2)>>2]=e;if((c[b+(e<<2)>>2]|0)==-1){c[b+524292+(d<<2)>>2]=e}else{c[b+524292+(d<<2)>>2]=c[b+524292+(e<<2)>>2]}f=1;g=f;return g|0}function kf(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0;f=a;a=b;b=d;d=e;if((f|0)==-1){c[b+4>>2]=0;c[b>>2]=0;return}if((c[b+4>>2]|0)==0){if((f|0)==144){c[b+4>>2]=1}else{b1[a&15](f,d);c[b>>2]=f}return}if((f|0)==0){b1[a&15](144,d)}else{e=1;while(1){if((e|0)>=(f|0)){break}b1[a&15](c[b>>2]|0,d);e=e+1|0}}c[b+4>>2]=0;return}function kg(b,d){b=b|0;d=d|0;var e=0;e=d;if((c[e+8>>2]|0)>>>0>=(c[e+12>>2]|0)>>>0){return}d=e+8|0;e=c[d>>2]|0;c[d>>2]=e+1;a[e]=b&255;return}function kh(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0;d=i;i=i+96|0;e=d|0;f=d+16|0;g=d+24|0;h=a;a=b;b=0;if((aC(aE(h|0)|0,g|0)|0)!=0){j=-1;k=j;i=d;return k|0}l=pS(c[g+28>>2]|0)|0;m=l;if((m|0)==0){j=-1;k=j;i=d;return k|0}bB(l|0,1,c[g+28>>2]|0,h|0)|0;do{if((ki(m,e,f,d+8|0)|0)==0){h=pS(c[e>>2]|0)|0;b=h;if((h|0)==0){break}kl(m+16|0,b,c[e>>2]|0,c[f>>2]|0);if((ay(b|0,c[e>>2]|0,1,a|0)|0)==0){pU(b);break}else{pU(b);pU(m);j=0;k=j;i=d;return k|0}}}while(0);pU(m);j=-1;k=j;i=d;return k|0}function ki(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0;f=a;a=b;b=d;d=e;if((p6(f|0,24448,4)|0)!=0){g=-1;h=g;return h|0}c[d>>2]=jk(f+4|0)|0;if((c[d>>2]|0)<0){g=-1;h=g;return h|0}c[a>>2]=jk(f+8|0)|0;if((c[a>>2]|0)<0){g=-1;h=g;return h|0}c[b>>2]=jk(f+12|0)|0;if((c[b>>2]|0)<0){g=-1;h=g;return h|0}else{g=0;h=g;return h|0}return 0}function kj(a,b,d){a=a|0;b=b|0;d=d|0;var e=0;e=a;a=b;c[e+8>>2]=a+d;c[e+12>>2]=a;c[e+4>>2]=(ji(c[e+8>>2]|0)|0)&65535;a=e+4|0;c[a>>2]=c[a>>2]&15;a=e+8|0;c[a>>2]=(c[a>>2]|0)-2;c[e>>2]=(ji(c[e+8>>2]|0)|0)&65535;a=e+8|0;c[a>>2]=(c[a>>2]|0)-2;a=(ji(c[e+8>>2]|0)|0)&65535;d=e+8|0;c[d>>2]=(c[d>>2]|0)-2;return a|0}function kk(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0;f=i;i=i+24|0;g=f|0;h=f+8|0;j=a;a=e;e=pS(a)|0;k=e;if((e|0)==0){e=c[q>>2]|0;bE(e|0,23384,(e=i,i=i+1|0,i=i+7>>3<<3,c[e>>2]=0,e)|0)|0;i=e;aV(1);return 0}c[h>>2]=j;c[h+4>>2]=j+b;c[h+8>>2]=k;c[h+12>>2]=k+a;kf(-1,0,g,h);while(1){if((c[h>>2]|0)>>>0>=(c[h+4>>2]|0)>>>0){break}a=h|0;b=c[a>>2]|0;c[a>>2]=b+1;kf(d[b]|0,8,g,h)}i=f;return k|0}function kl(b,c,d,e){b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+16|0;g=f|0;h=c;c=d;d=c;j=h+d|0;k=kj(g,b,e)|0;while(1){if((d|0)<=0){break}e=kr(g,9)|0;if((e&65535|0)<256){if(j>>>0<=h>>>0){bV(23352,140,31504,29024)}b=j-1|0;j=b;a[b]=e&255;d=d-1|0}else{do{if((e&65535|0)==318){l=6879}else{if((e&65535|0)==319){l=6879;break}if((e&65535|0)>=384){m=(e&64|0)!=0?3:2;if((e&32|0)!=0){e=(e&31)<<k-5&65535;e=(e&65535|(kr(g,k-5|0)|0)&65535)&65535;e=(e&65535)+544&65535}else{if((e&48|0)!=0){e=(e&15)<<1&65535;e=(e&65535|(kr(g,1)|0)&65535)&65535}else{e=(e&15)<<5&65535;e=(e&65535|(kr(g,5)|0)&65535)&65535;e=(e&65535)+32&65535}}}else{if((e&65535|0)>=320){m=((e&48)>>4)+4|0;if((e&8|0)!=0){e=(e&7)<<k-3&65535;e=(e&65535|(kr(g,k-3|0)|0)&65535)&65535;e=(e&65535)+544&65535}else{if((e&12|0)!=0){e=(e&3)<<3&65535;e=(e&65535|(kr(g,3)|0)&65535)&65535}else{e=(e&3)<<7&65535;e=(e&65535|(kr(g,7)|0)&65535)&65535;e=(e&65535)+32&65535}}}else{if((e&65535|0)>=288){m=((e&30)>>1)+8|0;if((e&1|0)!=0){e=kr(g,k)|0;e=(e&65535)+544&65535}else{e=kr(g,6)|0;if((e&32|0)!=0){e=e&65535&31}else{e=(e&65535)<<4&65535;e=(e&65535|(kr(g,4)|0)&65535)&65535;e=(e&65535)+32&65535}}}else{e=(e&31)<<3&65535;e=(e&65535|(kr(g,3)|0)&65535)&65535;m=23;while(1){if((e&65535|0)!=255){break}m=m+(e&65535)|0;e=kr(g,8)|0}m=m+(e&65535)|0;e=kr(g,7)|0;if((e&64|0)!=0){e=(e&63)<<k-6&65535;e=(e&65535|(kr(g,k-6|0)|0)&65535)&65535;e=(e&65535)+544&65535}else{if((e&32|0)!=0){e=e&65535&31}else{e=(e&65535)<<4&65535;e=(e&65535|(kr(g,4)|0)&65535)&65535;e=(e&65535)+32&65535}}}}}d=d-m|0;while(1){b=m;m=b-1|0;if((b|0)<=0){break}j=j-1|0;if(j>>>0<h>>>0){bV(23352,266,31504,27504)}if((j+(e&65535)+1|0)>>>0>=(h+c|0)>>>0){bV(23352,267,31504,26312)}a[j]=a[j+((e&65535)+1)|0]|0}}}while(0);if((l|0)==6879){l=0;e=(e&65535)<<4&65535;e=(e&65535|(kr(g,4)|0)&65535)&65535;m=(e&31)+14|0;d=d-m|0;while(1){b=m;m=b-1|0;if((b|0)<=0){break}e=kr(g,8)|0;if(j>>>0<=h>>>0){bV(23352,154,31504,29024)}b=j-1|0;j=b;a[b]=e&255}}}}i=f;return}function km(a,b){a=a|0;b=b|0;var c=0,d=0;c=b;do{if((c|0)==0){d=-1}else{if((kn(a,c)|0)<0){d=-1;break}else{d=0;break}}}while(0);return d|0}function kn(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0;f=i;i=i+40|0;g=f|0;h=b;b=e;e=0;do{if((ko(h)|0)!=0){if((kp(h,g)|0)==0){break}while(1){if((a7(g+1|0,23296)|0)!=0){j=(a[g+1|0]|0)==33}else{j=1}if(!j){k=6953;break}if((kq(h,g)|0)==0){k=6949;break}if((kp(h,g)|0)==0){k=6951;break}}if((k|0)==6953){if((d[g|0]|0)==0){l=-1;m=l;i=f;return m|0}n=ks(h,g)|0;o=n;if((n|0)==0){n=c[q>>2]|0;bE(n|0,28984,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p;l=-1;m=l;i=f;return m|0}n=0;r=0;s=d[g|0]|0;if((s|0)==6){r=1;n=j2(o,0,1,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((s|0)==3){r=1;n=kk(o,c[g+16>>2]|0,c[g+32>>2]|0)|0}else if((s|0)==1|(s|0)==2){r=1;n=o}else if((s|0)==9){r=1;n=j2(o,13,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((s|0)==8){r=1;n=j2(o,12,1,c[g+16>>2]|0,c[g+32>>2]|0,32)|0}else if((s|0)==5){r=1;n=j2(o,0,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}else if((s|0)==4){r=1;n=j5(o,c[g+16>>2]|0,c[g+32>>2]|0)|0}else if((s|0)!=7)if((s|0)==127){r=1;n=j2(o,16,0,c[g+16>>2]|0,c[g+32>>2]|0,0)|0}if((n|0)==0){if((r|0)!=0){r=c[q>>2]|0;bE(r|0,27456,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}else{r=c[q>>2]|0;s=d[g|0]|0;bE(r|0,26272,(p=i,i=i+8|0,c[p>>2]=s,p)|0)|0;i=p}e=1}else{while(1){s=aR(g+1|0,47)|0;if((s|0)==0){break}a[s]=95}s=ay(n|0,1,c[g+32>>2]|0,b|0)|0;if((s|0)!=(c[g+32>>2]|0)){s=c[q>>2]|0;r=bx(c[(bw()|0)>>2]|0)|0;bE(s|0,25200,(p=i,i=i+8|0,c[p>>2]=r,p)|0)|0;i=p;e=1}if((n|0)!=(o|0)){pU(n)}}pU(o);l=e;m=l;i=f;return m|0}else if((k|0)==6951){l=-1;m=l;i=f;return m|0}else if((k|0)==6949){l=-1;m=l;i=f;return m|0}}}while(0);l=-1;m=l;i=f;return m|0}function ko(a){a=a|0;var b=0,c=0,d=0,e=0,f=0,g=0,h=0;b=a;a=0;c=0;while(1){if((c|0)>=4){break}d=aN(b|0)|0;e=d;if((d|0)==-1){f=6991;break}if((e|0)==26){f=6993;break}c=c+1|0}if((f|0)==6993){a=1;c=e;e=b;a$(c|0,e|0)|0}else if((f|0)==6991){g=0;h=g;return h|0}g=a;h=g;return h|0}function kp(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+16|0;g=f|0;h=b;b=e;e=14;a[b|0]=-1;if((aN(h|0)|0)!=26){j=0;k=j;i=f;return k|0}l=aN(h|0)|0;m=l;if((l|0)==-1){j=0;k=j;i=f;return k|0}l=m>>7;a[b|0]=m&127;if((d[b|0]|0|0)==0){j=1;k=j;i=f;return k|0}if((d[b|0]|0|0)==1){e=e-4|0}do{if((bB(b+1|0,1,13,h|0)|0)==13){if((bB(g|0,1,e|0,h|0)|0)!=(e|0)){break}c[b+16>>2]=d[g|0]|0|(d[g+1|0]|0)<<8|(d[g+2|0]|0)<<16|(d[g+3|0]|0)<<24;c[b+20>>2]=d[g+4|0]|0|(d[g+5|0]|0)<<8;c[b+24>>2]=d[g+6|0]|0|(d[g+7|0]|0)<<8;c[b+28>>2]=d[g+8|0]|0|(d[g+9|0]|0)<<8;c[b+36>>2]=1;if((d[b|0]|0|0)==1){c[b+32>>2]=c[b+16>>2]}else{c[b+32>>2]=d[g+10|0]|0|(d[g+11|0]|0)<<8|(d[g+12|0]|0)<<16|(d[g+13|0]|0)<<24}a[b+13|0]=0;if((l|0)!=0){if((bB(g|0,1,12,h|0)|0)!=12){j=0;k=j;i=f;return k|0}do{if((d[b|0]|0|0)==2){if((d[g+3|0]|0|0)!=255){break}if((d[g+2|0]|0|0)!=253){break}if((d[g+1|0]|0|0)!=220){break}c[b+36>>2]=0}}while(0)}j=1;k=j;i=f;return k|0}}while(0);j=0;k=j;i=f;return k|0}function kq(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=a;a=c[b+16>>2]|0;b=0;while(1){if((b|0)>=(a|0)){e=7037;break}if((aN(d|0)|0)==-1){e=7034;break}b=b+1|0}if((e|0)==7034){f=0;g=f;return g|0}else if((e|0)==7037){f=1;g=f;return g|0}return 0}function kr(a,b){a=a|0;b=b|0;var d=0,e=0;d=a;a=b;b=d|0;c[b>>2]=c[b>>2]&65535;if((c[d+4>>2]|0)<(a|0)){b=d|0;c[b>>2]=c[b>>2]<<c[d+4>>2];if((c[d+8>>2]|0)>>>0<(c[d+12>>2]|0)>>>0){bV(23352,78,31432,25240);return 0}b=(ji(c[d+8>>2]|0)|0)&65535;e=d|0;c[e>>2]=c[e>>2]|b;b=d+8|0;c[b>>2]=(c[b>>2]|0)-2;a=a-(c[d+4>>2]|0)|0;c[d+4>>2]=16}b=d+4|0;c[b>>2]=(c[b>>2]|0)-a;b=d|0;c[b>>2]=c[b>>2]<<a;return(c[d>>2]|0)>>>16&65535|0}function ks(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=i;e=c[b+16>>2]|0;b=pS(e)|0;f=b;if((b|0)==0){b=c[q>>2]|0;bE(b|0,24416,(b=i,i=i+1|0,i=i+7>>3<<3,c[b>>2]=0,b)|0)|0;i=b;aV(1);return 0}if((bB(f|0,1,e|0,a|0)|0)==(e|0)){g=f;i=d;return g|0}pU(f);f=0;g=f;i=d;return g|0}function kt(f,g){f=f|0;g=g|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0;h=i;i=i+418176|0;j=h|0;k=h+1088|0;l=h+4160|0;m=h+280168|0;n=f;f=g;g=0;o=0;p=bB(j|0,1,1024,n|0)|0;o=o+p|0;do{if((o|0)>=3){if((d[j|0]|0|0)!=31){break}if((d[j+1|0]|0|0)!=157){break}r=a[j+2|0]&31;s=a[j+2|0]&128;t=1<<r;if((r|0)>16){u=-1;v=u;i=h;return v|0}g=o;w=9;x=511;y=(1<<w)-1|0;z=-1;A=0;B=0;C=24;D=(s|0)!=0?257:256;p7(m|0,0,256);E=255;while(1){if((E|0)<0){break}a[l+E|0]=E&255;E=E-1|0}L8828:while(1){L8830:while(1){F=C>>3;if((F|0)<=(o|0)){G=o-F|0}else{G=0}H=G;I=0;while(1){if((I|0)>=(H|0)){break}a[j+I|0]=a[j+(I+F)|0]|0;I=I+1|0}o=H;C=0;if(o>>>0<64){I=bB(j+o|0,1,1024,n|0)|0;p=I;if((I|0)<0){J=7075;break L8828}o=o+p|0}if((p|0)>0){K=o-((o|0)%(w|0)|0)<<3}else{K=(o<<3)-(w-1)|0}I=K;while(1){if((I|0)<=(C|0)){break L8830}if((D|0)>(x|0)){break}F=j+(C>>3)|0;E=(d[F|0]|0|(d[F+1|0]|0)<<8|(d[F+2|0]|0)<<16)>>(C&7)&y;C=C+w|0;do{}while(0);if((z|0)==-1){if((E|0)>=256){J=7092;break L8828}F=E;z=F;A=F;L=B;B=L+1|0;a[k+L|0]=F&255;continue}if((E|0)==256){if((s|0)!=0){J=7096;break}}F=E;L=l+276e3|0;if((E|0)>=(D|0)){if((E|0)>(D|0)){J=7099;break L8828}M=L-1|0;L=M;a[M]=A&255;E=z}while(1){if((E|0)<256){break}M=L-1|0;L=M;a[M]=a[l+E|0]|0;E=e[m+(E<<1)>>1]|0}M=d[l+E|0]|0;A=M;N=L-1|0;L=N;a[N]=M&255;M=l+276e3-L|0;N=M;if((B+M|0)>=1024){do{if((N|0)>(1024-B|0)){N=1024-B|0}if((N|0)>0){M=k+B|0;O=L;P=N;p3(M|0,O|0,P)|0;B=B+N|0}if((B|0)>=1024){if((ay(k|0,1,B|0,f|0)|0)!=(B|0)){J=7112;break L8828}B=0}L=L+N|0;P=l+276e3-L|0;N=P;}while((P|0)>0)}else{P=k+B|0;O=L;M=N;p3(P|0,O|0,M)|0;B=B+N|0}M=D;E=M;if((M|0)<(t|0)){b[m+(E<<1)>>1]=z&65535;a[l+E|0]=A&255;D=E+1|0}z=F}if((J|0)==7096){J=0;p7(m|0,0,256);D=256;C=C-1+((w<<3)-((C-1+(w<<3)|0)%(w<<3|0)|0))|0;w=9;x=511;y=(1<<w)-1|0;continue}C=C-1+((w<<3)-((C-1+(w<<3)|0)%(w<<3|0)|0))|0;w=w+1|0;if((w|0)==(r|0)){x=t}else{x=(1<<w)-1|0}y=(1<<w)-1|0}g=g+p|0;if((p|0)<=0){J=7123;break}}if((J|0)==7075){u=-1;v=u;i=h;return v|0}else if((J|0)==7099){bE(c[q>>2]|0,28912,(Q=i,i=i+1|0,i=i+7>>3<<3,c[Q>>2]=0,Q)|0)|0;i=Q;u=-1;v=u;i=h;return v|0}else if((J|0)==7092){bE(c[q>>2]|0,23240,(Q=i,i=i+8|0,c[Q>>2]=E,Q)|0)|0;i=Q;bE(c[q>>2]|0,28912,(Q=i,i=i+1|0,i=i+7>>3<<3,c[Q>>2]=0,Q)|0)|0;i=Q;u=-1;v=u;i=h;return v|0}else if((J|0)==7123){do{if((B|0)>0){if((ay(k|0,1,B|0,f|0)|0)==(B|0)){break}u=-1;v=u;i=h;return v|0}}while(0);u=0;v=u;i=h;return v|0}else if((J|0)==7112){u=-1;v=u;i=h;return v|0}}}while(0);u=-1;v=u;i=h;return v|0}function ku(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0;d=i;i=i+272|0;e=d|0;f=a;a=b;while(1){if((kv(f,e)|0)<0){break}if((cK(e+4|0)|0)==0){g=7141;break}bK(f|0,c[e+260>>2]|0,1)|0}if((g|0)==7141){h=kw(f,a,c[e+264>>2]|0,c[e>>2]|0)|0;j=h;i=d;return j|0}h=-1;j=h;i=d;return j|0}function kv(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;e=i;i=i+24|0;f=e|0;g=a;a=b;p7(a|0,0,272);if((bB(f|0,1,21,g|0)|0)!=21){h=-1;j=h;i=e;return j|0}b=d[f+20|0]|0;if((b|0)==3){c[a>>2]=jk(f+2|0)|0;c[a+260>>2]=jj(f+7|0)|0;c[a+264>>2]=jj(f+11|0)|0;c[a+268>>2]=(jc(g)|0)&65535;k=g;ja(k)|0;l=jg(g)|0;k=g;m=l-21|0;bK(k|0,m|0,1)|0}else if((b|0)==0){l=d[f|0]|0;c[a>>2]=jk(f+2|0)|0;c[a+260>>2]=jj(f+7|0)|0;c[a+264>>2]=jj(f+11|0)|0;n=(ja(g)|0)&255;bB(a+4|0,1,n|0,g|0)|0;c[a+268>>2]=(jc(g)|0)&65535;bK(g|0,l+2-24-n|0,1)|0}else if((b|0)==1){l=d[f|0]|0;c[a>>2]=jk(f+2|0)|0;c[a+260>>2]=jj(f+7|0)|0;c[a+264>>2]=jj(f+11|0)|0;n=(ja(g)|0)&255;bB(a+4|0,1,n|0,g|0)|0;c[a+268>>2]=(jc(g)|0)&65535;bK(g|0,l+2-26-n|0,1)|0;while(1){n=(jc(g)|0)&65535;l=n;if((n|0)==0){break}bK(g|0,l-2|0,1)|0}}else if((b|0)==2){l=(jh(f|0)|0)&65535;c[a>>2]=jk(f+2|0)|0;c[a+260>>2]=jj(f+7|0)|0;c[a+264>>2]=jj(f+11|0)|0;c[a+268>>2]=(jc(g)|0)&65535;bK(g|0,l+2-21|0,1)|0}else{h=-1;j=h;i=e;return j|0}h=0;j=h;i=e;return j|0}function kw(d,f,g,h){d=d|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;i=f;f=g;g=h;h=0;j=pT(25232,1)|0;k=j;if((j|0)==0){h=-1;l=h;return l|0}c[k+4>>2]=d;b[k+12>>1]=13;d=20;j=16;m=28;n=g;if((n|0)==762079281){b[k+12>>1]=12;d=18;m=10;j=8}else if((n|0)==762079284){b[k+12>>1]=12;o=7168}else if((n|0)==762079288){b[k+12>>1]=17}else if((n|0)==762079287){b[k+12>>1]=16}else if((n|0)==762079283){d=22;j=8;m=26}else if((n|0)==762079286){b[k+12>>1]=15}else if((n|0)==762079285){o=7168}else if((n|0)==762079282){d=6;m=10;j=18}else{h=1}if((h|0)==0){n=1<<e[k+12>>1];if((g|0)==762083955){p=1}else{p=(g|0)==762342706}g=p?254:253;p=pT(n,1)|0;c[k+8>>2]=p;q=p;if((p|0)!=0){p7(q|0,32,n|0);b0[d&31](k);n=n-1|0;while(1){if((c[k+24>>2]|0)>>>0>=f>>>0){o=7179;break}d=(b_[m&31](k)|0)&65535;if((c[k>>2]|0)!=0){o=7181;break}if((d|0)<=255){p=(aw(d|0,i|0)|0)&255;r=k+20|0;s=c[r>>2]|0;c[r>>2]=s+1;a[q+s|0]=p;p=k+20|0;c[p>>2]=c[p>>2]&n;p=k+24|0;c[p>>2]=(c[p>>2]|0)+1}else{d=d-g|0;p=c[k+20>>2]|0;s=p-((b_[j&31](k)|0)&65535)-1|0;p=k+24|0;c[p>>2]=(c[p>>2]|0)+d;while(1){p=d;d=p-1|0;if((p|0)==0){break}p=s;s=p+1|0;r=(aw(a[q+(p&n)|0]|0,i|0)|0)&255;p=k+20|0;t=c[p>>2]|0;c[p>>2]=t+1;a[q+t|0]=r;r=k+20|0;c[r>>2]=c[r>>2]&n}}}h=c[k>>2]|0;pU(q)}else{h=-1}}pU(k);l=h;return l|0}function kx(a){a=a|0;var b=0,d=0;b=a;if((e[b+12>>1]|0|0)<=13){c[b+36>>2]=14;c[b+32>>2]=4;d=b;kN(d);return}if((e[b+12>>1]|0|0)==16){c[b+36>>2]=17}else{c[b+36>>2]=16}c[b+32>>2]=5;d=b;kN(d);return}function ky(f){f=f|0;var g=0,h=0,i=0;g=f;f=b[g+8258+((e[g+14>>1]|0)>>8<<1)>>1]|0;if((f&65535|0)<(c[g+36>>2]|0)){kM(g,a[g+25104+(f&65535)|0]|0)}else{kM(g,8);h=-32768;do{if(((e[g+14>>1]|0)&(h&65535)|0)!=0){f=b[g+10808+((f&65535)<<1)>>1]|0}else{f=b[g+8770+((f&65535)<<1)>>1]|0}h=(h&65535)>>1&65535;}while((f&65535|0)>=(c[g+36>>2]|0));kM(g,(d[g+25104+(f&65535)|0]|0)-8&255)}if(f<<16>>16==0){i=f;return i|0}f=(1<<(f&65535)-1)+((kH(g,(f&65535)-1&255)|0)&65535)&65535;i=f;return i|0}function kz(f){f=f|0;var g=0,h=0,i=0;g=f;if((b[g+64>>1]|0)==0){b[g+64>>1]=kH(g,16)|0;kV(g,19,5,3);kX(g);kV(g,c[g+36>>2]&65535,c[g+32>>2]&65535,-1)}f=g+64|0;b[f>>1]=(b[f>>1]|0)-1&65535;f=b[g+66+(e[g+14>>1]>>4<<1)>>1]|0;if((f&65535|0)<510){kM(g,a[g+24594+(f&65535)|0]|0);h=f;return h|0}kM(g,12);i=-32768;do{if((e[g+14>>1]&(i&65535)|0)!=0){f=b[g+10808+((f&65535)<<1)>>1]|0}else{f=b[g+8770+((f&65535)<<1)>>1]|0}i=(i&65535)>>1&65535;}while((f&65535|0)>=510);kM(g,(d[g+24594+(f&65535)|0]|0)-12&255);h=f;return h|0}function kA(a){a=a|0;var d=0;d=a;c[d+56>>2]=314;b[d+60>>1]=60;kN(d);c[d+36>>2]=64;kS(d);kJ(d,0);kL(d,c[d+36>>2]&65535,d+25104|0,8,d+8258|0);return}function kB(a){a=a|0;var d=0,e=0,f=0,g=0,h=0,i=0;d=a;a=b[d+14870>>1]|0;e=b[d+14>>1]|0;f=0;do{a=b[d+14870+(a-((e<<16>>16|0)<0)<<1)>>1]|0;e=e<<16>>16<<1&65535;g=f+1&65535;f=g;if((g<<16>>16|0)==16){kM(d,16);e=b[d+14>>1]|0;f=0}}while((a|0)>0);kM(d,f&255);a=~a;kU(d,a);if((a|0)!=(c[d+44>>2]|0)){h=a;i=h&65535;return i|0}a=a+((kH(d,8)|0)&65535)|0;h=a;i=h&65535;return i|0}function kC(b){b=b|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;f=b;b=e[f+8258+((e[f+14>>1]|0)>>8<<1)>>1]|0;if((b|0)<(c[f+36>>2]|0)){kM(f,a[f+25104+b|0]|0);g=b;h=g<<6;i=f;j=kH(i,6)|0;k=j&65535;l=h+k|0;m=l&65535;return m|0}kM(f,8);n=e[f+14>>1]|0;do{if(((n&65535)<<16>>16|0)<0){b=e[f+10808+(b<<1)>>1]|0}else{b=e[f+8770+(b<<1)>>1]|0}n=n<<1;}while((b|0)>=(c[f+36>>2]|0));kM(f,(d[f+25104+b|0]|0)-8&255);g=b;h=g<<6;i=f;j=kH(i,6)|0;k=j&65535;l=h+k|0;m=l&65535;return m|0}function kD(a){a=a|0;var d=0;d=a;c[d+56>>2]=286;b[d+60>>1]=256;kN(d);kS(d);kT(d);return}function kE(a){a=a|0;var d=0,e=0,f=0,g=0;d=a;while(1){if((c[d+24>>2]|0)>>>0<=(c[d+28>>2]|0)>>>0){break}kO(d,(c[d+28>>2]|0)/64|0);a=d+28|0;e=(c[a>>2]|0)+64|0;c[a>>2]=e;if(e>>>0>=(c[d+40>>2]|0)>>>0){c[d+28>>2]=-1}}e=b[d+16126>>1]|0;a=b[d+14>>1]|0;f=0;while(1){if((e|0)<=0){break}e=b[d+14870+(e-((a<<16>>16|0)<0)<<1)>>1]|0;a=a<<16>>16<<1&65535;g=f+1&65535;f=g;if((g<<16>>16|0)==16){kM(d,16);a=b[d+14>>1]|0;f=0}}kM(d,f&255);e=~e-314|0;kR(d,e);return(e<<6)+((kH(d,6)|0)&65535)&65535|0}function kF(a){a=a|0;var d=0;d=a;c[d+56>>2]=286;b[d+60>>1]=256;kN(d);c[d+36>>2]=1024;return}function kG(c){c=c|0;var f=0,g=0,h=0,i=0;f=c;if((b[f+64>>1]|0)==0){b[f+64>>1]=kH(f,16)|0;kI(f);if((kH(f,1)|0)<<16>>16!=0){kK(f)}else{kJ(f,1)}kL(f,17,f+25104|0,8,f+8258|0)}c=f+64|0;b[c>>1]=(b[c>>1]|0)-1&65535;c=e[f+66+(e[f+14>>1]>>4<<1)>>1]|0;if((c|0)<286){kM(f,a[f+24594+c|0]|0)}else{kM(f,12);g=e[f+14>>1]|0;do{if(((g&65535)<<16>>16|0)<0){c=e[f+10808+(c<<1)>>1]|0}else{c=e[f+8770+(c<<1)>>1]|0}g=g<<1;}while((c|0)>=286);kM(f,(d[f+24594+c|0]|0)-12&255)}if((c|0)!=285){h=c;i=h&65535;return i|0}c=c+((kH(f,8)|0)&65535)|0;h=c;i=h&65535;return i|0}function kH(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;b=(e[c+14>>1]|0)>>16-(a&255)&65535;kM(c,a);return b|0}function kI(c){c=c|0;var e=0,f=0,g=0;e=c;c=0;L87:while(1){if((c|0)>=286){f=86;break}if((kH(e,1)|0)<<16>>16!=0){a[e+24594+c|0]=((kH(e,4)|0)&65535)+1&255}else{a[e+24594+c|0]=0}g=c+1|0;c=g;do{if((g|0)==3){if((d[e+24594|0]|0|0)!=1){break}if((d[e+24595|0]|0|0)!=1){break}if((d[e+24596|0]|0|0)==1){break L87}}}while(0)}if((f|0)==86){kL(e,286,e+24594|0,12,e+66|0);return}f=(kH(e,9)|0)&65535;p7(e+24594|0,0,286);c=0;while(1){if((c|0)>=4096){break}b[e+66+(c<<1)>>1]=f&65535;c=c+1|0}return}function kJ(d,e){d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=d;d=9152+(e<<6)|0;e=d;d=e+4|0;g=c[e>>2]|0;e=1<<16-g;h=0;i=0;while(1){if((i|0)>=(c[f+36>>2]|0)){break}while(1){if((c[d>>2]|0)!=(i|0)){break}g=g+1|0;d=d+4|0;e=e>>>1}a[f+25104+i|0]=g&255;b[f+14614+(i<<1)>>1]=h&65535;h=h+e|0;i=i+1|0}return}function kK(c){c=c|0;var e=0,f=0,g=0;e=c;c=0;L120:while(1){if((c|0)>=17){f=116;break}a[e+25104+c|0]=(kH(e,4)|0)&255;g=c+1|0;c=g;do{if((g|0)==3){if((d[e+25104|0]|0|0)!=1){break}if((d[e+25105|0]|0|0)!=1){break}if((d[e+25106|0]|0|0)==1){break L120}}}while(0)}if((f|0)==116){return}f=(kH(e,10)|0)&65535;c=0;while(1){if((c|0)>=17){break}a[e+24594+c|0]=0;c=c+1|0}c=0;while(1){if((c|0)>=256){break}b[e+66+(c<<1)>>1]=f&65535;c=c+1|0}return}function kL(a,f,g,h,j){a=a|0;f=f|0;g=g|0;h=h|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;k=i;i=i+120|0;l=k|0;m=k+40|0;n=k+80|0;o=a;a=f;f=g;g=h;h=j;j=a<<16>>16;p7(l|0,0,34);p=1;while(1){if(p>>>0>16){break}b[m+(p<<1)>>1]=1<<16-p&65535;p=p+1|0}p=0;while(1){if(p>>>0>=a<<16>>16>>>0){break}q=l+((d[f+p|0]|0)<<1)|0;b[q>>1]=(b[q>>1]|0)+1&65535;p=p+1|0}q=0;p=1;while(1){if(p>>>0>16){break}b[n+(p<<1)>>1]=q;q=(q&65535)+(ac(e[m+(p<<1)>>1]|0,e[l+(p<<1)>>1]|0)|0)&65535;p=p+1|0}if((q&65535|0)!=0){c[o>>2]=1;i=k;return}q=16-(g<<16>>16)|0;p=1;while(1){if(p>>>0>g<<16>>16>>>0){break}l=n+(p<<1)|0;b[l>>1]=(e[l>>1]|0)>>q&65535;l=m+(p<<1)|0;b[l>>1]=(e[l>>1]|0)>>q&65535;p=p+1|0}l=(e[n+((g<<16>>16)+1<<1)>>1]|0)>>q;r=1<<(g<<16>>16);if((l|0)!=0){p=l;while(1){if(p>>>0>=r>>>0){break}b[h+(p<<1)>>1]=0;p=p+1|0}}l=0;while(1){if((l|0)>=(a<<16>>16|0)){break}r=d[f+l|0]|0;if((r|0)!=0){s=(e[n+(r<<1)>>1]|0)+(e[m+(r<<1)>>1]|0)|0;if((r|0)<=(g<<16>>16|0)){p=e[n+(r<<1)>>1]|0;while(1){if(p>>>0>=s>>>0){break}b[h+(p<<1)>>1]=l&65535;p=p+1|0}}else{t=e[n+(r<<1)>>1]|0;p=t;u=h+(t>>>(q>>>0)<<1)|0;p=p<<(g<<16>>16);t=r-(g<<16>>16)|0;while(1){v=t-1|0;t=v;if((v|0)<0){break}if((e[u>>1]|0|0)==0){b[o+8770+(j<<1)>>1]=0;b[o+10808+(j<<1)>>1]=0;v=j;j=v+1|0;b[u>>1]=v&65535}if((p&32768|0)!=0){u=o+10808+((e[u>>1]|0)<<1)|0}else{u=o+8770+((e[u>>1]|0)<<1)|0}p=p<<1}b[u>>1]=l&65535}b[n+(r<<1)>>1]=s&65535}l=l+1|0}i=k;return}function kM(f,g){f=f|0;g=g|0;var h=0;h=f;f=g;while(1){if((f&255|0)<=(d[h+17|0]|0|0)){break}f=(f&255)-(d[h+17|0]|0)&255;b[h+14>>1]=((e[h+14>>1]|0)<<(d[h+17|0]|0))+((d[h+16|0]|0)>>8-(d[h+17|0]|0))&65535;a[h+16|0]=(aN(c[h+4>>2]|0)|0)&255;a[h+17|0]=8}g=h+17|0;a[g]=(d[g]|0)-(f&255)&255;b[h+14>>1]=((e[h+14>>1]|0)<<(f&255))+((d[h+16|0]|0)>>8-(f&255))&65535;g=h+16|0;a[g]=(d[g]|0)<<(f&255)&255;return}function kN(c){c=c|0;var d=0;d=c;b[d+14>>1]=0;a[d+16|0]=0;a[d+17|0]=0;kM(d,16);return}function kO(a,d){a=a|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=d;d=(c[e+48>>2]|0)+1|0;f=d+1|0;g=b[e+14870+(c[e+48>>2]<<1)>>1]|0;b[e+14870+(d<<1)>>1]=g;b[e+18406+(~(g<<16>>16)<<1)>>1]=d&65535;b[e+14870+(f<<1)>>1]=~(a+314)&65535;b[e+14870+(c[e+48>>2]<<1)>>1]=f&65535;b[e+12846+(d<<1)>>1]=b[e+12846+(c[e+48>>2]<<1)>>1]|0;b[e+12846+(f<<1)>>1]=0;b[e+19290+(d<<1)>>1]=b[e+19290+(c[e+48>>2]<<1)>>1]|0;if((c[e+48>>2]|0)==628){b[e+14102>>1]=-1;g=e+22826+(b[e+20546>>1]<<1)|0;b[g>>1]=(b[g>>1]|0)+1&65535}g=c[e+48>>2]&65535;b[e+21058+(f<<1)>>1]=g;b[e+21058+(d<<1)>>1]=g;g=f;c[e+48>>2]=g;d=g&65535;b[e+18406+(a+314<<1)>>1]=d;g=e+52|0;h=c[g>>2]|0;c[g>>2]=h+1;g=b[e+16638+(h<<1)>>1]|0;b[e+19290+(f<<1)>>1]=g;b[e+22826+(g<<16>>16<<1)>>1]=d;kR(e,a);return}
function kP(a,d,f){a=a|0;d=d|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0;g=a;a=d;d=f;f=0;h=a;i=h;j=h;while(1){if((j|0)>=(d|0)){break}h=b[g+14870+(j<<1)>>1]|0;k=h;if((h|0)<0){b[g+12846+(i<<1)>>1]=(((e[g+12846+(j<<1)>>1]|0)+1|0)/2|0)&65535;b[g+14870+(i<<1)>>1]=k&65535;i=i+1|0}h=b[g+19290+(j<<1)>>1]|0;f=h;if((b[g+22826+(h<<1)>>1]|0)==(j|0)){h=g+52|0;l=(c[h>>2]|0)-1|0;c[h>>2]=l;b[g+16638+(l<<1)>>1]=f&65535}j=j+1|0}i=i-1|0;j=d-1|0;l=d-2|0;while(1){if((j|0)<(a|0)){break}while(1){if((j|0)<(l|0)){break}b[g+12846+(j<<1)>>1]=b[g+12846+(i<<1)>>1]|0;b[g+14870+(j<<1)>>1]=b[g+14870+(i<<1)>>1]|0;j=j-1|0;i=i-1|0}m=(e[g+12846+(l<<1)>>1]|0)+(e[g+12846+(l+1<<1)>>1]|0)|0;k=a;while(1){if(m>>>0>=(e[g+12846+(k<<1)>>1]|0)>>>0){break}k=k+1|0}while(1){if((i|0)<(k|0)){break}b[g+12846+(j<<1)>>1]=b[g+12846+(i<<1)>>1]|0;b[g+14870+(j<<1)>>1]=b[g+14870+(i<<1)>>1]|0;j=j-1|0;i=i-1|0}b[g+12846+(j<<1)>>1]=m&65535;b[g+14870+(j<<1)>>1]=l+1&65535;j=j-1|0;l=l-2|0}m=0;j=a;while(1){if((j|0)>=(d|0)){break}a=b[g+14870+(j<<1)>>1]|0;i=a;if((a|0)<0){b[g+18406+(~i<<1)>>1]=j&65535}else{a=j&65535;b[g+21058+(i-1<<1)>>1]=a;b[g+21058+(i<<1)>>1]=a}a=e[g+12846+(j<<1)>>1]|0;if((a|0)==(m|0)){b[g+19290+(j<<1)>>1]=f&65535}else{l=g+52|0;k=c[l>>2]|0;c[l>>2]=k+1;l=b[g+16638+(k<<1)>>1]|0;b[g+19290+(j<<1)>>1]=l;k=l<<16>>16;f=k;b[g+22826+(k<<1)>>1]=j&65535;m=a}j=j+1|0}return}function kQ(a,d){a=a|0;d=d|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;f=a;a=d;d=b[f+19290+(a<<1)>>1]|0;g=b[f+22826+(d<<1)>>1]|0;h=g;if((g|0)==(a|0)){if((d|0)==(b[f+19290+(a+1<<1)>>1]|0)){g=f+22826+(d<<1)|0;b[g>>1]=(b[g>>1]|0)+1&65535;g=f+12846+(a<<1)|0;i=(b[g>>1]|0)+1&65535;b[g>>1]=i;if((i&65535|0)==(e[f+12846+(a-1<<1)>>1]|0)){b[f+19290+(a<<1)>>1]=b[f+19290+(a-1<<1)>>1]|0}else{i=f+52|0;g=c[i>>2]|0;c[i>>2]=g+1;i=b[f+16638+(g<<1)>>1]|0;b[f+19290+(a<<1)>>1]=i;b[f+22826+(i<<16>>16<<1)>>1]=a&65535}}else{i=f+12846+(a<<1)|0;g=(b[i>>1]|0)+1&65535;b[i>>1]=g;if((g&65535|0)==(e[f+12846+(a-1<<1)>>1]|0)){g=f+52|0;i=(c[g>>2]|0)-1|0;c[g>>2]=i;b[f+16638+(i<<1)>>1]=d&65535;b[f+19290+(a<<1)>>1]=b[f+19290+(a-1<<1)>>1]|0}}j=a;k=f;l=k+32|0;m=l;n=m+21026|0;o=n+(j<<1)|0;p=b[o>>1]|0;q=p<<16>>16;return q|0}i=b[f+14870+(a<<1)>>1]|0;g=b[f+14870+(h<<1)>>1]|0;b[f+14870+(a<<1)>>1]=g&65535;b[f+14870+(h<<1)>>1]=i&65535;if((i|0)>=0){r=h&65535;b[f+21058+(i-1<<1)>>1]=r;b[f+21058+(i<<1)>>1]=r}else{b[f+18406+(~i<<1)>>1]=h&65535}if((g|0)>=0){i=a&65535;b[f+21058+(g-1<<1)>>1]=i;b[f+21058+(g<<1)>>1]=i}else{b[f+18406+(~g<<1)>>1]=a&65535}a=h;h=f+22826+(d<<1)|0;b[h>>1]=(b[h>>1]|0)+1&65535;h=f+12846+(a<<1)|0;d=(b[h>>1]|0)+1&65535;b[h>>1]=d;if((d&65535|0)==(e[f+12846+(a-1<<1)>>1]|0)){b[f+19290+(a<<1)>>1]=b[f+19290+(a-1<<1)>>1]|0}else{d=f+52|0;h=c[d>>2]|0;c[d>>2]=h+1;d=b[f+16638+(h<<1)>>1]|0;b[f+19290+(a<<1)>>1]=d;b[f+22826+(d<<16>>16<<1)>>1]=a&65535}j=a;k=f;l=k+32|0;m=l;n=m+21026|0;o=n+(j<<1)|0;p=b[o>>1]|0;q=p<<16>>16;return q|0}function kR(a,d){a=a|0;d=d|0;var f=0;f=a;if((e[f+62>>1]|0)==32768){kP(f,628,(c[f+48>>2]|0)+1|0);b[f+62>>1]=b[f+14102>>1]|0;b[f+14102>>1]=-1}a=b[f+18406+(d+314<<1)>>1]|0;while(1){if((a|0)==628){break}a=kQ(f,a)|0}a=f+62|0;b[a>>1]=(b[a>>1]|0)+1&65535;return}function kS(a){a=a|0;var d=0,f=0,g=0,h=0;d=a;if((c[d+56>>2]|0)>>>0>=((e[d+60>>1]|0)+256-3+1|0)>>>0){f=512}else{f=(c[d+56>>2]|0)-1|0}c[d+44>>2]=f;f=0;while(1){if((f|0)>=628){break}b[d+16638+(f<<1)>>1]=f&65535;b[d+19290+(f<<1)>>1]=0;f=f+1|0}f=0;a=(c[d+56>>2]<<1)-2|0;while(1){if((f|0)>=(c[d+56>>2]|0)){break}b[d+12846+(a<<1)>>1]=1;b[d+14870+(a<<1)>>1]=~f&65535;b[d+18406+(f<<1)>>1]=a&65535;b[d+19290+(a<<1)>>1]=1;f=f+1|0;a=a-1|0}c[d+52>>2]=2;b[d+22828>>1]=(c[d+56>>2]|0)-1&65535;f=(c[d+56>>2]<<1)-2|0;while(1){if((a|0)<0){break}g=(e[d+12846+(f<<1)>>1]|0)+(e[d+12846+(f-1<<1)>>1]|0)&65535;b[d+12846+(a<<1)>>1]=g;b[d+14870+(a<<1)>>1]=f&65535;h=a&65535;b[d+21058+(f-1<<1)>>1]=h;b[d+21058+(f<<1)>>1]=h;if((g&65535|0)==(e[d+12846+(a+1<<1)>>1]|0|0)){g=b[d+19290+(a+1<<1)>>1]|0;b[d+19290+(a<<1)>>1]=g;b[d+22826+(g<<16>>16<<1)>>1]=a&65535}else{g=d+52|0;h=c[g>>2]|0;c[g>>2]=h+1;g=b[d+16638+(h<<1)>>1]|0;b[d+19290+(a<<1)>>1]=g;b[d+22826+(g<<16>>16<<1)>>1]=a&65535}f=f-2|0;a=a-1|0}return}function kT(a){a=a|0;var d=0,f=0;d=a;b[d+14102>>1]=1;b[d+16126>>1]=-315;b[d+19034>>1]=628;a=d+52|0;f=c[a>>2]|0;c[a>>2]=f+1;a=b[d+16638+(f<<1)>>1]|0;b[d+20546>>1]=a;b[d+22826+(a<<16>>16<<1)>>1]=628;c[d+48>>2]=628;b[d+62>>1]=0;c[d+40>>2]=1<<(e[d+12>>1]|0);c[d+28>>2]=64;return}function kU(a,d){a=a|0;d=d|0;var f=0;f=a;if((e[f+12846>>1]|0)==32768){kP(f,0,(c[f+56>>2]<<1)-1|0)}a=f+12846|0;b[a>>1]=(b[a>>1]|0)+1&65535;a=b[f+18406+(d<<1)>>1]|0;do{a=kQ(f,a)|0;}while((a|0)!=0);return}function kV(c,d,f,g){c=c|0;d=d|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0;h=c;c=d;d=f;f=g;g=kH(h,d&255)|0;i=g;if(g<<16>>16==0){j=kH(h,d&255)|0;k=0;while(1){if((k<<16>>16|0)>=(c<<16>>16|0)){break}a[h+25104+(k<<16>>16)|0]=0;k=k+1&65535}k=0;while(1){if((k<<16>>16|0)>=256){break}b[h+8258+(k<<16>>16<<1)>>1]=j;k=k+1&65535}return}k=0;while(1){if((k<<16>>16|0)>=(i<<16>>16|0)){break}j=(e[h+14>>1]|0)>>13&65535;if((j<<16>>16|0)==7){d=4096;while(1){if((d&65535&(e[h+14>>1]|0)|0)==0){break}d=(d&65535)>>1&65535;j=j+1&65535}}if((j<<16>>16|0)<7){l=3}else{l=(j<<16>>16)-3|0}kM(h,l&255);d=k;k=d+1&65535;a[h+25104+(d<<16>>16)|0]=j&255;if((k<<16>>16|0)==(f<<16>>16|0)){j=kH(h,2)|0;while(1){d=j-1&65535;j=d;if((d<<16>>16|0)<0){break}d=k;k=d+1&65535;a[h+25104+(d<<16>>16)|0]=0}}}while(1){if((k<<16>>16|0)>=(c<<16>>16|0)){break}j=k;k=j+1&65535;a[h+25104+(j<<16>>16)|0]=0}kL(h,c,h+25104|0,8,h+8258|0);return}function kW(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0;f=a;a=b;b=e;if((a|0)==0){return}e=~c[b+94560>>2];do{g=f;f=g+1|0;e=c[19696+((((d[g]|0)^e)&255)<<2)>>2]^e>>>8;g=a-1|0;a=g;}while((g|0)!=0);c[b+94560>>2]=~e;return}function kX(c){c=c|0;var d=0,f=0,g=0,h=0;d=c;c=kH(d,9)|0;f=c;if(c<<16>>16==0){g=kH(d,9)|0;h=0;while(1){if((h<<16>>16|0)>=510){break}a[d+24594+(h<<16>>16)|0]=0;h=h+1&65535}h=0;while(1){if((h<<16>>16|0)>=4096){break}b[d+66+(h<<16>>16<<1)>>1]=g;h=h+1&65535}return}h=0;while(1){if((h<<16>>16|0)>=(f<<16>>16|0)){break}g=b[d+8258+((e[d+14>>1]|0)>>8<<1)>>1]|0;if((g<<16>>16|0)>=19){c=128;do{if(((e[d+14>>1]|0)&(c&65535)|0)!=0){g=b[d+10808+(g<<16>>16<<1)>>1]|0}else{g=b[d+8770+(g<<16>>16<<1)>>1]|0}c=(c&65535)>>1&65535;}while((g<<16>>16|0)>=19)}kM(d,a[d+25104+(g<<16>>16)|0]|0);if((g<<16>>16|0)<=2){if(g<<16>>16!=0){if((g<<16>>16|0)==1){g=((kH(d,4)|0)&65535)+3&65535}else{g=((kH(d,9)|0)&65535)+20&65535}}else{g=1}while(1){c=g-1&65535;g=c;if((c<<16>>16|0)<0){break}c=h;h=c+1&65535;a[d+24594+(c<<16>>16)|0]=0}}else{c=h;h=c+1&65535;a[d+24594+(c<<16>>16)|0]=(g<<16>>16)-2&255}}while(1){if((h<<16>>16|0)>=510){break}g=h;h=g+1&65535;a[d+24594+(g<<16>>16)|0]=0}kL(d,510,d+24594|0,12,d+66|0);return}function kY(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0;d=a;a=b;if((a|0)==0){e=-1;f=e;return f|0}b=pS(94568)|0;if((b|0)==0){e=-1;f=e;return f|0}else{g=d;bK(g|0,10,1)|0;c[b+94564>>2]=a;a=d;d=b;kZ(a,d)|0;pU(b);e=0;f=e;return f|0}return 0}function kZ(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;f=i;g=b;b=e;e=1;c[b+600>>2]=0;h=b+600|0;do{j=1;k=bB(b+14|0,1,31,g|0)|0;if((aY(g|0)|0)!=0){aF(22864)}else{if((k|0)!=0){if((k|0)==31){c[b+94560>>2]=0;c[b+568>>2]=((d[b+43|0]|0)<<24)+((d[b+42|0]|0)<<16)+((d[b+41|0]|0)<<8)+(d[b+40|0]|0);a[b+43|0]=0;a[b+42|0]=0;a[b+41|0]=0;a[b+40|0]=0;kW(b+14|0,31,b);l=d[b+44|0]|0;k=bB(b+45|0,1,l|0,g|0)|0;if((aY(g|0)|0)!=0){aF(23840)}else{if((k|0)==(l|0)){a[b+45+l|0]=0;kW(b+45|0,l,b);l=d[b+28|0]|0;k=bB(b+301|0,1,l|0,g|0)|0;if((aY(g|0)|0)!=0){aF(25176)}else{if((k|0)==(l|0)){a[b+301+l|0]=0;kW(b+301|0,l,b);if((c[b+94560>>2]|0)==(c[b+568>>2]|0)){c[b+564>>2]=((d[b+19|0]|0)<<24)+((d[b+18|0]|0)<<16)+((d[b+17|0]|0)<<8)+(d[b+16|0]|0);c[b+560>>2]=((d[b+23|0]|0)<<24)+((d[b+22|0]|0)<<16)+((d[b+21|0]|0)<<8)+(d[b+20|0]|0);a[b+597|0]=a[b+25|0]|0;c[b+568>>2]=((d[b+39|0]|0)<<24)+((d[b+38|0]|0)<<16)+((d[b+37|0]|0)<<8)+(d[b+36|0]|0);k=pS(268)|0;m=k;if((k|0)!=0){c[h>>2]=m;h=m|0;c[m>>2]=0;c[m+4>>2]=c[b+564>>2];c[m+8>>2]=c[b+568>>2];l=0;while(1){k=a[b+45+l|0]|0;a[m+12+l|0]=k;if(k<<24>>24==0){break}l=l+1|0}if((c[b+560>>2]|0)!=0){l=d[b+597|0]|0;if((l|0)==0){j=k$(g,b)|0;j=1}else if((l|0)==2){j=k0(g,b)|0;j=1}else{j=k1(g,b)|0}if((j|0)!=0){n=370;break}o=c[b+600>>2]|0;while(1){l=o;m=l;if((l|0)==0){break}o=c[m>>2]|0;pU(m)}c[b+600>>2]=0;h=b+600|0;if((bK(g|0,c[b+560>>2]|0,1)|0)!=0){n=375;break}}else{j=0}}else{bE(c[q>>2]|0,28888,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}}else{l=c[q>>2]|0;bE(l|0,27416,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}}else{l=c[q>>2]|0;bE(l|0,26240,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}}}else{l=c[q>>2]|0;bE(l|0,24392,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}}}else{l=c[q>>2]|0;bE(l|0,23328,(p=i,i=i+1|0,i=i+7>>3<<3,c[p>>2]=0,p)|0)|0;i=p}}else{e=0}}}while((j|0)!=0^1);if((n|0)==375){aF(23208)}o=c[b+600>>2]|0;while(1){b=o;m=b;if((b|0)==0){break}o=c[m>>2]|0;pU(m)}i=f;return e|0}function k_(a,b){a=a|0;b=b|0;return c[b+94564>>2]|0}function k$(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0;d=i;e=a;a=b;b=0;f=c[a+600>>2]|0;while(1){if((b|0)!=0){g=0}else{g=(f|0)!=0}if(!g){break}if((cK(f+12|0)|0)!=0){h=0}else{h=k_(f+12|0,a)|0}c[a+94560>>2]=0;c[a+564>>2]=c[f+4>>2];if((c[a+564>>2]|0)>>>0>(c[a+560>>2]|0)>>>0){c[a+564>>2]=c[a+560>>2]}while(1){if((c[a+564>>2]|0)>>>0<=0){break}if((c[a+564>>2]|0)>>>0>16384){j=16384}else{j=c[a+564>>2]|0}k=j;if((bB(a+604|0,1,k|0,e|0)|0)!=(k|0)){l=418;break}m=a+560|0;c[m>>2]=(c[m>>2]|0)-k;kW(a+604|0,k,a);if((h|0)!=0){b=1;(ay(a+604|0,1,k|0,h|0)|0)!=(k|0)}m=a+564|0;c[m>>2]=(c[m>>2]|0)-k}if((l|0)==418){l=0;bf(31064,(k=i,i=i+1|0,i=i+7>>3<<3,c[k>>2]=0,k)|0)|0;i=k;if((aY(e|0)|0)!=0){aF(30784)}else{m=c[q>>2]|0;bE(m|0,30472,(k=i,i=i+1|0,i=i+7>>3<<3,c[k>>2]=0,k)|0)|0;i=k}b=1}f=c[f>>2]|0}i=d;return b|0}function k0(b,d){b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0;e=i;f=b;b=d;d=0;g=0;c[b+83068>>2]=0;c[b+83072>>2]=-16;c[b+83064>>2]=1;c[b+564>>2]=0;c[b+83060>>2]=0;h=0;while(1){if(h>>>0>=8){break}a[b+83076+h|0]=0;h=h+1|0}h=0;while(1){if(h>>>0>=768){break}a[b+83552+h|0]=0;h=h+1|0}j=b+16988|0;c[b+83040>>2]=j;c[b+83048>>2]=j-1024;j=b+82782|0;c[b+83044>>2]=j;c[b+83052>>2]=j;k=j;j=c[b+600>>2]|0;while(1){if((g|0)!=0){l=0}else{l=(j|0)!=0}if(!l){break}if((cK(j+12|0)|0)!=0){d=0}else{d=k_(j+12|0,b)|0}c[b+94560>>2]=0;c[b+564>>2]=c[j+4>>2];while(1){if((c[b+564>>2]|0)>>>0<=0){break}if((k|0)==(c[b+83044>>2]|0)){if((c[b+83040>>2]|0)>>>0>=(c[b+83048>>2]|0)>>>0){m=b+604|0;n=m-(c[b+83040>>2]|0)+16384|0;h=n;if((n|0)!=0){do{n=b+83040|0;o=c[n>>2]|0;c[n>>2]=o+1;n=m;m=n+1|0;a[n]=a[o]|0;o=h-1|0;h=o;}while((o|0)!=0)}c[b+83040>>2]=b+604;h=(c[b+83040>>2]|0)-m+16384|0;if((c[b+560>>2]|0)>>>0<h>>>0){h=c[b+560>>2]|0}if((bB(m|0,1,h|0,f|0)|0)!=(h|0)){p=457;break}o=b+560|0;c[o>>2]=(c[o>>2]|0)-h;m=m+h|0;if((c[b+83040>>2]|0)>>>0>=m>>>0){p=462;break}}if((c[b+83060>>2]|0)>>>0<=0){if((k2(b)|0)!=0){p=466;break}}if((c[b+83044>>2]|0)>>>0>=(b+82782|0)>>>0){o=(c[b+83044>>2]|0)-(b+16988)-65536|0;h=o;if((o|0)!=0){o=b+16988|0;c[b+83044>>2]=o;m=o+65536|0;do{o=m;m=o+1|0;n=a[o]|0;o=b+83044|0;r=c[o>>2]|0;c[o>>2]=r+1;a[r]=n;n=h-1|0;h=n;}while((n|0)!=0)}k=c[b+83044>>2]|0}c[b+83052>>2]=(c[b+83044>>2]|0)+(c[b+83060>>2]|0);if((c[b+83052>>2]|0)>>>0>(b+82782|0)>>>0){c[b+83052>>2]=b+82782}m=c[b+83044>>2]|0;k3(b);n=b+83060|0;c[n>>2]=(c[n>>2]|0)-((c[b+83044>>2]|0)-m)}h=(c[b+83044>>2]|0)-k|0;if(h>>>0>(c[b+564>>2]|0)>>>0){h=c[b+564>>2]|0}kW(k,h,b);if((d|0)!=0){g=1;(ay(k|0,1,h|0,d|0)|0)!=(h|0)}n=b+564|0;c[n>>2]=(c[n>>2]|0)-h;k=k+h|0}if((p|0)==457){p=0;bf(31064,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0)|0;i=n;if((aY(f|0)|0)!=0){aF(30784)}else{r=c[q>>2]|0;bE(r|0,30472,(n=i,i=i+1|0,i=i+7>>3<<3,c[n>>2]=0,n)|0)|0;i=n}g=1}else if((p|0)==466){p=0}else if((p|0)==462){p=0}j=c[j>>2]|0}i=e;return g|0}function k1(a,b){a=a|0;b=b|0;var d=0,e=0;a=i;d=0;e=c[b+600>>2]|0;while(1){if((e|0)==0){break}bE(c[q>>2]|0,22144,(b=i,i=i+8|0,c[b>>2]=e+12,b)|0)|0;i=b;e=c[e>>2]|0}i=a;return d|0}function k2(b){b=b|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;f=b;b=0;g=c[f+83068>>2]|0;h=c[f+83072>>2]|0;if((h|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}c[f+83056>>2]=g&7;g=g>>>3;i=h-3|0;h=i;if((i|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}do{if((b|0)==0){if((c[f+83056>>2]|0)!=3){break}k=0;while(1){if(k>>>0>=8){break}a[f+83076+k|0]=g&7;g=g>>>3;i=h-3|0;h=i;if((i|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}k=k+1|0}b=k4(8,7,f+83076|0,f+83084|0)|0}}while(0);if((b|0)==0){c[f+83060>>2]=(g&255)<<16;g=g>>>8;i=h-8|0;h=i;if((i|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}i=f+83060|0;c[i>>2]=(c[i>>2]|0)+((g&255)<<8);g=g>>>8;i=h-8|0;h=i;if((i|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}i=f+83060|0;c[i>>2]=(c[i>>2]|0)+(g&255);g=g>>>8;i=h-8|0;h=i;if((i|0)<0){h=h+16|0;i=f+83040|0;j=c[i>>2]|0;c[i>>2]=j+1;g=g+((d[j]|0)<<h+8)|0;j=f+83040|0;i=c[j>>2]|0;c[j>>2]=i+1;g=g+((d[i]|0)<<h)|0}}if((b|0)!=0){l=g;m=f;n=m+83068|0;c[n>>2]=l;o=h;p=f;q=p+83072|0;c[q>>2]=o;r=b;return r|0}if((c[f+83056>>2]|0)==1){l=g;m=f;n=m+83068|0;c[n>>2]=l;o=h;p=f;q=p+83072|0;c[q>>2]=o;r=b;return r|0}i=0;j=1;s=256;do{k=0;while(1){if(k>>>0>=20){break}a[f+83340+k|0]=g&15;g=g>>>4;t=h-4|0;h=t;if((t|0)<0){h=h+16|0;t=f+83040|0;u=c[t>>2]|0;c[t>>2]=u+1;g=g+((d[u]|0)<<h+8)|0;u=f+83040|0;t=c[u>>2]|0;c[u>>2]=t+1;g=g+((d[t]|0)<<h)|0}k=k+1|0}b=k4(20,6,f+83340|0,f+83360|0)|0;if((b|0)!=0){t=524;break}do{t=e[f+83360+((g&63)<<1)>>1]|0;u=t;if(t>>>0>=20){do{u=e[f+83360+((g>>>6&1)+(u<<1)<<1)>>1]|0;t=h;h=t-1|0;if((t|0)==0){h=h+16|0;t=f+83040|0;v=c[t>>2]|0;c[t>>2]=v+1;g=g+((d[v]|0)<<24)|0;v=f+83040|0;t=c[v>>2]|0;c[v>>2]=t+1;g=g+((d[t]|0)<<16)|0}g=g>>>1;}while(u>>>0>=20);k=6}else{k=d[f+83340+u|0]|0}g=g>>>(k>>>0);t=h-k|0;h=t;if((t|0)<0){h=h+16|0;t=f+83040|0;v=c[t>>2]|0;c[t>>2]=v+1;g=g+((d[v]|0)<<h+8)|0;v=f+83040|0;t=c[v>>2]|0;c[v>>2]=t+1;g=g+((d[t]|0)<<h)|0}t=u;if((t|0)==17|(t|0)==18){if((u|0)==17){k=4;w=3}else{k=6-j|0;w=19}w=w+((g&c[2736+(k<<2)>>2])+j)|0;g=g>>>(k>>>0);v=h-k|0;h=v;if((v|0)<0){h=h+16|0;v=f+83040|0;x=c[v>>2]|0;c[v>>2]=x+1;g=g+((d[x]|0)<<h+8)|0;x=f+83040|0;v=c[x>>2]|0;c[x>>2]=v+1;g=g+((d[v]|0)<<h)|0}while(1){if(i>>>0<s>>>0){v=w;w=v-1|0;y=(v|0)!=0}else{y=0}if(!y){break}v=i;i=v+1|0;a[f+83552+v|0]=0}}else if((t|0)==19){w=(g&1)+3+j|0;v=h;h=v-1|0;if((v|0)==0){h=h+16|0;v=f+83040|0;x=c[v>>2]|0;c[v>>2]=x+1;g=g+((d[x]|0)<<24)|0;x=f+83040|0;v=c[x>>2]|0;c[x>>2]=v+1;g=g+((d[v]|0)<<16)|0}g=g>>>1;v=e[f+83360+((g&63)<<1)>>1]|0;u=v;if(v>>>0>=20){do{u=e[f+83360+((g>>>6&1)+(u<<1)<<1)>>1]|0;v=h;h=v-1|0;if((v|0)==0){h=h+16|0;v=f+83040|0;x=c[v>>2]|0;c[v>>2]=x+1;g=g+((d[x]|0)<<24)|0;x=f+83040|0;v=c[x>>2]|0;c[x>>2]=v+1;g=g+((d[v]|0)<<16)|0}g=g>>>1;}while(u>>>0>=20);k=6}else{k=d[f+83340+u|0]|0}g=g>>>(k>>>0);t=h-k|0;h=t;if((t|0)<0){h=h+16|0;t=f+83040|0;v=c[t>>2]|0;c[t>>2]=v+1;g=g+((d[v]|0)<<h+8)|0;v=f+83040|0;t=c[v>>2]|0;c[v>>2]=t+1;g=g+((d[t]|0)<<h)|0}u=d[2832+((d[f+83552+i|0]|0)+17-u)|0]|0;while(1){if(i>>>0<s>>>0){t=w;w=t-1|0;z=(t|0)!=0}else{z=0}if(!z){break}t=i;i=t+1|0;a[f+83552+t|0]=u&255}}else{u=d[2832+((d[f+83552+i|0]|0)+17-u)|0]|0;t=i;i=t+1|0;a[f+83552+t|0]=u&255}}while(i>>>0<s>>>0);j=j-1|0;s=s+512|0;}while((s|0)==768);if((b|0)==0){b=k4(768,12,f+83552|0,f+84320|0)|0}l=g;m=f;n=m+83068|0;c[n>>2]=l;o=h;p=f;q=p+83072|0;c[q>>2]=o;r=b;return r|0}function k3(b){b=b|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;f=b;b=c[f+83068>>2]|0;g=c[f+83072>>2]|0;do{h=e[f+84320+((b&4095)<<1)>>1]|0;i=h;if(h>>>0>=768){b=b>>>12;h=g-12|0;g=h;if((h|0)<0){g=g+16|0;h=f+83040|0;j=c[h>>2]|0;c[h>>2]=j+1;b=b+((d[j]|0)<<g+8)|0;j=f+83040|0;h=c[j>>2]|0;c[j>>2]=h+1;b=b+((d[h]|0)<<g)|0}do{i=e[f+84320+((b&1)+(i<<1)<<1)>>1]|0;h=g;g=h-1|0;if((h|0)==0){g=g+16|0;h=f+83040|0;j=c[h>>2]|0;c[h>>2]=j+1;b=b+((d[j]|0)<<24)|0;j=f+83040|0;h=c[j>>2]|0;c[j>>2]=h+1;b=b+((d[h]|0)<<16)|0}b=b>>>1;}while(i>>>0>=768)}else{k=d[f+83552+i|0]|0;b=b>>>(k>>>0);h=g-k|0;g=h;if((h|0)<0){g=g+16|0;h=f+83040|0;j=c[h>>2]|0;c[h>>2]=j+1;b=b+((d[j]|0)<<g+8)|0;j=f+83040|0;h=c[j>>2]|0;c[j>>2]=h+1;b=b+((d[h]|0)<<g)|0}}if(i>>>0<256){h=f+83044|0;j=c[h>>2]|0;c[h>>2]=j+1;a[j]=i&255}else{i=i-256|0;j=i&31;k=j;h=c[2608+(j<<2)>>2]|0;k=d[2800+k|0]|0;do{if(k>>>0>=3){if((c[f+83056>>2]|0)!=3){l=598;break}k=k-3|0;h=h+((b&c[2736+(k<<2)>>2])<<3)|0;b=b>>>(k>>>0);j=g-k|0;g=j;if((j|0)<0){g=g+16|0;j=f+83040|0;m=c[j>>2]|0;c[j>>2]=m+1;b=b+((d[m]|0)<<g+8)|0;m=f+83040|0;j=c[m>>2]|0;c[m>>2]=j+1;b=b+((d[j]|0)<<g)|0}j=e[f+83084+((b&127)<<1)>>1]|0;k=j;h=h+j|0;k=d[f+83076+k|0]|0}else{l=598}}while(0);if((l|0)==598){l=0;h=h+(b&c[2736+(k<<2)>>2])|0;if((h|0)==0){h=c[f+83064>>2]|0}}b=b>>>(k>>>0);j=g-k|0;g=j;if((j|0)<0){g=g+16|0;j=f+83040|0;m=c[j>>2]|0;c[j>>2]=m+1;b=b+((d[m]|0)<<g+8)|0;m=f+83040|0;j=c[m>>2]|0;c[m>>2]=j+1;b=b+((d[j]|0)<<g)|0}c[f+83064>>2]=h;j=i>>>5&15;k=j;h=(c[2608+(j<<2)>>2]|0)+3|0;k=d[2800+k|0]|0;h=h+(b&c[2736+(k<<2)>>2])|0;b=b>>>(k>>>0);j=g-k|0;g=j;if((j|0)<0){g=g+16|0;j=f+83040|0;m=c[j>>2]|0;c[j>>2]=m+1;b=b+((d[m]|0)<<g+8)|0;m=f+83040|0;j=c[m>>2]|0;c[m>>2]=j+1;b=b+((d[j]|0)<<g)|0}if((f+16988+(c[f+83064>>2]|0)|0)>>>0<(c[f+83044>>2]|0)>>>0){n=(c[f+83044>>2]|0)+(-(c[f+83064>>2]|0)|0)|0}else{n=(c[f+83044>>2]|0)+65536+(-(c[f+83064>>2]|0)|0)|0}j=n;do{m=j;j=m+1|0;o=a[m]|0;m=f+83044|0;p=c[m>>2]|0;c[m>>2]=p+1;a[p]=o;o=h-1|0;h=o;}while((o|0)!=0)}if((c[f+83044>>2]|0)>>>0<(c[f+83052>>2]|0)>>>0){q=(c[f+83040>>2]|0)>>>0<(c[f+83048>>2]|0)>>>0}else{q=0}}while(q);c[f+83068>>2]=b;c[f+83072>>2]=g;return}function k4(a,c,f,g){a=a|0;c=c|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;h=a;a=c;c=f;f=g;g=0;i=0;j=0;k=1<<a;l=k;m=k;m=m>>>1;g=g+1&255;while(1){if((i|0)!=0){n=0}else{n=(g&255|0)<=(a|0)}if(!n){break}o=0;while(1){if((o|0)>=(h|0)){break}if((d[c+o|0]|0|0)==(g&255|0)){p=j;q=0;r=a;do{q=(q<<1)+(p&1)|0;p=p>>>1;k=r-1|0;r=k;}while((k|0)!=0);k=j+m|0;j=k;if(k>>>0>l>>>0){s=628;break}r=m;t=1<<(g&255);do{b[f+(q<<1)>>1]=o&65535;q=q+t|0;k=r-1|0;r=k;}while((k|0)!=0)}o=o+1|0}if((s|0)==628){s=0;i=1}m=m>>>1;g=g+1&255}do{if((i|0)==0){if((j|0)==(l|0)){break}o=j;while(1){if(o>>>0>=l>>>0){break}p=o;q=0;r=a;do{q=(q<<1)+(p&1)|0;p=p>>>1;n=r-1|0;r=n;}while((n|0)!=0);b[f+(q<<1)>>1]=0;o=o+1|0}t=l>>>1;j=j<<16;l=l<<16;m=32768;while(1){if((i|0)!=0){u=0}else{u=(g&255|0)<=16}if(!u){break}o=0;while(1){if((o|0)>=(h|0)){break}if((d[c+o|0]|0|0)==(g&255|0)){p=j>>>16;q=0;r=a;do{q=(q<<1)+(p&1)|0;p=p>>>1;n=r-1|0;r=n;}while((n|0)!=0);r=0;while(1){if(r>>>0>=((g&255)-a|0)>>>0){break}if((e[f+(q<<1)>>1]|0|0)==0){b[f+(t<<1<<1)>>1]=0;b[f+((t<<1)+1<<1)>>1]=0;n=t;t=n+1|0;b[f+(q<<1)>>1]=n&65535}q=(e[f+(q<<1)>>1]|0)<<1;q=q+(j>>>((15-r|0)>>>0)&1)|0;r=r+1|0}b[f+(q<<1)>>1]=o&65535;n=j+m|0;j=n;if(n>>>0>l>>>0){s=662;break}}o=o+1|0}if((s|0)==662){s=0;i=1}m=m>>>1;g=g+1&255}}}while(0);if((j|0)==(l|0)){v=i;return v|0}i=1;v=i;return v|0}function k5(a,b){a=a|0;b=b|0;var c=0,d=0;c=b;do{if((c|0)==0){d=-1}else{if((k6(a,c)|0)<0){d=-1;break}else{d=0;break}}}while(0);return d|0}function k6(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0;c=a;a=b;L883:do{if(88!=(aN(c|0)|0)){d=683}else{if(80!=(aN(c|0)|0)){d=683;break}if(75!=(aN(c|0)|0)){d=683;break}if(70!=(aN(c|0)|0)){d=683;break}b=aN(c|0)|0;b=b<<8;b=b|(aN(c|0)|0);b=b<<8;b=b|(aN(c|0)|0);b=b<<8;b=b|(aN(c|0)|0);do{if(83==(aN(c|0)|0)){if(81!=(aN(c|0)|0)){break}if(83!=(aN(c|0)|0)){break}if(72!=(aN(c|0)|0)){break}e=aN(c|0)|0;e=e<<8;e=e|(aN(c|0)|0);e=e<<8;e=e|(aN(c|0)|0);e=e<<8;e=e|(aN(c|0)|0);f=pS(b+3|0)|0;g=pS(e+100|0)|0;do{if((f|0)!=0){if((g|0)==0){break}do{if(1==(bB(f|0,b-8|0,1,c|0)|0)){if((k7(f,g,e)|0)!=(e|0)){break}if(1!=(ay(g|0,e|0,1,a|0)|0)){break}pU(f);pU(g);h=0;i=h;return i|0}}while(0);pU(f);pU(g);break L883}}while(0);break L883}}while(0)}}while(0);h=-1;i=h;return i|0}function k7(b,c,f){b=b|0;c=c|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0;g=i;i=i+56|0;h=g|0;j=b;b=c;c=f;f=0;k=h;p3(k|0,1320,56)|0;k=0;l=j+20|0;while(1){if((c|0)==0){m=751;break}n=l;l=n+1|0;o=d[n]|0;l=l+1|0;n=e[l>>1]|0;l=l+2|0;p=l;l=p+1|0;q=d[p]|0;q=q<<8;p=l;l=p+1|0;q=q|(d[p]|0);p=l;l=p+1|0;r=d[p]|0;r=r<<8;p=l;l=p+1|0;r=r|(d[p]|0);j=l+2|0;p=a[l+(q+2)|0]|0;s=a[l+(q+1)|0]|0;t=a[l+q|0]|0;a[l+(q+2)|0]=0;a[l+(q+1)|0]=0;a[l+q|0]=0;u=(k8(l,q+3>>2)|0)&65535;a[l+(q+2)|0]=p;a[l+(q+1)|0]=s;a[l+q|0]=t;if((u|0)!=(n|0)){m=707;break}if((o|0)==0){n=b;u=l;t=q;p3(n|0,u|0,t)|0;b=b+q|0;l=l+q|0;c=c-q|0;f=f+q|0;continue}if((o|0)!=1){m=711;break}c=c-r|0;f=f+r|0;q=q+3&65532;l=l+q|0;q=b+r|0;r=0;o=0;t=0;u=0;n=0;s=j;j=s+1|0;o=d[s]|0;s=b;b=s+1|0;a[s]=o&255;do{L924:do{if((u|0)>=8){do{if((k9(j,n,1)|0)!=0){n=n+1|0;v=8;if((v|0)==(r|0)){m=733;break}else{v=r;m=740;break}}else{n=n+1|0;if((k9(j,n,1)|0)==0){m=752;break L924}n=n+1|0;do{if((k9(j,n,1)|0)!=0){n=n+1|0;if((k9(j,n,1)|0)!=0){v=k9(j,n,3)|0;n=n+3|0;break}v=3;m=729}else{v=2;m=729}}while(0);if((m|0)==729){m=0;n=n+1|0}v=d[h+(r<<3)+v-17|0]|0;if((v|0)!=8){m=740;break}else{m=733;break}}}while(0);do{if((m|0)==733){m=0;if((t|0)>=20){w=1;break}w=0;m=742;break L924}else if((m|0)==740){m=0;w=4}}while(0);t=t+8|0;m=742}else{if((k9(j,n,1)|0)!=0){m=752;break}else{n=n+1|0;w=0;v=8;m=742;break}}}while(0);if((m|0)==752){m=0;n=n+1|0;do{if((k9(j,n,1)|0)!=0){n=n+1|0;if((k9(j,n,1)|0)==0){x=4;m=765;break}n=n+1|0;if((k9(j,n,1)|0)==0){x=6;m=765;break}n=n+1|0;if((k9(j,n,1)|0)==0){n=n+1|0;v=k9(j,n,3)|0;n=n+3|0;v=v+8|0;break}n=n+1|0;v=k9(j,n,5)|0;n=n+5|0;x=16;m=766}else{x=2;m=765}}while(0);if((m|0)==765){m=0;n=n+1|0;v=k9(j,n,1)|0;n=n+1|0;m=766}if((m|0)==766){m=0;v=v+x|0}do{if((k9(j,n,1)|0)!=0){w=12;y=-256}else{n=n+1|0;if((k9(j,n,1)|0)==0){w=8;y=0;break}w=14;y=-4352}}while(0);n=n+1|0;x=k9(j,n,w)|0;n=n+w|0;v=v-3|0;do{if((v|0)>=0){if((v|0)!=0){u=u-1|0}u=u-1|0;if((u|0)>=0){break}else{u=0;break}}}while(0);v=v+2|0;s=b-1+y+(-x|0)|0;do{p=s;s=p+1|0;z=b;b=z+1|0;a[z]=a[p]|0;v=v-1|0;}while((v|0)!=-1);p=s-1|0;s=p;o=d[p]|0}else if((m|0)==742){while(1){m=0;x=la(j,n,v)|0;n=n+v|0;o=o-x|0;p=b;b=p+1|0;a[p]=o&255;w=w-1|0;if((w|0)==-1){break}m=742}if((u|0)!=31){u=u+1|0}r=v}v=t;v=v>>3;t=t-v|0;}while(b>>>0<q>>>0);b=q}if((m|0)==711){k=2;A=f;B=A;i=g;return B|0}else if((m|0)==751){A=f;B=A;i=g;return B|0}else if((m|0)==707){k=1;A=f;B=A;i=g;return B|0}return 0}function k8(a,b){a=a|0;b=b|0;var d=0,e=0;d=a;a=b;b=0;while(1){e=a;a=e-1|0;if(e>>>0<=0){break}e=d;d=e+4|0;b=b^c[e>>2]}return(b^b>>>16)&65535|0}function k9(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0;e=a;a=b;e=e+((a|0)/8|0)|0;b=e;e=b+1|0;f=d[b]|0;f=f<<8;b=e;e=b+1|0;f=f|(d[b]|0);f=f<<8;f=f|(d[e]|0);f=f<<((a|0)%8|0);f=f&16777215;f=f>>24-c;return f|0}function la(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0;e=a;a=b;e=e+((a|0)/8|0)|0;b=e;e=b+1|0;f=d[b]|0;f=f<<8;b=e;e=b+1|0;f=f|(d[b]|0);f=f<<8;f=f|(d[e]|0);f=f<<((a|0)%8|0)+8;f=f>>32-c;return f|0}function lb(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;d=i;i=i+24|0;e=d|0;f=a;a=b;b=0;mL();p7(e|0,0,24);g=pS(8192)|0;h=g;if((g|0)==0){j=-1;k=j;i=d;return k|0}c[e>>2]=h;c[e+12>>2]=h+4096;c[e+20>>2]=4096;g=mR(2,16777216)|0;while(1){if((c[e+4>>2]|0)==(c[e+8>>2]|0)){l=bB(h|0,1,4096,f|0)|0;if((l|0)<0){m=801;break}c[e+8>>2]=l;c[e+4>>2]=0}l=mO(g,e)|0;if((c[e+16>>2]|0)!=0){n=c[e+12>>2]|0;o=c[e+16>>2]|0;p=a;ay(n|0,1,o|0,p|0)|0;c[e+16>>2]=0}if((l|0)==1){m=806;break}if((l|0)!=0){if((l|0)!=2){m=809;break}}}if((m|0)==801){b=-1}else if((m|0)!=806)if((m|0)==809){b=-1}mS(g);pU(h);j=b;k=j;i=d;return k|0}function lc(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;c[a>>2]=jg(d)|0;if((c[a>>2]|0)!=67324752){e=-1;f=e;return f|0}else{c[a+4>>2]=(jc(d)|0)&65535;c[a+8>>2]=(jc(d)|0)&65535;c[a+12>>2]=(jc(d)|0)&65535;c[a+16>>2]=(jc(d)|0)&65535;c[a+20>>2]=(jc(d)|0)&65535;c[a+24>>2]=jg(d)|0;c[a+28>>2]=jg(d)|0;c[a+32>>2]=jg(d)|0;c[a+36>>2]=(jc(d)|0)&65535;c[a+40>>2]=(jc(d)|0)&65535;e=0;f=e;return f|0}return 0}function ld(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0;d=i;i=i+1096|0;e=d|0;f=d+56|0;g=d+64|0;h=a;a=b;c[f>>2]=0;b=0;if((lc(h,e)|0)==-1){j=-1;k=j;i=d;return k|0}c[e+44>>2]=pS((c[e+36>>2]|0)+1|0)|0;c[e+48>>2]=pS((c[e+40>>2]|0)+1|0)|0;le(h,c[e+44>>2]|0,c[e+36>>2]|0)|0;le(h,c[e+48>>2]|0,c[e+40>>2]|0)|0;l=aT(h|0)|0;jA(g)|0;if((c[e+32>>2]|0)!=0){if((c[e+12>>2]|0)==0){c[f>>2]=lf(h,a,c[e+32>>2]|0,g)|0}else{g=h;m=a;jG(g,m,f,1)|0}if((c[f>>2]|0)!=(c[e+24>>2]|0)){b=-4}}pU(c[e+44>>2]|0);pU(c[e+48>>2]|0);bK(h|0,l+(c[e+28>>2]|0)|0,0)|0;if((c[e+8>>2]&8|0)!=0){e=h;jg(e)|0;e=h;jg(e)|0;e=h;jg(e)|0}j=b;k=j;i=d;return k|0}function le(b,c,d){b=b|0;c=c|0;d=d|0;var e=0;e=b;b=c;c=d;d=0;while(1){if((d|0)>=(c|0)){break}a[b+d|0]=(aN(e|0)|0)&255;d=d+1|0}a[b+d|0]=0;return 0}function lf(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,j=0;e=i;i=i+16744|0;f=e|0;g=a;a=b;b=c;c=d;d=-1;h=0;while(1){if((h|0)>=(b|0)){break}if((h+16738|0)<(b|0)){j=16738}else{j=b-h|0}li(g,f|0,j)|0;lj(a,f|0,j)|0;d=jB(f|0,j,d,c)|0;h=h+j|0}i=e;return~d|0}function lg(b){b=b|0;var d=0,e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;d=i;i=i+1080|0;e=d|0;f=d+56|0;g=b;b=0;while(1){h=aT(g|0)|0;b=lc(g,e)|0;if((b|0)==-1){j=850;break}k=aT(g|0)|0;l=c[e+36>>2]|0;if((l|0)>1023){l=1023}le(g,f|0,l)|0;a[f+l|0]=0;bK(g|0,k|0,0)|0;if((cK(f|0)|0)==0){j=854;break}bK(g|0,(c[e+28>>2]|0)+(c[e+36>>2]|0)+(c[e+40>>2]|0)|0,1)|0}if((b|0)!=-1){m=h;n=m;i=d;return n|0}else{m=-1;n=m;i=d;return n|0}return 0}function lh(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;c=a;a=lg(c)|0;if((a|0)<0){d=-1;e=d;return e|0}else{f=c;g=a;bK(f|0,g|0,0)|0;g=c;c=b;ld(g,c)|0;d=0;e=d;return e|0}return 0}function li(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=a;a=b;b=c;c=0;while(1){if((c|0)>=(b|0)){break}c=c+(bB(a+c|0,1,b-c|0,d|0)|0)|0}return c|0}function lj(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=a;a=b;b=c;c=0;while(1){if((c|0)>=(b|0)){break}c=c+(ay(a+c|0,1,b-c|0,d|0)|0)|0}return c|0}function lk(a){a=a|0;var b=0,d=0;b=a;a=0;while(1){if(a>>>0>=256){break}c[b+8848+(a<<2)>>2]=a;d=0;while(1){if(d>>>0>=8){break}c[b+8848+(a<<2)>>2]=(c[b+8848+(a<<2)>>2]|0)>>>1^((c[b+8848+(a<<2)>>2]&1|0)!=0?40961:0);d=d+1|0}a=a+1|0}return 1}function ll(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=b;b=pT(1,23716)|0;if((b|0)!=0){f=b;lk(f)|0;c[b>>2]=a;do{if((lm(b)|0)!=0){c[b+72>>2]=c[b+32>>2];L1149:while(1){if((bK(c[b>>2]|0,c[b+72>>2]|0,0)|0)!=0){g=892;break}if((ln(b)|0)==0){g=892;break}if((c[b+72>>2]|0)==0){g=894;break}if((d[b+98|0]|0|0)==1){continue}if((cK(b+106|0)|0)!=0){continue}do{if((d[b+69|0]|0|0)<=2){if((d[b+96|0]|0|0)>2){break}if((d[b+96|0]|0|0)==3){if((d[b+97|0]|0|0)>1){break}}if((bK(c[b>>2]|0,c[b+76>>2]|0,0)|0)!=0){continue L1149}c[b+4>>2]=e;if((d[b+69|0]|0|0)==0){h=lo(c[b+92>>2]|0,b)|0}else{if((d[b+69|0]|0|0)==1){h=lp(b)|0}else{if((d[b+69|0]|0|0)!=2){g=912;break L1149}h=lq(b)|0}}if((h|0)==0){g=916;break L1149}continue L1149}}while(0)}if((g|0)==894){pU(b);i=0;j=i;return j|0}else if((g|0)==916){break}else if((g|0)==912){break}else if((g|0)==892){break}}}while(0);pU(b)}i=-1;j=i;return j|0}function lm(d){d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=d;lA(e+8|0,20,e)|0;a[e+28|0]=0;d=jg(c[e>>2]|0)|0;c[e+28>>2]=d;if((d|0)!=-37443620){f=0;g=f;return g|0}c[e+32>>2]=jg(c[e>>2]|0)|0;c[e+36>>2]=jg(c[e>>2]|0)|0;a[e+40|0]=ja(c[e>>2]|0)|0;a[e+41|0]=ja(c[e>>2]|0)|0;if(34<(c[e+32>>2]|0)>>>0){h=(ja(c[e>>2]|0)|0)&255}else{h=0}a[e+42|0]=h&255;if(34<(c[e+32>>2]|0)>>>0){i=jg(c[e>>2]|0)|0}else{i=0}c[e+44>>2]=i;if(34<(c[e+32>>2]|0)>>>0){j=(jc(c[e>>2]|0)|0)&65535}else{j=0}b[e+48>>1]=j&65535;if(34<(c[e+32>>2]|0)>>>0){k=(ja(c[e>>2]|0)|0)&255}else{k=0}a[e+50|0]=k&255;c[e+52>>2]=0;c[e+56>>2]=0;c[e+60>>2]=0;f=1;g=f;return g|0}function ln(f){f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;g=f;f=jg(c[g>>2]|0)|0;c[g+64>>2]=f;if((f|0)!=-37443620){h=0;i=h;return i|0}a[g+68|0]=ja(c[g>>2]|0)|0;a[g+69|0]=ja(c[g>>2]|0)|0;c[g+72>>2]=jg(c[g>>2]|0)|0;c[g+76>>2]=jg(c[g>>2]|0)|0;b[g+80>>1]=jc(c[g>>2]|0)|0;b[g+82>>1]=jc(c[g>>2]|0)|0;b[g+84>>1]=jc(c[g>>2]|0)|0;c[g+88>>2]=jg(c[g>>2]|0)|0;c[g+92>>2]=jg(c[g>>2]|0)|0;a[g+96|0]=ja(c[g>>2]|0)|0;a[g+97|0]=ja(c[g>>2]|0)|0;a[g+98|0]=ja(c[g>>2]|0)|0;a[g+99|0]=ja(c[g>>2]|0)|0;c[g+100>>2]=jg(c[g>>2]|0)|0;b[g+104>>1]=jc(c[g>>2]|0)|0;lA(g+106|0,13,g)|0;a[g+119|0]=0;if((d[g+68|0]|0|0)==2){j=(jc(c[g>>2]|0)|0)&65535}else{j=0}b[g+120>>1]=j&65535;if((d[g+68|0]|0|0)==2){k=(ja(c[g>>2]|0)|0)&255}else{k=127}a[g+122|0]=k&255;if((d[g+68|0]|0|0)==2){l=(jc(c[g>>2]|0)|0)&65535}else{l=0}b[g+124>>1]=l&65535;if(0<(e[g+120>>1]|0|0)){m=(ja(c[g>>2]|0)|0)&255}else{m=0}a[g+126|0]=m&255;if(1<(e[g+120>>1]|0|0)){n=(ja(c[g>>2]|0)|0)&255}else{n=0}a[g+127|0]=n&255;lA(g+128|0,d[g+126|0]|0,g)|0;a[g+128+(d[g+126|0]|0)|0]=0;lA(g+384|0,d[g+127|0]|0,g)|0;a[g+384+(d[g+127|0]|0)|0]=0;n=(d[g+126|0]|0)+(d[g+127|0]|0)|0;if((n+2|0)>>>0<(e[g+120>>1]|0)>>>0){o=(jc(c[g>>2]|0)|0)&65535}else{o=0}b[g+640>>1]=o&65535;if((n+4|0)>>>0<(e[g+120>>1]|0)>>>0){p=je(c[g>>2]|0)|0}else{p=0}c[g+644>>2]=p;if((n+7|0)>>>0<(e[g+120>>1]|0)>>>0){q=(ja(c[g>>2]|0)|0)&255}else{q=0}a[g+648|0]=q&255;if((n+7|0)>>>0<(e[g+120>>1]|0)>>>0){r=(jc(c[g>>2]|0)|0)&65535}else{r=0}b[g+650>>1]=r&65535;h=1;i=h;return i|0}function lo(b,d){b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=b;b=d;d=0;while(1){if(0>=e>>>0){f=989;break}if(8192<e>>>0){g=8192}else{g=e}h=g;if((lA(b+652|0,h,b)|0)!=(h|0)){f=981;break}if((lt(b+652|0,h,b)|0)!=(h|0)){f=983;break}i=0;while(1){if(i>>>0>=h>>>0){break}d=d>>>8^c[b+8848+(((d^a[b+652+i|0])&255)<<2)>>2];i=i+1|0}e=e-h|0}if((f|0)==981){j=0;k=j;return k|0}else if((f|0)==983){j=0;k=j;return k|0}else if((f|0)==989){c[b+8844>>2]=d;j=1;k=j;return k|0}return 0}function lp(a){a=a|0;var b=0,d=0,e=0;b=a;if((lu(c[b>>2]|0,c[b+4>>2]|0,b+8848|0)|0)==0){d=1;e=d;return e|0}else{d=0;e=d;return e|0}return 0}function lq(f){f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;g=f;f=0;h=0;i=h|0;h=i;if(i>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}i=g+652|0;j=g+8844|0;k=0;l=f>>>((h-16|0)>>>0)&65535;m=h-16|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}L1291:while(1){if((l|0)==0){n=1199;break}m=f>>>((h-5|0)>>>0)&31;o=h-5|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}if((m|0)==0){o=f>>>((h-5|0)>>>0)&31;p=h-5|0;h=p;if(p>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}q=0;while(1){if(q>>>0>=256){break}b[g+22660+(q<<1)>>1]=o&65535;q=q+1|0}q=0;while(1){if(q>>>0>18){break}a[g+23697+q|0]=0;q=q+1|0}}else{q=0;while(1){if(q>>>0>=m>>>0){break}r=f>>>((h-3|0)>>>0)&7;o=h-3|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}if((r|0)==7){while(1){if((f>>>((h-1|0)>>>0)&1|0)==0){break}r=r+1|0;o=h-1|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}o=h-1|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}o=q;q=o+1|0;a[g+23697+o|0]=r&255;if((q|0)==3){r=f>>>((h-2|0)>>>0)&3;o=h-2|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}while(1){o=r;r=o-1|0;if(0>=o>>>0){break}o=q;q=o+1|0;a[g+23697+o|0]=0}}}while(1){if(q>>>0>18){break}o=q;q=o+1|0;a[g+23697+o|0]=0}if((lr(19,g+23697|0,8,g+22660|0,g)|0)==0){n=1045;break}}m=f>>>((h-9|0)>>>0)&511;o=h-9|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}if((m|0)==0){s=f>>>((h-9|0)>>>0)&511;o=h-9|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}q=0;while(1){if(q>>>0>=4096){break}b[g+13956+(q<<1)>>1]=s&65535;q=q+1|0}q=0;while(1){if(q>>>0>510){break}a[g+23172+q|0]=0;q=q+1|0}}else{q=0;while(1){if(q>>>0>=m>>>0){break}r=e[g+22660+((f>>>((h-8|0)>>>0)&255)<<1)>>1]|0;if(r>>>0<=18){o=h-(d[g+23697+r|0]|0)|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}else{o=h-8|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}do{if((f>>>((h-1|0)>>>0)&1|0)!=0){r=e[g+11914+(r<<1)>>1]|0}else{r=e[g+9872+(r<<1)>>1]|0}o=h-1|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}while(18<r>>>0)}if(r>>>0<=2){if((r|0)==0){r=1}else{if((r|0)==1){r=(f>>>((h-4|0)>>>0)&15)+3|0;o=h-4|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}else{r=(f>>>((h-9|0)>>>0)&511)+20|0;o=h-9|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}}while(1){o=r;r=o-1|0;if(0>=o>>>0){break}o=q;q=o+1|0;a[g+23172+o|0]=0}}else{o=q;q=o+1|0;a[g+23172+o|0]=r-2&255}}while(1){if(q>>>0>510){break}o=q;q=o+1|0;a[g+23172+o|0]=0}if((lr(511,g+23172|0,12,g+13956|0,g)|0)==0){n=1099;break}}m=f>>>((h-4|0)>>>0)&15;o=h-4|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}if((m|0)==0){t=f>>>((h-4|0)>>>0)&15;o=h-4|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}q=0;while(1){if(q>>>0>=256){break}b[g+22148+(q<<1)>>1]=t&65535;q=q+1|0}q=0;while(1){if(q>>>0>13){break}a[g+23683+q|0]=0;q=q+1|0}}else{q=0;while(1){if(q>>>0>=m>>>0){break}r=f>>>((h-3|0)>>>0)&7;o=h-3|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}if((r|0)==7){while(1){if((f>>>((h-1|0)>>>0)&1|0)==0){break}r=r+1|0;o=h-1|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}o=h-1|0;h=o;if(o>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}o=q;q=o+1|0;a[g+23683+o|0]=r&255}while(1){if(q>>>0>13){break}m=q;q=m+1|0;a[g+23683+m|0]=0}if((lr(14,g+23683|0,8,g+22148|0,g)|0)==0){n=1133;break}}while(1){m=l;l=m-1|0;if(0>=m>>>0){break}s=e[g+13956+((f>>>((h-12|0)>>>0)&4095)<<1)>>1]|0;if(s>>>0<=510){m=h-(d[g+23172+s|0]|0)|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}else{m=h-12|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}do{if((f>>>((h-1|0)>>>0)&1|0)!=0){s=e[g+11914+(s<<1)>>1]|0}else{s=e[g+9872+(s<<1)>>1]|0}m=h-1|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}while(510<s>>>0)}if(s>>>0<=255){m=i;i=m+1|0;a[m]=s&255;k=k>>>8^c[g+8848+(((k^s)&255)<<2)>>2];if((i|0)==(j|0)){if((lt(g+652|0,i-(g+652)|0,g)|0)!=(i-(g+652)|0)){n=1155;break L1291}i=g+652|0}}else{r=s-256+3|0;t=e[g+22148+((f>>>((h-8|0)>>>0)&255)<<1)>>1]|0;if(t>>>0<=13){m=h-(d[g+23683+t|0]|0)|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}else{m=h-8|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}do{if((f>>>((h-1|0)>>>0)&1|0)!=0){t=e[g+11914+(t<<1)>>1]|0}else{t=e[g+9872+(t<<1)>>1]|0}m=h-1|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}while(13<t>>>0)}if((t|0)==0){u=0}else{u=(1<<t-1)+(f>>>((h-(t-1)|0)>>>0)&(1<<t-1)-1)|0;m=h-(t-1)|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}m=g+652+(i-(g+652)-u-1&8191)|0;do{if(i>>>0<(j+(-r|0)|0)>>>0){if(m>>>0>=(j+(-r|0)|0)>>>0){n=1184;break}o=i+r|0;do{p=m;m=p+1|0;s=a[p]|0;k=k>>>8^c[g+8848+(((k^s)&255)<<2)>>2];p=i;i=p+1|0;a[p]=s&255;}while(i>>>0<o>>>0)}else{n=1184}}while(0);if((n|0)==1184){n=0;while(1){o=r;r=o-1|0;if(0>=o>>>0){break}o=m;m=o+1|0;s=a[o]|0;k=k>>>8^c[g+8848+(((k^s)&255)<<2)>>2];o=i;i=o+1|0;a[o]=s&255;if((m|0)==(j|0)){m=g+652|0}if((i|0)==(j|0)){if((lt(g+652|0,i-(g+652)|0,g)|0)!=(i-(g+652)|0)){n=1190;break L1291}i=g+652|0}}}}}l=f>>>((h-16|0)>>>0)&65535;m=h-16|0;h=m;if(m>>>0<16){f=(f<<16)+((jd(c[g>>2]|0)|0)&65535)|0;h=h+16|0}}if((n|0)==1045){v=0;w=v;return w|0}else if((n|0)==1099){v=0;w=v;return w|0}else if((n|0)==1133){v=0;w=v;return w|0}else if((n|0)==1155){v=0;w=v;return w|0}else if((n|0)==1190){v=0;w=v;return w|0}else if((n|0)==1199){if((lt(g+652|0,i-(g+652)|0,g)|0)!=(i-(g+652)|0)){v=0;w=v;return w|0}else{c[g+8844>>2]=k;v=1;w=v;return w|0}}return 0}function lr(a,c,f,g,h){a=a|0;c=c|0;f=f|0;g=g|0;h=h|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;j=i;i=i+120|0;k=j|0;l=j+40|0;m=j+80|0;n=a;a=c;c=f;f=g;g=h;h=1;while(1){if(h>>>0>16){break}b[k+(h<<1)>>1]=0;h=h+1|0}h=0;while(1){if(h>>>0>=n>>>0){break}o=k+((d[a+h|0]|0)<<1)|0;b[o>>1]=(b[o>>1]|0)+1&65535;h=h+1|0}b[m+2>>1]=0;h=1;while(1){if(h>>>0>16){break}b[m+(h+1<<1)>>1]=(e[m+(h<<1)>>1]|0)+((e[k+(h<<1)>>1]|0)<<16-h)&65535;h=h+1|0}if((e[m+34>>1]|0|0)!=0){p=0;q=p;i=j;return q|0}k=16-c|0;h=1;while(1){if(h>>>0>c>>>0){break}o=m+(h<<1)|0;b[o>>1]=(e[o>>1]|0)>>k&65535;b[l+(h<<1)>>1]=1<<c-h&65535;h=h+1|0}while(1){if(h>>>0>16){break}b[l+(h<<1)>>1]=1<<16-h&65535;h=h+1|0}h=(e[m+(c+1<<1)>>1]|0)>>k;if((h|0)!=0){r=1<<c;while(1){if((h|0)==(r|0)){break}o=h;h=o+1|0;b[f+(o<<1)>>1]=0}}o=n;s=1<<15-c;t=0;while(1){if(t>>>0>=n>>>0){break}u=d[a+t|0]|0;v=u;if((u|0)!=0){if(v>>>0<=c>>>0){h=0;while(1){if(h>>>0>=(e[l+(v<<1)>>1]|0)>>>0){break}b[f+(h+(e[m+(v<<1)>>1]|0)<<1)>>1]=t&65535;h=h+1|0}}else{r=e[m+(v<<1)>>1]|0;u=f+(r>>>(k>>>0)<<1)|0;h=v-c|0;while(1){if((h|0)==0){break}if((e[u>>1]|0|0)==0){b[g+9872+(o<<1)>>1]=0;b[g+11914+(o<<1)>>1]=0;w=o;o=w+1|0;b[u>>1]=w&65535}if((r&s|0)!=0){u=g+11914+((e[u>>1]|0)<<1)|0}else{u=g+9872+((e[u>>1]|0)<<1)|0}r=r<<1;h=h-1|0}b[u>>1]=t&65535}w=m+(v<<1)|0;b[w>>1]=(e[w>>1]|0)+(e[l+(v<<1)>>1]|0)&65535}t=t+1|0}p=1;q=p;i=j;return q|0}function ls(a){a=a|0;var b=0;b=a;c[b+48>>2]=9;c[b+52>>2]=512;c[b+44>>2]=258;return}function lt(a,b,d){a=a|0;b=b|0;d=d|0;return ay(a|0,1,b|0,c[d+4>>2]|0)|0}function lu(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0;f=i;g=pT(1,68)|0;if((g|0)!=0){c[g+16>>2]=b;c[g+20>>2]=d;c[g+48>>2]=9;c[g+52>>2]=512;c[g+44>>2]=258;c[g+64>>2]=e;c[g+8>>2]=pS(16648)|0;if((c[g+8>>2]|0)!=0){c[g+12>>2]=(c[g+8>>2]|0)+8196;do{if((bB(c[g+12>>2]|0,1,8182,c[g+16>>2]|0)|0)!=-1){c[g+28>>2]=pS(65616)|0;if((c[g+28>>2]|0)==0){break}c[g+4>>2]=pS(16020)|0;if((c[g+4>>2]|0)==0){pU(c[g+28>>2]|0);break}ls(g);L1652:while(1){c[g+32>>2]=ly(g)|0;while(1){if((c[g+32>>2]|0)==257){break L1652}if((c[g+32>>2]|0)!=256){break}ls(g);e=ly(g)|0;c[g+32>>2]=e;c[g+36>>2]=e;d=e&255;a[g+57|0]=d;a[g+56|0]=d;if((c[g+32>>2]|0)!=257){h=1284;break}}if((h|0)==1284){h=0;lz(a[g+57|0]|0,g);continue}c[g+40>>2]=c[g+32>>2];if((c[g+32>>2]|0)>>>0>=(c[g+44>>2]|0)>>>0){c[g+32>>2]=c[g+36>>2];d=a[g+56|0]|0;e=g|0;b=c[e>>2]|0;c[e>>2]=b+1;c[(c[g+4>>2]|0)+(b<<2)>>2]=d;if((c[g>>2]|0)>>>0>=4e3){d=c[q>>2]|0;bE(d|0,28800,(j=i,i=i+1|0,i=i+7>>3<<3,c[j>>2]=0,j)|0)|0;i=j}}while(1){if((c[g+32>>2]|0)>>>0<=255){break}d=a[(c[g+28>>2]|0)+(c[g+32>>2]<<3)+4|0]|0;b=g|0;e=c[b>>2]|0;c[b>>2]=e+1;c[(c[g+4>>2]|0)+(e<<2)>>2]=d;if((c[g>>2]|0)>>>0>=4e3){d=c[q>>2]|0;bE(d|0,28800,(j=i,i=i+1|0,i=i+7>>3<<3,c[j>>2]=0,j)|0)|0;i=j}c[g+32>>2]=c[(c[g+28>>2]|0)+(c[g+32>>2]<<3)>>2]}d=c[g+32>>2]&255;a[g+56|0]=d;a[g+57|0]=d;d=a[g+57|0]|0;e=g|0;b=c[e>>2]|0;c[e>>2]=b+1;c[(c[g+4>>2]|0)+(b<<2)>>2]=d;if((c[g>>2]|0)>>>0>=4e3){d=c[q>>2]|0;bE(d|0,28800,(j=i,i=i+1|0,i=i+7>>3<<3,c[j>>2]=0,j)|0)|0;i=j}while(1){if((c[g>>2]|0)==0){break}d=g|0;b=(c[d>>2]|0)-1|0;c[d>>2]=b;lz(c[(c[g+4>>2]|0)+(b<<2)>>2]|0,g)}lw(g);c[g+36>>2]=c[g+40>>2]}if((c[8916]|0)!=0){b=ay(c[g+8>>2]|0,1,c[8916]|0,c[g+20>>2]|0)|0;if((b|0)!=(c[8916]|0)){b=c[q>>2]|0;bE(b|0,23112,(j=i,i=i+1|0,i=i+7>>3<<3,c[j>>2]=0,j)|0)|0;i=j}lv(c[g+8>>2]|0,c[8916]|0,g)}pU(c[g+4>>2]|0);pU(c[g+28>>2]|0);pU(c[g+8>>2]|0);pU(g);k=0;l=k;i=f;return l|0}}while(0);pU(c[g+8>>2]|0)}pU(g)}k=-1;l=k;i=f;return l|0}function lv(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0;f=b;b=d;d=e;e=c[d+60>>2]|0;while(1){g=b;b=g-1|0;if((g|0)==0){break}g=f;f=g+1|0;e=e>>>8^c[(c[d+64>>2]|0)+(((e^a[g])&255)<<2)>>2]}c[d+60>>2]=e;return}function lw(b){b=b|0;var d=0;d=b;a[(c[d+28>>2]|0)+(c[d+44>>2]<<3)+4|0]=a[d+57|0]|0;c[(c[d+28>>2]|0)+(c[d+44>>2]<<3)>>2]=c[d+36>>2];b=d+44|0;c[b>>2]=(c[b>>2]|0)+1;if((c[d+44>>2]|0)>>>0<(c[d+52>>2]|0)>>>0){return}if((c[d+48>>2]|0)<13){b=d+48|0;c[b>>2]=(c[b>>2]|0)+1;c[d+52>>2]=c[d+52>>2]<<1}return}function lx(a){a=a|0;var b=0;b=a;return(c[b+20>>2]|0)-(c[b+24>>2]|0)|0}function ly(b){b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,r=0,s=0,t=0,u=0;e=i;f=b;b=((c[9190]|0)>>>0)%8|0;g=((c[9190]|0)>>>0)/8|0;c[9190]=(c[9190]|0)+(c[f+48>>2]|0);if(g>>>0>=8177){c[9190]=b+(c[f+48>>2]|0);h=8182-g|0;j=(c[f+12>>2]|0)+g|0;k=c[f+12>>2]|0;while(1){if((h|0)<=0){break}l=j;j=l+1|0;m=k;k=m+1|0;a[m]=a[l]|0;h=h-1|0}if((bB(k|0,1,g|0,c[f+16>>2]|0)|0)==-1){h=c[q>>2]|0;bE(h|0,26200,(h=i,i=i+1|0,i=i+7>>3<<3,c[h>>2]=0,h)|0)|0;i=h}g=0}k=(c[f+12>>2]|0)+g|0;g=d[k]|0;k=k+1|0;g=g|d[k]<<8;k=k+1|0;if((b|0)==0){n=g;o=f;p=o+48|0;r=c[p>>2]|0;s=6800+(r<<2)|0;t=c[s>>2]|0;u=n&t;i=e;return u|0}g=g>>>(b>>>0)|a[k]<<16-b;n=g;o=f;p=o+48|0;r=c[p>>2]|0;s=6800+(r<<2)|0;t=c[s>>2]|0;u=n&t;i=e;return u|0}function lz(b,d){b=b|0;d=d|0;var e=0,f=0;e=i;f=b;b=d;if((c[8916]|0)>>>0>=8182){d=ay(c[b+8>>2]|0,1,c[8916]|0,c[b+20>>2]|0)|0;if((d|0)!=(c[8916]|0)){d=c[q>>2]|0;bE(d|0,27336,(d=i,i=i+1|0,i=i+7>>3<<3,c[d>>2]=0,d)|0)|0;i=d}lv(c[b+8>>2]|0,c[8916]|0,b);c[8916]=0}d=c[8916]|0;c[8916]=d+1;a[(c[b+8>>2]|0)+d|0]=f&255;i=e;return}function lA(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;f=b;b=d;d=e;e=0;while(1){if(e>>>0>=b>>>0){g=1343;break}h=(ja(c[d>>2]|0)|0)&255;if((h|0)==-1){g=1339;break}i=f;f=i+1|0;a[i]=h&255;e=e+1|0}if((g|0)==1339){j=e;k=j;return k|0}else if((g|0)==1343){j=b;k=j;return k|0}return 0}function lB(a){a=a|0;var b=0;b=a;if((b|0)==0){return}else{lC(b);lD(b,b);return}}function lC(a){a=a|0;var b=0,e=0,f=0,g=0;b=a;a=0;while(1){if((a|0)>=2){break}lD(b,c[b+832+(a<<2)>>2]|0);lD(b,c[b+840+(a<<2)>>2]|0);lD(b,c[b+848+(a<<2)>>2]|0);lD(b,c[b+864+(a<<2)>>2]|0);a=a+1|0}a=0;while(1){if((a|0)>=(c[b+268>>2]|0)){break}e=(c[b+400>>2]|0)+(a*24|0)|0;if((c[e+16>>2]|0)!=0){f=0;while(1){if((f|0)>=(c[(c[b+128>>2]|0)+((d[e+13|0]|0)*2096|0)+4>>2]|0)){break}lD(b,c[(c[e+16>>2]|0)+(f<<2)>>2]|0);f=f+1|0}lD(b,c[e+16>>2]|0)}lD(b,c[e+20>>2]|0);a=a+1|0}if((c[b+128>>2]|0)!=0){a=0;while(1){if((a|0)>=(c[b+124>>2]|0)){break}f=(c[b+128>>2]|0)+(a*2096|0)|0;lD(b,c[f+8>>2]|0);lD(b,c[f+28>>2]|0);lD(b,c[f+32>>2]|0);lD(b,c[f+2084>>2]|0);if((c[f+2088>>2]|0)!=0){g=(c[f+2088>>2]|0)-4|0}else{g=0}lD(b,g);a=a+1|0}lD(b,c[b+128>>2]|0)}lD(b,c[b+264>>2]|0);lD(b,c[b+400>>2]|0);a=0;while(1){if((a|0)>=(c[b+404>>2]|0)){break}lD(b,c[(c[b+408>>2]|0)+(a*40|0)+4>>2]|0);a=a+1|0}lD(b,c[b+408>>2]|0);a=0;while(1){if((a|0)>=(c[b+4>>2]|0)){break}lD(b,c[b+804+(a<<2)>>2]|0);lD(b,c[b+812+(a<<2)>>2]|0);lD(b,c[b+820+(a<<2)>>2]|0);a=a+1|0}a=0;while(1){if((a|0)>=2){break}lD(b,c[b+856+(a<<2)>>2]|0);a=a+1|0}return}function lD(a,b){a=a|0;b=b|0;pU(b);return}function lE(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+24|0;f=e|0;g=e+8|0;h=e+16|0;j=a;a=b;b=d;if((lF(j,f,h,g)|0)==0){c[j+1192>>2]=0;c[j+1188>>2]=0;k=0;l=k;i=e;return l|0}c[f>>2]=lG(j,c[f>>2]|0,c[h>>2]|0,c[g>>2]|0)|0;g=0;while(1){if((g|0)>=(c[j+4>>2]|0)){break}c[j+808+(g<<2)>>2]=(c[j+804+(g<<2)>>2]|0)+(c[h>>2]<<2);g=g+1|0}c[j+1188>>2]=c[h>>2];c[j+1192>>2]=(c[h>>2]|0)+(c[f>>2]|0);if((a|0)!=0){c[a>>2]=c[j+4>>2]}if((b|0)!=0){c[b>>2]=j+808}k=c[f>>2]|0;l=k;i=e;return l|0}function lF(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0;f=i;i=i+24|0;g=f|0;h=f+8|0;j=f+16|0;k=a;a=d;d=e;if((mn(k,a,h,d,j,g)|0)!=0){l=mr(k,b,k+416+((c[g>>2]|0)*6|0)|0,c[a>>2]|0,c[h>>2]|0,c[d>>2]|0,c[j>>2]|0,a)|0;m=l;i=f;return m|0}else{l=0;m=l;i=f;return m|0}return 0}function lG(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,h=0,i=0,j=0,k=0,l=0;f=a;a=b;b=d;d=e;if((c[f+816>>2]|0)!=0){e=c[f+816>>2]|0;h=mq(f,e)|0;i=0;while(1){if((i|0)>=(c[f+4>>2]|0)){break}j=0;while(1){if((j|0)>=(e|0)){break}g[(c[f+804+(i<<2)>>2]|0)+(b+j<<2)>>2]=+g[(c[f+804+(i<<2)>>2]|0)+(b+j<<2)>>2]*+g[h+(j<<2)>>2]+ +g[(c[f+812+(i<<2)>>2]|0)+(j<<2)>>2]*+g[h+(e-1-j<<2)>>2];j=j+1|0}i=i+1|0}}i=c[f+816>>2]|0;c[f+816>>2]=a-d;e=0;while(1){if((e|0)>=(c[f+4>>2]|0)){break}h=0;while(1){if((d+h|0)>=(a|0)){break}g[(c[f+812+(e<<2)>>2]|0)+(h<<2)>>2]=+g[(c[f+804+(e<<2)>>2]|0)+(d+h<<2)>>2];h=h+1|0}e=e+1|0}if((i|0)==0){k=0;l=k;return l|0}if((a|0)<(d|0)){d=a}a=f+1180|0;c[a>>2]=(c[a>>2]|0)+(d-b);k=d-b|0;l=k;return l|0}function lH(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0;g=i;i=i+1200|0;h=g|0;j=b;b=d;d=e;if((j|0)==0){k=0;l=k;i=g;return l|0}lI(h,f);c[h+20>>2]=j;c[h+28>>2]=j+b;c[h+24>>2]=c[h+20>>2];c[h+32>>2]=b;a[h+36|0]=0;do{if((lJ(h)|0)!=0){b=lP(h)|0;if((b|0)==0){break}j=b;f=h;p3(j|0,f|0,1196)|0;lQ(b);k=b;l=k;i=g;return l|0}}while(0);if((d|0)!=0){c[d>>2]=c[h+104>>2]}lC(h);k=0;l=k;i=g;return l|0}function lI(a,b){a=a|0;b=b|0;var d=0,e=0;d=a;a=b;p7(d|0,0,1196);if((a|0)!=0){b=d+84|0;e=a;c[b>>2]=c[e>>2];c[b+4>>2]=c[e+4>>2];c[d+88>>2]=(c[d+88>>2]|0)+3&-4;c[d+96>>2]=c[d+88>>2]}c[d+100>>2]=0;c[d+104>>2]=0;c[d+20>>2]=0;c[d+128>>2]=0;c[d+1184>>2]=-1;return}function lJ(f){f=f|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0.0,N=0;h=i;i=i+1088|0;j=h|0;k=h+8|0;l=h+1008|0;m=h+1016|0;n=h+1024|0;o=f;f=0;p=0;if((lW(o)|0)==0){q=0;r=q;i=h;return r|0}if((a[o+1139|0]&2|0)==0){q=lL(o,34)|0;r=q;i=h;return r|0}if((a[o+1139|0]&4|0)!=0){q=lL(o,34)|0;r=q;i=h;return r|0}if((a[o+1139|0]&1|0)!=0){q=lL(o,34)|0;r=q;i=h;return r|0}if((c[o+880>>2]|0)!=1){q=lL(o,34)|0;r=q;i=h;return r|0}if((d[o+884|0]|0)!=30){q=lL(o,34)|0;r=q;i=h;return r|0}if(((lM(o)|0)&255|0)!=1){q=lL(o,34)|0;r=q;i=h;return r|0}if((lX(o,j|0,6)|0)==0){q=lL(o,10)|0;r=q;i=h;return r|0}if((lY(j|0)|0)==0){q=lL(o,34)|0;r=q;i=h;return r|0}if((lZ(o)|0)!=0){q=lL(o,34)|0;r=q;i=h;return r|0}c[o+4>>2]=(lM(o)|0)&255;if((c[o+4>>2]|0)==0){q=lL(o,34)|0;r=q;i=h;return r|0}if((c[o+4>>2]|0)>1){q=lL(o,5)|0;r=q;i=h;return r|0}c[o>>2]=lZ(o)|0;if((c[o>>2]|0)==0){q=lL(o,34)|0;r=q;i=h;return r|0}lZ(o)|0;lZ(o)|0;lZ(o)|0;s=lM(o)|0;t=s&15;u=(s&255)>>4;c[o+116>>2]=1<<t;c[o+120>>2]=1<<u;do{if((t|0)>=6){if((t|0)>13){break}do{if((u|0)>=6){if((u|0)>13){break}if((t|0)>(u|0)){q=lL(o,20)|0;r=q;i=h;return r|0}s=lM(o)|0;if((s&1|0)==0){q=lL(o,34)|0;r=q;i=h;return r|0}if((lW(o)|0)==0){q=0;r=q;i=h;return r|0}if((l_(o)|0)==0){q=0;r=q;i=h;return r|0}do{v=l0(o)|0;lN(o,v);a[o+1140|0]=0;}while((v|0)!=0);if((l_(o)|0)==0){q=0;r=q;i=h;return r|0}lO();if((l$(o)|0)!=5){q=lL(o,20)|0;r=q;i=h;return r|0}v=0;while(1){if((v|0)>=6){break}a[j+v|0]=(l$(o)|0)&255;v=v+1|0}if((lY(j|0)|0)==0){q=lL(o,20)|0;r=q;i=h;return r|0}c[o+124>>2]=(l2(o,8)|0)+1;c[o+128>>2]=lV(o,(c[o+124>>2]|0)*2096|0)|0;if((c[o+128>>2]|0)==0){q=lL(o,3)|0;r=q;i=h;return r|0}p7(c[o+128>>2]|0,0,(c[o+124>>2]|0)*2096|0|0);v=0;L1970:while(1){if((v|0)>=(c[o+124>>2]|0)){w=1634;break}x=0;y=(c[o+128>>2]|0)+(v*2096|0)|0;s=(l2(o,8)|0)&255;if((s&255|0)!=66){w=1510;break}s=(l2(o,8)|0)&255;if((s&255|0)!=67){w=1512;break}s=(l2(o,8)|0)&255;if((s&255|0)!=86){w=1514;break}s=(l2(o,8)|0)&255;c[y>>2]=((l2(o,8)|0)<<8)+(s&255);s=(l2(o,8)|0)&255;z=(l2(o,8)|0)&255;c[y+4>>2]=((l2(o,8)|0)<<16)+((z&255)<<8)+(s&255);z=l2(o,1)|0;if((z|0)!=0){A=0}else{A=l2(o,1)|0}a[y+23|0]=A&255;if((a[y+23|0]|0)!=0){B=l3(o,c[y+4>>2]|0)|0}else{C=lV(o,c[y+4>>2]|0)|0;c[y+8>>2]=C;B=C}if((B|0)==0){w=1522;break}if((z|0)!=0){z=0;C=(l2(o,5)|0)+1|0;while(1){if((z|0)>=(c[y+4>>2]|0)){break}D=l2(o,l1((c[y+4>>2]|0)-z|0)|0)|0;if((z+D|0)>(c[y+4>>2]|0)){w=1527;break L1970}p7(B+z|0,C&255|0,D|0);z=z+D|0;C=C+1|0}}else{E=0;while(1){if((E|0)>=(c[y+4>>2]|0)){break}if((d[y+23|0]|0)!=0){F=l2(o,1)|0}else{F=1}if((F|0)!=0){a[B+E|0]=(l2(o,5)|0)+1&255;x=x+1|0}else{a[B+E|0]=-1}E=E+1|0}}do{if((d[y+23|0]|0)!=0){if((x|0)<(c[y+4>>2]>>2|0)){break}if((c[y+4>>2]|0)>(c[o+16>>2]|0)){c[o+16>>2]=c[y+4>>2]}c[y+8>>2]=lV(o,c[y+4>>2]|0)|0;C=c[y+8>>2]|0;z=B;D=c[y+4>>2]|0;p3(C|0,z|0,D)|0;l4(o,B,c[y+4>>2]|0);B=c[y+8>>2]|0;a[y+23|0]=0}}while(0);if((a[y+23|0]|0)!=0){G=x}else{G=0;E=0;while(1){if((E|0)>=(c[y+4>>2]|0)){break}do{if((d[B+E|0]|0)>10){if((d[B+E|0]|0)==255){break}G=G+1|0}}while(0);E=E+1|0}}c[y+2092>>2]=G;H=0;if((a[y+23|0]|0)!=0){if((c[y+2092>>2]|0)!=0){c[y+8>>2]=lV(o,c[y+2092>>2]|0)|0;if((c[y+8>>2]|0)==0){w=1562;break}c[y+32>>2]=l3(o,c[y+2092>>2]<<2)|0;if((c[y+32>>2]|0)==0){w=1564;break}H=l3(o,c[y+2092>>2]<<2)|0;if((H|0)==0){w=1566;break}}x=(c[y+4>>2]|0)+(c[y+2092>>2]<<3)|0;if(x>>>0>(c[o+16>>2]|0)>>>0){c[o+16>>2]=x}}else{c[y+32>>2]=lV(o,c[y+4>>2]<<2)|0;if((c[y+32>>2]|0)==0){w=1558;break}}if((l5(y,B,c[y+4>>2]|0,H)|0)==0){w=1572;break}if((c[y+2092>>2]|0)!=0){c[y+2084>>2]=lV(o,(c[y+2092>>2]|0)+1<<2)|0;c[y+2088>>2]=lV(o,(c[y+2092>>2]|0)+1<<2)|0;if((c[y+2088>>2]|0)!=0){x=y+2088|0;c[x>>2]=(c[x>>2]|0)+4;c[(c[y+2088>>2]|0)-4>>2]=-1}l6(y,B,H)}if((a[y+23|0]|0)!=0){l4(o,H,c[y+2092>>2]<<2);l4(o,c[y+32>>2]|0,c[y+2092>>2]<<2);l4(o,B,c[y+4>>2]|0);c[y+32>>2]=0}mb(y);a[y+21|0]=(l2(o,4)|0)&255;if((d[y+21|0]|0)>2){w=1582;break}if((d[y+21|0]|0)>0){g[y+12>>2]=+mc(l2(o,32)|0);g[y+16>>2]=+mc(l2(o,32)|0);a[y+20|0]=(l2(o,4)|0)+1&255;a[y+22|0]=(l2(o,1)|0)&255;if((d[y+21|0]|0)==1){c[y+24>>2]=mi(c[y+4>>2]|0,c[y>>2]|0)|0}else{c[y+24>>2]=ac(c[y+4>>2]|0,c[y>>2]|0)|0}I=l3(o,c[y+24>>2]<<1)|0;if((I|0)==0){w=1588;break}E=0;while(1){if((E|0)>=(c[y+24>>2]|0)){break}x=l2(o,d[y+20|0]|0)|0;if((x|0)==-1){w=1592;break L1970}b[I+(E<<1)>>1]=x&65535;E=E+1|0}L2065:do{if((d[y+21|0]|0)==1){x=d[y+23|0]|0;do{if((x|0)!=0){if((c[y+2092>>2]|0)==0){break L2065}else{c[y+28>>2]=lV(o,ac(c[y+2092>>2]<<2,c[y>>2]|0)|0)|0;break}}else{c[y+28>>2]=lV(o,ac(c[y+4>>2]<<2,c[y>>2]|0)|0)|0}}while(0);if((c[y+28>>2]|0)==0){w=1602;break L1970}if((x|0)!=0){J=c[y+2092>>2]|0}else{J=c[y+4>>2]|0}D=J;E=0;while(1){if((E|0)>=(D|0)){break}if((x|0)!=0){K=c[(c[y+2088>>2]|0)+(E<<2)>>2]|0}else{K=E}z=K;C=1;L=0;while(1){if((L|0)>=(c[y>>2]|0)){break}M=+(e[I+(((((z|0)/(C|0)|0)>>>0)%((c[y+24>>2]|0)>>>0)|0)<<1)>>1]|0)*+g[y+16>>2]+ +g[y+12>>2];N=(ac(E,c[y>>2]|0)|0)+L|0;g[(c[y+28>>2]|0)+(N<<2)>>2]=M;C=ac(C,c[y+24>>2]|0)|0;L=L+1|0}E=E+1|0}l4(o,I,c[y+24>>2]<<1);a[y+21|0]=2;w=1623}else{c[y+28>>2]=lV(o,c[y+24>>2]<<2)|0;E=0;while(1){if((E|0)>=(c[y+24>>2]|0)){break}g[(c[y+28>>2]|0)+(E<<2)>>2]=+(e[I+(E<<1)>>1]|0)*+g[y+16>>2]+ +g[y+12>>2];E=E+1|0}l4(o,I,c[y+24>>2]<<1);w=1623}}while(0);if((w|0)==1623){w=0}do{if((d[y+21|0]|0)==2){if((d[y+22|0]|0)==0){break}E=1;while(1){if((E|0)>=(c[y+24>>2]|0)){break}g[(c[y+28>>2]|0)+(E<<2)>>2]=+g[(c[y+28>>2]|0)+(E-1<<2)>>2];E=E+1|0}a[y+22|0]=0}}while(0)}v=v+1|0}if((w|0)==1510){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1527){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1562){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1564){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1566){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1512){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1572){if((a[y+23|0]|0)!=0){l4(o,H,0)}q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1582){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1588){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1558){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1522){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1634){s=(l2(o,6)|0)+1&255;v=0;while(1){if((v|0)>=(s&255|0)){break}if((l2(o,16)|0)!=0){w=1637;break}v=v+1|0}if((w|0)==1637){q=lL(o,20)|0;r=q;i=h;return r|0}c[o+132>>2]=(l2(o,6)|0)+1;c[o+264>>2]=lV(o,(c[o+132>>2]|0)*1596|0)|0;v=0;L2149:while(1){if((v|0)>=(c[o+132>>2]|0)){w=1695;break}b[o+136+(v<<1)>>1]=(l2(o,16)|0)&65535;if((e[o+136+(v<<1)>>1]|0)>1){w=1643;break}if((e[o+136+(v<<1)>>1]|0)==0){w=1645;break}x=(c[o+264>>2]|0)+(v*1596|0)|0;D=-1;a[x|0]=(l2(o,5)|0)&255;E=0;while(1){if((E|0)>=(d[x|0]|0)){break}a[x+1+E|0]=(l2(o,4)|0)&255;if((d[x+1+E|0]|0)>(D|0)){D=d[x+1+E|0]|0}E=E+1|0}E=0;while(1){if((E|0)>(D|0)){break}a[x+33+E|0]=(l2(o,3)|0)+1&255;a[x+49+E|0]=(l2(o,2)|0)&255;if((a[x+49+E|0]|0)!=0){a[x+65+E|0]=(l2(o,8)|0)&255;if((d[x+65+E|0]|0)>=(c[o+124>>2]|0)){w=1660;break L2149}}L=0;while(1){if((L|0)>=(1<<d[x+49+E|0]|0)){break}b[x+82+(E<<4)+(L<<1)>>1]=(l2(o,8)|0)-1&65535;if((b[x+82+(E<<4)+(L<<1)>>1]|0)>=(c[o+124>>2]|0)){w=1665;break L2149}L=L+1|0}E=E+1|0}a[x+1588|0]=(l2(o,2)|0)+1&255;a[x+1589|0]=(l2(o,4)|0)&255;b[x+338>>1]=0;b[x+340>>1]=1<<d[x+1589|0]&65535;c[x+1592>>2]=2;E=0;while(1){if((E|0)>=(d[x|0]|0)){break}D=d[x+1+E|0]|0;L=0;while(1){if((L|0)>=(d[x+33+D|0]|0)){break}C=(l2(o,d[x+1589|0]|0)|0)&65535;b[x+338+(c[x+1592>>2]<<1)>>1]=C;C=x+1592|0;c[C>>2]=(c[C>>2]|0)+1;L=L+1|0}E=E+1|0}E=0;while(1){if((E|0)>=(c[x+1592>>2]|0)){break}b[k+(E<<2)>>1]=b[x+338+(E<<1)>>1]|0;b[k+(E<<2)+2>>1]=E&65535;E=E+1|0}bm(k|0,c[x+1592>>2]|0,4,80);E=0;while(1){if((E|0)>=(c[x+1592>>2]|0)){break}a[x+838+E|0]=b[k+(E<<2)+2>>1]&255;E=E+1|0}E=2;while(1){if((E|0)>=(c[x+1592>>2]|0)){break}c[l>>2]=0;c[m>>2]=0;ma(x+338|0,E,l,m);a[x+1088+(E<<1)|0]=c[l>>2]&255;a[x+1088+(E<<1)+1|0]=c[m>>2]&255;E=E+1|0}if((c[x+1592>>2]|0)>(p|0)){p=c[x+1592>>2]|0}v=v+1|0}if((w|0)==1643){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1645){D=(c[o+264>>2]|0)+(v*1596|0)|0;a[D|0]=(l2(o,8)|0)&255;b[D+2>>1]=(l2(o,16)|0)&65535;b[D+4>>1]=(l2(o,16)|0)&65535;a[D+6|0]=(l2(o,6)|0)&255;a[D+7|0]=(l2(o,8)|0)&255;a[D+8|0]=(l2(o,4)|0)+1&255;E=0;while(1){if((E|0)>=(d[D+8|0]|0)){break}a[D+9+E|0]=(l2(o,8)|0)&255;E=E+1|0}q=lL(o,4)|0;r=q;i=h;return r|0}else if((w|0)==1660){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1665){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1695){c[o+268>>2]=(l2(o,6)|0)+1;c[o+400>>2]=lV(o,(c[o+268>>2]|0)*24|0)|0;v=0;L2222:while(1){if((v|0)>=(c[o+268>>2]|0)){w=1730;break}D=(c[o+400>>2]|0)+(v*24|0)|0;b[o+272+(v<<1)>>1]=(l2(o,16)|0)&65535;if((e[o+272+(v<<1)>>1]|0)>2){w=1698;break}c[D>>2]=l2(o,24)|0;c[D+4>>2]=l2(o,24)|0;c[D+8>>2]=(l2(o,24)|0)+1;a[D+12|0]=(l2(o,6)|0)+1&255;a[D+13|0]=(l2(o,8)|0)&255;E=0;while(1){if((E|0)>=(d[D+12|0]|0)){break}C=0;z=(l2(o,3)|0)&255;if((l2(o,1)|0)!=0){C=(l2(o,5)|0)&255}a[n+E|0]=((C&255)<<3)+(z&255)&255;E=E+1|0}c[D+20>>2]=lV(o,d[D+12|0]<<4)|0;E=0;while(1){if((E|0)>=(d[D+12|0]|0)){break}L=0;while(1){if((L|0)>=8){break}if((d[n+E|0]&1<<L|0)!=0){x=(l2(o,8)|0)&65535;b[(c[D+20>>2]|0)+(E<<4)+(L<<1)>>1]=x;if((b[(c[D+20>>2]|0)+(E<<4)+(L<<1)>>1]|0)>=(c[o+124>>2]|0)){w=1711;break L2222}}else{b[(c[D+20>>2]|0)+(E<<4)+(L<<1)>>1]=-1}L=L+1|0}E=E+1|0}c[D+16>>2]=lV(o,c[(c[o+128>>2]|0)+((d[D+13|0]|0)*2096|0)+4>>2]<<2)|0;if((c[D+16>>2]|0)==0){w=1719;break}p7(c[D+16>>2]|0,0,c[(c[o+128>>2]|0)+((d[D+13|0]|0)*2096|0)+4>>2]<<2|0);E=0;while(1){if((E|0)>=(c[(c[o+128>>2]|0)+((d[D+13|0]|0)*2096|0)+4>>2]|0)){break}x=c[(c[o+128>>2]|0)+((d[D+13|0]|0)*2096|0)>>2]|0;z=E;C=lV(o,x)|0;c[(c[D+16>>2]|0)+(E<<2)>>2]=C;L=x-1|0;while(1){if((L|0)<0){break}a[(c[(c[D+16>>2]|0)+(E<<2)>>2]|0)+L|0]=((z|0)%(d[D+12|0]|0)|0)&255;z=(z|0)/(d[D+12|0]|0)|0;L=L-1|0}E=E+1|0}v=v+1|0}if((w|0)==1698){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1711){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1719){q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1730){c[o+404>>2]=(l2(o,6)|0)+1;c[o+408>>2]=lV(o,(c[o+404>>2]|0)*40|0)|0;v=0;L2269:while(1){if((v|0)>=(c[o+404>>2]|0)){w=1777;break}D=(c[o+408>>2]|0)+(v*40|0)|0;if((l2(o,16)|0)!=0){w=1733;break}c[D+4>>2]=lV(o,(c[o+4>>2]|0)*3|0)|0;if((l2(o,1)|0)!=0){a[D+8|0]=(l2(o,4)|0)&255}else{a[D+8|0]=1}if((d[D+8|0]|0)>(f|0)){f=d[D+8|0]|0}if((l2(o,1)|0)!=0){b[D>>1]=(l2(o,8)|0)+1&65535;L=0;while(1){if((L|0)>=(e[D>>1]|0)){break}z=(l2(o,(l1(c[o+4>>2]|0)|0)-1|0)|0)&255;a[(c[D+4>>2]|0)+(L*3|0)|0]=z;z=(l2(o,(l1(c[o+4>>2]|0)|0)-1|0)|0)&255;a[(c[D+4>>2]|0)+(L*3|0)+1|0]=z;if((d[(c[D+4>>2]|0)+(L*3|0)|0]|0)>=(c[o+4>>2]|0)){w=1743;break L2269}if((d[(c[D+4>>2]|0)+(L*3|0)+1|0]|0)>=(c[o+4>>2]|0)){w=1745;break L2269}if((d[(c[D+4>>2]|0)+(L*3|0)|0]|0)==(d[(c[D+4>>2]|0)+(L*3|0)+1|0]|0)){w=1747;break L2269}L=L+1|0}}else{b[D>>1]=0}if((l2(o,2)|0)!=0){w=1753;break}if((d[D+8|0]|0)>1){E=0;while(1){if((E|0)>=(c[o+4>>2]|0)){break}z=(l2(o,4)|0)&255;a[(c[D+4>>2]|0)+(E*3|0)+2|0]=z;if((d[(c[D+4>>2]|0)+(E*3|0)+2|0]|0)>=(d[D+8|0]|0)){w=1758;break L2269}E=E+1|0}}else{E=0;while(1){if((E|0)>=(c[o+4>>2]|0)){break}a[(c[D+4>>2]|0)+(E*3|0)+2|0]=0;E=E+1|0}}E=0;while(1){if((E|0)>=(d[D+8|0]|0)){break}l2(o,8)|0;a[D+9+E|0]=(l2(o,8)|0)&255;a[D+24+E|0]=(l2(o,8)|0)&255;if((d[D+9+E|0]|0)>=(c[o+132>>2]|0)){w=1770;break L2269}if((d[D+24+E|0]|0)>=(c[o+268>>2]|0)){w=1772;break L2269}E=E+1|0}v=v+1|0}if((w|0)==1733){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1743){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1745){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1747){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1753){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1758){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1770){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1772){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1777){c[o+412>>2]=(l2(o,6)|0)+1;v=0;while(1){if((v|0)>=(c[o+412>>2]|0)){w=1787;break}D=o+416+(v*6|0)|0;a[D|0]=(l2(o,1)|0)&255;b[D+2>>1]=(l2(o,16)|0)&65535;b[D+4>>1]=(l2(o,16)|0)&65535;a[D+1|0]=(l2(o,8)|0)&255;if((e[D+2>>1]|0)!=0){w=1780;break}if((e[D+4>>1]|0)!=0){w=1782;break}if((d[D+1|0]|0)>=(c[o+404>>2]|0)){w=1784;break}v=v+1|0}if((w|0)==1780){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1782){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1784){q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1787){md(o);c[o+816>>2]=0;v=0;while(1){if((v|0)>=(c[o+4>>2]|0)){break}c[o+804+(v<<2)>>2]=lV(o,c[o+120>>2]<<2)|0;c[o+812+(v<<2)>>2]=lV(o,(c[o+120>>2]<<2>>>0)/2|0)|0;c[o+820+(v<<2)>>2]=lV(o,p<<1)|0;v=v+1|0}if((me(o,0,c[o+116>>2]|0)|0)==0){q=0;r=q;i=h;return r|0}if((me(o,1,c[o+120>>2]|0)|0)==0){q=0;r=q;i=h;return r|0}c[o+108>>2]=c[o+116>>2];c[o+112>>2]=c[o+120>>2];v=c[o+120>>2]<<2>>>1;D=0;z=0;while(1){if((z|0)>=(c[o+268>>2]|0)){break}x=(c[o+400>>2]|0)+(z*24|0)|0;C=(((c[x+4>>2]|0)-(c[x>>2]|0)|0)>>>0)/((c[x+8>>2]|0)>>>0)|0;if((C|0)>(D|0)){D=C}z=z+1|0}c[o+12>>2]=ac(c[o+4>>2]|0,(D<<2)+4|0)|0;if(v>>>0>(c[o+12>>2]|0)>>>0){c[o+12>>2]=v}a[o+1141|0]=1;c[o+40>>2]=lx(o)|0;q=1;r=q;i=h;return r|0}}}}}else if((w|0)==1592){l4(o,I,c[y+24>>2]<<1);q=lL(o,20)|0;r=q;i=h;return r|0}else if((w|0)==1602){l4(o,I,c[y+24>>2]<<1);q=lL(o,3)|0;r=q;i=h;return r|0}else if((w|0)==1514){q=lL(o,20)|0;r=q;i=h;return r|0}}}while(0);q=lL(o,20)|0;r=q;i=h;return r|0}}while(0);q=lL(o,20)|0;r=q;i=h;return r|0}function lK(a,d,e){a=a|0;d=d|0;e=e|0;var f=0,h=0,j=0,k=0;f=i;i=i+8|0;h=f|0;j=a;a=d;d=e;e=0;while(1){if((e|0)>=(d|0)){break}g[h>>2]=+g[a+(e<<2)>>2]+384.0;k=(c[h>>2]|0)-1136656384|0;if((k+32768|0)>>>0>65535){k=(k|0)<0?-32768:32767}b[j+(e<<1)>>1]=k&65535;e=e+1|0}i=f;return}function lL(a,b){a=a|0;b=b|0;var d=0;d=a;a=b;c[d+104>>2]=a;if((c[d+100>>2]|0)!=0){return 0}if((a|0)==1){return 0}c[d+104>>2]=a;return 0}function lM(b){b=b|0;var d=0,e=0,f=0;d=b;if((c[d+20>>2]|0)>>>0>=(c[d+28>>2]|0)>>>0){c[d+100>>2]=1;e=0;f=e;return f|0}else{b=d+20|0;d=c[b>>2]|0;c[b>>2]=d+1;e=a[d]|0;f=e;return f|0}return 0}function lN(a,b){a=a|0;b=b|0;var d=0;d=a;a=d+20|0;c[a>>2]=(c[a>>2]|0)+b;if((c[d+20>>2]|0)>>>0<(c[d+28>>2]|0)>>>0){return}c[d+100>>2]=1;return}function lO(){var a=0,b=0,d=0;a=0;while(1){if((a|0)>=256){break}b=a<<24;d=0;while(1){if((d|0)>=8){break}b=b<<1^(b>>>0>=2147483648?79764919:0);d=d+1|0}c[35736+(a<<2)>>2]=b;a=a+1|0}return}function lP(a){a=a|0;return lV(a,1196)|0}function lQ(a){a=a|0;var b=0,d=0,e=0,f=0,g=0;b=i;i=i+24|0;d=b|0;e=b+8|0;f=b+16|0;g=a;if((lF(g,d,f,e)|0)==0){i=b;return}lG(g,c[d>>2]|0,c[f>>2]|0,c[e>>2]|0)|0;i=b;return}function lR(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0;f=i;i=i+8|0;g=f|0;h=a;a=e;e=lE(h,0,g)|0;if((e|0)>(a|0)){e=a}if((e|0)==0){j=e;i=f;return j|0}lS(b,d,0,c[h+4>>2]|0,c[g>>2]|0,0,e);j=e;i=f;return j|0}function lS(a,b,d,e,f,g,h){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0;g=a;a=b;b=d;d=e;e=f;f=h;if((g|0)<(d|0)){i=g}else{i=d}d=i;i=0;while(1){if((i|0)>=(d|0)){break}lK((c[a+(i<<2)>>2]|0)+(b<<1)|0,c[e+(i<<2)>>2]|0,f);i=i+1|0}while(1){if((i|0)>=(g|0)){break}p7((c[a+(i<<2)>>2]|0)+(b<<1)|0,0,f<<1|0);i=i+1|0}return}function lT(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0;f=i;i=i+8|0;g=f|0;h=b;c[g>>2]=d;if((h|0)==1){j=lR(a,h,g,e)|0;k=j;i=f;return k|0}else{j=0;k=j;i=f;return k|0}return 0}function lU(a,b,d,e){a=a|0;b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0;f=i;i=i+8|0;g=e;e=lH(a,b,f|0,0)|0;if((e|0)==0){h=-1;j=h;i=f;return j|0}b=c[e+4>>2]<<12;c[d>>2]=c[e+4>>2];d=0;a=0;k=b;l=pS(k<<1)|0;if((l|0)==0){lB(e);h=-2;j=h;i=f;return j|0}while(1){m=lT(e,c[e+4>>2]|0,l+(a<<1)|0,k-a|0)|0;if((m|0)==0){break}d=d+m|0;a=a+(ac(m,c[e+4>>2]|0)|0)|0;if((a+b|0)>(k|0)){k=k<<1;m=pV(l,k<<1)|0;if((m|0)==0){n=1938;break}l=m}}if((n|0)==1938){pU(l);lB(e);h=-2;j=h;i=f;return j|0}lB(e);c[g>>2]=l;h=d;j=h;i=f;return j|0}function lV(a,b){a=a|0;b=b|0;var d=0,e=0;d=b;d=d+3&-4;b=a+8|0;c[b>>2]=(c[b>>2]|0)+d;if((d|0)!=0){e=pS(d)|0;return e|0}else{e=0;return e|0}return 0}function lW(a){a=a|0;var b=0,c=0,d=0;b=a;if((ml(b)|0)!=0){c=mm(b)|0;d=c;return d|0}else{c=lL(b,30)|0;d=c;return d|0}return 0}function lX(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=d;if(((c[e+20>>2]|0)+a|0)>>>0>(c[e+28>>2]|0)>>>0){c[e+100>>2]=1;f=0;g=f;return g|0}else{d=b;b=c[e+20>>2]|0;h=a;p3(d|0,b|0,h)|0;h=e+20|0;c[h>>2]=(c[h>>2]|0)+a;f=1;g=f;return g|0}return 0}function lY(a){a=a|0;return(p6(a|0,1168,6)|0)==0|0}function lZ(a){a=a|0;var b=0;b=a;a=(lM(b)|0)&255;a=a+(((lM(b)|0)&255)<<8)|0;a=a+(((lM(b)|0)&255)<<16)|0;a=a+(((lM(b)|0)&255)<<24)|0;return a|0}function l_(b){b=b|0;var d=0,e=0,f=0,g=0;d=b;while(1){if((c[d+1144>>2]|0)!=-1){e=1974;break}if((lW(d)|0)==0){e=1970;break}if((a[d+1139|0]&1|0)!=0){e=1972;break}}if((e|0)==1974){c[d+1148>>2]=0;c[d+1160>>2]=0;c[d+1164>>2]=0;a[d+1140|0]=0;f=1;g=f;return g|0}else if((e|0)==1972){f=lL(d,32)|0;g=f;return g|0}else if((e|0)==1970){f=0;g=f;return g|0}return 0}function l$(a){a=a|0;var b=0;b=a;a=mo(b)|0;c[b+1160>>2]=0;return a|0}function l0(b){b=b|0;var e=0,f=0,g=0,h=0;e=b;if((c[e+1148>>2]|0)!=0){f=0;g=f;return g|0}do{if((c[e+1144>>2]|0)==-1){c[e+1152>>2]=(c[e+880>>2]|0)-1;if((lW(e)|0)==0){c[e+1148>>2]=1;f=0;g=f;return g|0}if((a[e+1139|0]&1|0)!=0){break}f=lL(e,32)|0;g=f;return g|0}}while(0);b=e+1144|0;h=c[b>>2]|0;c[b>>2]=h+1;b=d[e+884+h|0]|0;if((b|0)<255){c[e+1148>>2]=1;c[e+1152>>2]=(c[e+1144>>2]|0)-1}if((c[e+1144>>2]|0)>=(c[e+880>>2]|0)){c[e+1144>>2]=-1}if((d[e+1140|0]|0|0)!=0){bV(23008,1163,31296,31040);return 0}a[e+1140|0]=b&255;f=b;g=f;return g|0}function l1(b){b=b|0;var c=0,d=0,e=0;c=b;if(c>>>0<16384){if(c>>>0<16){d=a[8384+c|0]|0;e=d;return e|0}if(c>>>0<512){d=(a[8384+(c>>5)|0]|0)+5|0;e=d;return e|0}else{d=(a[8384+(c>>10)|0]|0)+10|0;e=d;return e|0}}if(c>>>0<16777216){if(c>>>0<524288){d=(a[8384+(c>>15)|0]|0)+15|0;e=d;return e|0}else{d=(a[8384+(c>>20)|0]|0)+20|0;e=d;return e|0}}if(c>>>0<536870912){d=(a[8384+(c>>25)|0]|0)+25|0;e=d;return e|0}if(c>>>0<2147483648){d=(a[8384+(c>>30)|0]|0)+30|0;e=d;return e|0}else{d=0;e=d;return e|0}return 0}function l2(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,i=0;d=a;a=b;if((c[d+1160>>2]|0)<0){e=0;f=e;return f|0}do{if((c[d+1160>>2]|0)<(a|0)){if((a|0)>24){g=l2(d,24)|0;g=g+((l2(d,a-24|0)|0)<<24)|0;e=g;f=e;return f|0}if((c[d+1160>>2]|0)==0){c[d+1156>>2]=0}while(1){if((c[d+1160>>2]|0)>=(a|0)){h=2037;break}b=mo(d)|0;if((b|0)==-1){break}i=d+1156|0;c[i>>2]=(c[i>>2]|0)+(b<<c[d+1160>>2]);b=d+1160|0;c[b>>2]=(c[b>>2]|0)+8}if((h|0)==2037){break}c[d+1160>>2]=-1;e=0;f=e;return f|0}}while(0);if((c[d+1160>>2]|0)<0){e=0;f=e;return f|0}else{g=c[d+1156>>2]&(1<<a)-1;h=d+1156|0;c[h>>2]=(c[h>>2]|0)>>>(a>>>0);h=d+1160|0;c[h>>2]=(c[h>>2]|0)-a;e=g;f=e;return f|0}return 0}function l3(a,b){a=a|0;b=b|0;a=b;a=a+3&-4;return pS(a)|0}function l4(a,b,c){a=a|0;b=b|0;c=c|0;pU(b);return}function l5(a,b,e,f){a=a|0;b=b|0;e=e|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;g=i;i=i+128|0;h=g|0;j=a;a=b;b=e;e=f;f=0;p7(h|0,0,128);k=0;while(1){if((k|0)>=(b|0)){break}if((d[a+k|0]|0|0)<255){l=2052;break}k=k+1|0}if((k|0)==(b|0)){if((c[j+2092>>2]|0)!=0){bV(23008,709,31560,23304);return 0}m=1;n=m;i=g;return n|0}o=f;f=o+1|0;mk(j,0,k,o,d[a+k|0]|0,e);o=1;while(1){if((o|0)>(d[a+k|0]|0|0)){break}c[h+(o<<2)>>2]=1<<32-o;o=o+1|0}o=k+1|0;while(1){if((o|0)>=(b|0)){break}k=d[a+o|0]|0;if((k|0)!=255){while(1){if((k|0)>0){p=(c[h+(k<<2)>>2]|0)!=0^1}else{p=0}if(!p){break}k=k-1|0}if((k|0)==0){l=2074;break}q=c[h+(k<<2)>>2]|0;c[h+(k<<2)>>2]=0;r=l7(q)|0;s=f;f=s+1|0;mk(j,r,o,s,d[a+o|0]|0,e);if((k|0)!=(d[a+o|0]|0|0)){s=d[a+o|0]|0;while(1){if((s|0)<=(k|0)){break}if((c[h+(s<<2)>>2]|0)!=0){bV(23008,737,31560,22120);return 0}c[h+(s<<2)>>2]=q+(1<<32-s);s=s-1|0}}}o=o+1|0}if((l|0)==2074){bV(23008,730,31560,22856);return 0}m=1;n=m;i=g;return n|0}function l6(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;g=b;b=e;e=f;if((a[g+23|0]|0)!=0){h=0;while(1){if((h|0)>=(c[g+2092>>2]|0)){break}f=l7(c[(c[g+32>>2]|0)+(h<<2)>>2]|0)|0;c[(c[g+2084>>2]|0)+(h<<2)>>2]=f;h=h+1|0}}else{f=0;h=0;while(1){if((h|0)>=(c[g+4>>2]|0)){break}if((mp(g,a[b+h|0]|0)|0)!=0){i=l7(c[(c[g+32>>2]|0)+(h<<2)>>2]|0)|0;j=f;f=j+1|0;c[(c[g+2084>>2]|0)+(j<<2)>>2]=i}h=h+1|0}if((f|0)!=(c[g+2092>>2]|0)){bV(23008,798,31536,25136)}}bm(c[g+2084>>2]|0,c[g+2092>>2]|0,4,58);c[(c[g+2084>>2]|0)+(c[g+2092>>2]<<2)>>2]=-1;if((d[g+23|0]|0)!=0){k=c[g+2092>>2]|0}else{k=c[g+4>>2]|0}f=k;h=0;while(1){if((h|0)>=(f|0)){break}if((d[g+23|0]|0)!=0){l=d[b+(c[e+(h<<2)>>2]|0)|0]|0}else{l=d[b+h|0]|0}k=l;if((mp(g,k&255)|0)!=0){i=l7(c[(c[g+32>>2]|0)+(h<<2)>>2]|0)|0;j=0;m=c[g+2092>>2]|0;while(1){if((m|0)<=1){break}n=j+(m>>1)|0;if((c[(c[g+2084>>2]|0)+(n<<2)>>2]|0)>>>0<=i>>>0){j=n;m=m-(m>>1)|0}else{m=m>>1}}if((c[(c[g+2084>>2]|0)+(j<<2)>>2]|0)!=(i|0)){bV(23008,828,31536,24336)}if((a[g+23|0]|0)!=0){c[(c[g+2088>>2]|0)+(j<<2)>>2]=c[e+(h<<2)>>2];a[(c[g+8>>2]|0)+j|0]=k&255}else{c[(c[g+2088>>2]|0)+(j<<2)>>2]=h}}h=h+1|0}return}function l7(a){a=a|0;var b=0;b=a;b=(b&-1431655766)>>>1|(b&1431655765)<<1;b=(b&-858993460)>>>2|(b&858993459)<<2;b=(b&-252645136)>>>4|(b&252645135)<<4;b=(b&-16711936)>>>8|(b&16711935)<<8;return b>>>16|b<<16|0}function l8(a){a=+a;var b=0.0;b=a;return+(b*b)}function l9(a,b){a=a|0;b=b|0;var c=0,d=0;c=a;a=b;if((e[c>>1]|0|0)<(e[a>>1]|0|0)){d=-1;return d|0}else{d=(e[c>>1]|0|0)>(e[a>>1]|0|0)|0;return d|0}return 0}function ma(a,b,d,f){a=a|0;b=b|0;d=d|0;f=f|0;var g=0,h=0,i=0;g=a;a=b;b=d;d=f;f=-1;h=65536;i=0;while(1){if((i|0)>=(a|0)){break}do{if((e[g+(i<<1)>>1]|0|0)>(f|0)){if((e[g+(i<<1)>>1]|0|0)>=(e[g+(a<<1)>>1]|0|0)){break}c[b>>2]=i;f=e[g+(i<<1)>>1]|0}}while(0);do{if((e[g+(i<<1)>>1]|0|0)<(h|0)){if((e[g+(i<<1)>>1]|0|0)<=(e[g+(a<<1)>>1]|0|0)){break}c[d>>2]=i;h=e[g+(i<<1)>>1]|0}}while(0);i=i+1|0}return}function mb(a){a=a|0;var e=0,f=0,g=0,h=0;e=a;a=0;while(1){if((a|0)>=1024){break}b[e+36+(a<<1)>>1]=-1;a=a+1|0}if((d[e+23|0]|0|0)!=0){f=c[e+2092>>2]|0}else{f=c[e+4>>2]|0}g=f;if((g|0)>32767){g=32767}a=0;while(1){if((a|0)>=(g|0)){break}if((d[(c[e+8>>2]|0)+a|0]|0|0)<=10){if((d[e+23|0]|0|0)!=0){h=l7(c[(c[e+2084>>2]|0)+(a<<2)>>2]|0)|0}else{h=c[(c[e+32>>2]|0)+(a<<2)>>2]|0}f=h;while(1){if(f>>>0>=1024){break}b[e+36+(f<<1)>>1]=a&65535;f=f+(1<<(d[(c[e+8>>2]|0)+a|0]|0))|0}}a=a+1|0}return}function mc(a){a=a|0;var b=0,c=0.0;b=a;a=b&2097151;if((b&-2147483648|0)!=0){c=-0.0- +(a>>>0>>>0)}else{c=+(a>>>0>>>0)}return+(+bc(+c,((b&2145386496)>>>21)-788|0))}function md(a){a=a|0;var b=0;b=a;do{}while((mo(b)|0)!=-1);return}function me(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=d;d=b>>1;f=b>>3;c[e+832+(a<<2)>>2]=lV(e,d<<2)|0;c[e+840+(a<<2)>>2]=lV(e,d<<2)|0;c[e+848+(a<<2)>>2]=lV(e,b>>2<<2)|0;do{if((c[e+832+(a<<2)>>2]|0)!=0){if((c[e+840+(a<<2)>>2]|0)==0){break}if((c[e+848+(a<<2)>>2]|0)==0){break}mf(b,c[e+832+(a<<2)>>2]|0,c[e+840+(a<<2)>>2]|0,c[e+848+(a<<2)>>2]|0);c[e+856+(a<<2)>>2]=lV(e,d<<2)|0;if((c[e+856+(a<<2)>>2]|0)==0){g=lL(e,3)|0;h=g;return h|0}mg(b,c[e+856+(a<<2)>>2]|0);c[e+864+(a<<2)>>2]=lV(e,f<<1)|0;if((c[e+864+(a<<2)>>2]|0)!=0){mh(b,c[e+864+(a<<2)>>2]|0);g=1;h=g;return h|0}else{g=lL(e,3)|0;h=g;return h|0}}}while(0);g=lL(e,3)|0;h=g;return h|0}function mf(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,h=0,i=0;e=a;a=b;b=c;c=d;d=e>>2;f=e>>3;h=0;i=0;while(1){if((i|0)>=(d|0)){break}g[a+(h<<2)>>2]=+U(+(+(i<<2|0)*3.141592653589793/+(e|0)));g[a+(h+1<<2)>>2]=-0.0- +V(+(+(i<<2|0)*3.141592653589793/+(e|0)));g[b+(h<<2)>>2]=+U(+(+(h+1|0)*3.141592653589793/+(e|0)/2.0))*.5;g[b+(h+1<<2)>>2]=+V(+(+(h+1|0)*3.141592653589793/+(e|0)/2.0))*.5;i=i+1|0;h=h+2|0}h=0;i=0;while(1){if((i|0)>=(f|0)){break}g[c+(h<<2)>>2]=+U(+(+(h+1<<1|0)*3.141592653589793/+(e|0)));g[c+(h+1<<2)>>2]=-0.0- +V(+(+(h+1<<1|0)*3.141592653589793/+(e|0)));i=i+1|0;h=h+2|0}return}function mg(a,b){a=a|0;b=b|0;var c=0;c=b;b=a>>1;a=0;while(1){if((a|0)>=(b|0)){break}g[c+(a<<2)>>2]=+V(+(+l8(+V(+((+(a|0)+.5)/+(b|0)*.5*3.141592653589793)))*1.5707963267948966));a=a+1|0}return}function mh(a,c){a=a|0;c=c|0;var d=0,e=0;d=a;a=c;c=(l1(d)|0)-1|0;e=d>>3;d=0;while(1){if((d|0)>=(e|0)){break}b[a+(d<<1)>>1]=(l7(d)|0)>>>((32-c+3|0)>>>0)<<2&65535;d=d+1|0}return}function mi(a,b){a=a|0;b=b|0;var c=0;c=a;a=b;b=~~+Q(+(+$(+(+aa(+(+(c|0)))/+(a|0)))));if((~~+Q(+(+T(+(+(b|0)+1.0),+(+(a|0)))))|0)<=(c|0)){b=b+1|0}if(+T(+(+(b|0)+1.0),+(+(a|0)))<=+(c|0)){bV(23008,853,31344,27304);return 0}if((~~+Q(+(+T(+(+(b|0)),+(+(a|0)))))|0)<=(c|0)){c=b;return c|0}else{bV(23008,854,31344,26152);return 0}return 0}function mj(a,b){a=a|0;b=b|0;var d=0,e=0;d=c[a>>2]|0;a=c[b>>2]|0;if(d>>>0<a>>>0){e=-1;return e|0}else{e=d>>>0>a>>>0|0;return e|0}return 0}function mk(b,d,e,f,g,h){b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0;i=b;b=d;d=e;e=f;if((a[i+23|0]|0)!=0){c[(c[i+32>>2]|0)+(e<<2)>>2]=b;a[(c[i+8>>2]|0)+e|0]=g&255;c[h+(e<<2)>>2]=d;return}else{c[(c[i+32>>2]|0)+(d<<2)>>2]=b;return}}function ml(a){a=a|0;var b=0,c=0;b=a;do{if(79!=((lM(b)|0)&255|0)){c=0}else{if(103!=((lM(b)|0)&255|0)){c=0;break}if(103!=((lM(b)|0)&255|0)){c=0;break}if(83!=((lM(b)|0)&255|0)){c=0;break}else{c=1;break}}}while(0);return c|0}function mm(b){b=b|0;var e=0,f=0,g=0,h=0,j=0,k=0,l=0;e=i;i=i+24|0;f=e|0;g=b;if(0!=((lM(g)|0)&255|0)){h=lL(g,31)|0;j=h;i=e;return j|0}a[g+1139|0]=lM(g)|0;b=lZ(g)|0;k=lZ(g)|0;lZ(g)|0;c[g+876>>2]=lZ(g)|0;lZ(g)|0;c[g+880>>2]=(lM(g)|0)&255;if((lX(g,g+884|0,c[g+880>>2]|0)|0)==0){h=lL(g,10)|0;j=h;i=e;return j|0}c[g+1168>>2]=-2;if((b|0)!=-1){l=2252}else{if((k|0)!=-1){l=2252}}if((l|0)==2252){k=(c[g+880>>2]|0)-1|0;while(1){if((k|0)<0){break}if((d[g+884+k|0]|0)<255){l=2255;break}k=k-1|0}if((k|0)>=0){c[g+1168>>2]=k;c[g+1172>>2]=b}}if((a[g+1141|0]|0)!=0){k=0;l=0;while(1){if((l|0)>=(c[g+880>>2]|0)){break}k=k+(d[g+884+l|0]|0)|0;l=l+1|0}k=k+((c[g+880>>2]|0)+27)|0;c[f>>2]=c[g+40>>2];c[f+4>>2]=(c[f>>2]|0)+k;c[f+8>>2]=c[f>>2];c[f+12>>2]=0;c[f+16>>2]=b;b=g+44|0;k=f;c[b>>2]=c[k>>2];c[b+4>>2]=c[k+4>>2];c[b+8>>2]=c[k+8>>2];c[b+12>>2]=c[k+12>>2];c[b+16>>2]=c[k+16>>2]}c[g+1144>>2]=0;h=1;j=h;i=e;return j|0}function mn(b,e,f,g,h,i){b=b|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0;j=b;b=e;e=f;f=g;g=h;h=i;c[j+1192>>2]=0;c[j+1188>>2]=0;while(1){if((c[j+100>>2]|0)!=0){k=2274;break}if((mN(j)|0)==0){k=2276;break}if((l2(j,1)|0)==0){k=2282;break}do{}while(-1!=(l$(j)|0))}if((k|0)==2276){l=0;m=l;return m|0}else if((k|0)==2274){l=0;m=l;return m|0}else if((k|0)==2282){i=l2(j,l1((c[j+412>>2]|0)-1|0)|0)|0;if((i|0)==-1){l=0;m=l;return m|0}if((i|0)>=(c[j+412>>2]|0)){l=0;m=l;return m|0}c[h>>2]=i;h=j+416+(i*6|0)|0;if((a[h|0]|0)!=0){n=c[j+120>>2]|0;o=l2(j,1)|0;p=l2(j,1)|0}else{p=0;o=0;n=c[j+116>>2]|0}i=n>>1;do{if((d[h|0]|0)!=0){if((o|0)!=0){k=2292;break}c[b>>2]=n-(c[j+116>>2]|0)>>2;c[e>>2]=n+(c[j+116>>2]|0)>>2}else{k=2292}}while(0);if((k|0)==2292){c[b>>2]=0;c[e>>2]=i}do{if((d[h|0]|0)!=0){if((p|0)!=0){k=2296;break}c[f>>2]=(n*3|0)-(c[j+116>>2]|0)>>2;c[g>>2]=(n*3|0)+(c[j+116>>2]|0)>>2}else{k=2296}}while(0);if((k|0)==2296){c[f>>2]=i;c[g>>2]=n}l=1;m=l;return m|0}return 0}function mo(b){b=b|0;var e=0,f=0,g=0;e=b;do{if((a[e+1140|0]|0)==0){if((c[e+1148>>2]|0)!=0){f=-1;g=f;return g|0}if((l0(e)|0)!=0){break}else{f=-1;g=f;return g|0}}}while(0);if((d[e+1140|0]|0)<=0){bV(23008,1177,31472,28776);return 0}b=e+1140|0;a[b]=(a[b]|0)-1&255;b=e+1164|0;c[b>>2]=(c[b>>2]|0)+1;f=(lM(e)|0)&255;g=f;return g|0}function mp(b,c){b=b|0;c=c|0;var d=0,e=0;d=c;do{if((a[b+23|0]|0)!=0){if((d&255|0)==255){bV(23008,778,31392,23824);return 0}e=1}else{if((d&255|0)==255){e=0;break}if((d&255|0)>10){e=1;break}else{e=0;break}}}while(0);return e|0}function mq(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;a=a<<1;if((a|0)==(c[d+116>>2]|0)){e=c[d+856>>2]|0;f=e;return f|0}if((a|0)!=(c[d+120>>2]|0)){bV(23008,2794,31440,22856);return 0}e=c[d+860>>2]|0;f=e;return f|0}function mr(f,h,j,k,l,m,n,o){f=f|0;h=h|0;j=j|0;k=k|0;l=l|0;m=m|0;n=n|0;o=o|0;var p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0,R=0.0,S=0.0;l=i;i=i+2568|0;p=l|0;q=l+1024|0;r=l+2048|0;s=l+2304|0;t=l+2312|0;u=f;f=h;h=j;j=k;k=m;m=n;n=o;o=c[u+108+(d[h|0]<<2)>>2]|0;v=(c[u+408>>2]|0)+((d[h+1|0]|0)*40|0)|0;w=o>>1;x=0;while(1){if((x|0)>=(c[u+4>>2]|0)){break}y=d[(c[v+4>>2]|0)+(x*3|0)+2|0]|0;c[p+(x<<2)>>2]=0;z=d[v+9+y|0]|0;if((e[u+136+(z<<1)>>1]|0)==0){A=2341;break}y=(c[u+264>>2]|0)+(z*1596|0)|0;do{if((l2(u,1)|0)!=0){z=c[1176+((d[y+1588|0]|0)-1<<2)>>2]|0;B=2;C=c[u+820+(x<<2)>>2]|0;b[C>>1]=(l2(u,(l1(z)|0)-1|0)|0)&65535;b[C+2>>1]=(l2(u,(l1(z)|0)-1|0)|0)&65535;D=0;while(1){if((D|0)>=(d[y|0]|0)){break}E=d[y+1+D|0]|0;F=d[y+33+E|0]|0;G=d[y+49+E|0]|0;H=(1<<G)-1|0;I=0;if((G|0)!=0){J=(c[u+128>>2]|0)+((d[y+65+E|0]|0)*2096|0)|0;if((c[u+1160>>2]|0)<10){ms(u)}I=c[u+1156>>2]&1023;I=b[J+36+(I<<1)>>1]|0;if((I|0)>=0){K=d[(c[J+8>>2]|0)+I|0]|0;L=u+1156|0;c[L>>2]=(c[L>>2]|0)>>>(K>>>0);L=u+1160|0;c[L>>2]=(c[L>>2]|0)-K;if((c[u+1160>>2]|0)<0){c[u+1160>>2]=0;I=-1}}else{I=mu(u,J)|0}if((a[J+23|0]|0)!=0){I=c[(c[J+2088>>2]|0)+(I<<2)>>2]|0}}J=0;while(1){if((J|0)>=(F|0)){break}K=b[y+82+(E<<4)+((I&H)<<1)>>1]|0;I=I>>G;if((K|0)>=0){L=(c[u+128>>2]|0)+(K*2096|0)|0;if((c[u+1160>>2]|0)<10){ms(u)}K=c[u+1156>>2]&1023;K=b[L+36+(K<<1)>>1]|0;if((K|0)>=0){M=d[(c[L+8>>2]|0)+K|0]|0;N=u+1156|0;c[N>>2]=(c[N>>2]|0)>>>(M>>>0);N=u+1160|0;c[N>>2]=(c[N>>2]|0)-M;if((c[u+1160>>2]|0)<0){c[u+1160>>2]=0;K=-1}}else{K=mu(u,L)|0}if((a[L+23|0]|0)!=0){K=c[(c[L+2088>>2]|0)+(K<<2)>>2]|0}L=B;B=L+1|0;b[C+(L<<1)>>1]=K&65535}else{K=B;B=K+1|0;b[C+(K<<1)>>1]=0}J=J+1|0}D=D+1|0}if((c[u+1160>>2]|0)==-1){A=2403;break}a[r+1|0]=1;a[r|0]=1;D=2;while(1){if((D|0)>=(c[y+1592>>2]|0)){break}B=d[y+1088+(D<<1)|0]|0;J=d[y+1088+(D<<1)+1|0]|0;G=mt(e[y+338+(D<<1)>>1]|0,e[y+338+(B<<1)>>1]|0,e[y+338+(J<<1)>>1]|0,b[C+(B<<1)>>1]|0,b[C+(J<<1)>>1]|0)|0;I=b[C+(D<<1)>>1]|0;H=z-G|0;E=G;if((H|0)<(E|0)){O=H<<1}else{O=E<<1}if((I|0)!=0){a[r+J|0]=1;a[r+B|0]=1;a[r+D|0]=1;if((I|0)>=(O|0)){if((H|0)>(E|0)){b[C+(D<<1)>>1]=I-E+G&65535}else{b[C+(D<<1)>>1]=G-I+H-1&65535}}else{if((I&1|0)!=0){b[C+(D<<1)>>1]=G-(I+1>>1)&65535}else{b[C+(D<<1)>>1]=G+(I>>1)&65535}}}else{a[r+D|0]=0;b[C+(D<<1)>>1]=G&65535}D=D+1|0}D=0;while(1){if((D|0)>=(c[y+1592>>2]|0)){break}if((a[r+D|0]|0)==0){b[C+(D<<1)>>1]=-1}D=D+1|0}}else{A=2403}}while(0);if((A|0)==2403){A=0;c[p+(x<<2)>>2]=1}x=x+1|0}if((A|0)==2341){P=lL(u,21)|0;Q=P;i=l;return Q|0}r=q;O=p;y=c[u+4>>2]<<2;p3(r|0,O|0,y)|0;x=0;while(1){if((x|0)>=(e[v>>1]|0)){break}if((c[p+(d[(c[v+4>>2]|0)+(x*3|0)|0]<<2)>>2]|0)!=0){if((c[p+(d[(c[v+4>>2]|0)+(x*3|0)+1|0]<<2)>>2]|0)==0){A=2411}}else{A=2411}if((A|0)==2411){A=0;c[p+(d[(c[v+4>>2]|0)+(x*3|0)+1|0]<<2)>>2]=0;c[p+(d[(c[v+4>>2]|0)+(x*3|0)|0]<<2)>>2]=0}x=x+1|0}x=0;while(1){if((x|0)>=(d[v+8|0]|0)){break}A=0;D=0;while(1){if((D|0)>=(c[u+4>>2]|0)){break}if((d[(c[v+4>>2]|0)+(D*3|0)+2|0]|0)==(x|0)){if((c[p+(D<<2)>>2]|0)!=0){a[t+A|0]=1;c[s+(A<<2)>>2]=0}else{a[t+A|0]=0;c[s+(A<<2)>>2]=c[u+804+(D<<2)>>2]}A=A+1|0}D=D+1|0}mv(u,s|0,A,w,d[v+24+x|0]|0,t|0);x=x+1|0}x=(e[v>>1]|0)-1|0;while(1){if((x|0)<0){break}t=o>>1;s=c[u+804+(d[(c[v+4>>2]|0)+(x*3|0)|0]<<2)>>2]|0;p=c[u+804+(d[(c[v+4>>2]|0)+(x*3|0)+1|0]<<2)>>2]|0;D=0;while(1){if((D|0)>=(t|0)){break}if(+g[s+(D<<2)>>2]>0.0){if(+g[p+(D<<2)>>2]>0.0){R=+g[s+(D<<2)>>2];S=+g[s+(D<<2)>>2]- +g[p+(D<<2)>>2]}else{S=+g[s+(D<<2)>>2];R=+g[s+(D<<2)>>2]+ +g[p+(D<<2)>>2]}}else{if(+g[p+(D<<2)>>2]>0.0){R=+g[s+(D<<2)>>2];S=+g[s+(D<<2)>>2]+ +g[p+(D<<2)>>2]}else{S=+g[s+(D<<2)>>2];R=+g[s+(D<<2)>>2]- +g[p+(D<<2)>>2]}}g[s+(D<<2)>>2]=R;g[p+(D<<2)>>2]=S;D=D+1|0}x=x-1|0}x=0;while(1){if((x|0)>=(c[u+4>>2]|0)){break}if((c[q+(x<<2)>>2]|0)!=0){p7(c[u+804+(x<<2)>>2]|0,0,w<<2|0)}else{D=u;p=v;s=x;t=o;A=c[u+804+(x<<2)>>2]|0;y=c[u+820+(x<<2)>>2]|0;mw(D,p,s,t,A,y,0)|0}x=x+1|0}x=0;while(1){if((x|0)>=(c[u+4>>2]|0)){break}mx(c[u+804+(x<<2)>>2]|0,o,u,d[h|0]|0);x=x+1|0}md(u);if((a[u+1141|0]|0)!=0){c[u+824>>2]=-w;c[u+1176>>2]=o-m;c[u+828>>2]=1;a[u+1141|0]=0}else{if((c[u+1176>>2]|0)!=0){j=j+(c[u+1176>>2]|0)|0;c[n>>2]=j;c[u+1176>>2]=0}else{do{if((c[u+816>>2]|0)==0){if((c[u+828>>2]|0)==0){break}}}while(0)}}if((c[u+1152>>2]|0)==(c[u+1168>>2]|0)){do{if((c[u+828>>2]|0)!=0){if((a[u+1139|0]&4|0)==0){break}n=(c[u+1172>>2]|0)-(o-m)|0;if(n>>>0>=((c[u+824>>2]|0)+m|0)>>>0){break}if(n>>>0<(c[u+824>>2]|0)>>>0){c[f>>2]=0}else{c[f>>2]=n-(c[u+824>>2]|0)}n=f;c[n>>2]=(c[n>>2]|0)+j;n=u+824|0;c[n>>2]=(c[n>>2]|0)+(c[f>>2]|0);P=1;Q=P;i=l;return Q|0}}while(0);c[u+824>>2]=(c[u+1172>>2]|0)-(w-j);c[u+828>>2]=1}if((c[u+828>>2]|0)!=0){w=u+824|0;c[w>>2]=(c[w>>2]|0)+(k-j)}c[f>>2]=m;P=1;Q=P;i=l;return Q|0}function ms(b){b=b|0;var d=0,e=0,f=0;d=b;if((c[d+1160>>2]|0)>24){return}if((c[d+1160>>2]|0)==0){c[d+1156>>2]=0}while(1){if((c[d+1148>>2]|0)!=0){if((a[d+1140|0]|0)==0){e=2487;break}}b=mo(d)|0;if((b|0)==-1){e=2489;break}f=d+1156|0;c[f>>2]=(c[f>>2]|0)+(b<<c[d+1160>>2]);b=d+1160|0;c[b>>2]=(c[b>>2]|0)+8;if((c[d+1160>>2]|0)>24){e=2492;break}}if((e|0)==2487){return}else if((e|0)==2489){return}else if((e|0)==2492){return}}function mt(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0;f=b;b=d;d=e-b|0;e=(ac(R(d|0)|0,a-f|0)|0)/(c-f|0)|0;if((d|0)<0){g=b-e|0;return g|0}else{g=b+e|0;return g|0}return 0}function mu(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0;f=b;b=e;ms(f);do{if((c[b+2084>>2]|0)!=0){g=2506}else{if((c[b+32>>2]|0)!=0){g=2506;break}bV(23008,1268,31608,29344);return 0}}while(0);if((c[b+4>>2]|0)>8){if((c[b+2084>>2]|0)!=0){g=2511}}else{if((c[b+32>>2]|0)==0){g=2511}}if((g|0)==2511){e=l7(c[f+1156>>2]|0)|0;h=0;i=c[b+2092>>2]|0;while(1){if((i|0)<=1){break}j=h+(i>>1)|0;if((c[(c[b+2084>>2]|0)+(j<<2)>>2]|0)>>>0<=e>>>0){h=j;i=i-(i>>1)|0}else{i=i>>1}}if((a[b+23|0]|0)==0){h=c[(c[b+2088>>2]|0)+(h<<2)>>2]|0}i=d[(c[b+8>>2]|0)+h|0]|0;if((c[f+1160>>2]|0)>=(i|0)){e=f+1156|0;c[e>>2]=(c[e>>2]|0)>>>(i>>>0);e=f+1160|0;c[e>>2]=(c[e>>2]|0)-i;k=h;l=k;return l|0}else{c[f+1160>>2]=0;k=-1;l=k;return l|0}}if((a[b+23|0]|0)!=0){bV(23008,1301,31608,29184);return 0}h=0;while(1){if((h|0)>=(c[b+4>>2]|0)){g=2535;break}if((d[(c[b+8>>2]|0)+h|0]|0)!=255){if((c[(c[b+32>>2]|0)+(h<<2)>>2]|0)==(c[f+1156>>2]&(1<<d[(c[b+8>>2]|0)+h|0])-1|0)){break}}h=h+1|0}if((g|0)==2535){g=f;lL(g,21)|0;c[f+1160>>2]=0;k=-1;l=k;return l|0}if((c[f+1160>>2]|0)>=(d[(c[b+8>>2]|0)+h|0]|0)){g=f+1156|0;c[g>>2]=(c[g>>2]|0)>>>((d[(c[b+8>>2]|0)+h|0]|0)>>>0);g=f+1160|0;c[g>>2]=(c[g>>2]|0)-(d[(c[b+8>>2]|0)+h|0]|0);k=h;l=k;return l|0}else{c[f+1160>>2]=0;k=-1;l=k;return l|0}return 0}function mv(f,g,h,j,k,l){f=f|0;g=g|0;h=h|0;j=j|0;k=k|0;l=l|0;var m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,J=0,K=0,L=0,M=0,N=0,O=0;m=i;i=i+48|0;n=m|0;o=m+8|0;p=m+16|0;q=m+24|0;r=m+32|0;s=m+40|0;t=f;f=g;g=h;h=j;j=k;k=l;l=(c[t+400>>2]|0)+(j*24|0)|0;u=e[t+272+(j<<1)>>1]|0;j=c[(c[t+128>>2]|0)+((d[l+13|0]|0)*2096|0)>>2]|0;v=(((c[l+4>>2]|0)-(c[l>>2]|0)|0)>>>0)/((c[l+8>>2]|0)>>>0)|0;w=c[t+96>>2]|0;x=ac(c[t+4>>2]|0,(v<<2)+4|0)|0;y=i;i=i+x|0;i=i+7>>3<<3;x=mC(y,c[t+4>>2]|0,v<<2)|0;y=0;while(1){if((y|0)>=(g|0)){break}if((a[k+y|0]|0)==0){p7(c[f+(y<<2)>>2]|0,0,h<<2|0)}y=y+1|0}do{if((u|0)==2){if((g|0)==1){break}z=0;while(1){if((z|0)>=(g|0)){break}if((a[k+z|0]|0)==0){A=2553;break}z=z+1|0}if((z|0)==(g|0)){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}E=0;L3255:while(1){if((E|0)>=8){A=2649;break}F=0;G=0;if((g|0)==2){while(1){if((F|0)>=(v|0)){break}H=c[l>>2]|0;I=H+(ac(F,c[l+8>>2]|0)|0)|0;c[n>>2]=I&1;c[o>>2]=I>>1;if((E|0)==0){I=(c[t+128>>2]|0)+((d[l+13|0]|0)*2096|0)|0;if((c[t+1160>>2]|0)<10){ms(t)}H=c[t+1156>>2]&1023;H=b[I+36+(H<<1)>>1]|0;if((H|0)>=0){J=d[(c[I+8>>2]|0)+H|0]|0;K=t+1156|0;c[K>>2]=(c[K>>2]|0)>>>(J>>>0);K=t+1160|0;c[K>>2]=(c[K>>2]|0)-J;if((c[t+1160>>2]|0)<0){c[t+1160>>2]=0;H=-1}}else{H=mu(t,I)|0}if((a[I+23|0]|0)!=0){H=c[(c[I+2088>>2]|0)+(H<<2)>>2]|0}if((H|0)==-1){A=2574;break L3255}c[(c[x>>2]|0)+(G<<2)>>2]=c[(c[l+16>>2]|0)+(H<<2)>>2]}y=0;while(1){if((y|0)<(j|0)){L=(F|0)<(v|0)}else{L=0}if(!L){break}H=c[l>>2]|0;I=H+(ac(F,c[l+8>>2]|0)|0)|0;H=b[(c[l+20>>2]|0)+(d[(c[(c[x>>2]|0)+(G<<2)>>2]|0)+y|0]<<4)+(E<<1)>>1]|0;if((H|0)>=0){if((mF(t,(c[t+128>>2]|0)+(H*2096|0)|0,f,n,o,h,c[l+8>>2]|0)|0)==0){A=2582;break L3255}}else{I=I+(c[l+8>>2]|0)|0;c[n>>2]=I&1;c[o>>2]=I>>1}y=y+1|0;F=F+1|0}G=G+1|0}}else{if((g|0)==1){while(1){if((F|0)>=(v|0)){break}I=c[l>>2]|0;H=I+(ac(F,c[l+8>>2]|0)|0)|0;c[p>>2]=0;c[q>>2]=H;if((E|0)==0){H=(c[t+128>>2]|0)+((d[l+13|0]|0)*2096|0)|0;if((c[t+1160>>2]|0)<10){ms(t)}I=c[t+1156>>2]&1023;I=b[H+36+(I<<1)>>1]|0;if((I|0)>=0){J=d[(c[H+8>>2]|0)+I|0]|0;K=t+1156|0;c[K>>2]=(c[K>>2]|0)>>>(J>>>0);K=t+1160|0;c[K>>2]=(c[K>>2]|0)-J;if((c[t+1160>>2]|0)<0){c[t+1160>>2]=0;I=-1}}else{I=mu(t,H)|0}if((a[H+23|0]|0)!=0){I=c[(c[H+2088>>2]|0)+(I<<2)>>2]|0}if((I|0)==-1){A=2603;break L3255}c[(c[x>>2]|0)+(G<<2)>>2]=c[(c[l+16>>2]|0)+(I<<2)>>2]}y=0;while(1){if((y|0)<(j|0)){M=(F|0)<(v|0)}else{M=0}if(!M){break}I=c[l>>2]|0;H=I+(ac(F,c[l+8>>2]|0)|0)|0;I=b[(c[l+20>>2]|0)+(d[(c[(c[x>>2]|0)+(G<<2)>>2]|0)+y|0]<<4)+(E<<1)>>1]|0;if((I|0)>=0){if((mJ(t,(c[t+128>>2]|0)+(I*2096|0)|0,f,g,p,q,h,c[l+8>>2]|0)|0)==0){A=2611;break L3255}}else{H=H+(c[l+8>>2]|0)|0;c[p>>2]=0;c[q>>2]=H}y=y+1|0;F=F+1|0}G=G+1|0}}else{while(1){if((F|0)>=(v|0)){break}H=c[l>>2]|0;I=H+(ac(F,c[l+8>>2]|0)|0)|0;c[r>>2]=(I|0)%(g|0)|0;c[s>>2]=(I|0)/(g|0)|0;if((E|0)==0){I=(c[t+128>>2]|0)+((d[l+13|0]|0)*2096|0)|0;if((c[t+1160>>2]|0)<10){ms(t)}H=c[t+1156>>2]&1023;H=b[I+36+(H<<1)>>1]|0;if((H|0)>=0){J=d[(c[I+8>>2]|0)+H|0]|0;K=t+1156|0;c[K>>2]=(c[K>>2]|0)>>>(J>>>0);K=t+1160|0;c[K>>2]=(c[K>>2]|0)-J;if((c[t+1160>>2]|0)<0){c[t+1160>>2]=0;H=-1}}else{H=mu(t,I)|0}if((a[I+23|0]|0)!=0){H=c[(c[I+2088>>2]|0)+(H<<2)>>2]|0}if((H|0)==-1){A=2631;break L3255}c[(c[x>>2]|0)+(G<<2)>>2]=c[(c[l+16>>2]|0)+(H<<2)>>2]}y=0;while(1){if((y|0)<(j|0)){N=(F|0)<(v|0)}else{N=0}if(!N){break}H=c[l>>2]|0;I=H+(ac(F,c[l+8>>2]|0)|0)|0;H=b[(c[l+20>>2]|0)+(d[(c[(c[x>>2]|0)+(G<<2)>>2]|0)+y|0]<<4)+(E<<1)>>1]|0;if((H|0)>=0){if((mJ(t,(c[t+128>>2]|0)+(H*2096|0)|0,f,g,r,s,h,c[l+8>>2]|0)|0)==0){A=2639;break L3255}}else{I=I+(c[l+8>>2]|0)|0;c[r>>2]=(I|0)%(g|0)|0;c[s>>2]=(I|0)/(g|0)|0}y=y+1|0;F=F+1|0}G=G+1|0}}}E=E+1|0}if((A|0)==2639){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2631){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2582){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2649){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2611){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2603){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2574){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}}}while(0);E=0;L3385:while(1){if((E|0)>=8){A=2692;break}s=0;r=0;while(1){if((s|0)>=(v|0)){break}if((E|0)==0){z=0;while(1){if((z|0)>=(g|0)){break}if((a[k+z|0]|0)==0){h=(c[t+128>>2]|0)+((d[l+13|0]|0)*2096|0)|0;if((c[t+1160>>2]|0)<10){ms(t)}N=c[t+1156>>2]&1023;N=b[h+36+(N<<1)>>1]|0;if((N|0)>=0){q=d[(c[h+8>>2]|0)+N|0]|0;p=t+1156|0;c[p>>2]=(c[p>>2]|0)>>>(q>>>0);p=t+1160|0;c[p>>2]=(c[p>>2]|0)-q;if((c[t+1160>>2]|0)<0){c[t+1160>>2]=0;N=-1}}else{N=mu(t,h)|0}if((a[h+23|0]|0)!=0){N=c[(c[h+2088>>2]|0)+(N<<2)>>2]|0}if((N|0)==-1){A=2668;break L3385}c[(c[x+(z<<2)>>2]|0)+(r<<2)>>2]=c[(c[l+16>>2]|0)+(N<<2)>>2]}z=z+1|0}}y=0;while(1){if((y|0)<(j|0)){O=(s|0)<(v|0)}else{O=0}if(!O){break}z=0;while(1){if((z|0)>=(g|0)){break}if((a[k+z|0]|0)==0){N=b[(c[l+20>>2]|0)+(d[(c[(c[x+(z<<2)>>2]|0)+(r<<2)>>2]|0)+y|0]<<4)+(E<<1)>>1]|0;if((N|0)>=0){h=c[f+(z<<2)>>2]|0;q=c[l>>2]|0;p=q+(ac(s,c[l+8>>2]|0)|0)|0;if((mG(t,(c[t+128>>2]|0)+(N*2096|0)|0,h,p,c[l+8>>2]|0,u)|0)==0){A=2682;break L3385}}}z=z+1|0}y=y+1|0;s=s+1|0}r=r+1|0}E=E+1|0}if((A|0)==2682){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2668){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}else if((A|0)==2692){B=w;C=t;D=C+96|0;c[D>>2]=B;i=m;return}}function mw(a,f,h,i,j,k,l){a=a|0;f=f|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;var m=0,n=0,o=0,p=0,q=0;l=a;a=f;f=j;j=k;k=i>>1;i=d[a+9+(d[(c[a+4>>2]|0)+(h*3|0)+2|0]|0)|0]|0;if((e[l+136+(i<<1)>>1]|0)==0){m=lL(l,21)|0;n=m;return n|0}h=(c[l+264>>2]|0)+(i*1596|0)|0;i=0;l=ac(b[j>>1]|0,d[h+1588|0]|0)|0;a=1;while(1){if((a|0)>=(c[h+1592>>2]|0)){break}o=d[h+838+a|0]|0;if((b[j+(o<<1)>>1]|0)>=0){p=ac(b[j+(o<<1)>>1]|0,d[h+1588|0]|0)|0;q=e[h+338+(o<<1)>>1]|0;mE(f,i,l,q,p,k);i=q;l=p}a=a+1|0}if((i|0)<(k|0)){o=i;while(1){if((o|0)>=(k|0)){break}i=f+(o<<2)|0;g[i>>2]=+g[i>>2]*+g[7328+(l<<2)>>2];o=o+1|0}}m=1;n=m;return n|0}function mx(a,b,d,f){a=a|0;b=b|0;d=d|0;f=f|0;var h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0.0,x=0.0,y=0,z=0,A=0,B=0.0,C=0.0,D=0.0,E=0.0;h=i;j=a;a=b;b=d;d=f;f=a>>1;k=a>>2;l=a>>3;m=c[b+96>>2]|0;n=i;i=i+(f<<2)|0;i=i+7>>3<<3;o=n;n=0;p=0;q=c[b+832+(d<<2)>>2]|0;r=o+(f-2<<2)|0;s=q;t=j|0;u=j+(f<<2)|0;while(1){if((t|0)==(u|0)){break}g[r+4>>2]=+g[t>>2]*+g[s>>2]- +g[t+8>>2]*+g[s+4>>2];g[r>>2]=+g[t>>2]*+g[s+4>>2]+ +g[t+8>>2]*+g[s>>2];r=r-8|0;s=s+8|0;t=t+16|0}t=j+(f-3<<2)|0;while(1){if(r>>>0<o>>>0){break}g[r+4>>2]=(-0.0- +g[t+8>>2])*+g[s>>2]-(-0.0- +g[t>>2])*+g[s+4>>2];g[r>>2]=(-0.0- +g[t+8>>2])*+g[s+4>>2]+(-0.0- +g[t>>2])*+g[s>>2];r=r-8|0;s=s+8|0;t=t-16|0}n=j;p=o;t=q+(f-8<<2)|0;s=p+(k<<2)|0;r=p|0;u=n+(k<<2)|0;v=n|0;while(1){if(t>>>0<q>>>0){break}w=+g[s+4>>2]- +g[r+4>>2];x=+g[s>>2]- +g[r>>2];g[u+4>>2]=+g[s+4>>2]+ +g[r+4>>2];g[u>>2]=+g[s>>2]+ +g[r>>2];g[v+4>>2]=w*+g[t+16>>2]-x*+g[t+20>>2];g[v>>2]=x*+g[t+16>>2]+w*+g[t+20>>2];w=+g[s+12>>2]- +g[r+12>>2];x=+g[s+8>>2]- +g[r+8>>2];g[u+12>>2]=+g[s+12>>2]+ +g[r+12>>2];g[u+8>>2]=+g[s+8>>2]+ +g[r+8>>2];g[v+12>>2]=w*+g[t>>2]-x*+g[t+4>>2];g[v+8>>2]=x*+g[t>>2]+w*+g[t+4>>2];t=t-32|0;u=u+16|0;v=v+16|0;s=s+16|0;r=r+16|0}r=(l1(a)|0)-1|0;mA(a>>4,n,f-1-0|0,-(a>>3)|0,q);mA(a>>4,n,f-1-k|0,-(a>>3)|0,q);my(a>>5,n,f-1-0|0,-(a>>4)|0,q,16);my(a>>5,n,f-1-l|0,-(a>>4)|0,q,16);my(a>>5,n,f-1-(l<<1)|0,-(a>>4)|0,q,16);my(a>>5,n,f-1-(l*3|0)|0,-(a>>4)|0,q,16);l=2;while(1){if((l|0)>=(r-3>>1|0)){break}s=a>>l+2;v=s>>1;u=1<<l+1;t=0;while(1){if((t|0)>=(u|0)){break}my(a>>l+4,n,f-1-(ac(s,t)|0)|0,-v|0,q,1<<l+3);t=t+1|0}l=l+1|0}while(1){if((l|0)>=(r-6|0)){break}t=a>>l+2;v=1<<l+3;s=t>>1;u=1<<l+1;y=q;z=f-1|0;A=a>>l+6;while(1){if((A|0)<=0){break}mz(u,n,z,-s|0,y,v,t);y=y+(v<<2<<2)|0;z=z-8|0;A=A-1|0}l=l+1|0}mD(a>>5,n,f-1|0,q,a);q=c[b+864+(d<<2)>>2]|0;l=p+(k-4<<2)|0;k=p+(f-4<<2)|0;while(1){if(l>>>0<p>>>0){break}r=e[q>>1]|0;g[k+12>>2]=+g[n+(r<<2)>>2];g[k+8>>2]=+g[n+(r+1<<2)>>2];g[l+12>>2]=+g[n+(r+2<<2)>>2];g[l+8>>2]=+g[n+(r+3<<2)>>2];r=e[q+2>>1]|0;g[k+4>>2]=+g[n+(r<<2)>>2];g[k>>2]=+g[n+(r+1<<2)>>2];g[l+4>>2]=+g[n+(r+2<<2)>>2];g[l>>2]=+g[n+(r+3<<2)>>2];l=l-16|0;k=k-16|0;q=q+4|0}if((p|0)!=(o|0)){bV(23008,2547,31376,30760)}q=c[b+848+(d<<2)>>2]|0;k=p;l=p+(f<<2)-16|0;while(1){if(k>>>0>=l>>>0){break}w=+g[k>>2]- +g[l+8>>2];x=+g[k+4>>2]+ +g[l+12>>2];B=+g[q+4>>2]*w+ +g[q>>2]*x;C=+g[q+4>>2]*x- +g[q>>2]*w;D=+g[k>>2]+ +g[l+8>>2];E=+g[k+4>>2]- +g[l+12>>2];g[k>>2]=D+B;g[k+4>>2]=E+C;g[l+8>>2]=D-B;g[l+12>>2]=C-E;w=+g[k+8>>2]- +g[l>>2];x=+g[k+12>>2]+ +g[l+4>>2];B=+g[q+12>>2]*w+ +g[q+8>>2]*x;C=+g[q+12>>2]*x- +g[q+8>>2]*w;D=+g[k+8>>2]+ +g[l>>2];E=+g[k+12>>2]- +g[l+4>>2];g[k+8>>2]=D+B;g[k+12>>2]=E+C;g[l>>2]=D-B;g[l+4>>2]=C-E;q=q+16|0;k=k+16|0;l=l-16|0}l=(c[b+840+(d<<2)>>2]|0)+(f<<2)-32|0;d=o+(f<<2)-32|0;o=j|0;k=j+(f-4<<2)|0;q=j+(f<<2)|0;f=j+(a-4<<2)|0;while(1){if(d>>>0<p>>>0){break}E=+g[d+24>>2]*+g[l+28>>2]- +g[d+28>>2]*+g[l+24>>2];C=(-0.0- +g[d+24>>2])*+g[l+24>>2]- +g[d+28>>2]*+g[l+28>>2];g[o>>2]=E;g[k+12>>2]=-0.0-E;g[q>>2]=C;g[f+12>>2]=C;B=+g[d+16>>2]*+g[l+20>>2]- +g[d+20>>2]*+g[l+16>>2];D=(-0.0- +g[d+16>>2])*+g[l+16>>2]- +g[d+20>>2]*+g[l+20>>2];g[o+4>>2]=B;g[k+8>>2]=-0.0-B;g[q+4>>2]=D;g[f+8>>2]=D;E=+g[d+8>>2]*+g[l+12>>2]- +g[d+12>>2]*+g[l+8>>2];C=(-0.0- +g[d+8>>2])*+g[l+8>>2]- +g[d+12>>2]*+g[l+12>>2];g[o+8>>2]=E;g[k+4>>2]=-0.0-E;g[q+8>>2]=C;g[f+4>>2]=C;B=+g[d>>2]*+g[l+4>>2]- +g[d+4>>2]*+g[l>>2];D=(-0.0- +g[d>>2])*+g[l>>2]- +g[d+4>>2]*+g[l+4>>2];g[o+12>>2]=B;g[k>>2]=-0.0-B;g[q+12>>2]=D;g[f>>2]=D;l=l-32|0;d=d-32|0;o=o+16|0;q=q+16|0;k=k-16|0;f=f-16|0}c[b+96>>2]=m;i=h;return}function my(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var h=0,i=0.0,j=0.0;h=e;e=f;f=b+(c<<2)|0;c=f+(d<<2)|0;d=a>>2;while(1){if((d|0)<=0){break}i=+g[f>>2]- +g[c>>2];j=+g[f-4>>2]- +g[c-4>>2];a=f|0;g[a>>2]=+g[a>>2]+ +g[c>>2];a=f-4|0;g[a>>2]=+g[a>>2]+ +g[c-4>>2];g[c>>2]=i*+g[h>>2]-j*+g[h+4>>2];g[c-4>>2]=j*+g[h>>2]+i*+g[h+4>>2];h=h+(e<<2)|0;i=+g[f-8>>2]- +g[c-8>>2];j=+g[f-12>>2]- +g[c-12>>2];a=f-8|0;g[a>>2]=+g[a>>2]+ +g[c-8>>2];a=f-12|0;g[a>>2]=+g[a>>2]+ +g[c-12>>2];g[c-8>>2]=i*+g[h>>2]-j*+g[h+4>>2];g[c-12>>2]=j*+g[h>>2]+i*+g[h+4>>2];h=h+(e<<2)|0;i=+g[f-16>>2]- +g[c-16>>2];j=+g[f-20>>2]- +g[c-20>>2];a=f-16|0;g[a>>2]=+g[a>>2]+ +g[c-16>>2];a=f-20|0;g[a>>2]=+g[a>>2]+ +g[c-20>>2];g[c-16>>2]=i*+g[h>>2]-j*+g[h+4>>2];g[c-20>>2]=j*+g[h>>2]+i*+g[h+4>>2];h=h+(e<<2)|0;i=+g[f-24>>2]- +g[c-24>>2];j=+g[f-28>>2]- +g[c-28>>2];a=f-24|0;g[a>>2]=+g[a>>2]+ +g[c-24>>2];a=f-28|0;g[a>>2]=+g[a>>2]+ +g[c-28>>2];g[c-24>>2]=i*+g[h>>2]-j*+g[h+4>>2];g[c-28>>2]=j*+g[h>>2]+i*+g[h+4>>2];f=f-32|0;c=c-32|0;h=h+(e<<2)|0;d=d-1|0}return}function mz(a,b,c,d,e,f,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;h=h|0;var i=0,j=0.0,k=0.0,l=0.0,m=0.0,n=0.0,o=0.0,p=0.0,q=0.0,r=0.0,s=0.0;i=e;e=f;f=h;j=+g[i>>2];k=+g[i+4>>2];l=+g[i+(e<<2)>>2];m=+g[i+(e+1<<2)>>2];n=+g[i+(e<<1<<2)>>2];o=+g[i+((e<<1)+1<<2)>>2];p=+g[i+((e*3|0)<<2)>>2];q=+g[i+((e*3|0)+1<<2)>>2];e=b+(c<<2)|0;c=e+(d<<2)|0;d=a;while(1){if((d|0)<=0){break}r=+g[e>>2]- +g[c>>2];s=+g[e-4>>2]- +g[c-4>>2];g[e>>2]=+g[e>>2]+ +g[c>>2];g[e-4>>2]=+g[e-4>>2]+ +g[c-4>>2];g[c>>2]=r*j-s*k;g[c-4>>2]=s*j+r*k;r=+g[e-8>>2]- +g[c-8>>2];s=+g[e-12>>2]- +g[c-12>>2];g[e-8>>2]=+g[e-8>>2]+ +g[c-8>>2];g[e-12>>2]=+g[e-12>>2]+ +g[c-12>>2];g[c-8>>2]=r*l-s*m;g[c-12>>2]=s*l+r*m;r=+g[e-16>>2]- +g[c-16>>2];s=+g[e-20>>2]- +g[c-20>>2];g[e-16>>2]=+g[e-16>>2]+ +g[c-16>>2];g[e-20>>2]=+g[e-20>>2]+ +g[c-20>>2];g[c-16>>2]=r*n-s*o;g[c-20>>2]=s*n+r*o;r=+g[e-24>>2]- +g[c-24>>2];s=+g[e-28>>2]- +g[c-28>>2];g[e-24>>2]=+g[e-24>>2]+ +g[c-24>>2];g[e-28>>2]=+g[e-28>>2]+ +g[c-28>>2];g[c-24>>2]=r*p-s*q;g[c-28>>2]=s*p+r*q;e=e+(-f<<2)|0;c=c+(-f<<2)|0;d=d-1|0}return}function mA(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,h=0.0,i=0.0;f=a;a=e;e=b+(c<<2)|0;c=e+(d<<2)|0;if((f&3|0)!=0){bV(23008,2142,31408,30440)}d=f>>2;while(1){if((d|0)<=0){break}h=+g[e>>2]- +g[c>>2];i=+g[e-4>>2]- +g[c-4>>2];f=e|0;g[f>>2]=+g[f>>2]+ +g[c>>2];f=e-4|0;g[f>>2]=+g[f>>2]+ +g[c-4>>2];g[c>>2]=h*+g[a>>2]-i*+g[a+4>>2];g[c-4>>2]=i*+g[a>>2]+h*+g[a+4>>2];a=a+32|0;h=+g[e-8>>2]- +g[c-8>>2];i=+g[e-12>>2]- +g[c-12>>2];f=e-8|0;g[f>>2]=+g[f>>2]+ +g[c-8>>2];f=e-12|0;g[f>>2]=+g[f>>2]+ +g[c-12>>2];g[c-8>>2]=h*+g[a>>2]-i*+g[a+4>>2];g[c-12>>2]=i*+g[a>>2]+h*+g[a+4>>2];a=a+32|0;h=+g[e-16>>2]- +g[c-16>>2];i=+g[e-20>>2]- +g[c-20>>2];f=e-16|0;g[f>>2]=+g[f>>2]+ +g[c-16>>2];f=e-20|0;g[f>>2]=+g[f>>2]+ +g[c-20>>2];g[c-16>>2]=h*+g[a>>2]-i*+g[a+4>>2];g[c-20>>2]=i*+g[a>>2]+h*+g[a+4>>2];a=a+32|0;h=+g[e-24>>2]- +g[c-24>>2];i=+g[e-28>>2]- +g[c-28>>2];f=e-24|0;g[f>>2]=+g[f>>2]+ +g[c-24>>2];f=e-28|0;g[f>>2]=+g[f>>2]+ +g[c-28>>2];g[c-24>>2]=h*+g[a>>2]-i*+g[a+4>>2];g[c-28>>2]=i*+g[a>>2]+h*+g[a+4>>2];a=a+32|0;e=e-32|0;c=c-32|0;d=d-1|0}return}function mB(a){a=a|0;var b=0,c=0.0,d=0.0,e=0.0,f=0.0;b=a;c=+g[b>>2]- +g[b-16>>2];d=+g[b>>2]+ +g[b-16>>2];e=+g[b-8>>2]+ +g[b-24>>2];f=+g[b-8>>2]- +g[b-24>>2];g[b>>2]=d+e;g[b-8>>2]=d-e;e=+g[b-12>>2]- +g[b-28>>2];g[b-16>>2]=c+e;g[b-24>>2]=c-e;e=+g[b-4>>2]- +g[b-20>>2];c=+g[b-4>>2]+ +g[b-20>>2];d=+g[b-12>>2]+ +g[b-28>>2];g[b-4>>2]=c+d;g[b-12>>2]=c-d;g[b-20>>2]=e-f;g[b-28>>2]=e+f;return}function mC(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0;e=b;b=d;d=a;a=d+(e<<2)|0;f=0;while(1){if((f|0)>=(e|0)){break}c[d+(f<<2)>>2]=a;a=a+b|0;f=f+1|0}return d|0}function mD(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0.0,h=0.0,i=0.0;f=+g[d+(e>>3<<2)>>2];e=b+(c<<2)|0;c=e+(-(a<<4)<<2)|0;while(1){if(e>>>0<=c>>>0){break}h=+g[e>>2]- +g[e-32>>2];i=+g[e-4>>2]- +g[e-36>>2];g[e>>2]=+g[e>>2]+ +g[e-32>>2];g[e-4>>2]=+g[e-4>>2]+ +g[e-36>>2];g[e-32>>2]=h;g[e-36>>2]=i;h=+g[e-8>>2]- +g[e-40>>2];i=+g[e-12>>2]- +g[e-44>>2];g[e-8>>2]=+g[e-8>>2]+ +g[e-40>>2];g[e-12>>2]=+g[e-12>>2]+ +g[e-44>>2];g[e-40>>2]=(h+i)*f;g[e-44>>2]=(i-h)*f;h=+g[e-48>>2]- +g[e-16>>2];i=+g[e-20>>2]- +g[e-52>>2];g[e-16>>2]=+g[e-16>>2]+ +g[e-48>>2];g[e-20>>2]=+g[e-20>>2]+ +g[e-52>>2];g[e-48>>2]=i;g[e-52>>2]=h;h=+g[e-56>>2]- +g[e-24>>2];i=+g[e-28>>2]- +g[e-60>>2];g[e-24>>2]=+g[e-24>>2]+ +g[e-56>>2];g[e-28>>2]=+g[e-28>>2]+ +g[e-60>>2];g[e-56>>2]=(h+i)*f;g[e-60>>2]=(h-i)*f;mB(e);mB(e-32|0);e=e-64|0}return}function mE(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var h=0,i=0,j=0,k=0,l=0;h=a;a=b;b=c;c=d;d=f;f=e-b|0;e=c-a|0;i=R(f|0)|0;j=a;a=b;b=0;k=(f|0)/(e|0)|0;if((f|0)<0){l=k-1|0}else{l=k+1|0}i=i-(ac(R(k|0)|0,e)|0)|0;if((c|0)>(d|0)){c=d}d=h+(j<<2)|0;g[d>>2]=+g[d>>2]*+g[7328+(a<<2)>>2];j=j+1|0;while(1){if((j|0)>=(c|0)){break}b=b+i|0;if((b|0)>=(e|0)){b=b-e|0;a=a+l|0}else{a=a+k|0}d=h+(j<<2)|0;g[d>>2]=+g[d>>2]*+g[7328+(a<<2)>>2];j=j+1|0}return}function mF(e,f,h,i,j,k,l){e=e|0;f=f|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;var m=0,n=0,o=0,p=0,q=0,r=0,s=0.0,t=0,u=0,v=0,w=0,x=0.0,y=0;m=e;e=f;f=h;h=i;i=j;j=k;k=l;l=c[h>>2]|0;n=c[i>>2]|0;o=c[e>>2]|0;if((d[e+21|0]|0)==0){p=lL(m,21)|0;q=p;return q|0}while(1){if((k|0)<=0){r=2841;break}s=0.0;if((c[m+1160>>2]|0)<10){ms(m)}t=c[m+1156>>2]&1023;t=b[e+36+(t<<1)>>1]|0;if((t|0)>=0){u=d[(c[e+8>>2]|0)+t|0]|0;v=m+1156|0;c[v>>2]=(c[v>>2]|0)>>>(u>>>0);v=m+1160|0;c[v>>2]=(c[v>>2]|0)-u;if((c[m+1160>>2]|0)<0){c[m+1160>>2]=0;t=-1}}else{t=mu(m,e)|0}if((t|0)<0){break}if((l+(n<<1)+o|0)>(j<<1|0)){o=(j<<1)-((n<<1)-l)|0}t=ac(t,c[e>>2]|0)|0;if((a[e+22|0]|0)!=0){w=0;while(1){if((w|0)>=(o|0)){break}x=+g[(c[e+28>>2]|0)+(t+w<<2)>>2]+s;u=(c[f+(l<<2)>>2]|0)+(n<<2)|0;g[u>>2]=+g[u>>2]+x;u=l+1|0;l=u;if((u|0)==2){l=0;n=n+1|0}s=x;w=w+1|0}}else{w=0;if((l|0)==1){u=(c[f+(l<<2)>>2]|0)+(n<<2)|0;g[u>>2]=+g[u>>2]+(+g[(c[e+28>>2]|0)+(t+w<<2)>>2]+s);l=0;n=n+1|0;w=w+1|0}u=c[f>>2]|0;v=c[f+4>>2]|0;while(1){if((w+1|0)>=(o|0)){break}y=u+(n<<2)|0;g[y>>2]=+g[y>>2]+(+g[(c[e+28>>2]|0)+(t+w<<2)>>2]+s);y=v+(n<<2)|0;g[y>>2]=+g[y>>2]+(+g[(c[e+28>>2]|0)+(t+w+1<<2)>>2]+s);n=n+1|0;w=w+2|0}if((w|0)<(o|0)){v=(c[f+(l<<2)>>2]|0)+(n<<2)|0;g[v>>2]=+g[v>>2]+(+g[(c[e+28>>2]|0)+(t+w<<2)>>2]+s);v=l+1|0;l=v;if((v|0)==2){l=0;n=n+1|0}}}k=k-o|0}if((r|0)==2841){c[h>>2]=l;c[i>>2]=n;p=1;q=p;return q|0}do{if((a[m+1140|0]|0)==0){if((c[m+1148>>2]|0)==0){break}p=0;q=p;return q|0}}while(0);p=lL(m,21)|0;q=p;return q|0}function mG(a,b,d,e,f,g){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0;h=a;a=b;b=d;d=e;e=f;do{if((g|0)==0){f=(e|0)/(c[a>>2]|0)|0;i=0;while(1){if((i|0)>=(f|0)){j=2854;break}if((mH(h,a,b+(d<<2)+(i<<2)|0,e-d-i|0,f)|0)==0){break}i=i+1|0}if((j|0)==2854){break}k=0;l=k;return l|0}else{i=0;while(1){if((i|0)>=(e|0)){j=2860;break}if((mI(h,a,b+(d<<2)|0,e-i|0)|0)==0){break}i=i+(c[a>>2]|0)|0;d=d+(c[a>>2]|0)|0}if((j|0)==2860){break}k=0;l=k;return l|0}}while(0);k=1;l=k;return l|0}function mH(b,d,e,f,h){b=b|0;d=d|0;e=e|0;f=f|0;h=h|0;var i=0,j=0.0,k=0,l=0,m=0.0,n=0;i=d;d=e;e=f;f=h;h=mK(b,i,e)|0;j=0.0;if((h|0)<0){k=0;l=k;return l|0}if((e|0)>(c[i>>2]|0)){e=c[i>>2]|0}h=ac(h,c[i>>2]|0)|0;b=0;while(1){if((b|0)>=(e|0)){break}m=+g[(c[i+28>>2]|0)+(h+b<<2)>>2]+j;n=d+((ac(b,f)|0)<<2)|0;g[n>>2]=+g[n>>2]+m;if((a[i+22|0]|0)!=0){j=m}b=b+1|0}k=1;l=k;return l|0}function mI(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var h=0,i=0,j=0,k=0.0,l=0,m=0.0;h=d;d=e;e=f;f=mK(b,h,e)|0;if((f|0)<0){i=0;j=i;return j|0}if((e|0)>(c[h>>2]|0)){e=c[h>>2]|0}f=ac(f,c[h>>2]|0)|0;if((a[h+22|0]|0)!=0){k=0.0;l=0;while(1){if((l|0)>=(e|0)){break}m=+g[(c[h+28>>2]|0)+(f+l<<2)>>2]+k;b=d+(l<<2)|0;g[b>>2]=+g[b>>2]+m;k=m+ +g[h+12>>2];l=l+1|0}}else{k=0.0;l=0;while(1){if((l|0)>=(e|0)){break}b=d+(l<<2)|0;g[b>>2]=+g[b>>2]+(+g[(c[h+28>>2]|0)+(f+l<<2)>>2]+k);l=l+1|0}}i=1;j=i;return j|0}function mJ(e,f,h,i,j,k,l,m){e=e|0;f=f|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;m=m|0;var n=0,o=0,p=0,q=0,r=0,s=0,t=0.0,u=0,v=0,w=0,x=0,y=0.0;n=e;e=f;f=h;h=i;i=j;j=k;k=l;l=m;m=c[i>>2]|0;o=c[j>>2]|0;p=c[e>>2]|0;if((d[e+21|0]|0)==0){q=lL(n,21)|0;r=q;return r|0}while(1){if((l|0)<=0){s=2938;break}t=0.0;if((c[n+1160>>2]|0)<10){ms(n)}u=c[n+1156>>2]&1023;u=b[e+36+(u<<1)>>1]|0;if((u|0)>=0){v=d[(c[e+8>>2]|0)+u|0]|0;w=n+1156|0;c[w>>2]=(c[w>>2]|0)>>>(v>>>0);w=n+1160|0;c[w>>2]=(c[w>>2]|0)-v;if((c[n+1160>>2]|0)<0){c[n+1160>>2]=0;u=-1}}else{u=mu(n,e)|0}do{if((a[e+23|0]|0)!=0){if((u|0)<(c[e+2092>>2]|0)){s=2912;break}bV(23008,1491,31640,29824);return 0}else{s=2912}}while(0);if((s|0)==2912){s=0}if((u|0)<0){break}v=m+(ac(o,h)|0)+p|0;if((v|0)>(ac(k,h)|0)){v=ac(k,h)|0;p=v-((ac(o,h)|0)-m)|0}u=ac(u,c[e>>2]|0)|0;if((a[e+22|0]|0)!=0){x=0;while(1){if((x|0)>=(p|0)){break}y=+g[(c[e+28>>2]|0)+(u+x<<2)>>2]+t;v=(c[f+(m<<2)>>2]|0)+(o<<2)|0;g[v>>2]=+g[v>>2]+y;v=m+1|0;m=v;if((v|0)==(h|0)){m=0;o=o+1|0}t=y;x=x+1|0}}else{x=0;while(1){if((x|0)>=(p|0)){break}v=(c[f+(m<<2)>>2]|0)+(o<<2)|0;g[v>>2]=+g[v>>2]+(+g[(c[e+28>>2]|0)+(u+x<<2)>>2]+t);v=m+1|0;m=v;if((v|0)==(h|0)){m=0;o=o+1|0}x=x+1|0}}l=l-p|0}if((s|0)==2938){c[i>>2]=m;c[j>>2]=o;q=1;r=q;return r|0}do{if((a[n+1140|0]|0)==0){if((c[n+1148>>2]|0)==0){break}q=0;r=q;return r|0}}while(0);q=lL(n,21)|0;r=q;return r|0}function mK(e,f,g){e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;g=e;e=f;f=-1;if((d[e+21|0]|0)==0){h=g;lL(h,21)|0}else{if((c[g+1160>>2]|0)<10){ms(g)}f=c[g+1156>>2]&1023;f=b[e+36+(f<<1)>>1]|0;if((f|0)>=0){h=d[(c[e+8>>2]|0)+f|0]|0;i=g+1156|0;c[i>>2]=(c[i>>2]|0)>>>(h>>>0);i=g+1160|0;c[i>>2]=(c[i>>2]|0)-h;if((c[g+1160>>2]|0)<0){c[g+1160>>2]=0;f=-1}}else{f=mu(g,e)|0}if((a[e+23|0]|0)!=0){if((f|0)>=(c[e+2092>>2]|0)){bV(23008,1397,31584,30080);return 0}}if((f|0)<0){do{if((a[g+1140|0]|0)==0){if((c[g+1148>>2]|0)==0){break}j=f;k=j;return k|0}}while(0);e=g;lL(e,21)|0}}j=f;k=j;return k|0}function mL(){var a=0,b=0,d=0;if((c[8528]|0)!=0){return}c[8528]=1;a=0;while(1){if(a>>>0>=256){break}b=a;d=0;while(1){if(d>>>0>=8){break}b=b>>>1^~((b&1)-1)&-306674912;d=d+1|0}c[33088+(a<<2)>>2]=b;a=a+1|0}return}function mM(a,b,e){a=a|0;b=b|0;e=e|0;var f=0;f=a;a=b;b=e;b=~b;while(1){if((a|0)==0){break}e=f;f=e+1|0;b=c[33088+(((d[e]|0)^b&255)<<2)>>2]^b>>>8;a=a-1|0}return~b|0}function mN(b){b=b|0;var d=0,e=0,f=0;d=b;do{if((c[d+1144>>2]|0)==-1){b=(lM(d)|0)&255;if((c[d+100>>2]|0)!=0){e=0;f=e;return f|0}if(79!=(b|0)){e=lL(d,30)|0;f=e;return f|0}if(103!=((lM(d)|0)&255|0)){e=lL(d,30)|0;f=e;return f|0}if(103!=((lM(d)|0)&255|0)){e=lL(d,30)|0;f=e;return f|0}if(83!=((lM(d)|0)&255|0)){e=lL(d,30)|0;f=e;return f|0}if((mm(d)|0)==0){e=0;f=e;return f|0}if((a[d+1139|0]&1|0)==0){break}c[d+1148>>2]=0;a[d+1140|0]=0;e=lL(d,32)|0;f=e;return f|0}}while(0);e=l_(d)|0;f=e;return f|0}function mO(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0;d=a;a=b;if((c[d+32>>2]|0)==0){mP(d)}b=c[a+4>>2]|0;e=c[a+16>>2]|0;f=mQ(d,a)|0;if((c[d+32>>2]|0)==0){if((f|0)==0){f=(c[a+4>>2]|0)==(c[a+8>>2]|0)?7:8}if((f|0)!=1){c[a+4>>2]=b;c[a+16>>2]=e}g=f;return g|0}do{if((f|0)==0){if((b|0)!=(c[a+4>>2]|0)){h=3027;break}if((e|0)!=(c[a+16>>2]|0)){h=3027;break}if((c[d+36>>2]|0)!=0){f=8}c[d+36>>2]=1}else{h=3027}}while(0);if((h|0)==3027){c[d+36>>2]=0}g=f;return g|0}function mP(a){a=a|0;var b=0;b=a;c[b>>2]=0;c[b+36>>2]=0;c[b+4>>2]=0;c[b+24>>2]=0;p7(b+64|0,0,48);p7(b+112|0,0,48);c[b+160>>2]=0;c[b+164>>2]=12;return}function mQ(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=a;a=b;c[e+16>>2]=c[a+4>>2];L3864:while(1){b=c[e>>2]|0;if((b|0)==7){f=3072;break}else if((b|0)==4){f=3053}else if((b|0)==6){f=3069;break}else if((b|0)==2){f=3045}else if((b|0)==1){f=3040}else if((b|0)==5){f=3061}else if((b|0)==0){if((mV(e,a)|0)==0){f=3036;break}c[e>>2]=1;g=mW(e)|0;if((g|0)!=0){f=3038;break}f=3040}else if((b|0)==3){f=3050}else if((b|0)==8){f=3082;break}else if((b|0)==9){break}do{if((f|0)==3040){f=0;if((c[a+4>>2]|0)==(c[a+8>>2]|0)){f=3041;break L3864}if((d[(c[a>>2]|0)+(c[a+4>>2]|0)|0]|0|0)==0){b=a+4|0;h=c[b>>2]|0;c[b>>2]=h+1;c[e+16>>2]=h;c[e>>2]=6;break}else{c[e+56>>2]=(d[(c[a>>2]|0)+(c[a+4>>2]|0)|0]|0)+1<<2;c[e+164>>2]=c[e+56>>2];c[e+160>>2]=0;c[e>>2]=2;f=3045;break}}}while(0);if((f|0)==3045){f=0;if((mV(e,a)|0)==0){f=3046;break}g=mX(e)|0;if((g|0)!=0){f=3048;break}c[e>>2]=3;f=3050}if((f|0)==3050){f=0;g=mY(e,a)|0;if((g|0)!=1){f=3051;break}c[e>>2]=4;f=3053}if((f|0)==3053){f=0;while(1){h=e+64|0;if(!((c[h>>2]&3|0)!=0|(c[h+4>>2]&0|0)!=0)){break}if((c[a+4>>2]|0)==(c[a+8>>2]|0)){f=3056;break L3864}h=a+4|0;b=c[h>>2]|0;c[h>>2]=b+1;if((d[(c[a>>2]|0)+b|0]|0|0)!=0){f=3058;break L3864}b=e+64|0;h=qf(c[b>>2]|0,c[b+4>>2]|0,1,0)|0;c[b>>2]=h;c[b+4>>2]=G}c[e>>2]=5;f=3061}if((f|0)==3061){f=0;if((c[e+28>>2]|0)==1){g=mT(e,a)|0;if((g|0)!=1){f=3063;break}}else{if((mU(e,a)|0)==0){f=3066;break}}c[e>>2]=1}}do{if((f|0)==3038){i=g;j=i;return j|0}else if((f|0)==3056){i=0;j=i;return j|0}else if((f|0)==3046){i=0;j=i;return j|0}else if((f|0)==3069){g=m$(e,a)|0;if((g|0)==1){c[e>>2]=7;f=3072;break}i=g;j=i;return j|0}else if((f|0)==3048){i=g;j=i;return j|0}else if((f|0)==3058){i=7;j=i;return j|0}else if((f|0)==3066){i=0;j=i;return j|0}else if((f|0)==3036){i=0;j=i;return j|0}else if((f|0)==3051){i=g;j=i;return j|0}else if((f|0)==3041){i=0;j=i;return j|0}else if((f|0)==3063){i=g;j=i;return j|0}}while(0);do{if((f|0)==3072){while(1){b=e+120|0;h=qf(c[b>>2]|0,c[b+4>>2]|0,(c[a+4>>2]|0)-(c[e+16>>2]|0)|0,0)|0;if(!((h&3|0)!=0|(G&0|0)!=0)){f=3079;break}if((c[a+4>>2]|0)==(c[a+8>>2]|0)){f=3075;break}h=a+4|0;b=c[h>>2]|0;c[h>>2]=b+1;if((d[(c[a>>2]|0)+b|0]|0|0)!=0){f=3077;break}}if((f|0)==3077){i=7;j=i;return j|0}else if((f|0)==3075){m0(e,a);i=0;j=i;return j|0}else if((f|0)==3079){m0(e,a);if((p6(e+88|0,e+136|0,24)|0)==0){c[e>>2]=8;f=3082;break}i=7;j=i;return j|0}}}while(0);do{if((f|0)==3082){g=mT(e,a)|0;if((g|0)==1){c[e+164>>2]=12;c[e>>2]=9;break}i=g;j=i;return j|0}}while(0);if((mV(e,a)|0)!=0){i=m1(e)|0;j=i;return j|0}else{i=0;j=i;return j|0}return 0}function mR(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=pS(1200)|0;if((a|0)==0){e=0;f=e;return f|0}c[a+32>>2]=d;c[a+1192>>2]=o_(d,b)|0;if((c[a+1192>>2]|0)!=0){mP(a);e=a;f=e;return f|0}pU(a);e=0;f=e;return f|0}function mS(a){a=a|0;var b=0;b=a;if((b|0)==0){return}o0(c[b+1192>>2]|0);pU(b);return}function mT(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=a;a=b;while(1){if((c[a+4>>2]|0)==(c[a+8>>2]|0)){f=3124;break}b=(c[e+24>>2]|0)>>>((c[e+4>>2]|0)>>>0)&255;g=a+4|0;h=c[g>>2]|0;c[g>>2]=h+1;if((b|0)!=(d[(c[a>>2]|0)+h|0]|0|0)){f=3126;break}h=e+4|0;c[h>>2]=(c[h>>2]|0)+8;if((c[e+4>>2]|0)>>>0>=32){f=3129;break}}if((f|0)==3126){i=7;j=i;return j|0}else if((f|0)==3129){c[e+24>>2]=0;c[e+4>>2]=0;i=1;j=i;return j|0}else if((f|0)==3124){i=0;j=i;return j|0}return 0}function mU(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0;e=a;a=b;while(1){if((c[e+4>>2]|0)>>>0>=(d[20832+(c[e+28>>2]|0)|0]|0)>>>0){f=3139;break}if((c[a+4>>2]|0)==(c[a+8>>2]|0)){f=3137;break}b=a+4|0;c[b>>2]=(c[b>>2]|0)+1;b=e+4|0;c[b>>2]=(c[b>>2]|0)+1}if((f|0)==3137){g=0;h=g;return h|0}else if((f|0)==3139){c[e+4>>2]=0;g=1;h=g;return h|0}return 0}function mV(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,i=0;d=a;a=b;if(((c[a+8>>2]|0)-(c[a+4>>2]|0)|0)>>>0<((c[d+164>>2]|0)-(c[d+160>>2]|0)|0)>>>0){e=(c[a+8>>2]|0)-(c[a+4>>2]|0)|0}else{e=(c[d+164>>2]|0)-(c[d+160>>2]|0)|0}b=e;e=d+168+(c[d+160>>2]|0)|0;f=(c[a>>2]|0)+(c[a+4>>2]|0)|0;g=b;p3(e|0,f|0,g)|0;g=a+4|0;c[g>>2]=(c[g>>2]|0)+b;g=d+160|0;c[g>>2]=(c[g>>2]|0)+b;if((c[d+160>>2]|0)==(c[d+164>>2]|0)){c[d+160>>2]=0;h=1;i=h;return i|0}else{h=0;i=h;return i|0}return 0}function mW(a){a=a|0;var b=0,e=0,f=0;b=a;if((p6(b+168|0,28760,6)|0)!=0){e=5;f=e;return f|0}a=mM(b+174|0,2,0)|0;if((a|0)!=(mZ(b+176|0)|0)){e=7;f=e;return f|0}if((d[b+174|0]|0|0)!=0){e=6;f=e;return f|0}c[b+28>>2]=d[b+175|0]|0;if((c[b+28>>2]|0)>>>0>15){e=6;f=e;return f|0}if((c[b+28>>2]|0)>>>0>1){e=2;f=e;return f|0}else{e=0;f=e;return f|0}return 0}function mX(b){b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=b;b=e+164|0;c[b>>2]=(c[b>>2]|0)-4;b=mM(e+168|0,c[e+164>>2]|0,0)|0;if((b|0)!=(mZ(e+168+(c[e+164>>2]|0)|0)|0)){f=7;g=f;return g|0}c[e+160>>2]=2;if((a[e+169|0]&63|0)!=0){f=6;g=f;return g|0}do{if((a[e+169|0]&64|0)!=0){if((m_(e,e+168|0,e+160|0,c[e+164>>2]|0)|0)==1){b=e+8|0;h=c[b+4>>2]|0;i=e+40|0;c[i>>2]=c[b>>2];c[i+4>>2]=h;break}f=7;g=f;return g|0}else{h=e+40|0;c[h>>2]=-1;c[h+4>>2]=-1}}while(0);do{if((a[e+169|0]&128|0)!=0){if((m_(e,e+168|0,e+160|0,c[e+164>>2]|0)|0)==1){h=e+8|0;i=c[h+4>>2]|0;b=e+48|0;c[b>>2]=c[h>>2];c[b+4>>2]=i;break}f=7;g=f;return g|0}else{i=e+48|0;c[i>>2]=-1;c[i+4>>2]=-1}}while(0);if(((c[e+164>>2]|0)-(c[e+160>>2]|0)|0)>>>0<2){f=7;g=f;return g|0}i=e+160|0;b=c[i>>2]|0;c[i>>2]=b+1;if((d[e+168+b|0]|0|0)!=33){f=6;g=f;return g|0}b=e+160|0;i=c[b>>2]|0;c[b>>2]=i+1;if((d[e+168+i|0]|0|0)!=1){f=6;g=f;return g|0}if(((c[e+164>>2]|0)-(c[e+160>>2]|0)|0)>>>0<1){f=7;g=f;return g|0}i=c[e+1192>>2]|0;b=e+160|0;h=c[b>>2]|0;c[b>>2]=h+1;b=o$(i,a[e+168+h|0]|0)|0;if((b|0)!=0){f=b;g=f;return g|0}while(1){if((c[e+160>>2]|0)>>>0>=(c[e+164>>2]|0)>>>0){j=3199;break}b=e+160|0;h=c[b>>2]|0;c[b>>2]=h+1;if((d[e+168+h|0]|0|0)!=0){j=3197;break}}if((j|0)==3197){f=6;g=f;return g|0}else if((j|0)==3199){c[e+160>>2]=0;j=e+64|0;c[j>>2]=0;c[j+4>>2]=0;j=e+72|0;c[j>>2]=0;c[j+4>>2]=0;f=0;g=f;return g|0}return 0}function mY(a,b){a=a|0;b=b|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;e=a;a=b;c[e+16>>2]=c[a+4>>2];c[e+20>>2]=c[a+16>>2];b=oN(c[e+1192>>2]|0,a)|0;f=e+64|0;g=qf(c[f>>2]|0,c[f+4>>2]|0,(c[a+4>>2]|0)-(c[e+16>>2]|0)|0,0)|0;c[f>>2]=g;c[f+4>>2]=G;f=e+72|0;g=qf(c[f>>2]|0,c[f+4>>2]|0,(c[a+16>>2]|0)-(c[e+20>>2]|0)|0,0)|0;c[f>>2]=g;c[f+4>>2]=G;f=e+64|0;g=c[f+4>>2]|0;h=e+40|0;i=c[h+4>>2]|0;do{if(!(g>>>0>i>>>0|g>>>0==i>>>0&(c[f>>2]|0)>>>0>(c[h>>2]|0)>>>0)){j=e+72|0;k=c[j+4>>2]|0;l=e+48|0;m=c[l+4>>2]|0;if(k>>>0>m>>>0|k>>>0==m>>>0&(c[j>>2]|0)>>>0>(c[l>>2]|0)>>>0){break}if((c[e+28>>2]|0)==1){c[e+24>>2]=mM((c[a+12>>2]|0)+(c[e+20>>2]|0)|0,(c[a+16>>2]|0)-(c[e+20>>2]|0)|0,c[e+24>>2]|0)|0}if((b|0)==1){l=e+40|0;do{if((c[l>>2]|0)!=(-1|0)|(c[l+4>>2]|0)!=(-1|0)){j=e+40|0;m=e+64|0;if(!((c[j>>2]|0)!=(c[m>>2]|0)|(c[j+4>>2]|0)!=(c[m+4>>2]|0))){break}n=7;o=n;return o|0}}while(0);l=e+48|0;do{if((c[l>>2]|0)!=(-1|0)|(c[l+4>>2]|0)!=(-1|0)){m=e+48|0;j=e+72|0;if(!((c[m>>2]|0)!=(c[j>>2]|0)|(c[m+4>>2]|0)!=(c[j+4>>2]|0))){break}n=7;o=n;return o|0}}while(0);l=e+64|0;j=qf(c[e+56>>2]|0,0,c[l>>2]|0,c[l+4>>2]|0)|0;l=e+88|0;m=qf(c[l>>2]|0,c[l+4>>2]|0,j,G)|0;c[l>>2]=m;c[l+4>>2]=G;l=e+88|0;m=qf(c[l>>2]|0,c[l+4>>2]|0,d[20832+(c[e+28>>2]|0)|0]|0,0)|0;c[l>>2]=m;c[l+4>>2]=G;l=e+72|0;m=e+96|0;j=qf(c[m>>2]|0,c[m+4>>2]|0,c[l>>2]|0,c[l+4>>2]|0)|0;c[m>>2]=j;c[m+4>>2]=G;c[e+104>>2]=mM(e+88|0,24,c[e+104>>2]|0)|0;m=e+80|0;j=qf(c[m>>2]|0,c[m+4>>2]|0,1,0)|0;c[m>>2]=j;c[m+4>>2]=G}n=b;o=n;return o|0}}while(0);n=7;o=n;return o|0}function mZ(a){a=a|0;var b=0;b=a;return d[b|0]|0|(d[b+1|0]|0)<<8|(d[b+2|0]|0)<<16|(d[b+3|0]|0)<<24|0}function m_(b,d,e,f){b=b|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;g=b;b=d;d=e;e=f;if((c[g+4>>2]|0)==0){f=g+8|0;c[f>>2]=0;c[f+4>>2]=0}while(1){if((c[d>>2]|0)>>>0>=e>>>0){h=3244;break}i=a[b+(c[d>>2]|0)|0]|0;f=d;c[f>>2]=(c[f>>2]|0)+1;f=i&127;j=qh(f|0,((f|0)<0?-1:0)|0,c[g+4>>2]|0)|0;f=G;k=g+8|0;l=c[k+4>>2]|f;c[k>>2]=c[k>>2]|j;c[k+4>>2]=l;if((i&128|0)==0){h=3237;break}l=g+4|0;c[l>>2]=(c[l>>2]|0)+7;if((c[g+4>>2]|0)==63){h=3242;break}}if((h|0)==3237){do{if((i&255|0)==0){if((c[g+4>>2]|0)==0){break}m=7;n=m;return n|0}}while(0);c[g+4>>2]=0;m=1;n=m;return n|0}else if((h|0)==3244){m=0;n=m;return n|0}else if((h|0)==3242){m=7;n=m;return n|0}return 0}function m$(a,b){a=a|0;b=b|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0;d=a;a=b;while(1){e=m_(d,c[a>>2]|0,a+4|0,c[a+8>>2]|0)|0;if((e|0)!=1){f=3252;break}b=c[d+112>>2]|0;if((b|0)==0){g=d+8|0;h=c[g+4>>2]|0;i=d+128|0;c[i>>2]=c[g>>2];c[i+4>>2]=h;h=d+128|0;i=d+80|0;if((c[h>>2]|0)!=(c[i>>2]|0)|(c[h+4>>2]|0)!=(c[i+4>>2]|0)){f=3255;break}c[d+112>>2]=1}else if((b|0)==1){i=d+8|0;h=d+136|0;g=qf(c[h>>2]|0,c[h+4>>2]|0,c[i>>2]|0,c[i+4>>2]|0)|0;c[h>>2]=g;c[h+4>>2]=G;c[d+112>>2]=2}else if((b|0)==2){b=d+8|0;h=d+144|0;g=qf(c[h>>2]|0,c[h+4>>2]|0,c[b>>2]|0,c[b+4>>2]|0)|0;c[h>>2]=g;c[h+4>>2]=G;c[d+152>>2]=mM(d+136|0,24,c[d+152>>2]|0)|0;h=d+128|0;g=qf(c[h>>2]|0,c[h+4>>2]|0,-1,-1)|0;c[h>>2]=g;c[h+4>>2]=G;c[d+112>>2]=1}h=d+128|0;g=c[h+4>>2]|0;b=0;if(!(g>>>0>b>>>0|g>>>0==b>>>0&(c[h>>2]|0)>>>0>0>>>0)){f=3261;break}}if((f|0)==3252){m0(d,a);j=e;k=j;return k|0}else if((f|0)==3255){j=7;k=j;return k|0}else if((f|0)==3261){j=1;k=j;return k|0}return 0}function m0(a,b){a=a|0;b=b|0;var d=0,e=0,f=0;d=a;a=b;b=(c[a+4>>2]|0)-(c[d+16>>2]|0)|0;e=d+120|0;f=qf(c[e>>2]|0,c[e+4>>2]|0,b,0)|0;c[e>>2]=f;c[e+4>>2]=G;c[d+24>>2]=mM((c[a>>2]|0)+(c[d+16>>2]|0)|0,b,c[d+24>>2]|0)|0;return}function m1(a){a=a|0;var b=0,e=0,f=0,g=0,h=0;b=a;if((p6(b+178|0,22944,2)|0)!=0){e=7;f=e;return f|0}a=mM(b+172|0,6,0)|0;if((a|0)!=(mZ(b+168|0)|0)){e=7;f=e;return f|0}a=b+120|0;g=c[a+4>>2]|0;h=(c[a>>2]|0)>>>2|g<<30;if((h|0)!=(mZ(b+172|0)|0)|(g>>>2|0<<30|0)!=0){e=7;f=e;return f|0}do{if((d[b+176|0]|0|0)==0){if((d[b+177|0]|0|0)!=(c[b+28>>2]|0)){break}e=1;f=e;return f|0}}while(0);e=7;f=e;return f|0}function m2(a,d,e,f,g,i,j){a=a|0;d=d|0;e=e|0;f=f|0;g=g|0;i=i|0;j=j|0;var k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;k=a;a=d;d=e;e=f;f=g;g=i;i=j;j=k|0;l=k+1536|0;m=l|0;n=j+316|0;do{if((e&255|0)>=240){if((e&255|0)>255){break}iw(k,a,e&255,f&255);return}}while(0);o=e&255;L4169:do{if((o|0)==172){p=g|0;c[p>>2]=c[p>>2]|16;if((f&15|0)!=0){ny(g+96|0,f&15)}if(((f&240)>>4|0)!=0){nz(g+96|0,(f&240)>>4)}return}else if((o|0)==2){if((f&255|0)==0){f=c[g+260>>2]&255}else{c[g+260>>2]=f&255}do{if((c[l+1280>>2]&32|0)!=0){if((i|0)!=0){if((c[l+1280>>2]&2048|0)!=0){break}}p=(f&240)>>4;if((p|0)==15){f=f&255&15;q=3529;break L4169}else if((p|0)==14){f=f&255&15;f=(f&255|32)&255;q=3612;break L4169}else{break}}}while(0);p=g|0;c[p>>2]=c[p>>2]|8;if((f&255|0)!=0){c[g+252>>2]=f&255;if((c[l+1280>>2]&1024|0)!=0){c[g+280>>2]=f&255}}else{if((c[g+252>>2]|0)<0){p=g+252|0;c[p>>2]=(c[p>>2]|0)*-1|0}}return}else if((o|0)==1){if((f&255|0)==0){f=c[g+260>>2]&255}else{c[g+260>>2]=f&255}do{if((c[l+1280>>2]&32|0)!=0){if((i|0)!=0){if((c[l+1280>>2]&2048|0)!=0){break}}p=(f&240)>>4;if((p|0)==15){f=f&255&15;q=3522;break L4169}else if((p|0)==14){f=f&255&15;f=(f&255|16)&255;q=3612;break L4169}else{break}}}while(0);p=g|0;c[p>>2]=c[p>>2]|8;if((f&255|0)!=0){c[g+252>>2]=-(f&255);if((c[l+1280>>2]&1024|0)!=0){c[g+280>>2]=f&255}}else{if((c[g+252>>2]|0)>0){p=g+252|0;c[p>>2]=(c[p>>2]|0)*-1|0}}return}else if((o|0)==0){if((f&255|0)!=0){c[g+144>>2]=0;c[g+148>>2]=((f&240)>>4)*100|0;c[g+152>>2]=(f&15)*100|0;c[g+160>>2]=3}return}else if((o|0)==123){if((f&15|0)!=0){p=g+4|0;c[p>>2]=c[p>>2]|16}else{p=g+4|0;c[p>>2]=c[p>>2]&-17}if(((f&15)<<2|0)!=0){ny(g+96|0,(f&15)<<2)}if(((f&240)>>4|0)!=0){nz(g+96|0,(f&240)>>4)}return}else if((o|0)==5){do{if((c[g+36>>2]|0)>>>0<(c[m+140>>2]|0)>>>0){if((c[(c[m+176>>2]|0)+((c[g+36>>2]|0)*764|0)+36>>2]|0)<=0){break}m7(l,g,d&255);p=g|0;c[p>>2]=c[p>>2]|4;q=3415;break L4169}}while(0);return}else if((o|0)==7){p=g|0;c[p>>2]=c[p>>2]|32;if((f&15|0)!=0){ny(g+112|0,f&15)}if(((f&240)>>4|0)!=0){nz(g+112|0,(f&240)>>4)}return}else if((o|0)==121){p=g+4|0;c[p>>2]=c[p>>2]|8;c[g+252>>2]=-(f&255);p=f&255;c[g+260>>2]=p;if((p|0)==0){p=g+4|0;c[p>>2]=c[p>>2]&-9}return}else if((o|0)==127){q=3372}else if((o|0)==4){p=g|0;c[p>>2]=c[p>>2]|16;if(((f&15)<<2|0)!=0){ny(g+96|0,(f&15)<<2)}if(((f&240)>>4|0)!=0){nz(g+96|0,(f&240)>>4)}return}else if((o|0)==3){if((c[l+1280>>2]&131072|0)!=0){do{if((d&255|0)==0){if((c[g+272>>2]|0)!=0){break}return}}while(0)}do{if((c[g+36>>2]|0)>>>0<(c[m+140>>2]|0)>>>0){if((c[(c[m+176>>2]|0)+((c[g+36>>2]|0)*764|0)+36>>2]|0)<=0){break}m7(l,g,d&255);if(f<<24>>24!=0){c[g+280>>2]=f&255;if((c[l+1280>>2]&1024|0)!=0){c[g+260>>2]=f&255}}else{f=c[g+280>>2]&255}if(f<<24>>24!=0){c[g+276>>2]=f&255}p=g|0;c[p>>2]=c[p>>2]|4;return}}while(0);return}else if((o|0)==175){q=3522}else if((o|0)==176){q=3529}else if((o|0)==6){p=g|0;c[p>>2]=c[p>>2]|16;q=3415}else if((o|0)==163){q=3536}else if((o|0)==171){q=3540}else if((o|0)==135){if(((f&240)>>4|0)==0){p=j+20|0;c[p>>2]=(c[p>>2]|0)-(f&15);if((c[j+20>>2]|0)<32){c[j+20>>2]=32}}else{if(((f&240)>>4|0)==1){p=j+20|0;c[p>>2]=(c[p>>2]|0)+(f&15);if((c[j+20>>2]|0)>255){c[j+20>>2]=255}}else{if((f&255|0)<20){f=20}c[j+20>>2]=f&255}}h[j+40>>3]=+h[l+1256>>3]*+h[l+1248>>3]/+(c[j+20>>2]|0);return}else if((o|0)==16){if((f&255|0)>(c[l+1272>>2]|0)){c[j+312>>2]=c[l+1272>>2]}else{c[j+312>>2]=f&255}return}else if((o|0)==17){while(1){if(f<<24>>24!=0){q=3567;break}p=c[g+236>>2]&255;f=p;if((p&255|0)==0){q=3581;break}}if((q|0)==3567){p=g|0;c[p>>2]=c[p>>2]|1048576;c[g+236>>2]=f&255;r=(f&240)>>4;s=f&15;if((c[l+1280>>2]&32|0)!=0){do{if((s|0)==15){if((r|0)==0){q=3571;break}c[g+228>>2]=0;c[g+232>>2]=r}else{q=3571}}while(0);if((q|0)==3571){do{if((r|0)==15){if((s|0)==0){q=3574;break}c[g+228>>2]=0;c[g+232>>2]=-s}else{q=3574}}while(0);if((q|0)==3574){c[g+228>>2]=r-s;c[g+232>>2]=0}}}else{c[g+228>>2]=r-s;c[g+232>>2]=0}}return}else if((o|0)==33){q=3612}else if((o|0)==128){if((f&255|0)<=(c[l+1268>>2]|0)){c[g+52>>2]=f&255}return}else if((o|0)==129){do{if((f&255|0)==0){p=c[g+248>>2]&255;f=p;if((p&255|0)!=0){break}return}}while(0);if((c[l+1280>>2]&32|0)!=0){r=(f&240)>>4;s=f&15;do{if((r|0)==15){if((s|0)==0){break}c[g+248>>2]=f&255;f=f&255&15;q=3642;break L4169}}while(0);do{if((s|0)==15){if((r|0)==0){break}c[g+248>>2]=f&255;f=f&255&240;q=3642;break L4169}}while(0)}p=g|0;c[p>>2]=c[p>>2]|512;if(f<<24>>24!=0){r=(f&240)>>4;s=f&15;c[g+248>>2]=f&255;if((s|0)==0){c[g+240>>2]=r}else{if((r|0)==0){c[g+240>>2]=-s}else{p=g|0;c[p>>2]=c[p>>2]&-513}}}return}else if((o|0)==124){p=g+4|0;c[p>>2]=c[p>>2]|1;c[g+212>>2]=f&255;if((f&255|0)==0){p=g+4|0;c[p>>2]=c[p>>2]&-2}return}else if((o|0)==130){q=3642}else if((o|0)==166){q=3646}else if((o|0)==131){p=f&255;if((p|0)==0){iH(k,a,0)}else if((p|0)==1){iH(k,a,2)}else if((p|0)==2){iH(k,a,3)}else if((p|0)==3){iE(k,a,0)}else if((p|0)==4){iE(k,a,1)}else if((p|0)==5){iE(k,a,2)}else if((p|0)==6){iE(k,a,3)}return}else if((o|0)==132){c[g+332>>2]=f&255;return}else if((o|0)==133){c[g+336>>2]=f&255;return}else if((o|0)==169){return}else if((o|0)==170){return}else if((o|0)==156|(o|0)==157|(o|0)==159|(o|0)==158){if((f&255|0)!=0){if((e&255|0)==159){q=3665}else{if((e&255|0)==158){q=3665}}if((q|0)==3665){c[g+188>>2]=(f&240)>>4;c[g+192>>2]=((f&240)>>4)+1;c[g+196>>2]=0}do{if((e&255|0)==157){q=3668}else{if((e&255|0)==158){q=3668;break}c[g+304>>2]=-(f&15)}}while(0);if((q|0)==3668){c[g+304>>2]=f&15}p=(f&240)>>4;c[g+312>>2]=p;c[g+316>>2]=p}if((e&255|0)==159){q=3673}else{if((e&255|0)==158){q=3673}}if((q|0)==3673){p=g|0;c[p>>2]=c[p>>2]|262144}p=g|0;c[p>>2]=c[p>>2]|16384;return}else if((o|0)==115){p=g|0;c[p>>2]=c[p>>2]|16384;c[g+304>>2]=-(f&255);c[g+312>>2]=1;c[g+316>>2]=1;return}else if((o|0)==116){p=g|0;c[p>>2]=c[p>>2]|16384;c[g+304>>2]=f&255;c[g+312>>2]=1;c[g+316>>2]=1;return}else if((o|0)==117){p=g|0;c[p>>2]=c[p>>2]|32768;c[g+308>>2]=-(f&255);return}else if((o|0)==118){p=g|0;c[p>>2]=c[p>>2]|32768;c[g+308>>2]=f&255;return}else if((o|0)==138){p=g|0;c[p>>2]=c[p>>2]|524288;if(((f&15)<<4|0)!=0){ny(g+128|0,(f&15)<<4)}if(((f&240)>>4|0)!=0){nz(g+128|0,(f&240)>>4)}return}else if((o|0)==139){nA(g+128|0,f&3);return}else if((o|0)==27){if(f<<24>>24!=0){c[g+188>>2]=f&255}if(d<<24>>24!=0){c[g+192>>2]=(c[g+188>>2]&15)+1;c[g+196>>2]=(c[g+188>>2]&240)>>4}p=g|0;c[p>>2]=c[p>>2]|262144;return}else if((o|0)==29){if((f&255|0)!=0){c[g+208>>2]=f&255}else{f=c[g+208>>2]&255}if(((f&240)>>4|0)==0){f=(f&255|16)&255}if((f&15|0)==0){f=(f&255|1)&255}c[g+200>>2]=f&255;return}else if((o|0)==20){c[g+60>>2]=f&255;return}else if((o|0)==21){b[g+88>>1]=f&255;return}else if((o|0)==22){c[g+48>>2]=f&255;return}else if((o|0)==25){p=g|0;c[p>>2]=c[p>>2]|2;if(f<<24>>24!=0){c[g+292>>2]=(f&15)-((f&240)>>4)}return}else if((o|0)==137){p=g|0;c[p>>2]=c[p>>2]|2;if(f<<24>>24!=0){if(((f&240)>>4|0)==15){c[g+292>>2]=0;c[g+296>>2]=f&15}else{if((f&15|0)==15){c[g+292>>2]=0;c[g+296>>2]=-((f&240)>>4)}else{p=g|0;c[p>>2]=c[p>>2]|2;c[g+292>>2]=(f&15)-((f&240)>>4);c[g+296>>2]=0}}}return}else if((o|0)==13){c[j+316>>2]=1;c[j+332>>2]=(((f&240)>>4)*10|0)+(f&15);return}else if((o|0)==14){e=(f&255)>>4&255;f=f&255&15;p=e&255;if((p|0)==4){nA(g+96|0,f&3)}else if((p|0)==5){f=(f&255)<<4&255;q=3646;break}else if((p|0)==6){if((f&255|0)==0){c[(c[n+24>>2]|0)+(a<<3)>>2]=c[j+8>>2]}else{if((c[(c[n+24>>2]|0)+(a<<3)+4>>2]|0)!=0){t=(c[n+24>>2]|0)+(a<<3)+4|0;u=(c[t>>2]|0)-1|0;c[t>>2]=u;if((u|0)!=0){u=a+1|0;a=u;c[n+20>>2]=u}else{if((c[l+1280>>2]&1|0)!=0){c[(c[n+24>>2]|0)+(a<<3)>>2]=(c[j+8>>2]|0)+1}}}else{if((c[(c[n+24>>2]|0)+(a<<3)>>2]|0)<=(c[j+8>>2]|0)){c[(c[n+24>>2]|0)+(a<<3)+4>>2]=f&255;u=a+1|0;a=u;c[n+20>>2]=u}}}}else if((p|0)==1){q=3522;break}else if((p|0)==2){q=3529;break}else if((p|0)==3){c[g+68>>2]=f&255}else if((p|0)==10){q=3514;break}else if((p|0)==11){q=3518;break}else if((p|0)==12){u=g|0;c[u>>2]=c[u>>2]|262144;c[g+188>>2]=(f&255)+1;c[g+192>>2]=c[g+188>>2];c[g+196>>2]=16}else if((p|0)!=13)if((p|0)==14){c[j+324>>2]=f&255}else if((p|0)==15){c[g+320>>2]=f&255}else if((p|0)==7){nA(g+112|0,f&3)}else if((p|0)==9){p=g|0;c[p>>2]=c[p>>2]|262144;if((f&255|0)!=0){c[g+188>>2]=f&255}c[g+192>>2]=c[g+188>>2]&15;c[g+196>>2]=0}return}else if((o|0)==120){p=g+4|0;c[p>>2]=c[p>>2]|8;c[g+252>>2]=f&255;p=f&255;c[g+260>>2]=p;if((p|0)==0){p=g+4|0;c[p>>2]=c[p>>2]&-9}return}else if((o|0)==136){if((c[n+40>>2]|0)==0){c[n+36>>2]=f&255;c[n+40>>2]=1}return}else if((o|0)==113){if((f&255|0)!=0){c[g+144>>2]=0;c[g+148>>2]=(f&15)*100|0;c[g+152>>2]=0;c[g+156>>2]=((f&240)>>4)*-100|0;c[g+160>>2]=4}return}else if((o|0)==112){if((f&255|0)!=0){c[g+144>>2]=((f&240)>>4)*-100|0;c[g+148>>2]=0;c[g+152>>2]=(f&15)*100|0;c[g+160>>2]=3}return}else if((o|0)==114){if((f&255|0)!=0){c[g+144>>2]=(f&15)*100|0;c[g+148>>2]=(f&15)*100|0;c[g+152>>2]=0;c[g+160>>2]=3}return}else if((o|0)==11){c[j+316>>2]=1;c[j+320>>2]=f&255;return}else if((o|0)==12){p=g|0;c[p>>2]=c[p>>2]|4096;c[g+72>>2]=f&255;return}else if((o|0)==122){do{if((c[g+36>>2]|0)>>>0<(c[m+140>>2]|0)>>>0){if((c[(c[m+176>>2]|0)+((c[g+36>>2]|0)*764|0)+36>>2]|0)<=0){break}p=g+4|0;c[p>>2]=c[p>>2]|4;m7(l,g,d&255);c[g+276>>2]=f&255;if((f&255|0)==0){p=g+4|0;c[p>>2]=c[p>>2]&-5}return}}while(0);return}else if((o|0)==15){do{if((c[j+28>>2]&1|0)==0){if((f&255|0)<32){break}q=3540;break L4169}}while(0);q=3536}else if((o|0)==173){q=3514}else if((o|0)==174){q=3518}else if((o|0)==160){if(f<<24>>24!=0){c[g+212>>2]=f&255}p=g|0;c[p>>2]=c[p>>2]|1;return}else if((o|0)==161){if(f<<24>>24!=0){c[g+212>>2]=-(f&255)}p=g|0;c[p>>2]=c[p>>2]|1;return}else if((o|0)==165){p=g|0;c[p>>2]=c[p>>2]|64;if(f<<24>>24!=0){r=(f&240)>>4;s=f&15;if((s|0)==0){c[g+216>>2]=r}else{if((r|0)==0){c[g+216>>2]=-s}else{p=g|0;c[p>>2]=c[p>>2]&-65}}}return}else if((o|0)==164){p=g|0;c[p>>2]=c[p>>2]|8192;if(f<<24>>24!=0){r=(f&240)>>4;s=f&15;if((s|0)==0){c[g+220>>2]=r}else{if((r|0)==0){c[g+220>>2]=-s}else{p=g|0;c[p>>2]=c[p>>2]&-8193}}}return}else if((o|0)==125){p=g+4|0;c[p>>2]=c[p>>2]|1;c[g+212>>2]=-(f&255);if((f&255|0)==0){p=g+4|0;c[p>>2]=c[p>>2]&-2}return}else if((o|0)==126){if(f<<24>>24!=0){c[j+16>>2]=f&255}q=3372}else if((o|0)==8){c[g+288>>2]=f&255;return}else if((o|0)==9){p=g|0;c[p>>2]=c[p>>2]|256;if(f<<24>>24!=0){p=(f&255)<<8;c[g+80>>2]=p;c[g+84>>2]=p}else{c[g+84>>2]=c[g+80>>2]}return}else if((o|0)==10){q=3415}else{return}}while(0);do{if((q|0)==3372){c[g+4>>2]=0;return}else if((q|0)==3522){o=g|0;c[o>>2]=c[o>>2]|128;if(f<<24>>24!=0){c[g+256>>2]=-(f&255)<<2}else{if((c[g+252>>2]|0)>0){o=g+252|0;c[o>>2]=(c[o>>2]|0)*-1|0}}return}else if((q|0)==3529){o=g|0;c[o>>2]=c[o>>2]|128;if(f<<24>>24!=0){c[g+256>>2]=(f&255)<<2}else{if((c[g+252>>2]|0)<0){o=g+252|0;c[o>>2]=(c[o>>2]|0)*-1|0}}return}else if((q|0)==3536){if(f<<24>>24!=0){c[j+16>>2]=f&255}return}else if((q|0)==3540){if(f<<24>>24!=0){if((f&255|0)<20){f=20}c[j+20>>2]=f&255;h[j+40>>3]=+h[l+1256>>3]*+h[l+1248>>3]/+(c[j+20>>2]|0)}return}else if((q|0)==3612){o=g|0;c[o>>2]=c[o>>2]|128;o=(f&240)>>4;if((o|0)==1){c[g+256>>2]=-(f&15)}else if((o|0)==2){c[g+256>>2]=f&15}return}else if((q|0)==3415){while(1){q=0;if((c[l+1280>>2]&32|0)==0){break}r=(f&240)>>4;s=f&15;if((s|0)==15){if((r|0)!=0){q=3418;break}}if((r|0)==15){if((s|0)!=0){q=3421;break}}if((f&255|0)!=0){q=3426;break}o=c[g+224>>2]&255;f=o;if((o&255|0)==0){q=3425;break}q=3415}if((q|0)==3425){q=3426}else if((q|0)==3418){c[g+224>>2]=f&255;f=(f&255)>>4&255;q=3514;break}else if((q|0)==3421){c[g+224>>2]=f&255;f=f&255&15;q=3518;break}o=g|0;c[o>>2]=c[o>>2]|1;if(f<<24>>24!=0){r=(f&240)>>4;s=f&15;if((c[l+1280>>2]&512|0)!=0){o=f&255;c[g+224>>2]=o;if((o|0)!=0){if((s|0)!=0){v=-s|0}else{v=r}c[g+212>>2]=v}}else{if((s|0)==0){c[g+212>>2]=r}else{if((r|0)==0){c[g+212>>2]=-s}else{o=g|0;c[o>>2]=c[o>>2]&-2}}}}if((c[l+1280>>2]&32|0)!=0){if(((c[g+224>>2]&240)>>4|0)==15){q=3448}else{if((c[g+224>>2]&15|0)==15){q=3448}}if((q|0)==3448){o=g|0;c[o>>2]=c[o>>2]|64;c[g+216>>2]=c[g+212>>2]}}return}else if((q|0)==3642){o=g|0;c[o>>2]=c[o>>2]|1024;if(f<<24>>24!=0){c[g+244>>2]=((f&240)>>4)-(f&15)}return}else if((q|0)==3646){c[g+32>>2]=((f&255)-128&65535)<<16>>16;return}}while(0);if((q|0)==3514){l=g|0;c[l>>2]=c[l>>2]|64;if(f<<24>>24!=0){c[g+216>>2]=f&255}return}else if((q|0)==3518){q=g|0;c[q>>2]=c[q>>2]|64;if(f<<24>>24!=0){c[g+216>>2]=-(f&255)}return}}function m3(a,d,e,f){a=a|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0;g=a;a=d;d=e;e=f;f=g+28|0;c[e>>2]=0;if((~c[g>>2]&1|0)!=0){h=d;i=h;return i|0}if((c[g+4>>2]|0)<=0){h=d;i=h;return i|0}d=(c[g+4>>2]|0)-1<<1;j=b[f+(d<<1)>>1]|0;do{if((a|0)<(j|0)){if((d|0)==0){break}do{d=d-2|0;j=b[f+(d<<1)>>1]|0;if((d|0)>0){k=(j|0)>(a|0)}else{k=0}}while(k);l=b[f+(d+1<<1)>>1]|0;m=b[f+(d+2<<1)>>1]|0;do{if((c[g>>2]&4|0)!=0){if((d|0)!=(c[g+24>>2]<<1|0)){break}d=(c[g+20>>2]|0)-1<<1}}while(0);h=((ac((b[f+(d+3<<1)>>1]|0)-l|0,a-j|0)|0)/(m-j|0)|0)+l|0;i=h;return i|0}}while(0);c[e>>2]=1;h=b[f+(d+1<<1)>>1]|0;i=h;return i|0}function m4(a,d,e){a=a|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;f=a;a=d;d=e;e=f+28|0;if((a|0)<65535){a=a+1|0}if((~c[f>>2]&1|0)!=0){g=a;h=g;return h|0}if((c[f+4>>2]|0)<=0){g=a;h=g;return h|0}i=c[f>>2]&4;j=c[f>>2]&2;k=c[f+20>>2]<<1;l=c[f+24>>2]<<1;m=c[f+12>>2]<<1;n=c[f+16>>2]<<1;if((c[f>>2]&16|0)!=0){do{if((d|0)!=0){o=3790}else{if((j|0)==0){o=3790;break}if((a|0)>(b[e+(n<<1)>>1]|0)){a=b[e+(m<<1)>>1]|0}}}while(0);if((o|0)==3790){if((i|0)!=0){if((a|0)>(b[e+(l<<1)>>1]|0)){a=b[e+(k<<1)>>1]|0}}}}else{do{if((d|0)==0){if((j|0)==0){break}if((a|0)<=(b[e+(m<<1)>>1]|0)){break}a=b[e+(m<<1)>>1]|0}}while(0);do{if((i|0)!=0){if((a|0)<=(b[e+(l<<1)>>1]|0)){break}do{if((d|0)!=0){if((j|0)==0){o=3805;break}if((m|0)!=(l|0)){o=3805}}else{o=3805}}while(0);if((o|0)==3805){a=b[e+(k<<1)>>1]|0}}}while(0)}g=a;h=g;return h|0}function m5(a,d){a=a|0;d=d|0;var e=0,f=0,g=0,h=0;e=a;a=e+28|0;if((~c[e>>2]&1|0)!=0){f=0;g=f;return g|0}h=(c[e+4>>2]|0)-1<<1;if((d|0)<=(b[a+(h<<1)>>1]|0)){f=0;g=f;return g|0}if((b[a+(h+1<<1)>>1]|0)==0){f=-1;g=f;return g|0}else{f=1;g=f;return g|0}return 0}function m6(a,b,d,e,f,h){a=a|0;b=b|0;d=d|0;e=e|0;f=f|0;h=h|0;var i=0,j=0.0,k=0.0,l=0.0,m=0.0;i=b;b=d;d=e;e=f;f=h;j=+(a|0);if((i|0)<0){i=0}else{if((i|0)>255){i=255}}if((b|0)<0){b=0}else{if((b|0)>255){b=255}}k=+g[9344+(i<<2)>>2];k=k*(6.28318530717958/j);j=+g[10704+(b>>1<<2)>>2];l=(1.0-j)*k;if(l>2.0){l=2.0}m=1.0/(k*k);l=(j-l)/k+m;c[d>>2]=~~(1.0/(1.0+l)*65536.0);c[e>>2]=~~((l+m)/(1.0+l)*65536.0);c[f>>2]=~~((-0.0-m)/(1.0+l)*65536.0);return}function m7(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,i=0;g=b;b=e;e=f;f=(c[g+176>>2]|0)+((c[b+36>>2]|0)*764|0)|0;i=(c[f+756>>2]|0)+(d[f+512+(c[b+16>>2]<<1)|0]<<6)|0;do{if((e|0)>=1){if((e|0)>128){break}if((c[b+36>>2]|0)>>>0>=(c[g+140>>2]|0)>>>0){break}e=e-1|0;h[b+264>>3]=+hA(e+(c[i+12>>2]|0)+(a[f+512+(c[b+16>>2]<<1)+1|0]|0)|0,c[i+16>>2]|0,c[g+1280>>2]&4096)}}while(0);c[b+272>>2]=+h[b+24>>3]<+h[b+264>>3]?1:-1;return}function m8(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;d=a;a=b;b=c;if((d|0)>(a|0)){d=a;e=d;return e|0}if((d|0)<(b|0)){d=b}e=d;return e|0}function m9(a,b,e,f,g,h){a=a|0;b=b|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0;i=a;a=b;b=e;e=f;f=g;g=h;h=c[i+4792>>2]|0;j=c[i+4800>>2]|0;k=(d[i+60|0]|0)&32;l=i+4836|0;if((i|0)!=(c[l+32792>>2]|0)){c[l+32792>>2]=i;c[l+32796>>2]=c[i+52>>2];c[l+32800>>2]=(c[l+32796>>2]|0)+1800;c[l+32804>>2]=(c[l+32796>>2]|0)+1400;c[l+32808>>2]=(c[l+32796>>2]|0)+1480;c[l+32812>>2]=(c[l+32796>>2]|0)+1600;c[l+32816>>2]=(c[l+32796>>2]|0)+1680;c[l+32840>>2]=c[i+4796>>2];c[l+32844>>2]=c[i+4804>>2];c[l+32832>>2]=c[i+4784>>2];c[l+32836>>2]=c[i+4788>>2]}if((k&255|0)!=0){m=(c[l+32796>>2]|0)+1200|0}else{m=c[l+32800>>2]|0}k=m;while(1){m=b;b=m-1|0;if((m|0)==0){break}m=h+(c[l+32840>>2]|0)|0;h=m;c[l+32824>>2]=c[(c[l+32832>>2]|0)+(m>>>23<<2)>>2];m=j+(c[l+32844>>2]|0)|0;j=m;c[l+32828>>2]=c[(c[l+32836>>2]|0)+(m>>>23<<2)>>2];c[l+32820>>2]=0;m=c[l+32796>>2]|0;while(1){if(m>>>0>=k>>>0){break}na(m,l);m=m+200|0}m=(m8(c[l+32820>>2]|0,268427264,-268435456)|0)>>13;if((g|0)!=0){n=ac(m,f)|0;o=a;a=o+4|0;c[o>>2]=(c[o>>2]|0)+n}n=ac(m,e)|0;m=a;a=m+4|0;c[m>>2]=(c[m>>2]|0)+n}c[i+4792>>2]=h;c[i+4800>>2]=j;return}function na(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,i=0;f=b;b=e;c[b+32848>>2]=0;e=f|0;g=nv(e,b)|0;if(g>>>0<4095){if((a[e+73|0]|0)!=0){h=((ac(c[e+40>>2]|0,c[b+32828>>2]|0)|0)>>>0)/256|0;i=e+36|0;c[i>>2]=(c[i>>2]|0)+h}else{h=e+36|0;c[h>>2]=(c[h>>2]|0)+(c[e+40>>2]|0)}if((a[f+161|0]|0)!=0){h=(c[f+172>>2]|0)+(c[f+176>>2]|0)>>d[f+161|0];c[f+176>>2]=c[f+172>>2];i=c[(c[(c[e+76>>2]|0)+((((((c[e+36>>2]|0)+h|0)>>>0)/8192|0)&2047)<<2)>>2]|0)+(g<<2)>>2]|0;c[f+172>>2]=i;h=c[f+164>>2]|0;c[h>>2]=(c[h>>2]|0)+i}else{i=c[f+164>>2]|0;c[i>>2]=(c[i>>2]|0)+(c[(c[(c[e+76>>2]|0)+(((((c[e+36>>2]|0)>>>0)/8192|0)&2047)<<2)>>2]|0)+(g<<2)>>2]|0)}}else{c[f+176>>2]=c[f+172>>2];c[f+172>>2]=0}e=f+80|0;g=nv(e,b)|0;if(g>>>0>=4095){return}if((a[e+73|0]|0)!=0){f=((ac(c[e+40>>2]|0,c[b+32828>>2]|0)|0)>>>0)/256|0;i=e+36|0;c[i>>2]=(c[i>>2]|0)+f}else{f=e+36|0;c[f>>2]=(c[f>>2]|0)+(c[e+40>>2]|0)}f=b+32820|0;c[f>>2]=(c[f>>2]|0)+(c[(c[(c[e+76>>2]|0)+((((((c[e+36>>2]|0)+(c[b+32848>>2]|0)|0)>>>0)/8192|0)&2047)<<2)>>2]|0)+(g<<2)>>2]|0);return}function nb(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;a=b+4836|0;c[b+36>>2]=0;nd(b,1,0);nd(b,2,0);nd(b,3,0);nd(b,4,0);d=255;while(1){if((d|0)<32){break}nd(b,d,0);d=d-1|0}d=0;while(1){if((d|0)>=(c[b+56>>2]|0)){break}e=(c[b+52>>2]|0)+(d*200|0)|0;f=0;while(1){if((f|0)>=2){break}c[e+(f*80|0)+76>>2]=c[a+4>>2];c[e+(f*80|0)+48>>2]=536870912;c[e+(f*80|0)+52>>2]=536870913;c[e+(f*80|0)+56>>2]=0;f=f+1|0}d=d+1|0}return}function nc(a,b){a=a|0;b=b|0;var c=0,e=0,f=0;c=a;if((b&1|0)!=0){e=0;f=e;return f|0}else{e=(d[c+33|0]|0)&(d[c+34|0]|0|128)&255;f=e;return f|0}return 0}function nd(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;g=b;b=e;e=f;f=g+4836|0;h=b&224;if((h|0)==96){i=c[3112+((b&31)<<2)>>2]|0;if((i|0)==-1){return}else{nn(g,i,e);return}}else if((h|0)==0){j=b&31;if((j|0)==1){if((a[g|0]&1|0)!=0){a[g+4808|0]=e&32;if((a[g+4808|0]|0)==0){k=0;while(1){if((k|0)>=(c[g+56>>2]|0)){break}c[(c[g+52>>2]|0)+(k*200|0)+76>>2]=c[f+4>>2];c[(c[g+52>>2]|0)+(k*200|0)+156>>2]=c[f+4>>2];k=k+1|0}}}return}else if((j|0)==3){c[g+44>>2]=256-e<<4;return}else if((j|0)==2){c[g+40>>2]=256-e<<2}else if((j|0)==4){return}return}else if((h|0)==32){i=c[3112+((b&31)<<2)>>2]|0;if((i|0)==-1){return}else{nq(g,i,e);return}}else if((h|0)==160){if((b|0)==189){c[g+4784>>2]=(c[f+8>>2]|0)+(((e&128|0)!=0?512:0)<<2);c[g+4788>>2]=(c[f+12>>2]|0)+(((e&64|0)!=0?512:0)<<2);a[g+60|0]=e&63;return}if((b&15|0)>8){return}l=(c[g+52>>2]|0)+((b&15)*200|0)|0;if((b&16|0)!=0){j=e>>5&1;m=(e&31)<<8|c[l+180>>2]&255;if((d[l+196|0]|0)!=(j|0)){k=j&255;a[l+196|0]=k;if(k<<24>>24!=0){c[l+176>>2]=0;c[l+172>>2]=0;nr(l|0);nr(l+80|0)}else{ns(l|0);ns(l+80|0)}}}else{m=c[l+180>>2]&7936|e}if((c[l+180>>2]|0)!=(m|0)){c[l+180>>2]=m;c[l+192>>2]=c[32192+(m>>6<<2)>>2];c[l+188>>2]=(c[g+688+((m&1023)<<2)>>2]|0)>>>((7-(m>>10)|0)>>>0);a[l+184|0]=(c[l+180>>2]|0)>>>9&255;do{if((c[g+36>>2]&64|0)!=0){if((c[l+180>>2]&256|0)==0){break}m=l+184|0;a[m]=(d[m]|1)&255}}while(0);nt(l,l|0);nt(l,l+80|0)}return}else if((h|0)==128){i=c[3112+((b&31)<<2)>>2]|0;if((i|0)==-1){return}else{no(g,i,e);return}}else if((h|0)==192){if((b&15|0)>8){return}l=(c[g+52>>2]|0)+((b&15)*200|0)|0;m=e>>1&7;if((m|0)!=0){n=9-m|0}else{n=0}a[l+161|0]=n&255;a[l+160|0]=e&1;nu(l,f);return}else if((h|0)==64){i=c[3112+((b&31)<<2)>>2]|0;if((i|0)==-1){return}else{nm(g,i,e);return}}else if((h|0)==224){i=c[3112+((b&31)<<2)>>2]|0;if((i|0)==-1){return}l=(c[g+52>>2]|0)+(((i|0)/2|0)*200|0)|0;if((a[g+4808|0]|0)!=0){c[l+((i&1)*80|0)+76>>2]=(c[f+4>>2]|0)+((e&3)<<11<<2)}return}else{return}}function ne(b,d,e){b=b|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0;f=9;g=37688;g=g+(f*200|0)|0;h=pS(g)|0;if((h|0)==0){i=0;j=i;return j|0}p7(h|0,0,g|0);g=h;h=h+37688|0;c[g+37624>>2]=0;c[g+37628>>2]=0;if((nf(g+4836|0)|0)==-1){i=0;j=i;return j|0}else{c[g+52>>2]=h;h=h+(f*200|0)|0;a[g|0]=b&255;c[g+4>>2]=d;c[g+8>>2]=e;c[g+56>>2]=f;ng(g);nb(g);i=g;j=i;return j|0}return 0}function nf(a){a=a|0;var b=0,d=0,e=0;b=a;a=b+32788|0;c[a>>2]=(c[a>>2]|0)+1;if((c[b+32788>>2]|0)>1){d=0;e=d;return e|0}c[b+32792>>2]=0;if((np(b)|0)!=0){d=0;e=d;return e|0}else{a=b+32788|0;c[a>>2]=(c[a>>2]|0)-1;d=-1;e=d;return e|0}return 0}function ng(a){a=a|0;var b=0,d=0.0,e=0.0,f=0.0,g=0,i=0,j=0;b=a;if((c[b+8>>2]|0)!=0){d=+(c[b+4>>2]|0)/+(c[b+8>>2]|0)/72.0}else{d=0.0}h[b+16>>3]=d;h[b+24>>3]=1.0/(+(c[b+4>>2]|0)/72.0);nl(b,141280,1956e3);a=0;while(1){if((a|0)>=1024){break}c[b+688+(a<<2)>>2]=~~(+h[b+16>>3]*+(a|0)*16.0*128.0/2.0);a=a+1|0}if((c[b+8>>2]|0)!=0){e=4294967296.0/+(c[b+8>>2]|0)*3.7*(+(c[b+4>>2]|0)/36.0e5)}else{e=0.0}c[b+4796>>2]=~~e;if((c[b+8>>2]|0)!=0){f=4294967296.0/+(c[b+8>>2]|0)*6.4*(+(c[b+4>>2]|0)/36.0e5);g=~~f;i=b;j=i+4804|0;c[j>>2]=g;return}else{f=0.0;g=~~f;i=b;j=i+4804|0;c[j>>2]=g;return}}function nh(a){a=a|0;var b=0;b=a;ni(b+4836|0);pU(b);return}function ni(a){a=a|0;var b=0;b=a;if((c[b+32788>>2]|0)!=0){a=b+32788|0;c[a>>2]=(c[a>>2]|0)-1}if((c[b+32788>>2]|0)!=0){return}else{c[b+32792>>2]=0;nk(b);return}}function nj(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0;g=b;b=f;if((e&1|0)==0){a[g+32|0]=b&255;h=g;i=h+33|0;j=a[i]|0;k=j&255;l=k>>7;return l|0}if((c[g+4828>>2]|0)!=0){b1[c[g+4828>>2]&15](c[g+4832>>2]|0,0)}nd(g,d[g+32|0]|0,b);h=g;i=h+33|0;j=a[i]|0;k=j&255;l=k>>7;return l|0}function nk(a){a=a|0;var b=0;b=a;pU(c[b>>2]|0);pU(c[b+4>>2]|0);pU(c[b+8>>2]|0);pU(c[b+12>>2]|0);return}function nl(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0.0;e=a;a=b;b=d;d=0;while(1){if((d|0)>=4){break}c[e+388+(d<<2)>>2]=0;c[e+88+(d<<2)>>2]=0;d=d+1|0}d=4;while(1){if((d|0)>60){break}f=+h[e+16>>3];if((d|0)<60){f=f*(+(d&3|0)*.25+1.0)}f=f*+(1<<(d>>2)-1|0);f=f*268435456.0;c[e+88+(d<<2)>>2]=~~(f/+(a|0));c[e+388+(d<<2)>>2]=~~(f/+(b|0));d=d+1|0}d=60;while(1){if((d|0)>=76){break}c[e+88+(d<<2)>>2]=268435455;c[e+388+(d<<2)>>2]=c[e+628>>2];d=d+1|0}return}function nm(b,e,f){b=b|0;e=e|0;f=f|0;var g=0,h=0,i=0;g=b;b=e;e=f;f=(c[g+52>>2]|0)+(((b|0)/2|0)*200|0)|0;h=f+((b&1)*80|0)|0;b=e>>6;if((b|0)!=0){i=3-b|0}else{i=31}a[h+28|0]=i&255;c[h>>2]=~~(+(e&63|0)*32.0);if((c[g+36>>2]&128|0)!=0){return}c[h+4>>2]=(c[h>>2]|0)+((c[f+192>>2]|0)>>>((d[h+28|0]|0)>>>0));return}function nn(a,b,e){a=a|0;b=b|0;e=e|0;var f=0,g=0,h=0,i=0;f=a;a=b;b=e;e=(c[f+52>>2]|0)+(((a|0)/2|0)*200|0)+((a&1)*80|0)|0;a=b>>4;g=b&15;if((a|0)!=0){h=f+88+(a<<2<<2)|0}else{h=37264}c[e+12>>2]=h;c[e+60>>2]=c[(c[e+12>>2]|0)+((d[e+29|0]|0)<<2)>>2];if((d[e+45|0]|0|0)==2){c[e+56>>2]=c[e+60>>2]}if((g|0)!=0){i=f+388+(g<<2<<2)|0}else{i=37264}c[e+16>>2]=i;c[e+64>>2]=c[(c[e+16>>2]|0)+((d[e+29|0]|0)<<2)>>2];if((d[e+45|0]|0|0)!=1){return}c[e+56>>2]=c[e+64>>2];return}function no(a,b,e){a=a|0;b=b|0;e=e|0;var f=0;f=a;a=b;b=e;e=(c[f+52>>2]|0)+(((a|0)/2|0)*200|0)+((a&1)*80|0)|0;c[e+20>>2]=c[32e3+(b>>4<<2)>>2];if((d[e+45|0]|0|0)==1){c[e+52>>2]=c[e+20>>2]}c[e+24>>2]=f+388+((b&15)<<2<<2);c[e+68>>2]=c[(c[e+24>>2]|0)+((d[e+29|0]|0)<<2)>>2];if((d[e+45|0]|0|0)!=0){return}c[e+56>>2]=c[e+68>>2];return}function np(a){a=a|0;var b=0,d=0,e=0,f=0,g=0.0,h=0,i=0,j=0;b=a;a=pS(65536)|0;c[b>>2]=a;if((a|0)==0){d=0;e=d;return e|0}a=pS(32768)|0;c[b+4>>2]=a;if((a|0)==0){pU(c[b>>2]|0);d=0;e=d;return e|0}a=pS(4096)|0;c[b+8>>2]=a;if((a|0)==0){pU(c[b>>2]|0);pU(c[b+4>>2]|0);d=0;e=d;return e|0}a=pS(4096)|0;c[b+12>>2]=a;if((a|0)==0){pU(c[b>>2]|0);pU(c[b+4>>2]|0);pU(c[b+8>>2]|0);d=0;e=d;return e|0}a=0;while(1){if((a|0)>=4095){break}f=~~(67108863.0/+T(10.0,+(+(a|0)*.0234375/20.0)));c[(c[b>>2]|0)+(a<<2)>>2]=f;c[(c[b>>2]|0)+(a+8192<<2)>>2]=-(c[(c[b>>2]|0)+(a<<2)>>2]|0);a=a+1|0}a=4095;while(1){if((a|0)>=8192){break}c[(c[b>>2]|0)+(a+8192<<2)>>2]=0;c[(c[b>>2]|0)+(a<<2)>>2]=0;a=a+1|0}a=(c[b>>2]|0)+16380|0;c[(c[b+4>>2]|0)+4096>>2]=a;c[c[b+4>>2]>>2]=a;a=1;while(1){if((a|0)>512){break}g=+V(+(+(a|0)*6.283185307179586/2048.0));g=+bq(+(1.0/g))*20.0;f=~~(g/.0234375);h=(c[b>>2]|0)+(f<<2)|0;c[(c[b+4>>2]|0)+(1024-a<<2)>>2]=h;c[(c[b+4>>2]|0)+(a<<2)>>2]=h;h=(c[b>>2]|0)+(f+8192<<2)|0;c[(c[b+4>>2]|0)+(2048-a<<2)>>2]=h;c[(c[b+4>>2]|0)+(a+1024<<2)>>2]=h;a=a+1|0}a=0;while(1){if((a|0)>=2048){break}if((a|0)<1024){i=c[(c[b+4>>2]|0)+(a<<2)>>2]|0}else{i=(c[b>>2]|0)+16384|0}c[(c[b+4>>2]|0)+(a+2048<<2)>>2]=i;c[(c[b+4>>2]|0)+(a+4096<<2)>>2]=c[(c[b+4>>2]|0)+(((a|0)%1024|0)<<2)>>2];if((((a|0)/512|0)&1|0)!=0){j=(c[b>>2]|0)+16384|0}else{j=c[(c[b+4>>2]|0)+(a+4096<<2)>>2]|0}c[(c[b+4>>2]|0)+(a+6144<<2)>>2]=j;a=a+1|0}a=0;while(1){if((a|0)>=4096){break}g=+T(+(+(4095-a|0)/4096.0),8.0)*4096.0;c[b+16+(a<<2)>>2]=~~g;c[b+16+(a+4096<<2)>>2]=a;a=a+1|0}c[b+32784>>2]=4095;a=0;while(1){if((a|0)>=512){break}g=(+V(+(+(a|0)*6.283185307179586/512.0))+1.0)/2.0;c[(c[b+8>>2]|0)+(a<<2)>>2]=~~(g*42.666666666666664);c[(c[b+8>>2]|0)+(a+512<<2)>>2]=~~(g*204.79999999999998);a=a+1|0}a=0;while(1){if((a|0)>=512){break}g=+V(+(+(a|0)*6.283185307179586/512.0))*15.36;c[(c[b+12>>2]|0)+(a<<2)>>2]=~~(g*.07+256.0);c[(c[b+12>>2]|0)+(a+512<<2)>>2]=~~(g*.14+256.0);a=a+1|0}d=1;e=d;return e|0}function nq(b,d,e){b=b|0;d=d|0;e=e|0;var f=0;f=d;d=e;e=(c[b+52>>2]|0)+(((f|0)/2|0)*200|0)|0;b=e+((f&1)*80|0)|0;c[b+32>>2]=c[32128+((d&15)<<2)>>2];a[b+8|0]=((d&16|0)!=0?0:2)&255;a[b+44|0]=(d&32)>>5&255;a[b+73|0]=d&64;a[b+72|0]=d&128;nt(e,b);return}function nr(b){b=b|0;var d=0;d=b;c[d+36>>2]=0;a[d+45|0]=2;c[d+56>>2]=c[d+60>>2];c[d+48>>2]=0;c[d+52>>2]=268435456;return}function ns(b){b=b|0;var e=0;e=b;if((d[e+45|0]|0|0)<=0){return}a[e+45|0]=0;if((c[e+48>>2]&268435456|0)==0){c[e+48>>2]=268435456}c[e+52>>2]=536870912;c[e+56>>2]=c[e+68>>2];return}function nt(b,e){b=b|0;e=e|0;var f=0;f=b;b=e;c[b+40>>2]=ac(c[f+188>>2]|0,c[b+32>>2]|0)|0;e=(d[f+184|0]|0)>>(d[b+8|0]|0);if((d[b+29|0]|0|0)!=(e|0)){a[b+29|0]=e&255;c[b+60>>2]=c[(c[b+12>>2]|0)+(e<<2)>>2];c[b+64>>2]=c[(c[b+16>>2]|0)+(e<<2)>>2];c[b+68>>2]=c[(c[b+24>>2]|0)+(e<<2)>>2]}c[b+4>>2]=(c[b>>2]|0)+((c[f+192>>2]|0)>>>((d[b+28|0]|0)>>>0));return}function nu(a,b){a=a|0;b=b|0;var e=0,f=0;e=a;a=b;b=a+32820|0;if((d[e+160|0]|0|0)!=0){f=b}else{f=a+32848|0}c[e+164>>2]=f;c[e+168>>2]=b;return}function nv(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,i=0;f=b;b=e;e=f+48|0;g=(c[e>>2]|0)+(c[f+56>>2]|0)|0;c[e>>2]=g;if((g|0)>=(c[f+52>>2]|0)){g=d[f+45|0]|0;if((g|0)==0){c[f+48>>2]=536870912;c[f+52>>2]=536870913;c[f+56>>2]=0}else if((g|0)==2){a[f+45|0]=1;c[f+48>>2]=268435456;c[f+52>>2]=c[f+20>>2];c[f+56>>2]=c[f+64>>2]}else if((g|0)==1){c[f+48>>2]=c[f+20>>2];c[f+52>>2]=536870912;if((a[f+44|0]|0)!=0){c[f+56>>2]=0}else{a[f+45|0]=0;c[f+56>>2]=c[f+68>>2]}}}g=(c[f+4>>2]|0)+(c[b+16+(c[f+48>>2]>>16<<2)>>2]|0)|0;if((d[f+72|0]|0)!=0){h=c[b+32824>>2]|0;i=g+h|0;return i|0}else{h=0;i=g+h|0;return i|0}return 0}function nw(a){a=a|0;var b=0;b=a;a=b+12|0;c[a>>2]=(c[a>>2]|0)+(c[b+4>>2]|0);a=b+12|0;c[a>>2]=(c[a>>2]|0)%64|0;return}function nx(a,b){a=a|0;b=b|0;c[a+12>>2]=b;return}function ny(a,b){a=a|0;b=b|0;c[a+8>>2]=b;return}function nz(a,b){a=a|0;b=b|0;c[a+4>>2]=b;return}function nA(a,b){a=a|0;b=b|0;c[a>>2]=b;return}function nB(a){a=a|0;var b=0,d=0,e=0,f=0;b=a;if((c[b+4>>2]|0)==0){d=0;e=d;return e|0}a=c[b>>2]|0;if((a|0)==3){f=((a9()|0)&511)-256|0;d=ac(f,c[b+8>>2]|0)|0;e=d;return e|0}else if((a|0)==0|(a|0)==1|(a|0)==2|(a|0)==4){d=ac(c[144+(c[b>>2]<<8)+(c[b+12>>2]<<2)>>2]|0,c[b+8>>2]|0)|0;e=d;return e|0}else{d=0;e=d;return e|0}return 0}function nC(a,b,d){a=a|0;b=b|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;e=a;a=b;b=d;d=e|0;if((a|0)==0){f=-1;g=f;return g|0}if((jl(a)|0)!=1179603533){f=-1;g=f;return g|0}h=jl(a)|0;if((jl(a)|0)!=1095977286){f=-1;g=f;return g|0}i=0;while(1){if((i|0)>=2){break}j=jl(a)|0;h=jl(a)|0;k=(aT(a|0)|0)+h|0;l=j;if((l|0)==1463899717){j=a;m=(ja(a)|0)&255;bK(j|0,m|0,1)|0;m=((jc(a)|0)&65535)+1|0;c[(c[d+180>>2]|0)+(b*52|0)+32>>2]=m;h=(jc(a)|0)&65535;m=0;while(1){if((m|0)>=(h|0)){break}jc(a)|0;j=((jc(a)|0)&65535)<<8;c[(c[d+180>>2]|0)+(m*52|0)+32>>2]=j;jc(a)|0;jc(a)|0;m=m+1|0}gQ(e,a,2,(c[d+180>>2]|0)+(b*52|0)|0,0)|0;i=i+1|0}else if((l|0)==1229869908){bK(a|0,(ja(a)|0)&255|0,1)|0;jc(a)|0;bK(a|0,24,1)|0;ja(a)|0;ja(a)|0;ja(a)|0;ja(a)|0;ja(a)|0;ja(a)|0;c[(c[d+176>>2]|0)+(b*764|0)+36>>2]=1;c[c[(c[d+176>>2]|0)+(b*764|0)+756>>2]>>2]=64;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+8>>2]=128;c[(c[(c[d+176>>2]|0)+(b*764|0)+756>>2]|0)+40>>2]=b;i=i+1|0}bK(a|0,k|0,0)|0}f=0;g=f;return g|0}function nD(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=c;c=0;if((b|0)<896){f=896-b|0;g=f;return g|0}do{if((d[e+2|0]|0|0)==172){if((d[e+3|0]|0|0)!=29){break}if((d[e+c|0]|0|0)>127){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=31){break}if((d[e+(c+10+(b<<3))|0]|0|0)>15){h=4208;break}b=b+1|0}if((h|0)==4208){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=128){h=4217;break}if((d[e+(c+768+b)|0]|0|0)>127){h=4214;break}b=b+1|0}if((h|0)==4214){f=-1;g=f;return g|0}else if((h|0)==4217){f3(0,a,0);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function nE(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;f=i;i=i+2048|0;g=f|0;h=f+1024|0;j=f+1536|0;k=b;b=e;e=-1;l=0;p7(h|0,0,512);p7(j|0,0,512);m=ja(k)|0;n=ja(k)|0;jd(k)|0;o=jl(k)|0;f0(b,20)|0;p=0;while(1){if((p|0)>=31){break}f0(b,22)|0;q=(jd(k)|0)&65535;jo(b,q&65535);l=l+(q<<1)|0;jm(b,ja(k)|0);jm(b,ja(k)|0);jo(b,jd(k)|0);jo(b,jd(k)|0);p=p+1|0}q=0;while(1){if((q&255|0)>=128){break}c[h+((q&255)<<2)>>2]=jl(k)|0;if((c[h+((q&255)<<2)>>2]|0)==0){r=4232;break}q=q+1&255}q=q-1&255;p=0;while(1){if((p|0)>=((q&255)-1|0)){break}c[j+(p<<2)>>2]=(c[h+(p+1<<2)>>2]|0)-(c[h+(p<<2)>>2]|0);p=p+1|0}jm(b,m);jm(b,n);bK(k|0,768,0)|0;f$(b,k,128)|0;jq(b,1294879534);p=0;while(1){if((p|0)>=(q&255|0)){break}bK(k|0,c[h+(p<<2)>>2]|0,0)|0;jl(k)|0;jl(k)|0;jl(k)|0;p7(g|0,0,1024);n=0;while(1){if((n|0)>=4){break}m=0;while(1){if((m|0)>=64){break}j=(m<<4)+(n<<2)|0;r=0;s=0;t=0;u=0;v=ja(k)|0;do{if((v&128|0)!=0){m=m+((v&127&255)-1)|0}else{w=ja(k)|0;t=((v&192)>>2|(w&255)>>4&15)&255;u=v&255&63;if((u&255|0)==63){u=e}else{if(u<<24>>24!=0){u=(u&255)-11&255}}if((u&255|0)==0){u=u+1&255}a[g+j|0]=t&255&240;if((u&255|0)!=(e&255|0)){x=g+j|0;a[x]=(d[x]|0|(d[5440+((u&255)<<1)|0]|0))&255;a[g+(j+1)|0]=a[5441+((u&255)<<1)|0]|0}if((w&15|0)==7){s=0;r=0;a[g+(j+2)|0]=(t&255)<<4&240;break}else{s=w&255&15;r=ja(k)|0;a[g+(j+2)|0]=((t&255)<<4&240|s&255)&255;a[g+(j+3)|0]=r;break}}}while(0);m=m+1|0}n=n+1|0}ay(g|0,1024,1,b|0)|0;p=p+1|0}bK(k|0,o|0,0)|0;f$(b,k,l)|0;i=f;return 0}function nF(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;e=a;a=b;b=c;c=0;if((b|0)<21){f=21-b|0;g=f;return g|0}h=(ji(e+c|0)|0)&65535;if((h|0)>31){f=-1;g=f;return g|0}i=0;j=0;while(1){if((j|0)>=(h|0)){k=4285;break}l=((ji(e+c+14|0)|0)&65535)<<1;m=((ji(e+c+18|0)|0)&65535)<<1;n=((ji(e+c+20|0)|0)&65535)<<1;if((l|0)>65535){k=4276;break}if((m|0)>65535){k=4276;break}if((n|0)>65535){k=4276;break}if((m+n|0)>(l|0)){k=4278;break}if((d[e+(c+16+(j<<3))|0]|0|0)>15){k=4280;break}if((d[e+(c+17+(j<<3))|0]|0|0)>64){k=4282;break}i=i+l|0;j=j+1|0}if((k|0)==4282){f=-1;g=f;return g|0}else if((k|0)==4280){f=-1;g=f;return g|0}else if((k|0)==4285){if((i|0)<=2){f=-1;g=f;return g|0}l=(h<<3)+2|0;j=jk(e+c+2|0)|0;h=jk(e+c+6|0)|0;i=jk(e+c+10|0)|0;do{if((h|0)>(j|0)){if((i|0)<=(j|0)){break}if((i|0)<=(h|0)){break}if((h-j|0)>128){f=-1;g=f;return g|0}if((j|0)<(l|0)){f=-1;g=f;return g|0}if((b|0)<(c+h-1|0)){f=c+h-1-b|0;g=f;return g|0}m=j;while(1){if((m|0)>=(h-1|0)){break}if((d[e+(c+m)|0]|0|0)>128){k=4302;break}m=m+1|0}if((k|0)==4302){f=-1;g=f;return g|0}if((d[e+(c+h-1)|0]|0|0)!=255){f=-1;g=f;return g|0}if((i|0)>65535){f=-1;g=f;return g|0}else{f3(0,a,0);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}else if((k|0)==4276){f=-1;g=f;return g|0}else if((k|0)==4278){f=-1;g=f;return g|0}return 0}function nG(c,f){c=c|0;f=f|0;var g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;g=i;i=i+448|0;h=g|0;j=g+8|0;k=g+136|0;l=g+392|0;m=c;c=f;f=0;p7(j|0,0,128);p7(h|0,0,5);p7(k|0,0,128);f0(c,20)|0;n=(jd(m)|0)&255;o=jl(m)|0;jl(m)|0;p=jl(m)|0;q=0;while(1){if((q|0)>=(n&255|0)){break}f0(c,22)|0;r=(jd(m)|0)&65535;jo(c,r&65535);f=f+(r<<1)|0;jm(c,ja(m)|0);jm(c,ja(m)|0);jo(c,jd(m)|0);jo(c,jd(m)|0);q=q+1|0}p7(l|0,0,50);q=n&255;while(1){if((q|0)>=31){break}ay(l|0,30,1,c|0)|0;q=q+1|0}l=aT(m|0)|0;bK(m|0,o|0,0)|0;q=0;do{s=ja(m)|0;o=q;q=o+1|0;a[j+o|0]=s;}while((s&255|0)!=255);a[j+(q-1)|0]=0;jm(c,q-1&255);jm(c,127);q=0;o=0;while(1){if((q|0)>=128){break}jm(c,a[j+q|0]|0);if((d[j+q|0]|0|0)>(o&255|0)){o=a[j+q|0]|0}q=q+1|0}jq(c,1294879534);bK(m|0,l|0,0)|0;q=0;while(1){if((q|0)>(o&255|0)){break}b[k+(q<<1)>>1]=jd(m)|0;q=q+1|0}q=0;while(1){if((q|0)>(o&255|0)){break}bK(m|0,e[k+(q<<1)>>1]|0|0,0)|0;l=0;while(1){if((l|0)>=256){break}p7(h|0,0,5);s=ja(m)|0;do{if((s&128|0)==0){t=ja(m)|0;u=((s&255)<<4&48|(t&255)>>4&15)&255;a[h|0]=a[5440+((u&255)<<1)|0]|0;a[h+1|0]=a[5441+((u&255)<<1)|0]|0;v=(s&255)>>2&31;j=h|0;a[j]=(d[j]|0|v&240)&255;a[h+2|0]=(v&255)<<4&240;w=t&255&15;j=h+2|0;a[j]=(d[j]|0|w&255)&255;x=0;a[h+3|0]=x;j=h|0;n=c;ay(j|0,4,1,n|0)|0}else{if((s&255|0)==255){p7(h|0,0,5);n=h|0;j=c;ay(n|0,4,1,j|0)|0;break}else{t=ja(m)|0;j=ja(m)|0;u=((s&255)<<4&48|(t&255)>>4&15)&255;a[h|0]=a[5440+((u&255)<<1)|0]|0;a[h+1|0]=a[5441+((u&255)<<1)|0]|0;v=(s&255)>>2&31;n=h|0;a[n]=(d[n]|0|v&240)&255;a[h+2|0]=(v&255)<<4&240;w=t&255&15;n=h+2|0;a[n]=(d[n]|0|w&255)&255;x=j;a[h+3|0]=x;j=h|0;n=c;ay(j|0,4,1,n|0)|0;break}}}while(0);l=l+1|0}q=q+1|0}bK(m|0,p|0,0)|0;f$(c,m,f)|0;i=g;return 0}function nH(b,c,e){b=b|0;c=c|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;f=b;b=c;c=e;e=0;if((c|0)<1084){g=1084-c|0;h=g;return h|0}i=d[f+(e+950)|0]|0;do{if((i|0)!=0){if((i|0)>127){break}j=0;while(1){if((j|0)>=31){k=4380;break}l=((d[f+(e+42+(j*30|0))|0]|0)<<8)+(d[f+(e+43+(j*30|0))|0]|0)|0;m=((d[f+(e+46+(j*30|0))|0]|0)<<8)+(d[f+(e+47+(j*30|0))|0]|0)|0;n=((d[f+(e+48+(j*30|0))|0]|0)<<8)+(d[f+(e+49+(j*30|0))|0]|0)|0;l=l<<1;m=m<<1;n=n<<1;if((l|0)>65535){k=4372;break}if((m|0)>65535){k=4372;break}if((n|0)>65535){k=4372;break}if((m+n|0)>(l+2|0)){k=4374;break}if((d[f+(e+44+(j*30|0))|0]|0|0)>15){k=4377;break}if((d[f+(e+45+(j*30|0))|0]|0|0)>64){k=4377;break}j=j+1|0}if((k|0)==4380){o=((d[f+(e+1080)|0]|0)<<24)+((d[f+(e+1081)|0]|0)<<16)+((d[f+(e+1082)|0]|0)<<8)+(d[f+(e+1083)|0]|0)|0;if((o|0)<1084){g=-1;h=g;return h|0}m=0;j=0;while(1){if((j|0)>=(i|0)){break}n=d[f+(e+952+j)|0]|0;if((n|0)>(m|0)){m=n}if((n|0)>127){k=4387;break}j=j+1|0}if((k|0)==4387){g=-1;h=g;return h|0}j=j+2|0;while(1){if((j|0)==128){break}if((d[f+(e+952+j)|0]|0|0)!=0){k=4393;break}j=j+1|0}if((k|0)==4393){g=-1;h=g;return h|0}m=m+1|0;n=0;i=999999;if((c|0)<(e+(m<<2<<1)+1085|0)){g=e+(m<<2<<1)+1085-c|0;h=g;return h|0}j=0;while(1){if((j|0)>=(m<<2|0)){break}l=((d[f+(e+(j<<1)+1084)|0]|0)<<8)+(d[f+(e+(j<<1)+1085)|0]|0)|0;if((l|0)>(o|0)){k=4403;break}if((l|0)<1084){k=4403;break}if((l|0)>(n|0)){n=l}if((l|0)<(i|0)){i=l}j=j+1|0}if((k|0)==4403){g=-1;h=g;return h|0}j=i;L5574:while(1){if((j|0)>=(l|0)){k=4427;break}do{if((a[f+(e+j)|0]&192|0)!=192){if((a[f+(e+j)|0]&192|0)==128){j=j+2|0;break}if((a[f+(e+j)|0]&192|0)==64){if((a[f+(e+j)|0]&63|0)==0){if((d[f+(e+j+1)|0]|0|0)==0){k=4419;break L5574}}j=j+1|0;break}if((a[f+(e+j)|0]&192|0)!=0){break}if((d[f+(e+j)|0]|0|0)>19){k=4423;break L5574}j=j+3|0}}while(0);j=j+1|0}if((k|0)==4423){g=-1;h=g;return h|0}else if((k|0)==4427){f3(f,b,20);g=0;h=g;return h|0}else if((k|0)==4419){g=-1;h=g;return h|0}}else if((k|0)==4377){g=-1;h=g;return h|0}else if((k|0)==4374){g=-1;h=g;return h|0}else if((k|0)==4372){g=-1;h=g;return h|0}}}while(0);g=-1;h=g;return h|0}function nI(b,e){b=b|0;e=e|0;var f=0,g=0,h=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;f=i;i=i+3128|0;g=f|0;h=f+1080|0;j=b;b=e;e=0;bB(g|0,1080,1,j|0)|0;ay(g|0,1080,1,b|0)|0;k=0;while(1){if((k|0)>=31){break}e=e+(((ji(g+(k*30|0)+42|0)|0)&65535)<<1)|0;k=k+1|0}k=0;l=0;while(1){if((k|0)>=128){break}if((d[g+(k+952)|0]|0|0)>(l|0)){l=d[g+(k+952)|0]|0}k=k+1|0}l=l+1|0;jq(b,1294879534);m=jl(j)|0;k=0;while(1){if((k|0)>=(l|0)){break}n=0;while(1){if((n|0)>=4){break}c[h+(k<<4)+(n<<2)>>2]=(jd(j)|0)&65535;n=n+1|0}k=k+1|0}k=0;while(1){if((k|0)>=(l|0)){break}p7(g|0,0,1024);n=0;while(1){if((n|0)>=4){break}bK(j|0,c[h+(k<<4)+(n<<2)>>2]|0,0)|0;o=0;while(1){if((o|0)>=64){break}p=g+((o<<4)+(n<<2))|0;q=ja(j)|0;do{if((q&192|0)==0){r=p;p=r+1|0;a[r]=q;r=p;p=r+1|0;a[r]=ja(j)|0;r=p;p=r+1|0;a[r]=ja(j)|0;r=p;p=r+1|0;a[r]=ja(j)|0}else{if((q&192|0)==192){o=o+(q&63)|0;break}if((q&192|0)==64){p=p+2|0;r=p;p=r+1|0;a[r]=q&255&15;r=p;p=r+1|0;a[r]=ja(j)|0;break}if((q&192|0)==128){r=p;p=r+1|0;a[r]=ja(j)|0;r=p;p=r+1|0;a[r]=ja(j)|0;r=p;p=r+1|0;a[r]=(q&255)<<4&240;break}else{break}}}while(0);o=o+1|0}n=n+1|0}ay(g|0,1024,1,b|0)|0;k=k+1|0}bK(j|0,m|0,0)|0;f$(b,j,e)|0;i=f;return 0}function nJ(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0;e=a;a=b;b=c;c=0;if((b|0)<285){f=285-b|0;g=f;return g|0}do{if((d[e|0]|0|0)==70){if((d[e+1|0]|0|0)!=67){break}if((d[e+2|0]|0|0)!=45){break}if((d[e+3|0]|0|0)!=77){break}if((d[e+(c+4)|0]|0|0)!=1){f=-1;g=f;return g|0}if((d[e+(c+5)|0]|0|0)!=0){f=-1;g=f;return g|0}b=0;while(1){if((b|0)>=31){h=4500;break}if((d[e+(c+37+(b<<3))|0]|0|0)>64){h=4497;break}b=b+1|0}if((h|0)==4497){f=-1;g=f;return g|0}else if((h|0)==4500){f3(e+10|0,a,20);f=0;g=f;return g|0}}}while(0);f=-1;g=f;return g|0}function nK(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,j=0;c=i;i=i+128|0;d=a;a=b;b=0;p7(c|0,0,128);jl(d)|0;jd(d)|0;jl(d)|0;f$(a,d,20)|0;jl(d)|0;e=0;while(1){if((e|0)>=31){break}f0(a,22)|0;f=(jd(d)|0)&65535;g=f;jo(a,f&65535);b=b+(g<<1)|0;jm(a,ja(d)|0);jm(a,ja(d)|0);jo(a,jd(d)|0);g=(jd(d)|0)&65535;if((g|0)==0){g=1}jo(a,g&65535);e=e+1|0}jl(d)|0;g=ja(d)|0;f=g;jm(a,g);jm(a,ja(d)|0);jl(d)|0;e=0;g=0;while(1){if((e|0)>=(f&255|0)){break}h=ja(d)|0;j=h;jm(a,h);if((j&255|0)>(g&255|0)){g=j}e=e+1|0}while(1){if((e|0)>=128){break}jm(a,0);e=e+1|0}jq(a,1294879534);jl(d)|0;e=0;while(1){if((e|0)>(g&255|0)){break}f$(a,d,1024)|0;e=e+1|0}jl(d)|0;f$(a,d,b)|0;i=c;return 0}function nL(a,b,c){a=a|0;b=b|0;c=c|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0;c=a;a=b;b=0;if((jk(c+192|0)|0)!=1397706311){e=-1;f=e;return f|0}g=((d[c+(b+10)|0]|0)<<24)+((d[c+(b+11)|0]|0)<<16)+((d[c+(b+12)|0]|0)<<8)+(d[c+(b+13)|0]|0)|0;do{if((g|0)>2){if((g|0)>=1048560){break}h=0;i=0;while(1){if((i|0)>=16){j=4542;break}k=((d[c+(b+(i<<1)+14)|0]|0)<<8)+(d[c+(b+(i<<1)+15)|0]|0)|0;l=((d[c+(b+(i<<1)+78)|0]|0)<<8)+(d[c+(b+(i<<1)+79)|0]|0)|0;if((